{ *******************************************************************
  *			    This file is part of the WMFH package				*
  ******************************************************************* }


{
	  This program translates the text of a help file into the format
	  required for use with:
 
	  Hlp_unit.pas	A Turbo Pascal unit for displaying help	files


			 Copyright (c) 1997   Gianfranco Boggio-Togna                              *

							C.P. 14021                                   
					  I-20140 Milano (Italy)                             

					 e-mail: gbt@computer.org                                                     *


	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

}

{DEFINE DEBUG}
{$R+}

PROGRAM Make_HLP ;

USES	Dos, HLP_Type ;

TYPE
	Buffer 	   = ARRAY  [0..MAX_SECTION_LENGTH] OF Char ;
	Buffer_ptr = ^ Buffer ;

CONST
		Copyright:	String =
					'WMFH 1.0 - Copyright 1997 Gianfranco Boggio-Togna' ;

VAR
	input_file_name:	String ;
	output_file_name:	String ;
	input_file:			Text ;
	output_file:		File ;
	input_line:			String ;

	section_table:		ARRAY [0..MAX_SECTIONS-1] OF Section ;
	area_table:			ARRAY [1..MAX_AREAS] OF Area ;

	section_names:		ARRAY [0..MAX_SECTIONS-1] OF Line_ptr ;

	identifier: 		Integer ;

	section_lines:		Integer ;
	max_section_lines:	Integer ;

	section_index: 		Integer ;
	area_index: 		Integer ;

	section_count: 		Integer ;
	area_count: 		Integer ;

	end_of_input:		Boolean ;

	pass:				Integer ;

	line_number:		Integer ;

	errors:  			Boolean ;

	section_buffer:		Buffer_ptr ;

	buffer_length:		Word ;

	des:				Descriptor ;


{ ************************************************************************
  *																		 *
  *				     Write error message and die 						 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	fatal_error (s: String) ;
BEGIN
	Writeln ('MAKEHELP Fatal error - ',s) ;
	Halt ;
END ;


{ ************************************************************************
  *																		 *
  *					       Write error message 							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	error_message (s: String; i: Integer) ;
VAR
	t:	Integer ;
BEGIN
	errors := True ;
	Writeln ('MAKEHELP Error at line ', line_number, ':  ',s ) ;	
	Writeln	(input_line) ;
	IF  i <> 0  THEN
	  BEGIN
		FOR  t := 1  TO  i-1  DO
	  		Write (' ') ;
		Write ('^') ;
	  END ;
	Writeln (' ') ;
END ;


{ ************************************************************************
  *																		 *
  *				     Check the command line parameters					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE   check_parameters ;

BEGIN

	IF  (ParamCount = 0)  OR  (ParamCount > 2) THEN
		fatal_error ('Invalid number of command line parameters') ;

	input_file_name := ParamStr(1) ;
	IF  Pos('.', input_file_name) = 0  THEN
		input_file_name := input_file_name + '.txt' ;
	    
	 Assign (input_file, input_file_name) ;	     	   
{$i-}							   
	 Reset (input_file) ;				
{$i+}							   
	 IF  IOResult <> 0  THEN				   
	 	fatal_error ('Cannot open ' + ParamStr(1)) ;

	 output_file_name := Copy (input_file_name,	1,
	 											Pos('.', input_file_name)-1) ;
	 output_file_name := output_file_name + '.hlp' ;

	IF  ParamCount = 2  THEN
	  BEGIN
	    output_file_name := ParamStr(2) ;
	    IF  Pos('.', output_file_name) = 0  THEN
	    output_file_name := output_file_name + '.hlp' ;
	  END ;
  
	IF  input_file_name = output_file_name  THEN
		fatal_error ('Same file for input and output') ;

	Assign (output_file, output_file_name) ;	     	   
{$i-}							   
	Rewrite (output_file, 1) ;				
{$i+}							   
	IF  IOResult <> 0  THEN				   
	   	fatal_error ('Cannot open ' + ParamStr(2)) ;
END ;


{ ************************************************************************
  *																		 *
  *				 Read a number at current position in input line		 *
  *																		 *
  ************************************************************************ }
			
FUNCTION  read_number (VAR i: Integer) : Integer ;
VAR
	n:	Integer ;
BEGIN
	IF  (i <= Length(input_line))  AND  (input_line[i] IN ['0'..'9'])  THEN
	  BEGIN
		n := 0 ;
		WHILE  (i <= Length(input_line))  AND  
			   (input_line[i] IN ['0'..'9'])  DO
		BEGIN
			n := n * 10 + Ord(input_line[i]) - Ord('0') ;
			Inc (i) ;
		END ;
		read_number := n ;
	  END
	ELSE
	  read_number := -1 ;
END ;


{ ************************************************************************
  *																		 *
  *						      Read a line  								 *
  *																		 *
  ************************************************************************ }
			
FUNCTION	read_a_line : Boolean ;
VAR
	work:	String ;
BEGIN

{$i-}							   
	Readln (input_file, input_line) ;
{$i+}							   
	IF  Eof (input_file)  THEN
		read_a_line := False 
	ELSE
		IF  IOResult <> 0  THEN
	    	fatal_error ('Error on input file') 
		ELSE
		  BEGIN
			WHILE  input_line[Length(input_line)] = ESC  DO 
			  BEGIN
{$i-}							   
				Readln (input_file, work) ;
{$i+}							   
				IF  Eof (input_file)  THEN
			    	fatal_error ('Missing continuation line') ;

				input_line[0] := Chr(Length(input_line)-1) ;
				IF  Length(input_line) + Length(work) > 256  THEN
			    	fatal_error ('Line longer than 255') ;
				input_line := input_line + work ;
			  END ;
			read_a_line := True ; 
			Inc (line_number) ;
		  END ;
END ;


{ ************************************************************************
  *																		 *
  *					  Process an area definition   						 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	process_area_definition (i: Integer) ;

VAR
	area_number: Integer ;

BEGIN

	IF  pass = 1  THEN
	  BEGIN
		IF  area_index <> 0  THEN
		  BEGIN
			IF  (section_index = -1)  AND
				(area_table [area_index].area_first <> -1)  THEN
				error_message ('Misplaced \area statement', 0) ;
			area_table [area_index].area_last := section_index  ;
		  END ;
		WHILE  (i <= Length(input_line))  AND  (input_line[i] = ' ')  DO
	  		Inc (i) ;
		IF  i > Length(input_line)  THEN
			error_message ('Missing area number', i) 
		ELSE
		  IF  NOT (input_line[i] IN ['0'..'9'])  THEN
		  	error_message ('\area must be followed by a number', i) 
		  ELSE
			BEGIN
		  		area_number := read_number (i) ;
				IF  (area_number <> (area_index + 1))  OR
				    (area_number > MAX_AREAS)  THEN
				  	error_message ('Invalid area number', i) 
				ELSE
				  BEGIN
					IF  area_table [area_number].area_first <> -1 THEN
						error_message ('Misplaced \area statement', 0) 
					ELSE
					  BEGIN
					  	area_index := area_number ;
					  	area_count := area_number ;
					    area_table [area_number].area_first := -1 ;
					  END ;
				  END ;
			END ;
	  END ;		
END	;


{ ************************************************************************
  *																		 *
  *				    Process character(s) definition						 *
  *																		 *
  ************************************************************************ }
			
FUNCTION  	get_characters (VAR i: Integer; VAR s: String) : String ;

VAR
	c, k, n: 	Integer ;


FUNCTION	get_a_char (VAR i: Integer): Integer ;
BEGIN
	n := read_number (i) ;
	IF  n <> -1  THEN
	  BEGIN
		IF  n > 255  THEN
		  BEGIN
			error_message ('Character code > 255', i) ;
			get_characters := '' ;
		  END 
		ELSE
		  BEGIN
			get_a_char := n ;
			s := s + Chr(n) ;
		  END ;
		END
	ELSE
	  BEGIN
		error_message ('Invalid or missing character code', i) ;
		get_characters := '' ;
		get_a_char := -1 ;
	  END ;
END ;

BEGIN
		s := '' ;
		get_characters := s ;
		Inc (i) ;	

		WHILE  (i <= Length(input_line))  AND  (input_line[i] <> '}')  DO
		  BEGIN

			c := get_a_char (i);

			IF  i > Length(input_line)  THEN
				Exit ;

			IF  input_line[i] = '*'  THEN
			  BEGIN
				Inc (i) ;
				n := read_number (i) ;
				IF  (n <= 0)  OR  (n > MAX_LINE_LENGTH)  THEN
				  BEGIN
					error_message ('Invalid or missing repeat count', i) ;
					get_characters := '' ;
				  END
				ELSE
				  	FOR  k := 1  TO  n-1  DO
						s := s + Chr(c) ;
				END ;

			IF  input_line[i] = '-'  THEN
			  BEGIN
				Inc (i) ;
				n := read_number (i) ;
				IF  (n = -1)  OR  (n < c)  THEN
				  BEGIN
					error_message ('Invalid character range', i) ;
					get_characters := '' ;
				  END
				ELSE
					FOR  c := c+1  TO  n  DO
						s := s + Chr(c) ;
			  END ;

			IF  i > Length(input_line)  THEN
				Exit ;

			IF  input_line[i] = ','  THEN
				Inc (i) ;

		  END ;

		  IF  i > Length(input_line)  THEN
			BEGIN
			  error_message ('Missing }', i) ;
			  get_characters := '' ;
			END 
		  ELSE
			BEGIN
			  Inc (i) ;
			  get_characters := s ;
			END ;

END ;



{ ************************************************************************
  *																		 *
  *					Process an attribute definition						 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  	get_attribute (VAR i: Integer; VAR  fg, bg: Integer) ;

VAR
	n:	Integer ;

BEGIN
		fg := -1 ;
		bg := -1 ;
		Inc (i) ;	

		IF  i > Length(input_line)  THEN
			error_message ('No attribute value', i) 
		ELSE
		  BEGIN
			n := read_number (i) ;
			IF  n <> -1  THEN
			  BEGIN
			    IF  n > 15  THEN
				 	error_message ('Foreground palette number > 15', i) 
			  	ELSE
					fg := n ;
				IF  i > Length(input_line)  THEN  
				  BEGIN
					error_message ('Missing )', i) ;
					fg := -1 ;
				  END ;	
			  END 
			ELSE
			  IF  (input_line[i] <> ',')  AND  (input_line[i] <> ')')  THEN
			       error_message ('Invalid or missing foreground palette number', i) ;

			IF  (i <= Length(input_line))  AND  (input_line[i] = ',')  THEN
			  BEGIN
			  	Inc (i) ;
			    n := read_number (i) ;
			    IF  n <> -1  THEN
			  	  BEGIN
			    	IF  n > 15  THEN
				 		error_message ('Background palette number > 15', i) 
			  		ELSE
						bg := n ;
			  	  END
				ELSE
				   	error_message ('Invalid or missing background palette number', i)	;
			  END ;

			IF  (i > Length(input_line))  OR  (input_line[i] <> ')')  THEN  
			  BEGIN
			  	error_message ('Missing )', i) ;
				fg := -1 ;
				bg := -1 ;
			  END 
			ELSE
			  Inc (i) ;
		  END ;
END ;


{ ************************************************************************
  *																		 *
  *						 Process a section identifier					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  	get_identifier (VAR i: Integer; VAR  id: Integer) ;

VAR
	n:		  Integer ;
	id_start: Integer ;
BEGIN

		id_start := i - 2 ;
		IF  i > Length(input_line)  THEN
			error_message ('Missing section identifier', i) 
		ELSE
		  BEGIN
			id := read_number (i) ;
			IF  id <> -1  THEN
			  BEGIN
				IF  i > Length(input_line)  THEN  
				  BEGIN
					error_message ('Missing quote', i) ;
					id := -1 ;
				  END 	
				ELSE
				  BEGIN
					IF  input_line[i] <> ''''  THEN
						error_message ('Missing quote', i) 
					ELSE
						Inc (i) ;
				  END ;
			  END 
			ELSE
				error_message ('Missing or invalid section identifier', i) 

		  END ;

		IF  pass = 2  THEN
			Delete (input_line, id_start, i - id_start) ;  	

END ;



{ ************************************************************************
  *																		 *
  *				       normalize a section name	  						 *
  *																		 *
  ************************************************************************ }
			
FUNCTION	normalized_string (VAR i: Integer;
							   delimiter: Char;
							   within_definition: Boolean) :  String ;

VAR
	s:			String ;
	ws:			String ;
	j:			Integer ;
	t:			Integer ;
	w:			Integer ;
	save:			Integer ;
	string_end: Boolean ;

BEGIN

	string_end := False ;
	j := 1 ;

	WHILE  (i <= Length(input_line))  AND  (NOT string_end)  DO
		IF  input_line[i] = ESC  THEN
		  BEGIN
			Inc (i) ;
			IF  input_line [i]  = ESC  THEN
					BEGIN
					  s[j] := ESC ;
					  Inc (j) ;
					  Inc (i) ;
				    END 
			ELSE  IF  input_line [i]  = '('  THEN
					get_attribute (i, t, w)
			ELSE  IF  input_line [i]  = '='  THEN
					BEGIN
					  IF  within_definition  THEN
						error_message ('\= not allowed in definition', i) ;
					  Inc (i) ;
					  IF  pass = 2  THEN
						BEGIN
						  save := i - 2 ;
						  s := normalized_string (i, delimiter, False) ;
						  delete (input_line, save, i - save) ;
						  i := save ;
						  normalized_string := s ;
						  Exit ;
						END ;
					END
			ELSE  IF  input_line [i]  = '{'  THEN
					BEGIN
						IF  get_characters (i, ws) <> ''  THEN
						  BEGIN
							s[0] := Chr(j-1) ;
				  			s := s + ws ;
							Inc (j, Length(ws)) ;
						  END ;						  	
					END
			ELSE  IF  input_line [i]  = ''''  THEN
					BEGIN
					  IF  NOT within_definition  THEN
						error_message ('Identifer only allowed in section definition', i) ;
					  Inc (i) ;
					  get_identifier (i, identifier) ;
					END
			ELSE  IF  input_line [i]  = '>'  THEN
					BEGIN
					  s[j] := '>' ;
					  Inc (j) ;
					  Inc (i) ;
				    END 
			ELSE  IF  input_line [i]  = ']'  THEN
					BEGIN
					  s[j] := ']' ;
					  Inc (j) ;
					  Inc (i) ;
				    END 
			ELSE
			  BEGIN
				error_message ('Invalid escape sequence', i) ;
				Inc (i) ;
			  END ;
		  END
		ELSE
		  BEGIN
			IF  input_line [i] = delimiter  THEN
				string_end := True 
			ELSE
			  BEGIN
				s[j] := input_line [i] ;
				IF  s[j] = Chr(9)  THEN
					s[j] := ' ' ;
				IF  (s[j] <> ' ')  OR  ( (j > 1)  AND  (s[j-1] <> ' '))  THEN  
		  			Inc (j) ;
				Inc (i) ;
			  END ;
		  END ;

	IF  (delimiter <> Chr(0))  AND  (NOT  string_end)  THEN
		error_message ('Missing closing bracket', i) ;

	Dec (j) ;
	IF  (j > 0)  AND  (s[j] = ' ')  THEN
		Dec (j) ;

	s[0] := Chr(j) ;
	FOR  t := 1  TO  j  DO
		s[t] := Upcase(s[t]) ;
	normalized_string := s ;

END ;


{ ************************************************************************
  *																		 *
  *				    Write a section to the output file					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	write_section ;
BEGIN
		IF  section_lines > max_section_lines  THEN
			max_section_lines := section_lines ;
		section_table [section_index].section_address := FilePos (output_file) ;
		section_table [section_index].section_length := buffer_length ;
		BlockWrite (output_file, section_buffer^, buffer_length) ;
		buffer_length := 0 ;
END ;

{ ************************************************************************
  *																		 *
  *						Process a section definition					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	process_section_definition (i: Integer) ;

VAR
	s:	String ;
	p:	Line_ptr ;
	n:	Integer ;
	t:	Integer ;

BEGIN

	IF  pass = 2  THEN
	  BEGIN
		IF  section_index <> -1  THEN
			write_section ;
		section_lines := 1 ;
	  END ;

	WHILE  (i <= Length(input_line))  AND  (input_line[i] = ' ')  DO
  		Inc (i) ;
	IF  i > Length(input_line)  THEN
		error_message ('Missing section name', i) 
	ELSE
	  BEGIN
		identifier := -1 ;
		s := normalized_string (i, Chr(0), True) ;
		n :=  Length(s) ;
		IF  n <> 0  THEN
		  BEGIN
			IF  n > MAX_LINE_LENGTH  THEN
				error_message ('Section name too long', i) 
			ELSE
			  BEGIN
				t := 0 ;
				WHILE  (t <= section_count) AND
				       (s <> section_names[t]^)  DO
				  	Inc (t) ;
			  END ;
		  END 
		ELSE
			error_message ('No section name ', i) ;
	  END ;

	IF  pass = 1  THEN
	  BEGIN
		IF  t <= section_count  THEN
			error_message ('Duplicate section name ', 0) 
		ELSE
		  BEGIN
			GetMem (p, n+1) ;
			Move (s, p^, n+1) ;
			Inc (section_index) ;
			Inc (section_count) ;
			section_names [section_index] := p ;
			section_table [section_index].section_address := 0 ;
			section_table [section_index].section_identifier := identifier ;
			IF  area_table [area_index].area_first = -1  THEN
				    area_table [area_index].area_first := section_index ;
		  END ;
	END ;

	IF  pass = 2  THEN
	  BEGIN
		IF  t > section_count  THEN
			error_message ('Undefined section name: '''+ s, i) 
		ELSE
		  BEGIN
			section_index := t ;
			buffer_length := Length (input_line) + 1 ;
			Move (input_line, section_buffer^, buffer_length) ;
		  END ;
	END ;

END	;


{ ************************************************************************
  *																		 *
  *					  Process a definition								 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	process_definition (i: Integer) ;
BEGIN
	IF    	  Pos ('area', input_line)  = i  THEN
				process_area_definition	(i + Length('area'))
	ELSE IF  Pos ('section', input_line)  = i  THEN
				process_section_definition (i + Length('section'))
	ELSE IF  Pos ('end', input_line)  = i  THEN
			 	end_of_input := True ;
END ;


{ ************************************************************************
  *																		 *
  *					   Process the section text							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	process_text ;

VAR
	i:				Integer ;
	delim:			Char ;
	insert_point:	Integer ;
	work1, work2:	Integer ;
	work_string:	String ;

PROCEDURE	insert_reference (insert_point: Integer; s: String) ;
VAR
	t:	Integer ;
	ss:	String ;
BEGIN
	IF  s = 'DUMMY'  THEN
		Insert ('32767 ', input_line, insert_point) 
	ELSE
	  BEGIN
		t := 0 ;
		WHILE  (t <= section_count) AND	(s <> section_names[t]^)  DO
  	  	Inc (t) ;
		IF  t > section_count  THEN
	   	error_message ('''' + s + ''' undefined', 0) ;
		Str (t, ss) ;
		ss := ss + ' ' ;
		Insert (ss, input_line, insert_point) ;
	END	;
END	;

BEGIN

	i := 1 ;
	
	WHILE  i <= Length(input_line)  DO
	  BEGIN
		WHILE  (i <= Length(input_line))  AND  (input_line[i] <> ESC)  DO
	  	  Inc (i) ;
		IF  i <= Length(input_line)  THEN
		  BEGIN
		  	Inc (i) ;
			IF  input_line [i]  = '('  THEN
				get_attribute (i, work1, work2)
			ELSE IF  input_line [i]  = '{'  THEN
				work_string := get_characters (i, work_string) 
			ELSE IF  (input_line [i]  = '<') OR (input_line [i]  = '[') THEN
				   BEGIN
					 IF  input_line [i]  = '<'  THEN
					 	delim := '>'
					 ELSE
					 	delim := ']' ;
					 Inc (i) ;
					 insert_point := i ;
					 work_string := normalized_string (i, delim, False) ;
					 IF  pass = 2   THEN
					   BEGIN
						 insert_reference (insert_point, work_string) ;
						 Inc (i) ;
					   END ;
				   END
			ELSE
			   ;	
		  END ;
	  END ;


	IF  pass = 2  THEN
	  BEGIN
		work1 := Length(input_line) ;
		IF  buffer_length + work1 > MAX_SECTION_LENGTH  THEN
			error_message ('Maximum section length exceeded', 0) 
		ELSE
		  BEGIN
			Move (input_line, section_buffer^[buffer_length], work1 + 1) ;
	    	Inc (buffer_length, work1 + 1) ;
			Inc (section_lines) ;
		  END ;
	  END ;

END ;



{ ************************************************************************
  *																		 *
  *						Process a line of text							 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	process_a_line ;

VAR
	i:	Integer ;

BEGIN

	i := 1 ;
	WHILE  (i <=  Length(input_line))  AND  (input_line[i] = ' ')  DO
		Inc (i) ;

	IF  i <  Length(input_line)  THEN
	  BEGIN
		IF  input_line[i] = ESC  THEN
		  BEGIN
			IF  input_line[i+1] IN  ['a'..'z','A'..'Z']  THEN
				process_definition (i+1)
			ELSE
				IF  input_line[i+1] <> '*' THEN
					process_text ;
		  END
		ELSE
			process_text ;
	  END
	ELSE
		process_text ;
END ;


{ ************************************************************************
  *																		 *
  *						   Initialize the program						 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE	initialize ;
VAR
	i:	Integer ;
BEGIN
	section_count := -1 ;
	max_section_lines := -1 ;
	section_lines := 1 ;
	area_index := 0 ;
	FOR  i := 1  TO  MAX_AREAS  DO
	  BEGIN
		area_table [i].area_first := -1 ;
		area_table [i].area_last := 0 ;
	  END ;		
END ;

{$IFDEF DEBUG}

PROCEDURE  list_sections ;
VAR
	i, n:	Integer ;
	p:		^ String ;
	x:		Word ;
BEGIN
{$i-}							   
    Reset (output_file, 1) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
    	fatal_error ('Cannot reopen ' + output_file_name) ;

	writeln ('SECTIONS') ;
	writeln ;
	FOR  i := 0  TO  section_count  DO
	  BEGIN
		writeln (i, ' ', section_names[i]^,
				 '   ', section_table[i].section_address,
				 '   ', section_table[i].section_length ) ;
		writeln ;

{$i-}							   
		Seek (output_file, section_table[i].section_address) ;
{$i+}							   
	    n :=  IOResult ;				   
	    IF  n <> 0  THEN				   
    		writeln ('Seek ' , n) ;

{$i-}							   
		BlockRead (output_file, section_buffer^,
								section_table[i].section_length, x) ;
{$i+}							   
    	writeln ('Readx ' , x) ;

	    n :=  IOResult ;				   
	    IF  n <> 0  THEN				   
    		writeln ('Read ' , n) ;

		n := 0  ;
		WHILE  n < section_table[i].section_length  DO
		  BEGIN
		  	p := Addr(section_buffer^[n]) ;
			writeln (p^) ;
			n := n + Length(p^) + 1;
		  END ;

	  END ;

END ;

PROCEDURE  list_areas ;
VAR
	i:	Integer ;
BEGIN
	writeln ;
	writeln ('AREAS (', line_number, ' lines)') ;
	writeln ;
	FOR  i := 1  TO  area_count  DO
		writeln (i, ' ', area_table[i].area_first, ' ',area_table[i].area_last) ;
END ;

{$ENDIF}


{ ************************************************************************
  *																		 *
  *			    First pass: check input and collect section names		 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  pass_one	;
BEGIN
	end_of_input := False ;
	pass := 1 ;
	section_index := -1 ;
	line_number := 0 ;

	WHILE  (NOT  end_of_input)  AND  read_a_line  DO
		process_a_line ;

	IF  section_count < 0  THEN
		error_message ('No sections', 0) 
	ELSE
		IF  area_table [area_index].area_first <> -1  THEN
			area_table [area_index].area_last := section_index ;
END ;

{ ************************************************************************
  *																		 *
  *				 Second pass:  generate the output file					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  pass_two	;
BEGIN
{$i-}							   
    Reset (input_file) ;				
{$i+}							   
    IF  IOResult <> 0  THEN				   
    	fatal_error ('Cannot open ' + input_file_name) ;

	end_of_input := False ;
	pass := 2 ;
	line_number := 0 ;
	section_index := -1 ;

	GetMem (section_buffer, MAX_SECTION_LENGTH) ;
	buffer_length := 0 ;

	WHILE  (NOT  end_of_input)  AND  read_a_line  DO
		process_a_line ;

	IF  section_index <> -1  THEN  
		write_section ;
END ;


{ ************************************************************************
  *																		 *
  *					 Write the section and area tables 					 *
  *																		 *
  ************************************************************************ }
			
PROCEDURE  write_tables	;
	
BEGIN

	des.start_address 	  := FilePos (output_file) ;
	des.area_count		  := area_count ;		
	des.section_count 	  := section_count ;		
	des.max_section_lines := max_section_lines ;		

{$i-}							   
	BlockWrite (output_file, area_table, Sizeof(area) * area_count) ;
{$i+}							   
	    IF  IOResult <> 0  THEN				   
			fatal_error ('Error writing output file') ;
{$i-}							   
	BlockWrite (output_file, section_table, Sizeof(section) * (section_count+1)) ;
{$i+}							   
	    IF  IOResult <> 0  THEN				   
			fatal_error ('Error writing output file') ;

{$i-}							   
	BlockWrite (output_file, des, Sizeof(des)) ;
{$i+}							   
	    IF  IOResult <> 0  THEN				   
			fatal_error ('Error writing output file') ;

	Close (output_file) ;

{$IFDEF DEBUG}
	Writeln ;
	Writeln ('Start address: ', des.start_address, '   Areas: ', area_count,
			 '   Sections: ', section_count, '   Lines: ', max_section_lines ) ;
{$ENDIF}
END ;

{ ************************************************************************
  *																		 *
  *						  The main program								 *
  *																		 *
  ************************************************************************ }
			
BEGIN

	initialize ;
	check_parameters ;

	errors := False ;
	pass_one ;
{$IFDEF DEBUG}
		list_areas ; 
{$ENDIF}

	IF  NOT  errors THEN
	  BEGIN
		pass_two ;
{$IFDEF DEBUG}
		list_areas ; 
{$ENDIF}
	  END ;
	
	IF  NOT  errors THEN
	  BEGIN
		write_tables ;
{$IFDEF DEBUG}
		list_sections ; 
{$ENDIF}
	  END ;
END.





















