(*****************************************************************************

  Tables
    Version 1.11

  Purpose:
    This unit hold several routines designed to solve tables for optimal
    solutions.

  How it works:
    These routines do not search through all the possible solutions to find
      the optimal one, since this would take an enormous amount of time.
    Instead the problem is solved using a specially modified form of the
      Hungarian algorithm.

  Features:
    Use of large arrays up to the size of Table_Limit.
    Segmented algorithm to simplify understanding.
    Selected functions and procedures to simplify coding.
    Simple error checking.

  Limitations:
    Values in the table must be integers.

  Versions:
    1.10 - Altered routines to make them operate quicker.
    1.11 - Added the Write_Table procedure.

  Copyright 1987, 1992, all rights reserved
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Tables;

  Interface

    Const
      List_Limit = 50;
      Table_Limit = 50;

    Type
      Value_Type = Integer;
      Table_Array_Type = Array [ 1 .. Table_Limit, 1 .. Table_Limit ] of Value_Type;
      Table_Type = Record
                     Size: Byte;
                     Data: Table_Array_Type;
                   End;
      Solution_Node_Type = Record
                             Job,
                             Cost: Integer;
                           End;
      Solution_Array_Type = Array [ 1 .. Table_Limit ] of Solution_Node_Type;
      Solution_Type = Record
                        Size: Byte;
                        Cost: LongInt;
                        Data: Solution_Array_Type;
                      End;
      Solution_List_Type = Record
                             Cost: LongInt;
                             Amount: Byte;
                             Data: Array[ 1 .. List_Limit ] of Solution_Type;
                           End;

(***********************************************************

  Procedure: Minimize solve single.

    This procedure is a method of solving the general
    m-resource and m-activity assignment problem for total
    effectiveness.  The problem is minimized and all the
    values on the table should be positive.

***********************************************************)

    Procedure Minimize_Solve_Single( Var Table: Table_Type; Var Solution: Solution_Type );

(***********************************************************

  Procedure: Minimize solve multiple.

    This procedure allows for finding multiple solutions
    instead of just a single solution.  The solutions are
    stored in list which can hold only a finite amount of
    solutions.

***********************************************************)

    Procedure Minimize_Solve_Multiple( Var Table: Table_Type; Var List: Solution_List_Type );

(***********************************************************

  Procedure: Maximize solve single.

    This procedure is a method of solving the general
    m-resource and m-activity assignment problem for total
    effectiveness.  The problem is maximized and all the
    values on the table should be positive.

***********************************************************)

    Procedure Maximize_Solve_Single( Var Table: Table_Type; Var Solution: Solution_Type );

(***********************************************************

  Procedure: Maximize solve multiple.

    This procedure allows for finding multiple solutions
    instead of just a single solution.  The solutions are
    stored in list which can hold only a finite amount of
    solutions.

***********************************************************)

    Procedure Maximize_Solve_Multiple( Var Table: Table_Type; Var List: Solution_List_Type );

(***********************************************************

  Procedure: Print out the solution.

    This procedure is designed to print out the solution
    generated by optimizing the table.

***********************************************************)

    Procedure Print_Solution( Var OutFile: Text; Var Solution: Solution_Type );

(***********************************************************

  Procedure: Print out the list of solutions.

    This procedure is designed to print out all the
    solutions that were found.  It is used to print out the
    final data.  It is also very useful for debugging the
    programs.

***********************************************************)

    Procedure Print_Solutions( Var OutFile: Text; Var List: Solution_List_Type );

(***********************************************************

  Procedure: Read in the table.

    This procedure reads in the table from the standard text
    file of the user supplied name.  The first element of
    the file must be the size of the table followed by all
    the values of the table in order of location.
    The table must be a table of integers.

***********************************************************)

    Procedure Read_Table( Var InFile: Text; Var Table: Table_Type );

(***********************************************************

  Procedure: Write out the table.

    This procedure writes out the table to the text file in
    standard formation.  The first element of the file will
    be the size of the table followed by all the values of
    the table in order of location.  The table will be a
    table of integers.

***********************************************************)

    Procedure Write_Table( Var OutFile: Text; Var Table: Table_Type );

(***********************************************************

  Procedure: Print out the table.

    This procedure is left in here so that the programmer
    can use it to print out the whole table when needed.  It
    is intended mostly for debugging programs.

***********************************************************)

    Procedure Print_Table( Var OutFile: Text; Var Table: Table_Type );

{----------------------------------------------------------------------------}

  Implementation

    Const
      Maximum: Value_Type = 32767;
      Starting_Value = 2147483647;

    Type
      Invalid_Type = Array[ 1 .. Table_Limit ] of Word;
      Save_Type = Record
                    Row,
                    Column: Byte;
                  End;
      Stack_Pointer_Type = ^Stack_Element_Type;
      Stack_Element_Type = Record
                             Value: Byte;
                             Next: Stack_Pointer_Type;
                           End;
      Alternative_Type = Record
                           Current_Row: Byte;
                           Current_Count: Word;
                           Stack: Stack_Pointer_Type;
                           Tries: Array [ 1 .. Table_Limit ] of Byte;
                         End;

    Var
      Alternative: Alternative_Type;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Minimize the rows.
    This procedure finds the minimal column
    element in each of the rows and adjusts that
    row to minimize that element.

*************************************************)

    Procedure Minimize_Rows( Var Table: Table_Type );
      Var
        Row,
        Column: Byte;
        Row_Minimum: Value_Type;
      Begin
        for Row := 1 to Table.Size do
          Begin
            Row_Minimum := Table.Data[ Row, 1 ];
            For Column := 2 to Table.Size do
              If ( Table.Data[ Row, Column ] < Row_Minimum )
                then
                  Row_Minimum := Table.Data[ Row, Column ];
            For Column := 1 to Table.Size do
              Table.Data[ Row, Column ] := Table.Data[ Row, Column ] - Row_Minimum;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Minimize the columns.
    This procedure finds the minimal row element
    in each of the columns and adjusts that column
    to minimize that element.

*************************************************)

    Procedure Minimize_Columns( Var Table: Table_Type );
      Var
        Row,
        Column: Byte;
        Column_Minimum: Value_Type;
      Begin
        For Column := 1 to Table.Size do
          Begin
            Column_Minimum := Table.Data[ 1, Column ];
            For Row := 2 to Table.Size do
              if ( Table.Data[ Row, Column ] < Column_Minimum )
                then
                  Column_Minimum := Table.Data[ Row, Column ];
            For Row := 1 to Table.Size do
              Table.Data[ Row, Column ] := Table.Data[ Row, Column ] - Column_Minimum;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize marks.
    This procedure initializes the marked columns
    and rows to zero.

*************************************************)

    Procedure Initialize_Marks( Var Invalid_Row, Invalid_Column: Invalid_Type; Size: Byte );
      Var
        Row: Byte;
      Begin
        For Row := 1 to Size do
          Invalid_Row[ Row ] := 0;
        Invalid_Column := Invalid_Row;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Mark Rows
    This procedure finds rows with exactly one
    unmarked zero in them and crosses out the
    entire column so that additional assignments
    will exclude them.

*************************************************)

    Procedure Mark_Rows( Var Table: Table_Type; Var Invalid_Row, Invalid_Column: Invalid_Type; Var Number_of_Zeros,
                         No_Made: Word; Var Loop, Save_Row: Byte );
      Var
        Row,
        Column,
        Save_Column,
        Number_Zeros_in_Row: Byte;
      Begin
        For Row := 1 to Table.Size do
          Begin
            Number_Zeros_in_Row := 0;
            If ( Invalid_Row[ Row ] = 0 )
              then
                Begin
                  For Column := 1 to Table.Size do
                    If ( Invalid_Column[ Column ] = 0 )
                      then
                        If ( Table.Data[ Row, Column ] = 0 )
                          then
                            Begin
                              Inc( Number_Zeros_in_Row );
                              Inc( Number_Of_Zeros );
                              Save_Row := Row;
                              Save_Column := Column;
                            End;
                  If ( Number_Zeros_in_Row = 1 )
                    then
                      Begin
                        Invalid_Row[ Row ] := Save_Column;
                        Invalid_Column[ Save_Column ] := 1;
                        Inc( No_Made );
                        Inc( Loop );
                      End;
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Mark columns.
    This procedure searches the columns for a
    single unmarked zero and marks them.  It
    crosses out the other zeros in the same row
    so that they won't be assigned.

*************************************************)

    Procedure Mark_Columns( Var Table: Table_Type; Var Invalid_Row, Invalid_Column: Invalid_Type; Var Number_of_Zeros,
                            No_Made: Word; Var Loop: Byte );
      Var
        Row,
        Column,
        Column_Pointer,
        Number_Zeros_in_Column: Byte;
      Begin
        For Row := 1 to Table.Size do
          Begin
            Number_Zeros_in_Column := 0;
            If ( Invalid_Column[ Row ] = 0 )
              then
                Begin
                  For Column := 1 to Table.Size do
                    If ( Invalid_Row[ Column ] = 0 )
                      then
                        If ( Table.Data[ Column, Row ] = 0 )
                          then
                            Begin
                              Inc( Number_Zeros_in_Column );
                              Inc( Number_of_Zeros );
                              Column_Pointer := Column;
                            End;
                  If ( Number_Zeros_in_Column = 1 )
                    then
                      Begin
                        Invalid_Column[ Row ] := 1;
                        Invalid_Row[ Column_Pointer ] := Row;
                        Inc( No_Made );
                        Inc( Loop );
                      End;
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Do assignments.
    This procedure does the assignments until
    there are no more valid assignments that it
    can do.

*************************************************)

    Procedure Do_Assignments( Var Table: Table_Type; Var Invalid_Row, Invalid_Column: Invalid_Type; Var Number_of_Zeros,
                              No_Made: Word; Var Save_Row: Byte; Var Finished: Boolean );
      Var
        Loop: Byte;
      Begin
        Repeat
          Loop := 0;
          Number_Of_Zeros := 0;
          Mark_Rows( Table, Invalid_Row, Invalid_Column, Number_Of_Zeros, No_Made, Loop, Save_Row );
          Mark_Columns( Table, Invalid_Row, Invalid_Column, Number_Of_Zeros, No_Made, Loop );
          If ( No_Made = Table.Size )
            then
              Finished := True;
        Until ( ( Loop = 0 ) or Finished );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function:  Find first column.
    This function finds the first column that is
    not invalid and returns that column number.

*************************************************)

    Function Find_First_Column( Size: Byte; Invalid_Column: Invalid_Type ): Byte;
      Var
        Column: Byte;
      Begin
        Column := 1;
        While ( Column <= Size ) and ( Invalid_Column[ Column ] <> 0 ) do
          Inc( Column );
        If ( Column > Size )
          then
            Begin
              WriteLn( 'Error in Find_First_Column: All columns are invalid.' );
              Halt;
            End;
        Find_First_Column := Column;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find next column.
    This function is intended to be called with
    the value that was given in Find_First_Column
    it will return the next column number that is
    valid, or will return zero if there isn't any.

*************************************************)

    Function Find_Next_Column( Last_Column, Size: Byte; Invalid_Column: Invalid_Type ): Byte;
      Var
        Column: Byte;
      Begin
        If ( Invalid_Column[ Last_Column ] <> 0 )
          then
            Begin
              WriteLn( 'Error in Find_Next_Column: Current column is invalid.' );
              Halt;
            End;
        Column := Succ( Last_Column );
        While ( Column <= Size ) and ( Invalid_Column[ Column ] <> 0 ) do
          Inc( Column );
        If ( Column <= Size )
          then
            Find_Next_Column := Column
          else
            Find_Next_Column := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: No more columns.
    This function acts like Find_Next_Column,
    except that instead of returning the next
    column, it returns true if there are no more
    columns.

*************************************************)

    Function No_More_Columns( Last_Column, Size: Byte; Invalid_Column: Invalid_Type ): Boolean;
      Var
        Count: Byte;
      Begin
        Count := Succ( Last_Column );
        While ( Count <= Size ) and ( Invalid_Column[ Count ] <> 0 ) do
          Inc( Count );
        No_More_Columns := ( Count > Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Push.
    This procedure pushes the given value onto the
    stack.

*************************************************)

    Procedure Push( Var Stack: Stack_Pointer_Type; Data: Byte );
      Var
        Temporary_Pointer: Stack_Pointer_Type;
      Begin
        New( Temporary_Pointer );
        Temporary_Pointer^.Value := Data;
        Temporary_Pointer^.Next := Stack;
        Stack := Temporary_Pointer;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Pop.
    This procedure pops the given value off of the
    stack.

*************************************************)

    Procedure Pop( Var Stack: Stack_Pointer_Type; Var Data: Byte );
      Var
        Temporary_Pointer: Stack_Pointer_Type;
      Begin
        If ( Stack = Nil )
          then
            Begin
              WriteLn( 'Error in Pop: Stack empty.' );
              Halt;
            End;
        Temporary_Pointer := Stack;
        Data := Stack^.Value;
        Stack := Stack^.Next;
        Dispose( Temporary_Pointer );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Clear down.
    This procedure clears only the rows that are
    down from the current row.

*************************************************)

    Procedure Clear_Down( Current_Row: Byte );
      Var
        Row: Byte;
      Begin
        For Row := Pred( Current_Row ) downto 1 do
          Alternative.Tries[ Row ] := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Pick arbitrary row.
    This procedure is called when there is more
    than one possible assignment in the particular
    row.

*************************************************)

    Procedure Pick_Arbitrary_Row( Var Table: Table_Type; Var Invalid_Row, Invalid_Column: Invalid_Type; Row: Byte );
      Var
        Column: Byte;
      Begin
        If ( Alternative.Tries[ Row ] = 0 )
          then
            Begin
              Column := Find_First_Column( Table.Size, Invalid_Column );
              Alternative.Tries[ Row ] := Column;
              Push( Alternative.Stack, Alternative.Current_Row );
              Alternative.Current_Row := Row;
              Inc( Alternative.Current_Count );
            End
          else
            If ( Row = Alternative.Current_Row )
              then
                Begin
                  Column := Find_Next_Column( Alternative.Tries[ Alternative.Current_Row ], Table.Size, Invalid_Column );
                  Alternative.Tries[ Row ] := Column;
                  Clear_Down( Alternative.Current_Row );
                  If No_More_Columns( Column, Table.Size, Invalid_Column )
                    then
                      Begin
                        Pop( Alternative.Stack, Alternative.Current_Row );
                        Dec( Alternative.Current_Count );
                      End;
                End
              else
                Column := Alternative.Tries[ Row ];
        Invalid_Row[ Row ] := Column;
        Invalid_Column[ Column ] := 1;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Attempt assignment.
    This procedure attempts to complete the
    assignment

*************************************************)

    Procedure Attempt_Assignment( Var Table: Table_Type; Var Invalid_Row: Invalid_Type; Var Finished: Boolean );
      Var
        Done: Boolean;
        Save_Row: Byte;
        No_Made,
        Number_Of_Zeros: Word;
        Invalid_Column: Invalid_Type;
      Begin
        Initialize_Marks( Invalid_Row, Invalid_Column, Table.Size );
        No_Made := 0;
        Repeat
          Do_Assignments( Table, Invalid_Row, Invalid_Column, Number_of_Zeros, No_Made, Save_Row, Finished );
          If ( not Finished )
            then
              If ( Number_Of_Zeros <> 0 )
                then
                  Begin
                    Pick_Arbitrary_Row( Table, Invalid_Row, Invalid_Column, Save_Row );
                    Inc( No_Made );
                    Done := False;
                  End
                else
                  Done := True;
        Until ( Done or Finished );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Check rows.
    This procedure checks all the rows for which
    assignments haven't been made.

*************************************************)

    Procedure Check_Rows( Limit: Byte; Var Row_Check, Column_Check, Invalid_Row: Invalid_Type );
      Var
        Row: Byte;
      Begin
        For Row := 1 to Limit do
          Begin
            Row_Check[ Row ] := 0;
            Column_Check[ Row ] := 0;
            If ( Invalid_Row[ Row ] = 0 )
              then
                Row_Check[ Row ] := Row;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Eliminate rows.
    This procedure draws lines through all the
    unchecked rows and checked columns.

*************************************************)

    Procedure Eliminate_Rows( Size: Byte; Var Row_Check, Column_Check, Invalid_Row: Invalid_Type; Var Done: Boolean );
      Var
        Row,
        Column: Byte;
      Begin
        For Row := 1 to Size do
          If ( Column_Check[ Row ] > 0 )
            then
              For Column := 1 to Size do
                If ( Column_Check[ Row ] = Invalid_Row[ Column ] )
                  then
                    Begin
                      Row_Check[ Column ] := Column;
                      Done := False;
                    End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Check columns and eliminate rows.
    This procedure checks columns not already
    checked which have a zero in the checked rows,
    then checks the remaining rows that have
    assignments in checked columns until there are
    no more left.  Then it draws lines through all
    the unchecked rows and checked columns.

*************************************************)

    Procedure Check_Columns( Var Table: Table_Type; Var Row_Check, Column_Check, Invalid_Row: Invalid_Type );
      Var
        Row,
        Column: Byte;
        Done,
        No_Check: Boolean;
      Begin
        Repeat
          Done := True;
          No_Check := True;
          For Row := 1 to Table.Size do
            If ( Row_Check[ Row ] <> 0 )
              then
                For Column := 1 to Table.Size do
                  If ( Column_Check[ Column ] = 0 )
                    then
                      If ( Table.Data[ Row, Column ] = 0 )
                        then
                          Begin
                            Column_Check[ Column ] := Column;
                            No_Check := False;
                          End;
          If ( not No_Check )
            then
              Eliminate_Rows( Table.Size, Row_Check, Column_Check, Invalid_Row, Done );
        Until Done;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Find minimum.
    This procedure finds the minimum value of the
    uncrossed elements.

*************************************************)

    Procedure Find_Minimum( Var Table: Table_Type; Var Row_Check, Column_Check: Invalid_Type; Var Minimum: Value_Type );
      Var
        Row,
        Column: Byte;
      Begin
        Minimum := Maximum;
        For Row := 1 to Table.Size do
          If ( Row_Check[ Row ] <> 0 )
            then
              For Column := 1 to Table.Size do
                If ( Column_Check[ Column ] = 0 )
                  then
                    If ( Table.Data[ Row, Column ] < Minimum )
                      then
                        Minimum := Table.Data[ Row, Column ];
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Adjust remaining elements.
    This procedure adjusts the remaining elements
    so that some of them can be eliminated from
    the solution.

*************************************************)

    Procedure Adjust_Elements( Var Table: Table_Type; Var Row_Check, Column_Check: Invalid_Type; Minimum: Value_Type );
      Var
        Row,
        Column: Byte;
      Begin
        For Row := 1 to Table.Size do
          For Column := 1 to Table.Size do
            If ( Row_Check[ Row ] >= 0 )
              then
                If ( Row_Check[ Row ] <> 0 )
                  then
                    Begin
                      If ( Column_Check[ Column ] = 0 )
                        then
                          Table.Data[ Row, Column ] := Table.Data[ Row, Column ] - Minimum;
                    End
                  else
                    If ( Column_Check[ Column ] > 0 )
                      then
                        Table.Data[ Row, Column ] := Table.Data[ Row, Column ] + Minimum;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reassign table.
    This procedure adjusts the remaining elements
    in the table so that they can be eliminated
    from the final solution.

*************************************************)

    Procedure Reassign_Table( Var Table: Table_Type; Var Invalid_Row: Invalid_Type );
      Var
        Minimum: Value_Type;
        Row_Check,
        Column_Check: Invalid_Type;
      Begin
        Check_Rows( Table.Size, Row_Check, Column_Check, Invalid_Row );
        Check_Columns( Table, Row_Check, Column_Check, Invalid_Row );
        Find_Minimum( Table, Row_Check, Column_Check, Minimum );
        Adjust_Elements( Table, Row_Check, Column_Check, Minimum );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Negate values.
    This procedure makes all the values in the
    table negative so that the maximal solution
    can be found.

*************************************************)

    Procedure Negate_Values( Var Table: Table_Type );
      Var
        Row,
        Column: Byte;
      Begin
        For Row := 1 to Table.Size do
          For Column := 1 to Table.Size do
            Table.Data[ Row, Column ] := - Table.Data[ Row, Column ];
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Minimize Hungarian algorithm
    This procedure is a method of solving the
    general m-resource, m-activity assignment
    problem for total effectiveness by least
    cost.

*************************************************)

    Procedure Minimize_Hungarian_Algorithm( Table: Table_Type; Var Invalid_Row: Invalid_Type );
      Var
        Done,
        Finished: Boolean;
      Begin
        Minimize_Rows( Table );
        Minimize_Columns( Table );
        Repeat
          Finished := False;
          Attempt_Assignment( Table, Invalid_Row, Finished );
          If ( not Finished )
            then
              Reassign_Table( Table, Invalid_Row );
        Until Finished;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Maximize Hungarian algorithm
    This procedure is a method of solving the
    general m-resource, m-activity assignment
    problem for total effectiveness by maximum
    cost.

*************************************************)

    Procedure Maximize_Hungarian_Algorithm( Table: Table_Type; Var Invalid_Row: Invalid_Type );
      Var
        Done,
        Finished: Boolean;
      Begin
        Negate_Values( Table );
        Minimize_Rows( Table );
        Minimize_Columns( Table );
        Repeat
          Finished := False;
          Attempt_Assignment( Table, Invalid_Row, Finished );
          If ( not Finished )
            then
              Reassign_Table( Table, Invalid_Row );
        Until Finished;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Calculate the cost.
    This procedure translates the values from the
    Invalid_Row table to the Solution table while
    calculating the cost of the solution with the
    values from the table.

*************************************************)

    Procedure Calculate_Cost( Var Table: Table_Type; Invalid_Row: Invalid_Type; Var Solution: Solution_Type );
      Var
        Row,
        Column: Byte;
      Begin
        Solution.Cost := 0;
        Solution.Size := Table.Size;
        For Row := 1 to Solution.Size do
          Begin
            Column := Invalid_Row[ Row ];
            Solution.Data[ Row ].Job := Column;
            Solution.Data[ Row ].Cost := Table.Data[ Row, Column ];
            Solution.Cost := Solution.Cost + Solution.Data[ Row ].Cost;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize tries.
    This procedure initializes everything that is
    necessary to generate multiple solutions to
    the table.

*************************************************)

    Procedure Initialize_Tries;
      Var
        Row: Byte;
      Begin
        Alternative.Current_Row := 0;
        Alternative.Current_Count := 0;
        Alternative.Stack := Nil;
        For Row := 1 to Table_Limit do
          Alternative.Tries[ Row ] := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Deallocate tries.
    This procedure eliminates the tries list after
    the solutions are found.

*************************************************)

    Procedure Deallocate_Tries;
      Var
        Pointer: Stack_Pointer_Type;
      Begin
        While ( Alternative.Stack <> Nil ) Do
          Begin
            Pointer := Alternative.Stack;
            Alternative.Stack := Alternative.Stack^.Next;
            Dispose( Pointer );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize list.
    This procedure is designed to initialize the
    solution list.

*************************************************)

    Procedure Initialize_List( Var List: Solution_List_Type );
      Begin
        List.Cost := Starting_Value;
        List.Amount := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in the list.
    This procedure is designed to put a solution
    in the list of the solutions that have been
    worked out so far.

*************************************************)

    Procedure Put_In_List( Var List: Solution_List_Type; Var Solution: Solution_Type );
      Begin
        If ( Solution.Cost < List.Cost )
          then
            Begin
              List.Amount := 1;
              List.Cost := Solution.Cost;
              List.Data[ 1 ] := Solution;
            End
          else
            If ( List.Amount < List_Limit )
              then
                If ( Solution.Cost = List.Cost )
                  then
                    Begin
                      Inc( List.Amount );
                      List.Data[ List.Amount ] := Solution;
                    End
                  else
                    WriteLn( 'Error in Put_In_List: Solution cost is greater than current cost.' )
              else
                WriteLn( 'Error in Put_In_List: Out of list space.' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: No more tries.
    This function returns true if there are no
    more possible solutions to the table.

*************************************************)

    Function No_More_Tries: Boolean;
      Begin
        No_More_Tries := ( Alternative.Current_Count = 0 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Minimize.
    This procedure solves the minimal general
    m-resource, m-activity assignment problem for
    total effectiveness.

*************************************************)

    Procedure Minimize( Var Table: Table_Type; Var Solution: Solution_Type );
      Var
        Invalid_Row: Invalid_Type;
      Begin
        Minimize_Hungarian_Algorithm( Table, Invalid_Row );
        Calculate_Cost( Table, Invalid_Row, Solution )
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Maximize.
    This procedure solves the maximum general
    m-resource, m-activity assignment problem for
    total effectiveness.

*************************************************)

    Procedure Maximize( Var Table: Table_Type; Var Solution: Solution_Type );
      Var
        Invalid_Row: Invalid_Type;
      Begin
        Maximize_Hungarian_Algorithm( Table, Invalid_Row );
        Calculate_Cost( Table, Invalid_Row, Solution )
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Minimize solve single.
    As previously defined.

*************************************************)

    Procedure Minimize_Solve_Single( Var Table: Table_Type; Var Solution: Solution_Type );
      Begin
        Initialize_Tries;
        Minimize( Table, Solution );
        Deallocate_Tries;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Maximize solve single.
    As previously defined.

*************************************************)

    Procedure Maximize_Solve_Single( Var Table: Table_Type; Var Solution: Solution_Type );
      Begin
        Initialize_Tries;
        Maximize( Table, Solution );
        Deallocate_Tries;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Minimize solve multiple.
    As previously defined.

*************************************************)

    Procedure Minimize_Solve_Multiple( Var Table: Table_Type; Var List: Solution_List_Type );
      Var
        Solution: Solution_Type;
      Begin
        Initialize_Tries;
        Initialize_List( List );
        Repeat
          Minimize( Table, Solution );
          Put_In_List( List, Solution );
        Until No_More_Tries;
        Deallocate_Tries;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Maximize solve multiple.
    As previously defined.

*************************************************)

    Procedure Maximize_Solve_Multiple( Var Table: Table_Type; Var List: Solution_List_Type );
      Var
        Solution: Solution_Type;
      Begin
        Initialize_Tries;
        Initialize_List( List );
        Repeat
          Maximize( Table, Solution );
          Put_In_List( List, Solution );
        Until No_More_Tries;
        Deallocate_Tries;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print out the solution.
    As previously defined.

*************************************************)

    Procedure Print_Solution( Var OutFile: Text; Var Solution: Solution_Type );
      Var
        Row: Byte;
      Begin
        WriteLn( OutFile, '----------Solution----------' );
        For Row := 1 to Solution.Size do
          WriteLn( OutFile, 'Person ', Row, ' gets job ', Solution.Data[ Row ].Job );
        WriteLn( OutFile, 'The optimal solution costs is ', Solution.Cost );
        WriteLn( OutFile, '-----------End-----------' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print out the set of solutions.
    As previously defined.

*************************************************)

    Procedure Print_Solutions( Var OutFile: Text; Var List: Solution_List_Type );
      Var
        Count: Byte;
      Begin
        For Count := 1 to List.Amount do
          Begin
            WriteLn( OutFile, '----------Solution number ', Count, '----------' );
            WriteLn( OutFile );
            Print_Solution( OutFile, List.Data[ Count ] );
            WriteLn( OutFile );
          End;
        WriteLn( OutFile, '-------------- End of solutions --------------' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read in the table.
    As previously defined.

*************************************************)

    Procedure Read_Table( Var InFile: Text; Var Table: Table_Type );
      Var
        Row,
        Column: Byte;
        Value: Value_Type;
      Begin
        ReadLn( InFile, Table.Size );
        For Row := 1 to Table.Size do
          Begin
            For Column := 1 to Table.Size do
              Begin
                Read( InFile, Value );
                If ( Value < 0 )
                  then
                    Begin
                      WriteLn( 'Error in Read_In: Negative value found!' );
                      Halt;
                    End;
                Table.Data[ Row, Column ] := Value;
              End;
            ReadLn( InFile );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write out the table.
    As previously defined.

*************************************************)

    Procedure Write_Table( Var OutFile: Text; Var Table: Table_Type );
      Var
        Row,
        Column: Byte;
      Begin
        WriteLn( OutFile, Table.Size );
        For Row := 1 to Table.Size do
          Begin
            For Column := 1 to Table.Size do
              Write( OutFile, Table.Data[ Row, Column ], ' ' );
            WriteLn( OutFile );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print out the table.
    As previously defined.

*************************************************)

    Procedure Print_Table( Var OutFile: Text; Var Table: Table_Type );
      Var
        Row,
        Column: Byte;
      Begin
        WriteLn( OutFile, '-------- Table of size ', Table.Size, '--------' );
        WriteLn( OutFile );
        For Row := 1 to Table.Size do
          Begin
            For Column := 1 to Table.Size do
              Write( OutFile, Table.Data[ Row, Column ], ' ' );
            WriteLn( OutFile );
          End;
        WriteLn( OutFile );
      End;

{----------------------------------------------------------------------------}

  End.

