{$F+} { Compiler Directive: Generate far procedure calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Search
    version 1.13

  This unit holds procedures designed to perform system searches for files.

  Purpose:
    These function are designed to find the file on the given drive.  They
    are especially useful, for when supporting program files are not in the
    current directory.

  How it works:
    The procedures use the standard DOS function calls to perform searches
    throughout the file directory structure.

  Features:
    Can search every directory in the current disk drive for the specified
      files.
    Superfind can perform the search across the multiple disk drives.

  Limitations:
    Note that when compiling under Speed Pascal/2, Find_Location may
      not always find the file.
    Also note that though under DOS, Super_Find will not generate any error
      messages on empty diskette drives, it will generate error windows
      under OS/2 which should then be set to return the error code to the
      program.

  Versions:
    1.10 - Added the superfind function to increase power and functions.
    1.11 - Updated superfind to include multiple drives.
    1.12 - Added the superfind with date function.

  Copyright 1989, 1995 All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Search;

  Interface

    Uses
      DOS,
     {$IFNDEF OS2}
      String_Utilities;
     {$ELSE}
      OS2Supp,
      String_U;
     {$ENDIF}

(***********************************************************

  Function: Find location of file in drive.

    This function will search all the directories for the
    first location of a file with the specified name.  Then
    it will return the path.  If the value passed is '',
    then it looks in the current path.
    Drive_Path is the specified path to begin the search.
      'A:\'           => looks in the specified drive.
      'A:\Path_Name\  => looks in the specified drive and
                         path.
    File_Name is the specified name of the file to look for.
    Question marks act as wild cards.

***********************************************************)

    Function Find_Location( Drive_Path, File_Name: String ): String;

(***********************************************************

  Function: Find file in the drive.

    This function will search all the directories for the
    first location of a file with the specified name.  Then
    it will return the filename and the path.  If the value
    passed is '' then it looks in the current path.
    Drive_Path is the specified path to begin the search.
      'A:\'           => looks in the specified drive.
      'A:\Path_Name\  => looks in the specified drive and
                         path.
    File_Name is the specified name of the file to look for.
    Question marks act as wild cards.

***********************************************************)

    Function Find_File( Drive_Path, File_Name: String ): String;

(***********************************************************

  Function: Super find.

    This function will search for the specified file across
    the whole system. If the file is found, the path-name
    combination is returned or the path depending on the
    value of the switch.

    First, the current path is searched.  If the file isn't
    found, then the current drive is completely searched.
    If that doesn't work, the available drives are each
    searched or alphabetical order.
    File_Name is the specified name of the file to look for.
    Question marks act as wild cards.

***********************************************************)

    Function Super_Find( File_Name: String; Return_Path_Only: Boolean ): String;

(***********************************************************

  Function: Super find with date.

    This function will search for the specified file with
    the given date across the whole system. If the file is
    found, the path-name combination is returned or the path
    depending on the value of the switch.

    First, the current path is searched.  If the file isn't
    found, then the current drive is completely searched.
    If that doesn't work, the available drives are each
    searched or alphabetical order.
    File_Name is the specified name of the file to look for.
    Month is the month from the file's attributes.
    Day is the day of the month from the file's attributes.
    Year is the year from the file's attributes.
    Question marks act as wild cards.

***********************************************************)

    Function Super_Find_With_Date( File_Name: String; Month, Day: Byte; Year: Word; Return_Path_Only: Boolean ): String;

{----------------------------------------------------------------------------}

  Implementation

    Type
      Drive_Map_Type = Set of Char;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Stretch.
    This procedure takes the name and stretches
    it in the Long_Name variable.  Spaces are
    added normally before the extension.

*************************************************)

  Procedure Stretch( Name: String; Var Long_Name: String );
    Var
      Where: Byte;
    Begin
      Long_Name := Name;
      Where := Pos( '.', Long_Name );
      If ( Where <> 0 )
        then
          Begin
            While ( ( Length( Long_Name ) < 12 ) and ( Where < 10 ) ) do
              Begin
                Insert( ' ', Long_Name, Where );
                Where := Pos( '.', Long_Name );
              End
          End;
      While ( Length( Long_Name ) < 12 ) do
        Long_Name := Long_Name + ' ';
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Identical.
    This function returns true if the given look
    file name matched the find file name.

*************************************************)

    Function Identical( Look_Name, Find_Name: String ): Boolean;
      Var
        Count: Byte;
        Same: Boolean;
      Begin
        Stretch( Look_Name, Look_Name );
        Same := True;
        Count := 1;
        Repeat
          If ( Find_Name[ Count ] <> '?' ) and ( Look_Name[ Count ] <> Find_Name[ Count ] )
            then
              Same := False
            else
              Inc( Count );
        Until ( ( not Same ) or ( Count > 12 ) );
        Identical := Same;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Identical with Date.
    This function returns true if the given look
    file name matched the find file name and the
    dates match.

*************************************************)

    Function Identical_With_Date( Look_Name, Find_Name: String; Look_Date, Find_Date: DateTime ): Boolean;
      Var
        Count: Byte;
        Same,
        Date_Okay: Boolean;
      Begin
        Date_Okay := ( ( Look_Date.Day = Find_Date.Day ) and ( Look_Date.Month = Find_Date.Month ) and
                       ( Look_Date.Year = Find_Date.Year ) );
        Stretch( Look_Name, Look_Name );
        Same := True;
        Count := 1;
        Repeat
          If ( Find_Name[ Count ] <> '?' ) and ( Look_Name[ Count ] <> Find_Name[ Count ] )
            then
              Same := False
            else
              Inc( Count );
        Until ( ( not Same ) or ( Count > 12 ) );
        Identical_With_Date := ( Same and Date_Okay );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Search for location.
    This function tries to find the location of
    a file matching the given file name in the
    path allowed by Real_Path.  The path is
    returned in the Real_Name variable.

*************************************************)

    Function Search_For_Location( Var Real_Path, File_Name, Real_Name: String ): Boolean;
      Var
        Done: Boolean;
        Temp_Path: String;
        Search_Record: SearchRec;
      Begin
       {$I-}
        FindFirst( Real_Path + '*.*', AnyFile, Search_Record );
       {$I+}
        Done := False;
        Stretch( File_Name, File_Name );
        While ( ( DOSError = 0 ) and ( Not Done ) ) do
          Begin
            If ( ( ( Search_Record.Attr and Directory ) <> 0 ) and
                 ( Search_Record.Name <> '.' ) and ( Search_Record.Name <> '..' ) )
              then
                Begin
                  Temp_Path := Real_Path + Search_Record.Name + '\';
                  If Search_For_Location( Temp_Path, File_Name, Real_Name )
                    then
                      Begin
                        Done := True;
                        Real_Path := Temp_Path;
                      End;
                End
              else
                If Identical( Search_Record.Name, File_Name )
                  then
                    Begin
                      Real_Name := Search_Record.Name;
                      Done := True;
                    End;
            FindNext( Search_Record )
          End;
        Search_For_Location := Done;
       {$IFDEF OS2}
        FindClose( Search_Record );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Search for location with date.
    This function tries to find the location of
    a file matching the given file name and date
    in the path allowed by Real_Path.  The path is
    returned in the Real_Name variable.

*************************************************)

    Function Search_For_Location_With_Date( Var Real_Path, File_Name, Real_Name: String; Date: DateTime ): Boolean;
      Var
        Done: Boolean;
        Temp_Path: String;
        Search_Record: SearchRec;
        Find_Time: DateTime;
      Begin
       {$I-}
        FindFirst( Real_Path + '*.*', AnyFile, Search_Record );
       {$I+}
        Done := False;
        Stretch( File_Name, File_Name );
        While ( ( DOSError = 0 ) and ( Not Done ) ) do
          Begin
            UnpackTime( Search_Record.Time, Find_Time );
            If ( ( ( Search_Record.Attr and Directory ) <> 0 ) and
                 ( Search_Record.Name <> '.' ) and ( Search_Record.Name <> '..' ) )
              then
                Begin
                  Temp_Path := Real_Path + Search_Record.Name + '\';
                  If Search_For_Location_With_Date( Temp_Path, File_Name, Real_Name, Date )
                    then
                      Begin
                        Done := True;
                        Real_Path := Temp_Path;
                      End;
                End
              else
                If Identical_With_Date( Search_Record.Name, File_Name, Date, Find_Time )
                  then
                    Begin
                      Real_Name := Search_Record.Name;
                      Done := True;
                    End;
            FindNext( Search_Record )
          End;
        Search_For_Location_With_Date := Done;
       {$IFDEF OS2}
        FindClose( Search_Record );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find location.
    As previously defined.

*************************************************)

    Function Find_Location( Drive_Path, File_Name: String ): String;
      Var
        Path,
        Real_Name: String;
      Begin
        If ( Drive_Path = '' )
          then
            Begin
              GetDir( 0, Path );
              Path := Path + '\';
            End
          else
            Path := Drive_Path;
        Capitalize( File_Name );
        If Search_For_Location( Path, File_Name, Real_Name )
          then
            Find_Location := Path
          else
            Find_Location := '';
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find file.
    As previously defined.

*************************************************)

    Function Find_File( Drive_Path, File_Name: String ): String;
      Var
        Path,
        Real_Name: String;
      Begin
        If ( Drive_Path = '' )
          then
            Begin
              GetDir( 0, Path );
              Path := Path + '\';
            End
          else
            Path := Drive_Path;
        Capitalize( File_Name );
        If Search_For_Location( Path, File_Name, Real_Name )
          then
            Find_File := Path + Real_Name
          else
            Find_File := '';
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Map drives.
    This procedure scans the entire drive spectum
    and notes where the drives exist and are
    ready to use.

*************************************************)

    Procedure Map_Drives( Var Map: Drive_Map_Type );
      Var
        Character: Char;
        Current_Path: String;
      Begin
        GetDir( 0, Current_Path );
        Map := [];
        For Character := 'A' to 'Z' do
          Begin
           {$IFDEF OS2}
            Change_Error_Handling( True, False );
           {$ENDIF}
           {$I-}
            ChDir( Character + ':\' );
           {$I+}
           {$IFDEF OS2}
            Change_Error_Handling( True, True );
           {$ENDIF}
            Case IOResult of
              0: Map := Map + [ Character ];
             {$IFDEF OS2}
              3: Map := Map + [ Character ];
             {$ENDIF}
              else ;
            End; { Case }
          End;
        ChDir( Current_Path );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Search other drives.
    This procedure searches the other drives in
    the drive map for the first file of the given
    file name.

*************************************************)

    Function Search_Other_Drives( Var Drive_Map: Drive_Map_Type; Var File_Name, Real_Name: String ): String;
      Var
        Found: Boolean;
        Drive: Char;
        Path: String;
      Begin
        Found := False;
        Drive := 'Z';
        Repeat
          If ( Drive in Drive_Map )
            then
              Begin
                Path := Drive + ':\';
                If Search_For_Location( Path, File_Name, Real_Name )
                  then
                    Found := True;
                Drive_Map := Drive_Map - [ Drive ];
              End;
          Drive := Pred( Drive );
        Until Found or ( Drive_Map = [] ) or ( Drive < 'A' );
        If Found
          then
            Search_Other_Drives := Path
          else
            Search_Other_Drives := '';
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Search other drives with date.
    This procedure searches the other drives in
    the drive map for the first file of the given
    file name and the given date.

*************************************************)

    Function Search_Other_Drives_With_Date( Var Drive_Map: Drive_Map_Type; Var File_Name, Real_Name: String;
                                            Date: DateTime ): String;
      Var
        Found: Boolean;
        Drive: Char;
        Path: String;
      Begin
        Found := False;
        Drive := 'Z';
        Repeat
          If ( Drive in Drive_Map )
            then
              Begin
                Path := Drive + ':\';
                If Search_For_Location_With_Date( Path, File_Name, Real_Name, Date )
                  then
                    Found := True;
                Drive_Map := Drive_Map - [ Drive ];
              End;
          Drive := Pred( Drive );
        Until Found or ( Drive_Map = [] ) or ( Drive < 'A' );
        If Found
          then
            Search_Other_Drives_With_Date := Path
          else
            Search_Other_Drives_With_Date := '';
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Super find.
    As previously defined.

*************************************************)

    Function Super_Find( File_Name: String; Return_Path_Only: Boolean ): String;
      Var
        Path,
        Save_Path,
        Real_Name: String;
        Drive_Map: Drive_Map_Type;
      Begin
        GetDir( 0, Path );
        Save_Path := Path;
        Path := Path + '\';
        Capitalize( File_Name );
        If Search_For_Location( Path, File_Name, Real_Name )
          then
            Begin
              If Return_Path_Only
                then
                  Super_Find := Path
                else
                  Super_Find := Path + Real_Name;
            End
          else
            Begin
              Path := Copy( Path, 1, 3 );
              If Search_For_Location( Path, File_Name, Real_Name )
                then
                  Begin
                    If Return_Path_Only
                      then
                        Super_Find := Path
                      else
                        Super_Find := Path + Real_Name;
                  End
                else
                  Begin
                    Map_Drives( Drive_Map );
                    Drive_Map := Drive_Map - [ Path[ 1 ] ];
                    Path := Search_Other_Drives( Drive_Map, File_Name, Real_Name );
                    If ( Path = '' )
                      then
                        Super_Find := ''
                      else
                        If Return_Path_Only
                          then
                            Super_Find := Path
                          else
                            Super_Find := Path + Real_Name;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Super find with date.
    As previously defined.

*************************************************)

    Function Super_Find_With_Date( File_Name: String; Month, Day: Byte; Year: Word; Return_Path_Only: Boolean ): String;
      Var
        Path,
        Save_Path,
        Real_Name: String;
        Date: DateTime;
        Drive_Map: Drive_Map_Type;
      Begin
        GetDir( 0, Path );
        Save_Path := Path;
        Date.Day := Day;
        Date.Month := Month;
        Date.Year := Year;
        Path := Path + '\';
        Capitalize( File_Name );
        If Search_For_Location_With_Date( Path, File_Name, Real_Name, Date )
          then
            Begin
              If Return_Path_Only
                then
                  Super_Find_With_Date := Path
                else
                  Super_Find_With_Date := Path + Real_Name;
            End
          else
            Begin
              Path := Copy( Path, 1, 3 );
              If Search_For_Location_With_Date( Path, File_Name, Real_Name, Date )
                then
                  Begin
                    If Return_Path_Only
                      then
                        Super_Find_With_Date := Path
                      else
                        Super_Find_With_Date := Path + Real_Name;
                  End
                else
                  Begin
                    Map_Drives( Drive_Map );
                    Drive_Map := Drive_Map - [ Path[ 1 ] ];
                    Path := Search_Other_Drives_With_Date( Drive_Map, File_Name, Real_Name, Date );
                    If ( Path = '' )
                      then
                        Super_Find_With_Date := ''
                      else
                        If Return_Path_Only
                          then
                            Super_Find_With_Date := Path
                          else
                            Super_Find_With_Date := Path + Real_Name;
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

  End.
