(*****************************************************************************

  Program: B-tree demonstration.

  Purpose:
    To demonstrate the abilities of the b-tree unit.

  Features:
    This program demonstrates the abilities of the b-tree unit by generating
      a series of random records and storing them in the tree.  Then
      immediately afterwards, it performs a search to verify the key was
      stored properly.  Finally, it checks for all the keys by finding them
      in order of file placement and performs another search for them.
      Lastly, it deletes the tree record by record.

  Copyright 1995, All rights reserved.
    P. Renaud.

  Compilers:
    Turbo Pascal version 4.0 to 6.0
    Speed Pascal/2 version 1.5

  Systems:
    MS-DOS, MDOS, OS/2.

*****************************************************************************)

Program B_Tree_Demonstration( Input, Output );

  Uses
    BTree;

  Const
   { Defines the amount of records to generate. }
    Amount = 200;
   { Defines the size of the record body. }
    Data_Range = 50;
   { Defines the size of the record key. }
    Key_Range = 6;  { Must be smaller than Data Range }

  Type
   { Defines the record structure. }
    Our_Data_Type = array[ 1 .. Data_Range ] of Char;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Create char.
    This function generates a random character and
    returns it.

*************************************************)

  Function Create_Char: Char;
    Var
      Character: Char;
    Begin
      Repeat
        Character := Chr( Random( 125 ) );
      Until ( Character in [ 'A' .. 'Z', 'a' .. 'z' ] );
      Create_Char := Character;
    End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: List tree.
    This function lists the entire tree that it
    can recover from the b-tree file.

*************************************************)

  Procedure List_Tree( Var Tree: Tree_Type );
    Var
      Data: Our_Data_Type;
    Begin
      If Find_First_In_Tree( Tree, Data )
        then
          Begin
            Write( Data, ' ' );
            While Find_Next_In_Tree( Tree, Data ) do
              Write( Data, ' ' )
          End;
    End;

{-----------------------------------------------------------------------------}

(*************************************************

  Main program.
    Create the tree.
    Verify the tree.
    Destroy the tree.
    Verify that the tree is destroyed.

*************************************************)

  Var
    Count,
    Inner,
    Counter: Integer;
    The_Tree: Tree_Type;
    Data: Our_Data_Type;
    Okay: Boolean;
  Begin
    WriteLn( 'B-tree demonstration program.' );
    WriteLn( 'Copyright 1994, All rights reserved.' );
    WriteLn( 'by P. Renaud.' );
    WriteLn;
    WriteLn( 'This program tests and demonstrates some of the abilities of the B-Tree' );
    WriteLn( 'unit.  First the tree is created and filled with 200 randomly generated' );
    WriteLn( 'records.  After the records are inserted, they are verified.  Next, the' );
    WriteLn( 'program will traverse the tree forwards and count how many records it' );
    WriteLn( 'finds.  This is followed by a backwards count.  Finally, the records are' );
    WriteLn( 'deleted from the tree in forward order to test the deletion routines.' );
    WriteLn;
    WriteLn( 'Note:  This program will run much slower from a floppy disk when ' );
    WriteLn( '       Node_Maximum_Length in the BTree unit is greater than 512.' );
    WriteLn;
    WriteLn( 'Please press [enter] to continue.' );
    ReadLn;

    Randomize;
    Open_Tree_File( The_Tree, 'Data.Tre', SizeOf( Our_Data_Type ), 1, Key_Range );

    For Counter := 1 to Amount do
      Begin
        For Count := 1 to Data_Range do
          Data[ Count ] := Create_Char;
        If Insert_In_Tree( The_Tree, Data )
          then
            Begin
              WriteLn( Counter:7, ' Inserted the key: ', Data );
              For Inner := Succ( Key_Range ) to Data_Range do
                Data[ Inner ] := ' ';
              WriteLn( Counter:7, '  Looking for key: ', Data );
              If Find_In_Tree( The_Tree, Data )
                then
                  WriteLn( Counter:7, '    Found the key: ', Data )
                else
                  Begin
                    WriteLn( Counter:7, ' Did not find the key ', Data );
                    Halt;
                  End;
            End;
      End;

    WriteLn( 'Searching for nodes in the tree.' );

    Counter := 0;
    If Find_First_In_Tree( The_Tree, Data )
      then
        Repeat
          WriteLn( 'Found ', Data, ' in the tree.' );
          Inc( Counter )
        Until not Find_Next_In_Tree( The_Tree, Data ) or ( Counter > Amount );
    If ( Counter < Amount )
      then
        Begin
          WriteLn( 'Error: Missing record.' );
          Halt;
        End;
    WriteLn;
    WriteLn;

    Counter := 0;
    If Find_Last_In_Tree( The_Tree, Data )
      then
        Repeat
          WriteLn( 'Found ', Data, ' in the tree.' );
          Inc( Counter )
        Until not Find_Previous_In_Tree( The_Tree, Data ) or ( Counter > Amount );
    If ( Counter < Amount )
      then
        Begin
          WriteLn( 'Error: Missing record.' );
          Halt;
        End;

    WriteLn( 'Deleteing nodes in tree.' );

    Count := Amount;
    While Find_First_In_Tree( The_Tree, Data ) do
      Begin
        WriteLn( Count:7, ' Deleting ', Data, ' from tree.' );
        If not Delete_From_Tree( The_Tree, Data )
          then
            Begin
              WriteLn( 'Error: Record disappeared.' );
              Halt;
            End;
        Dec( Count );
      End;

    WriteLn;
    WriteLn( 'Press enter to continue.' );
    ReadLn;

    List_Tree( The_Tree );
    Close_Tree_File( The_Tree );
    Erase( The_Tree.The_File );

  End.

