program     FITDEMO;                                                      (*
*                                                                        *
*                    Sample program for the demonstration                *
*                      of OptiVec data-fitting functions                 *
*                              for Borland Pascal                        *
*                                                                        *
*                                                                        *
*   Copyright 1996-2002 by OptiCode - Dr. Martin Sander Software Dev.    *
*                                                                        *
*   before compiling, please check and correct the BGI path statement    *
*   below as necessary!                                                  *
**************************************************************************)

{$E-} {$N+} {$D+} {$F+}
    (* The Force-Far-Calls option is absolutely necessary for all
       linfit and nonlinfit applications!                         *)
uses VecLib, VDstd, VDmath, VIstd, MDstd, VDnlfit, VDmnlfit,
     Vgraph, DOS, Graph;

const BGIDIR = '\BP\BGI\';
      polydeg = 5;     (* refers to the polynomial fitting example *)
var  XExp, YExp, YFit,
     YExp2, YFit2, YExp3, YFit3: dVector;
     sizex, xx, yy:              UInt;
     FitPars:                    array[0..6] of double;
         (* the highest number of parameters we will have in our examples *)
     ParStatus:                  array[0..6] of Integer;
     Opt:                        VD_NONLINFITOPTIONS;
     ExpList:                    array[0..2] of VD_EXPERIMENT;
     DataText:                   string[20];

 {the following function is needed to avoid the CRT unit,
   as it does not work with Win95B }
function ReadKeyNew: Char;
var xchar: Char;
begin
    asm mov ax, $0700; int $21; xor ah, ah; mov word ptr xchar, ax end;
    ReadKeyNew := xchar;
end;

    (* the following function is used with VD_linfit: *)
procedure PolyModel( BasFuncs:dVector; x:double; nfuncs:UInt );
var i:UInt;
begin
   (* This function fills the vector BasFuncs with powers of x.
      VD_linfit will then determine the coefficient for each of these
      powers.
      Note that the coefficients do not occur in the model function!
      The basis functions with known coefficients (whose fitting has
      been disabled) are still part of the model and must be calculated.
      You will see below that the values of the known (disabled)
      coefficients  must be set prior to calling VD_linfit.          *)
     BasFuncs^ := 1.0;
     for i:=1 to nfuncs-1 do
          VD_Pelement( BasFuncs,i )^ := VD_element( BasFuncs, i-1 ) * x;
end;

   (* the following function is used with VD_nonlinfit: *)
procedure VPolyModel( Y, X:dVector; size:UInt );
    (* Here, the model function has to fill a whole result vector,
       using your first guess of FitPars. In contrast to the
       linear case, now the coefficients are explicitly used in the
       model function. You must initialize FitPars with something,
       even if you have no idea about the result.
       FitPars must be global, so that the model function can access the
       parameters. With VD_nonlinfit, you can use just any functions
       in your model.
       For better comparison, we use the same polynomial approximation
       as before, but now we code it as if we didn't know that a
       polynomial actually is linear in its coefficients (and as if
       we didn't know either how to efficiently code a polynomial at all). *)
var i:  UInt;
    xi: double;
begin
    for i:=0 to size-1 do
    begin
        xi  := VD_element( X, i );
        VD_Pelement( Y, i )^ := FitPars[0]
            + FitPars[1] * xi
            + FitPars[2] * xi * xi
            + FitPars[3] * xi * xi * xi
            + FitPars[4] * xi * xi * xi * xi
            + FitPars[5] * xi * xi * xi * xi * xi;
                  (* FitPars is a static array and can be accessed
                     with the [ ] operator  *)
   end;
end;

   (* the following function is used with VD_multiNonlinfit: *)
procedure VSineModel( Y, X:dVector; size, iExperiment:UInt );
    (* According to the parameter iExperiment, the model function must
       choose the correct parameters for the calculation of the model
       function. The model function itself is the same for all experiments. *)
var omega, phase, amp: double;
begin
    case iExperiment of
        0: begin
               phase := FitPars[1];
               amp   := FitPars[4];
           end;
        1: begin
               phase := FitPars[2];
               amp   := FitPars[5];
           end;
        2: begin
               phase := FitPars[3];
               amp   := FitPars[6];
           end;
    end;
    omega := FitPars[0];
              (* we assume this parameter to be the same for all *)
    VDx_sin( Y, X, size, omega, phase, amp );
end;

begin
   writeln; writeln;
   writeln( 'This is a short demonstration of OptiVec data-fitting' );
   writeln( 'for Pascal in DOS real mode.' );
   writeln( 'Hit any key to continue  or abort with [ESC] !' );
   if ReadKeyNew = #27 then Halt;

   sizex := 200;
   XExp  := VD_vector( sizex );
   YExp  := VD_vector( sizex );
   YFit  := VD_vector( sizex );
   YExp2 := VD_vector( sizex );
   YExp3 := VD_vector( sizex );
   YFit2 := VD_vector( sizex );
   YFit3 := VD_vector( sizex );

   VD_ramp( XExp, sizex, 0, 1.0/(sizex-1) ); (* "experimental" x-axis from 0 to 1 *)
   VD_cubic( YExp, XExp, sizex );        (* fake "measured" y-data as y = x^3 *)
   VD_noise( YFit, sizex, 1, 0.005 );
   VD_addV( YExp, YExp, YFit, sizex ); (* add 1% peak-to-peak "experimental noise" *)

   V_initgraph( BGIDIR );  (* this should be your correct path, see above! *)
   yy := getmaxy div 15;
   xx := round( getmaxx * 0.3 );
   V_setPlotRegion( xx, yy, getmaxx, getmaxy-yy );
   VD_xyAutoPlot( XExp, YExp, sizex, PS_NULL+SY_CROSS, LIGHTGREEN );
     SetViewport( 1, 1, xx, getmaxy, FALSE ); clearviewport;
     SetTextJustify( LeftText, TopText );
     SetTextStyle( DefaultFont, HorizDir, 1 );
     SetColor( WHITE );
     OutTextXY( 2,  20, 'Suppose these are' );
     OutTextXY( 2,  40, 'your experimental' );
     OutTextXY( 2,  60, 'data points.' );
     OutTextXY( 2,  80, '(Actually, they consist' );
     OutTextXY( 2, 100, 'of a simple cubic with' );
     OutTextXY( 2, 120, '1% added noise.)' );
     if ReadKeyNew = #27 then Halt;

        (* now fit your data to one of the simplest models, a polynomial *)
   VD_polyfit( @FitPars, polydeg, XExp, YExp, sizex );
   VD_poly( YFit, XExp, sizex, @FitPars, polydeg ); (* calculate fit curve *)
   V_continuePlot;   (* go back to the last plotting region *)
   VD_xy2AutoPlot( XExp, YExp, sizex, PS_NULL or SY_CROSS, LIGHTGREEN,
                   XExp, YFit, sizex, PS_SOLID, LIGHTRED );
     SetViewport( 1, 1, xx, getmaxy, FALSE ); clearviewport;
     SetTextJustify( LeftText, TopText );
     SetTextStyle( DefaultFont, HorizDir, 1 );
     SetColor( WHITE );

     OutTextXY( 2, 10, 'The red curve is a' );
     OutTextXY( 2, 30, 'fifth-order polynomial' );
     OutTextXY( 2, 50, 'fitted to your data.' );
     OutTextXY( 2, 70, 'Without noise, the' );
     OutTextXY( 2, 90, 'coefficients should' );
     OutTextXY( 2,110, 'have been:' );
     OutTextXY( 2,130, '{0, 0, 0, 1.0, 0, 0}' );

     OutTextXY( 2,160, 'Actually, we got:' );
     Str( FitPars[0]:7:5, DataText );
     OutTextXY( 2,180, 'a0 = ' + DataText );
     Str( FitPars[1]:7:5, DataText );
     OutTextXY( 2,200, 'a1 = ' + DataText );
     Str( FitPars[2]:7:5, DataText );
     OutTextXY( 2,220, 'a2 = ' + DataText );
     Str( FitPars[3]:7:5, DataText );
     OutTextXY( 2,240, 'a3 = ' + DataText );
     Str( FitPars[4]:7:5, DataText );
     OutTextXY( 2,260, 'a4 = ' + DataText );
     Str( FitPars[5]:7:5, DataText );
     OutTextXY( 2,280, 'a5 = ' + DataText );
     OutTextXY( 2,310, 'Note how even moderate' );
     OutTextXY( 2,330, 'noise leads to rather' );
     OutTextXY( 2,350, 'large errors in the' );
     OutTextXY( 2,370, 'fit parameters, if' );
     OutTextXY( 2,390, 'there are too many' );
     OutTextXY( 2,410, '"free" parameters.' );
     if ReadKeyNew = #27 then Halt;

        (* now refine your fit by switching to a general linear model,
           giving you the chance to consider only the uneven terms    *)
   ParStatus[0] := 0;
   ParStatus[2] := 0;
   ParStatus[4] := 0;  (* disable fitting of even terms *)
   FitPars[0] := 0.0;  (* the disabled fitting parameters must be *)
   FitPars[2] := 0.0;  (*  initialized before calling VD_linfit ! *)
   FitPars[4] := 0.0;  (* set them to the known value, 0.0 *)
   ParStatus[1] := 1;
   ParStatus[3] := 1;
   ParStatus[5] := 1;  (* enable fitting of uneven terms *)
   VD_linfit( @FitPars, @ParStatus, polydeg+1,
              XExp, YFit, sizex,
              @PolyModel );
   VD_poly( YFit, XExp, sizex, @FitPars, polydeg );
             (* calculate new fit curve  *)
   V_continuePlot;   (* go back to the last plotting region *)
   VD_xy2AutoPlot( XExp, YExp, sizex, PS_NULL + SY_CROSS, LIGHTGREEN,
                   XExp, YFit, sizex, PS_SOLID, LIGHTRED );
     SetViewport( 1, 1, xx, getmaxy, FALSE ); clearviewport;
     SetTextJustify( LeftText, TopText );
     SetTextStyle( DefaultFont, HorizDir, 1 );
     SetColor( WHITE );
     OutTextXY(  2, 10, 'Suppose you know that' );
     OutTextXY(  2, 30, 'the coefficients of' );
     OutTextXY(  2, 50, 'all even terms are 0.' );
     OutTextXY(  2, 70, 'Then you fit to your' );
     OutTextXY(  2, 90, 'own linear model,' );
     OutTextXY(  2,110, 'consisting only of' );
     OutTextXY(  2,130, 'uneven terms.' );
     OutTextXY(  2,160, 'Now we get:' );
     OutTextXY(  2,180, 'a0 = 0 (fix)' );
     Str( FitPars[1]:7:5, DataText );
     OutTextXY(  2,200, 'a1 = ' + DataText );
     OutTextXY(  2,220, 'a2 = 0 (fix)' );
     Str( FitPars[3]:7:5, DataText );
     OutTextXY(  2,240, 'a3 = ' + DataText );
     OutTextXY(  2,260, 'a4 = 0 (fix)' );
     Str( FitPars[5]:7:5, DataText );
     OutTextXY(  2,280, 'a5 = ' + DataText );
     OutTextXY(  2,310, 'This is about as close' );
     OutTextXY(  2,330, 'as we can get in the' );
     OutTextXY(  2,350, 'presence of noise.' );
   if ReadKeyNew = #27 then Halt;

        (* here, we mis-use a non-linear fitting algorithm
           for our simple problem. ParStatus remains set as before *)
   VD_getNonlinfitOptions( Opt );
   Opt.FigureOfMerit := 0;  (* choose least-square fitting *)
   Opt.AbsTolChi     := 1.e-6;
   Opt.FracTolChi    := 1.e-3; (* makes the fit fast, but not very accurate *)
   Opt.LevelOfMethod := 3;  (* if you fear you might jump into a
         local rather than the true global parameter optimum, try
         LevelOfMethod := 7 - but only if you have time to wait for the result *)
   VD_setNonlinfitOptions( Opt );

   FitPars[0] := 0.0;
   FitPars[2] := 0.0;
   FitPars[4] := 0.0;  (* set known coefficients to the value, 0.0, as before *)
   FitPars[1] := 1.5;
   FitPars[3] := 1.5;
   FitPars[5] := 1.5;  (* you must provide some guess here!
     all fitting parameters must be initialized before calling VD_nonlinfit ! *)
   VD_nonlinfit( @FitPars, @ParStatus, polydeg+1,
                 XExp, YExp, sizex,
                 @VPolyModel, nil );
      (* If you know the derivatives with respect to each parameter, put
         your knowledge into a DerivModel function and replace the 'nil'
         parameter with it. (Actually, here we do know; but let's assume
         we don't, and have VD_nonlinfit call the numeric differentiation
         procedure. *)
   VPolyModel( YFit, XExp, sizex );  (* get fit curve from model *)

   V_continuePlot;   (* go back to the last plotting region *)
   VD_xy2AutoPlot( XExp, YExp, sizex, PS_NULL or SY_CROSS, LIGHTGREEN,
                   XExp, YFit, sizex, PS_SOLID, LIGHTRED );
     SetViewport( 1, 1, xx, getmaxy, FALSE ); clearviewport;
     SetTextJustify( LeftText, TopText );
     SetTextStyle( DefaultFont, HorizDir, 1 );
     SetColor( WHITE );
     OutTextXY(  2, 10, 'Let us fire with the' );
     OutTextXY(  2, 30, '"cannon" of a non-' );
     OutTextXY(  2, 50, 'linear fit on our' );
     OutTextXY(  2, 70, 'simple "sparrow"' );
     OutTextXY(  2, 90, 'problem.' );
     OutTextXY(  2,110, 'It takes much longer' );
     OutTextXY(  2,130, 'to find the result...' );
     OutTextXY(  2,160, 'But finally we get:' );
     OutTextXY(  2,180, 'a0 = 0 (fix)' );
     Str( FitPars[1]:7:5, DataText );
     OutTextXY(  2,200, 'a1 = ' + DataText );
     OutTextXY(  2,220, 'a2 = 0 (fix)' );
     Str( FitPars[3]:7:5, DataText );
     OutTextXY(  2,240, 'a3 = ' + DataText );
     OutTextXY(  2,260, 'a4 = 0 (fix)' );
     Str( FitPars[5]:7:5, DataText );
     OutTextXY(  2,280, 'a5 = ' + DataText );
     OutTextXY(  2,310, 'That is virtually the' );
     OutTextXY(  2,330, 'same as before.' );
   if ReadKeyNew = #27 then Halt;

        (* finally, let's suppose you have several experimental
           curves, measuring the same physical process under
           slightly different conditions.
           Say, we have a vibration, and each measurement
           begins with a different phase and has a somewhat
           different amplitude, but the same frequency.  *)
     VD_ramp( XExp, sizex, 0, 1.0/(sizex-1) ); (* x-axis again from 0 to 1 *)
     VDx_sin( YExp,  XExp, sizex, 15.0,  0.0, 1.2 );  (* first 'measurement' *)
     VDx_sin( YExp2, XExp, sizex, 15.0,  0.5, 1.0 );  (* second 'measurement'*)
     VDx_sin( YExp3, XExp, sizex, 15.0, -1.8, 0.75 ); (* third 'measurement' *)

   V_continuePlot;   (* go back to the last plotting region *)
   VD_xy2AutoPlot( XExp, YExp,  sizex, PS_NULL + SY_CROSS, LIGHTGREEN,
                   XExp, YExp2, sizex, PS_NULL + SY_CROSS, LIGHTBLUE );
   VD_xyDataPlot(  XExp, YExp3, sizex, PS_NULL + SY_CROSS, YELLOW );
     SetViewport( 1, 1, xx, getmaxy, FALSE ); clearviewport;
     SetTextJustify( LeftText, TopText );
     SetTextStyle( DefaultFont, HorizDir, 1 );
     SetColor( WHITE );
     OutTextXY(  2, 10, 'Finally, let us fit' );
     OutTextXY(  2, 25, 'several sets of experi-' );
     OutTextXY(  2, 40, 'mental data at once' );
     OutTextXY(  2, 55, '(sine waves with the' );
     OutTextXY(  2, 70, 'same frequency, but' );
     OutTextXY(  2, 85, 'different phases and' );
     OutTextXY(  2,100, 'amplitudes):' );
     OutTextXY(  2,115, 'First you see the' );
     OutTextXY(  2,130, '"experimental" data' );
     OutTextXY(  2,145, 'Please wait (may take' );
     OutTextXY(  2,160, 'several minutes)...' );

         (* cram your experiments into the array of VD_EXPERIMENT structs: *)
   ExpList[0].X := XExp;   ExpList[0].Y := YExp;   ExpList[0].size := sizex;
   ExpList[1].X := XExp;   ExpList[1].Y := YExp2;  ExpList[1].size := sizex;
   ExpList[2].X := XExp;   ExpList[2].Y := YExp3;  ExpList[2].size := sizex;
          (* we are not using the InvVar and WeightOfExperiment fields
             of ExpList, as we are not weighting the data. *)

   VI_equC( @ParStatus, 7, 1 ); (* we have 1 frequency, 3 phases, and 3 amplitudes,
                  and all these 7 parameters are unknown.
                  We must provide a first guess for each of them: *)
   FitPars[0] := 10.0;  (* the frequency term *)
   FitPars[1] :=  0.0;
   FitPars[2] :=  0.0;
   FitPars[3] :=  0.0;  (* the three phases  *)
   FitPars[4] :=  1.5;
   FitPars[5] :=  1.5;
   FitPars[6] :=  1.5;  (* the three amplitudes *)
   VD_getNonlinfitOptions( Opt );
   Opt.AbsTolChi  := 1.e-8;
   Opt.FracTolChi := 1.e-6; (* force higher accuracy to avoid premature break-off *)
     (*  Unlike almost every other fitting routine available, you
         can get a result even for input parameters much farther off
         from the true value than the 'guesses' chosen above.
         But then you must run VD_multiNonlinfit at 'full power' and
         enable the following line:                  *)
     (* Opt.LevelOfMethod = 7; *)
   VD_setNonlinfitOptions( Opt );

   VD_multiNonlinfit( @FitPars, @ParStatus, 7,
                      @ExpList, 3,
                      @VSineModel,
                      nil ); (* Again, we pretend we don't know the derivatives
                bring the phases into the range -PI < phase < + PI *)
   VD_modC( @(FitPars[1]), @(FitPars[1]), 3, 2.0*PI );
   if FitPars[1] > PI then FitPars[1] := FitPars[1] - 2.0*PI
   else if FitPars[1] < -PI then FitPars[1] := FitPars[1] + 2.0*PI;
   if FitPars[2] > PI then FitPars[2] := FitPars[2] - 2.0*PI
   else if FitPars[2] < -PI then FitPars[2] := FitPars[2] + 2.0*PI;
   if FitPars[3] > PI then FitPars[3] := FitPars[3] - 2.0*PI
   else if FitPars[3] < -PI then FitPars[3] := FitPars[3] + 2.0*PI;
   VSineModel( YFit,  XExp, sizex, 0 );  (* get fit curves from your model *)
   VSineModel( YFit2, XExp, sizex, 1 );
   VSineModel( YFit3, XExp, sizex, 2 );

   V_continuePlot;   (* go back to the last plotting region *)
   VD_xy2AutoPlot( XExp, YExp,  sizex, PS_NULL + SY_CROSS, LIGHTGREEN,
                   XExp, YExp2, sizex, PS_NULL + SY_CROSS, LIGHTBLUE );
   VD_xyDataPlot(  XExp, YExp3, sizex, PS_NULL + SY_CROSS, YELLOW );
   VD_xyDataPlot(  XExp, YFit, sizex, PS_SOLID, LIGHTRED );
   VD_xyDataPlot(  XExp, YFit2, sizex, PS_SOLID, LIGHTRED );
   VD_xyDataPlot(  XExp, YFit3, sizex, PS_SOLID, LIGHTRED );
     SetViewport( 1, 1, xx, getmaxy, FALSE ); clearviewport;
     SetTextJustify( LeftText, TopText );
     SetTextStyle( DefaultFont, HorizDir, 1 );
     SetColor( WHITE );
     OutTextXY(  2, 10, 'Several sets of experi-' );
     OutTextXY(  2, 25, 'mental data at once' );
     OutTextXY(  2, 40, '(sine waves with the' );
     OutTextXY(  2, 55, 'same frequency, but' );
     OutTextXY(  2, 70, 'different phases and' );
     OutTextXY(  2, 85, 'amplitudes):' );
     OutTextXY(  2,110, 'Here are the results' );
     OutTextXY(  2,125, '(in brackets: "true")' );
     Str( FitPars[0]:7:5, DataText );
     OutTextXY(  2,140, 'freq = ' + DataText + '(15.0)' );
     Str( FitPars[1]:7:5, DataText );
     OutTextXY(  2,155, 'ph1  = ' + DataText + '(0.0)' );
     Str( FitPars[2]:7:5, DataText );
     OutTextXY(  2,170, 'ph2  = ' + DataText + '(0.5)' );
     Str( FitPars[3]:7:5, DataText );
     OutTextXY(  2,185, 'ph3  = ' + DataText + '(-1.8)' );
     Str( FitPars[4]:7:5, DataText );
     OutTextXY(  2,200, 'amp1 = ' + DataText + '(1.2)' );
     Str( FitPars[5]:7:5, DataText );
     OutTextXY(  2,215, 'amp2 = ' + DataText + '(1.0)' );
     Str( FitPars[6]:7:5, DataText );
     OutTextXY(  2,230, 'amp3 = ' + DataText + '(0.75)' );

   ReadKeyNew;
   closegraph;
   V_freeAll;
end.

