{$U-$V-}   {disable CNTRL C, disable strict string length type checking}

            { Basic Graphics Package.     Version 2.0  07 Nov 87}

const
  GxC80       = $03;            {colour 80-column text mode}
  GxEga       = $10;            {colour EGA mode}
  GxVga       = $12;            {colour VGA mode - PS/2 systems only}
  GxMaxRow    = 349;            {row addresses are 0-349, (0-479 for PS/2}
  GxMaxCol    = 639;            {col addresses are 0-639}
  GxIndent    =  78;            {to give square default viewport}
  GxIndexRng  =  15;            {colour index range; 0 is index of background}
  GxColourRng =  63;            {colour palette range}
  GxSet       =   0;            {write in overwrite (set) mode}
  GxXor       =   1;            {write using xor mode}
  BELL        =  ^G;
  EgaBase     = $A000;          {base address of EGA board}

var
  GxBeamX, GxBeamY: real;              {current beam position - world coords}
  GxPage:       integer;
  GxPalette:    array [0..GxIndexRng] of integer;
  GxBackground: integer;               {background index for all palettes}
  GxIndex:      integer;               {current colour index}
  GxText:       integer;               {current colour for text}
  GxTextBack:   integer;               {current background colour for text}
  GxBorderIndex:integer;               {current border index}
  GxClip2d:     boolean;               {true if clipping, false otherwise}
  GxMode:       integer;               {current writing mode:
                                        0 = overwrite, 1 = xor write}

  GxWxt, GxWyt, GxWxb, GxWyb:  real;     {Current Window's coordinates}
  GxVxt, GxVyt, GxVxb, GxVyb:  integer;  {Current Viewport's coordinates}
  GxSx, GxSy: real;                      {Current View's scale factors}
  GxTx, GxTy: real;                      {Current View's translation constants}

  GxResult: record
              ax, bx, cx, dx, bp,
              si, di, ds, es, flags: integer;   {record for DOS interface}
            end;

            { Graphics Segment Sub-Picture Package. }

const
  GxSegNoMax = 100;
  GxScribe   =   0;
  GxDisplay  =   1;
  GxErase    =   2;

type
  GxName     = string [15];
  GxSegRng   = 1 .. GxSegNoMax;

  GxDrawOp   = (GxNoOp, GxOpMoveTo, GxOpPlotAt, GxOpDrawTo,
                GxOpRelMoveTo, GxOpRelDrawTo,
                GxOpRelPlotAt, GxOpLineIndex);
  GxLine     = ^GxLinRcd;
  GxLinRcd = record                    {line record within a segment}
               op: GxDrawOp;
               x1, y1: real;
               next: GxLine;
             end;

  GxSegAttribute = ^GxSegRcd;
  GxSegRcd = record                    {segment attribute record}
               visible: boolean;
               mode:    integer;       {GxSet or GxXor}
               priority: integer;
               pivotX, pivotY: real;
               first, last: GxLine;
             end;
  GxSegTable    = array [GxSegRng] of GxSegAttribute;  {seg list by name}
  GxPrioTable   = array [GxSegRng] of integer;         {seg list by priority}

var

{Segments and their attributes}

  GxSegOpen: boolean;                  {flag set true if a segment is open}
  GxSegCount: integer;                 {total number of segments defined}
  GxSegment: GxSegTable;               {segment list by name}
  GxPrioList: GxPrioTable;             {segment list by priority}
  GxPriority: integer;                 {Priority for all newly created segs.}
  GxVisibility: boolean;               {Visibility for all new segments}

{the currently open segment's attributes}

  GxSegNo:   integer;
  GxCurrSeg: GxSegAttribute;
  GxPivotX, GxPivotY: real;            {pivot point coords for rotations}

procedure GxNormalize;
  begin
  Port [$03CE] := 5;                   {select Mode register}
  Port [$03CF] := 0;                   {write mode = 0, as expected by BIOS}
  Port [$03CE] := 8;                   {select Bit Mask register}
  Port [$03CF] := $FF;                 {all points writeable}
  end {GxNormalize};

procedure WritePixel (x, y: integer);

  {write pixel in current line index colour at x, y in IBM (top-down) coords.}

  var
    offset: integer;
    latch: byte;
  begin
  Port [$03CE] := 8;                     {select Bit Mask register}
  Port [$03CF] := $80 shr (x and 7);     {set Bit Mask register}
  offset := y * 80 + x div 8;
  latch := Mem [EgaBase: offset];        {set processor latches}
  Mem [EgaBase: offset] := GxIndex;
  Port [$03CF] := $FF;                   {reset all points writeable}
  end {WritePixel};

procedure DrawLine (x1, y1, x2, y2: integer);
  {Bresenham's Algorithm for drawing lines in raster graphics.  See
   Foley and van Dam, "Fundamentals of Interactive Computer Graphics"
   Addison-Wesley, 1982, pp433-436.  Generalized for lines of all slopes.}

            {This version written for maximum speed}
  var
    x, y: integer;               {current pixel coords}
    stepx, stepy: integer;
    dx, dy: integer;
    d, incr1, incr2: integer;
    offset: integer;
    latch: byte;

  begin

 {initialize starting values for frame buffer and x, y}

  x := x1;  y := y1;
 {write initial entry into frame buffer for x1, y1}

  Port [$03CE] := 5;  {select Mode register and ...}
  Port [$03CF] := 2;  {... set to write}
  Port [$03CE] := 8;  {select Bit Mask register}
    {draw point at x, y}
  Port [$03CF] := $80 shr (x and 7);     {set Bit Mask register}
  offset := y * 80 + x div 8;
  latch := Mem [EgaBase: offset];
  Mem [EgaBase: offset] := GxIndex;

  {compute constants for the Algorithm}

  dx := x2 - x1;  dy := y2 - y1;
  if dx < 0 then
    begin
    stepx := -1;  dx := - dx;
    end
  else
    stepx := 1;
  if dy < 0 then
    begin
    stepy := -1;  dy := - dy;
    end
  else
    stepy := 1;

  {Compute the points and write their images into the frame buffer}

  if dy < dx then
    begin
    incr2 := (dy - dx) shl 1;
    d     := incr2 + dx;
    incr1 := d + dx;
    while x <> x2 do
      begin
      x := x + stepx;
      if d < 0 then
        d := d + incr1
      else
        begin
        d := d + incr2;
        y := y + stepy;
        end { d >= 0};
         {draw point at x, y}
      Port [$03CF] := $80 shr (x and 7);     {set Bit Mask register}
      offset := y * 80 + x div 8;
      latch := Mem [EgaBase: offset];
      Mem [EgaBase: offset] := GxIndex;
      end { while x <> x2};
    end {dy < dx}
  else
    begin
    incr2 := (dx - dy) shl 1;
    d     := incr2 + dy;
    incr1 := d + dy;
    while y <> y2 do
      begin
      y := y + stepy;
      if d < 0 then
        d := d + incr1
      else
        begin
        d := d + incr2;
        x := x + stepx;
        end { d >= 0};
        {draw point at x, y}
      Port [$03CF] := $80 shr (x and 7);     {set Bit Mask register}
      offset := y * 80 + x div 8;
      latch := Mem [EgaBase: offset];
      Mem [EgaBase: offset] := GxIndex;
      end { while y <> y2};
    end {dy >= dx};
  GxNormalize;
  end {DrawLine};

procedure clip2D (x1, y1, x2, y2: real);

{If any portion of the line is visible then clip the input line against
 the current window, and draw it}

  label 99;
  const
    left = 1;  right = 2; bottom = 4; top = 8;
  var
    c, c1, c2: byte;
    x, y: real;

  begin
    c1 := 0;
    if x1 < GxWxb then c1 := c1 + left else
    if x1 > GxWxt then c1 := c1 + right;
    if y1 < GxWyb then c1 := c1 + bottom else
    if y1 > GxWyt then c1 := c1 + top;
    c2 := 0;
    if x2 < GxWxb then c2 := c2 + left else
    if x2 > GxWxt then c2 := c2 + right;
    if y2 < GxWyb then c2 := c2 + bottom else
    if y2 > GxWyt then c2 := c2 + top;
  while (c1 <> 0) or (c2 <> 0) do
    begin
    if (c1 and c2) <> 0 then     {line is completely invisible}
      goto 99;
        {clipping is necessary}
    c := c1;
    if c = 0 then c := c2;
    if (left and c) <> 0 then          {crosses x = GxWxb}
      begin
      x := GxWxb;
      y := y1 + (GxWxb - x1) * (y2 - y1) / (x2 - x1);
      end
    else if (right and c) <> 0 then     {crosses x =  GxWxt}
      begin
      x := GxWxt;
      y := y1 + (GxWxt - x1) * (y2 - y1) / (x2 - x1);
      end
    else if (bottom and c) <> 0 then     {crosses y = GxWyb}
      begin
      x := x1 + (GxWyb - y1) * (x2 - x1) / (y2 - y1);
      y := GxWyb;
      end
     else if (top and c) <> 0 then     {crosses y = GxWyt}
       begin
       x := x1 + (GxWyt - y1) * (x2 - x1) / (y2 - y1);
       y := GxWyt;
       end;
    if c = c1 then
      begin
      x1 := x; y1 := y;
      c1 := 0;
      if x1 < GxWxb then c1 := c1 + left else
      if x1 > GxWxt then c1 := c1 + right;
      if y1 < GxWyb then c1 := c1 + bottom else
      if y1 > GxWyt then c1 := c1 + top;
      end
    else
      begin
      x2 := x; y2 := y;
      c2 := 0;
      if x2 < GxWxb then c2 := c2 + left else
      if x2 > GxWxt then c2 := c2 + right;
      if y2 < GxWyb then c2 := c2 + bottom else
      if y2 > GxWyt then c2 := c2 + top;
      end;
    end;

      { draw line }

  DrawLine (round (x1 * GxSx + GxTx),
              round (y1 * GxSy + GxTy),
              round (x2 * GxSx + GxTx),
              round (y2 * GxSy + GxTy));
  99:
  end {clip2D};

procedure GxMakeEntry (operator: GxDrawOp; x, y: real);
  const
    MinMem = 200.0;                   {200 paragraphs of 16 bytes each}
  var
    entry:    GxLine;
    TrueFree: real;
  begin
  TrueFree := MaxAvail;
  if TrueFree < 0.0 then
    TrueFree := TrueFree + 65536.0;   {No. of free paragraphs (16 bytes) of
                                       free memory left}
  if TrueFree > MinMem then
    begin
    new (entry);
    with entry^ do
      begin
      op     := operator;
      x1     := x;  y1 := y;
      next   := nil;
      end;
    with GxCurrSeg^ do
      begin
      last^.next := entry;
      last := last^.next;
      end;
    end;
  end {GxMakeEntry};

procedure GxMoveTo(x, y: real);

  {Absolute move in world coordinates}

  begin
  GxBeamX := x;
  GxBeamY := y;
  end {GxMoveTo};

procedure GxPlotAt (x, y: real);
  var
    Vx, Vy: integer;
    visible: boolean;
    offset: integer;
    latch: byte;

  begin

  GxBeamX := x;
  GxBeamY := y;

    {Test visibility against window.}

  if GxClip2d then
    if (x <= GxWxb) then visible := false
    else visible := (x <= GxWxt);
    if visible then
      if (y <= GxWyb) then visible := false
      else visible := (y <= GxWyt)
  else visible := true;

  if visible then
    begin
    Vx := round (x * GxSx + GxTx);
    Vy := round (y * GxSy + GxTy);
    Port [$03CE] := 5;  {select Mode register and ...}
    Port [$03CF] := 2;  {... set to write}
    Port [$03CE] := 8;

      {draw point at Vx, Vy}

    Port [$03CF] := $80 shr (Vx and 7);     {set Bit Mask register}
    offset := Vy * 80 + Vx div 8;
    latch := Mem [EgaBase: offset];
    Mem [EgaBase: offset] := GxIndex;
    GxNormalize;
    end;
  end {GxPlotAt};

procedure GxDrawTo (x, y: real);

  {absolute draw in world coordinates}

  begin
  if GxClip2d then
    Clip2D (GxBeamX, GxBeamY, x, y)
  else
    DrawLine (round (GxBeamX * GxSx + GxTx),
                round (GxBeamY * GxSy + GxTy),
                round (x * GxSx + GxTx),
                round (y * GxSy + GxTy));
  GxBeamX := x;  GxBeamY := y;
  end {GxDrawTo};

procedure GxRelMoveTo (x, y: real);
  begin
  GxMoveTo (GxBeamX + x, GxBeamY + y);
  end {GxRelMoveTo};

procedure GxRelPlotAt (x, y: real);
  begin
  GxPlotAt (GxBeamX + x, GxBeamY + y);
  end {GxRelPlotAt};

procedure GxRelDrawTo (x, y: real);
  begin
  GxDrawTo (GxBeamX + x, GxBeamY + y);
  end {GxRelDrawTo};


{
*****************************************************************************
              THE USER'S PACKAGE OF 2-D ROUTINES STARTS HERE.
}

procedure pause;
  begin
  write (BELL);  readln;
  end {pause};

procedure ColourMap (reg, colour: integer);
  begin
  reg := reg and GxIndexRng;
  colour := colour and GxColourRng;
  with GxResult do
    begin
    ax := $1000;
    bx := (colour shl 8) or reg;
    end;
  Intr ($10, GxResult);
  GxPalette [reg] := colour;
  end {ColourMap};

procedure DefaultMap;
  {set up the default colour map -
   0: black;      1: dark blue;     2: dark green;   3: dark cyan;
   4: dark red;   5: dark magenta;  6: dark yellow;  7: dark grey;
   8: light grey; 9: full blue;    10: full green;  11: full cyan;
  12: full red   13: full magenta; 14: full yellow; 15: white.}

  var
    k: integer;

  begin
  for k := 0 to 6 do
    ColourMap (k, k); {black, dark: blue, green, cyan, red, magenta, yellow}
  ColourMap (7, 56);                   {dark grey}
  ColourMap (8, 7);                    {light grey}
  for k := 1 to 7 do
    ColourMap (k+8, 9*k);  {full: blue, green, cyan, red, magenta, yellow;
                            white}
  end {DefaultMap};

procedure Border (colour: integer);

  {draw the current viewport outline in the index 'colour'}
  var
    index: integer;
  begin
  if colour <> GxBackground then
    begin
    index := GxIndex;  GxIndex := colour;
    DrawLine (GxVxb, GxVyb, GxVxb, GxVyt);
    DrawLine (GxVxb, GxVyt, GxVxt, GxVyt);
    DrawLine (GxVxt, GxVyt, GxVxt, GxVyb);
    DrawLine (GxVxt, GxVyb, GxVxb, GxVyb);
    GxIndex := index;
    GxBorderIndex := colour;
    end;
  end {border};

procedure Window (xb, yb, xt, yt: real);

  {define the user's window boundaries}

  begin
  if xb > xt then window (xt, yb, xb, yt)  else
  if yb > yt then window (xb, yt, xt, yb)  else
  if xb = xt then window (0.0, yb, 1.0, yt) else
  if yb = yt then window (xb, 0.0, xt, 1.0)
  else
    begin
    GxWxb := xb;  GxWxt := xt;
    GxWyb := yb;  GxWyt := yt;
    GxSx := (GxVxt - GxVxb) / (GxWxt - GxWxb);
    GxSy := (GxVyt - GxVyb) / (GxWyt - GxWyb);
    GxTx := GxVxb - GxWxb * GxSx;
    GxTy := GxVyb - GxWyb * GxSy;
    end;
  end {Window};

procedure Viewport (xb, yb, xt, yt: integer);

  {define the user's viewport boundaries.  (xb, yb), (xt, yt) are bottom left
   and top right corners respectively.}

  begin
  if xb > xt then viewport (xt, yb, xb, yt) else
  if yb > yt then viewport (xb, yt, xt, yb) else
  if xb = xt then viewport (0, yb, GxMaxCol, yt) else
  if yb = yt then viewport (xb, 0, xt, GxMaxRow) else
  if xb < 0  then viewport (0, yb, xt, yt)  else
  if xt > GxMaxCol then viewport (xb, yb, GxMaxCol, yt) else
  if yb < 0  then viewport (xb, 0, xt, yt)  else
  if yt > GxMaxRow then viewport (xb, yb, xt, GxMaxRow) else
    begin
    yb := GxMaxRow - yb; yt := GxMaxRow - yt;    {invert IBM's y-direction, that
    is,
                                    shift origin to lower left from top left}
    GxVxb := xb;  GxVxt := xt;
    GxVyb := yb;  GxVyt := yt;
    GxSx := (GxVxt - GxVxb) / (GxWxt - GxWxb);
    GxSy := (GxVyt - GxVyb) / (GxWyt - GxWyb);
    GxTx := GxVxb - GxWxb * GxSx;
    GxTy := GxVyb - GxWyb * GxSy;
    Border (GxBorderIndex);
    end;
  end {Viewport};

procedure GraphicsOpen;
  var
    colour: integer;
    k: integer;

  begin
  with GxResult do
      ax := GxEga;        {ah = 0, al = $10;  640x350, 80x25, colour}
                          {ah = 0, al = $12;  640x480, PS/2 colour}
  intr ($10, GxResult);

  GxPage := 0;
  GxBeamX := 0.0;  GxBeamY := 0.0;
  GxClip2d := true;

    {write mode - overwrite}

  GxMode := GxSet;
  Port [$03CE] := 3;   {select data rotate and function register...}
  Port [$03CF] := 0;   {... and set it to 'No Change' mode}

    {colours}

  DefaultMap;
  GxIndex       := GxIndexRng;          {White lines}
  GxBackGround  := GxPalette [0];       {black background}
  GxText        := GxIndexRng;          {white text}
  GxTextBack    := GxBackground;        {black background}
  GxBorderIndex := GxBackground;        {do not draw viewport boundary}

    {windows and viewports}

  window (0.0, 0.0, 1.0, 1.0);
  viewport (0, GxIndent, GxMaxCol-GxIndent, GxMaxRow);

  {Initialize segments}

  GxSegCount:=   0;              {total no. of defined segments}
  GxSegNo   :=  -1;              {currently open segment no. (-1 = none)}
  GxCurrSeg := nil;
  GxSegOpen := false;
  GxPivotX  := 0.0;  GxPivotY := 0.0;
  for k := 1 to GxSegNoMax do
    GxSegment [k] := nil;
  GxPriority      := 0;          {default priority for next segment to be
                                  opened}
  GxVisibility    := true;       {default visibility is VISIBLE for next
                                  segment to be opened}

  end {GraphicsOpen};

procedure GraphicsClose;
  begin
  Port [$03CE] := 3;   {select data rotate and function register...}
  Port [$03CF] := 0;   {... and set it to 'No Change' mode}
  with GxResult do
    begin
    ax := GxC80;
    intr ($10, GxResult);
    end;
  end {GraphicsClose};

procedure WriteModeSet;

  {Define the current graphics writing mode to SET or overwrite mode}

  begin
  GxMode := GxSet;
  Port [$03CE] := 3;   {select data rotate and function register...}
  Port [$03CF] := 0;   {... and set it to overwrite mode}
  end {WriteModeSet};

procedure WriteModeXor;

  {Define the current graphics writing mode to XOR or see-through mode}

  begin
  GxMode := GxXor;
  Port [$03CE] := 3;   {select data rotate and function register...}
  Port [$03CF] := $18;   {... and set it to XOR mode}
  end {WriteModeXor};

procedure Transform (x, y, xscale, yscale, theta: real;
                     var x1, y1: real);

  {Scale, and rotate (x, y) theta radians clockwise about origin, returning new
   coordinates in (xpos, ypos)}

  var
    SinTheta, CosTheta: real;
    xa, ya:   real;
  begin
    SinTheta := sin (theta);
    CosTheta := cos (theta);

    xa := xscale * x;
    ya := yscale * y;
    x1 := (  xa * CosTheta + ya * SinTheta);
    y1 := (- xa * SinTheta + ya * CosTheta);
  end {Transform};

procedure LineIndex (colour: integer);

{select a colour within the current palette for drawing}

  begin
  if colour in [0..GxIndexRng] then
    GxIndex := colour
  else
    GxIndex := 0;
  if GxSegOpen then
    GxMakeEntry (GxOpLineIndex, colour, 0);
  end {LineIndex};

procedure SetCursor (x, y, colour, background: integer);

  {Move the alpha mode cursor to (x, y) in lower left origin coordinates,
   and set the text and background colours.  A negative value for x will
   leave the x-coordinate of the cursor unchanged, similarly for y.

   'Colour' can be:
       black, blue, green, cyan, red, magenta, brown, lightgray,
       darkgray, lightblue, lightgreen, lightcyan, lightred, lightmagenta,
       yellow, white.  'blink' can be added to colour:  red + blink.

    'Background' can be any of the first eight of the above.

   Setting colour or background parameters to -1 will leave the last setting
   of the parameter(s) unchanged.}

  begin
  if x < 1 then x := WhereX;  if x > 80 then x := 1 + (x mod 80);
  if y < 1 then y := WhereY;  if y > 25 then y := 1 + (y mod 25);

  if colour in [black .. white] then
    GxText := colour;
  TextColor (GxText);
  if background in [black .. lightgray] then
    GxTextBack := background;
  TextBackGround (GxTextBack);
  GoToXY (x, 26 - y);   {convert user's y to IBM's top left origin}
  end {SetCursor};


procedure Alpha (x, y: integer; colour, background: integer);

  {clear the screen and change to text mode 80 x 25 of character colour
   'colour' with background colour 'background', placing the cursor at x, y
   in lower left origin coordinates.  For negative values of x, y the
   corresponding coordinate is left unchanged.

   For permitted colours see 'SetCursor' above.}

  begin
  with GxResult do
    begin
    ax := GxC80;
    intr ($10, GxResult);
    end;

  SetCursor (x, y, colour, background);
  end {Alpha};

procedure Graphics (Pal, Index: integer);

  {clear the screen and change to graphics mode.
   Pal= -1:  select current colour map;
   Pal = 0:  select default colour map;

   Index =  n:  (n in 0..15)  set line colour index to n in palette Pal
   Index = -1;  set line colour index to current value}

  {NOTE: In the CGA package, Pal is used to select a palette, in which case
         -1 <= Pal, Index <= 3}
  var
    reg: integer;
    k: integer;
  begin
  with GxResult do
    ax := GxEga;        {ah = 0, al = $10;  640x350, 80x25, colour}
  intr ($10, GxResult);
  if pal = -1 then
    for reg := 0 to GxIndexRng do
      ColourMap (reg, GxPalette [reg])   {restore user's colour map after
                                          screen reset}
  else
    begin  {restore default colour map}
    for k := 0 to 6 do
      ColourMap (k, k);
    ColourMap (7, 56);  ColourMap (8, 7);
    for k := 1 to 7 do
      ColourMap (k+8, k*9);
    end;
  if Index = -1 then Index := GxIndex;
  LineIndex (Index);
  Border (GxBorderIndex);
  end {graphics};

procedure GraphicsMode;

  {return to graphics mode - ONLY after calling AlphaMode for text output in
   EGA graphics mode.
   This is NOT equivalent to GraphicsOpen.  It will FAIL unpredictably if
   called after SetCursor or Alpha have been invoked!}

  begin
  Port [$03CE] := 3;      {select data rotate and function register...}
  if GxMode = GxSet then
    Port [$03CF] := 0     {... set it to 'No Change' mode}
  else
    Port [$03CF] := $18;  {... set it to 'Xor' mode}
  Port [$03CE] := 5;      {select Mode register and ...}
  Port [$03CF] := 2;      {... set to write mode 2}
  end {GraphicsMode};

procedure AlphaMode;
  {enable text output in EGA Graphics mode.  Note:  GraphicsMode must be
   called after text output completed before any more drawing can done}
  begin
  Port [$03CE] := 5;   {select Mode register and ...}
  Port [$03CF] := 0;   {... set to write mode 0}
  end {AlphaMode};

  { BASIC 2-D DRAWING ROUTINES }

procedure ClipOn2d;
  {Apply 2D clipping to all drawing operations using the current window.  If
   it is known that ALL of the drawing will be in the window then it is much
   faster to turn clipping off - see ClipOff2d below.}
  begin
  GxClip2d := true;
  end {ClipOn2d};

procedure ClipOff2d;
  {Turn off 2D clipping for all drawing operations. If it is known that ALL
   of the drawing will be in the window then it will be done much faster if
   clipping is turned off.}
  begin
  GxClip2d := false;
  end {ClipOn2d};

procedure MoveTo (xb, yb: real);

  {Move the current beam position to (xb, yb) in the current GxIndex colour.
   If a segment is currently open then add the move to the segment.}

  var
    lin: GxLine;

  begin
  GxMoveTo (xb, yb);
  if GxSegOpen then
    with GxCurrSeg^ do
      GxMakeEntry (GxOpMoveTo, xb - PivotX, yb - PivotY);
  end {MoveTo};

procedure DrawTo (xb, yb: real);

  {draw a line from current beam position to (xb, yb) in the current GxIndex
   colour, using the current GxMode of writing.
   If a segment is currently open then add the line to the segment, and draw
   to the screen if segment is marked visible.  Draw to the screen if no
   segment currently open.}

  var
    lin: GxLine;

  begin
  if not GxSegOpen then
    GxDrawTo (xb, yb)
  else
    with GxCurrSeg^ do
    begin
    if visible then
      GxDrawTo (xb, yb)
    else
      GxMoveTo (xb, yb);

    GxMakeEntry (GxOpDrawTo, xb - PivotX, yb - PivotY);
    end;
  end {DrawTo};

procedure PlotAt (xb, yb: real);

  {Plot a point at (xb, yb) in the current GxIndex colour, using the current
   GxMode of writing.
   If a segment is currently open then add the point to the segment, and draw
   to the screen if segment is marked visible.  Draw to the screen if no
   segment currently open.}

  var
    lin: GxLine;

  begin
  if not GxSegOpen then
    GxPlotAt (xb, yb)
  else
    with GxCurrSeg^ do
    begin
    if visible then
      GxPlotAt (xb, yb)
    else
      GxMoveTo (xb, yb);

    GxMakeEntry (GxOpPlotAt, xb - PivotX, yb - PivotY);
    end;
  end {PlotAt};

procedure RelMoveTo (x, y: real);
  begin
  MoveTo (GxBeamX + x, GxBeamY + y);
  end {RelMoveTo};

procedure RelPlotAt (x, y: real);
  begin
  PlotAt (GxBeamX + x, GxBeamY + y);
  end {RelPlotAt};

procedure RelDrawTo (x, y: real);
  begin
  DrawTo (GxBeamX + x, GxBeamY + y);
  end {RelDrawTo};

