
/*
 *	pc2ps
 *
 *	Converts PC-extended text to postscript
 *	based on a program by Stephen Frede, UNSW, Australia
 *	written by Roland Giersig, ATI, Austria
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define	TRUE		1
#define	FALSE		0
#define	INCH		72.0		/* no. postscript units / inch */
#define	CM		28.35		/* postscript units / cm */
#define NEW_PAGE	014		/* ^L forces a new page */

#define OPAGEOFFSET	(1.0*CM)	/* default hor. offset if option used */
#define DTOPMARGIN	(0.5*CM)	/* default top margin */
#define OTOPMARGIN	(1.5*CM)	/* default top margin if option used */
#define	DFONTSIZE	10.0		/* default font size (in points) */
#define	OFONTSIZE	12.0		/* default font size if option used */
#define CHARSPERLINE	10000		/* no. of chars/line */
#define LINESPERPAGE	10000		/* no. of lines/page */
#define	ROTATION	0.0		/* default orientation */
#define FONT		"PCfont"
#define	TABSIZE		8



FILE		*ostr;

char  usage[] = "\
Usage: PC2PS [options] Files... > PostscriptPrinterPort (ie. LPT1:)\n\
Converts PC-extended ASCII text from Files to Postscript on STDOUT. Options:\n\
 -R[angle]... rotate output by 90 or [angle]. Default 0.\n\
 -S[size].... set fontsize to 12 points or [size]. Default 10 points.\n\
 -O[offset].. set left margin to 1cm or [offset]. Default 0.\n\
 -OE[offset]. set left margin on even pages to 1cm or [offset]. Default 0.\n\
 -M[top]..... set top margin to 1.5cm or [top]. Default 0.5cm.\n\
 -W[width]... set number of chars per line to 72 or [width].\n\
 -L[length].. set lines per page to 60 or [length].\n\
 -I.......... accept input from STDIN.\n\
 -?.......... print this help message.\n";

char  name[] = "\
PC2PS: Converts Extended ASCII to Postscript.\n\
       Version 1.1, 05-05-91, by Roland Giersig\n\n";


int	tabsize,charsperline,  /* tab spacing in # chars, # chars per line */
	linesperpage;          /* # lines per page */

void process(FILE*);
void init(double, double, double, double, double, double, char*, double);
void pch(int);

void main(int argc,char **argv)
{
	int	status = 0;  /* exit status (no. errors occured) */
	int	iflag = 0, downloadflag = 0;
	double	opageoffset,
		epageoffset,
		topmargin,
		fontsize,
		linepitch,
		spacing,
		rotation;
	char	*fontname;
	FILE	*istr;

	fontsize = DFONTSIZE;
	linepitch = 0.0;
	opageoffset = 0.0;
	epageoffset = 0.0;
	topmargin = DTOPMARGIN;
	spacing = 0.0;
	tabsize = TABSIZE;
	charsperline = CHARSPERLINE;
	linesperpage = LINESPERPAGE;
	rotation = ROTATION;
	fontname = FONT;
	ostr = stdout;
	if (argc < 2) status++;
	argv++;		/* skip program name */
	while(*argv && (**argv == '-' || **argv == '/'))
	{
		char	c;

		(*argv)++;	/* skip the '-' */
		c = (char)tolower(**argv);  /* option letter */
		(*argv)++;	/* skip option letter */
		switch(c)
		{
			case 'o':	/* offset */
				if (**argv != 'e') {
					if(**argv == '\0')
						opageoffset = OPAGEOFFSET;
					else
						opageoffset = atof(*argv) * CM;
					epageoffset = opageoffset;
					}
				else {
					(*argv)++;
					if (**argv == '\0')
						epageoffset = OPAGEOFFSET;
					else 
						epageoffset = atof(*argv) * CM;
					}
				break;

			case 'm':	/* top margin */
				if(**argv == '\0')
					topmargin = OTOPMARGIN;
				else
					topmargin = atof(*argv) * CM;
				break;

			case 'r':	/* rotation */
				if(**argv == '\0')
					rotation = 90.0;
				else
					rotation = atof(*argv);
				break;

			case 's':	/* font size */
				if(**argv == '\0')
					fontsize = OFONTSIZE;
				else
					fontsize = atof(*argv);
				break;

			case 't':	/* tab size */
				if(**argv == '\0')
					tabsize = 4;
				else
					tabsize = (int) atof(*argv);
				break;

			case 'w':	/* linewidth (chars per line) */
				if(**argv == '\0')
					charsperline = 72;
				else
					charsperline = (int) atof(*argv);
				break;

			case 'l':	/* lines per page) */
				if(**argv == '\0')
					linesperpage = 60;
				else
					linesperpage = (int) atof(*argv);
				break;

			case 'i': iflag++;
				break;

			case '?':	/* usage - options */
				fprintf(stderr, usage);
				exit(status);
				break;

			default:
		                (*argv)--; (*argv)--;
				fprintf(stderr, "Illegal option: [%s]\n",
					*argv);
		                (*argv)++; (*argv)++;
				status++;
				break;
		} /* end switch */
	    argv++;
	} /* end while */
	fprintf(stderr, name);
	if(status)
	{
		fprintf(stderr, usage);
		exit(status);
		/* NOTREACHED */
	}
	if(linepitch == 0)
		linepitch = fontsize + 2;
	spacing *= fontsize;
	init(fontsize, opageoffset, epageoffset, topmargin, linepitch,
	   rotation, fontname, spacing);
	if(! *argv) {
	    if (iflag)
		process(stdin);
	    else {
		fprintf(stderr, usage);
		exit(status);
	    }
	}
	else while(*argv)
	{
		if((istr = fopen(*argv, "r")) == NULL)
		{
			perror(*argv);
			status++;
		}
		else
		{
			fprintf(stderr, "%%%% Processing File %s ...", *argv);
			process(istr);
			fclose(istr);
		}
		argv++;
	}
	putc('\004', ostr);
	exit(status);
}



void process(FILE *istr)
{
	register char	ch;
	register int	x,	/* used for tab calculations and folding */
		 	linenum;

        linenum = 1;
	x = 0;
	putc('(', ostr);
	while((ch=(char)getc(istr)) != EOF && ch != '\x1A')
	{

		if(ch == '\t')
		{
			int	n = x + tabsize - (x % tabsize);

			while(x < n) {
					pch(' ');
					x++;
				}
		}
		else if(ch == '\n')
		{
			fprintf(ostr, ") n\n");
			linenum = linenum + 1;
			if (linenum > linesperpage) {
				fprintf(ostr,"p\n");
				linenum = 1;
			}
			x = 0;
			putc('(', ostr);
		}
		else if(ch == '\r')
		{
			fprintf(ostr, ") r\n");
			x = 0;
			putc('(', ostr);
		}
		else if(ch == NEW_PAGE)
		{
			fprintf(ostr, ") n p\n");
			linenum = 1;
			x = 0;
			putc('(', ostr);
		}
		else
		{
			if (x >= charsperline) {
				fprintf(ostr, ") n\n");
				linenum = linenum + 1;
				if (linenum > linesperpage) {
					fprintf(ostr,"p\n");
					linenum = 1;
				}
				x = 0;
				putc('(', ostr);
			}
			pch(ch);
			x++;
		}
	}
	fprintf(ostr, ") n p\n\f");
}

void pch(ch)
int	ch;
{
	if(ch < ' ' || ch > '~')
		fprintf(ostr, "\\%3.3o", (unsigned char)ch);
	else
	{
		if(ch == '(' || ch == ')' || ch == '\\')
			putc('\\', ostr);
		putc(ch, ostr);
	}
}

char	*inittab[] = {

	/* print a page and start a new one */

	"/p",
	"{ 0.0 coffset sub 0 translate",
        "  /temp coffset def",
        "  /coffset noffset def",
	"  /noffset temp def",
	"  coffset 0 translate",
	"  copypage erasepage newpath 0 pgtop moveto",
	"} def",

	"/n",
	"{ spacing 0 3 -1 roll ashow",	  /* show the string given as an arg */
	"  0 linepitch rmoveto",	  /* now move down a line; linepitch is -'ve */
	"  /y currentpoint exch pop def", /* save the new y posn */
	"  y 0 lt",			  /* test if the next line would be below the bottom margin */
	"  { p }",			  /* if so, print the page, and move to the top of a new page */
	"  { 0 y moveto } ifelse",	  /* else go to where the next line is due to start */
	"} def",

	"/r",
	"{ spacing 0 3 -1 roll ashow",	  /* show the string given as an arg */
	"  /y currentpoint exch pop def", /* save y */
	"  0 y moveto", 		  /* and then move to the beginning of the current line */
	"} def",
	(char *)0 };

/* PCfont uses about 36k */

char * initPCfont[] = {

"FontDirectory /PCfont known {stop} if",
"serverdict begin 0 exitserver",
"23 dict dup begin /FontMatrix [.001 0 0 .00112 0 0]def /FontType 3 def /FontBBox [0 -200 600 800]def /Encoding 256 array def /Notice (PCfont 1991) def /FullName (PCfont)def /FontName /PCfont def",
"/isFixedPitch true def Encoding dup 000 /null put dup 001 /Wsmiley put dup 002 /Bsmiley put dup 003 /heart put dup 004 /diamond put dup 005 /club put dup 006 /spade put dup 007 /Bbullet put dup 008 /Wbullet put dup 009",
"/Bcircle put dup 010 /Wcircle put dup 011 /male put dup 012 /female put dup 013 /quarternote put dup 014 /sixteenthnote put dup 015 /sun put dup 016 /pointerright put dup 017 /pointerleft put dup 018 /Varrowboth put dup 019",
"/exclamdbl put dup 020 /paragraph put dup 021 /section put dup 022 /cursorblock put dup 023 /floor put dup 024 /arrowup put dup 025 /arrowdown put dup 026 /arrowright put dup 027 /arrowleft put dup 028 /smallLLsingle put dup 029",
"/arrowboth put dup 030 /pointerup put dup 031 /pointerdown put dup 032 /blank put dup 033 /exclam put dup 034 /quotedbl put dup 035 /numbersign put dup 036 /dollar put dup 037 /percent put dup 038 /ampersand put dup 039",
"/quotesingle put dup 040 /parenleft put dup 041 /parenright put dup 042 /asterisk put dup 043 /plus put dup 044 /comma put dup 045 /hyphen put dup 046 /period put dup 047 /slash put dup 048 /zero put dup 049",
"/one put dup 050 /two put dup 051 /three put dup 052 /four put dup 053 /five put dup 054 /six put dup 055 /seven put dup 056 /eight put dup 057 /nine put dup 058 /colon put dup 059 /semicolon put dup 060 /less put dup 061",
"/equal put dup 062 /greater put dup 063 /question put dup 064 /at put dup 065 /A put dup 066 /B put dup 067 /C put dup 068 /D put dup 069 /E put dup 070 /F put dup 071 /G put dup 072 /H put dup 073 /I put dup 074",
"/J put dup 075 /K put dup 076 /L put dup 077 /M put dup 078 /N put dup 079 /O put dup 080 /P put dup 081 /Q put dup 082 /R put dup 083 /S put dup 084 /T put dup 085 /U put dup 086 /V put dup 087 /W put dup 088",
"/X put dup 089 /Y put dup 090 /Z put dup 091 /bracketleft put dup 092 /backslash put dup 093 /bracketright put dup 094 /asciicircum put dup 095 /underscore put dup 096 /quoteleft put dup 097 /a put dup 098 /b put dup 099",
"/c put dup 100 /d put dup 101 /e put dup 102 /f put dup 103 /g put dup 104 /h put dup 105 /i put dup 106 /j put dup 107 /k put dup 108 /l put dup 109 /m put dup 110 /n put dup 111 /o put dup 112 /p put dup 113 /q put dup 114",
"/r put dup 115 /s put dup 116 /t put dup 117 /u put dup 118 /v put dup 119 /w put dup 120 /x put dup 121 /y put dup 122 /z put dup 123 /braceleft put dup 124 /brokenbar put dup 125 /braceright put dup",
"126 /asciitilde put dup 127 /Delta put dup 128 /Ccedilla put dup 129 /udieresis put dup 130 /eacute put dup 131 /acircumflex put dup 132 /adieresis put dup 133 /agrave put dup 134 /aring put dup 135 /ccedilla put dup",
"136 /ecircumflex put dup 137 /edieresis put dup 138 /egrave put dup 139 /idieresis put dup 140 /icircumflex put dup 141 /igrave put dup 142 /Adieresis put dup 143 /Aring put dup 144 /Eacute put dup 145 /ae put",
"dup 146 /AE put dup 147 /ocircumflex put dup 148 /odieresis put dup 149 /ograve put dup 150 /ucircumflex put dup 151 /ugrave put dup 152 /ydieresis put dup 153 /Odieresis put dup 154 /Udieresis put dup 155 /cent put dup",
"156 /sterling put dup 157 /yen put dup 158 /point put dup 159 /integral put dup 160 /aacute put dup 161 /iacute put dup 162 /oacute put dup 163 /uacute put dup 164 /ntilde put dup 165 /Ntilde put dup 166 /aunder put dup",
"167 /ounder put dup 168 /questiondown put dup 169 /smallULsingle put dup 170 /smallURsingle put dup 171 /half put dup 172 /quarter put dup 173 /exclamdown put dup 174 /lessdbl put dup 175 /greaterdbl put dup",
"176 /lightbox put dup 177 /mediumbox put dup 178 /darkbox put dup 179 /Vsingle put dup 180 /VsingleTleftsingle put dup 181 /VsingleTleftdbl put dup 182 /VdblTleftsingle put dup 183 /VdblURsingle put dup",
"184 /VsingleURdbl put dup 185 /VdblTleftdbl put dup 186 /Vdbl put dup 187 /VdblURdbl put dup 188 /VdblLRdbl put dup 189 /VdblLRsingle put dup 190 /VsingleLRdbl put dup 191 /VsingleURsingle put dup",
"192 /VsingleLLsingle put dup 193 /HsingleTupsingle put dup 194 /HsingleTdownsingle put dup 195 /VsingleTrightsingle put dup 196 /Hsingle put dup 197 /VsingleXsingle put dup 198 /VsingleTrightdbl put dup 199 /VdblTrightsingle put dup",
"200 /VdblLLdbl put dup 201 /VdblULdbl put dup 202 /HdblTupdbl put dup 203 /HdblTdowndbl put dup 204 /VdblTrightdbl put dup 205 /Hdbl put dup 206 /VdblXdbl put dup 207 /HdblTupsingle put dup 208 /HsingleTupdbl put dup",
"209 /HdblTdownsingle put dup 210 /HsingleTdowndbl put dup 211 /VdblLLsingle put dup 212 /VsingleLLdbl put dup 213 /VsingleULdbl put dup 214 /VdblULsingle put dup 215 /VdblXsingle put dup 216 /VsingleXdbl put dup",
"217 /VsingleLRsingle put dup 218 /VsingleULsingle put dup 219 /allblack put dup 220 /botblack put dup 221 /leftblack put dup 222 /rightblack put dup 223 /topblack put dup 224 /alpha put dup 225 /beta put dup 226 /Gamma put dup",
"227 /pi put dup 228 /Sigma put dup 229 /sigma put dup 230 /mu put dup 231 /tau put dup 232 /Phi put dup 233 /theta put dup 234 /Omega put dup 235 /delta put dup 236 /infinity put dup 237 /phi put dup 238 /element put dup",
"239 /intersection put dup 240 /equivalence put dup 241 /plusminus put dup 242 /greaterequal put dup 243 /lessequal put dup 244 /integraltp put dup 245 /integralbt put dup 246 /divide put dup 247 /approxequal put dup",
"248 /degree put dup 249 /bullet put dup 250 /dotmath put dup 251 /radical put dup 252 /eta put dup 253 /squared put dup 254 /block put 255 /blank put /_uC { 0 0 moveto /Courier findfont 1000 scalefont setfont show } def",
"/_uS { 0 0 moveto /Symbol findfont 1000 scalefont setfont show } def /dotless_i { newpath 110 20 moveto 490 20 lineto 300 20 moveto 300 400 lineto 150 400 lineto 22 setlinewidth 1 setlinecap stroke } def /squash { 1 dict begin",
"/cr exch def save 1 .75 scale cr _uC restore end } def /lt {lineto} def /mt {moveto} def /a {arc} def /an {arcn} def /cp {closepath} def /slw {setlinewidth} def /slc {setlinecap} def /CharacterDefs 255 dict def",
"CharacterDefs begin /null { 0 598 mt 230 598 lt 184 460 lt 0 460 lt cp 92 0 mt 92 184 lt 230 184 lt 368 598 lt 598 598 lt 598 460 lt 377.2 460 lt 230 0 lt cp fill } def /Wsmiley { 300 300 275 0 360 a 370 230",
"mt 300 300 100 315 225 an 20 slw 1 slc stroke 230 370 25 0 360 a fill 370 370 25 0 360 a fill } def /Bsmiley { 300 300 275 0 360 a 352.5 247.5 mt 370 230 25 135 315 an 300 300 125 315 225 an 230 230 25 225 45 an",
"300 300 75 225 315 a 230 335 mt 230 370 50 270 -90 an 370 370 50 270 -90 an eofill } def /heart {(\\251) _uS} def /diamond {(\\250) _uS} def /club {(\\247) _uS} def /spade {(\\252) _uS} def /Bbullet { (\\267) _uC } def",
"/Wbullet { 0 -200 mt 600 -200 lt 600 800 lt 0 800 lt 0 -200 lt 300 300 100 225 585 a eofill } def /Bcircle { 300 300 100 225 585 a 20 slw stroke } def /Wcircle { 0 -200 mt 600 -200 lt 600 800 lt 0 800 lt 0 -200",
"lt 300 300 125 225 585 a 300 300 75 225 585 a eofill } def /male { newpath 300 200 200 70 -290 arcn 575 625 lineto 575 500 moveto 575 625 lineto 450 625 lineto 20 slw 1 setlinecap stroke } def",
"/female { newpath 300 400 200 270 -90 arcn 300 0 lineto 150 100 moveto 450 100 lineto 20 slw 1 setlinecap stroke } def /quarternote { save 1 .5 scale 200 400 100 0 360 arc fill restore 290 200",
"moveto 290 658 lineto 430 840 230 235 270 arc 20 slw 0 setlinecap 290 610 lineto gsave fill grestore stroke } def /sixteenthnote { save 1 .5 scale 200 400 100 0 360 arc 330 200 moveto 340 200 100 0 360 arc fill restore",
"290 200 moveto 290 658 lineto 430 840 230 235 270 arc 430 610 moveto 430 790 230 270 235 arcn 430 610 moveto 430 100 lineto 20 slw 1 setlinecap stroke } def /sun { newpath 300 300 150 270 -90 arcn 300 0 lineto",
"180 180 moveto 100 100 lineto 150 300 moveto 25 300 lineto 180 420 moveto 100 500 lineto 300 450 moveto 300 600 lineto 420 420 moveto 500 500 lineto 450 300 moveto 575 300 lineto 420 180 moveto 500 100 lineto",
"20 slw 1 setlinecap stroke } def /pointerright { 600 750 450 270 210 an 600 -150 450 150 90 an fill } def /pointerleft { 150 750 450 330 270 an 150 -150 450 90 30 an fill } def /Varrowboth { 300 700 mt 550 500 lt",
"350 500 lt 350 200 lt 550 200 lt 300 0 lt 50 200 lt 250 200 lt 250 500 lt 50 500 lt cp fill } def /exclamdbl { save -100 0 translate (!) _uC 200 0 translate (!) _uC restore } def /paragraph { (\\266) _uC }",
"def /section { (\\247) _uC } def /cursorblock { 50 -100 mt 550 -100 lt 550 200 lt 50 200 lt cp fill } def /floor { 300 700 mt 550 500 lt 350 500 lt 350 200 lt 550 200 lt 300 0 lt 50 200 lt 250 200 lt 250 500 lt",
"50 500 lt cp fill 0 0 mt 600 0 lt 20 slw stroke } def /arrowup { 300 700 mt 550 500 lt 350 500 lt 350 0 lt 250 0 lt 250 500 lt 50 500 lt cp fill } def /arrowdown { 300 0 mt 550 200 lt 350 200 lt 350 700 lt 250 700 lt",
"250 200 lt 50 200 lt cp fill } def /arrowright { 50 300 mt 550 300 lt 300 400 lt 550 300 mt 300 200 lt 1 slc 25 slw stroke } def /arrowleft { 550 300 mt 50 300 lt 300 400 lt 50 300 mt 300 200 lt 1 slc 25 slw stroke } def",
"/smallLLsingle { 100 400 mt 100 275 lt 500 275 lt 20 slw stroke } def /arrowboth { 550 300 mt 50 300 lt 200 400 lt 50 300 mt 200 200 lt 550 300 mt 400 400 lt 550 300 mt 400 200 lt 1 slc 20 slw stroke } def",
"/pointerup { 50 50 mt 300 600 lt 550 50 lt cp fill } def /pointerdown { 0 550 mt 300 50 lt 550 550 lt cp fill } def /blank {} def /exclam { (!) _uC } def /quotedbl { (\") _uC } def /numbersign { (#) _uC } def",
"/dollar { ($) _uC } def /percent { (%) _uC } def",
"/ampersand {(&) _uC} def /quotesingle {(\') _uC} def /parenleft { (\\050) _uC } def /parenright { (\\051) _uC } def /asterisk { (*) _uC } def /plus { (+) _uC } def",
"/comma { (,) _uC } def /hyphen { (-) _uC } def /period { (.) _uC } def /slash { (/) _uC } def /zero { (0) _uC } def /one { (1) _uC } def /two { (2) _uC } def /three { (3) _uC } def /four { (4) _uC } def",
"/five { (5) _uC } def /six { (6) _uC } def /seven { (7) _uC } def /eight { (8) _uC } def /nine { (9) _uC } def /colon { (:) _uC } def /semicolon { (;) _uC } def /less { (<) _uC } def /equal { (=) _uC } def",
"/greater { (>) _uC } def /question { (?) _uC } def /at { (@) _uC } def /A { (A) _uC } def /B { (B) _uC } def /C { (C) _uC } def /D { (D) _uC } def /E { (E) _uC } def /F { (F) _uC } def /G { (G) _uC } def",
"/H { (H) _uC } def /I { (I) _uC } def /J { (J) _uC } def /K { (K) _uC } def /L { (L) _uC } def /M { (M) _uC } def /N { (N) _uC } def /O { (O) _uC } def /P { (P) _uC } def /Q { (Q) _uC } def /R { (R) _uC } def",
"/S { (S) _uC } def /T { (T) _uC } def /U { (U) _uC } def /V { (V) _uC } def /W { (W) _uC } def /X { (X) _uC } def /Y { (Y) _uC } def /Z { (Z) _uC } def /bracketleft { ([) _uC } def /backslash { (\\\\) _uC } def",
"/bracketright { (]) _uC } def /asciicircum { (^) _uC } def /underscore { 0 -200 mt 600 -200 lt 20 slw stroke } def /quoteleft { (`) _uC } def /a { (a) _uC } def /b { (b) _uC } def /c { (c) _uC } def /d { (d) _uC }",
"def /e { (e) _uC } def /f { (f) _uC } def /g { (g) _uC } def /h { (h) _uC } def /i { (i) _uC } def /j { (j) _uC } def /k { (k) _uC } def /l { (l) _uC } def /m { (m) _uC } def /n { (n) _uC } def /o { (o) _uC } def",
"/p { (p) _uC } def /q { (q) _uC } def /r { (r) _uC } def /s { (s) _uC } def /t { (t) _uC } def /u { (u) _uC } def /v { (v) _uC } def /w { (w) _uC } def /x { (x) _uC } def /y { (y) _uC } def /z { (z) _uC } def",
"/braceleft { ({) _uC } def /brokenbar { (|) _uC } def /braceright { (}) _uC } def /asciitilde { (~) _uC } def /Delta { 50 0 mt 50 200 lt 300 500 lt 550 200 lt 550 0 lt cp 0 setlinejoin 20 slw stroke } def",
"/Ccedilla { (C) _uC (\\313) _uC } def /udieresis { (u) _uC (\\310) _uC } def /eacute { (e) _uC (\\302) _uC } def /acircumflex { (a) _uC (\\303) _uC } def /adieresis { (a) _uC (\\310) _uC } def /agrave { (a) _uC (\\301) _uC } def",
"/aring { (a) _uC (\\312) _uC } def /ccedilla { (c) _uC (\\313) _uC } def /ecircumflex { (e) _uC (\\303) _uC } def /edieresis { (e) _uC (\\310) _uC } def /egrave { (e) _uC (\\301) _uC } def /idieresis { dotless_i (\\310) _uC } def",
"/icircumflex { dotless_i (\\303) _uC } def /igrave { dotless_i (\\301) _uC } def /Adieresis { (A) squash (\\310) _uC } def /Aring { (A) squash (\\312) _uC } def /Eacute { (E) squash (\\302) _uC } def",
"/ae { save .6 1 scale -75 0 translate (a) _uC 360 0 translate (e) _uC restore } def /AE { 120 540 moveto 480 540 lineto 480 420 lineto 300 540 moveto 300 20 lineto 250 20 moveto 500 20 lineto 500 160 lineto 250 540 moveto",
"80 20 lineto 30 20 moveto 180 20 lineto 170 280 moveto 400 280 lineto 400 240 moveto 400 320 lineto 20 slw 1 setlinecap stroke } def /ocircumflex { (o) _uC (\\303) _uC } def /odieresis { (o) _uC (\\310) _uC } def",
"/ograve { (o) _uC (\\301) _uC } def /ucircumflex { (u) _uC (\\303) _uC } def /ugrave { (u) _uC (\\301) _uC } def /ydieresis { (y) _uC (\\310) _uC } def /Odieresis { (O) squash (\\310) _uC } def /Udieresis { (U) squash (\\310) _uC } def",
"/cent { (\\242) _uC } def /sterling { (\\243) _uC } def /yen { (\\245) _uC } def /point { save .6 1 scale (P) _uC .5 .5 scale 1000 0 translate (t) _uC restore } def /integral { save 1 .75 scale (\\362) _uS restore } def",
"/aacute { (a) _uC (\\302) _uC } def /iacute { dotless_i (\\302) _uC } def /oacute { (o) _uC (\\302) _uC } def /uacute { (u) _uC (\\302) _uC } def /ntilde { (n) _uC (\\304) _uC } def /Ntilde { (N) squash (\\304) _uC } def",
"/aunder { save 175 200 translate .6 .6 scale (a) _uC 100 -200 mt 500 -200 lt 20 slw stroke restore } def /ounder { save 175 200 translate .6 .6 scale (o) _uC 100 -200 mt 500 -200 lt 20 slw stroke restore } def",
"/questiondown { (\\277) _uC } def /smallULsingle { 25 100 mt 25 300 lt 575 300 lt 20 slw stroke } def /smallURsingle { 25 300 mt 575 300 lt 575 100 lt 20 slw stroke } def /half { save .5 .5 scale 0 600 translate (1) _uC",
"500 -600 translate (2) _uC restore 50 0 moveto 500 600 lineto 10 slw 1 setlinecap stroke } def /quarter { save .5 .5 scale 0 600 translate (1) _uC 500 -600 translate (4) _uC restore 50 0 moveto 500 600 lineto",
"10 slw 1 setlinecap stroke } def /exclamdown { (\\241) _uC } def /lessdbl { (\\253) _uC } def /greaterdbl { (\\273) _uC } def /lightbox { 2 dict begin 50 85 600 { /col exch def -150 71 800 { /row exch def col row",
"5 0 360 arc fill } for } for end } def /mediumbox { 2 dict begin 50 85 600 { /col exch def -150 71 800 { /row exch def col row 15 0 360 arc fill } for } for end } def /darkbox { 2 dict begin 50 85 600 { /col exch def",
"-150 71 800 { /row exch def col row 25 0 360 arc fill } for } for end } def /Vsingle { 300 -205 mt 300 805 lt 20 slw stroke } def /VsingleTleftsingle { 300 -205 mt 300 805 lt 0 300 mt 300 300 lt 20 slw stroke } def",
"/VsingleTleftdbl { 300 -205 mt 300 805 lt 0 400 mt 300 400 lt 0 200 mt 300 200 lt 20 slw stroke } def /VdblTleftsingle { 200 -205 mt 200 805 lt 400 -205 mt 400 805 lt 0 300 mt 200 300 lt 20 slw stroke } def",
"/VdblURsingle { 0 300 mt 400 300 lt 400 -205 lt 200 300 mt 200 -205 lt 20 slw stroke } def /VsingleURdbl { 300 -205 mt 300 400 lt 0 400 lt 0 200 mt 300 200 lt 20 slw stroke } def",
"/VdblTleftdbl { 400 -205 mt 400 805 lt 200 -205 mt 200 200 lt 0 200 lt 0 400 mt 200 400 lt 200 805 lt 20 slw stroke } def /Vdbl { 400 -205 mt 400 805 lt 200 -205 mt 200 805 lt 20 slw stroke } def",
"/VdblURdbl { 400 -205 mt 400 400 lt 0 400 lt 0 200 mt 200 200 lt 200 -205 lt 20 slw stroke } def /VdblLRdbl { 200 805 mt 200 400 lt 0 400 lt 400 805 mt 400 200 lt 0 200 lt 20 slw stroke } def",
"/VdblLRsingle { 200 805 mt 200 300 lt 400 805 mt 400 300 lt 0 300 lt 20 slw stroke } def /VsingleLRdbl { 300 805 mt 300 200 lt 0 200 lt 0 400 mt 300 400 lt 20 slw stroke } def",
"/VsingleURsingle { 300 -205 mt 300 300 lt 0 300 lt 20 slw stroke } def /VsingleLLsingle { 300 805 mt 300 300 lt 600 300 lt 20 slw stroke } def /HsingleTupsingle { 0 300 mt 600 300 lt 300 300 mt 300 805 lt 20 slw stroke } def",
"/HsingleTdownsingle { 0 300 mt 600 300 lt 300 300 mt 300 -205 lt 20 slw stroke } def /VsingleTrightsingle { 300 -205 mt 300 805 lt 300 300 mt 600 300 lt 20 slw stroke } def /Hsingle { 0 300 mt 600 300 lt 20 slw stroke } def",
"/VsingleXsingle { 300 -205 mt 300 805 lt 0 300 mt 600 300 lt 20 slw stroke } def /VsingleTrightdbl { 300 -205 mt 300 805 lt 300 200 mt 600 200 lt 300 400 mt 600 400 lt 20 slw stroke } def",
"/VdblTrightsingle { 400 -205 mt 400 805 lt 200 -205 mt 200 805 lt 400 300 mt 600 300 lt 20 slw stroke } def /VdblLLdbl { 400 805 mt 400 400 lt 600 400 lt 200 805 mt 200 200 lt 600 200 lt 20 slw stroke } def",
"/VdblULdbl { 400 -205 mt 400 200 lt 600 200 lt 200 -205 mt 200 400 lt 600 400 lt 20 slw stroke } def /HdblTupdbl { 0 200 mt 600 200 lt 0 400 mt 200 400 lt 200 805 lt 400 805 mt 400 400 lt 600 400 lt 20 slw stroke } def",
"/HdblTdowndbl { 0 400 mt 600 400 lt 0 200 mt 200 200 lt 200 -205 lt 400 -205 mt 400 200 lt 600 200 lt 20 slw stroke } def /VdblTrightdbl { 200 -205 mt 200 805 lt 400 -205 mt 400 200 lt 600 200 lt 600 400 mt",
"400 400 lt 400 805 lt 20 slw stroke } def /Hdbl { 0 200 mt 600 200 lt 0 400 mt 600 400 lt 20 slw stroke } def /VdblXdbl { 400 -205 mt 400 200 lt 600 200 lt 600 400 mt 400 400 lt 400 805 lt 200 -205 mt 200 200 lt 0 200 lt",
"0 400 mt 200 400 lt 200 805 lt 20 slw stroke } def /HdblTupsingle { 0 200 mt 600 200 lt 0 400 mt 600 400 lt 300 400 mt 300 805 lt 20 slw stroke } def /HsingleTupdbl { 0 300 mt 600 300 lt 200 300 mt 200 805 lt",
"400 300 mt 400 805 lt 20 slw stroke } def /HdblTdownsingle { 0 200 mt 600 200 lt 0 400 mt 600 400 lt 300 200 mt 300 -205 lt 20 slw stroke } def /HsingleTdowndbl { 0 300 mt 600 300 lt 200 300 mt 200 -205 lt",
"400 300 mt 400 -205 lt 20 slw stroke } def /VdblLLsingle { 200 805 mt 200 300 lt 600 300 lt 400 805 mt 400 300 lt 20 slw stroke } def /VsingleLLdbl { 300 805 mt 300 200 lt 600 200 lt 300 400 mt 600 400 lt 20 slw stroke } def",
"/VsingleULdbl { 300 -205 mt 300 400 lt 600 400 lt 300 200 mt 600 200 lt 20 slw stroke } def /VdblULsingle { 200 -205 mt 200 300 lt 600 300 lt 400 -205 mt 400 300 lt 20 slw stroke } def",
"/VdblXsingle { 400 -205 mt 400 805 lt 200 -205 mt 200 805 lt 0 300 mt 600 300 lt 20 slw stroke } def /VsingleXdbl { 300 -205 mt 300 805 lt 0 200 mt 600 200 lt 0 400 mt 600 400 lt 20 slw stroke } def",
"/VsingleLRsingle { 0 300 mt 300 300 lt 300 805 lt 20 slw stroke } def /VsingleULsingle { 300 -205 mt 300 300 lt 600 300 lt 20 slw stroke } def /allblack { 0 -205 mt 600 -205 lt 600 805 lt 0 805 lt cp fill } def",
"/botblack { 0 -205 mt 600 -205 lt 600 300 lt 0 300 lt cp fill } def /leftblack { 0 -205 mt 300 -205 lt 300 805 lt 0 805 lt cp fill } def /rightblack { 300 -205 mt 600 -205 lt 600 805 lt 300 805 lt cp fill } def",
"/topblack { 0 300 mt 600 300 lt 600 805 lt 0 805 lt cp fill } def /alpha { save .8 .8 scale (\\141) _uS restore } def /beta { save .9 1 scale  (\\373) _uC restore } def",
"/Gamma { save .8 .8 scale (\\107) _uS restore } def /pi { save .8 .8 scale (\\160) _uS restore } def /Sigma { save .8 .8 scale (\\123) _uS restore } def /sigma { save .8 .8 scale (\\163) _uS restore } def",
"/mu { save .8 .8 scale (\\155) _uS restore } def /tau { save .8 .8 scale (\\164) _uS restore } def /Phi { save .75 1 scale (\\106) _uS restore } def /theta { save .8 .8 scale (\\161) _uS restore } def",
"/Omega { save .6 .9 scale (\\127) _uS restore } def /delta { save .8 .8 scale (\\144) _uS restore } def",
"/infinity { save .75 1 scale (\\245) _uS restore } def /phi { save .8 .7 scale (\\146) _uS restore } def /element { save .8 .8 scale (\\316) _uS restore } def",
"/intersection { save .5 .8 scale (\\307) _uS restore } def /equivalence {(\\272) _uS} def /plusminus {(\\261) _uS} def",
"/greaterequal {(\\263) _uS} def /lessequal {(\\243) _uS} def /integraltp { save .75 .75 scale (\\363) _uS restore } def /integralbt { save .75 .75 scale (\\365) _uS restore } def /divide {(\\270) _uS} def",
"/approxequal {(\\273) _uS} def /degree {(\\260) _uS} def /bullet {(\\267) _uS} def /dotmath {(\\327) _uS} def /radical { 100 300 mt 150 350 lt 350 0 lt 550 600 lt 25 slw stroke } def",
"/eta { save .8 .8 scale (\\150) _uS restore } def /squared { save 100 350 translate",
".45 .45 scale (2) _uC restore } def /block { 50 0 mt 50 400 lt 550 400 lt 550 0 lt fill } def /blank {} def end /BuildChar { 0 begin /char exch def /fontdict exch def /charname fontdict /Encoding get char get def",
"fontdict begin 600 0 0 -200 600 800 setcachedevice CharacterDefs charname get exec end end } def /BuildChar load 0 3 dict put /UniqueID 847667 def end /PCfont exch definefont pop",
(char *)0 };


void init(fontsize, opageoffset, epageoffset, topmargin, linepitch, rotation, fontname, spacing)
double	fontsize,
	opageoffset,
	epageoffset,
	topmargin,
	linepitch,
	spacing,
	rotation;
char	*fontname;
{
	register char	**p;
	register char	i;


	fprintf(ostr, "\004\n");

	p = initPCfont;
	while(*p)	      /* download PCfont */
	     fprintf(ostr, "%s\n", *p++);
	fprintf(ostr, "\004\n");

	p = inittab;
	while(*p)
		fprintf(ostr, "%s\n", *p++);
	fprintf(ostr, "/%s findfont %.1f scalefont setfont\n",
		fontname, fontsize);
	fprintf(ostr, "/linepitch %.1f def\n", -linepitch);
	fprintf(ostr, "/spacing %.1f def\n", spacing);
	fprintf(ostr, "/coffset %.1f def\n", opageoffset + 4);
	fprintf(ostr, "/noffset %.1f def\n", epageoffset + 4);
	/* apply rotation transformation, if any */
	if(rotation != 0.0)
		fprintf(ostr, "%.1f rotate\n", rotation);
	/* get current imageable area */
	fprintf(ostr, "clippath pathbbox\n");
	/* save the upper right y coordinate */
	fprintf(ostr, "/pgtop exch def\n");
	/* save lower left y; translate origin to lower left */
	fprintf(ostr, "pop /y exch def y translate\n");
	/* subtract old lower left from upper right to get top of page */
	/* then subtract top margin to set top of page */
	fprintf(ostr, "/pgtop pgtop y sub %.1f sub linepitch add def\n", topmargin);
	/* apply horizontal offset, if any */
	/* unfortunately, a slight fudge factor is required here */
	fprintf(ostr, "coffset 0 translate\n");
	/* move to top of page, ready to start printing */
	fprintf(ostr, "newpath 0 pgtop moveto\n");
}
