//  ____________________________________________________
// |                                                    |
// |  Project:     POWER VIEW INTERFACE                 |
// |  File:        PVEDITOR.CPP                         |
// |  Compiler:    WPP386 (10.6)                        |
// |                                                    |
// |  Subject:     Text editor implementation           |
// |                                                    |
// |  Author:      Emil Dotchevski                      |
// |____________________________________________________|
//
// E-mail: zajo@geocities.com
// URL:    http://www.geocities.com/SiliconValley/Bay/3577

#define uses_fcntl
#define uses_io
#define uses_stdio
#define uses_string
#define uses_app
#define uses_check
#define uses_ctype
#define uses_dc
#define uses_desk
#define uses_dialog
#define uses_errno
#define uses_editor
#define uses_help
#define uses_label
#define uses_lbox
#define uses_icons
#define uses_input
#define uses_stddlg
#define uses_system

#define DECLARE_PVEDITOR
#include "PVuses.h"
#undef DECLARE_PVEDITOR

#define smEXTEND                        0x01
#define smDOUBLE                        0x02

#define sfSEARCH_FAILED                 ((uint)-1)

#ifdef SYNTAXHILIGHT
#define MAX_RESERVED_WORD_LENGTH        16
#else
#define MAX_RESERVED_WORD_LENGTH        0
#endif

/*
STATIC DATA
*/
  static uint first_keys[] =
  {
    50,
    kCTRL_A,      cmeWORD_LEFT,
    kCTRL_C,      cmePAGE_DOWN,
    kCTRL_D,      cmeCHAR_RIGHT,
    kCTRL_E,      cmeLINE_UP,
    kCTRL_F,      cmeWORD_RIGHT,
    kCTRL_G,      cmeDEL_CHAR,
    kCTRL_H,      cmeBACK_SPACE,
    kCTRL_J,      cmeGOTO_LINE,
    kCTRL_K,      0xFF02,
    kCTRL_L,      cmeSEARCH_AGAIN,
    kCTRL_M,      cmeNEW_LINE,
    kCTRL_O,      cmeINDENT_MODE,
    kCTRL_Q,      0xFF01,
    kCTRL_R,      cmePAGE_UP,
    kCTRL_S,      cmeCHAR_LEFT,
    kCTRL_T,      cmeDEL_WORD,
    kALT_BS,      cmUNDO,
    kCTRL_V,      cmeINS_MODE,
    kCTRL_W,      cmeSCROLL_UP,
    kCTRL_X,      cmeLINE_DOWN,
    kCTRL_Y,      cmeDEL_LINE,
    kCTRL_Z,      cmeSCROLL_DOWN,
    kLEFT,        cmeCHAR_LEFT,
    kSHIFT_LEFT,  cmeCHAR_LEFT,
    kRIGHT,       cmeCHAR_RIGHT,
    kSHIFT_RIGHT, cmeCHAR_RIGHT,
    kCTRL_LEFT,   cmeWORD_LEFT,
    kCTRL_RIGHT,  cmeWORD_RIGHT,
    kHOME,        cmeLINE_START,
    kSHIFT_HOME,  cmeLINE_START,
    kEND,         cmeLINE_END,
    kSHIFT_END,   cmeLINE_END,
    kUP,          cmeLINE_UP,
    kSHIFT_UP,    cmeLINE_UP,
    kDOWN,        cmeLINE_DOWN,
    kSHIFT_DOWN,  cmeLINE_DOWN,
    kPG_UP,       cmePAGE_UP,
    kSHIFT_PG_UP, cmePAGE_UP,
    kPG_DN,       cmePAGE_DOWN,
    kSHIFT_PG_DN, cmePAGE_DOWN,
    kCTRL_PG_UP,  cmeTEXT_START,
    kCTRL_PG_DN,  cmeTEXT_END,
    kCTRL_HOME,   cmeSCREEN_TOP,
    kCTRL_END,    cmeSCREEN_BOTTOM,
    kTAB,         cmeTAB,
    kINS,         cmeINS_MODE,
    kDEL,         cmeDEL_CHAR,
    kSHIFT_INS,   cmPASTE,
    kSHIFT_DEL,   cmCUT,
    kCTRL_INS,    cmCOPY
  };
  static uint quick_keys[] =
  {
    20,
    'A',          cmeREPLACE,
    'C',          cmeTEXT_END,
    'D',          cmeLINE_END,
    'E',          cmeSCREEN_TOP,
    'F',          cmeFIND,
    'H',          cmeDEL_START,
    'R',          cmeTEXT_START,
    'S',          cmeLINE_START,
    'X',          cmeSCREEN_BOTTOM,
    'Y',          cmeDEL_END,
    '0',          cmeGMARK0,
    '1',          cmeGMARK1,
    '2',          cmeGMARK2,
    '3',          cmeGMARK3,
    '4',          cmeGMARK4,
    '5',          cmeGMARK5,
    '6',          cmeGMARK6,
    '7',          cmeGMARK7,
    '8',          cmeGMARK8,
    '9',          cmeGMARK9
  };
  static uint block_keys[] =
  {
    18,
    'B',          cmeSTART_SELECT,
    'C',          cmPASTE,
    'H',          cmeHIDE_SELECT,
    'K',          cmCOPY,
    'Y',          cmCUT,
    'I',          cmeINDENT,
    'U',          cmeUNINDENT,
    'S',          cmSAVE,
    '0',          cmeMARK0,
    '1',          cmeMARK1,
    '2',          cmeMARK2,
    '3',          cmeMARK3,
    '4',          cmeMARK4,
    '5',          cmeMARK5,
    '6',          cmeMARK6,
    '7',          cmeMARK7,
    '8',          cmeMARK8,
    '9',          cmeMARK9
  };
  static uint *key_map[] =
  {
    first_keys,
    quick_keys,
    block_keys
  };
#ifdef CPPHILIGHT
 static Tset cpp_symbols;
#endif
#ifdef ASMHILIGHT
 static Tset asm_symbols;
#endif
#ifdef A51HILIGHT
 static Tset a51_symbols;
#endif


/*
STATIC FUNCTIONS
*/
  static int count_lines( char *buf, uint count )
  {
    uint cnt;

    cnt = 0;
    count++;
    while( --count )
      if( *( buf++ ) == '\r' ) cnt++;
    return cnt;
  }

  static uint scan_key_map( uint *key_map, uint key_code )
  {
    uint count, key, cmd;

    count = *( key_map++ ) + 1;
    while( --count )
    {
      key = *( key_map++ );
      cmd = *( key_map++ );
      if( key == key_code ) return cmd;
    }
    return 0;
  }

  static uint scan( char *block, uint size, char *str )
  {
    uint length, counter;

    length = strlen( str );
    if( size >= length )
    {
      counter = size - length + 1;
      while( --counter )
      {
        if( memcmp( block, str, length ) == 0 ) return size - length - counter;
        block++;
      }
    }
    return sfSEARCH_FAILED;
  }

  static uint iscan( char *block, uint size, char *str )
  {
    char s[256], *_s, *_block;
    uint length, _length, counter;

    length = strlen( str );
    if( size >= length )
    {
      strcpy( s, str );
      strupr( s );
#ifdef CYR
      cyr_strupr( s );
#endif
      counter = size - length + 1;
      while( --counter )
      {
        _length = length + 1; _block = block; _s = s;
        while( --_length )
        {
#ifdef CYR
          char c=cyr_toupper(toupper(*(_block++)));
#else
          char c=toupper(*(_block++));
#endif
          if( c!=*(_s++) ) goto differrent;
        }
        return size - length - counter;
      differrent:
        block++;
      }
    }
    return sfSEARCH_FAILED;
  }

#ifdef SYNTAXHILIGHT
  static boolean must_hilight( char *file_name, char *extensions )
  {
    if( extensions!=NULL )
    {
      char ext[_MAX_EXT];
      _splitpath( file_name, NULL, NULL, NULL, ext );
      int l=strlen( ext );
      if( l<2 ) return 0;
      for( char *c=extensions; *c; c++ )
        if( *c=='.' && !memcmp( ext, c, l ) && !isalnum(*(c+l)) )
          return 1;
    }
    return 0;
  }
#endif //SYNTAXHILIGHT


/*
Ttext_editor
*/

//public:

  static init_text_editor( Ttext_editor *text_editor, Teditor *editor )
  {
    text_editor->date = text_editor->time = 0;
    text_editor->file_name = STRDUP("");
    text_editor->booleans = ebCAN_UNDO;
    text_editor->assoc_number = 0;
    text_editor->editor = editor;
    if( text_editor->edit_buffer == NULL )
    {
      text_editor->editor_dialog( edOUT_OF_MEMORY, NULL );
      text_editor->buf_size = 0;
    }
    text_editor->set_buf_len( 0 );
    ~text_editor->valid_chars;
    text_editor->valid_chars << 9;
    for( int i=32; i<256; i++ ) text_editor->valid_chars << i;
  }

  Ttext_editor::Ttext_editor( uint _buf_size, Teditor *_editor )
  {
    buf_size = _buf_size;
    edit_buffer = (char *) MALLOC( buf_size+1 );
    edit_buffer[buf_size]=0;
    init_text_editor( this, _editor );
    booleans |= ( ebDISPOSE_BUFFER + ebCAN_GROW );
  }

  Ttext_editor::Ttext_editor( char *_edit_buffer, uint _buf_size, Teditor *_editor )
  {
    buf_size = _buf_size;
    edit_buffer = _edit_buffer;
    init_text_editor( this, _editor );
  }

  Ttext_editor::~Ttext_editor( void )
  {
    if( booleans & ebDISPOSE_BUFFER ) FREE( edit_buffer );
    FREE( file_name );
    edit_buffer = NULL;
  }

  boolean Ttext_editor::set_buf_size( uint new_size )
  {
    uint n;
    void *new_pos;

    if( !( booleans & ebCAN_GROW ) ) return new_size <= buf_size;
    if( !new_size )
      new_size = 0x1000;
    else
#ifndef __386__
      if( new_size > 0xF000 )
        new_size = 0xFFF0;
      else
#endif
#ifdef __386__
        new_size = ( new_size + 0x0FFF ) & 0xFFFFF000;
#else
        new_size = ( new_size + 0x0FFF ) & 0xF000;
#endif
    if( new_size != buf_size )
    {
      if( new_size > buf_size )
      {
        if( ( new_pos = REALLOC( edit_buffer, new_size+1 ) ) == NULL ) return 0;
        edit_buffer = (char *) new_pos;
        edit_buffer[new_size]=0;
      }
      n = buf_len - cur_ptr + del_count;
      memmove( edit_buffer + new_size - n, edit_buffer + buf_size - n, n );
      buf_size = new_size;
      gap_len = buf_size - buf_len;
    }
    return 1;
  }

  void Ttext_editor::set_buf_len( uint length )
  {
    sel_start     = 0;
    sel_end       = 0;
    cur_after_eol = 0;
    cur_ptr       = 0;
    cur_pos_x     = 0;
    cur_pos_y     = 0;
    del_count     = 0;
    ins_count     = 0;
    booleans     &= ~( ebMODIFIED + ebSELECTING );
    booleans     |= ebDIRTY;
    buf_len       = length;
    gap_len       = buf_size - length;
    limit_xl      = MAX_LINE_LENGTH;
    limit_yl      = count_lines( edit_buffer+gap_len, buf_len ) + 1;
    update( 0, 0, 0 );
  }

  void Ttext_editor::set_cur_ptr( uint p, char select_mode )
  {
    uint anchor;

    if( !( select_mode & smEXTEND ) )
      anchor = p;
    else
      if( cur_ptr == sel_start )
        anchor = sel_end;
      else
        anchor = sel_start;
    if( p < anchor )
    {
      if( select_mode & smDOUBLE )
      {
        p = prev_line( next_line( p ) );
        anchor = next_line( prev_line( anchor ) );
      }
      set_select( p, anchor, 1 );
    }
    else
    {
      if( select_mode & smDOUBLE )
      {
        p = next_line( p );
        anchor = prev_line( next_line( anchor ) );
      }
      set_select( anchor, p, 0 );
    }
  }

  void Ttext_editor::set_cur_xy( int _x, uint _y, char select_mode )
  {
    uint p, le;
    int n;

    set_cur_ptr( line_move( cur_ptr, _y - cur_pos_y ), select_mode );
    p = line_start( cur_ptr );
    le = line_end( cur_ptr );
    _x = max( 0, _x );
    n = 0;
    while( ( p < le ) && ( n < _x ) )
      n += ( buf_char( p++ ) == '\t' )? ( tab_size - ( n % tab_size ) ) : 1;
    set_cur_ptr( p, select_mode );
    if( n < _x ) cur_after_eol = _x - n;
    update( 0, 0, 0 );
  }

  void Ttext_editor::set_select( uint new_start, uint new_end, int cur_start )
  {
    uint p, l, beg, end;

    if( cur_start ) p = new_start; else p = new_end;
    beg = sel_start;
    end = sel_end;
    if( p != cur_ptr )
    {
      if( p > cur_ptr )
      {
        l = p - cur_ptr;
        memmove( edit_buffer+cur_ptr, edit_buffer+cur_ptr+gap_len, l );
        cur_pos_y += count_lines( edit_buffer+cur_ptr, l );
        cur_ptr = p;
      }
      else
      {
        l = cur_ptr - p;
        cur_ptr = p;
        cur_pos_y -= count_lines( edit_buffer+cur_ptr, l );
        memmove( edit_buffer+cur_ptr+gap_len, edit_buffer+cur_ptr, l );
      }
      cur_pos_x = char_pos( line_start( cur_ptr ), p );
      del_count = 0;
      ins_count = 0;
      set_buf_size( buf_len );
    }
    sel_start = new_start;
    sel_end = new_end;
    cur_after_eol = 0;
    if( ( ( beg != sel_start ) || ( end != sel_end ) ) &&
        ( ( beg != end ) || ( sel_start != sel_end ) ) ) booleans |= ebDIRTY;
    update( 0, 0, 0 );
  }

  void Ttext_editor::hide_select( void )
  {
    set_select( cur_ptr, cur_ptr, 0 );
    booleans &= ~ebSELECTING;
  }

  boolean Ttext_editor::has_selection( void )
  {
    return sel_start != sel_end;
  }

  uint Ttext_editor::lines_selected( void )
  {
    if( !has_selection() ) return 0;
    return count_lines( edit_buffer+buf_ptr( sel_start ), sel_end-sel_start ) + 1;
  }

  void Ttext_editor::undo( void )
  {
    uint length;

    if( del_count || ins_count )
    {
      sel_start = cur_ptr - ins_count;
      sel_end = cur_ptr;
      length = del_count;
      del_count = 0;
      ins_count = 0;
      insert_buffer( edit_buffer, cur_ptr + gap_len - length, length, 0, 1 );
    }
  }

  boolean Ttext_editor::insert_text( char *text, uint length, int select_text )
  {
    return insert_buffer( text, 0, length, booleans & ebCAN_UNDO, select_text );
  }

  boolean Ttext_editor::insert_string( char *text, int select_text )
  {
    return insert_text( text, strlen( text ), select_text );
  }

  boolean Ttext_editor::insert_from( Ttext_editor *text_editor )
  {
    return insert_buffer( text_editor->edit_buffer,
      text_editor->buf_ptr( text_editor->sel_start ),
      text_editor->sel_end - text_editor->sel_start,
      booleans & ebCAN_UNDO,
      (Ttext_editor *) clipboard == this );
  }

  int Ttext_editor::smart_tab( void )
  {
    uint p, pe, i, cp;

    p = prev_line( line_start( cur_ptr ) );
    pe = line_end( p );
    cp = char_pos( line_start( cur_ptr ), cur_ptr ) + cur_after_eol;
    i = 0;
    while( ( p < pe ) && ( i < cp ) )
    {
      if( edit_buffer[p] == '\t' )
        i += tab_size - ( i % tab_size );
      else
        i++;
      p++;
    }
    while( ( p < pe ) &&
           ( edit_buffer[p] != ' ' ) && ( ( edit_buffer[p] != '\t' ) ) )
    {
      i++;
      p++;
    }
    while( p < pe )
    {
      if( edit_buffer[p] == ' ' )
        i++;
      else
        if( edit_buffer[p] == '\t' )
          i += tab_size - ( i % tab_size );
        else
          break;
      p++;
    }
    if( p >= pe )  return cp + tab_size - ( cp % tab_size );
    return i;
  }

  void Ttext_editor::new_line( void )
  {
    static char cr_lf[2] = { 13, 10 };
    uint i;

    if( !has_selection() )
    {
      i = cur_ptr;
      while( i && ( buf_char( i - 1 ) == ' ' ) ) i--;
      delete_range( i, cur_ptr, 0 );
    }
    cur_after_eol = 0;
    insert_text( cr_lf, 2, 0 );
    if( editor_flags & efAUTO_INDENT )
    {
      i = prev_line( line_start( cur_ptr ) );
      if( ( edit_buffer[i] != ' ' ) && ( edit_buffer[i] != '\t' ) ) return;
      cur_after_eol = smart_tab();
      if( cur_ptr < line_end( cur_ptr ) ) insert_text( NULL, 0, 0 );
    }
  }

  void Ttext_editor::delete_select( void )
  {
    insert_text( NULL, 0, 0 );
  }

  void Ttext_editor::delete_range( uint start_ptr, uint end_ptr, int del_select )
  {
    if( !has_selection() || !del_select )
    {
      set_select( cur_ptr, end_ptr, 1 );
      delete_select();
      set_select( start_ptr, cur_ptr, 0 );
    }
    delete_select();
  }

  boolean Ttext_editor::clip_copy( void )
  {
    if( ( clipboard != NULL ) && ( clipboard != this ) )
    {
      booleans &= ~ebSELECTING;
      return clipboard->insert_from( this );
    }
    return 0;
  }

  void Ttext_editor::clip_cut( void )
  {
    if( clip_copy() )
    {
      delete_select();
      clear_eol_spaces();
    }
  }

  void Ttext_editor::clip_paste( void )
  {
    if( ( clipboard != NULL ) && ( clipboard != this ) )
      insert_from( clipboard );
  }

  void Ttext_editor::get_text( char *text, uint ofs, uint length )
  {
    length++;
    while( --length )
      *text++ = buf_char( ofs++ );
  }

  void Ttext_editor::get_line_str( char *str, uint length )
  {
    uint s, l;

    s = line_start( cur_ptr );
    l = min( length, line_end( cur_ptr ) - s );
    get_text( str, s, l ); str[l] = 0;
  }

  void Ttext_editor::get_word_str( char *str, uint length )
  {
    uint s;

    if( buf_char( cur_ptr ) & word_chars )
    {
      s = prev_word( next_word( cur_ptr ) );
      length++;
      while( --length && ( s < buf_len ) &&
             ( ( ( *str = buf_char( s++ ) ) & word_chars ) || ( *str == '.' ) ) )
        str++;
    }
    *str = 0;
  }

  uint Ttext_editor::next_char( uint p )
  {
    char *n;

    if( ( p == buf_len ) || ( ++p == buf_len ) ) return p;
    n = edit_buffer;
    if( p >= cur_ptr ) n += gap_len;
    if( *( (word *) ( n + p - 1 ) ) == 0x0A0D ) p++;
    return p;
  }

  uint Ttext_editor::next_line( uint p )
  {
    return next_char( line_end( p ) );
  }

  uint Ttext_editor::next_word( uint p )
  {
    while( ( p < buf_len ) && ( buf_char( p ) & word_chars ) )
      p = next_char( p );
    while( ( p < buf_len ) && !( buf_char( p ) & word_chars ) )
      p = next_char( p );
    return p;
  }

  uint Ttext_editor::prev_char( uint p )
  {
    char *n;

    if( ( p == 0 ) || ( --p == 0 ) ) return p;
    n = edit_buffer;
    if( p >= cur_ptr ) n += gap_len;
    if( *( (word *) ( n + p - 1 ) ) == 0x0A0D ) p--;
    return p;
  }

  uint Ttext_editor::prev_line( uint p )
  {
    return line_start( prev_char( p ) );
  }

  uint Ttext_editor::prev_word( uint p)
  {
    while( ( p > 0 ) && !( buf_char( prev_char( p ) ) & word_chars ) )
      p = prev_char( p );
    while( ( p > 0 ) && ( buf_char( prev_char( p ) ) & word_chars ) )
      p = prev_char( p );
    return p;
  }

  uint Ttext_editor::line_start( uint p )
  {
    while( ( p-- > 0 ) && ( buf_char( p ) != 0x0D ) );
    if( buf_char( ++p ) == '\n' ) p++;
    return p;
  }

  uint Ttext_editor::line_end( uint p )
  {
    while( ( p < buf_len ) && ( buf_char( p ) != 0x0D ) )
      p++;
    return p;
  }

  uint Ttext_editor::line_move( uint p, int count )
  {
    int pos;
    uint i;

    i = p;
    p = line_start( p );
    pos = char_pos( p, i );
    while( count )
    {
      i = p;
      if( count < 0 )
      {
        p = prev_line( p );
        count++;
      }
      else
      {
        p = next_line( p );
        count--;
      }
    }
    if( p != i ) p = char_ptr( p, pos );
    return p;
  }

  char Ttext_editor::buf_char( uint p )
  {
    if( p >= cur_ptr ) p += gap_len;
    return *( edit_buffer + p );
  }

  uint Ttext_editor::buf_ptr( uint p )
  {
    if( p >= cur_ptr ) p += gap_len;
    return p;
  }

  boolean Ttext_editor::search( char *find_str, uint options )
  {
    uint i, pos;

    pos = cur_ptr;
    do
    {
      if( options & efCASE_SENSITIVE )
        i = scan( edit_buffer + buf_ptr( pos ), buf_len - pos + 1, find_str );
      else
        i = iscan( edit_buffer + buf_ptr( pos ), buf_len - pos + 1, find_str );
      if( ( i != sfSEARCH_FAILED ) )
      {
        i += pos;
        if( !( options & efWHOLE_WORDS_ONLY ) ||
            !( ( i && ( word_chars & buf_char( i - 1 ) ) ) ||
             ( ( i + strlen( find_str ) != buf_len ) &&
               ( word_chars & buf_char( i + strlen( find_str ) ) ) ) ) )
        {
          set_select( i, i + strlen( find_str ), 0 );
          return 1;
        }
        else
          pos = i + 1;
      }
    }
    while( i != sfSEARCH_FAILED );
    return 0;
  }

  static char *__edit_buffer;
  static word *__draw_buffer;
  static uint __from;
  static int __ready;
  static int __width;

  static boolean format_to( uint to, word ch )
  {
    register uint count;
    char *source;

    if( to < __from ) return 1;
    count = to - __from;
    source = __edit_buffer + __from;
    loop:
      while( count && ( ( __from++, lo( ch ) = *source++ ) >= ' ' ) )
      {
        count--;
        *__draw_buffer++ = ch;
        if( ++__ready >= __width ) return 0;
      }
      if( !count ) return 1;
      if( lo( ch ) == 13 ) return 0;
      count--;
      if( lo( ch ) == '\t' )
      {
        lo( ch ) = ' ';
        do
        {
          *__draw_buffer++ = ch;
          if( ++__ready >= __width ) return 0;
        }
        while( __ready % tab_size );
        goto loop;
      }
      lo( ch ) = ' ';
      *__draw_buffer++ = ch;
      if( ++__ready >= __width ) return 0;
      goto loop;
  }

  void Ttext_editor::format_line( word *draw_buf, uint line_ptr, int width, char normal, char selected )
  {
    word color, __normal, __selected;

    __edit_buffer = edit_buffer;
    __draw_buffer = draw_buf;
    __from = line_ptr;
    __ready = 0;
    __width = width;
    __normal = normal << 8;
    __selected = selected << 8;
    if( format_to( sel_start, color = __normal ) &&
        format_to( cur_ptr, color = __selected ) &&
        ( __from += gap_len, format_to( sel_end + gap_len, color = __selected ) ) &&
        format_to( buf_size, color = __normal ) );
    __width -= __ready;
    lo( color ) = ' ';
    while( __width-- > 0 )
      *(__draw_buffer++) = color;
  }

  void Ttext_editor::set_name( char *_file_name )
  {
    char buffer[_MAX_PATH];
    fexpand( strcpy(buffer,_file_name) );
    FREE( file_name );
    file_name=STRDUP(buffer);
  }

  boolean Ttext_editor::load( void )
  {
    uint length;
    long fsize;
    int handle;
    unsigned num_read;
    char ctrl_z = 0;
    boolean result = 0;

    if( file_name==NULL ) return 0;
    length = 0;
    switch( _dos_open( file_name, O_RDONLY, &handle) )
    {
      case 2:
        result = 1;
        break;
      case 0:
        fsize = filelength( handle );
        if( fsize )
        {
          lseek( handle, fsize-1, 0 );
          if( _dos_read( handle, &ctrl_z, 1, &num_read ) )
            editor_dialog( edREAD_ERROR, file_name );
          else
            if( ctrl_z == 0x1A ) fsize--;
          lseek( handle, 0, 0 );
        }
        if(
#ifndef __386__
            ( fsize > 0xFFF0 ) ||
#endif
            !set_buf_size( (uint) fsize ) )
          editor_dialog( edOUT_OF_MEMORY, NULL );
        else
        {
          if( _dos_read( handle, edit_buffer+buf_size-(uint)fsize, (uint)fsize, &num_read ) )
            editor_dialog( edREAD_ERROR, file_name );
          else
          {
            result = 1;
            length = (uint) fsize;
          }
        }
        _dos_close( handle );
        break;
      default:
        editor_dialog( edREAD_ERROR, file_name );
    }
    set_buf_len( length );
    return result;
  }

  boolean Ttext_editor::save( void )
  {
    int handle;
    unsigned num_written;
    char drive[_MAX_DRIVE];
    char dir[_MAX_DIR];
    char fname[_MAX_FNAME];
    char ext[_MAX_EXT];
    char bak[_MAX_PATH];

    if( file_name==NULL ) return 0;
    boolean result = 0;
    if( editor_flags & efBACKUP_FILES )
    {
      _splitpath( file_name, drive, dir, fname, ext );
      _makepath( bak, drive, dir, fname, ".BAK" );
      remove( bak );
      rename( file_name, bak );
    }
    if( _dos_creat( file_name, _A_NORMAL, &handle ) )
      editor_dialog( edCREATE_ERROR, file_name );
    else
    {
      if( _dos_write( handle, edit_buffer, cur_ptr, &num_written ) ||
          ( num_written != cur_ptr ) ||
          _dos_write( handle, edit_buffer+cur_ptr+gap_len, buf_len - cur_ptr, &num_written ) ||
          ( num_written != buf_len - cur_ptr ) )
        editor_dialog( edWRITE_ERROR, file_name );
      else
      {
        booleans &= ~ebMODIFIED;
        booleans |= ebDIRTY;
        update( 0, 0, 0 );
        result = 1;
      }
      _dos_close( handle );
    }
    if( result )
    {
      get_date_time( file_name, date, time );
      broadcast( cmSTAMP_CHECK );
    }
    return result;
  }

//protected:

  int Ttext_editor::char_pos( uint p, uint target )
  {
    int pos;

    pos = 0;
    while( p < target )
    {
      if( buf_char( p ) == '\t' )
        pos += tab_size - ( pos % tab_size );
      else
        pos++;
      p++;
    }
    return pos;
  }

  uint Ttext_editor::char_ptr( uint p, int target )
  {
    int pos;

    pos = 0;
    while( ( pos < target ) && ( p < buf_len ) && ( buf_char( p ) != 13 ) )
    {
      if( buf_char( p ) == '\t' )
        pos += tab_size - ( pos % tab_size );
      else
        pos++;
      p++;
    }
    if( pos > target ) p--;
    return p;
  }

  boolean Ttext_editor::insert_buffer( char *p, uint offset, uint length, int allow_undo, int select_text )
  {
    uint sel_len, del_len, sel_lines, lines, new_size;
    uint cur_ptr_inserted;
    int lines_inserted;
    int bytes_inserted;

    booleans &= ~ebSELECTING;
    sel_len = sel_end - sel_start;
    if( ( !sel_len ) && ( !length ) && ( !cur_after_eol ) ) return 1;
    del_len = 0;
    if( allow_undo )
      if( cur_ptr == sel_start )
        del_len = sel_len;
      else
        if( sel_len > ins_count ) del_len = sel_len - ins_count;
    new_size = ( buf_len + del_count - sel_len + del_len ) + length + cur_after_eol;
    if( new_size > buf_len + del_count )
      if(
#ifndef __386__
          ( new_size > 0xFFF0 ) ||
#endif
          !set_buf_size( new_size ) )
      {
        editor_dialog( edOUT_OF_MEMORY, NULL );
        sel_end = sel_start;
        return 0;
      }
    sel_lines = count_lines( edit_buffer+buf_ptr( sel_start ), sel_len );
    if( cur_ptr == sel_end )
    {
      if( allow_undo )
      {
        if( del_len > 0 )
          memmove( edit_buffer+cur_ptr+gap_len-del_count-del_len, edit_buffer+sel_start, del_len );
        ins_count -= ( sel_len - del_len );
      }
      cur_ptr = sel_start;
      cur_pos_y -= sel_lines;
    }
    if( cur_after_eol > 0 )
      memset( edit_buffer+cur_ptr, ' ', cur_after_eol );
    if( length > 0 )
      memmove( edit_buffer+cur_ptr+cur_after_eol, p+offset, length );
    length += cur_after_eol;
    cur_after_eol = 0;
    lines = count_lines( edit_buffer + cur_ptr, length );
    cur_ptr_inserted = cur_ptr;
    lines_inserted = lines - sel_lines;
    bytes_inserted = length - sel_len;
    cur_ptr += length;
    cur_pos_y += lines;
    limit_yl += lines_inserted;
    if( editor != NULL )
      editor->scroll_to( editor->delta_x, max( 0, min( editor->delta_y, limit_yl - editor->yl ) ) );
    cur_pos_x = char_pos( line_start( cur_ptr ), cur_ptr );
    if( !select_text ) sel_start = cur_ptr;
    sel_end = cur_ptr;
    buf_len += bytes_inserted;
    gap_len -= bytes_inserted;
    if( allow_undo )
    {
      del_count += del_len;
      ins_count += length;
    }
    if( clipboard != this ) booleans |= ebMODIFIED;
    set_buf_size( buf_len + del_count );
    booleans |= ebDIRTY;
    update( cur_ptr_inserted, bytes_inserted, lines_inserted );
    return 1;
  }

  void Ttext_editor::clear_eol_spaces( void )
  {
    uint p, xx;

    if( cur_ptr == line_end( cur_ptr ) )
    {
      p = cur_ptr;
      while( p && ( buf_char( p - 1 ) == ' ' ) ) p--;
      xx = cur_ptr - p;
      delete_range( p, cur_ptr, 1 );
      cur_after_eol = xx;
    }
  }

  void Ttext_editor::start_select( void )
  {
    hide_select();
    booleans |= ebSELECTING;
  }

  void Ttext_editor::update( uint cur_ptr_inserted, int bytes_inserted, int lines_inserted )
  {
    Tfile_editor *e;

    if( editor == NULL ) return;
    if( assoc_number == 1 )
      editor->update( cur_ptr_inserted, bytes_inserted, lines_inserted );
    else
      for( e=open_editors; e!=NULL; e=e->next_editor )
        if( e->text_editor == this )
          e->update( cur_ptr_inserted, bytes_inserted, lines_inserted );
    booleans &= ~ebDIRTY;
  }

#pragma off( unreferenced )
  uint Ttext_editor::editor_dialog( int dialog, void *info )
  {
    if( editor == NULL ) return cmCANCEL;
    return editor->editor_dialog( dialog, info );
  }
#pragma on( unreferenced )


#ifdef SYNTAXHILIGHT

/*
Tsyntax_editor
*/

//public:

  Tsyntax_editor::Tsyntax_editor( uint _buf_size, Teditor *_editor ):
    Ttext_editor( _buf_size, _editor )
  {
  }

  Tsyntax_editor::Tsyntax_editor( char *_edit_buffer, uint _buf_size, Teditor *_editor ):
    Ttext_editor( _edit_buffer, _buf_size, _editor )
  {
  }

  void Tsyntax_editor::set_name( char *_file_name )
  {
    Ttext_editor::set_name( _file_name );
    syntax_type=0;
#ifdef CPPHILIGHT
    if( must_hilight(file_name,cpp_hilight_extensions) ) syntax_type=syCPP;
#endif
#ifdef ASMHILIGHT
    if( must_hilight(file_name,asm_hilight_extensions) ) syntax_type=syASM;
#endif
#ifdef A51HILIGHT
    if( must_hilight(file_name,a51_hilight_extensions) ) syntax_type=syA51;
#endif
  }

#ifdef CPPHILIGHT
  #define RW_NUM (sizeof(rw)/sizeof(*rw))
  static void syntax_cpp( word *draw_buf, int width )
  {
    static char *rw[] = {
      "asm", "auto", "break", "case", "catch", "char", "class", "const",
      "continue",  "default", "delete", "do", "double", "else", "enum",
      "extern", "float", "for", "friend", "goto", "if", "inline", "int",
      "long", "new", "operator", "private", "protected", "public",
      "register", "return", "short", "signed", "sizeof", "static", "struct",
      "switch", "template", "this", "throw", "try", "typedef", "union",
      "unsigned", "virtual", "void", "volatile", "while"
    };
    int len = ++width;
    word *b;
    { //initialize the buffer
      int i=len;
      for( word *b=draw_buf; --i; hi(*b++) &= 0xF0 );
    }
    { //skip ending blank space
      char c = cpp_syntax_colors[cppWHITE];
      for( word *b=draw_buf+len-1; lo(*--b)==' '; )
        if( --len ) hi(*b) |= c; else return;
    }
    { //skip leading blank space
      char c = cpp_syntax_colors[cppWHITE];
      for( b=draw_buf; lo(*b)==' ' && --len; hi(*b++) |= c );
    }
    if( lo(*b)=='#' )
    { //preprocessor
      char c = cpp_syntax_colors[cppPREPROCESSOR];
      while( --len )
      {
        if( lo(*b)=='/' )
          switch( lo(*(b+1)) )
          {
            case '*':
            { /* comment */
              char c = cpp_syntax_colors[cppCOMMENTS], old=0;
              ++len;
              while( --len )
              {
                hi(*b++) |= c;
                if( old=='*' && lo(*(b-1))=='/' ) break;
                old = lo(*(b-1));
              }
              if( !len ) return;
              continue;
            }
            case '/':
              c = cpp_syntax_colors[cppCOMMENTS];
          }
        hi(*b++) |= c;
      }
      return;
    }
    while( len>1 )
    { //until the end of the buffer
      if( lo(*b)=='/' && lo(*(b+1))=='/' )
      { //comment
        char c = cpp_syntax_colors[cppCOMMENTS];
        while( --len ) hi(*b++) |= c;
        return;
      }
      if( lo(*b)=='/' && lo(*(b+1))=='*' )
      { /* comment */
        char c = cpp_syntax_colors[cppCOMMENTS], old=0;
        while( --len )
        {
          hi(*b++) |= c;
          if( old=='*' && lo(*(b-1))=='/' ) break;
          old = lo(*(b-1));
        }
        continue;
      }
      if( lo(*b)=='"' || lo(*b)=='\'' )
      { //string
        char ch=lo(*b);
        char c = cpp_syntax_colors[cppSTRINGS];
        do
        {
          if( lo(*b)=='\\' )
            if( --len ) hi(*b++) |= c; else return;
          if( --len ) hi(*b++) |= c; else return;
        } while( lo(*b)!=ch );
        if( --len ) hi(*b++) |= c; else return;
        continue;
      }
      if( isdigit(lo(*b)) )
      { //number
        char c = cpp_syntax_colors[cppNUMBERS];
        while( (isalnum(lo(*b))||lo(*b)=='.') && --len ) hi(*b++) |= c;
        continue;
      }
      if( __iscsymf(lo(*b)) )
      { //reserved word or identifier
        int i=0, l=len;
        char ch, wrd[MAX_RESERVED_WORD_LENGTH] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 };
        for( int n=0; --l && __iscsym(wrd[n]=ch=b[n]); n++ )
        {
          for( ; i<RW_NUM && rw[i][n]!=wrd[n]; i++ );
          if( i==RW_NUM || memcmp(wrd,rw[i],n) ) break;
        }
        if( i<RW_NUM && !rw[i][n] )
        {
          char c = cpp_syntax_colors[cppRESERVED_WORDS];
          for( n++; --n && --len; hi(*b++) |= c );
        }
        else
        {
          char c = cpp_syntax_colors[cppIDENTIFIERS];
          while( __iscsym(lo(*b)) && --len ) hi(*b++) |= c;
        }
        continue;
      }
      if( cpp_symbols&lo(*b) )
        //symbols
        hi(*b++) |= cpp_syntax_colors[cppSYMBOLS];
      else
        //white space
        hi(*b++) |= cpp_syntax_colors[cppWHITE];
      --len;
    }
  }
  #undef RW_NUM
#endif //CPPHILIGHT

#ifdef ASMHILIGHT
  #define RW_NUM (sizeof(rw)/sizeof(*rw))
  static void syntax_asm( word *draw_buf, int width )
  {
    static char *rw[] = {
      "AAA", "AAD", "AAM", "AAS", "ABS", "ADC", "ADD", "AH", "AL", "ALIGN",
      "ALN", "AND", "ARG", "ARPL", "ASS", "ASSUME", "AT", "AX", "BASIC",
      "BH", "BL", "BOUND", "BP", "BRK", "BSF", "BSR", "BSS", "BSWAP", "BT",
      "BTC", "BTR", "BTS", "BX", "BYTE", "CALL", "CALLMETHOD", "CALLPROC",
      "CATSTR", "CBW", "CDQ", "CH", "CL", "CLC", "CLD", "CLI", "CLTS",
      "CMC", "CMP", "CMPS", "CMPSB", "CMPSD", "CMPSW", "CMPXCHG", "CODE",
      "CODEPTR", "CODESEG", "COMM", "COMMENT", "COMMON", "COMPACT", "CONST",
      "CPP", "CR0", "CR2", "CR3", "CS", "CWD", "CWDE", "CX", "DAA", "DAS",
      "DATA", "DATAPTR", "DATASEG", "DB", "DD", "DEC", "DF", "DGROUP", "DH",
      "DI", "DISPLAY", "DIV", "DL", "DOS", "DOSSEG", "DP", "DQ", "DR0",
      "DR1", "DR2", "DR3", "DR6", "DR7", "DS", "DT", "DUP", "DW", "DWORD",
      "DX", "EAX", "EBP", "EBX", "ECX", "EDI", "EDX", "ELSE", "ELSEIF",
      "ELSEIF1", "ELSEIF2", "ELSEIFB", "ELSEIFDEF", "ELSEIFDIF",
      "ELSEIFDIFI", "ELSEIFE", "ELSEIFIDN", "ELSEIFIDNI", "ELSEIFNB",
      "ELSEIFNDEF", "EMUL", "END", "ENDIF", "ENDM", "ENDP", "ENDS", "ENTER",
      "ENTERD", "ENTERW", "ENUM", "EQ", "EQU", "ERR", "ERRIF", "ERRIF1",
      "ERRIF2", "ERRIFB", "ERRIFDEF", "ERRIFDIF", "ERRIFDIFI", "ERRIFE",
      "ERRIFIDN", "ERRIFIDNI", "ERRIFNB", "ERRIFNDEF", "ES", "ESC", "ESI",
      "ESP", "EVEN", "EVENDATA", "EXECONLY", "EXECREAD", "EXIT", "EXITCODE",
      "EXITM", "EXTRN", "F2XM1", "FABS", "FADD", "FADDP", "FAR", "FARDATA",
      "FARSTACK", "FAR_BSS", "FAR_DATA", "FASTIMUL", "FBLD", "FBSTP",
      "FCHS", "FCLEX", "FCOM", "FCOMP", "FCOMPP", "FCOS", "FDECSTP",
      "FDISI", "FDIV", "FDIVP", "FDIVR", "FDIVRP", "FENI", "FFREE", "FIADD",
      "FIARQQ", "FICOM", "FICOMP", "FICRQQ", "FIDIV", "FIDIVR", "FIDRQQ",
      "FIERQQ", "FIFRQQ", "FIGRQQ", "FILD", "FIMUL", "FINCSTP", "FINIT",
      "FISRQQ", "FIST", "FISTP", "FISUB", "FISUBR", "FIWRQQ", "FJARQQ",
      "FJCRQQ", "FJFRQQ", "FJGRQQ", "FJSRQQ", "FLAT", "FLD", "FLD1",
      "FLDCW", "FLDENV", "FLDL2E", "FLDL2T", "FLDLG2", "FLDLN2", "FLDPI",
      "FLDZ", "FLIPFLAG", "FMUL", "FMULP", "FNCLEX", "FNDISI", "FNENI",
      "FNINIT", "FNOP", "FNSAVE", "FNSTCW", "FNSTENV", "FNSTSW", "FORTRAN",
      "FPATAN", "FPREM", "FPREM1", "FPTAN", "FRNDINT", "FRSTOR", "FS",
      "FSAVE", "FSCALE", "FSETPM", "FSIN", "FSINCOS", "FSQRT", "FST",
      "FSTCW", "FSTENV", "FSTP", "FSTSW", "FSUB", "FSUBP", "FSUBR",
      "FSUBRP", "FTST", "FUCOM", "FUCOMP", "FUCOMPP", "FWAIT", "FWORD",
      "FXAM", "FXCH", "FXTRACT", "FYL2X", "FYL2XP1", "GE", "GETFIELD",
      "GLOBAL", "GOTO", "GROUP", "GS", "GT", "GTP", "HIGH", "HLT", "HUGE",
      "ICG", "IDEAL", "IDIV", "IF", "IF1", "IF2", "IFB", "IFDEF", "IFDIF",
      "IFDIFI", "IFE", "IFIDN", "IFIDNI", "IFNB", "IFNDEF", "IMUL", "IN",
      "INC", "INCLUDE", "INCLUDELIB", "INS", "INSB", "INSD", "INSTR",
      "INSW", "INT", "INTO", "INVD", "INVLPG", "IRET", "IRETD", "IRETW",
      "IRP", "IRPC", "JA", "JAE", "JB", "JBE", "JC", "JCXZ", "JE", "JECXZ",
      "JG", "JGE", "JL", "JLE", "JMP", "JMPMETHOD", "JNA", "JNAE", "JNB",
      "JNBE", "JNC", "JNE", "JNG", "JNGE", "JNL", "JNLE", "JNO", "JNP",
      "JNS", "JNZ", "JO", "JP", "JPE", "JPO", "JS", "JUMPS", "JZ", "LABEL",
      "LABEL", "LAHF", "LAR", "LARGE", "LARGESTACK", "LCO", "LDS", "LE",
      "LEA", "LEAVE", "LEAVED", "LEAVEW", "LENGTH", "LES", "LFS", "LGDT",
      "LGS", "LIDT", "LLDT", "LMSW", "LOCAL", "LOCALS", "LOCK", "LODS",
      "LODSB", "LODSD", "LODSW", "LOOP", "LOOPD", "LOOPDE", "LOOPDNE",
      "LOOPDNZ", "LOOPDZ", "LOOPE", "LOOPNE", "LOOPNZ", "LOOPW", "LOOPWE",
      "LOOPWNE", "LOOPWNZ", "LOOPWZ", "LOOPZ", "LOW", "LSL", "LSS", "LT",
      "LTR", "M400", "M500", "M510", "M520", "MACRO", "MASK", "MASKFLAG",
      "MASM", "MASM51", "MCP", "MEDIUM", "MEMORY", "MEMPAGE", "METHOD",
      "MOD", "MODEL", "MOV", "MOVS", "MOVSB", "MOVSD", "MOVSW", "MOVSX",
      "MOVZX", "MUL", "MULTERRS", "NAME", "NE", "NEAR", "NEARSTACK", "NEG",
      "NOEMUL", "NOJUMPS", "NOLANGUAGE", "NOLOCALS", "NOMASM51",
      "NOMULTERRS", "NOP", "NORMAL", "NOSMART", "NOT", "NOTHING", "NOWARN",
      "OBJDEF", "OBJEND", "ODDFAR", "ODDNEAR", "OFFSET", "OPI", "OPP",
      "OPS", "OR", "ORG", "OS2", "OS_DOS", "OS_OS2", "OUT", "OUTS", "OUTSB",
      "OUTSD", "OUTSW", "OVF", "P186", "P286", "P286N", "P287", "P386",
      "P386N", "P387", "P486", "P486N", "P487", "P8086", "P8087", "PAGE",
      "PARA", "PASCAL", "PDC", "PNO87", "POP", "POPA", "POPAD", "POPAW",
      "POPF", "POPFD", "POPFW", "POPSTATE", "PQK", "PRIVATE", "PRO", "PROC",
      "PROC", "PROCBEG", "PROCDESC", "PROCEND", "PROCTYPE", "PROLOG", "PTR",
      "PUBLIC", "PUBLICDLL", "PURGE", "PUSH", "PUSHA", "PUSHAD", "PUSHAW",
      "PUSHF", "PUSHFD", "PUSHFW", "PUSHSTATE", "PWORD", "QUIRKS", "QWORD",
      "RADIX", "RCL", "RCR", "READONLY", "READWRITE", "RECORD", "REP",
      "REPE", "REPNE", "REPNZ", "REPT", "REPZ", "REQ", "RES", "REST", "RET",
      "RETCODE", "RETF", "RETN", "RETURNS", "ROL", "ROR", "SAHF", "SAL",
      "SAR", "SBB", "SCAS", "SCASB", "SCASD", "SCASW", "SEG", "SEGCS",
      "SEGDS", "SEGES", "SEGFS", "SEGGS", "SEGMENT", "SEGSS", "SETA",
      "SETAE", "SETB", "SETBE", "SETC", "SETE", "SETFIELD", "SETFLAG",
      "SETG", "SETGE", "SETL", "SETLE", "SETNA", "SETNAE", "SETNB",
      "SETNBE", "SETNC", "SETNE", "SETNG", "SETNGE", "SETNL", "SETNLE",
      "SETNO", "SETNP", "SETNS", "SETNZ", "SETO", "SETP", "SETPE", "SETPO",
      "SETS", "SETZ", "SGDT", "SHL", "SHLD", "SHORT", "SHR", "SHRD", "SI",
      "SIDT", "SIZE", "SIZESTR", "SLDT", "SMALL", "SMALLSTACK", "SMART",
      "SMSW", "SP", "SS", "ST", "STACK", "STARTUP", "STARTUPCODE", "STC",
      "STD", "STDCALL", "STI", "STOS", "STOSB", "STOSD", "STOSW", "STR",
      "STRUC", "STRUC", "SUB", "SUB", "SUBSTR", "SUBTTL", "SYMTYPE",
      "SYSCALL", "T100", "T101", "T200", "T250", "T300", "T310", "T320",
      "TABLE", "TBLINIT", "TBLINST", "TBLPTR", "TBYTE", "TCHUGE", "TEST",
      "TESTFLAG", "THIS", "TINY", "TITLE", "TPASCAL", "TPI", "TR3", "TR4",
      "TR5", "TR6", "TR7", "TYPE", "TYPEDEF", "UDATASEG", "UFARDATA", "UNI",
      "UNINIT", "UNION", "UNKNOWN", "USE16", "USE32", "USES", "VARARG",
      "VERR", "VERSION", "VERW", "VIRTUAL", "WAIT", "WARN", "WBINVD",
      "WHILE", "WIDTH", "WINDOWS", "WORD", "XADD", "XCHG", "XLAT", "XLATB",
      "XOR", "_BSS", "_DATA", "_TEXT"
    };
    int len = ++width;
    word *b;
    { //initialize the buffer
      int i=len;
      for( word *b=draw_buf; --i; hi(*b++) &= 0xF0 );
    }
    { //skip ending blank space
      char c = asm_syntax_colors[asmWHITE];
      for( word *b=draw_buf+len-1; lo(*--b)==' '; )
        if( --len ) hi(*b) |= c; else return;
    }
    { //skip leading blank space
      char c = asm_syntax_colors[asmWHITE];
      for( b=draw_buf; lo(*b)==' ' && --len; hi(*b++) |= c );
    }
    while( len>1 )
    { //until the end of the buffer
      if( lo(*b)==';' )
      { //comment
        char c = asm_syntax_colors[asmCOMMENTS];
        while( --len ) hi(*b++) |= c;
        return;
      }
      if( lo(*b)=='\'' )
      { //string
        char c = asm_syntax_colors[asmSTRINGS];
        hi(*b++) |= c; if( !--len ) continue;
        while( lo(*b)!='\'' && --len ) hi(*b++) |= c;
        if( len ) hi(*b++) |= c;
        continue;
      }
      if( isdigit(lo(*b)) )
      { //number
        uint l=len; word *s=b; char c;
        while( isalnum(lo(*b)) && --len ) c=lo(*b++);
        switch( toupper(c) )
        {
          case 'H': c=asm_syntax_colors[asmHEXADECIMALS]; break;
          case 'B': c=asm_syntax_colors[asmBINARIES];     break;
          default:  c=asm_syntax_colors[asmDECIMALS];
        }
        for( l-=len-1; --l; hi(*s++)|=c );
      }
      if( __iscsymf(lo(*b)) )
      { //reserved word or identifier
        int i=0, l=len;
        char ch, wrd[MAX_RESERVED_WORD_LENGTH] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 };
        for( int n=0; --l && __iscsym(wrd[n]=ch=toupper(lo(b[n]))); n++ )
        {
          for( ; i<RW_NUM && rw[i][n]!=wrd[n]; i++ );
          if( i==RW_NUM || memcmp(wrd,rw[i],n) ) break;
        }
        if( i<RW_NUM && !rw[i][n] )
        {
          char c = asm_syntax_colors[asmRESERVED_WORDS];
          for( n++; --n && --len; hi(*b++) |= c );
        }
        else
        {
          char c = asm_syntax_colors[asmIDENTIFIERS];
          while( __iscsym(lo(*b)) && --len ) hi(*b++) |= c;
        }
        continue;
      }
      if( asm_symbols&lo(*b) )
        //symbols
        hi(*b++) |= asm_syntax_colors[asmSYMBOLS];
      else
        //white space
        hi(*b++) |= asm_syntax_colors[asmWHITE];
      --len;
    }
  }
  #undef RW_NUM
#endif //ASMHILIGHT

#ifdef A51HILIGHT
  #define RW_NUM (sizeof(rw)/sizeof(*rw))
  static void syntax_a51( word *draw_buf, int width )
  {
    static char *rw[] = {
      "A", "AB", "ACALL", "ADD", "ADDC", "AJMP", "AND", "ANL", "AR0",
      "AR1", "AR2", "AR3", "AR4", "AR5", "AR6", "AR7", "AT", "BIT",
      "BITADDRESSABLE", "BSEG", "C", "CALL", "CJNE", "CLR", "CODE", "CPL",
      "CSEG", "DA", "DATA", "DB", "DBIT", "DEC", "DIV", "DJNZ", "DPTR",
      "DS", "DSEG", "DW", "END", "EQ", "EQU", "EXTI0", "EXTI1", "EXTRN",
      "GE", "GT", "HIGH", "IDATA", "INBLOCK", "INC", "INPAGE", "ISEG", "JB",
      "JBC", "JC", "JMP", "JNB", "JNC", "JNZ", "JZ", "LCALL", "LE", "LJMP",
      "LOW", "LT", "MOD", "MOV", "MOVC", "MOVX", "MUL", "NAME", "NE", "NOP",
      "NOT", "NUMBER", "OR", "ORG", "ORL", "PAGE", "PC", "POP", "PUBLIC",
      "PUSH", "R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7", "RESET",
      "RET", "RETI", "RL", "RLC", "RR", "RRC", "RSEG", "SEGMENT", "SET",
      "SETB", "SHL", "SHR", "SINT", "SJMP", "SUBB", "SWAP", "TIMER0",
      "TIMER1", "UNIT", "USING", "XCH", "XCHD", "XDATA", "XOR", "XRL",
      "XSEG"
    };
    int len = ++width;
    word *b;
    { //initialize the buffer
      int i=len;
      for( word *b=draw_buf; --i; hi(*b++) &= 0xF0 );
    }
    { //skip ending blank space
      char c = a51_syntax_colors[a51WHITE];
      for( word *b=draw_buf+len-1; lo(*--b)==' '; )
        if( --len ) hi(*b) |= c; else return;
    }
    { //skip leading blank space
      char c = a51_syntax_colors[a51WHITE];
      for( b=draw_buf; lo(*b)==' ' && --len; hi(*b++) |= c );
    }
    while( len>1 )
    { //until the end of the buffer
      if( lo(*b)==';' )
      { //comment
        char c = a51_syntax_colors[a51COMMENTS];
        while( --len ) hi(*b++) |= c;
        return;
      }
      if( lo(*b)=='\'' )
      { //string
        char c = a51_syntax_colors[a51STRINGS];
        hi(*b++) |= c; if( !--len ) continue;
        while( lo(*b)!='\'' && --len ) hi(*b++) |= c;
        if( len ) hi(*b++) |= c;
        continue;
      }
      if( isdigit(lo(*b)) )
      { //number
        uint l=len; word *s=b; char c;
        while( isalnum(lo(*b)) && --len ) c=lo(*b++);
        switch( toupper(c) )
        {
          case 'H': c=a51_syntax_colors[a51HEXADECIMALS]; break;
          case 'B': c=a51_syntax_colors[a51BINARIES];     break;
          default:  c=a51_syntax_colors[a51DECIMALS];
        }
        for( l-=len-1; --l; hi(*s++)|=c );
      }
      if( __iscsymf(lo(*b)) )
      { //reserved word or identifier
        int i=0, l=len;
        char ch, wrd[MAX_RESERVED_WORD_LENGTH] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 };
        for( int n=0; --l && __iscsym(wrd[n]=ch=toupper(lo(b[n]))); n++ )
        {
          for( ; i<RW_NUM && rw[i][n]!=wrd[n]; i++ );
          if( i==RW_NUM || memcmp(wrd,rw[i],n) ) break;
        }
        if( i<RW_NUM && !rw[i][n] )
        {
          char c = a51_syntax_colors[a51RESERVED_WORDS];
          for( n++; --n && --len; hi(*b++) |= c );
        }
        else
        {
          char c = a51_syntax_colors[a51IDENTIFIERS];
          while( __iscsym(lo(*b)) && --len ) hi(*b++) |= c;
        }
        continue;
      }
      if( a51_symbols&lo(*b) )
        //symbols
        hi(*b++) |= a51_syntax_colors[a51SYMBOLS];
      else
        //white space
        hi(*b++) |= a51_syntax_colors[a51WHITE];
      --len;
    }
  }
  #undef RW_NUM
#endif //A51HILIGHT

  void Tsyntax_editor::format_line( word *draw_buf, uint line_ptr, int width, char normal, char selected )
  {
    Ttext_editor::format_line( draw_buf, line_ptr, width, normal, selected );
    switch( syntax_type )
    {
#ifdef CPPHILIGHT
      case syCPP:
        syntax_cpp( draw_buf, width );
        break;
#endif
#ifdef ASMHILIGHT
      case syASM:
        syntax_asm( draw_buf, width );
        break;
#endif
#ifdef A51HILIGHT
      case syA51:
        syntax_a51( draw_buf, width );
        break;
#endif
    }
  }

#endif //SYNTAXHILIGHT


/*
Teditor
*/
#define __file_name             text_editor->file_name
#define __date                  text_editor->date
#define __time                  text_editor->time
#define __assoc_number          text_editor->assoc_number
#define __edit_buffer           text_editor->edit_buffer
#define __buf_size              text_editor->buf_size
#define __buf_len               text_editor->buf_len
#define __gap_len               text_editor->gap_len
#define __cur_ptr               text_editor->cur_ptr
#define __sel_start             text_editor->sel_start
#define __sel_end               text_editor->sel_end
#define __cur_pos_x             text_editor->cur_pos_x
#define __cur_pos_y             text_editor->cur_pos_y
#define __cur_after_eol         text_editor->cur_after_eol
#define __limit_xl              text_editor->limit_xl
#define __limit_yl              text_editor->limit_yl
#define __del_count             text_editor->del_count
#define __ins_count             text_editor->ins_count
#define __valid_chars           text_editor->valid_chars
#define __booleans              text_editor->booleans

#define __set_buf_size          text_editor->set_buf_size
#define __set_buf_len           text_editor->set_buf_len
#define __set_cur_ptr           text_editor->set_cur_ptr
#define __set_cur_xy            text_editor->set_cur_xy
#define __set_select            text_editor->set_select
#define __hide_select           text_editor->hide_select
#define __has_selection         text_editor->has_selection
#define __lines_selected        text_editor->lines_selected
#define __undo                  text_editor->undo
#define __insert_text           text_editor->insert_text
#define __insert_string         text_editor->insert_string
#define __insert_from           text_editor->insert_from
#define __new_line              text_editor->new_line
#define __smart_tab             text_editor->smart_tab
#define __delete_select         text_editor->delete_select
#define __delete_range          text_editor->delete_range
#define __clip_copy             text_editor->clip_copy
#define __clip_cut              text_editor->clip_cut
#define __clip_paste            text_editor->clip_paste
#define __get_text              text_editor->get_text
#define __get_line_str          text_editor->get_line_str
#define __get_word_str          text_editor->get_word_str
#define __next_char             text_editor->next_char
#define __next_line             text_editor->next_line
#define __next_word             text_editor->next_word
#define __prev_char             text_editor->prev_char
#define __prev_line             text_editor->prev_line
#define __prev_word             text_editor->prev_word
#define __line_start            text_editor->line_start
#define __line_end              text_editor->line_end
#define __line_move             text_editor->line_move
#define __buf_char              text_editor->buf_char
#define __buf_ptr               text_editor->buf_ptr
#define __search                text_editor->search
#define __format_line           text_editor->format_line
#define __set_name              text_editor->set_name
#define __load                  text_editor->load
#define __save                  text_editor->save
#define __char_pos              text_editor->char_pos
#define __char_ptr              text_editor->char_ptr
#define __insert_buffer         text_editor->insert_buffer
#define __clear_eol_spaces      text_editor->clear_eol_spaces
#define __start_select          text_editor->start_select

//public:

  Teditor::Teditor( Ttext_editor *_text_editor, int _xl, int _yl ):
    Titem( _xl, _yl )
  {
    int i;

    text_editor = _text_editor;
    __assoc_number++;
    indicator = NULL;
    cur_ptr_svd   = 0;
    cur_pos_x_svd = 0;
    cur_pos_y_svd = 0;
    draw_ptr      = 0;
    draw_line     = 0;
    delta_x       = 0;
    delta_y       = 0;
    for( i = 0; i < 10; i++ )
      marks[i] = (uint) -1;
#ifndef NOMOUSE
    set_events_mask( evMOUSE_REP, 1 );
#endif
    ~__valid_chars;
    __valid_chars << 9;
    for( i = 32; i < 256; i++ )
      __valid_chars << i;
    ~word_chars;
    for( i = '0'; i <= '9'; i++ )
      word_chars << i;
    for( i = 'A'; i <= 'Z'; i++ )
      word_chars << i;
    for( i = 'a'; i <= 'z'; i++ )
      word_chars << i;
    word_chars << '_';
#ifdef CYR
    for( i = ''; i <= ''; i++ )
      word_chars << i;
    for( i = ''; i <= ''; i++ )
      word_chars << i;
#endif
  }

  Teditor::~Teditor( void )
  {
    if( --__assoc_number == 0 ) DELETE( text_editor );
  }

  boolean Teditor::cursor_visible( void )
  {
    return ( __cur_pos_y >= delta_y ) && ( __cur_pos_y < delta_y + yl );
  }

  void Teditor::track_cursor( boolean center )
  {
    uint cpx;

    cpx = __cur_pos_x + __cur_after_eol;
    if( center )
      scroll_to( cpx - xl, __cur_pos_y - ( yl >> 1 ) );
    else
      scroll_to( max( cpx         - xl + 1, min( delta_x, cpx         ) ),
                 max( __cur_pos_y - yl + 1, min( delta_y, __cur_pos_y ) ) );
  }

  void Teditor::scroll_to( int _x, int _y )
  {
    _x = max( 0, min( _x, __limit_xl - xl ) );
    _y = max( 0, min( _y, __limit_yl - yl ) );
    if( ( _x != delta_x ) || ( _y != delta_y ) )
    {
      delta_x = _x;
      delta_y = _y;
      if( abs( delta_y - draw_line ) > abs( delta_y - __cur_pos_y ) )
        draw_ptr = __line_move( __cur_ptr, delta_y - __cur_pos_y );
      else
        draw_ptr = __line_move( draw_ptr,  delta_y - draw_line );
      draw_line = delta_y;
      __booleans |= ebDIRTY;
      update( 0, 0, 0 );
    }
  }

  uint Teditor::get_mouse_ptr( int local_x, int local_y )
  {
    local_x = max( 0, min( local_x, xl - 1 ) );
    local_y = max( 0, min( local_y, yl - 1 ) );
    return __char_ptr( __line_move( draw_ptr, local_y + delta_y - draw_line ),
             local_x + delta_x );
  }

  void Teditor::find( void )
  {
    if( editor_dialog( edFIND, NULL ) == cmCANCEL ) return;
    editor_flags &= ~efDO_REPLACE;
    do_search_replace();
  }

  void Teditor::replace( void )
  {
    if( editor_dialog( edREPLACE, NULL ) == cmCANCEL ) return;
    editor_flags |= efDO_REPLACE;
    do_search_replace();
  }

//protected:

  void Teditor::get_focused( void )
  {
    Titem::get_focused();
    current_editor = text_editor;
    if( ( text_editor == clipboard ) ||
        ( ( __cur_pos_y == cur_pos_y_svd ) &&
          ( __cur_pos_x + __cur_after_eol == cur_pos_x_svd ) ) )
      update_commands();
    else
    {
      __set_cur_ptr( cur_ptr_svd, 0 );
      __set_cur_xy( cur_pos_x_svd, cur_pos_y_svd, 0 );
    }
  }

  boolean Teditor::release_focus( void )
  {
    if( !Titem::release_focus() ) return 0;
    current_editor = NULL;
    return 1;
  }

  void Teditor::calc_bounds( int delta_xl, int delta_yl )
  {
    Titem::calc_bounds( delta_xl, delta_yl );
    if( ( grow_mode & gmGROW_HOR ) &&
        ( delta_x + xl > __limit_xl ) &&
        ( __limit_xl > xl ) ) delta_x = ( __limit_xl - xl );
    if( ( grow_mode & gmGROW_VER ) &&
        ( delta_y + yl > __limit_yl ) &&
        ( __limit_yl > yl ) ) delta_y = ( __limit_yl - yl );
  }

  void Teditor::draw( void )
  {
    int dx, dy, dxl, dyl;
    uint line_ptr, i, length;
    word buf[MAX_LINE_LENGTH+MAX_RESERVED_WORD_LENGTH];
    char normal, selected;

    curs_type = ( editor_flags & efOVERWRITE )? 2 : 1;
    curs_x = ( __cur_pos_x + __cur_after_eol ) - delta_x;
    curs_y = __cur_pos_y - delta_y;
    get_clip_rect( dx, dy, dxl, dyl );
    if( state( isDISABLED ) )
      normal = selected = disabled_attr;
    else
    {
      normal = text_attr;
      selected = selected_attr;
    }
    i = dy + delta_y;
    if( i >= __limit_yl ) return;
    draw_ptr = __line_start( __line_move( draw_ptr, i - draw_line ) );
    draw_line = i;
    line_ptr = draw_ptr;
    length = delta_x + xl;
    for( i = 0; i < dyl; i++, dy++ )
    {
      __format_line( buf, line_ptr, length+MAX_RESERVED_WORD_LENGTH, normal, selected );
      goto_xy( -delta_x, dy );
      print( buf, length );
      line_ptr = __next_line( line_ptr );
    }
  }

  void Teditor::event_handler( Tevent &ev )
  {
    char select_mode;
    uint line_num, p, end, s_start, s_end;

    Titem::event_handler( ev );
    if( !state( isFOCUSED ) ) return;
    convert_event( ev );
    select_mode = 0;
    if( ( __booleans & ebSELECTING ) || ( get_shifts() & smSHIFT ) ) select_mode = smEXTEND;
    switch( ev.code )
    {
#ifndef NOMOUSE
      int dx, dy;
      case evMOUSE_DOWN:
        if( ev.INSIDE )
        {
          if( ev.CLICKS ) select_mode |= smDOUBLE;
          do
          {
            if( ev.code == evMOUSE_REP )
            {
              dx = delta_x; dy = delta_y;
              if( ev.LOCAL_X < 0 ) dx--;
              if( ev.LOCAL_X >= xl ) dx++;
              if( ev.LOCAL_Y < 0 ) dy--;
              if( ev.LOCAL_Y >= yl ) dy++;
              scroll_to( dx, dy );
            }
            __set_cur_ptr( get_mouse_ptr( ev.LOCAL_X, ev.LOCAL_Y ), select_mode );
            select_mode |= smEXTEND;
            handled( ev );
          }
          while( get_mouse( ev, evMOUSE_DRAG | evMOUSE_REP ) );
          handled( ev );
        }
        break;
#endif
      case evKEY_PRESS:
        if( ev.ASCII < 256 )
        {
          if( ev.ASCII >= ' ' ) handled( ev );
          if( __valid_chars & ev.ASCII )
          {
            if( __cur_ptr == __line_end( __cur_ptr ) )
            {
              if( ev.ASCII == ' ' )
              {
                __cur_after_eol++;
                goto done;
              }
            }
            else
              if( ( editor_flags & efOVERWRITE ) && !__has_selection() ) __sel_end = __next_char( __cur_ptr );
            __insert_text( (char *) &ev.ASCII, 1, 0 );
            item_acted = this;
            goto done;
          }
        }
        break;
      case evCOMMAND:
        switch( ev.CMD_CODE )
        {
          case cmeFIND:
            find();
            break;
          case cmeREPLACE:
            replace();
            break;
          case cmeSEARCH_AGAIN:
            do_search_replace();
            break;
          case cmeGOTO_LINE:
            line_num = editor_dialog( edGOTO_LINE, &__limit_yl );
            if( line_num )
              __set_cur_xy( 0, line_num - 1, select_mode );
            break;
          case cmCUT:
            __clip_cut();
            item_acted = this;
            break;
          case cmCOPY:
            __clip_copy();
            item_acted = this;
            break;
          case cmPASTE:
            __clip_paste();
            item_acted = this;
            break;
          case cmUNDO:
            __undo();
            item_acted = this;
            break;
          case cmCLEAR:
            __delete_select();
            __clear_eol_spaces();
            item_acted = this;
            break;
          case cmeSCREEN_TOP:
            __set_cur_xy( __cur_pos_x, delta_y, select_mode );
            break;
          case cmeSCREEN_BOTTOM:
            __set_cur_xy( __cur_pos_x, delta_y + yl - 1, select_mode );
            break;
          case cmeSCROLL_UP:
            scroll_to( delta_x, delta_y - 1 );
            goto ret;
          case cmeSCROLL_DOWN:
            scroll_to( delta_x, delta_y + 1 );
            goto ret;
          case cmeCHAR_LEFT:
            if( __cur_after_eol > 0 )
              __cur_after_eol--;
            else
              if( __cur_ptr > __line_start( __cur_ptr ) )
                __set_cur_ptr( __prev_char( __cur_ptr ), select_mode );
            break;
          case cmeCHAR_RIGHT:
            if( __cur_ptr == __line_end( __cur_ptr ) )
              __cur_after_eol++;
            else
              __set_cur_ptr( __next_char( __cur_ptr ), select_mode );
            break;
          case cmeWORD_LEFT:
            __set_cur_ptr( __prev_word( __cur_ptr ), select_mode );
            break;
          case cmeWORD_RIGHT:
            __set_cur_ptr( __next_word( __cur_ptr ), select_mode );
            break;
          case cmeLINE_START:
            __set_cur_ptr( __line_start( __cur_ptr ), select_mode );
            break;
          case cmeLINE_END:
            __set_cur_ptr( __line_end( __cur_ptr ), select_mode );
            break;
          case cmeLINE_UP:
            __set_cur_xy( __cur_pos_x + __cur_after_eol, __cur_pos_y - 1, select_mode );
            break;
          case cmeLINE_DOWN:
            __set_cur_xy( __cur_pos_x + __cur_after_eol, __cur_pos_y + 1, select_mode );
            break;
          case cmePAGE_UP:
            scroll_to( delta_x, delta_y - ( yl - 1 ) );
            __set_cur_xy( __cur_pos_x + __cur_after_eol, __cur_pos_y - ( yl - 1 ), select_mode );
            break;
          case cmePAGE_DOWN:
            scroll_to( delta_x, delta_y + ( yl - 1 ) );
            __set_cur_xy( __cur_pos_x + __cur_after_eol, __cur_pos_y + ( yl - 1 ), select_mode );
            break;
          case cmeTEXT_START:
            __set_cur_ptr( 0, select_mode );
            break;
          case cmeTEXT_END:
            __set_cur_ptr( __buf_len, select_mode );
            break;
          case cmeTAB:
            if( editor_flags & efHARD_TABS )
              __insert_text( "\t", 1, 0 );
            else
            {
              __cur_after_eol = __smart_tab() - __cur_pos_x;
              if( __cur_ptr < __line_end( __cur_ptr ) ) __insert_text( NULL, 0, 0 );
            }
            break;
          case cmeNEW_LINE:
            __new_line();
            item_acted = this;
            break;
          case cmeBACK_SPACE:
            if( __has_selection() )
            {
              __delete_select();
              __clear_eol_spaces();
            }
            else
              if( __cur_after_eol > 0 )
                __cur_after_eol--;
              else
              {
                __delete_range( __prev_char( __cur_ptr ), __cur_ptr, 1 );
                __clear_eol_spaces();
              }
            item_acted = this;
            break;
          case cmeDEL_CHAR:
            if( __has_selection() )
              __delete_select();
            else
            {
              __insert_text( NULL, 0, 0 );
              __delete_range( __cur_ptr, __next_char( __cur_ptr ), 0 );
            }
            __clear_eol_spaces();
            item_acted = this;
            break;
          case cmeDEL_WORD:
            if( __has_selection() )
              __delete_select();
            else
            {
              if( __cur_ptr == __buf_len ) break;
              __insert_text( NULL, 0, 0 );
              p = __next_char( __cur_ptr );
              while( ( p < __buf_len ) &&
                     ( __buf_char( p ) & word_chars ) )
                p = __next_char( p );
              while( ( p < __buf_len ) && ( __buf_char( p ) == ' ' ) )
                p = __next_char( p );
              __delete_range( __cur_ptr, p, 0 );
            }
            __clear_eol_spaces();
            item_acted = this;
            break;
          case cmeDEL_START:
            __delete_range( __line_start( __cur_ptr ), __cur_ptr, 0 );
            item_acted = this;
            break;
          case cmeDEL_END:
            __delete_range( __cur_ptr, __line_end( __cur_ptr ), 0 );
            __clear_eol_spaces();
            item_acted = this;
            break;
          case cmeDEL_LINE:
            __delete_range( __line_start( __cur_ptr ), __next_line( __cur_ptr ), 0 );
            item_acted = this;
            break;
          case cmeINS_MODE:
            editor_flags ^= efOVERWRITE;
            break;
          case cmeSTART_SELECT:
            __start_select();
            break;
          case cmeHIDE_SELECT:
            __hide_select();
            break;
          case cmeINDENT_MODE:
            editor_flags ^= efAUTO_INDENT;
            break;
          case cmeINDENT:
            p = __line_start( __sel_start );
            end = __line_start( __sel_end );
            s_start = __sel_start; s_end = end;
            while( p < end )
            {
              if( __buf_char( p ) != '\r' )
              {
                __set_cur_ptr( p, 0 );
                __insert_text( " ", 1, 0 );
                s_end++;
                end++;
              }
              p = __next_line( p );
            }
            __set_select( s_start, s_end, 1 );
            break;
          case cmeUNINDENT:
            p = __line_start( __sel_start );
            end = __line_start( __sel_end );
            s_start = __sel_start; s_end = end;
            while( p < end )
            {
              if( __buf_char( p ) == ' ' )
              {
                __set_cur_ptr( p, 0 );
                __delete_range( p, p + 1, 0 );
                s_end--;
                end--;
              }
              p = __next_line( p );
            }
            __set_select( s_start, s_end, 1 );
            break;
          case cmeMARK0: case cmeMARK1: case cmeMARK2: case cmeMARK3: case cmeMARK4:
          case cmeMARK5: case cmeMARK6: case cmeMARK7: case cmeMARK8: case cmeMARK9:
            marks[ev.CMD_CODE - cmeMARK0] = __cur_ptr;
            break;
          case cmeGMARK0: case cmeGMARK1: case cmeGMARK2: case cmeGMARK3: case cmeGMARK4:
          case cmeGMARK5: case cmeGMARK6: case cmeGMARK7: case cmeGMARK8: case cmeGMARK9:
            if( marks[ev.CMD_CODE - cmeGMARK0] != (uint) -1 )
              __set_cur_ptr( marks[ev.CMD_CODE - cmeGMARK0], select_mode );
            track_cursor( !cursor_visible() );
            break;
          default:
            return;
        }
      done:
        track_cursor( 0 );
      ret:
        update( 0, 0, 0 );
        handled( ev );
    }
  }

  void Teditor::convert_event( Tevent &ev )
  {
    static int key_state = 0;
    uint key;

    if( ev.code == evKEY_PRESS )
    {
      key = ev.ASCII;
      if( key_state )
      {
        if( ( lo( key ) >= 0x01 ) && ( lo( key ) <= 0x1A ) ) key += 0x40;
        if( ( lo( key ) >= 'a' ) && ( lo( key ) <= 'z' ) ) key -= ( 'a' - 'A' );
      }
      key = scan_key_map( key_map[ key_state ], key );
      key_state = 0;
      if( key )
        if( key >= 0xFF00 )
        {
          key_state = lo( key );
          handled( ev );
        }
        else
          if( cdisabled( key ) )
            handled( ev );
          else
          {
            ev.code = evCOMMAND;
            ev.CMD_CODE = key;
          }
    }
  }

  void Teditor::update( uint cur_ptr_inserted, int bytes_inserted, int lines_inserted )
  {
    uint p;

    curs_type = ( editor_flags & efOVERWRITE )? 2 : 1;
    curs_x = ( __cur_pos_x + __cur_after_eol ) - delta_x;
    curs_y = __cur_pos_y - delta_y;
    if( owner == NULL ) return;
    if( state( isFOCUSED ) )
    {
      update_commands();
      cur_pos_x_svd = __cur_pos_x + __cur_after_eol;
      cur_pos_y_svd = __cur_pos_y;
      cur_ptr_svd   = __cur_ptr;
    }
    if( bytes_inserted != 0 )
    {
      if( !state( isFOCUSED ) && ( draw_ptr >= cur_ptr_inserted ) )
      {
        draw_ptr      = max( 0, draw_ptr      + bytes_inserted );
        draw_line     = max( 0, draw_line     + lines_inserted );
        delta_y       = max( 0, delta_y       + lines_inserted );
        cur_ptr_svd   = max( 0, cur_ptr_svd   + bytes_inserted );
        cur_pos_y_svd = max( 0, cur_pos_y_svd + lines_inserted );
      }
      for( p = 0; p < 10; p++ )
        if( ( marks[p] != (uint) -1 ) && ( marks[p] >= cur_ptr_inserted ) )
          marks[p] = max( cur_ptr_inserted, marks[p] + bytes_inserted );
    }
    if( __booleans & ebDIRTY )
      redraw();
    else
      if( indicator != NULL ) indicator->redraw();
  }

  void Teditor::update_commands( void )
  {
    boolean hs, clp;

    clp = ( ( clipboard != NULL ) && ( clipboard != text_editor ) );
    hs = __has_selection();
    cstate( cmCUT, clp && hs );
    cstate( cmCOPY, clp && hs );
    cstate( cmPASTE, clp && clipboard->has_selection() );
    cstate( cmUNDO, __del_count || __ins_count );
    cstate( cmCLEAR, hs );
    cstate( cmeINDENT, hs );
    cstate( cmeUNINDENT, hs );
    for( uint i = cmUSER50; i <= cmUSER99; i++ )
      cenable( i );
    cenable( cmeFIND );
    cenable( cmeREPLACE );
    cenable( cmeSEARCH_AGAIN );
    cenable( cmeGOTO_LINE );
    cenable( cmPRINT );
  }

  #pragma off( unreferenced )
  uint Teditor::editor_dialog( int dialog, void *info )
  {
    return cmCANCEL;
  }
  #pragma on( unreferenced )

  void Teditor::do_search_replace( void )
  {
    uint i;
    int xx, yy;

    if( *find_str == 0 ) return;
    do
    {
      i = cmCANCEL;
      if( !__search( find_str, editor_flags ) )
      {
        if( ( editor_flags & ( efREPLACE_ALL + efDO_REPLACE ) ) != ( efREPLACE_ALL + efDO_REPLACE ) )
          editor_dialog( edSEARCH_FAILED, find_str );
      }
      else
      {
        track_cursor( !cursor_visible() );
        if( editor_flags & efDO_REPLACE )
        {
          i = cmYES;
          if( editor_flags & efPROMPT_ON_REPLACE )
          {
            xx = (int) ( __cur_pos_x - delta_x );
            yy = (int) ( __cur_pos_y - delta_y );
            make_global( xx, yy, xx, yy );
            i = editor_dialog( edREPLACE_PROMPT, &yy );
          }
          if( i == cmYES )
          {
            __insert_string( replace_str, 0 );
            track_cursor( 0 );
            item_acted = this;
          }
        }
      }
    }
    while( ( i != cmCANCEL ) && ( editor_flags & efREPLACE_ALL ) );
  }


/*
Tmemo
*/

//public:

  Tmemo::Tmemo( char *_buffer, uint buf_size, int _xl, int _yl ):
    Teditor( NEW( Ttext_editor( buf_size - 1, this ) ), _xl, _yl )
  {
    buffer = _buffer;
    __booleans &= ~( ebCAN_UNDO | ebCAN_GROW );
    __insert_string( buffer, 0 );
    __set_cur_ptr( 0, 0 );
  }

  Tmemo *memo( char *title, char *buffer, uint buf_size, int _xl, int _yl )
  {
    Tmemo *m;
    Thscroll_bar *hbar;
    Tvscroll_bar *vbar;
    Tbox *b;

    m = NEW( Tmemo( buffer, buf_size, _xl, _yl ) );
    vbar = NEW( Tvscroll_bar( m->yl, m->text_editor->limit_yl, m->delta_y ) );
    vbar->set_flags( sfHIDEABLE+sfHANDLE_KEYBOARD, 0 );
    vbar->icon_up  ->shortcut = kCTRL_W;
    vbar->icon_down->shortcut = kCTRL_Z;
    hbar = NEW( Thscroll_bar( m->xl+1, m->text_editor->limit_xl, m->delta_x ) );
    hbar->set_flags( sfHIDEABLE+sfHANDLE_KEYBOARD, 0 );
    hbar->delta = 1;
    m->put_in( vbar, m->xl, 0 );
    m->put_in( hbar, -1, m->yl );
    b = NEW( Tbox( title, m ) );
    b->resize( b->xl-2, b->yl-1 );
    m->put_in( b, -1, -1 );
    hspace(); vspace();
    put_item( m, b->xl, b->yl-1 );
    return m;
  }

//protected:

  void Tmemo::ok_item( void )
  {
    __get_text( buffer, 0, __buf_len );
    buffer[__buf_len] = 0;
  }


/*
Tfile_editor
*/

//public:

  static Ttext_editor *registrate_editor( Teditor *_this, char *file_name )
  {
    if( *file_name )
    {
      char fn[_MAX_PATH];
      fexpand( strcpy( fn, file_name ) );
      for( Tfile_editor *e=open_editors; e!=NULL; e=e->next_editor )
        if( !strcmp( e->__file_name, fn ) )
          return e->text_editor;
    }
#ifdef SYNTAXHILIGHT
    Tsyntax_editor *t = NEW( Tsyntax_editor( 0x1000, _this ) );
#else
    Ttext_editor *t = NEW( Ttext_editor( 0x1000, _this ) );
#endif
    t->set_name( file_name );
    return t;
  }

  Tfile_editor::Tfile_editor( char *_file_name, int _xl, int _yl ):
    Teditor( registrate_editor( this, _file_name ), _xl, _yl )
  {
    next_editor = open_editors;
    open_editors = this;
    if( *_file_name )
    {
      get_date_time( _file_name, __date, __time );
      if( *__file_name && (__assoc_number==1) && state( isVALID ) )
        set_state( isVALID, __load() );
    }
  }

  Tfile_editor::~Tfile_editor( void )
  {
    Tfile_editor *e = open_editors;
    if( e == this )
      open_editors = next_editor;
    else
    {
      while( e->next_editor != this )
        e = e->next_editor;
      e->next_editor = next_editor;
    }
    if( __assoc_number>1 )
      for( e=open_editors; e!=NULL; e=e->next_editor )
        if( e->text_editor==text_editor )
        {
          text_editor->editor = e;
          break;
        }
  }

  boolean Tfile_editor::valid( uint command )
  {
    int d;

    if( !Teditor::valid( command ) ) return 0;
    if( command==cmVALID ) return 1;
    if( (__booleans&ebMODIFIED) && (__assoc_number==1 || command!=cmDONE) )
    {
      if( *__file_name==0 ) d = edSAVE_UNTITLED; else d = edSAVE_MODIFY;
      switch( editor_dialog( d, __file_name ) )
      {
        case cmYES:
          return save();
        case cmNO:
          __booleans &= ~ebMODIFIED;
          return 1;
        case cmCANCEL:
          return 0;
      }
    }
    return 1;
  }

  boolean Tfile_editor::save( void )
  {
    if( *__file_name==0 ) return save_as();
    return __save();
  }

  boolean Tfile_editor::save_as( void )
  {
    char buf[_MAX_PATH];
    strcpy( buf, __file_name );
    if( editor_dialog( edSAVE_AS, buf )==cmCANCEL ) return 0;
    if( strcmp( buf, __file_name ) )
    {
      if( __assoc_number>1 )
      {
        Ttext_editor *old = text_editor;
        __assoc_number--;
#ifdef SYNTAXHILIGHT
        text_editor = NEW( Tsyntax_editor( old->buf_size, this ) );
#else
        text_editor = NEW( Ttext_editor( old->buf_size, this ) );
#endif
        __assoc_number++;
        __insert_text( old->edit_buffer, old->cur_ptr, 0 );
        __insert_text( old->edit_buffer+old->cur_ptr+old->gap_len, old->buf_len-old->cur_ptr, 0 );
        __set_cur_ptr( old->cur_ptr, 0 );
        scroll_to( delta_x, delta_y );
        if( !old->assoc_number ) DELETE( old );
      }
      __set_name( buf );
      if( owner!=NULL && text_editor!=clipboard )
      {
        ((Tedit_window *)owner)->set_title( __file_name );
        ((Tedit_window *)owner)->vbar->count = &__limit_yl;
        ((Tedit_window *)owner)->vbar->beg_print = &delta_y;
        ((Tedit_window *)owner)->hbar->count = &__limit_xl;
        ((Tedit_window *)owner)->hbar->beg_print = &delta_x;
      }
    }
    boolean result = __save();
    if( text_editor==clipboard ) __set_name( "" );
    return result;
  }

//protected:

  void Tfile_editor::event_handler( Tevent &ev )
  {
    uint d, t;
    char s[_MAX_PATH];

    Teditor::event_handler( ev );
    if( ev.code == evCOMMAND )
      switch( ev.CMD_CODE )
      {
        case cmSAVE:
          save();
          handled( ev );
          break;
        case cmSAVE_ALL:
          if( __booleans & ebMODIFIED ) save();
          break;
        case cmSAVE_AS:
          save_as();
          handled( ev );
        case cmSTAMP_CHECK:
          if( __date && *__file_name )
          {
            if( get_date_time(__file_name,d,t) && (__date!=d || __time!=t) )
            {
              min_path( strcpy( s, __file_name ) );
              short_path( s, 30 );
#ifdef CYR
              if( ynh( "/   \"%s\"  .\n\n   ?", s ) )
#else
              if( ynh( "Date/time stamp has been changed for file \"%s\".\n\nReload from disk?", s ) )
#endif
              {
                __load();
                redraw();
                track_cursor( 0 );
              }
              __date = d;
              __time = t;
            }
          }
      }
  }

#ifndef NOHIST
  static void *_find_history = NULL;
#endif
  static uint find_dialog( void )
  {
#ifndef NOHELP
    _help( htD_FIND );
#endif
#ifdef CYR
    dialog( "" );
#else
    dialog( "Find" );
#endif
#ifndef NOHIST
    if( _find_history ) _history( _find_history ); else _find_history = _history( find_dialog );
#endif
#ifdef CYR
    input( "|~  :", find_str, 80, 20 );
    check( "|~  / ", editor_flags, efCASE_SENSITIVE );
    check( "  |~", editor_flags, efWHOLE_WORDS_ONLY );
#else
    input( "|~Find what:", find_str, 80, 20 );
    check( "|~Case sensitive", editor_flags, efCASE_SENSITIVE );
    check( "|~Whole words only", editor_flags, efWHOLE_WORDS_ONLY );
#endif
    if( bkch() && *find_str ) return cmOK;
    return cmCANCEL;
  }

  static uint replace_dialog( void )
  {
#ifndef NOHELP
    _help( htD_REPLACE );
#endif
#ifdef CYR
    dialog( "" );
#else
    dialog( "Replace" );
#endif
#ifndef NOHIST
    if( _find_history ) _history( _find_history ); else _find_history = _history( replace_dialog );
#endif
#ifdef CYR
    input( "  |~:", find_str, 80, 20 );
#else
    input( "|~Find what:   ", find_str, 80, 20 );
#endif
#ifndef NOHIST
    _history( (char *) replace_dialog + 1 );
#endif
#ifdef CYR
    input( "  |~: ", replace_str, 80, 20 );
    check( "|~  / ", editor_flags, efCASE_SENSITIVE );
    check( "  |~", editor_flags, efWHOLE_WORDS_ONLY );
    check( "  |~ ", editor_flags, efREPLACE_ALL );
    check( "|~  ", editor_flags, efPROMPT_ON_REPLACE );
#else
    input( "|~Replace with:", replace_str, 80, 20 );
    check( "|~Case sensitive", editor_flags, efCASE_SENSITIVE );
    check( "|~Whole words only", editor_flags, efWHOLE_WORDS_ONLY );
    check( "Replace |~all occurences", editor_flags, efREPLACE_ALL );
    check( "|~Prompt on replace", editor_flags, efPROMPT_ON_REPLACE );
#endif
    if( bkch() && *find_str ) return cmOK;
    return cmCANCEL;
  }

  static int goto_dialog( int max_line )
  {
    static int line_num = 1;

#ifndef NOHELP
    _help( htD_GOTO_LINE_NUMBER );
#endif
#ifdef CYR
    dialog( "   " );
#else
    dialog( "Go to line number" );
#endif
#ifndef NOHIST
    _history( goto_dialog );
#endif
#ifdef CYR
    iinput( "   :", line_num, 1, max_line );
#else
    iinput( "Enter new line number:", line_num, 1, max_line );
#endif
    if( bkch() ) return line_num;
    return 0;
  }

  void Tfile_editor::update_commands( void )
  {
    Teditor::update_commands();
    cenable( cmSAVE );
    cenable( cmSAVE_AS );
  }

  void Tfile_editor::update( uint cur_ptr_inserted, int bytes_inserted, int lines_inserted )
  {
    Teditor::update( cur_ptr_inserted, bytes_inserted, lines_inserted );
    if( editors_supervisor!=NULL && text_editor!=clipboard )
      editors_supervisor( __file_name, cur_ptr_inserted, bytes_inserted, lines_inserted );
  }

  uint Tfile_editor::editor_dialog( int dialog, void *info )
  {
    char s[256];

    if( ( dialog == edREAD_ERROR ) ||
        ( dialog == edWRITE_ERROR ) ||
        ( dialog == edCREATE_ERROR ) ||
        ( dialog == edSAVE_MODIFY ) ||
        ( dialog == edSAVE_UNTITLED ) )
    {
      strcpy( s, (char *) info );
      min_path( s );
      short_path( s, 30 );
    }
    switch( dialog )
    {
      case edOUT_OF_MEMORY:
        _terror();
#ifdef CYR
        ok( "     ." );
#else
        ok( "Not enough memory for this operation." );
#endif
        return cmCANCEL;
      case edREAD_ERROR:
        _terror();
#ifdef CYR
        ok( "   %s", s );
#else
        ok( "Can't read file %s", s );
#endif
        return cmCANCEL;
      case edWRITE_ERROR:
        _terror();
#ifdef CYR
        ok( "    %s", s );
#else
        ok( "Can't write file %s", s );
#endif
        return cmCANCEL;
      case edCREATE_ERROR:
        _terror();
#ifdef CYR
        ok( "    %s",s );
#else
        ok( "Can't create file %s",s );
#endif
        return cmCANCEL;
      case edSAVE_MODIFY:
        _tconfirm(); _iwarning();
#ifdef CYR
        return ync( " %s   .\n\n    ?", s );
#else
        return ync( "File %s has been modified.\n\nSave changes?", s );
#endif
      case edSAVE_UNTITLED:
        _tconfirm(); _iwarning();
#ifdef CYR
        return ync( "     ?" );
#else
        return ync( "Do you want to save untitled file?" );
#endif
      case edSAVE_AS:
        _new_file();
        if( filter_str != NULL ) _filters( filter_str );
#ifdef CYR
        return get_file( " ", (char *) info );
#else
        return get_file( "Save file as", (char *) info );
#endif
      case edFIND:
        __get_word_str( find_str, 80 );
        return find_dialog();
      case edSEARCH_FAILED:
#ifdef CYR
        ok( " \"%s\"   .", info );
#else
        ok( "String \"%s\" not found.", info );
#endif
        return cmCANCEL;
      case edREPLACE:
        __get_word_str( find_str, 80 );
        return replace_dialog();
      case edREPLACE_PROMPT:
        _pstandard(); _iquestion();
        if( *( (int *) info ) > desktop_y + ( desktop_yl>>1 ) )
          _dialog_xy( xCENTER, ( desktop_yl>>2 ) - 4 );
        else
          _dialog_xy( xCENTER, desktop_yl - ( desktop_yl>>2 ) - 5 );
#ifdef CYR
        return ync( "     ?" );
#else
        return ync( "Replace this occurence?" );
#endif
      case edGOTO_LINE:
        return goto_dialog( *( (uint *) info ) );
    }
    return cmCANCEL;
  }


/*
Tindicator
*/

//public:

  class Tindicator: public Titem
  {
    public:
      Tindicator( void );

    protected:
      virtual void draw( void );
  };

  Tindicator::Tindicator( void ):
    Titem( 1, 1 )
  {
    char s[256];

    set_flags( ifSELECTABLE, 0 );
    ltoa( MAX_LINE_LENGTH, s, 10 );
    resize( 9 + strlen( s ), 1 );
    set_events_mask( ~evCOMMAND, 0 );
    drag_mode = dmDRAG_VER;
  }

//protected:

  void Tindicator::draw( void )
  {
    char c, m;
    Teditor *editor;

    editor = (Teditor *) owner;
    if( editor->__booleans & ebMODIFIED ) m = '*'; else m = ' ';
    if( owner->state( isACTIVE ) ) c = '1'; else c = '0';
    txtf( "|%c|i %c%5u:%u      ", c, m, editor->cur_pos_y_svd + 1, editor->cur_pos_x_svd + 1 );
  }


/*
Tedit_window
*/
  static Tindicator *indicator;

  Tedit_window::Tedit_window( char *fn, int _xl, int _yl ):
    Tfile_window( editor->delta_x, editor->__limit_xl,
                  editor->delta_y, editor->__limit_yl,
                  ( indicator = NEW( Tindicator ), indicator->xl ),
                  fn, editor = NEW( Tfile_editor( fn, _xl-i_sb_up_len-1, _yl-2 ) ) )
  {
    if( *editor->__file_name==0 )
#ifdef CYR
      set_title( "[ ]" );
#else
      set_title( "[Untitled]" );
#endif
    editor->grow_mode = gmGROW_BOTH;
    editor->put_in( indicator, -1, editor->yl );
    editor->indicator = indicator;
    hbar->set_flags( sfHANDLE_KEYBOARD, 0 );
    vbar->set_flags( sfHANDLE_KEYBOARD, 0 );
  }

  Tedit_window::~Tedit_window( void )
  {
    _command_info( this, 0 );
    broadcast( cmeEDITOR_CLOSE );
  }

  void Tedit_window::save_status( Tedit_window_status &st )
  {
    if( state( isICONIZED ) )
    {
      st.icon_x = x;
      st.icon_y = y;
    }
    else
    {
      st.icon_x = icon_x;
      st.icon_y = icon_y;
    }
    if( state( isICONIZED|wsMAXIMIZED ) )
    {
      st.x  = x_saved;
      st.y  = y_saved;
      st.xl = xl_saved;
      st.yl = yl_saved;
    }
    else
    {
      st.x  = x;
      st.y  = y;
      st.xl = xl;
      st.yl = yl;
    }
    st.state = state_word;
    st.cur_pos_y = editor->__cur_pos_y;
    st.cur_pos_x = editor->__cur_pos_x + editor->__cur_after_eol;
    st.delta_x = editor->delta_x;
    st.delta_y = editor->delta_y;
    for( int i = 0; i < 10; i++ )
      st.marks[i] = editor->marks[i];
    strcpy( st.file_name, editor->__file_name );
  }

  void Tedit_window::restore_status( Tedit_window_status &st )
  {
    set_state( st.state, 0 );
    drag( st.x, st.y );
    resize( st.xl, st.yl );
    icon_x = st.icon_x;
    icon_y = st.icon_y;
    set_state( st.state, 1 );
    editor->scroll_to( st.delta_x, st.delta_y );
    editor->__set_cur_xy( st.cur_pos_x, st.cur_pos_y, 0 );
    for( int i = 0; i < 10; i++ )
      editor->marks[i] = st.marks[i];
    editor->cur_ptr_svd   = editor->__cur_ptr;
    editor->cur_pos_x_svd = editor->__cur_pos_x;
    editor->cur_pos_y_svd = editor->__cur_pos_y;
  }

  Tedit_window *edit_file( char *fn )
  {
    Tfile_editor *f;
    char buf[_MAX_PATH];

    fexpand( strcpy( buf, fn ) );
    f = open_editors;
    while( f != NULL )
    {
      if( !strcmp(buf,f->__file_name) )
      {
#ifndef NOHELP
        __help();
#endif
        f->owner->focus();
        f->owner->set_state( isICONIZED, 0 );
        return (Tedit_window *) f->owner;
      }
      f = f->next_editor;
    }
    return open_file( fn );
  }

  Tedit_window *open_file( char *fn )
  {
    Tedit_window *ew;

    _context( cxEDITOR );
    ew = NEW( Tedit_window( fn, desktop->xl, desktop->yl ) );
    if( !is_valid( ew ) ) return NULL;
    insert_window( ew, 0, 0 );
    return ew;
  }

  Tedit_window *open_file( Tedit_window_status &st )
  {
    Tedit_window *ew=open_file( st.file_name );
    if( ew != NULL ) ew->restore_status( st );
    return ew;
  }

//protected:

  void Tedit_window::initialize( void )
  {
    Tfile_window::initialize();
    _command_info( this, 0 );
    broadcast( cmeEDITOR_OPEN );
  }


/*
Tclipboard_window
*/
  class Tclipboard_window: public Tscroll_window
  {
    public:
      Tclipboard_window( void );
      virtual void set_state( uint _state_word, char enable );
      virtual boolean valid( uint command );
  };

  Tclipboard_window *clipboard_window;
  static Tfile_editor *tmp_clipboard;

  Tclipboard_window::Tclipboard_window( void ):
    Tscroll_window( tmp_clipboard->delta_x, tmp_clipboard->__limit_xl,
                    tmp_clipboard->delta_y, tmp_clipboard->__limit_yl,
                    ( indicator = NEW( Tindicator ), indicator->xl - 1 ),
                    "Clipboard",
                    tmp_clipboard = NEW( Tfile_editor( "", desktop->xl-i_sb_up_len-1, desktop->yl-2 ) ) )
  {
    clipboard = tmp_clipboard->text_editor;
    set_state( isHIDDEN, 1 );
    commands<<cmSAVE_AS;
    clipboard->booleans &= ~ebCAN_UNDO;
    tmp_clipboard->grow_mode = gmGROW_BOTH;
    tmp_clipboard->put_in( indicator, -1, tmp_clipboard->yl );
    tmp_clipboard->indicator = indicator;
    hbar->set_flags( sfHANDLE_KEYBOARD, 0 );
    vbar->set_flags( sfHANDLE_KEYBOARD, 0 );
  }

  void Tclipboard_window::set_state( uint _state_word, char enable )
  {
    Tscroll_window::set_state( _state_word, enable );
    if( _state_word & isHIDDEN )
      if( enable )
      {
        if( state( isSELECTED ) ) owner->tab_next( -1 );
        if( state( isSELECTED ) ) set_state( isSELECTED, 0 );
      }
      else
        focus();
  }

  boolean Tclipboard_window::valid( uint command )
  {
    if( command == cmDONE )
    {
      set_state( isHIDDEN, 1 );
      return 0;
    }
    return Tscroll_window::valid( command );
  }

  void show_clipboard( void )
  {
    if( clipboard_window != NULL )
    {
      clipboard_window->set_state( isHIDDEN, 0 );
      clipboard_window->set_state( isICONIZED, 0 );
    }
  }


void __init_editors( void )
{
  int i;

  *find_str = 0;
  *replace_str = 0;
  ~word_chars;
  for( i = '0'; i <= '9'; i++ )
    word_chars << i;
  for( i = 'A'; i <= 'Z'; i++ )
    word_chars << i;
  for( i = 'a'; i <= 'z'; i++ )
    word_chars << i;
  word_chars << '_';
#ifdef CYR
  for( i = ''; i <= ''; i++ )
    word_chars << i;
  for( i = ''; i <= ''; i++ )
    word_chars << i;
#endif
#ifdef SYNTAXHILIGHT
    if( scr_bw )
    {
#ifdef CPPHILIGHT
      static char bw_cpp_colors[16] = { 7,7,15,7,15,7,7,7 };
      memcpy( cpp_syntax_colors, bw_cpp_colors, sizeof(cpp_syntax_colors) );
#endif
#ifdef ASMHILIGHT
      static char bw_asm_colors[16] = { 7,7,15,7,15,7,7,7,7 };
      memcpy( asm_syntax_colors, bw_asm_colors, sizeof(asm_syntax_colors) );
#endif
#ifdef A51HILIGHT
      static char bw_a51_colors[16] = { 7,7,15,7,15,7,7,7,7 };
      memcpy( a51_syntax_colors, bw_a51_colors, sizeof(a51_syntax_colors) );
#endif
    }
#endif
#ifdef CPPHILIGHT
  ~cpp_symbols;
  cpp_symbols << '{' << '}' << '[' << ']' << '(' << ')' << ';' << ','
              << '.' << '<' << '>' << '=' << '&' << '|' << '^' << '~'
              << '!' << '+' << '-' << '*' << '/' << '?' << ':' << '%';
#endif
#ifdef ASMHILIGHT
  ~asm_symbols;
  asm_symbols << '[' << ']' << '{' << '|' << '}' << '$' << '%' << '('
              << ')' << '*' << '+' << ',' << '-' << '.' << '/' << ':'
              << '<' << '=' << '>' << '?' << '@';
#endif
#ifdef A51HILIGHT
  ~a51_symbols;
  a51_symbols << '[' << ']' << '{' << '|' << '}' << '$' << '%' << '('
              << ')' << '*' << '+' << ',' << '-' << '.' << '/' << ':'
              << '<' << '=' << '>' << '?' << '@' << '#';
#endif
  _context( cxEDITOR );
  clipboard_window = NEW( Tclipboard_window );
  desktop->put_in( clipboard_window, 0, 0 );
}
