/* Stub code for the graphic primitives */

#include <stdlib.h>
#include <graphics.h>
#include <dos.h>
#include <conio.h>

#include "mlvalues.h"
#include "alloc.h"
#include "memory.h"
#include "fail.h"

#define Unit Atom(0)

static int graph_opened = 0;
static int graph_driver, graph_mode;
static int mx, my;

#define Cvt(y) (my - (y))

static void graphic_failure(msg)
     char * msg;
{
  raise_with_string(GRAPHIC_FAILURE_EXN, msg);
}

static void check_graph_error()
{
  int errorcode;
  errorcode = graphresult();
  if (errorcode != grOk) graphic_failure(grapherrormsg(errorcode));
}

struct graph_mode {
  char * name;
  char driver;
  char mode;
};

static struct graph_mode mode_table[] = {
  "cga", CGA, CGAC1,
  "ega_low", EGA, EGALO,
  "ega_high", EGA, EGAHI,
  "vga_low", VGA, VGALO,
  "vga_medium", VGA, VGAMED,
  "vga_high", VGA, VGAHI
};

struct rgb {
  unsigned char r;
  unsigned char g;
  unsigned char b;
};

value gr_close_graph();

value gr_open_graph(mode)	/* ML */
     value mode;
{
  int i;
  char * pathtodrivers;
  
  if (graph_opened) gr_close_graph();
  graph_driver = DETECT;
  graph_mode = 0;
  for (i = 0; i < sizeof(mode_table) / sizeof(struct graph_mode); i++) {
    if (strcmp(String_val(mode), mode_table[i].name) == 0) {
      graph_driver = mode_table[i].driver;
      graph_mode = mode_table[i].mode;
      break;
    }
  }
  (pathtodrivers = getenv("BGIDRIVERS")) != NULL ||
     (pathtodrivers = getenv("CAMLLIB")) != NULL ||	  
     (pathtodrivers = "");
  initgraph(&graph_driver, &graph_mode, pathtodrivers);
  check_graph_error();
  graph_opened = 1;
  mx = getmaxx();
  my = getmaxy();
  moveto(0, my);
  return Unit;
}

value gr_close_graph()	/* ML */
{
  if (graph_opened) closegraph();
  graph_opened = 0;
  return Unit;
}

value gr_clear_graph()	/* ML */
{
  cleardevice();
  return Unit;
}

value gr_size_x()	/* ML */
{
  return Val_long(mx+1);
}

value gr_size_y()	/* ML */
{
  return Val_long(my+1);
}

static int color_from_rgb(rgb)
     unsigned long rgb;
{
  int r, g, b, c;

  r = (rgb >> 16) & 0xFF;
  g = (rgb >> 8) & 0xFF;
  b = rgb & 0xFF;
  if (graph_driver == CGA) {    /* 4 color mode */
    if (g >= 0x80 && r >= 0x80)
      c = CGA_WHITE;
    else if (g >= 0x80)
      c = CGA_LIGHTCYAN;
    else if (r >= 0x80)
      c = CGA_LIGHTMAGENTA;
    else
      c = 0;
  } else {                      /* 16 color mode */
    c = 0;
    if (b >= 0x80) { c |= BLUE; b -= 0x80; }
    if (g >= 0x80) { c |= GREEN; g -= 0x80; }
    if (r >= 0x80) { c |= RED; r -= 0x80; }
    if (r + g + b >= 0x40 + 0x40 + 0x40) c |= DARKGRAY;
  }
  return c;
}

static unsigned long table_color4[] = {
  0x000000,                     /* black */
  0x00FFFF,                     /* light cyan */
  0xFF00FF,                     /* light magenta */
  0xFFFFFF                      /* white */
};

static unsigned long table_color16[] = {
  0x000000,                     /* black */
  0x0000FF,                     /* blue */
  0x00FF00,                     /* green */
  0x00C0C0,                     /* cyan */
  0xFF0000,                     /* red */
  0xC000C0,                     /* magenta */
  0xC0C000,                     /* brown */
  0xAAAAAA,                     /* light gray */
  0x555555,                     /* dark gray */
  0x2020FF,                     /* light blue */
  0x20FF20,                     /* light green */
  0x20C0C0,                     /* light cyan */
  0xFF2020,                     /* light red */
  0xC020C0,                     /* light magenta */
  0xC0C020,                     /* yellow */
  0xFFFFFF                      /* white */
};

static unsigned long rgb_from_color(c)
     int c;
{
  if (graph_driver == CGA)
    return table_color4[c];
  else
    return table_color16[c];
}

value gr_set_color(color)	/* ML */
     value color;
{
  int c;

  c = color_from_rgb(Long_val(color));
  setcolor(c);
  setfillstyle(SOLID_FILL, c);
  return Unit;
}

value gr_plot(x, y)	/* ML */
     value x, y;
{
  putpixel(Int_val(x), Cvt(Int_val(y)), getcolor());
  return Unit;
}

value gr_point_color(x, y)	/* ML */
     value x, y;
{
  int c;

  c = getpixel(Int_val(x), Cvt(Int_val(y)));
  return Val_long(rgb_from_color(c));
}

value gr_moveto(x, y)	/* ML */
     value x, y;
{
  moveto(Int_val(x), Cvt(Int_val(y)));
  return Unit;
}

value gr_current_point()	/* ML */
{
  value res;
  res = alloc_tuple(2);
  Field(res, 0) = Val_int(getx());
  Field(res, 1) = Val_int(Cvt(gety()));
  return res;
}

value gr_lineto(x, y)	/* ML */
     value x, y;
{
  lineto(Int_val(x), Cvt(Int_val(y)));
  return Unit;
}

value gr_draw_arc(argv, argc)	/* ML */
     value * argv;
     int argc;
{
  ellipse(Int_val(argv[0]),
          Cvt(Int_val(argv[1])),
          360 - Int_val(argv[5]),
          360 - Int_val(argv[4]),
          Int_val(argv[2]),
          Int_val(argv[3]));
  return Unit;
}

value gr_set_line_width(v)	/* ML */
     value v;
{
  setlinestyle(SOLID_LINE, 0, Int_val(v) <= 1 ? NORM_WIDTH : THICK_WIDTH);
  return Unit;
}

static value draw_text(s)
     char * s;
{
  int x, y, w, h;
  x = getx();
  y = gety();
  w = textwidth(s);
  h = textheight(s);
  outtextxy(x, y - h + 1, s);
  moveto(x + w, y);
  return Unit;
}

value gr_draw_char(c)	/* ML */
     value c;
{
  char s[2];
  s[0] = Int_val(c);
  s[1] = 0;
  draw_text(s);
  return Unit;
}

value gr_draw_string(s)	/* ML */
     value s;
{
  draw_text(String_val(s));
  return Unit;
}

value gr_text_size(s)	/* ML */
     value s;
{
  value res;
  res = alloc_tuple(2);
  Field(res, 0) = Val_int(textwidth(String_val(s)));
  Field(res, 1) = Val_int(textheight(String_val(s)));
  return res;
}

value gr_fill_rect(vx, vy, vw, vh)	/* ML */
     value vx, vy, vw, vh;
{
  int x, y, w, h;
  x = Int_val(vx);
  y = Int_val(vy);
  w = Int_val(vw);
  h = Int_val(vh);
  y = Cvt(y + h - 1);
  bar(x, y, x + w - 1, y + h - 1);
  return Unit;
}

value gr_fill_arc(argv, argc)	/* ML */
     value * argv;
     int argc;
{
  sector(Int_val(argv[0]),
          Cvt(Int_val(argv[1])),
          360 - Int_val(argv[5]),
          360 - Int_val(argv[4]),
          Int_val(argv[2]),
          Int_val(argv[3]));
  check_graph_error();
  return Unit;
}

value gr_fill_poly(v)	/* ML */
     value v;
{
  int numpoints;
  int * points;
  int i;
  int * p;

  numpoints = Wosize_val(v);
  points = (int *) stat_alloc(numpoints * 2 * sizeof(int));
  for (i = 0, p = points; i < numpoints; i++) {
    *p++ = Int_val(Field(Field(v, i), 0));
    *p++ = Int_val(Field(Field(v, i), 1));
  }
  fillpoly(numpoints, points);
  stat_free((char *) points);
  check_graph_error();
  return Unit;
}

static value new_bitmap(width, height)
     int width, height;
{
  unsigned bsize;
  mlsize_t wsize;

  bsize = imagesize(0, 0, width, height);
  if (bsize >= 0xFFFF - 3) graphic_failure("image too big");
  wsize = (bsize + 3) >> 2;
  if (wsize == 0)
    return Atom(Abstract_tag);
  if (wsize <= Max_young_wosize)
    return alloc(wsize, Abstract_tag);
  else
    return alloc_shr(wsize, Abstract_tag);
}

struct image {
  value width;                  /* Width, in pixels */
  value height;                 /* Height, in pixels */
  value data;                   /* Image data */
  value mask;                   /* Image mask (or Val_long(0) if empty) */
};

value gr_draw_image(image, vx, vy)	/* ML */
     struct image * image;
	value vx, vy;
{
  int x, y, h;

  h = Int_val(image->height);
  x = Int_val(vx);
  y = Cvt(Int_val(vy) + h - 1);
  if (Is_long(image->mask)) {
    putimage(x, y, (char *) image->data, COPY_PUT);
  } else {
    putimage(x, y, (char *) image->mask, AND_PUT);
    putimage(x, y, (char *) image->data, OR_PUT);
  }
  return Unit;
}

value gr_create_image(vw, vh)		/* ML */
	value vw, vh;
{
  int w, h;
  struct image * res;
  Push_roots(roots, 1)
#define bitmap roots[0]
  w = Int_val(vw);
  h = Int_val(vh);
  bitmap = new_bitmap(w, h);
  res = (struct image *) alloc_tuple(4);
  res->width = Val_int(w);
  res->height = Val_int(h);
  res->data = bitmap;
  res->mask = Val_int(0);
  Pop_roots();
  return (value) res;
#undef bitmap
}

value gr_blit_image(image, vx, vy)	/* ML */
	struct image * image;
	value vx, vy;
{
  int x, y, w, h;

  w = Int_val(image->width);
  h = Int_val(image->height);
  x = Int_val(vx);
  y = Cvt(Int_val(vy) + h - 1);
  getimage(x, y, x + w, y + h, (char *) image->data);
  return Atom(0);
}

value gr_make_image(color_matrix)	/* ML */
     value color_matrix;
{
  int width, height;
  int i, j;
  struct image * res;
  value row;
  long rgb;
  char * backup;
  unsigned bsize;
  int has_transp;
  Push_roots(roots, 3);
#define v roots[0]
#define bm_data roots[1]
#define bm_mask roots[2]

  v = color_matrix;
  height = Wosize_val(v);
  if (height == 0) {
    width = 0;
  } else {
    width = Wosize_val(Field(v, 0));
    for (i = 1; i < height; i++) {
      if (width != Wosize_val(Field(v, i)))
        graphic_failure("make_image: non-rectangular matrix");
    }
  }
  bsize = imagesize(0, 0, width, height);
  if (bsize == 0xFFFF) graphic_failure("make_image: too big");
  backup = stat_alloc(bsize);
  getimage(0, 0, width, height, backup);
  setviewport(0, 0, width, height, 1);
  clearviewport();
  
  has_transp = 0;
  for (j = 0; j < height; j++) {
    row = Field(v, j);
    for(i = 0; i < width; i++) {
      rgb = Long_val(Field(row, i));
      if (rgb == -1)
        has_transp = 1;
      else
        putpixel(i, j, color_from_rgb(rgb));
    }
  }
  bm_data = new_bitmap(width, height);
  getimage(0, 0, width, height, (char *) bm_data);

  if (has_transp) {
    clearviewport();
    for (j = 0; j < height; j++) {
      row = Field(v, j);
      for(i = 0; i < width; i++) {
        if (Long_val(Field(row, i)) == -1) putpixel(i, j, 0xF);
      }
    }
    bm_mask = new_bitmap(width, height);
    getimage(0, 0, width, height, (char *) bm_mask);
  } else {
    bm_mask = Val_int(0);
  }
  putimage(0, 0, backup, COPY_PUT);
  stat_free(backup);
  setviewport(0, 0, mx, my, 1);

  res = (struct image *) alloc_tuple(4);
  res->width = Val_int(width);
  res->height = Val_int(height);
  res->data = bm_data;
  res->mask = bm_mask;
  Pop_roots();
  return (value) res;
#undef bm_data
#undef bm_mask
}

static value alloc_int_vect(size)
     mlsize_t size;
{
  value res;
  mlsize_t i;
  
  if (size <= Max_young_wosize) {
    res = alloc(size, 0);
  } else {
    res = alloc_shr(size, 0);
  }
  for (i = 0; i < size; i++) {
    Field(res, i) = Val_long(0);
  }
  return res;
}

value gr_dump_image(image)	/* ML */
     struct image * image;
{
  int height, width, i, j;
  unsigned char * p;
  int c;
  unsigned bsize;
  char * backup;
  Push_roots(roots, 2);
#define img ((struct image *) roots[0])
#define matrix (roots[1])

  img = image;
  height = Int_val(img->height);
  width  = Int_val(img->width);
  bsize = imagesize(0, 0, width, height);
  if (bsize == 0xFFFF) graphic_failure("dump_image: too big");
  backup = stat_alloc(bsize);
  getimage(0, 0, width, height, backup);
  matrix = alloc_int_vect(height);
  for (i = 0; i < height; i++) {
    value row = alloc_int_vect(width);
    modify(&Field(matrix, i), row);
  }
  putimage(0, 0, (char *) img->data, COPY_PUT);  
  for (i = 0; i < height; i++)
    for (j = 0; j < width; j++)
      Field(Field(matrix, i), j) = Val_long(rgb_from_color(getpixel(j, i)));
  if (img->mask != Val_long(0)) {
    putimage(0, 0, (char *) img->mask, COPY_PUT);
    for (i = 0; i < height; i++)
      for (j = 0; j < width; j++) {
        c = getpixel(j, i);
        if (c > 0) Field(Field(matrix, i), j) = Val_long(-1);
      }
  }
  putimage(0, 0, backup, COPY_PUT);
  stat_free(backup);
  setviewport(0, 0, mx, my, 1);
  Pop_roots();
  return matrix;
#undef img
#undef matrix
}

static int mouse_initialized = 0;

static void init_mouse()
{
  union REGS r;

  if (!mouse_initialized) {
    r.x.ax = 0;
    int86(0x33, &r, &r);
    if (r.x.ax == 0) graphic_failure("no mouse installed");
    mouse_initialized = 1;
  }
}

static void get_mouse(x, y, buttons)
     int * x, * y, * buttons;
{
  union REGS r;
  init_mouse();
  r.x.ax = 3;
  int86(0x33, &r, &r);
  *x = r.x.cx;
  *y = r.x.dx;
  *buttons = r.x.bx != 0;
}

static void show_mouse(flag)
	int flag;
{
  union REGS r;
  init_mouse();
  r.x.ax = flag ? 1 : 2;
  int86(0x33, &r, &r);
}

value gr_wait_event(events)     /* ML */
     value events;
{
  int button_down, button_up, keypress, motion, poll, mouse;
  value res;
  int x, y, b, kbd, c, x0, y0, b0;
  
  enter_blocking_section();
  for (button_down = button_up = keypress = motion = poll = 0;
       Tag_val(events) == 1;
       events = Field(events, 1)) {
    switch (Tag_val(Field(events, 0))) {
      case 0: button_down = 1; break;
      case 1: button_up = 1; break;
      case 2: keypress = 1; break;
      case 3: motion = 1; break;
      case 4: poll = 1; break;
    }
  }
  mouse = motion | button_down | button_up;
  if (poll) {
    get_mouse(&x, &y, &b);
    kbd = kbhit();
    c = 0;
  } else {
    if (mouse) { show_mouse(1); get_mouse(&x0, &y0, &b0); }
    kbd = 0;
    c = 0;
    while (1) {
      if (mouse) {
        get_mouse(&x, &y, &b);
	if (motion && (x != x0 || y != y0)) break;
	if (button_down && b0 == 0 && b == 1) break;
	if (button_up && b0 == 1 && b == 0) break;
        b0 = b;
      }
      if (keypress) {
        if (kbhit()) { kbd = 1; c = getch(); break; }
      }
    }
    if (mouse) show_mouse(0);
  }
  leave_blocking_section();
  res = alloc_tuple(5);
  Field(res, 0) = Val_int(x);
  Field(res, 1) = Val_int(Cvt(y));
  Field(res, 2) = Atom(b);
  Field(res, 3) = Atom(kbd);
  Field(res, 4) = Val_int(c);
  return res;
}
  
value gr_sound(freq, dur)	/* ML */
     value freq, dur;
{
  enter_blocking_section();
  sound(Int_val(freq));
  delay(Int_val(dur));
  nosound();
  leave_blocking_section();
  return Unit;
}
