/* MOUSE.I: Source code for mouse interface functions.             */
/*    Must #include dos.h before this #include to define the       */
/*    register set used to pass args to device driver              */
/* --------------------------------------------------------------- */
/* Define int for mouse device driver */
#define  callMDD int86(0x33, &inreg, &outreg)

/* Define sorting macros used locally */
#define  lower (x, y)  (x < y) ? x : y
#define  upper (x, y)  (x > y) ? x : y

/* STATIC REGISTERS USED THROUGHOUT */
union REGS  inreg, outreg;

/* STRUCTURES USED BY THESE FUNCTIONS */
typedef struct {
  int exists,                           /* TRUE if mouse is present */
      nButtons;                           /* # of buttons on mouse */
} resetRec;                                  /* returned by mReset */

typedef struct {
  int buttonStatus,       /* bits 0-2 on if corresp button is down */
      opCount,                  /* # times button has been clicked */
      column, row;                                     /* position */
} locRec;                 /* returned by mPos, mPressed, mReleased */

typedef struct {
  int hCount,                           /* net horizontal movement */
  vCount;                                 /* net vertical movement */
} moveRec;                                  /* returned by mMotion */
/* --------------------------------------------------------------- */
/* Following are implementations of the Microsoft mouse functions  */

resetRec *mReset ()
/* Resets mouse to default state. Returns pointer to a structure   */
/* indicating whether or not mouse is installed and, if so, how    */
/* many buttons it has.                                            */
/* Always call this function during program initialization.        */
{
static resetRec  m;

  inreg.x.ax = 0;                                    /* function 0 */
  callMDD;
  m.exists = outreg.x.ax;
  m.nButtons = outreg.x.bx;
  return ( &m );
} /* ------------------------ */
void mShow (void)
/* Makes the mouse cursor visible. Don't call if cursor is already */
/* visible, and alternate with calls to mHide.                     */
{
  inreg.x.ax = 1;                                    /* function 1 */
  callMDD;
} /* ------------------------ */
void mHide (void)
/* Makes mouse cursor invisible. Movement and button activity are  */
/* still tracked. Do not call if cursor is already hidden, and     */
/* alternate with calls to mShow                                   */
{
  inreg.x.ax = 2;                                    /* function 2 */
  callMDD;
} /* ------------------------ */
locRec *mPos (void)
/* Gets mouse cursor position and button status, returns pointer   */
/* to structure containing this info                               */
{
static locRec  m;

  inreg.x.ax = 3;                                    /* function 3 */
  callMDD;
  m.buttonStatus = outreg.x.bx;                   /* button status */
  m.column = outreg.x.cx;                        /* horiz position */
  m.row = outreg.x.dx;                            /* vert position */
  return (&m);
} /* ------------------------ */
void  mMoveto (int newCol, int newRow)
/* Move mouse cursor to new position */
{
  inreg.x.ax = 4;                                    /* function 4 */
  inreg.x.cx = newCol;
  inreg.x.dx = newRow;
  callMDD;
} /* ------------------------ */
locRec  *mPressed (int button)
/* Gets pressed info about named button: current status (up/down), */
/* times pressed since last call, position at most recent press.   */
/* Resets count and position info. Button 0 is left, 1 is right on */
/* Microsoft mouse.                                                */
/* Returns pointer to locRec structure containing info.            */
{
static locRec  m;

  inreg.x.ax = 5;                                    /* function 5 */
  inreg.x.bx = button;              /* request for specific button */
  callMDD;
  m.buttonStatus = outreg.x.ax;
  m.opCount = outreg.x.bx;
  m.column  = outreg.x.cx;
  m.row     = outreg.x.dx;
  return (&m);
} /* ------------------------ */
locRec  *mReleased (int button)
/* Same as mPressed, except gets released info about button */
{
static locRec  m;

  inreg.x.ax = 6;                                    /* function 6 */
  inreg.x.bx = button;              /* request for specific button */
  callMDD;
  m.buttonStatus = outreg.x.ax;
  m.opCount = outreg.x.bx;
  m.column  = outreg.x.cx;
  m.row     = outreg.x.dx;
  return (&m);
} /* ------------------------ */
void  mColRange (int hmin, int hmax)
/* Sets min and max horizontal range for mouse cursor. Moves   */
/* cursor inside range if outside when called. Swaps values if */
/* hmin and hmax are reversed.                                 */
{
  inreg.x.ax = 7;                                /* function 7 */
  inreg.x.cx = hmin;
  inreg.x.dx = hmax;
  callMDD;
} /* ------------------------ */
void  mRowRange (int vmin, int vmax)
/* Same as mHminmax, except sets vertical boundaries. */
{
  inreg.x.ax = 8;                                    /* function 8 */
  inreg.x.cx = vmin;
  inreg.x.dx = vmax;
  callMDD;
} /* ------------------------ */
void  mGraphCursor (int hHot, int vHot, unsigned maskSeg,
                    unsigned maskOfs)
/* Sets graphic cursor shape */
{
struct SREGS  seg;

  inreg.x.ax = 9;                                    /* function 9 */
  inreg.x.bx = hHot;                /* cursor hot spot: horizontal */
  inreg.x.cx = vHot;                  /* cursor hot spot: vertical */
  inreg.x.dx = maskOfs;
  seg.es = maskSeg;
  int86x (0x33, &inreg, &outreg, &seg);
} /* ------------------------ */
void  mTextCursor (int curstype, unsigned arg1, unsigned arg2)
/* Sets text cursor type, where 0 = software and 1 = hardware)     */
/* For software cursor, arg1 and arg2 are the screen and cursor    */
/*   masks.                                                        */
/* For hardware cursor, arg1 and arg2 specify scan line start/stop */
/*   i.e. cursor shape.                                            */
{
  inreg.x.ax = 10;                                  /* function 10 */
  inreg.x.bx = curstype;
  inreg.x.cx = arg1;
  inreg.x.dx = arg2;
  callMDD;
} /* ------------------------ */
moveRec  *mMotion (void)
/* Reports net motion of cursor since last call to this function   */
{
static moveRec  m;

  inreg.x.ax = 11;                                  /* function 11 */
  callMDD;
  m.hCount = _CX;                                /* net horizontal */
  m.vCount = _DX;                                  /* net vertical */
  return (&m);
} /* ------------------------ */
void mInstTask (unsigned mask, unsigned taskSeg, unsigned taskOfs)
/* Installs a user-defined task to be executed upon one or more    */
/*   mouse events specified by mask.                               */
{
struct SREGS  seg;

  inreg.x.ax = 12;                                  /* function 12 */
  inreg.x.cx = mask;
  inreg.x.dx = taskOfs;
  seg.es = taskSeg;
  int86x (0x33, &inreg, &outreg, &seg);
} /* ------------------------ */
void mLpenOn (void)
/* Turns on light pen emulation. This is the default condition.    */
{
  inreg.x.ax = 13;                                  /* function 13 */
  callMDD;
} /* ------------------------ */
void mLpenOff (void)
/* Turns off light pen emulation. */
{
  inreg.x.ax = 14;                                  /* function 14 */
  callMDD;
} /* ------------------------ */
void mRatio (int horiz, int vert)
/* Sets mickey-to-pixel ratio, where ratio is R/8. Default is 16   */
/*   for vertical, 8 for horizontal */
{
  inreg.x.ax = 15;                                  /* function 15 */
  inreg.x.cx = horiz;
  inreg.x.dx = vert;
  callMDD;
} /* ------------------------ */
