/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPUTIL.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/5/21
**
** DESCRIPTION: General IP utilities.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   IPUTIL.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPUTIL.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:24:28   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:15:10   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPUTIL.C_V   1.1   21 Nov 1990 14:24:28   etstjan  $";
#endif

#include      <stdio.h>
#include      <stdlib.h>
#include      <string.h>
#include      <power3.h>
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines */
#include      "ipcodes.h"              /* and ip return codes */
#include      "iplib.h"                /* include ip library */
#include      "ipif.h"                 /* interface layer */
#include      "iploop.h"               /* loopback interface */
#include      "ipether.h"              /* ethernet interface */
#include      "ipevents.h"             /* event processing */
#include      "iplayer.h"              /* general ip layer header file */

/* global variables */ 
IPSTAT        IPStat;                  /* structure with IP statistics  */
IPADDRESS     *HostRoutes;             /* host routing table            */
IPADDRESS     *NetRoutes;              /* network routing table         */
IPADDRESS     *DefaultRoutes;          /* default routing table         */
#ifdef DEBUGANY
  extern PWWIN  *IPWin;                /* IP report window              */
#endif

/**************************************************************
** NAME:        IPReport
** SYNOPSIS:    void IPReport(int ThisComm,
**                            CIPHEAD *ThisHeader,
**                            DATALINK *ThisPacket)
**              
** DESCRIPTION: IP report facility.
**              ThisHeader will be freed, if not NULL and
**              ThisPacket will be discarded, if not NULL. 
** RETURNS:
**            
**************************************************************/
void IPReport(int ThisComm, CIPHEAD *ThisHeader, DATALINK *ThisPacket)
{

  if (ThisComm == NOSPACE) {
    IPStat.TimesNoBuf++;
    }
  else {
#ifdef DEBUGANY
    pw_printf(IPWin, "\nIPReport. Code : %d", ThisComm);
#endif
    }

#ifdef DEBUGROUT
  if (ThisComm == ROUTFAIL) {
    pw_printf(IPWin, "\nBad Guy: 0x%08lx  Destination: 0x%08lx",
                      ntohl(ThisHeader->Source), ntohl(ThisHeader->Destin));
    }
#endif

  if (ThisHeader != NULL) {
    IPBufFree(ThisHeader);
    }
  if (ThisPacket != NULL) {
    DiscardAll(ThisPacket);
    }
}


/**************************************************************
** NAME:        GetFirstHop
** SYNOPSIS:    int GetFirstHop(IFDESCRIPTOR **FirstHopIf,
**                              ADDRESS_T *FirstHop,
**                              ADDRESS_T Destin, BYTE Tos,
**                              int *BCast, DIR_CHOICE Direct);
**
** DESCRIPTION: Finds the first IP host to which the packet
**              should be sent to reach host 'Destin'. The
**              address of the first IP host is returned in
**              *FirstHop, the address of the interface
**              descriptor through which the packet should
**              be sent in **FirstHopIf.
**              If the packet should be broadcasted by the
**              link layer, *BCast is set true and the
**              the interfaces through which the packet
**              should be sent are linked by 'BCastLink'
**              in the interface descriptor.
**              Direct determines if the packet should be
**              sent to a connected network only or if
**              intermediate hops may be used.
**
** RETURNS:     NO_ERR      --> everything just fine
**              NOT_ON_NET  --> not on connected network
**                              (Direct was 'DIRECT_ONLY')
**              NOT_REACHED --> no first hop address found
**************************************************************/
int GetFirstHop(IFDESCRIPTOR **FirstHopIf, ADDRESS_T *FirstHop,
                ADDRESS_T Destin, BYTE Tos,
                int *BCast, DIR_CHOICE Direct)
{
  IFDESCRIPTOR  *ListIf;
  IFDESCRIPTOR  *BrdCastIf;
  IPADDRESS     *ListAdd;
  ADDRESS_T     CurMask;

  /* first, test for limited broadcast */
  if ((*BCast = (Destin == LIMBRCAST) || (Destin == NONSTBRCAST)) != 0) {
    *FirstHopIf = FirstIf;
    for (ListIf = FirstIf; ListIf != NULL; ListIf = ListIf->NextIf) {
      ListIf->BCastLink = ListIf->NextIf;
      }
    return NO_ERR;
    }

  /* then, check the host routing table */
  ListAdd = HostRoutes;
  while ((ListAdd = IPAddressSearch(Destin, UNMASKED, ListAdd)) != NULL) {
    if (ListAdd->Flags & UP) {
      *FirstHopIf = ListAdd->AddressIf;
      if (ListAdd->Flags & HOST) {
        *FirstHop = Destin;
        return NO_ERR;
        }
      if (Direct == DIRECT_ONLY) {          /* first hop was a gateway */
        return NOT_ON_NET;
        }
      *FirstHop = ListAdd->FirstHop;
      return NO_ERR;
      }
    }

  /* no solution found yet, so try the network routing table */
  /* first, test for a directed or all subnets broadcast     */
  CurMask = GetNetMask(Destin);

  if ((*BCast = (((Destin & ~CurMask) == ~CurMask) ||
		 ((Destin & ~CurMask) == 0l))) != 0) {
    BrdCastIf = NULL;
    for (ListAdd = NetRoutes; ListAdd != NULL;
         ListAdd = ListAdd->NextAddress) {
      if (((ListAdd->HostAddress & CurMask) == (Destin & CurMask)) &&
          (ListAdd->Flags & HOST) && (ListAdd->Flags & UP)) {
        if (BrdCastIf == NULL) {
          BrdCastIf = *FirstHopIf = ListAdd->AddressIf;
          }
        else {
          BrdCastIf->BCastLink = ListAdd->AddressIf;
          BrdCastIf = ListAdd->AddressIf;
          }
        }
      }
    if (BrdCastIf != NULL) {
      BrdCastIf->BCastLink = NULL;
      return NO_ERR;
      }
    *BCast = 0;
    }

  /* not a directed broadcast to a connected network */
  for (ListAdd = NetRoutes;
       (ListAdd != NULL) &&
       (ntohl(ListAdd->HostAddress & ListAdd->NetMask) <=
        ntohl(Destin & ListAdd->NetMask));
       ListAdd = ListAdd->NextAddress) {
    if (((Destin & ListAdd->NetMask) ==
         (ListAdd->HostAddress & ListAdd->NetMask))
         && (ListAdd->Flags & UP)) {
      *FirstHopIf = ListAdd->AddressIf;
      if (ListAdd->Flags & HOST) {
        ListAdd->AddressIf->BCastLink = NULL;
        *BCast = (((Destin & ListAdd->HostMask) == ListAdd->HostMask) ||
                 ((Destin & ListAdd->HostMask) == 0l));
        *FirstHop = Destin;
        return NO_ERR;
        }
      if (Direct == DIRECT_ONLY) {
        return NOT_ON_NET;
        }
      *FirstHop = ListAdd->FirstHop;
      return NO_ERR;
      }
    }

  /* still no succes ---> take a default gateway */
  for (ListAdd = DefaultRoutes; ListAdd != NULL;
       ListAdd = ListAdd->NextAddress) {
    if (ListAdd->Flags & UP) {
      *FirstHopIf = ListAdd->AddressIf;
      if (ListAdd->Flags & HOST) {
        *FirstHop = Destin;
        return NO_ERR;
        }
      if (Direct == DIRECT_ONLY) {
        return NOT_ON_NET;
        }
      *FirstHop = ListAdd->FirstHop;
      return NO_ERR;
      }
    }        
  return NOT_REACHED;
}

/**************************************************************
** NAME:        AddRoute
** SYNOPSIS:    int AddRoute(NETHOST NetHost, ADDRESS_T Destin,
**                           ADDRESS_T FirstHop,
**                           ADDRESS_T NetMask);
**              
** DESCRIPTION: Adds a route for Destin to the routing table.
**              FirstHop is the first hop address for this
**              destination.
**              NetHost values:
**                - NETSTAT    : add static net route
**                - NETMODIFY  : add volatile net route
**                - HOSTSTAT   : add static host route
**                - HOSTMODIFY : add volatile host route
**
** RETURNS:     NO_ERR     -->   no error
**              NOT_FOUND  -->   FirstHop not on connected
**                               network
**              NOSPACE    -->   no buffer space left
**************************************************************/
int AddRoute(NETHOST NetHost, ADDRESS_T Destin,
             ADDRESS_T FirstHop, ADDRESS_T NetMask)
{
  IPADDRESS    *ResPoint;
  IPADDRESS    **AddQueue;
  IPADDRESS    *SourceIPAdd;
  ADDRESS_T    SubNetMask;
  USHORT       StFlag = UP;

  SubNetMask = NetMask - GetNetMask(Destin);

  switch(NetHost) {
    case NETSTAT :    StFlag |= STATIC;
    case NETMODIFY :  AddQueue = &NetRoutes;
                      break;
    case HOSTSTAT :   StFlag |= STATIC;
    case HOSTMODIFY : AddQueue = &HostRoutes;
                      break;
    default :         return ILL_REQUEST;
    }
  if (Destin == INADDR_ANY) {
    AddQueue = &DefaultRoutes;
    }

  if ((SourceIPAdd = IPAddressSearch(FirstHop, MASKED,
                                     HostAddresses)) == NULL) {
    return NOT_FOUND;
    }

  if ((ResPoint = AddIPAddress(Destin, GetNetMask(Destin), SubNetMask,
                               NULL, AddQueue)) == NULL) {
    return NOSPACE;
    }

  ResPoint->FirstHop = FirstHop;
  ResPoint->AddressIf = SourceIPAdd->AddressIf;
  if (SourceIPAdd->HostAddress == FirstHop) 
    ResPoint->Flags = HOST | StFlag;         /* on connected network */
  else 
    ResPoint->Flags = StFlag;                /* reachable via gateway */

  return NO_ERR;
}

/**************************************************************
** NAME:        ModRoute
** SYNOPSIS:    void ModRoute(CIPHEAD *IPHeader,
**                            CIPHEAD *RecIPHeader,
**                            ADDRESS_T NewAdd)
**             
** DESCRIPTION: Modifies a routing table as response to an
**              ICMP Redirect Message. Only the host routing
**              table can be modified.
**              If already an entry exists for this host, it
**              is modified only if it is not marked as
**              'STATIC'. If an existing entry is modified,
**              it is marked as 'MODIFIED'.
**              An entry is created or modified only if the
**              new gateway is on a connected network.
** RETURNS:    
**************************************************************/
void ModRoute(CIPHEAD *IPHeader, CIPHEAD *RecIPHeader,
              ADDRESS_T NewAdd)
{
  IPADDRESS     *CurRoute;
  IPADDRESS     *SourceIPAdd;
  ADDRESS_T     ThisMask;
  ADDRESS_T     OldFirstHop;
  IFDESCRIPTOR  *DummyIf;
  USHORT        StFlag = UP;
  int           Dummy;

  /* first, check if gateway at this moment is the gateway which sent */
  /* the redirect and if the proposed address is reachable.           */
  /* (rfc 1122, section 3.3.1.2)                                      */ 
  if ((GetFirstHop(&DummyIf, &OldFirstHop, RecIPHeader->Destin,
                   RecIPHeader->Tos, &Dummy, HOPS_ALLOWED) != NO_ERR) ||
      (OldFirstHop != IPHeader->Source) ||
      ((SourceIPAdd = IPAddressSearch(NewAdd, MASKED,
                                     HostAddresses)) == NULL)) {
    return;
    }

  ThisMask = GetNetMask(NewAdd);

  if ((CurRoute = IPAddressSearch(RecIPHeader->Destin, UNMASKED,
                                  HostRoutes)) != NULL) {
    if (CurRoute->Flags & STATIC) {
      return;
      }
    else {
      AddIPAddress(RecIPHeader->Destin, ThisMask, INADDR_ANY,
                   CurRoute->AddressIf, &HostRoutes);        
      StFlag |= MODIFIED;
      }
    }
  else {
    if ((CurRoute = AddIPAddress(RecIPHeader->Destin, ThisMask, INADDR_ANY,
                                 NULL, &HostRoutes)) == NULL) {
      IPReport(NOSPACE, NULL, NULL);
      return;
      }
    }
    
  CurRoute->FirstHop = NewAdd;
  CurRoute->AddressIf = SourceIPAdd->AddressIf;
  if (SourceIPAdd->HostAddress == NewAdd) {
    CurRoute->Flags = HOST | StFlag;         /* on connected network */
    }
  else {
    CurRoute->Flags = StFlag;                /* reachable via gateway */
    }
  IPStat.NrModRed++;
}
      
/**************************************************************
** NAME:        GetHostAdd
** SYNOPSIS:    int IPGetHostAdd(ADDRESS_T *HostAdd,
**                               ADDRESS_T DestAdd,
**                               BYTE Tos);
** 
** DESCRIPTION: Determines which host address to use for
**              reaching destination address DestAdd.
**              The host address is copied into HostAdd.
** RETURNS:     NO_ERR -->   no error
**              else   -->   error code
**************************************************************/
int IPGetHostAdd(ADDRESS_T *HostAdd, ADDRESS_T DestAdd, BYTE Tos)
{
  IFDESCRIPTOR *TempIf;
  int          RetCode;
  int          Dummy;

  if ((RetCode = GetFirstHop(&TempIf, HostAdd, DestAdd, Tos,
                             &Dummy, HOPS_ALLOWED)) != NO_ERR) {
    return RetCode;
    }
  *HostAdd = TempIf->FirstAdd->ThisAddress;
  return NO_ERR;
}

/**************************************************************
** NAME:        AddIPAddress
** SYNOPSIS:    IPADDRESS *AddIPAddress(
**                            ADDRESS_T NewAddress,
**                            ADDRESS_T NetMask,  
**                            ADDRESS_T SubNetMask,
**                            IFDESCRIPTOR *OwningIf,
**                            IPADDRESS **ListAdd);
** DESCRIPTION: Adds NewAddress to ListAdd.
**              Addresses are sorted when added. OwningIf
**              contains the address of the interface
**              descriptor which 'owns' the address.
**              When an  address already exists, it will
**              be updated. NetMask and SubNetMask contain
**              the network address mask and the subnetwork
**              address mask.
** RETURNS:     NULL    --> no buffer space
**              else    --> pointer to allocated structure
**************************************************************/
IPADDRESS *AddIPAddress(ADDRESS_T NewAddress, ADDRESS_T NetMask,
                        ADDRESS_T SubNetMask, IFDESCRIPTOR *OwningIf,
                        IPADDRESS **ListAdd)
{
  IPADDRESS   *CurAddress = *ListAdd;
  IPADDRESS   **PrevAddress = ListAdd;
  IPADDRESS   *NewAd;

  while ((CurAddress != NULL) &&
         (ntohl(NewAddress) > ntohl(CurAddress->HostAddress))) {
    PrevAddress = &(CurAddress->NextAddress);
    CurAddress = CurAddress->NextAddress;
    }
 
  /* address not found -> make new */
  if ((CurAddress == NULL) || (NewAddress != CurAddress->HostAddress) ||
      (CurAddress->AddressIf != OwningIf)) {
    if ((NewAd = IPBufGet(sizeof(IPADDRESS))) == NULL ) {
      return NULL;
      }
    *PrevAddress = NewAd;
    NewAd->NextAddress = CurAddress;
    }
  else {
    NewAd = CurAddress;
    }

  NewAd->HostAddress = NewAddress;
  NewAd->NetWorkPart = NewAddress & NetMask;
  NewAd->SubNetPart = (NetMask | SubNetMask) & NewAddress;
  NewAd->NetWorkMask = NetMask;
  NewAd->SubNetMask = SubNetMask;
  NewAd->NetMask = NetMask | SubNetMask;
  NewAd->HostMask = ~NetMask;
  NewAd->SubHostMask = ~(NetMask | SubNetMask);
  NewAd->AddressIf = OwningIf;

  return NewAd;
}

/**************************************************************
** NAME:        IPAddressSearch
** SYNOPSIS:    IPADDRESS *IPAddressSearch(
**                              ADDRESS_T Search_Add,
**                              MASKCHOICE MaskSearch,
**                              IPADDRESS *AddList);
** DESCRIPTION: Searches for address Search_Add in the
**              address list AddList. If the Address is
**              not found and 'MaskSearch' is true, another
**              addresses search will be performed with 
**              the IP Address masked with the NetMask before
**              comparing.
** RETURNS:     NULL  -->   Address not found
**              else  -->   Address of IFADDRESS structure
**************************************************************/
IPADDRESS *IPAddressSearch(ADDRESS_T Search_Add, MASKCHOICE MaskSearch,
                           IPADDRESS *ListAdd)
{
  IPADDRESS  *AddPoint = ListAdd;
  IPADDRESS  *MaskMatch = NULL;
  ADDRESS_T  Mask;

  for (AddPoint = ListAdd; AddPoint != NULL; AddPoint = AddPoint->NextAddress) {
    Mask = AddPoint->NetMask;
    if ((AddPoint->HostAddress & Mask) == (Search_Add & Mask)) {
      MaskMatch = AddPoint;
      }
    if (ntohl(AddPoint->HostAddress) >= ntohl(Search_Add)) {
      break;
      }
    }
  if (AddPoint == NULL) {
    return (MaskSearch == MASKED) ? MaskMatch : NULL;
    }
  if (MaskSearch == UNMASKED) {
    return (AddPoint->HostAddress) == (Search_Add) ? AddPoint : NULL;
    }
  return MaskMatch;
}


/**************************************************************
** NAME:        DotConvert
** SYNOPSIS:    int DotConvert(ADDRESS_T *AddStore,
**                             char *DotAddress);
**
** DESCRIPTION: Converts an IP Address in dotted decimal
**              notation to an ADDRESS_T variable. The
**              address of the variable which accepts the
**              converted IP Address should be in 'AddStore'.
** RETURNS:     0    -->   no error
**              1    -->   illegal address format
**************************************************************/
int DotConvert(ADDRESS_T *AddStore, char *DotAddress)
{
  int  j, Tmp[4];

  j = sscanf(DotAddress, "%i.%i.%i.%i", Tmp, Tmp + 1, Tmp + 2, Tmp + 3);
  if (j < 4) 
    return 1;
  for (j = 0; j < 4; j++) 
    if ((*((BYTE *)(AddStore) + j) = (BYTE)Tmp[j]) > 255) 
      return 1;
  return 0;
}

