/* [DOC] File ****************************************************
** MODULE INFORMATION*
**********************
**      FILE    NAME:       PWGOPEN.C
**      SYSTEM  NAME:       POWER
**      VERSION NUMBER      3.00
**
** Descr:       Open window & Close windoe functions for
**              Programmer Oriented Window Environment Routines.
**              (NoC) July 1989 - Ling Thio Software Productions.
******************************************************************
** ADMINISTRATIVE INFORMATION*
******************************
**      ORIGINAL AUTHOR:    Ling Thio
**      CREATION DATE:      89/07/10
**      CHANGES:            none
*****************************************************************/

#include "pwinc.h"                          /* main include file */

extern int    _pwb_nlevels;

static void pwg_border(PWWIN *win, int ttype);

/* [DOC] Function *********************************************
** Name:        pw_nrow                                 [API]
** SYNOPSIS:    int pw_nrow(void)
** Descr:       Returns screen height in number of lines.
** RETURNS:     Returns screen height in number of lines.
** SEE ALSO:    pw_ncol, pw_winncol, pw_winnrow
**************************************************************/
int pw_nrow(void)
{
    return _pwv_scrnrow;
}

/* [DOC] Function *********************************************
** Name:        pw_ncol                                 [API]
** SYNOPSIS:    int pw_ncol(void)
** Descr:       Returns screen width in number of characters.
** RETURNS:     Returns screen width in number of characters.
** SEE ALSO:    pw_nrow, pw_winncol, pw_winnrow
**************************************************************/
int pw_ncol(void)
{
    return _pwv_scrncol;
}

/* [DOC] Function *********************************************
** Name:        pw_winsrow                              [API]
** SYNOPSIS:    int pw_winsrow(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Returns window's row position.
** RETURNS:     Returns window's row position.
** SEE ALSO:    pw_winscol
**************************************************************/
int pw_winsrow(PWWIN *win)
{
    return win->srow;
}

/* [DOC] Function *********************************************
** Name:        pw_winscol                              [API]
** SYNOPSIS:    int pw_winscol(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Returns window's column position.
** RETURNS:     Returns window's column position.
** SEE ALSO:    pw_winsrow, pw_ncol, pw_nrow
**************************************************************/
int pw_winscol(PWWIN *win)
{
    return win->scol;
}

/* [DOC] Function *********************************************
** Name:        pw_winnrow                              [API]
** SYNOPSIS:    int pw_winnrow(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Returns window's height in number of lines.
** RETURNS:     Returns window's height in number of lines.
** SEE ALSO:    pw_winncol, pw_ncol, pw_nrow
**************************************************************/
int pw_winnrow(PWWIN *win)
{
    return win->nrow;
}

/* [DOC] Function *********************************************
** Name:        pw_winncol                              [API]
** SYNOPSIS:    int pw_winncol(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Returns window's width in number of characters.
** RETURNS:     Returns window's width in number of characters.
** SEE ALSO:    pw_winnrow
**************************************************************/
int pw_winncol(PWWIN *win)
{
    return win->ncol;
}

/* [DOC] Function *********************************************
** Name:        pw_open                                   [API]
** SYNOPSIS:    PWWIN *pw_open(srow, scol, nrow, ncol,
**                             header, mode, type)
**              int  srow           Top left row
**              int  scol           Top left column
**              int  nrow           Window length in rows
**              int  ncol           Window width in columns
**              char *header;       Text in top left of window
**              int  mode           Window mode
**              int  type;          Window type (PWW_...)
** Descr:       Opens a window.
**              The 'pw_open' function opens a window of type
**                'type', and makes this the active window.
**              'mode' determines the window mode:
**                  PWM_DEFAULT     Border+Shade+Show
**                  PWM_NOBORDER    No borders
**                  PWM_NOSHADE     No shading
**                  PWM_NOSHOW      Hide window
**                  PWM_NOHEADER    Don't diplay header
**              All positions are given as an offset, relative
**                to the top left corner (borders exclusive).
**              The character string 'header' will be displayed
**                on the left of the top borderline.
**              If the PWK_HELP key is pressed, the helpfile will
**                be searched for the 'header' text of the active
**                window (see also pw_pop).
**              The following window types are defined in POWER.H
**                  PWW_NORMAL
**                  PWW_HEADER
**                  PWW_MENU
**                  PWW_DATA
**                  PWW_INPUT
**                  PWW_ERROR
**                  PWW_HELP
**                  PWW_SPECIAL
** RETURNS:     The 'pw_open' function returns a PWWIN pointer
**                to the opened window.
** SEE ALSO:    pw_close
**************************************************************/
PWWIN *pw_open(int srow, int scol, int nrow, int ncol, char *header, int mode, int type)
{
    PWWIN *win;
    PWWIN *activewin;
    int level = 0;

    if ((win = malloc(sizeof(PWWIN))) != NULL)
    {
        win->scol     = scol;
        win->srow     = srow;
        win->ncol     = ncol;
        win->nrow     = nrow;
        win->status   = mode;
        win->wtype    = type;
        win->ttype    = PWT_PLAIN;
        pwg_initcoord(win);
        if ((win->buffer = pwg_malloc((win->ncol+2)*(win->nrow+2)*2)) != NULL)
        {
            win->header   = header;
            pwg_initwin(win);
            activewin = pwb_activewin();
            if ((win->status & PWM_NOSHADE) == 0)
                win->status |= STAT_SHADE;
            if (win->status & PWM_PUSH)
                level = -1;
            if (pwb_open(win, level))
            {                                       /* too many windows open */
                pwg_free(win->buffer);
                free(win);
                pw_error("More than %d windows open", MAX_WINS-1);
                pw_exit(1);
            }
            if ((activewin) && (activewin != pwb_activewin()) &&
                !(win->status & PWM_NOKEYINFO))
            {
                pwg_border(activewin, PWT_BORDER);
                pwg_keyinfo("", "");
            }
            win->crow = win->ccol = 0;              /* top left of window */
        }
        else
        {
            free(win);
            return NULL;
        }
    }
    return win;
}

/* [DOC] Function *********************************************
** Name:        pw_close                                  [API]
** SYNOPSIS:    void pw_close(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Closes a window.
**              The 'pw_close' function closes window 'win',
**                previously openend by function pw_open().
** RETURNS:     void
** SEE ALSO:    pw_open
**************************************************************/
void pw_close(PWWIN *win)
{
    PWWIN *activewin;

    pwb_close(win);
    pwg_free(win->buffer);
    free(win);
    if ((activewin = pwb_activewin()) != NULL)
    {
        pwg_border(activewin, PWT_ACTIVEBORDER);
        pwg_keyinfo(activewin->keyleft, activewin->keyright);
    }
    else
        pwg_keyinfo("", "");
}

/* [DOC] Function *********************************************
** Name:        pw_pop                                  [API]
** SYNOPSIS:    void pw_pop(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Pops a window to the top.
**              The 'pw_pop' function places the window 'win'
**                on top off all other windows, currently opened,
**                and makes this the active window.
** RETURNS:     void
** EXAMPLE:       ....
**              PWWIN *win1, *win2;
**                ....
**              win1 = pw_open(4, 4, 10, 5, "First",
**                             PWM_DEFAULT, PWW_NORMAL);
**              win2 = pw_open(6, 8, 10, 5, "Second",
**                             PWM_DEFAULT, PWW_NORMAL);
**              pw_pop(win1);
**                ....
**************************************************************/
void pw_pop(PWWIN *win)
{
    PWWIN *activewin;

    if (win != wintable[_pwb_nlevels-1])
    {
        activewin = pwb_activewin();
        pwb_pop(win);
        if ((activewin) && (activewin != pwb_activewin()))
        {
            pwg_border(activewin, PWT_BORDER);
            pwg_border(win, PWT_ACTIVEBORDER);
            pwg_keyinfo(win->keyleft, win->keyright);
        }
    }
}

/* [DOC] Function *********************************************
** Name:        pw_show                                 [API]
** SYNOPSIS:    void pw_show(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Redraws a hided window.
** RETURNS:     void
** SEE ALSO:    pw_hide
**************************************************************/
void pw_show(PWWIN *win)
{
    pwb_show(win);
}

/* [DOC] Function *********************************************
** Name:        pw_hide                                 [API]
** SYNOPSIS:    void pw_hide(win)
**              PWWIN *win          Pointer to window handle
** Descr:       hides a visible window.
** RETURNS:     void
** SEE ALSO:    pw_show
**************************************************************/
void pw_hide(PWWIN *win)
{
    pwb_hide(win);
}

static void pwg_border(PWWIN *win, int ttype)
{
    if ((win->status & PWM_NOBORDER) == 0)
    {
        pw_block(win,        -1,        -1,         1, win->ncol+2, ttype);
        pw_block(win,         0,        -1, win->nrow,           1, ttype);
        pw_block(win,         0, win->ncol, win->nrow,           1, ttype);
        pw_block(win, win->nrow,        -1,         1, win->ncol+2, ttype);
    }
}

/* [DOC] Function *********************************************
** Name:        pw_shade                                  [API]
** SYNOPSIS:    void pw_shade(win)
**              PWWIN *win          Pointer to window handle
** Descr:       Povide a window with shading.
**              The 'pw_shade' function shows a shadow, under
**                and to the right of the window 'win'.
**              The shading color depends on the monitor mode
**                configuration and the current attribute.
** RETURNS:     void
**************************************************************/
void pw_shade(PWWIN *win)
{
    if (!(win->status & STAT_HIDE) &&       /* visible ? */
        !(win->status & STAT_SHADE))        /* not yet shaded */
    {
        pwb_shade(win);
        win->status ^= STAT_SHADE;
    }
}

void pwg_initcoord(PWWIN *win)
{
    int bordersize = (win->status & PWM_NOBORDER) ? 0 : 1;

    if (win->srow == PWC_TOP)    win->srow = 1+bordersize;
    if (win->scol == PWC_LEFT)   win->scol = bordersize;
    if (win->srow == PWC_CENTER) win->srow = (_pwv_scrnrow-win->nrow)/2;
    if (win->scol == PWC_CENTER) win->scol = (_pwv_scrncol-win->ncol)/2;
    if (win->srow == PWC_BOTTOM) win->srow = _pwv_scrnrow-1-win->nrow-bordersize;
    if (win->scol == PWC_RIGHT)  win->scol = _pwv_scrncol-win->ncol-bordersize;
    if (win->nrow == PWC_BOTTOM) win->nrow = _pwv_scrnrow-1-win->srow-bordersize;
    if (win->ncol == PWC_RIGHT)  win->ncol = _pwv_scrncol-win->scol-bordersize;
    if (win->nrow == PWC_CENTER) win->nrow = _pwv_scrnrow-(win->srow*2);
    if (win->ncol == PWC_CENTER) win->ncol = _pwv_scrncol-(win->scol*2);
}

void pwg_initwin(PWWIN *win)
{
    win->tscol    = win->scol;
    win->tsrow    = win->srow;
    win->tncol    = win->ncol;
    win->tnrow    = win->nrow;
    win->scrptr   = _pwv_scrbuf;
    win->levptr   = _pwv_levbuf;
    win->winptr   = win->buffer;
    win->keyleft  = NULL;
    win->keyright = NULL;
    if ((win->status & PWM_NOBORDER) == 0)
    {
        win->tscol--;
        win->tsrow--;
        win->tncol += 2;
        win->tnrow += 2;
    }
}


