/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       Bufm.C
**     SYSTEM   NAME:       IP Buffer Management
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1990/2/1
**
** DESCRIPTION: Contains functions for buffer management
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   BUFM.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/BUFM/VCS/BUFM.C_V  $
**              
**                 Rev 1.0   01 Feb 1991 10:16:46   etstjan
**              Initial revision.
*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <error.h>
#include <futil.h>
#include <power3.h>
#include <dnpap.h>

#include "bufm.h"

#define  MAGIC_1  0xAAAA                /* is pattern 1010101010101010 */
#define  MAGIC_2  0x5555                /* is pattern 0101010101010101 */


#define  MAGIC    unsigned int          /* type of magic field     */
#define  BYTE     unsigned char         /* define byte type */

typedef struct heap_cbl {               /* structure of a heap control block */
  struct heap_cbl *Previous;            /* previous heap control block */
  struct heap_cbl *Next;                /* next heap control block */
  BM_SIZE_T       Length;               /* length of free block */
  unsigned int    BufferNr;             /* number of buffer */
  MAGIC           UsedFlag;             /* 0 indicates block is free */
  } HEAP_CBL;

typedef struct buf_cbl {                /* info for each buffer */
  HEAP_CBL        *Buffer;              /* pointer to start of buffer */
  BM_SIZE_T       MaxSize;              /* maximum size in this buffer */
  } BUF_CBL;

#ifdef DEBUGBM
  extern PWWIN *IPWin;                  /* IP report window */
#endif

BUF_CBL      *Start;                    /* array with pointers to start of */
                                        /* allocation buffer */
BM_SIZE_T    SaveBufferSize;            /* contains length of buffers */
int          MaxBuf = 0;                /* number of buffers */



int BufInit (void)
{
    FILE    *ConFile = NULL;
    int     LineNr, Ret = 0;
    long    TmpStor;
    char    ConfBuf[255];
    USHORT  BufSize = 0x8000;
    int     BufNr = 1;

    if ((ConFile = fopen(ConfigFile, "r")) == NULL)
    {
        ERR_ERR(ERR_BUF, ERR_CRITICAL, "Initialization file '%s' not found", ConfigFile);
        return -1;
    }
    if (FFindSection(ConFile, "BUFFER") < 0)
    {
        ERR_ERR(ERR_BUF, ERR_CRITICAL, "Section '[BUFFER]' not found");
        fclose(ConFile);
        return -1;
    }
    if ((LineNr = FGetVar(ConFile, "buffersize", ConfBuf, 255, UPC)) > 0)
    {
        if (((TmpStor = atol(ConfBuf)) <= 0) || (TmpStor > 0xffff)) 
            ERR_ERR(ERR_BUF, ERR_WARNING, "Illegal value for Buffersize, Taking default");
        else 
            BufSize = (WORD)TmpStor;
    }
    if ((LineNr = FGetVar(ConFile, "numberbuffers", ConfBuf, 255, UPC)) > 0)
    {
        if (((TmpStor = atol(ConfBuf)) <= 0) || (TmpStor > 0xffff)) 
            ERR_ERR(ERR_BUF, ERR_WARNING, "Illegal value for Numberbuffers, Taking default");
        else 
            BufNr = (int)TmpStor;
    }
    if ((Ret = IPBufInit(BufNr, BufSize)) != NO_ERR)
    {
        ERR_ERR(ERR_BUF, ERR_CRITICAL, "Initializing Buffer Management Code: %d", Ret);
        fclose(ConFile);
    }
    return (Ret);
}






/**************************************************************
** NAME:        IPBufInit                                 [API]
** SYNOPSIS:    int IPBufInit(unsigned int NrBuffers,
**                            BM_SIZE_T BufSize);
** DESCRIPTION: Initializes the Buffer Management.
**              Allocates NrBuffers buffers of BufSize bytes.
** SEE ALSO:    IPBufGet, IPBufFree, IPBufEnd
** RETURNS:     0        --> no error
**              IACT_ERR --> not closed yet
**              IMEM_ERR --> memory allocation error
**************************************************************/
int IPBufInit(unsigned int NrBuffers, BM_SIZE_T BufSize)

{
  int i, j;                             /* used for general purposes */

  if (MaxBuf != 0)                      /* Init_Buffers already invoced */
    return IACT_ERR;

  MaxBuf = NrBuffers;                   /* store number of buffers */

  /* get space for pointer array */
  if ((Start = calloc(NrBuffers, sizeof(BUF_CBL))) == NULL)
    return IMEM_ERR;

  /* get space for buffers */
  for (i = 0; i < MaxBuf; i++) {
    if ((Start[i].Buffer = malloc(BufSize)) == NULL) { 
      for (j = 0; j < i; j++)           /* free everything allocated */
        free(Start[j].Buffer);
      free(Start);
      MaxBuf = 0;                       /* and reset maxbuf */
      return IMEM_ERR;                  /* and return an error code */
      }
    Start[i].Buffer->Previous = NULL;
    Start[i].Buffer->Next = NULL;
    /* set length and maximum available */
    Start[i].Buffer->Length = BufSize - sizeof(HEAP_CBL);
    Start[i].MaxSize = Start[i].Buffer->Length;
    Start[i].Buffer->BufferNr = i;    /* set buffernr */
    Start[i].Buffer->UsedFlag = 0;    /* block ain't used yet */
    /* set first free block in buffer */
    }
  SaveBufferSize = BufSize;             /* set SaveBufSize */
  return NO_ERR;
}

/**************************************************************
** NAME:        IPBufEnd                                  [API]
** SYNOPSIS:    void IPBufEnd(void);
** DESCRIPTION: Deallocates all memory used for buffering.
** SEE ALSO:    IPBufInit, IPBufGet, IPBufFree
** RETURNS:
**
**************************************************************/
void IPBufEnd(void)
{
int i;

  for (i = 0; i < MaxBuf; i++) {
    free(Start[i].Buffer);              /* deallocate reserved memory */
    }
  free(Start);
  MaxBuf = 0;                           /* buf. man. free again */
}

/**************************************************************
** NAME:        IPBufGet                                  [API]
**
** SYNOPSIS:    void *IPBufGet(BM_SIZE_T Size);
** DESCRIPTION: Allocates a buffer in the buffer management.
**              Returns a pointer to a piece of memory
**              of at least Size bytes. Size can not
**              exceed BufSize minus some overhead.
**              Check bytes are placed at the top and
**              bottom of the block for control purposes.
** SEE ALSO:    IPBufInit, IPBufEnd, IPBufFree
** RETURNS:     NULL  -->   not enough memory (error)
**              else  -->   void pointer to data block
**************************************************************/
void *IPBufGet(BM_SIZE_T Size)
{
  HEAP_CBL         *CheckIt;            /* pointer in the linked list */
  HEAP_CBL         *NewPoint;           /* pointer to new heap_cbl address */
  BYTE             *DataArea;           /* pointer to first byte of data area */
  BM_SIZE_T        AreaSize;            /* minimum area to be found */
  int              i;                   /* general purpose counter */
  int              UpdateSizes = 0;     /* should maxsizes be updated? */

  /* for alignment purposes, make sure that size is even */
  if ((Size & 1) == 1) {                /* odd number of bytes required */
    Size++;                             /* so increment size by 1 */
    }

  /* When searching a block, the size must be enough to contain
     a MAGIC at the end and an additional heap control block. */
  AreaSize = Size + sizeof(MAGIC) + sizeof(HEAP_CBL);

  /* First, look if there is a buffer with enough space for request. */
  for(i = 0; (i < MaxBuf) && (Start[i].MaxSize < AreaSize); i++)  
    ;

  if (i == MaxBuf) {                    /* no bufferspace found */
    return NULL;                        /* return NULL pointer */
    }

  for (CheckIt = Start[i].Buffer;       /* data block should be in buffer i */
       CheckIt != NULL; CheckIt = CheckIt->Next) {
    if (!CheckIt->UsedFlag &&             /* block must be free */
        (CheckIt->Length >= AreaSize)) {  /* and large enough */
      break;                              /* then a block has been found */
      }
    }

  if (CheckIt == NULL) {                 /* no block could be found */
    return NULL;                         /* this should not occur, security */
    }

  /* fill heap control block and generate a new one for the remainder */

  /* generate new heap_cbl */
  NewPoint = (HEAP_CBL *)((BYTE *)CheckIt + AreaSize);
  NewPoint->Previous = CheckIt;     /* set prev pointer to this heap_cbl */
  NewPoint->Next = CheckIt->Next;   /* complete linked list */

  if (NewPoint->Next != NULL) {
    NewPoint->Next->Previous = NewPoint;  /* adjust next block previous */
    }

  NewPoint->Length = CheckIt->Length - AreaSize;
                                    /* set rest length */
  NewPoint->BufferNr = i;           /* and set buffer number */
  NewPoint->UsedFlag = 0;           /* data block not used yet */

  /* set the update maximum sizes array flag, if necessary */
  if (CheckIt->Length == Start[i].MaxSize) { /* the maximum size has been used */
    UpdateSizes++;
    }

  /* fill heap_cbl */
                                    /* set pointer to next heap_cbl */
  CheckIt->Next = NewPoint;
  CheckIt->Length = Size;           /* set length of data block */
  CheckIt->UsedFlag = MAGIC_1;      /* set magic number 1 -> used */
  DataArea = (BYTE *)CheckIt + sizeof(HEAP_CBL);
                                             /* set data area pointer */
  *((MAGIC *)(DataArea + Size)) = MAGIC_2;   /* set magic number 2 */

  /* update maximum sizes array, if necessary */
  if (UpdateSizes) {                              /* flag has been set */
    Start[i].MaxSize = 0;                         /* reset maxsizes */
    for (CheckIt = Start[i].Buffer;               /* so update maximum size */
      CheckIt != NULL;
      CheckIt = CheckIt->Next) {
      if (!CheckIt->UsedFlag &&              /* if free data block */
          CheckIt->Length > Start[i].MaxSize) { 
        Start[i].MaxSize = CheckIt->Length;  /* then this is the greatest */
        }
      }
    }
  return DataArea;                     /* and return pointer to data area */
}

/**************************************************************
** NAME:        IPBufFree1                                [API]
** SYNOPSIS:    int IPBufFree1(void *Point);
**
** DESCRIPTION: This function releases the memory block
**              indicated by Point. If the block is not
**              recognized as allocated by IPBufGet, an
**              error code will be returned, otherwise
**              the block will be available again.
** RETURNS:     NO_ERR    -->  no error
**              FREE_ERR  -->  block not an IPBuf block
**************************************************************/
int IPBufFree1(void *Point)
{
  HEAP_CBL      *ThisHeapCbl;           /* pointer to this heap cbl */
  HEAP_CBL      *PrevHeapCbl;           /* pointer to previous heap cbl */
  HEAP_CBL      *NextHeapCbl;           /* pointer to next heap cbl */
  BYTE          *ParamArea;             /* pointer into parameter area */
  int           Test = 0;               /* used in validity test */

  if (Point == NULL) {                  /* quit if pointer is NULL */
    return FREE_ERR;
    }

  ParamArea = (BYTE *)Point;
  ThisHeapCbl = (HEAP_CBL *)(ParamArea - sizeof(HEAP_CBL));

  /* check if this block could have been located by Give_Space */
  /* check for magics */
  Test += (ThisHeapCbl->UsedFlag == MAGIC_1) ? 0 : 1;
  Test += (*(MAGIC *)(ParamArea + ThisHeapCbl->Length) == MAGIC_2) ? 0 : 1;

  if (Test != 0) {                      /* quit if an error occured */
    return FREE_ERR;
    }

  PrevHeapCbl = ThisHeapCbl->Previous;  /* there may be a previous block */
  NextHeapCbl = ThisHeapCbl->Next;      /* a next block always exists !! */

  /* check if previous block is also free */
  if (PrevHeapCbl != NULL &&
	!PrevHeapCbl->UsedFlag) {           /* previous block also free */
    PrevHeapCbl->Next = NextHeapCbl;    /* continue linked list  */
    NextHeapCbl->Previous = PrevHeapCbl;/* complete linked list */
                                        /* adjust length         */
    PrevHeapCbl->Length +=
	 ThisHeapCbl->Length + sizeof(MAGIC) + sizeof(HEAP_CBL);
    ThisHeapCbl = PrevHeapCbl;          /* adjust current pointer */
    }
  else {                                /* free this block */
    ThisHeapCbl->Length += sizeof(MAGIC);
    ThisHeapCbl->UsedFlag = 0;
    }

  /* check if next block is also free */
  if (!NextHeapCbl->UsedFlag) {         /* next block also free */
    /* combine these blocks */
    ThisHeapCbl->Next = NextHeapCbl->Next;    /* continue linked list  */
    if (NextHeapCbl->Next != NULL) {          /* and once again */
      NextHeapCbl->Next->Previous = ThisHeapCbl;
      }
                                        /* adjust length         */
    ThisHeapCbl->Length +=              /* with length of block made free */
	 NextHeapCbl->Length + sizeof(HEAP_CBL);
    }

  /* adjust maximum sizes array, if necessary */
  if (ThisHeapCbl->Length > Start[ThisHeapCbl->BufferNr].MaxSize) {
    Start[ThisHeapCbl->BufferNr].MaxSize = ThisHeapCbl->Length;
    }
  return NO_ERR;                        /* return succes code */
}

/**************************************************************
** NAME:        IPBufCheck                                [API]
** SYNOPSIS:    int IPBufCheck(void);
** DESCRIPTION: Performs validity checks on the Buffer
**              Management.
** RETURNS:     NO_ERR   -->   no error
**              IPBUF_MAGERR -->  magic error found
**              IPBUF_TOTERR -->  total size error
**              IPBUF_CYCERR -->  order of list error
**************************************************************/
int IPBMCheck(void)
{
  HEAP_CBL      *HeapPoint;             /* pointer in heap */
  BYTE          *CheckPoint;            /* pointer to check bytes */
  BM_SIZE_T     SizeCollect;            /* collect overhead bytes */
  BM_SIZE_T     SizeData;               /* collect total used bytes */
  BM_SIZE_T     SizeFree;               /* collect total free bytes */
  int           Test = 0;               /* used for detecting errors */
  int           i;                      /* buffer pointer array index */

  for (i=0; i < MaxBuf; i++) {              /* get all buffers */
    SizeCollect = SizeData = SizeFree = 0;  /* clear size counters */
    for (HeapPoint = Start[i].Buffer;       /* get start pnt of buffer */
         HeapPoint != NULL;
         HeapPoint = HeapPoint->Next) {
      /* collect free, used and overhead size */
      SizeCollect += sizeof(HEAP_CBL);      /* add HeapCbl overhead */
      if (HeapPoint->UsedFlag)  {           /* if it is allocated data */
        /* test for MAGIC_1 */
        CheckPoint = (BYTE *)HeapPoint + sizeof(HEAP_CBL) + HeapPoint->Length;
        Test += (HeapPoint->UsedFlag == MAGIC_1) ? 0 : 1;
        Test += (*(MAGIC *)CheckPoint == MAGIC_2) ? 0 : 1;

        if (Test != 0) {                    /* quit if an error has occured */
          return MAGERR;                    /* return magic error */
          }
        SizeData += HeapPoint->Length;      /* collect used data size */
        SizeCollect += sizeof(MAGIC);       /* and some more overhead */
        }

      else {                                /* it is a free block */
        SizeFree += HeapPoint->Length;      /* collect free data size */
        }

      /* check for cycle and order */
      if (HeapPoint->Next != NULL) {        /* if there exists an other block */
        if ((HeapPoint->Next->Previous != HeapPoint) ||
            (HeapPoint->Next <= HeapPoint)) {
          return CYCERR;                    /* return cycle error */
          }
        }
      }
    /* check if total number of bytes in this buffer is still the same */
    if ((SizeData + SizeFree + SizeCollect) != SaveBufferSize) {
      return TOTERR;                        /* if it is not , return error */
      }
    }
  return NO_ERR;                            /* no error occured */

}

/**************************************************************
** NAME:        FreeMem                                   [API]
** SYNOPSIS:    unsigned long FreeMem(void);
**           
** DESCRIPTION: Returns the free memory (in bytes).
**              This is all free memory in
**              the memory management allocation blocks.
** RETURNS:     The free memory in bytes.
**************************************************************/
unsigned long FreeMem(void)
{
  HEAP_CBL      *HeapPoint;            /* pointer in heap */
  unsigned long SizeFree = 0;          /* collect total free bytes */
  int           i;

  for (i = 0; i < MaxBuf; i++) {
    for (HeapPoint = Start[i].Buffer;  /* get start pnt of buffer */
      HeapPoint != NULL;
      HeapPoint = HeapPoint->Next) {
      if (!HeapPoint->UsedFlag)  {  
        SizeFree += HeapPoint->Length; /* collect free data size */
        }
      }
    }
  return SizeFree;                     
}

/**************************************************************
** NAME:        IPBufFree2                                [API]
** SYNOPSIS:    int IPBufFree2(void *Point, int FileLine,
**                             char *FileName)
**             
** DESCRIPTION: This is a debug version of IPBufFree.
**              If the symbol 'DEBUGBM' is defined, this
**              function will be called instead of IPBufFree.
**              When an error occures, the line number and
**              filename of the caller will be printed in
**              the report window.
** RETURNS:     NO_ERR    -->  no error
**              FREE_ERR  -->  block not an IPBuf block
**************************************************************/
int IPBufFree2(void *Point, int FileLine, char *FileName)
{
  int RetCode;

  if ((RetCode = IPBufFree1(Point)) != NO_ERR) {
#ifdef DEBUGBM 
    pw_printf(IPWin, "\nError found in Free!  File: %s  Line: %d", FileName, FileLine);
#endif
    }
  return RetCode;
}
