Unit serial2a;
{ a serial communications unit which reads and writes directly
  to serial ports COM1 and COM2. Also uses BIOS calls to set
  communications parameters for COM1 and COM2. }

{ intended primarily for reading electronic balances, some of
  which have funny handshakes. 6/92 D. Lunney }

{ NOTE: the real constant maxloops is the max number of retries
  before the I/O procedures give up and time out.  The boolean
  variable TimeOut is set TRUE if either input or output is
  unsuccessful after maxloops tries.  This keeps the system
  from hanging up. Maxloops can be changed from the calling
  program. }

{ Copyright 1992-1996 East Carolina University, Greenville, NC, USA. }
{ Author: David Lunney, Professor of Chemistry, ECU }

(*      ADDRESS: Department of Chemistry
                 East Carolina University
                 Greenville, NC 27858-4353
                 USA


        INQUIRIES REGARDING THIS PROGRAM SHOULD BE DIRECTED
        TO DAVID LUNNEY AT chlunney@ecuvm.cis.ecu.edu OR
        LUNNEY@DELPHI.COM                                            *)

(*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    A COPY OF THE GNU GENERAL PUBLIC SOFTWARE LICENSE HAS BEEN
    PROVIDED WITH THIS PROGRAM IN THE FILE "LICENSE.TXT"       *)


    { No Warranty.  EAST CAROLINA UNIVERSITY DISCLAIMS AND MAKES NO
    REPRESENTATIONS AND EXTENDS NO WARRANTIES, EITHER EXPRESS OR
    IMPLIED. THERE ARE NO EXPRESS OR IMPLIED WARRANTIES OF
    MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  The User
    shall assume all liability for all damages whatsoever that may or
    do arise from the User's use, inability to use, performance, or
    storage of SERIAL2A.  East Carolina University shall not be
    liable to the User for any loss, claims, damages or demands
    whatsoever made by the User or made against the User by any other
    party, due to or arising from the performance of, use of, and/or
    inability to use SERIAL2A by the User and/or anyone else. }

{ derived from SERIAL2.PAS  }

{ routines

Procedure DumReadByte(PortNo: Byte; Var inbyte: Byte; InShake: String4);

Procedure DumReadALine(PortNo: Byte; Var InString: String;
                               InShake, delim: String4);

  added 7-26-96 to read  Ohaus TP200 balance which keeps
  its handshake active after its port is dead. }

{ modified not to halt if input errors are detected. 7/96 dl }

{ Modified 5/5/96 to include boolean function InStatusOK which
  checks for RDR bit and errors.  If InStatusOK is TRUE, there
  is a byte in the input register, and there are no parity,
  framing, or overrun errors: added to accommodate ill-designed
  instruments with no input handshake lines at all. dl }


{ Modified 7/21/93 to allow CR, CR/LF, LF, or even LF/CR
  as delimiters in the routines that output or input strings. }

{ modified not to hang if there is no response from I/O device
  6/28/92 dl }

{ modified to set baud rate, etc. with a bios call 6/29/92 dl }

{ modified to set baud rate, etc. CORRECTLY with a bios call 6/10/93 dl }

{ Sets up 8250 UART chip in a PC to operate in
  polling mode for input and output. }

{ DERIVED FROM PORTSET0.PAS }

{ Data on 8250 taken from the Waite Group's MS-DOS
  Developer's  Guide, 2nd ed., Howard W. Sams Co.,
  1989, pp. 458-459. }

(*
                        INTRODUCTION

   In an IBM PC clone, the computer is configured to be the Data
Terminal Equipment(DTE) and the external device (modem, etc.)
is configured to be the Data Communications Equipment (DCE).

The signals relevant to this program are:

                       Outputs from Computer

        name            abbrev.     pin on DB-25   pin on DB-9
   Transmitted data       TD             2              3
   Ready to send          RTS            4              7
   Data terminal ready    DTR           20              4


                       Inputs to Computer

        name            abbrev.     pin on DB-25   pin on DB-9
   Received data          RD             3              2
   Clear to send          CTS            5              8
   Data set ready         DSR            6              6


Other input signals, not used in this program, are Receive Line
Signal DetectOr (RLSD) and Ring IndicatOr (RI).

                                                        *)

(*                       INTERFACE                      *)

Interface
   Uses crt, DOS;

Const

  { base addresses of COM1 and COM2 }
  ComBase: Array[1..2] Of Word =($3F8, $2F8);
  { offsets of UART registers }
  TXBuffer: Byte = $00; { transmit-receive buffer }
  IntEnableReg: Byte = $01; { interrupt enable register }
  IntIDReg: Byte = $02; { interrupt identification register }
  LineContReg: Byte = $03; { line control register }
  ModemContReg: Byte = $04; { modem control register }
  LineStatReg: Byte = $05; { line status register }
  ModemStatReg: Byte = $06; { modem status register }

  { bit masks for setting and checking reg. bits }

  { following bits are in the interrupt enable register(base+1) }
  { Included here for future use; NOT used in this program! }
  RDAIntbit: Byte = $01;   { receieved data available(RDA) bit }
  THREIntbit:   Byte = $02;   { xmit holding reg empty bit }
  RLSIntbit: Byte = $04; { receive line status bit }
  MSIntbit: Byte = $08;  { modem status bit }

  { following bits are in the line control register(base+3) }
  SevenBits: Byte = $02;   { seven-bit data }
  EightBits: Byte = $03;   { eight-bit data }
  OneStopBit: Byte = $00;  { one stop bit }
  TwoStopBits: Byte = $04; { two stop bits }
  NoParity: Byte = $00;    { no parity }
  OddParity: Byte = $08;   { odd parity }
  EvenParity: Byte = $18;  { even parity }
  BreakBit: Byte = $40;    { sets BREAK }

  DLAB: Byte = $80; { setting msb of line control register
  (at base+3) enables baud rate divisor
  to be loaded into the receive buffer
  and interrupt enable regs, lo byte first.}

  { following bits are in the modem control register(base+4) }
  SetDTRbit: Byte = $01;   { sets DTR }
  SetRTSbit: Byte = $02;   { sets RTS }

  { following bits are in the line status register(base+5) }
  RDRbit: Byte = $01;   { read data ready bit }
  OverRunBit: Byte = $02;  { overrun error }
  ParityErrorBit: Byte = $04;  { parity error }
  FramingErrorBit: Byte = $08;  { framing error }
  BreakDetectBit: Byte = $10;   { break detect }
  THREBit: Byte = $20;   { Xmit holding reg. empty }
  TXMTBit: Byte = $40;   { Xmit reg. empty }

  { following bits are in the modem status register(base+6) }
  DeltaCTSbit: Byte = $01;   { change in CTS }
  DeltaDSRbit: Byte = $02;   { change in DSR }
  DeltaRIbit: Byte = $02;    { change in ring indicatOr (RI) }
  DeltaRLSDbit: Byte = $04;  { change in receive line signal detect }
  CTSbit: Byte = $10;   { CTS }
  DSRbit: Byte = $20;   { DSR }
  RIbit: Byte = $40;    { ring indicatOr (RI) }
  RLSDbit: Byte = $80;  { receive line signal detect }

  { characters }
  CR:   Char = ^M;
  LF:   Char = ^J;
  NUL:  Char = ^@;
  ACK:  Char = ^F;
  BEL:  Char = ^G;
  SP:   Char = #32;
  XON:  Char =  ^Q;
  XOFF: Char =  ^S;

  { ASCII bytes for characters }
  CRbyte:   Byte = $0D;
  LFbyte:   Byte = $0A;
  NULbyte:  Byte = $00;
  ACKbyte:  Byte = $06;
  BELbyte:  Byte = $07;
  SPbyte:   Byte = $20;
  XONbyte:  Byte = $11;
  XOFFbyte: Byte = $13;

  { legal baud rates for PC clones }
  legalbaud: Array[0..7] Of Word =(110, 150, 300, 600, 1200,
  2400, 4800, 9600);

  { max number of loops while waiting for I/O }
  maxloops: Real = 1.0E+05;

  (*                  TYPE DECLARATIONS                     *)

Type
  String4 = String[4];


  (*             GLOBAL VARIABLE DECLARATIONS                *)

Var
  xfile: Text; { debugging file }

  DTRset, RTSset, CTSset, DSRset, THRE,
    TXMT, RDR, InputError:  Array[1..2] Of Boolean;

  { NOTE: These boolean variables keep track of current state of
  handshake lines, status bits, etc. in the UART, but are not
  valid until after calls to the procedures which check, set,
  or clear the corresponding bits.  DTRset and RTSset are
  initialized by a call to InitHandShakes, which clears DTR
  and RTS. }

  timeout: Boolean;
  { set TRUE if unit times out waiting for input or output }

  (*                      PROCEDURE LIST                         *)
  { The procedures in the first group are the ones most likely
  to be called by an external program. Their names explain their
  functions. }

  { BIOSInitComPort initializes the port using a bios call. }
Procedure BIOSInitComPort(PortNo: Byte; Var baud: Word; DataBits: Byte;
                                 parity: Char; StopBits: Byte);
Procedure InitHandshakes(PortNo: Byte);
Procedure ReadByte2(PortNo: Byte; Var inbyte: Byte; InShake: String4);
Procedure DumReadByte(PortNo: Byte; Var inbyte: Byte; InShake: String4);
Function InStatusOK(PortNo: Byte): Boolean;
Procedure GetFirstByte(portno: Byte);
Procedure ReadALine4(PortNo: Byte; Var InString: String;
                               InShake, delim: String4);

Procedure DumReadALine(PortNo: Byte; Var InString: String;
                               InShake, delim: String4);

Procedure SendByte(PortNo, outbyte: Byte; OutShake: String4);
Procedure SendALine2(PortNo: Byte; OutString: String;
                                OutShake, delim: String4);


{                                                          }
{ The procedures in the second group are not likely to be used
  by an external program except for testing. They are called
  by the above procedures.}

Procedure CheckLegalBaud(Var Baud: Word; Var baudbyte: Byte);
Procedure SetLegalBaud(Var Baud: Word);
Procedure SetRTS(PortNo: Byte);
Procedure ClearRTS(PortNo: Byte);
Procedure SetDTR(PortNo: Byte);
Procedure ClearDTR(PortNo: Byte);
Procedure CheckCTS(PortNo: Byte; Var SetClear: Byte);
Procedure CheckDSR(PortNo: Byte; Var SetClear: Byte);
Procedure CheckTHRE(PortNo: Byte; Var MT: Byte);
Procedure CheckTXMT(PortNo: Byte; Var MT: Byte);
Procedure CheckRDR(PortNo: Byte; Var InReddy: Byte);
Procedure CheckErrors(PortNo: Byte);


(*                    IMPLEMENTATION                     *)

Implementation

Procedure SetLegalBaud(Var Baud: Word);
{ changes an illegal baud rate to the nearest legal one }

   Var
     i, ndx: Byte;
     legal: Boolean;
     minerr, err, diff: Real;

   Begin
     WriteLn;
     WriteLn;
     WriteLn(bel, 'Illegal baud rate. ');
     WriteLn;
     WriteLn('Setting baud rate to nearest legal value.');
     WriteLn;
     WriteLn('Legal values are:');
     WriteLn;
     For i := 0 To 7 Do WriteLn(LegalBaud[i]);
     WriteLn;
     {find min. relative error}
     minerr := 1.0E+06;
     For i := 0 To 7 Do
     Begin
       diff := Abs(1.0 * baud - 1.0 * LegalBaud[i]);
       err := diff / LegalBaud[i];
       If err < minerr Then
       Begin
         minerr := err;
         ndx := i;
       End;
     End;
     WriteLn;
     { set new rate to nearest legal value }
     baud := LegalBaud[ndx];
     WriteLn('New baud rate is ', baud);
     WriteLn;
   End;

Procedure CheckLegalBaud(Var Baud: Word; Var baudbyte: Byte);
{ checks that the baud rate is a legal one and assigns baudbyte}

   Var
     i: Word;
     legal: Boolean;

   Begin
     legal := False;
     Repeat
       For i := 0 To 7 Do If baud = LegalBaud[i] Then
       Begin
         legal := True;
         baudbyte := i;
       End;
       If Not legal Then SetLegalBaud(baud);
     Until legal;
   End;

Procedure SetRTS(PortNo: Byte);

Var
  OutByte: Byte;

Begin
  OutByte := $00;
  { check status of other line and preserve it }
  If DTRset[PortNo] Then OutByte := SetDTRbit Else OutByte := $00;
  { construct control byte }
  OutByte := OutByte + SetRTSbit;
  { set boolean variable }
  RTSset[PortNo] := True;
  { send control byte to line control reg }
  Port[ComBase[PortNo] + ModemContReg] := OutByte;
End;


Procedure ClearRTS(PortNo: Byte);

Var
  OutByte: Byte;

Begin
  OutByte := $00;
  { check status of other line and preserve it }
  If DTRset[PortNo] Then OutByte := SetDTRbit Else OutByte := $00;
  { construct control byte }
  OutByte := OutByte And (Not SetRTSbit);
  { set boolean variable }
  RTSset[PortNo] := False;
  { send control byte to line control reg }
  Port[ComBase[PortNo] + ModemContReg] := OutByte;
End;


Procedure SetDTR(PortNo: Byte);

Var
  OutByte: Byte;

Begin
  OutByte := $00;
  { check status of other line and preserve it }
  If RTSset[PortNo] Then OutByte := SetRTSbit Else OutByte := $00;
  { construct control byte }
  OutByte := OutByte + SetDTRbit;
  { set boolean variable }
  DTRset[PortNo] := True;
  { send control byte to line control reg }
  Port[ComBase[PortNo] + ModemContReg] := OutByte;
End;


Procedure ClearDTR(PortNo: Byte);

Var
  OutByte: Byte;

Begin
  OutByte := $00;
  { check status of other line and preserve it }
  If RTSset[PortNo] Then OutByte := SetRTSbit Else OutByte := $00;
  { construct control byte }
  OutByte := OutByte And (Not SetDTRbit);
  { set boolean variable }
  DTRset[PortNo] := False;
  { send control byte to line control reg }
  Port[ComBase[PortNo] + ModemContReg] := OutByte;
End;

Procedure InitHandshakes(PortNo: Byte);
  { clears both DTR and RTS handshakes to initialize
   their corresponding boolean variables }
Begin
  ClearDTR(PortNo);
  ClearRTS(PortNo);
End;

Procedure CheckCTS(PortNo: Byte; Var SetClear: Byte);

{ Checks status of CTS.  If SetClear = 0, the bit
  is cleared; otherwise, it is set. }

(*     uses these constants:

      CTSbit: byte = $10;   { CTS }
      DSRbit: byte = $20;   { DSR }
                                                     *)


Var
  InByte: Byte;

Begin
  InByte := Port[ComBase[PortNo] + ModemStatReg];
  SetClear := InByte And CTSbit;
  { set boolean variable }
  If SetClear > 0 Then CTSset[PortNo] := True
  Else CTSset[PortNo] := False;
End;


Procedure CheckDSR(PortNo: Byte; Var SetClear: Byte);

{ Checks status of DSR.  If SetClear = 0, the bit
  is cleared; otherwise, it is set. }

(*     uses these constants:

      CTSbit: byte = $10;   { CTS }
      DSRbit: byte = $20;   { DSR }
                                                     *)


Var
  InByte: Byte;

Begin
  InByte := Port[ComBase[PortNo] + ModemStatReg];
  SetClear := InByte And DSRbit;
  { set boolean variable }
  If SetClear > 0 Then DSRset[PortNo] := True
  Else DSRset[PortNo] := False;
End;


Procedure CheckTHRE(PortNo: Byte; Var MT: Byte);

{ Checks status of transmit holding register. If MT = 0, the register
  is full, otherwise, it is empty. }

Var
  InByte: Byte;

Begin
  InByte := Port[ComBase[PortNo] + LineStatReg];
  MT := InByte And THREbit;
  { set boolean variable }
  If MT > 0 Then THRE[PortNo] := True
  Else THRE[PortNo] := False;
End; { Procedure CheckTHRE }


Procedure CheckTXMT(PortNo: Byte; Var MT: Byte);

{ Checks status of transmit register. If MT = 0, the register
  is full, otherwise, it is empty. }

Var
  InByte: Byte;

Begin
  InByte := Port[ComBase[PortNo] + LineStatReg];
  MT := InByte And TXMTBit;
  { set boolean variable }
  If MT > 0 Then TXMT[PortNo] := True
  Else TXMT[PortNo] := False;
End;


Procedure CheckRDR(PortNo: Byte; Var InReddy: Byte);

{ Checks status of receive data ready bit. If InReddy = 0, no data
  ready; otherwise, received data ready. }

Var
  InByte: Byte;

Begin
  InByte := Port[ComBase[PortNo] + LineStatReg];
  InReddy := InByte And RDRBit;
  { set boolean variable }
  If InReddy > 0 Then RDR[PortNo] := True
  Else RDR[PortNo] := False;
End;


Procedure CheckErrors(PortNo: Byte);
{ Checks for error bits in line staus reg. }
(*

    uses these constants:

   OverRunBit: byte = $02;       { overrun error }
   ParityErrorBit: byte = $04;   { parity error }
   FramingErrorBit: byte = $08;  { framing error }

                                                  *)

Var
  InByte, ErrBit: Byte;

Begin

  InputError[PortNo] := False;
  InByte := Port[ComBase[PortNo] + LineStatReg];
  ErrBit := InByte And OverRunBit;
  { set boolean variable }
  If ErrBit > 0 Then InputError[PortNo] := True;
  ErrBit := InByte And ParityErrorBit;
  { set boolean variable }
  If ErrBit > 0 Then InputError[PortNo] := True;
  ErrBit := InByte And FramingErrorBit;
  { set boolean variable }
  If ErrBit > 0 Then InputError[PortNo] := True;
End;


Procedure BIOSInitComPort(PortNo: Byte; Var baud: Word; DataBits: Byte;
                                 parity: Char; StopBits: Byte);

{ This procedure uses a BIOS call to initialize com port baud rate,
  parity, stop bits, and word length. See Waite Group book, p. 467. }

Var CPUregs: Registers;
  i: Word;
  baudbyte: Byte;

Begin
  CheckLegalBaud(Baud, baudbyte);
  parity := UpCase(parity);
  With CPUregs Do
  Begin
    AX := 0;
    DX := PortNo - 1;         { Port no. - 1 in DX }
    AX := baudbyte ShL 5;     { Set baud rate part of AX }
    Case parity Of
      'O': AX := AX +(1 ShL 3);
      'E': AX := AX +(3 ShL 3);
      { 'N' doesn't appear because zero in these bits is 'no parity' }
    End;
    AX := AX +((StopBits - 1) ShL 2);   {Set AX for stop bits}
    AX := AX  +(DataBits - 5);          {Set AX for data bits}
    Intr($14, CPUregs);         {Call Bios to initialize the port}
  End;
  { clear interrupts and DLAB.  Is this needed? }
  (*   Port[ComBase[PortNo] + LineContReg] := $00;    *)
End;


Procedure ReadByte2(PortNo: Byte; Var inbyte: Byte; InShake: String4);

{ does not halt if input error is detected dl 7-26-96 }
{ Reads a byte after changing status of handshake line(s), etc. }
{ Conversion to type char must be done by the calling program.}

{ The string variable InShake can take on values 'DTR', 'RTS',
  'BOTH', or 'NONE' and determines which handshake lines are used. }

Var
  DataStat: Byte;
  nloops: Real;  {quit after maxloops tries }

Begin
  { set device handshake(s) }
  If ((InShake = 'DTR') Or (InShake = 'BOTH')) Then SetDTR(PortNo);
  If ((InShake = 'RTS') Or (InShake = 'BOTH')) Then SetRTS(PortNo);
  TimeOut := False;
  nloops := 0.0;
  Repeat
    Delay(1);
    nloops := nloops + 1.0;
    { check for data available }
    CheckRDR(PortNo, DataStat);
  Until((DataStat > 0) Or (nloops = maxloops));
  { clear device handshake(s) }
  If ((InShake = 'DTR') Or (InShake = 'BOTH')) Then ClearDTR(PortNo);
  If ((InShake = 'RTS') Or (InShake = 'BOTH')) Then ClearRTS(PortNo);
  If nloops >= maxloops Then
  Begin
    TimeOut := True;
    Write(bel, bel); { Timed out waiting for input }
    Exit;
  End;
  CheckErrors(PortNo);
  If InputError[PortNo] Then
  Begin
    Write(bel, bel); { , 'Input error..'); }
    Exit;
  End;
  InByte := Port[ComBase[PortNo]];
End;

Procedure DumReadByte(PortNo: Byte; Var inbyte: Byte; InShake: String4);
{ Reads a byte after changing status of handshake line(s), etc. }

{ this routine checks InStatusOK to catch ALL input errors.
  dl 7-26-96  dl  }

{ Conversion to type char must be done by the calling program.}

{ The string variable InShake can take on values 'DTR', 'RTS',
  'BOTH', or 'NONE' and determines which handshake lines are used. }


Var
  DataStat: Byte;
  nloops: Real;  {quit after maxloops tries }

Begin
  { set device handshake(s) }
  If ((InShake = 'DTR') Or (InShake = 'BOTH')) Then SetDTR(PortNo);
  If ((InShake = 'RTS') Or (InShake = 'BOTH')) Then SetRTS(PortNo);
  TimeOut := False;
  nloops := 0.0;
  Repeat
    Delay(1);
    nloops := nloops + 1.0;
    { check for data available AND no errors }
  Until((InStatusOK(PortNo)) Or (nloops = maxloops));
    { clear device handshake(s) }
  If ((InShake = 'DTR') Or (InShake = 'BOTH')) Then ClearDTR(PortNo);
  If ((InShake = 'RTS') Or (InShake = 'BOTH')) Then ClearRTS(PortNo);
  If nloops >= maxloops Then
  Begin
    TimeOut := True;
    Write(bel, bel); { Timed out waiting for input }
    Exit;
  End;
  InByte := Port[ComBase[PortNo]];
End;



Function InStatusOK(PortNo: Byte): Boolean;
{  checks for RDR bit and errors.  If InStatusOK is TRUE, there
  is a byte in the input register, and there are no parity,
  framing, or overrun errors. Added to accommodate ill-designed
  instruments with no input handshake lines at all. dl }

{ USES NO HANDSHAKE SIGNALS!! DOES NOT LOOP LIKE ReadByte!! }

Var
  RDRStat: Byte;  { set > 0 for a byte in the RD register }

Begin
  InStatusOK := True;
  CheckRDR(PortNo, RDRStat);
  { variable RDR[Portno] set TRUE if data byte ready }
  CheckErrors(PortNo);
  { variable InputError[Portno] set TRUE if any data error }
  If ((Not RDR[Portno]) Or (InputError[portno])) Then
    InStatusOK := False;
End;

Procedure GetFirstByte(portno: Byte);
Var
  inreddy, inbyte: Byte;

Begin
  CheckRDR(PortNo, inreddy);
  If inreddy > 0 Then
    ReadByte2(PortNo, inbyte, 'NONE');
  (* writeln('First byte in UART = ', chr(inbyte)); *)
End;


Procedure ReadALine4(PortNo: Byte; Var InString: String;
                               InShake, delim: String4);

{ adapted from Procedure ReadALine, which looks for CR/LF as a delimiter.
  This one  allows CR, LF, CR/LF, or LF/CR as delimiters. The string
  variable delim can take on values 'CR', 'LF', 'CRLF', OR 'LFCR'. }

{ Reads a string of characters(bytes, actually) and stops at delimiter. }

{ The string variable InShake can take on values 'DTR', 'RTS',
  'BOTH', or 'NONE' and determines which handshake lines are used. }

{ modified not to give range check error if no bytes at all are read
          4/1/96 dl  }

{ modified not to give range check error if no delimiter is
    encountered in the input stream 8-4-96 dl }

Var
  i, numbytes, stringlen:  Byte;
  inbyte,    { byte read from UART }
  lastbyte,  { last byte of delimiter }
  numdelim: Byte;  { number of characters(bytes) in delimiter }
  inbyteray: Array[1..255] Of Byte; { bytes read from UART }
  maxbytes:  word;

Begin
  If ((delim = 'CRLF') Or (delim = 'LFCR')) Then
    numdelim := 2 Else numdelim := 1;
  If ((delim = 'LF') Or (delim = 'CRLF')) Then
    lastbyte := LFbyte Else lastbyte := CRbyte;

  maxbytes := 255 + numdelim;  { max string length = 255 }
  numbytes := 0;
  Repeat
    numbytes := numbytes + 1;
    ReadByte2(PortNo, inbyteray[numbytes], InShake);
    { writeln(xfile, inbyteray[numbytes], '  ',  chr(inbyteray[numbytes]));}
  Until((inbyteray[numbytes] = lastbyte) Or (TimeOut) or (numbytes = maxbytes));

  If TimeOut Then instring := 'Timed out waiting for input.'
  Else If InputError[PortNo] Then instring :=  'Input error.'
  Else
  Begin
    { convert byte array to string and discard last one or two
    characters. }
    stringlen := numbytes - numdelim;
    instring[0] := Chr(stringlen); { put string length in location 0 }
    For i := 1 To stringlen Do InString[i] := Chr(inbyteray[i]);
  End;
End;

Procedure DumReadALine(PortNo: Byte; Var InString: String;
                               InShake, delim: String4);

{ adapted from Procedure ReadALine, which looks for CR/LF as a delimiter.
  This one  allows CR, LF, CR/LF, or LF/CR as delimiters. The string
  variable delim can take on values 'CR', 'LF', 'CRLF', OR 'LFCR'. }

{ Reads a string of characters(bytes, actually) and stops at delimiter. }

{ The string variable InShake can take on values 'DTR', 'RTS',
  'BOTH', or 'NONE' and determines which handshake lines are used. }

{ modified not to give range check error if no bytes at all are read
          4/1/96 dl  }

{ this version uses DumReadByte instead of ReadByte }

Var
  i, numbytes, stringlen: Byte;
  inbyte,    { byte read from UART }
  lastbyte,  { last byte of delimiter }
  numdelim: Byte;  { number of characters(bytes) in delimiter }
  inbyteray: Array[1..255] Of Byte; { bytes read from UART }

Begin
  If ((delim = 'CRLF') Or (delim = 'LFCR')) Then
    numdelim := 2
  Else numdelim := 1;
  If ((delim = 'LF') Or (delim = 'CRLF')) Then
    lastbyte := LFbyte
  Else lastbyte := CRbyte;

  numbytes := 0;
  Repeat
    numbytes := numbytes + 1;
    DumReadByte(PortNo, inbyteray[numbytes], InShake);
    { writeln(xfile, inbyteray[numbytes], '  ',  chr(inbyteray[numbytes]));}
  Until((inbyteray[numbytes] = lastbyte) Or (TimeOut));
  { convert byte array to string and discard last one or two
  characters. }
  If TimeOut Then instring := 'Timed out waiting for input.'
  Else
  Begin
    stringlen := numbytes - numdelim;
    instring[0] := Chr(stringlen); { put string length in location 0 }
    For i := 1 To stringlen Do InString[i] := Chr(inbyteray[i]);
  End;
End;


Procedure SendByte(PortNo, outbyte: Byte; OutShake: String4);
{ Sends a byte after checking status of handshake line(s), etc. }
{ The calling program must convert characters to bytes
  before making a calls to this procedure. }
{ the string variable OutShake can take on values 'CTS', 'DSR',
  'BOTH' or 'NONE' and determines which handshake lines are used. }

Var
  Reddy: Boolean;
  Stat: Byte;
  nloops: Real;

Begin
  TimeOut := False;
  nloops := 0.0;
  Repeat
    nloops := nloops + 1.0;
    Reddy := True;
    { check device handshake(s) }
    If ((OutShake = 'CTS') Or (OutShake = 'BOTH')) Then CheckCTS(PortNo, Stat);
    If Stat = 0 Then Reddy := False;
    If ((OutShake = 'DSR') Or (OutShake = 'BOTH')) Then CheckDSR(PortNo, Stat);
    If Stat = 0 Then Reddy := False;
    { check transmit holding register }
    Delay(1);
    CheckTHRE(PortNo, Stat);
    If Stat = 0 Then Reddy := False;
  Until((Reddy) Or (nloops = maxloops));
  If nloops >= maxloops Then
  Begin
    TimeOut := True;
    Write(bel, bel); {, 'Timed out waiting for output handshake.');}
    Exit;
  End;
  Port[ComBase[PortNo]] := outbyte;
End;

Procedure SendALine2(PortNo: Byte; OutString: String;
                                      OutShake, delim: String4);

{ adapted from SendALine, which only had CR/LF as a delimiter }

{ Sends a string of characters(bytes, actually) and winds it
   up with a delimiter. }

{  This one  allows CR, LF, CR/LF, or LF/CR as delimiters. The string
  variable delim can take on values 'CR', 'LF', 'CRLF', OR 'LFCR'. }

{ the string variable OutShake can take on values 'CTS', 'DSR',
  'BOTH' or 'NONE' and determines which handshake lines are used. }

Var
  i, stringlen: Byte;
  outbyteray: Array[1..255] Of Byte;

Begin
  { convert to byte }
  stringlen := Length(OutString);
  For i := 1 To stringlen Do outbyteray[i] := Ord(OutString[i]);
  For i := 1 To stringlen Do
  Begin
    SendByte(PortNo, outbyteray[i], OutShake);
    If timeout Then Exit;
  End;
  { send delimiter byte(s) }
  If ((delim = 'CR') Or (delim = 'CRLF')) Then
    SendByte(PortNo, CRbyte, OutShake);
  If ((delim = 'LF') Or (delim = 'CRLF')) Then
    SendByte(PortNo, LFbyte, OutShake);
  If delim = 'LFCR' Then SendByte(PortNo, CRbyte, OutShake);
End;
End.


