//--------------------------------------------------------------------------
//
//      SCRCOPY.CPP: Screen save demo for DOS TSR class.
//      (c) J.English 1993 (je@unix.brighton.ac.uk)
//
//      This example saves an image of the screen to disk when the
//      hotkey is pressed.  It demonstrates a critical-error handler.
//
//--------------------------------------------------------------------------

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <bios.h>
#include <dos.h>
#include "tsr.h"

//----- Error messages for TSR status values -----
const char* status_msg [] = {
    "success",
    "DOS version 3 or higher required",
    "multiple TSR instances declared in program", 
    "can't allocate stack",
    "can't allocate multiplex function code",
    "already loaded",
    "can't make program resident",
    "startup error"
};

//----- Error messages for unload result -----
const char* unload_msg [] = {
    "unloaded",
    "not loaded",
    "can't be unhooked",
    "can't be unloaded",
    "environment can't be unloaded"
};

//----- The derived TSR class -----
class ScreenCopy : public TSR
{
  private:
    virtual void main (int);            // body of screen copy TSR
    char filename [200];                // file to save screen image to
  public:
    ScreenCopy () :                     // constructor
        TSR ("TSR screen copier demo")  // ... call base class constructor
        { }                             // ... nothing else needed

    TSR::critical_code critical_error (int n);
                                        // critical error handler
    void open (const char* name)        { strcpy (filename, name); }
    const char* file ()                 { return filename; }
};

//----- The main program -----
int main (int argc, char** argv)
{
    ScreenCopy copier;

    if (argc == 2 && argv[1][0] != '/')
    {
        copier.open (argv[1]);
        cout << "Installing " << copier.name() << " in memory\n";
        cout << "Press ALT-F10 to save screen to " << argv[1] << "\n";
        copier.run (TSR::ALT + TSR::F10);
        cout << "Error installing " << copier.name() << ": "
             << status_msg [copier.status()] << "\n";
    }
    else if (argc == 2 && argv[1][0] == '/' && toupper (argv[1][1]) == 'U')
    {   cout << copier.name() << " "
             << unload_msg [copier.unload()] << "\n";
    }
    else
    {   cout << "Usage: SCRCOPY [filename | /U]\n";
        return -1;
    }
    return copier.status ();
}

//----- Resident part of TSR -----
void ScreenCopy::main (int)
{
    union REGS r;                           // get video mode
    r.h.ah = 0x0F;
    int86 (0x10, &r, &r);

    if (r.h.al > 3)                         // exit if not text mode 0-3
        return;

    char far* screen =                      // address of screen
        (char far*) MK_FP(0xB800 + r.h.bh * 256, 0);
    char line [82];                         // line buffer for screen image
    ofstream f (file());                    // open output file

    for (int i = 0; i < 25; i++)            // copy 25 rows
    {   for (int c = 0; c < r.h.ah; c++)    // copy each column
        {   line [c] = *screen;
            screen += 2;
        }
        line [c]   = '\n';                  // bad luck if char 0A on screen!
        line [c+1] = '\0';
        f << line;
    }
    f.close ();
}

//----- Critical error handler -----
TSR::critical_code ScreenCopy::critical_error (int n)
{
    union REGS r;                           // get video mode
    r.h.ah = 0x0F;
    int86 (0x10, &r, &r);
    unsigned far* screen =                  // address of screen
        (unsigned far*) MK_FP(0xB800 + r.h.bh * 256, 0);

    unsigned line [80];                     // save top line of screen
    int i;
    for (i = 0; i < r.h.ah; i++)
        line[i] = screen[i];

    char message [81];                      // display message
    sprintf (message,
             "Critical error %02X: Retry, Ignore or Fail?", n);
    for (i = 0; i < r.h.ah; i++)
        screen[i] = 0x1E00 + (i < strlen(message) ? message[i] : ' ');

    TSR::critical_code result;              // get response from user
    char c;
    do {
        c = toupper (bioskey (0));
        if (c == 'R')
            result = TSR::RETRY;
        if (c == 'I')
            result = TSR::IGNORE;
        if (c == 'F')
            result = TSR::FAIL;
    } while (c != 'R' && c != 'I' && c != 'F');

    for (i = 0; i < r.h.ah; i++)        // restore top line
        screen[i] = line[i];

    return result;                          // return user's response
}

