#ifndef _PRSOUND_            
#define _PRSOUND_           /* Mark this header already included */

#ifdef __cplusplus
 extern "C" {
#endif

#include "audio.h"

/* Result codes */
#define SOUND_NOT_FOUND                          -17
#define SOUND_OUT_OF_SPACE                       -18
#define SOUND_DELETED                            -19
#define SOUND_LOAD_ERROR                         -20
#define SOUND_ALREADY_LOADED                     -21

/* Sound Request Modes */
#define SOUND_INACTIVE                             0
#define SOUND_ACTIVE                               1
#define SOUND_START_PLAYING                        2
#define SOUND_STOP_PLAYING                         4

/* 3D Sound Options */
#define SOUND_NORMAL_PANNING                       0
#define SOUND_3D_PANNING                           1
#define SOUND_NORMAL_VOLUME                        0
#define SOUND_3D_VOLUME                            1
#define SOUND_DOPPLER_OFF                          0
#define SOUND_DOPPLER_ON                           1


typedef struct
{
 char *                 name;
 LPAUDIOWAVE            data;
} PR_SOUND;                             /* WAV file type */


typedef struct
{
 PR_DWORD active;               /* 1 if active,
                                   2 if waiting to be played,
                                   4 if stop is requested */
 PR_DWORD voice;                /* Voice number */
 PR_DWORD sound;                /* Sound number */

 PR_DWORD soundnum;             /* number in 3D sound list */
 void *   soundptr;             /* Pointer to 3D sound not from list */

 PR_DWORD volume;               /* Current Volume (0-64) */
 PR_DWORD panning;              /* Current Panning (0-256) */
 PR_DWORD pitch;                /* Original pitch */

 /* 3D parameters */
 PR_DWORD panning_mode;         /* 1 if 3D panning */
 PR_DWORD volume_mode;          /* 1 if 3D volume */
 PR_DWORD shift_mode;           /* 1 if doppler shift */
 PR_DWORD pitch_slide;

 PR_REAL  panning_factor;
 PR_REAL  volume_factor;
 PR_REAL  shift_factor;

 PR_DWORD current_pitch;
 PR_REAL  x, y, z;              /* Current 3D position */
 PR_REAL  lastdistance;         /* Last 3D position */
 PR_ENTITY *Attached_Entity;    /* Entity that controls the sound */

} PR_3DSOUND;                   /* 3D Sound type */


typedef struct {
  PR_WORD  ProductID;
  char     Name[30];
  PR_DWORD Formats;
} PR_SOUNDDEVICE;               /* Sound Device type */
extern PR_SOUNDDEVICE PRSND_DeviceList[30];


extern PR_DWORD      PRSND_NumDevices;
extern PR_DWORD      PRSND_MaxVoices;
extern char *        PRSND_SoundPath;                              
extern PR_SOUND *    PRSND_Sounds;         /* List of sounds */        
extern PR_DWORD      PRSND_NumSounds;      /* Number of sounds in the list */
extern LPAUDIOMODULE PRSND_Song;
extern HAC           PRSND_VoiceList[AUDIO_MAX_VOICES];
extern PR_3DSOUND *  PRSND_3DSoundList[AUDIO_MAX_VOICES];



PR_DWORD PRSND_AddSound (char *filename, LPAUDIOWAVE data);
/* -----------------------------------------------------------------------
   Function:   PRSND_AddSound
   Purpose:    Adds a sound to the PRSND_Sounds array
   Parameters: filename - Filename of the table
               data     - Pointer to the wave data
   Return:     Sound number
   ----------------------------------------------------------------------- */


PR_3DSOUND *PRSND_Alloc3DSound (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_Alloc3DSound
   Purpose:    Allocates space for a 3D Sound
   Parameters: None
   Return:     Pointer to the structure
   ----------------------------------------------------------------------- */


void PRSND_AllocSounds (PR_DWORD maxsound);
/* -----------------------------------------------------------------------
   Function:   PRSND_AllocSounds
   Purpose:    Allocates room for a list of sounds
   Parameters: maxsound - Number of sounds to allocate space for
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_CloseAudio (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_CloseAudio
   Purpose:    Closes the audio system
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_CloseVoices (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_CloseVoices
   Purpose:    Removes all of the voices
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_DeleteAllSounds (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_DeleteAllSounds
   Purpose:    Deletes all the sound from the PRSND_Sounds array
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PRSND_DeleteSound (char *filename);
/* -----------------------------------------------------------------------
   Function:   PRSND_DeleteSound
   Purpose:    Deletes a sound from the PRSND_Sounds array 
   Parameters: filename
   Return:     SOUND_NOT_FOUND or SOUND_DELETED
   ----------------------------------------------------------------------- */


void PRSND_FindDevices (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_FindDevices
   Purpose:    Adds all of the available devices to the PRSND_DeviceList array
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PRSND_FindFreeVoice (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_FindFreeVoice
   Purpose:    Finds the next available voice
   Parameters: None
   Return:     Voice number
   ----------------------------------------------------------------------- */


PR_DWORD PRSND_FindSound (char *filename);
/* -----------------------------------------------------------------------
   Function:   PRSND_FindSound
   Purpose:    Finds the sound number given its filename
   Parameters: filename - Filename of the sound
   Return:     SOUND_NOT_FOUND if the file is not in the PRSND_Sounds array
               or Sound number
   ----------------------------------------------------------------------- */


void PRSND_FreeSong (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_FreeSong
   Purpose:    Removes a song from memory
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PRSND_Get3DSoundStatus (PR_3DSOUND *soundptr);
/* -----------------------------------------------------------------------
   Function:   PRSND_Get3DSoundStatus
   Purpose:    Returns 1 if the sound is playing, 0 otherwise
   Parameters: soundptr - Pointer to the 3D sound
   Return:     1 if playing, 0 otherwise
   ----------------------------------------------------------------------- */


PR_DWORD PRSND_GetLength (PR_DWORD sound);
/* -----------------------------------------------------------------------
   Function:   PRSND_GetLength
   Purpose:    Returns the length of a sample
   Parameters: sound - Sounds number
   Return:     Length of sample
   ----------------------------------------------------------------------- */


void PRSND_Initialize (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_Initialize
   Purpose:    Initializes the audio system
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_InitializeVoices (PR_DWORD maxvoice);
/* -----------------------------------------------------------------------
   Function:   PRSND_InitializeVoices
   Purpose:    Initializes the voices used for sound effects
   Parameters: maxvoice - Number of voices available for effects
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_LoadSong (char *filename);
/* -----------------------------------------------------------------------
   Function:   PRSND_LoadSong
   Purpose:    Loads a song from disk
   Parameters: filename - Filename of the song
   Return:     None
   ----------------------------------------------------------------------- */


PR_DWORD PRSND_LoadSound (char *filename);
/* -----------------------------------------------------------------------
   Function:   PRSND_LoadSound
   Purpose:    Loads a sound from disk
   Parameters: filename - Filename of the sound
   Return:     SOUND_OUT_OF_SPACE, SOUND_LOAD_ERROR, SOUND_ALREADY_LOADED or
               sound number
   ----------------------------------------------------------------------- */


void PRSND_OpenAudio (PR_DWORD device, PR_DWORD format, PR_DWORD rate);
/* -----------------------------------------------------------------------
   Function:   PRSND_OpenAudio
   Purpose:    Opens an audio device
   Parameters: device - Device number
               format - Format flags
               rate   - Mixing rate
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Play3DSound (PR_3DSOUND *soundptr, PR_DWORD num);
/* -----------------------------------------------------------------------
   Function:   PRSND_Play3DSound
   Purpose:    Sets a 3D sound to start playing
   Parameters: soundptr - Pointer to the 3D sound
               num - Sound number
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_PlaySong (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_PlaySong
   Purpose:    Starts playing a song
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_SetLoopEnd (PR_DWORD sound, PR_DWORD pos);
/* -----------------------------------------------------------------------
   Function:   PRSND_SetLoopEnd
   Purpose:    Sets the end position of sound loop
   Parameters: sound - Sound number
               pos - Position within sample
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_SetLoopMode (PR_DWORD sound, PR_DWORD mode);
/* -----------------------------------------------------------------------
   Function:   PRSND_SetLoopMode
   Purpose:    Sets the looping mode of a sample
   Parameters: sound - Sound number
               mode - Looping mode
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_SetLoopStart (PR_DWORD sound, PR_DWORD pos);
/* -----------------------------------------------------------------------
   Function:   PRSND_SetLoopStart
   Purpose:    Sets the start position of sound loop
   Parameters: sound - Sound number
               pos - Position within sample
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_SetOrigin (PR_REAL x, PR_REAL y, PR_REAL z);
/* -----------------------------------------------------------------------
   Function:   PRSND_SetOrigin
   Purpose:    Sets the position of the listener
   Parameters: x, y, z - 3D coordinate of listener
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_SetSongVolume (PR_DWORD volume);
/* -----------------------------------------------------------------------
   Function:   PRSND_SetSongVolume
   Purpose:    Sets the volume of the song
   Parameters: volume - between 0 and 64
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_SetSoundPath (char *pathname);
/* -----------------------------------------------------------------------
   Function:   PRSND_SetSoundPath
   Purpose:    Sets the path prefix for sounds 
   Parameters: pathname - Path of sounds data files
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_StopSong (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_StopSong
   Purpose:    Stops a song while playing
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundCoordinate (PR_3DSOUND *soundptr, PR_REAL x, PR_REAL y, PR_REAL z, PR_ENTITY *ent);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundCoordinate
   Purpose:    Sets the coordinate of a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               x,y,z - 3D coordinate
               ent - Pointer to entity it belongs to (NULL if none)
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundPanning (PR_3DSOUND *soundptr, PR_DWORD pan);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundPanning
   Purpose:    Sets the panning value for a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               pan - value between 0 and 256 (128 = middle)
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundPanningFactor (PR_3DSOUND *soundptr, PR_REAL factor);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundPanningFactor
   Purpose:    Sets the panning factor for a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               factor - Distance to a full right or left pan
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundPanningMode (PR_3DSOUND *soundptr, PR_DWORD mode);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundPanningMode
   Purpose:    Sets the panning mode for a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               mode - Panning mode 
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundShiftFactor (PR_3DSOUND *soundptr, PR_REAL factor);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundShiftFactor
   Purpose:    Sets the Doppler shift factor for a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               factor - shift factor based on velocity
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundShiftMode (PR_3DSOUND *soundptr, PR_DWORD mode);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundShiftMode
   Purpose:    Sets the Doppler shift mode for a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               mode - Doppler shift mode
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundShiftSlide (PR_3DSOUND *soundptr, PR_DWORD slide);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundShiftSlide
   Purpose:    Sets the Doppler shift slide value for a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               slide - Slide value
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundVolume (PR_3DSOUND *soundptr, PR_DWORD vol);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundVolume
   Purpose:    Sets the volume of a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               vol - A value between 0 dn 63
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundVolumeFactor (PR_3DSOUND *soundptr, PR_REAL factor);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundVolumeFactor
   Purpose:    Sets the volume factor of a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               factor - Distance where a sound cannot be heard
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Set3DSoundVolumeMode (PR_3DSOUND *soundptr, PR_DWORD mode);
/* -----------------------------------------------------------------------
   Function:   PRSND_Set3DSoundVolumeMode
   Purpose:    Sets the volume mode of a 3D sound
   Parameters: soundptr - Pointer to the 3D sound
               mode - Volume mode
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_Update3DSound (PR_3DSOUND *sound);
/* -----------------------------------------------------------------------
   Function:   PRSND_Update3DSound
   Purpose:    Updates a 3D sound 
   Parameters: sound - Pointer to the 3D sound
   Return:     None
   ----------------------------------------------------------------------- */


void PRSND_UpdateVoices (void);
/* -----------------------------------------------------------------------
   Function:   PRSND_UpdateVoices
   Purpose:    Updates all of the active 3D voices
   Parameters: None
   Return:     None
   ----------------------------------------------------------------------- */


#ifdef __cplusplus
}
#endif

#endif



