/***************************************************************

	sfs.h           Space Flight Simulator
			General Header

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

/***    Version number -- for all modules */

#define VERSION "1.01"

/***    Definitions for all sfs modules */

#ifndef CR
#define CR      0x0d
#endif

#ifndef LF
#define LF      0x0a
#endif

#ifndef TRUE
#define TRUE    1
#define FALSE   0
#endif

#define FAR_SEGMENT
#define FMALLOC         malloc

#define FT_POINTS       64              /* Points in forward track buffer */
#define ORBITS          (16)            /* Number of orbits tracked     */
#define GT_POINTS       512             /* Size of ground track buffer  */
#define PE_POINTS       64              /* Size of Perspective buffer   */
#define OR_COLORS       6               /* Colors for orbits */
#define HORDEGS         30.0            /* Horizontal degrees in window
					   for visual display */
#define DEFAULTFD       "earth.fd"      /* default focal data file */
#define DEFAULTSD       "null.spd"      /* default surface data point file */
#define DEFAULTORB      "orb.spd"       /* default orb point file */
#define DEFAULTGRID     "latlon.spd"    /* default grid point file */
#define SFS_DEFAULTPROG "default.sfs"   /* default program file */
#define SFS_CHAINFN     "chain"         /* name for chian file */
#define SFS_EXITARG     "exit"          /* argument denotes exit from
					   program */
#define SFS_CONTARG     "cont"          /* argument denotes continuation
					   (do not show logo) */
#define SFS_FILESIZE    36              /* size for filename buffers */

/***    Definitions of names for chain files */

#define SFSCH_EXIT      "exit"          /* exit from SFS */
#define SFSCH_SIMUL     "sfsx.app"      /* chain to simulation module */
#define SFSCH_MODEL     "sfsm.app"      /* chain to modeling module */
#define SFSCH_ATLAS     "sfsa.app"      /* chain to atlas module */

/***    Definitions of Display Types */

#define SFSD_VISUAL     0               /* visual simulation display */
#define SFSD_PERSP      1               /* distant perspective display */
#define SFSD_GROUND     2               /* ground track display */

/***    Definitions of Visual Simulation Elements */

#define VI_GRIDFRONT    1               /* show front of grid */
#define VI_GRIDBACK     2               /* show back of grid */
#define VI_SURFRONT     4               /* show front of surface features */
#define VI_SURBACK      8               /* show back of surface features */
#define VI_ORBOUTLINE   16              /* show outline of orb */
#define VI_ORBBLANK     32              /* blank the orb before drawing */
#define VI_POINTS       64              /* show points */
#define VI_PTITLES      128             /* show point titles */
#define VI_STITLE       256             /* show screen title */
#define VI_CROSSH       512             /* show crosshair */

/***    Alignments for Visual Simulation DIsplay */

#define AL_HORIZON      0
#define AL_CENTER       1

/***************************************************************

	language-specific definitions

	English is taken as a default, and if SFS_TITLE
	has not been defined, the English definitions will be
	utilized. To translate into other languages, we suggest
	that the following section be copied, a language name
	defined as one, and then each of the definitions rendered
	into the language. Example:

	;#define        LATIN   1
	;
	;
	;#if    LATIN
	;#define        SFS_TITLE        " Astronautica Simulanda "
	;......etc.
	;#endif

	SFS_TITLE will then be defined, and the English language
	tags will not be called.

***************************************************************/

#ifndef SFS_TITLE
#define SFS_TITLE       "Space Flight Simulator"
#define VERABB          "ver."
#define SFS_LINKERR     "SFS failed to link to the Main Module.\n"
#define ERR_LTITLE      "Link error"
#define SFS_CHERR       "SFS failed to chain to %s\n"
#define ERR_TTITLE      "SFS termination error"
#define DI_MEMERR       "Cannot allocate memory for display initialization"
#define DI_DSMEMERR     "Cannot allocate memory for display structures"
#define DI_DMEMERR      "Cannot allocate memory for display initialization"
#define VIS_TITLE       "Visual: %s from %s (Orbit %d)"
#define PER_TITLE       "Perspective: %s orbits"
#define GTR_TITLE       "Ground Track: %s orbits"
#define FT_NOTICE       "Initializing forward track for orbit %d"
#define FT_CALC         "Calculating forward track for orbit %d"
#define GT_DRAW         "Drawing ground track"
#define GT_WTITLE       " Ground Track: %s orbit(s)"
#define GTERR_NODATA    "gt_track(): data is not ready."
#define PE_CALC         "Calculating perspective points for orbit %d"
#define PE_DRAW         "Drawing distant perspective"
#define PE_WTITLE       " %s from %.0lf km; point = %ld seconds "
#define PRERR_OPEN      "Cannot open program file <%s> "
#define PR_NOTICE       "Interpreting program file %s; %s "
#define PRERR_PRI       "Periapsis must be > 0 "
#define PRERR_APO       "Apoapsis must be > 0 and <= periapsis"
#define PRERR_INC       "Inclination must be > %.1lf and < %.1lf"
#define PRERR_LAN       "Long. of asc. node must be > %.1lf and < %.1lf"
#define PRERR_ARP       "Argument of periapsis must be >= %.1lf and < %.1lf"
#define PRMEMERR_NAME   "Cannot allocate storage for orbit name."
#define PRERR_TF        "Time factor must be > 0"
#define PRERR_UI        "Update interval must be > 0 "
#define PRERR_TL        "Trig level must be 1 or 2"
#define PRERR_INS       "Insertion must be > 0"
#define PRERR_UNKNOWN   "Unknown command %s "
#define PRMEMERR_SORBIT "Cannot allocate storage for sfs orbit."
#define PRMEMERR_AORBIT "Cannot allocate storage for as orbit."
#define PRERR_WRITE     "Cannot open program file %s for writing."
#define PRMEMERR_FOCUS  "Cannot allocate storage for as focus."
#define VI_DRAW         "Drawing visual simulation"
#define VI_WTITLE       " Visual Simulation: %s from %s "
#define	VI_POS		"Position"
#define	VI_ORBIT	"Orbit"
#define	VI_LAT		"LAT"
#define	VI_LON		"LON"
#define VI_ALT          "ALT"
#define TF_TITLE        "Set Time Factor"
#define TF_LEGEND       "The time factor is the ratio between computed time and real time.  A time factor of 1 is equivalent to real-time.  Enter the time factor as a whole number."
#define TF_PROMPT       "Time Factor:  "
#define UI_TITLE        "Set Screen Update Interval"
#define UI_LEGEND       "The screen update interval determines the amount of time (in seconds) between screen updates.  Enter the screen update interval in seconds."
#define UI_PROMPT       "Update Interval:   "
#define TPL_TITLE       "Set Trig Precision Level"
#define TPL_LEGEND      "The Trig Precision Level specifies the accuracy of trig calculations:  1 = fast, 2 = accurate.  Enter 1 or 2. "
#define TPL_PROMPT      "Precision Level:  "
#define MINERR          "ERROR in entry:  Value entered must be more than"
#define MAXERR          "ERROR in entry:  Value entered must be less than"
#define PTA             "Please try again"
#define	SFS_TOGGLE	"Toggle"
#define PR_PARAMS       "parameters for orbit"
#define PR_ENDOF        "end of file"
#define PR_CREATED      "created"
#endif

/***    Structure definitions for all sfs modules */

struct ft_pt                            /* forward track point structure */
   {
   double  latitude;                    /* latitude of the point */
   double  longitude;                   /* longitude of the point */
   double  altitude;                    /* altitude of the point */
   };

struct sfs_orbit
   {
   struct as_orbit *aorbit;             /* as orbit structure */
   char *name;                          /* name of this orbit */
   struct ft_pt ft_buffer[ FT_POINTS ]; /* Buffer for forward track */
   int ft_bufstart;                     /* Start of forward track buffer */
   int ft_bufpos;                       /* End of forward track buffer  */
   long ft_inc;                         /* Time increment for f.t. */
   double gt_buffer[ GT_POINTS ][ 2 ];  /* Ground track buffer */
   int gt_bufstart;                     /* Start of ground track buffer */
   int gt_bufpos;                       /* End of ground track buffer */
   struct ft_pt pe_buffer[ PE_POINTS ]; /* Perspective data buffer */
   long pe_inc;                         /* Time increment for perspective */
   long ft_t;
   long last_time;
   char         *orbfile[ SFS_FILESIZE ];       /* orb filename */
   char         *gridfile[ SFS_FILESIZE ];      /* grid filename */
   char         *surfile[ SFS_FILESIZE ];       /* surface feature filename */
#ifdef  USEPOINTS
   char         *pdfile[ SFS_FILESIZE ];        /* point data filename */
#endif
   struct spj_pt o_start;                    /* start orb structure */
   struct spj_pt o_end;                      /* end orb structure */
   struct spj_pt g_start;                    /* start grid structure */
   struct spj_pt g_end;                      /* end grid structure */
   struct spj_pt s_start;                    /* start surface structure */
   struct spj_pt s_end;                      /* end surface structure */
#ifdef  USEPOINTS
   struct spj_pt p_start;                    /* start point structure */
   struct spj_pt p_end;                      /* end point structure */
#endif
   double ssp_lat;              /* latitude return from or_ssp() */
   double ssp_lon;              /* longitude return from or_ssp() */
   long ssp_r;                  /* orbit distance return from or_ssp() */
   long ssp_n;                  /* orbit number return from or_ssp() */
   double nssp_lat;             /* next latitude return from or_ssp() */
   double nssp_lon;             /* next longitude return from or_ssp() */
   long nssp_r;                 /* next orbit distance return from or_ssp() */
   long nssp_n;                 /* next orbit number return from or_ssp() */
   double ssp_rot;              /* rotation for orbit */
   };

/***    Structure for displays */

struct sfs_display
   {
   int type;                    /* type of display: SFSD_xxx */
   int elements;                /* display elements */
   int orbit;                   /* orbit number for visual */
   struct as_focus *focus;      /* focus for perspective or ground track */
   int gt_xsize;                /* Size of ground track map, x */
   int gt_ysize;                /* Size of ground track map, y */
   int gt_mapbase;              /* Bottom of gt map */
   int gt_mapedge;              /* Left edge of groundtrack map */
   };

/***    External declarations for all sfs modules */

extern double vi_rotate();
extern long sfs_inc;
extern long sfs_insertion;
extern long sfs_tfactor;

/***    Colors */

extern  int     cl_mback;               /* Main background color        */
extern  int     cl_mfore;               /* Main foreground color        */
extern  int     cl_marker;              /* Color for prompt markers     */
extern  int     cl_grid;                /* Color for planetary grid     */
extern  int     cl_surface;             /* Color for planetary surface  */
extern  int     cl_orbits[ OR_COLORS ]; /* Colors for different orbits  */


