/* Copyright (C) 1993 Morten Welinder
This file is not part of GNU Make, but part of the MSDOS port of GNU Make.
This file may be copied under the same conditions as GNU Make:

GNU Make is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Make is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Make; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <dir.h>
#include "make.h"
#include "variable.h"
#include "dosstuff.h"
#ifdef __GNUC__
#include <process.h>
#endif

#ifdef __GNUC__
/* Borland's Turbo C has this in libc -- don't know about MS-C.  */

static char *
searchpath (char *cmd)
{
  char result[MAXPATHLEN + 1];

  if (search_path (cmd, getenv ("PATH"), result))
    return result;
  else
    return 0;
}
#endif

static int
searchfor (buf, size, txt)
     char *buf, *txt;
     unsigned size;
{
  unsigned len = strlen (txt);
  char first = *txt;

 again:
  while (*buf != first && size != 0)
    buf++, size--;
  if (size > len)
    if (strcmp ((size--, ++buf), txt + 1))
      goto again;
    else
      return 1;
  else
    return 0;
}

static int
wrapped_aout_or_coff (int f)
{
  unsigned short header[3];
  unsigned long stub_offset;
  unsigned long header_offset;

  header[0] = 0;
  read (f, header, sizeof (header));
  if (header[0] == 0x5a4d)
    {
      header_offset = (long) header[2]*512L;
      if (header[1])
	header_offset += (long) header[1] - 512L;
      lseek (f, header_offset - 4, SEEK_SET);
      read (f, &stub_offset, 4);
      header[0] = 0;
      read (f, header, sizeof (header));
      if (header[0] == 0x010b || header[0] == 0x014c)
	return 1;
    }
  return 0;
}

static int
member (char *var, char *cmd)
{
  struct variable *prgs = lookup_variable (var, strlen (var));
  char *p, *tok;

  if (prgs && prgs->value)
    {
      p = strcpy (alloca (strlen (prgs->value) + 1), prgs->value);
      tok = strtok (p, ":");
      while (tok && strcmp (tok, cmd))
	tok = strtok (NULL, ":");
      return (tok != NULL);
    }
  return 0;
}

/* Determine how command PATH (which is neither an ms-dos internal command
   nor a batch file) can receive a command line of more that 128 chars.

   Returns: 0 (fail), 1 (by environment), or 2 (by @-file).  */

static int
enhanced_parameters (char *path)
{
  char *cmd;
  int f;

  cmd = path + strlen (path);
  while (cmd != path
	 && cmd[-1] != '\\'
	 && cmd[-1] != '/'
	 && cmd[-1] != ':')
    cmd--;

  if (member ("ENVARGS", cmd)) return 1;
  if (member ("ATFILEARGS", cmd)) return 2;

  f = open (path, O_RDONLY | O_BINARY);
  if (f >= 0)
    {
      int found, r;
      unsigned n, siz = 0x7ff0;
      char *buf;

      if (wrapped_aout_or_coff (f))
	found = 1;
      else
	{
	  buf = xmalloc (siz);
	  lseek (f, 0L, SEEK_SET);
	  r = read (f, buf, siz);
	  if (r >= 0)
	    n = r;
	  else
	    n = 0;
	  found = searchfor (buf, n, "_argc");
	  free (buf);
	}
      close (f);
      if (found) return 1;
    }
  return 0;
}


static char *
make_cmdline (path, argv, useenv, useatfile)
     char **argv, *path, **useatfile;
     int *useenv;
{
  int len, batch, first;
  char **vp, *res, *p, *q;
  FILE *fp;
  extern char *getenv();

  *useenv = 0;
  *useatfile = NULL;
  batch = !strcmp (path + (strlen (path) - 4), ".bat");
  if (batch)
    {
      strcpy (path, getenv ("COMSPEC"));
      vp = argv;
      len = 3;
    }
  else
    {
      vp = argv + 1;
      len = 0;
    }
  for (; *vp; vp++)
    len += strlen (*vp) + 1;

  if (len < 127)
    {
      res = p = xmalloc (len + 1);
      if (batch)
	{
	  strcpy (p, " /c");
	  p += 3;
	  vp = argv;
	}
      else
	vp = argv + 1;
      for (; *vp; vp++)
	{
	  *p++ = ' ';
	  strcpy (p, *vp);
	  p += strlen (*vp);
	}
      *p = 0;
      return res;
    }
  else if (!batch)
    /* The command line is too long for MSDOS -- try some tricks.  */
    switch (enhanced_parameters (path))
      {
      case 1:
	if (debug_flag)
	  puts ("Using environment for arguments.");
	res = xmalloc (1);
	*res = '\0';
	*useenv = 1;
	return res;
      case 2:
	p = getenv ("TMP");
	if (!p) p = getenv ("TEMP");
	if (!p) p = "\\";
	*useatfile = p = strcpy (xmalloc (strlen (p) + 20), p);
	while (*p)
	  if (*p++ == '/')
	    p[-1] = '\\';
	if (**useatfile && *useatfile[strlen (*useatfile) - 1] != '\\')
	  strcat (*useatfile, "\\");
	strlwr (*useatfile);
	strcat (*useatfile, "mkXXXXXX");
	mktemp (*useatfile);
	if (debug_flag)
	  printf ("Using temporary file \"%s\" for arguments.\n", *useatfile);
	fp = fopen (*useatfile, "wb");
	if (fp)
	  {
	    for (first = 1, vp = argv + 1; *vp; vp++, first = 0)
	      fprintf (fp, "%s%s", first ? "" : " ", *vp);
	    fclose (fp);
	  }
	else
	  return NULL;

	len = strlen (*useatfile);
	res = xmalloc (len + 2);
	res [0] = '@';
	strcpy (res + 1, *useatfile);
	return res;
      }

  /* The command line is too long for MSDOG and we don't know how to
     bypass MSDOG for the command in question.  In other words, we lose.  */
  return NULL;
}

#ifdef __GNUC__
static char **
make_c_environment (char **env, int useenv, char **argv)
{
  if (useenv)
    {
      int count_e, count_a, i;
      char **p, **q, *r, **res;

      for (p = env, count_e = 0; *p; p++, count_e++);
      for (p = argv, count_a = 0; *p; p++, count_a++);

      q = res = (char **) xmalloc ((count_e + count_a + 2) * sizeof (char **));
      for (p = env; *p; p++)
	*q++ = strdup (*p);
      r = xmalloc (20);
      sprintf (r, "_argc=%d", count_a);
      *q++ = r;
      for (i = 0, p = argv; *p; p++)
	{
	  r = xmalloc (strlen (*p) + 10);
	  sprintf (r, "_argv%d=%s", i++, *p);
	  *q++ = r;
	}
      *q = 0;
      return res;
    }
  else
    return env;
}

#else /* not __GNUC__ */

/* Make the Dos representation of the environment, i.e., a string of the
   form `var1=val1\000var2=\val2\000' followed by an extra `\000'.  If
   useenv is non-zero then all arguments are put into the environment as,
   e.g., _argv4=val4.  NULL is returned on failure, otherwise the returned
   entity is malloc'ed and must be freed later.  */

static char*
make_dos_environment (length, env, argv, useenv)
     char **env, **argv;
     int *length, useenv;
{
  char *e, *res, **ep, **ap;
  unsigned long l;
  int i, argc;

  l = 2;
  for (ep = env; *ep; ep++)
    l += strlen (*ep) + 1;
  if (useenv)
    {
      argc = 0;
      for (ap = argv; *ap; ap++, argc++)
	l += (6 + 1 + 1) + (argc >= 10) + (argc >= 100) + strlen (*ap);
      l += (6 + 1 + 1) + (argc >= 10) + (argc >= 100);
    }

  if (l > 0x7ff0L)
    return NULL;
  e = res = xmalloc (l);
  *length = l;

  for (ep = env; *ep; ep++)
    {
      i = strlen (*ep);
      strcpy (e, *ep);
      e += i + 1;
    }
  if (useenv)
    {
      e += sprintf (e, "_argc=%d", argc) + 1;
      i = 0;
      for (ap = argv; *ap; ap++, i++)
	e += sprintf (e, "_argv%d=%s", i, *ap) + 1;
    }

  *e++ = 0; /* Terminate environment.  */

  return res;
}
#endif /* not __GNUC__ */


int
swap_spawnve (path, argv, env)
     char *path;
     char **argv, **env;
{
  char *cmdline = NULL, *useatfile = NULL;
  void *newenv = NULL;
  char *darg[3];
  int res = -1, envlen, useenv;

  path = dosfilename (alloca (MAXPATH + 1), path);
  cmdline = make_cmdline (path, argv, &useenv, &useatfile);
  if (cmdline == NULL)
    {
      errno = ENOENT;
      goto out;
    }

#ifdef __GNUC__
  newenv = make_c_environment (env, useenv, argv);
#else
  newenv = make_dos_environment (&envlen, env, argv, useenv);
#endif
  if (newenv == NULL)
    {
      errno = EINVAL;
      goto out;
    }

#ifdef __GNUC__
  /* We use here that djgpp's spawn* don't use the environment passing
     method -- if they did we would lose.  */
  darg[0] = path;
  darg[1] = darg[2] = 0;
  res = spawnve (P_WAIT, path,
		 useenv ? darg
		 : (useatfile ? (darg[1] = cmdline, darg)
		    : argv),
		 newenv);
#else
  res = spawn_child (path, cmdline, newenv, envlen);
#endif

 out:
  if (newenv)
#ifdef __GNUC__
    if (useenv)
      {
	char **p = newenv;
	while (*p)
	  free (*p++);
	free (newenv);
      }
#else
    free (newenv);
#endif
  if (cmdline) free (cmdline);
  if (useatfile)
    {
      unlink (useatfile);
      free (useatfile);
    }

  return res;
}

int
swap_spawnvpe (path, argv, env)
     char *path;
     char *argv, **env;
{
  char *cmd;
  int l = strlen (path);
  int dot = 0, inpath = 0;

  cmd = path + l - 1;
  while (cmd != path && dot == 0)
    switch (*cmd--)
      {
      case '.':
        if (!inpath) dot = 1;
        break;
      case '/':
      case '\\':
      case ':':
        cmd = path;
	inpath = 1;
      }

  if (dot)
    cmd = searchpath (path);
  else
    {
      path = strcpy (alloca (l + 5), path);
      strcpy (path + l, ".exe");
      cmd = searchpath (path);
      if (cmd == NULL)
        {
          strcpy (path + l + 1, "com");
          cmd = searchpath (path);
          if (cmd == NULL)
            {
              strcpy (path + l + 1, "bat");
              cmd = searchpath (path);
            }
        }
    }

  if (cmd == NULL)
    return (errno = ENOENT);
  return swap_spawnve (cmd, argv, env);
}
