/* GETDCWD.C - implementation of _getdcwd() for MS-DOS GNU C (DJGPP)
 ******************************************************************************
 *
 *	char *_getdcwd(int drive, char *buffer, int maxlen)
 *	char *_fullpath(char *buffer, const char *pathname, size_t maxlen)
 *
 ******************************************************************************
 * edit history is at the end of the file
 ******************************************************************************
 * Copyright 1995 by the Summer Institute of Linguistics, Inc.
 *
 * This file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; see the file COPYING.LIB.  If
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave,
 * Cambridge, MA 02139, USA.
 */
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <dos.h>

#define _MAX_PATH 260

/*****************************************************************************
 * NAME
 *    _getdcwd
 * ARGUMENTS
 *    drive  - drive number (0 = current, 1 = A:, etc.)
 *    buffer - pointer to buffer for current directory, or NULL
 *    maxlen - length of buffer if not NULL
 * DESCRIPTION
 *    Find the current working directory for the given drive, and store it in
 *    the buffer (or allocate a buffer if the given one is NULL).
 * RETURN VALUE
 *    pointer to the string containing the current drive and directory, or NULL
 *    if out of memory
 */
char *_getdcwd(int drive, char *buffer, int maxlen)
{
int olddrive;
char cwd_string[_MAX_PATH];
size_t length;
/*
 *  get the current directory on the given drive
 */
olddrive = getdisk();
if (drive == 0)
    drive = olddrive;		/* Microsoft 0 => current drive */
else
    --drive;			/* Microsoft starts at 1, DJGPP starts at 0 */
if (drive != olddrive)
    {
    if (setdisk(drive) == 0xFF)
	{
	setdisk(olddrive);
	errno = ENODEV;
	return (char *)NULL;
	}
    }
getwd(cwd_string);
if (drive != olddrive)
    setdisk(olddrive);
/*
 *  store the drive and directory in the buffer
 */
length = strlen(cwd_string) + 1;
if (buffer == (char *)NULL)
    {
    if (length < maxlen)
	length = maxlen;
    buffer = malloc(length);
    if (buffer == (char *)NULL)
	{
	errno = ENOMEM;
	return (char *)NULL;
	}
    }
else if (length > maxlen)
    {
    errno = ERANGE;
    return (char *)NULL;
    }
strcpy(buffer, cwd_string);

return( buffer );
}

/*****************************************************************************
 * NAME
 *    fix_pathname
 * ARGUMENTS
 *    path - a file pathname to fix into standard Unix form
 * DESCRIPTION
 *    convert the pathname to lowercase with forward slashes, and get rid of
 *    as many ./ and ../ components as possible
 * RETURN VALUE
 *    0 if final form is okay, 1 if a ../ cannot be removed
 */
static int fix_pathname(char *path)
{
char *p, *q;

for ( p = path ; *p ; ++p )
    {
    if (isascii(*p) && isupper(*p))
	*p = tolower(*p);
    else if (*p == '\\')
	*p = '/';
    }
while ((p = strstr(path, "/./")) != (char *)NULL)
    memmove(p, p+2, strlen(p+1));
while ((p = strstr(path, "/../")) != (char *)NULL)
    {
    *p = '\0';
    q = strrchr(path, '/');
    if (q == (char *)NULL)
	{
	*p = '/';
	return( 1 );
	}
    memmove(q, p+3, strlen(p+2));
    }
return( 0 );
}

/*****************************************************************************
 * NAME
 *    _fullpath
 * ARGUMENTS
 *    buffer   - pointer to buffer for receiving full pathname, or NULL
 *    pathname - relative pathname of a file
 *    maxlen   - size of buffer (if it's not NULL)
 * DESCRIPTION
 *    convert a relative pathname to a full pathname (including drive letter)
 * RETURN VALUE
 *    pointer to the full pathname for the given relative pathname
 */
char *_fullpath(char *buffer, const char *pathname, size_t maxlen)
{
char *p;
char *dcwd;
char *path;
char *sep;
unsigned count_removed;

path = buffer;
if (buffer == (char *)NULL)
    maxlen = _MAX_PATH;
if (isascii(pathname[0]) && isalpha(pathname[0]) && (pathname[1] == ':'))
    {
    if ((pathname[2] == '/') || (pathname[2] == '\\'))
	{
	if (strlen(pathname) >= maxlen)
	    return( (char *)NULL );
	dcwd = (char *)NULL;
	sep = (char *)NULL;
	}
    else
	{
	dcwd = _getdcwd(pathname[0] & 0x1F, (char *)NULL, 0);
	pathname += 2;		/* skip over the drive designation */
	if (strcmp(dcwd+1, ":/") == 0)
	    sep = "";
	else
	    sep = "/";
	}
    }
else if ((pathname[0] == '/') || (pathname[0] == '\\'))
    {
    dcwd = _getdcwd(0, (char *)NULL, 0);
    dcwd[2] = '\0';
    sep = "";
    }
else
    {
    dcwd = _getdcwd(0, (char *)NULL, 0);
    if (strcmp(dcwd+1, ":/") == 0)
	sep = "";
    else
	sep = "/";
    }
if (path == (char *)NULL)
    {
    path = malloc(_MAX_PATH);
    if (path == (char *)NULL)
	return( (char *)NULL );
    maxlen = _MAX_PATH;
    }
if (dcwd == (char *)NULL)
    {
    strcpy(path, pathname);
    if (fix_pathname(path))
	{
	if (path != buffer)
	    free(path);
	return( (char *)NULL );
	}
    return( path );
    }
else
    {
    do  {
	count_removed = 0;
	while ((strncmp(pathname,"./",2)==0) || (strncmp(pathname,".\\",2)==0))
	    {
	    pathname += 2;
	    ++count_removed;
	    }
	if ((strncmp(pathname,"../",3)==0) || (strncmp(pathname,"..\\",3)==0))
	    {
	    p = strrchr(dcwd, '/');
	    if ((p != (char *)NULL) && (p[1] != '\0'))
		{
		*p = '\0';
		pathname += 3;
		++count_removed;
		}
	    else
		break;
	    }
	} while (count_removed != 0);
    }
if (strlen(dcwd) + strlen(sep) + strlen(pathname) >= maxlen)
    {
    free(dcwd);
    if (path != buffer)
	free(path);
    return( (char *)NULL );
    }
strcat(strcat(strcpy(path, dcwd), sep), pathname);
free(dcwd);
if (fix_pathname(path))
    {
    if (path != buffer)
	free(path);
    return( (char *)NULL );
    }
return( path );
}

/******************************************************************************
 * EDIT HISTORY
 ******************************************************************************
 *  3-Mar-95	SRMc - write _getdcwd() function for DJGPP
 * 15-Mar-95	SRMc - write _fullpath() function for DJGPP
 */
