#ifndef stone_h
#define stone_h

#define stone_unix   1 /* Linux directory reading */
#define stone_djgpp  0 /* findfirst/findnext directory reading */
#define stone_gcc    1 /* GCC extensions (Print formatting) */

#define stone_elf    1 /* Intel ELF support */
#define stone_coff   1 /* DJGPP COFF support */
#define stone_pe     1 /* Windows PE support (Glorified COFF) */
#define stone_omf    1 /* Object Module Format */

#define stone_resize(PTR, NUM) ((PTR) = realloc (PTR, sizeof (*(PTR)) * (NUM)))
#define stone_allot(PTR, NUM)  ((PTR) = malloc (sizeof (*(PTR)) * (NUM)))

typedef struct stone_compiler stone_compiler;
typedef struct stone_object stone_object;
typedef struct stone_object_rel stone_object_rel;
typedef struct stone_object_sym stone_object_sym;
typedef struct stone_search stone_search;

#define csst_import  0
#define csst_export  1
#define csst_unknown 2

#define csr_unknown   0
#define csr_relative  1         /* Subtract the destination by the source */
#define csr_absolute  2         /* Just put the destination in */
#define csr_relative2 3         /* For ELF, don't ask :) */
#define csr_absolute2 4         /* Ditto */

struct stone_object_sym
{
  char *name;                   /* Symbol identifier */
  int obj;                      /* Pointer for where it resolved to */
  int sym;                      /* Symbol at that location */
  int type;                     /* csst */
  int off;
};

struct stone_object_rel
{
  int off, sym;                 /* Offset and import symbol entry */
  int type;                     /* Signal type */
  int base;                     /* Base offset (Taken from source) */
};

struct stone_object
{
  char *src;                    /* Source file (For debugging) */
  char *code;                   /* Imported code */
  int len;                      /* Code length */

  int link;                     /* Set if linked */
  int user;                     /* User count */
    
  int nsym;                     /* Number of symbols */
  stone_object_sym *sym;        /* List of symbols */

  int nrel;                     /* Number of relocations */
  stone_object_rel *rel;        /* List of resolutions */
};

/* Compiler specification */
struct stone_compiler
{
  stone_compiler *next;
  char *id; /* Compiler identifier */
  char *name; /* Compiler name */
  int nstep; /* Number of compilation steps */
  char **step; /* Compile steps */
  char *srcext; /* Source file extension */
  char *objext; /* Object file extension */
};

/* Search path entry */
struct stone_search
{
  stone_search *next; /* Later in the search list */
  char *path; /* Path text */
};

extern int stone_nobj; /* Number of objects loaded */
extern stone_object *stone_obj; /* List of objects loaded */
extern stone_compiler *stone_compiler_list; /* Linked list of compilers */
extern stone_search *stone_search_list; /* Linked list of search paths */
extern int stone_is_setup; /* Whether stone_setup has been called */
extern void (*stone_error_cur) (char *str); /* Called by stone_report to print out the error */
extern int stone_nerror; /* Number of stone_report calls */

/* Although symbol names have _ prefixes, you don't need to.  If you leave   */
/* it off, it will match with _ prefixed symbols and not.                    */

int stone_export (char *name, void *ptr); /* Exports symbol, returns success */

extern int (*stone_sym_proc) (char *id, int *sym); /* Called by stone_sym_find */
int stone_sym_default (char *id, int *symptr); /* Default searching mechanism */

void *stone_sym (char *id); /* Find a symbol in the system and return it's pointer */
int /*[obj]*/ stone_sym_find (char *id, int *sym); /* Search for symbol */
void *stone_sym_ptr (int obj, char *id); /* Get the symbol pointer or NULL */
int stone_sym_match (char *a, char *b); /* Return whether the symbols match */
void *stone_ptr (char *id); /* Get symbol pointer throughout system */
int stone_list_unlink (void); /* List unlinked symbols, sorted */

int stone_inside (void *ptr, int *obj, int *sym); /* Search the objects/symbols for the closest symbol to ptr */

int stone_free (int obj); /* Try to free an object, return success */
int stone_link (int obj); /* Make sure all links are made and return success */
int stone_load (char *file); /* Load an object and return index or < 0 */
int stone_link_all (void); /* Links all objects, returns success */
int stone_link_range (int low, int high); /* Links from low to high objects, returns success */

/* These two commands ignore anything with "." as the first letter in the filename */
/* They also link the files they load. */
int stone_load_dir (char *dir); /* Compile all C files and load in .o files and directories */
int stone_load_subdir (char *base); /* Call obj_load_dir with the subdirectories of this directory */

void stone_add_compiler (char *id, char *name, char *srcext, char *objext, /* char *step [] */.../* NULL */); /* Add compiler to the list. */
void stone_add_search (char *path); /* Add search path to front of list */

void stone_clear_obj (stone_object *); /* Clear the object data */
void stone_free_obj (stone_object *); /* Free the object data */

#if stone_gcc
void stone_report (char *str, ...) __attribute__ ((format (printf, 1, 2)));
#else
void stone_report (char *str, ...);
#endif /* !stone_gcc */
void stone_setup (void); /* Setup stonewheel, automatically called */

/* Don't call these directly -- use stone_load */
#if stone_coff || stone_pe
int stone_load_coff (FILE *file, char *filename);
#endif /* stone_coff || stone_pe */

#if stone_elf
int stone_load_elf (FILE *file, char *filename);
#endif /* stone_elf */

#if stone_omf
int stone_load_omf (FILE *file, char *filename);
#endif /* stone_omf */

#endif /* stone_h */
