/*
 * Copyright (c) 1996, 1997, 1998, 1999
 *             Shigio Yamaguchi. All rights reserved.
 * Copyright (c) 1999, 2000
 *             Tama Communications Corporation. All rights reserved.
 *
 * This file is part of GNU GLOBAL.
 *
 * GNU GLOBAL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * GNU GLOBAL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <assert.h>
#include <ctype.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "gparam.h"
#include "conf.h"
#include "die.h"
#include "locatestring.h"
#include "makepath.h"
#include "strbuf.h"
#include "strmake.h"
#include "test.h"

static FILE	*fp;
static STRBUF	*ib;
static char	*line;
static int	allowed_nest_level = 8;
static int	opened;

static void	trim(char *);
static char	*readrecord(const char *);
static void	includelabel(STRBUF *, const char *, int);

#ifndef isblank
#define isblank(c)	((c) == ' ' || (c) == '\t')
#endif

static void
trim(l)
char	*l;
{
	char	*f, *b;
	int	colon = 0;

	for (f = b = l; *f; f++) {
		if (colon && isblank(*f))
			continue;
		colon = 0;
		if ((*b++ = *f) == ':')
			colon = 1;
	}
	*b = 0;
}
static char	*
readrecord(label)
const char *label;
{
	char	*p, *q;
	int	flag = STRBUF_NOCRLF;

	rewind(fp);
	while ((p = strbuf_fgets(ib, fp, flag)) != NULL) {
		flag &= ~STRBUF_APPEND;
		if (*p == '#')
			continue;
		if (strbuf_unputc(ib, '\\')) {
			flag |= STRBUF_APPEND;
			continue;
		}
		trim(p);
		for (;;) {
			if ((q = strmake(p, "|:")) == NULL)
				die("illegal configuration file format (%s).", p);
			if (!strcmp(label, q)) {
				if (!(p = locatestring(p, ":", MATCH_FIRST)))
					die("illegal configuration file format.");
				p = strdup(p);
				if (!p)
					die("short of memory.");
				return p;
			}
			p += strlen(q);
			if (*p == ':')
				break;
			else if (*p == '|')
				p++;
			else
				assert(0);
		}
	}
	return NULL;
}
static	void
includelabel(sb, label, level)
STRBUF	*sb;
const char *label;
int	level;
{
	char	*savep, *p, *q;

	if (++level > allowed_nest_level)
		die("nested include= (or tc=) over flow.");
	if (!(savep = p = readrecord(label)))
		die("label '%s' not found.", label);
	while ((q = locatestring(p, ":include=", MATCH_FIRST)) || (q = locatestring(p, ":tc=", MATCH_FIRST))) {
		char	inclabel[MAXPROPLEN+1], *c = inclabel;

		strbuf_nputs(sb, p, q - p);
		q = locatestring(q, "=", MATCH_FIRST) + 1;
		while (*q && *q != ':')
			*c++ = *q++;
		*c = 0;
		includelabel(sb, inclabel, level);
		p = q;
	}
	strbuf_puts(sb, p);
	free(savep);
}
/*
 * configpath: get path of configuration file.
 */
char *
configpath() {
	static char config[MAXPATHLEN+1];
	char *p;

	if ((p = getenv("GTAGSCONF")) != NULL) {
		if (!test("r", p))
			config[0] = 0;
		else
			strcpy(config, p);
	} else if ((p = getenv("HOME")) && test("r", makepath(p, GTAGSRC, NULL)))
		strcpy(config, makepath(p, GTAGSRC, NULL));
	else if (test("r", GTAGSCONF))
		strcpy(config, GTAGSCONF);
	else if (test("r", DEBIANCONF))
		strcpy(config, DEBIANCONF);
	else
		config[0] = 0;
	return config;
}
/*
 * openconf: load configuration file.
 *
 *	go)	line	specified entry
 */
void
openconf()
{
	const char *label, *config;
	STRBUF	*sb;

	assert(opened == 0);

	config = configpath();
	/*
	 * if configuration file is not found, default values are set
	 * for upper compatibility.
	 */
	if (*config == 0) {
		sb = strbuf_open(0);
		strbuf_putc(sb, ':');
		strbuf_puts(sb, "suffixes=");
		strbuf_puts(sb, DEFAULTSUFFIXES);
		strbuf_putc(sb, ':');
		strbuf_puts(sb, "skip=");
		strbuf_puts(sb, DEFAULTSKIP);
		strbuf_putc(sb, ':');
		strbuf_puts(sb, "format=standard:");
		strbuf_puts(sb, "extractmethod:");
		strbuf_puts(sb, "GTAGS=gctags %s:");
		strbuf_puts(sb, "GRTAGS=gctags -r %s:");
		strbuf_puts(sb, "GSYMS=gctags -s %s:");
		strbuf_puts(sb, "sort_command=sort:");
		strbuf_puts(sb, "sed_command=sed:");
		line = strdup(strbuf_value(sb));
		if (!line)
			die("short of memory.");
		strbuf_close(sb);
		opened = 1;
		return;
	}
	if ((label = getenv("GTAGSLABEL")) == NULL)
		label = "default";
	if (!(fp = fopen(config, "r")))
		die("cannot open '%s'.", config);
	ib = strbuf_open(MAXBUFLEN);
	sb = strbuf_open(0);
	includelabel(sb, label, 0);
	line = strdup(strbuf_value(sb));
	strbuf_close(ib);
	strbuf_close(sb);
	fclose(fp);
	opened = 1;
	return;
}
/*
 * getconfn: get property number
 *
 *	i)	name	property name
 *	o)	num	value (if not NULL)
 *	r)		1: found, 0: not found
 */
int
getconfn(name, num)
const char *name;
int	*num;
{
	char	*p;
	char	buf[MAXPROPLEN+1];

	if (!opened)
		openconf();
	snprintf(buf, sizeof(buf), ":%s#", name);
	if ((p = locatestring(line, buf, MATCH_FIRST)) != NULL) {
		p += strlen(buf);
		if (num != NULL)
			*num = atoi(p);
		return 1;
	}
	return 0;
}
/*
 * getconfs: get property string
 *
 *	i)	name	property name
 *	o)	sb	string buffer (if not NULL)
 *	r)		1: found, 0: not found
 */
int
getconfs(name, sb)
const char *name;
STRBUF	*sb;
{
	char	*p;
	char	buf[MAXPROPLEN+1];
	int	all = 0;
	int	exist = 0;

	if (!opened)
		openconf();
	if (!strcmp(name, "suffixes") || !strcmp(name, "skip"))
		all = 1;
	snprintf(buf, sizeof(buf), ":%s=", name);
	p = line;
	while ((p = locatestring(p, buf, MATCH_FIRST)) != NULL) {
		if (exist && sb)
			strbuf_putc(sb, ',');		
		exist = 1;
		for (p += strlen(buf); *p && *p != ':'; p++) {
			if (*p == '\\')	/* quoted charactor */
				p++;
			if (sb)
				strbuf_putc(sb, *p);
		}
		if (!all)
			break;
	}
	/*
	 * It may be that these code should be moved to applications.
	 * But nothing cannot start without them.
	 */
	if (!exist) {
		exist = 1;
		if (!strcmp(name, "suffixes")) {
			if (sb)
				strbuf_puts(sb, DEFAULTSUFFIXES);
		} else if (!strcmp(name, "skip")) {
			if (sb)
				strbuf_puts(sb, DEFAULTSKIP);
		} else if (!strcmp(name, "sort_command")) {
			if (sb)
				strbuf_puts(sb, "sort");
		} else if (!strcmp(name, "sed_command")) {
			if (sb)
				strbuf_puts(sb, "sed");
		} else
			exist = 0;
	}
	return exist;
}
/*
 * getconfb: get property bool value
 *
 *	i)	name	property name
 *	r)		1: TRUE, 0: FALSE
 */
int
getconfb(name)
const char *name;
{
	char	buf[MAXPROPLEN+1];

	if (!opened)
		openconf();
	snprintf(buf, sizeof(buf), ":%s:", name);
	if (locatestring(line, buf, MATCH_FIRST) != NULL)
		return 1;
	return 0;
}
/*
 * getconfline: print loaded config entry.
 */
char *
getconfline()
{
	if (!opened)
		openconf();
	return line;
}
void
closeconf()
{
	if (!opened)
		return;
	free(line);
	line = NULL;
	opened = 0;
}
