/* (c) 1993-1998 by Michael Haardt. */
/* #includes */ /*{{{C}}}*//*{{{*/
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2

#include "config.h"

#ifdef HAVE_NL_TYPES_H
#include <nl_types.h>
#else
#define nl_catd int
#define catopen(name, oflag) 0
#define catgets(catd, set_id, msg_id, msg) msg
#endif

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "getopt.h"
#include "misc.h"
/*}}}*/
/* #defines */ /*{{{*/
#define LATIN1

#define USAGE1    catgets(catd,1,1,"Usage: deroff [-w] [-s] [-ml] [-ms] [-mm] [-C] [file ...]\n")
#define USAGE2    catgets(catd,1,2,"       deroff [--word-list] [--skip-headers] [--skip-lists] [--cpp-line] [file ...]\n")
#define USAGE3    catgets(catd,1,3,"       deroff --version\n")
#define USAGE4    catgets(catd,1,4,"Try deroff -h or deroff --help for more information.\n")
#define USAGE5    catgets(catd,1,5,"Remove roff, tbl, eqn, refer and pic constructs from documents.\n")
#define USAGE6    catgets(catd,1,6,"-w, --word-list        output a word list\n")
#define USAGE7    catgets(catd,1,7,"-s, --skip-headers     skip headers\n")
#define USAGE8    catgets(catd,1,8,"-ml, --skip-lists      suppress lists\n")
#define USAGE9    catgets(catd,1,9,"-ms, -mm               ignored for compatibility\n")
#define USAGE10   catgets(catd,1,10,"-C, --cpp-line         output cpp(1)-style #line directives\n")
#define CANTOPEN  catgets(catd,1,11,"deroff: Can't open %s: %s\n")
#define CANTCLOSE catgets(catd,1,12,"deroff: Can't close standard output: %s\n")
#define NOMEM     catgets(catd,1,13,"deroff: %s\n")
#define SOREPEATS catgets(catd,1,14,"deroff: ignoring repeated request to process %s\n")

#define HASHLINE  if (cpp) if (file) printf("#line %d \"%s\"\n",lineShould,file); else printf("#line %d\n",lineShould);
/*}}}*/

/* variables */ /*{{{*/
static int specletter;
static int pic=0,tbl=0,eqn=0,refer=0,refer2=0,macro=0,inlist=0,inheader=0;
static int words=0,skipheaders=0,skiplists=0,cpp=0;
static char linebuf[512],*s;
static struct tm now;
static nl_catd catd;
static char **source;
static const char *file;
static int lineShould,lineIs;
static int sourceSize;
static int sourceCapacity;
/*}}}*/
/* forwards */ /*{{{*/
static int exch(void);
static int word(void);
static int numb(void);
static int nomacline(void);
static void deroff(FILE *fp, const char *fileName);
/*}}}*/

/* condputchar */ /*{{{*/
static int condputchar(int c)
{
  static int nls=2;
  static int lastc='\n';

  if (!pic && !tbl && !eqn && !refer && !refer2 && !macro && (!skiplists || !inlist) && (!skipheaders || !inheader))
  {
    if (c=='\n') 
    {
      if (++nls>2)
      {
        return c;
      }
    }
    else
    {
      nls=0;
    }
    if (lastc=='\n' && (c!='\n' || lineIs+1!=lineShould))
    {
      if (lineIs!=lineShould) HASHLINE
      lineIs=lineShould;
    }
    if (c=='\n') ++lineIs;
    lastc=c;
    return putchar(c);
  }
  else
  {
    return c;
  }
}
/*}}}*/

/* prch */ /*{{{*/
static int prch(char ch)
{
  return (ch && ch!=' ' && ch!='\t' && ch!='\n');
}
/*}}}*/
/* white */ /*{{{*/
static int white(char ch)
{
  return (ch==' ' || ch=='\t' || ch=='\n');
}
/*}}}*/
/* digit */ /*{{{*/
static int digit(int ch)
{
  return (ch>='0' && ch<='9');
}
/*}}}*/
/* letter */ /*{{{*/
/* Notes */ /*{{{*/
/*

The underscore is of course not really a letter, but commonly used in C
identifiers, which should not be splitted into parts.  The latin1 stuff
is not finished.

*/
/*}}}*/
static int letter(char ch)
{
  return (isalpha(ch) || (ch=='_')
#ifdef LATIN1
  || ch==(char)0xe4 || ch==(char)0xc4 || ch==(char)0xf6 || ch==(char)0xd6 || ch==(char)0xfc || ch==(char)0xdc || ch==(char)0xdf
#endif
  );
}
/*}}}*/

/* comment */ /*{{{*/
/*

comment ::= '\\' '"' character ...

*/
static int comment(void)
{
  if (*s=='\\' && *(s+1)=='"')
  {
    while (*s && *s!='\n') ++s;
    return 1;
  }
  return 0;
}
/*}}}*/
/* font */ /*{{{*/
/*

font ::= '\\' 'f' '(' prch prch
font ::= '\\' 'f' prch
font ::= '\\' 'f' '[' prch ... ']'

*/
static int font(void)
{
  if (*s=='\\' && *(s+1)=='f')
  {
    if (*(s+2)=='(' && prch(*(s+3)) && prch(*(s+4))) { s+=5; return 1; }
    else if (*(s+2)=='[')
    {
      s+=2; while (prch(*s) && *s!=']') ++s;
      if (*s==']') ++s;
    }
    else if (prch(*(s+2))) { s+=3; return 1; }
  }
  return 0;
}
/*}}}*/
/* numreq */ /*{{{*/
/*

numreq ::= '\\' [ 'h' | 'v' | 'w' ]  '\'' exch '\''

*/
static int numreq(void)
{
  if (*s=='\\' && (*(s+1)=='h' || *(s+1)=='v' || *(s+1)=='w') && *(s+2)=='\'')
  {
    ++macro;
    s+=3;
    while (*s!='\'' && exch());
    if (*s=='\'') ++s;
    --macro;
    return 1;
  }
  return 0;
}
/*}}}*/
/* size */ /*{{{*/
/*

size ::= '\\' 's' digit { digit }
size ::= '\\' 's' [ '+' | '-' ] digit { digit }

*/
static int size(void)
{
  if (*s=='\\' && *(s+1)=='s' && (digit(*(s+2)) || ((*(s+2)=='-' || *(s+2)=='+') && digit(*(s+3)))))
  {
    s+=3; while (digit(*s)) ++s;
    return 1;
  }
  return 0;
}
/*}}}*/
/* var */ /*{{{*/
/*

var ::= '\\' ( '*' | 'n' ) '(' prch prch
var ::= '\\' ( '*' | 'n' ) prch

*/
static int var(void)
{
  if (*s=='\\' && (*(s+1)=='*' || *(s+1)=='n'))
  {
    if (*(s+1)=='n' && *(s+3)=='d' && *(s+4)=='y')
    {
      if (now.tm_mday/10) condputchar(now.tm_mday/10+'0');
      condputchar(now.tm_mday%10+'0');
      s+=5;
      return 1;
    }
    else if (*(s+2)=='(') { if (prch(*(s+3)) && prch(*(s+4))) s+=5; return 1; }
    else if (prch(*(s+2))) { s+=3; return 1; }
  }
  return 0;
}
/*}}}*/
/* spec */ /*{{{*/
static int spec(void)
{
  specletter=0;
  if (*s=='\\' && *(s+1)=='(' && prch(*(s+2)) && prch(*(s+3)))
  {
    switch (*(s+2)|((*(s+3))<<8))
    {
      case 'e'|('n'<<8): if (!words) condputchar('-'); break;
      case 'e'|('m'<<8): if (!words) { condputchar('-'); condputchar('-'); } break;
      case 'l'|('q'<<8): if (!words) { condputchar('`'); condputchar('`'); } break;
      case 'r'|('q'<<8): if (!words) { condputchar('\''); condputchar('\''); } break;
      case 'B'|('q'<<8): if (!words) { condputchar(','); condputchar(','); } break;
#ifdef LATIN1
      case ':'|('a'<<8): condputchar((char)0xe4); specletter=1; break;
      case ':'|('A'<<8): condputchar((char)0xc4); specletter=1; break;
      case ':'|('o'<<8): condputchar((char)0xf6); specletter=1; break;
      case ':'|('O'<<8): condputchar((char)0xd6); specletter=1; break;
      case ':'|('u'<<8): condputchar((char)0xfc); specletter=1; break;
      case ':'|('U'<<8): condputchar((char)0xdc); specletter=1; break;
      case 's'|('s'<<8): condputchar((char)0xdf); specletter=1; break;
#endif
    }
    s+=4; return 1;
  }
  else if (*s=='\\' && *(s+1)=='%')
  {
    specletter=1;
    s+=2;
    return 1;
  }
  else return 0;
}
/*}}}*/
/* esc */ /*{{{*/
static int esc(void)
{
  if (*s=='\\' && *(s+1))
  {
    switch (*(s+1))
    {
      case '`':
      case '\'':
      case ' ':
      case '-': if (!words) condputchar(*(s+1)); break;
      case 'e': if (!words) condputchar('\\'); break;
    }
    s+=2; return 1;
  }
  else return 0;
}
/*}}}*/

/* to do */ /*{{{*/
/*
ereq ::= '\\' 'z' '\'' exch { exch } '\''
ereq ::= '\\' 'z' prch
*/
/*}}}*/

/* exch */ /*{{{*/
static int exch(void)
{
  return (comment() || font() || size() || numreq() || var() || spec() || esc() || word() || numb());
}
/*}}}*/
/* targ -- non-quoted text argument */ /*{{{*/
static int targ(void)
{
  if (!exch()) if (*s && !white(*s)) { if (!words) condputchar(*s); ++s; } else return 0;
  while (1)
  {
    if (!exch()) if (*s && !white(*s)) 
    {
      if (!words) condputchar(*s);
      ++s;
    }
    else return 1;
  }
}
/*}}}*/
/* qarg -- quoted argument */ /*{{{*/
static int qarg(void)
{
  if (*s=='"')
  {
    ++s;
    while (*s && *s!='"')
    {
      if (!exch()) if (*s)
      {
        if (!words) condputchar(*s);
        ++s;
      }
    }
    return 1;
  }
  else return 0;
}
/*}}}*/
/* macr -- macro */ /*{{{*/
static int macr(void)
{
  int nobody;

  if (*s=='.' || *s=='\'')
  {
    ++s;
    switch (*s)
    {
      case '\\': if (*(s+1)=='"') return 1; else break;
      case '[': refer=1; return 1;
      case ']':
      {
        refer=0;
        ++s;
        return nomacline();
      }
      case '.': macro=0; return 1;
    }
    nobody=0;
    switch (*s|(*(s+1)<<8))
    {
      case 'S'|('H'<<8):
      {
        if (strncmp(s+2," SYNOPSIS",8)==0) { if (!words) condputchar('\n'); inheader=1; }
        else if (strncmp(s+2," \"SYNOPSIS",9)==0) { if (!words) condputchar('\n'); inheader=1; }
        else if (strncmp(s+2," BERSICHT",10)==0) { if (!words) condputchar('\n'); inheader=1; }
        else if (strncmp(s+2," \"BERSICHT",11)==0) { if (!words) condputchar('\n'); inheader=1; }
        else { inheader=0; if (!words) condputchar('\n'); }
        nobody=1;
        break;
      }
      case 'S'|('S'<<8):
      case 'I'|('P'<<8):
      case 'H'|(' '<<8): if (!words) condputchar('\n'); nobody=1; break;
      case 'I'|(' '<<8):
      case 'I'|('R'<<8):
      case 'I'|('B'<<8):
      case 'B'|(' '<<8):
      case 'B'|('R'<<8):
      case 'B'|('I'<<8):
      case 'R'|(' '<<8):
      case 'R'|('B'<<8):
      case 'R'|('I'<<8):
      case 'A'|('B'<<8): break;
      case ']'|(' '<<8): refer=0; break;
      case 'P'|('S'<<8): if (white(*(s+2))) pic=1; return 1;
      case 'P'|('E'<<8): if (white(*(s+2))) pic=0; return 1;
      case 'T'|('S'<<8): if (white(*(s+2))) tbl=1; return 1;
      case 'T'|('E'<<8): if (white(*(s+2))) tbl=0; return 1;
      case 'E'|('Q'<<8): if (white(*(s+2))) eqn=1; return 1;
      case 'E'|('N'<<8): if (white(*(s+2))) eqn=0; return 1;
      case 'R'|('1'<<8): if (white(*(s+2))) refer2=1; return 1;
      case 'R'|('2'<<8): if (white(*(s+2))) refer2=0; return 1;
      case 'd'|('e'<<8): macro=1; return 1;
      case 'B'|('L'<<8):
      case 'V'|('L'<<8):
      case 'A'|('L'<<8):
      case 'L'|('B'<<8):
      case 'R'|('L'<<8):
      case 'M'|('L'<<8):
      case 'D'|('L'<<8): if (white(*(s+2))) inlist=1; return 1;
      case 'B'|('V'<<8): if (*(s+2)=='L' && white(*(s+3))) inlist=1; return 1;
      case 'L'|('E'<<8): if (white(*(s+2))) inlist=0; return 1;
      case 'P'|('\n'<<8):
      {
        if (!words) { condputchar('\n'); condputchar('\n'); }
        return 1;
      }
      case 's'|('o'<<8):
      {
        char *t;
        FILE *fp;
        int oldLineShould;

        s+=2;
        while (white(*s)) ++s;
        t=s;
        while (*t && *t!='\n') ++t;
        *t='\0';
        if ((fp=fopen(s,"r"))==(FILE*)0)
        {
          fprintf(stderr,CANTOPEN,s,strerror(errno));
          exit(1);
        }
        oldLineShould=lineShould;
        deroff(fp,s);
        lineIs=lineShould=oldLineShould;
        HASHLINE
        fclose(fp);
        return 1;
      }
      case 's'|('p'<<8): condputchar('\n'); condputchar('\n'); return 1;
      default: return 1;
    }
    if (skipheaders && nobody) return 1;
    while (white(*s)) ++s;
    while (*s && !white(*s)) ++s;
    while (white(*s)) ++s;
    while (1)
    {
      if (!qarg() && !targ())
      {
        if (*s)
        {
          if (!words) condputchar(*s);
          ++s;
        }
        else return 1;
      }
    }
  }
  else return 0;
}
/*}}}*/
/* numb */ /*{{{*/
static int numb(void)
{
  if (((*s=='-' || *s=='+') && digit(*(s+1))) || digit(*s))
  {
    if (!words) condputchar(*s);
    ++s;
    while (digit(*s)) { if (!words) condputchar (*s); ++s; }
    return 1;
  }
  return 0;
}
/*}}}*/
/* word */ /*{{{*/
static int word(void)
{
  if (letter(*s))
  {
    condputchar(*s);
    ++s;
    while (1)
    {
      if (spec() && !specletter) break;
      else if (letter(*s)) { condputchar(*s); ++s; } else break;
    }
    if (words) condputchar('\n');
    return 1;
  }
  return 0;
}
/*}}}*/
/* text */ /*{{{*/
static int text(void)
{
  while (1)
  {
    if (!exch()) if (*s) { if (!words) condputchar(*s); ++s; } else break;
  }
  return 1;
}
/*}}}*/
/* nomacline */ /*{{{*/
static int nomacline(void)
{
  return text();
  if (!pic && !tbl && !eqn && !refer && !refer2 && !macro && (!skiplists || !inlist) && (!skipheaders || !inheader)) return text(); else return 1;
}
/*}}}*/
/* doLine */ /*{{{*/
static int doLine(void)
{
  return (macr() || nomacline());
}
/*}}}*/
/* deroff */ /*{{{*/
static void deroff(FILE *fp, const char *fileName)
{
  lineShould=1;
  lineIs=1;
  if (fileName)
  {
    int i;

    for (i=0; i<sourceSize; ++i) if (strcmp(source[i],fileName)==0)
    {
      fprintf(stderr,SOREPEATS,fileName);
      return;
    }
    if (sourceSize==sourceCapacity && (source=realloc(source,sizeof(char*)*(sourceCapacity*=2)))==(char**)0)
    {
      fprintf(stderr,NOMEM,strerror(errno));
      exit(2);
    }
    if ((source[sourceSize]=malloc(strlen(fileName)+1))==(char*)0)
    {
      fprintf(stderr,NOMEM,strerror(errno));
      exit(2);
    }
    strcpy(source[sourceSize],fileName);
    file=source[sourceSize];
    ++sourceSize;
    HASHLINE
  }
  while (fgets(linebuf,sizeof(linebuf),fp))
  {
    s=linebuf;
    doLine();
    ++lineShould;
  }
}
/*}}}*/

/* main */ /*{{{*/
int main(int argc, char *argv[])
{
  /* variables */ /*{{{*/
  FILE *in;
  int usage=0;
  int c;
  time_t nowt;
  static struct option lopts[]=
  {
    { "word-list", no_argument, 0, 'w' },
    { "skip-headers", no_argument, 0, 's' },
    { "skip-lists", no_argument, 0, 'L' },
    { "cpp-line", no_argument, 0, 'C' },
    { "help", no_argument, 0, 'h' },
    { "version", no_argument, 0, 'v' },
    { (const char*)0, 0, 0, '\0' }
  };
  /*}}}*/

  setlocale(LC_ALL,"");
  catd=catopen("deroff",0);
  nowt=time((time_t*)0);
  now=*localtime(&nowt);
  if ((source=malloc(sizeof(char*)*(sourceCapacity=64)))==0)
  {
    fprintf(stderr,NOMEM,strerror(errno));
    exit(2);
  }
  /* parse arguments */ /*{{{*/
  while ((c=getopt_long(argc,argv,"wsm:C?h",lopts,(int*)0))!=EOF) switch(c)
  {
    case 'w': words=1; break;
    case 's': skipheaders=1; break;
    case 'L': skiplists=1; break;
    case 'm':
    {
      if (strcmp(optarg,"l")==0) skiplists=1;
      else if (strcmp(optarg,"s") && strcmp(optarg,"m")) usage=1;
      break;
    }
    case 'C': cpp=1; break;
    case 'h': usage=2; break;
    case 'v': printf("deroff " VERSION "\n"); exit(0);
    default: usage=1;
  }
  if (usage==1)
  {
    fprintf(stderr,USAGE1);
    fprintf(stderr,USAGE2);
    fprintf(stderr,USAGE3);
    fprintf(stderr,"\n");
    fprintf(stderr,USAGE4);
    exit(1);
  }
  if (usage==2)
  {
    fprintf(stderr,USAGE1);
    fprintf(stderr,USAGE2);
    fprintf(stderr,USAGE3);
    fprintf(stderr,"\n");
    fprintf(stderr,USAGE5);
    fprintf(stderr,"\n");
    fprintf(stderr,USAGE6);
    fprintf(stderr,USAGE7);
    fprintf(stderr,USAGE8);
    fprintf(stderr,USAGE9);
    fprintf(stderr,USAGE10);
    exit(0);
  }
  /*}}}*/
  /* deroff stdin or files, if any */ /*{{{*/
  if (optind<argc) while (optind<argc)
  {
    int i;

    if ((in=fopen(argv[optind],"r"))==(FILE*)0)
    {
      fprintf(stderr,CANTOPEN,argv[optind],strerror(errno));
      exit(1);
    }
    for (i=0; i<sourceSize; ++i) free(source[i]);
    sourceSize=0;
    deroff(in,argv[optind]);
    fclose(in);
    ++optind;
  }
  else
  {
    deroff(stdin,(const char*)0);
  }
  if (fclose(stdout)==-1)
  {
    fprintf(stderr,CANTCLOSE,strerror(errno));
    return 1;
  }
  /*}}}*/
  return 0;
}
/*}}}*/
