/* $Id: simple.c,v 1.9 2003/03/02 18:43:56 richdawe Exp $ */

/*
 * simple.c - Test program for simple tests of pakke's dependency functions.
 * Copyright (C) 2000-2003 by Richard Dawe
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <libpakke/package.h>
#include <libpakke/packlist.h>
#include <libpakke/packdep.h>

/* --------
 * - main -
 * -------- */

int
main (int argc, char *argv[])
{
  /* Fake package list */
  PACKAGE_INFO  packages[2];

  /* Package with dependency */
  PACKAGE_INFO *package     = &packages[0];

  /* Target package, satisfies or breaks dep. */
  PACKAGE_INFO *dep_package = &packages[1];
  PACKAGE_DEP   dep;

  /* For failures */
  PACKAGE_DEP *failed_dep[3] = { NULL, NULL, NULL };
  const int failed_dep_max   = 2;

  const char PACKAGE_NAME[]     = "foo";
  const char DEP_PACKAGE_NAME[] = "bar";

  const int testcases[] = {
    PACKAGE_DEP_REQUIRES,
    PACKAGE_DEP_DEPENDS_ON,
    PACKAGE_DEP_CONFLICTS_WITH,
    PACKAGE_DEP_REPLACES,
    PACKAGE_DEP_PROVIDES,
    PACKAGE_DEP_INSTALL_BEFORE,
    PACKAGE_DEP_INSTALL_AFTER,
    PACKAGE_DEP_NONE /* terminator */
  };

  int verbose = 0;
  int i;
  int ret;

  /* Verbose operation, for manual testing. */
  if (   (getenv("VERBOSE") != NULL)
      && (strcmp(getenv("VERBOSE"), "n") != 0))
    verbose = 1;

  /* Parse arguments. */
  for (i = 1; i < argc; i++) {
    /* Verbose operation, for manual testing. */
    if (   (strcmp(argv[i], "--verbose") == 0)
	|| (strcmp(argv[i], "-v") == 0))
      verbose = 1;
  }

  /* Set up package list */
  memset(&packages, 0, sizeof(packages));
  memset(&dep, 0, sizeof(dep));

  package->q_back     = dep_package->q_forw = NULL;
  package->q_forw     = dep_package;
  dep_package->q_back = package;

  package->deps[0] = &dep;
  package->deps[1] = NULL;

  /* Set up package with dep. */
  strcpy(package->name, PACKAGE_NAME);
  strcpy(dep.name, DEP_PACKAGE_NAME);

  /* Set up target package. */
  strcpy(dep_package->name, DEP_PACKAGE_NAME);

  /* Run testcases */
  for (i = 0; testcases[i] != PACKAGE_DEP_NONE; i++) {
    /* Initialise dep for this testcase */
    dep.dep      = NULL;
    dep.dep_type = testcases[i];

    /* Generate cross-links */
    packlist_xref(package);

    if ((testcases[i] != PACKAGE_DEP_PROVIDES) && (dep.dep != dep_package)) {
      /* Cross-linking failed */
      if (verbose)
	printf("Cross-linking for test case %i failed: %s: %s\n", i + 1,
	       package_dep_type_string(dep.dep_type),
	       package_dep_string(&dep));
      return(EXIT_FAILURE);
    }

    /* Is the dependency satisfied? */
    ret = package_check_deps(package, dep.dep_type,
			     failed_dep, failed_dep_max);

    if (!ret) {
      /* Did we expect a failure? */
      switch(dep.dep_type) {
      case PACKAGE_DEP_CONFLICTS_WITH:
      case PACKAGE_DEP_REPLACES:
      case PACKAGE_DEP_INSTALL_BEFORE:
	/* Yep */
	if (verbose)
	  printf("Dependencies for test case %i failed, as expected: "
		 "%s: %s\n", i + 1,
		 package_dep_type_string(dep.dep_type),
		 package_dep_string(&dep));
	break;

      case PACKAGE_DEP_REQUIRES:
      case PACKAGE_DEP_DEPENDS_ON:
      case PACKAGE_DEP_INSTALL_AFTER: /* no-op dependency */
      case PACKAGE_DEP_PROVIDES: /* no-op dependency */
      default:
	/* Nope */
	if (verbose)
	  printf("Dependencies for test case %i failed unexpectedly: "
		 "%s: %s\n", i + 1,
		 package_dep_type_string(dep.dep_type),
		 package_dep_string(&dep));
	return(EXIT_FAILURE);
	break;
      }
    }
  }

  return(EXIT_SUCCESS);
}
