//
// Module Name: AvkCom.h
//
// Description: Common EXTERNAL definitions for all AVK source files
//
// Status: Version 1 Release 2
//
// Copyright Intel Corp. 1990, 1993
// Copyright IBM Corp. 1990, 1993
// All Rights Reserved
//

#ifndef AVKCOM_H_INCLUDED	// Ignore multiple includes
#define AVKCOM_H_INCLUDED	// Flag file as included

//***********************************************************************
// Device Capability Information
//   Set as integer constants in AVK.INI
//   Returned in device capabilities information structure
//***********************************************************************


// External Video keying types
#define AVK_VID_KEY_NONE	0	// Nothing attached to DVI board
#define AVK_VID_KEY_DEFAULT	0	// Default, from AVK.INI
#define AVK_VID_KEY_ANALOG	0x01	// External analog keying (Y cable)
#define AVK_VID_KEY_DIGITAL	0x02	// External digitial keying (VFC) 




// Audio Capture Impedance.
// These definitions are synonymous with the impedance values
// in the AVK.INI file.
#define AVK_AUD_CAPT_NONE	0	// None
#define AVK_AUD_CAPT_DEFAULT	0	// Default, from AVK.INI
#define	AVK_AUD_CAPT_LO_LINE	1
#define AVK_AUD_CAPT_MIC	2
#define AVK_AUD_CAPT_CD		3

// There are synomyms for IMPEDANCE that are used by various
// software. These different spellings are guaranteed
// to remain synonymous as shown.
#define AVK_AUD_NONE		AVK_AUD_CAPT_NONE	// None
#define AVK_AUD_DEFAULT		AVK_AUD_CAPT_DEFAULT	// Default, from AVK.INI
#define AVK_AUD_LO_LINE		AVK_AUD_CAPT_LO_LINE
#define AVK_AUD_MIC		AVK_AUD_CAPT_MIC
#define AVK_AUD_CD		AVK_AUD_CAPT_CD

// Similarly, these synonyms are used in AVK API calls...
#define AVK_AUD_NONE		AVK_AUD_CAPT_NONE	// None
#define AVK_AUD_DEFAULT		AVK_AUD_CAPT_DEFAULT	// Default, from AVK.INI
#define AVK_LO_LINE		AVK_AUD_CAPT_LO_LINE
#define	AVK_MIC			AVK_AUD_CAPT_MIC
#define	AVK_CD_LINE		AVK_AUD_CAPT_CD

// These symbols are obsolete, and should be replaced by the ones above.
// They have been retained to avoid compilation problems in some programs. 
// They are semantically incorrect, and should not be used!
#define	AVK_AUD_CAPT_MIC_HIGH	1
#define AVK_AUD_CAPT_MIC_LOW	2
#define AVK_AUD_CAPT_LINE_INPUT	3
// these synonyms are similarly obsolete...
#define	AVK_AUD_MIC_HIGH	AVK_AUD_CAPT_MIC_HIGH
#define AVK_AUD_MIC_LOW		AVK_AUD_CAPT_MIC_LOW
#define AVK_AUD_LINE		AVK_AUD_CAPT_LINE_INPUT

#define AVK_IMPEDANCE_MAX_TOKEN 3	//largest impedance value supported
// audchannel values... how to record or playback audio...
#define AVK_AUD_LEFT	1	//unsupported
#define AVK_AUD_RIGHT	2	//unsupported
#define AVK_AUD_MIX	3
#define AVK_AUD_STEREO	4
#define AVK_AUD_MAX	AVK_AUD_STEREO


//
// macros to position and recover the bits/sample (with an audio alg nr)...
//
#define  SET_AVK_AUD_PER_SAMP(A) ((A&15) <<8)   // e.g., 4 or 8 or 0 (for 16).
#define  GET_AVK_AUD_PER_SAMP(A) ((A >>8) & 15)

//
// the algorithm symbols used for AvkAudStrmFormat
//
#define  AVK_ADPCM4  (AVK_AUD_ALG_ADPCM4 | (SET_AVK_AUD_PER_SAMP(4)))
#define  AVK_PCM8    (AVK_AUD_ALG_PCM8   | (SET_AVK_AUD_PER_SAMP(8)))



// Video Capture Input Source. These symbols correspond to
// values in AVK.INI, for capture source.
#define AVK_CAPT_SOURCE_NONE		0	// None
#define AVK_CAPT_SOURCE_DEFAULT		0	// Default, from AVK.INI
#define AVK_CAPT_SOURCE_COMPOSITE_1	1	// Red input line
#define AVK_CAPT_SOURCE_COMPOSITE_2	2	// Green input line
#define AVK_CAPT_SOURCE_SUPER_VHS	3	// Super VHS (YC)
#define AVK_CAPT_SOURCE_RGB_SYNC_GREEN	4	// RGB, Sync on Green
#define AVK_CAPT_SOURCE_RGB_SEP_SYNC	5	// RGB, separate Sync


//***********************************************************************
// Portability Data Types and Conventions 
//***********************************************************************

//***********************************************************************
// Portable data types for AVK functions and data definitions 
//   Windows.h typedefs BOOL, BYTE, WORD, DWORD 
//   PM uses [U]SHORT, [U]LONG 
//   ANSI C does not guarantee portable signed/unsigned 8-bit

typedef char     	I8;	//  8-bit signed Char - may not be portable
typedef short     	I16;	// 16-bit signed Integer 
typedef long    	I32; 	// 32-bit signed Integer 

typedef unsigned char	U8;	//  8-bit unsigned Word 
typedef unsigned short	U16;	// 16-bit unsigned Word 
typedef unsigned long	U32;	// 32-bit unsigned Word 

//**********************************************************************
// the avk api, and avkio call mechanisms...

//to use the 32bit compiler, define this symbol before including AVKCOM.H...
#ifdef _32BIT                    // Compilation with IBM C SET2 compiler
 #define AVK_CALL I16 _Far16 _Pascal
 #define AVK_CALLBACK _Far16 _Pascal
#else                                  // other Compilations
 #ifdef _OS2
  #define  AVK_CALL  I16 far pascal _loadds
  #define  AVK_CALLBACK  far pascal _loadds 
 #else
  #define  AVK_CALL  I16 far pascal
  #define  AVK_CALLBACK  far pascal 
 #endif
#endif

//***********************************************************************
// Device Open Flags

#define AVK_OPEN_EXCLUSIVE 0x0001  // Open device - allow no others
#define AVK_DEV_OPEN_EXCLUSIVE  AVK_OPEN_EXCLUSIVE   // required device open flags.
#define AVK_OPEN_SHARE	   0x0000  // Open device shared - reserved; UNSUPPORTED


//***********************************************************************
// AVK Standard Data Definitions 
//***********************************************************************

//***********************************************************************
// Bitmap Data Definitions 
//***********************************************************************

//***********************************************************************
// Bitmap Format Identifiers:
//   Fields defined as xybb: 
//     xy: x & y planar subsample ratio (0=packed, 1,2,4) 
//     bb = bits per pixel 
// Commonly used in the AVKIO API and in many existing A/V data files

// Packed bitmaps, all bits of pixel in same byte/word/longword 
//	(see above for bit field definitions)
// The definitions for  8-bit are reserved; unsupported.
// The definitions for 12-bit are reserved; unsupported.
// The definitions for 32-bit are reserved; unsupported.

#define AVK_BM_8	0x0008	// Packed bitmap, 8 bits per pixel 
#define AVK_BM_16	0x0010	// Packed bitmap, 16 bits per pixel 
#define AVK_BM_32	0x0020	// Packed bitmap, 32 bits per pixel 

// Planar bitmaps: 8 bits per pixel, 3 separate planes
//	(see above for bit field definitions)
#define AVK_BM_9	0x4408	// Planar, Subsampled 1/4 horiz & vert 
#define AVK_BM_10	0x2408	// Planar, Subsampled 1/2 horiz & 1/4 vert ?
#define AVK_BM_12	0x2208	// Planar, Subsampled 1/2 horiz & vertical 
#define AVK_BM_24	0x1108	// Planar, Non-Subsampled (1/1) horiz & vert

// The following macro converts data file bitmap values into
// the corresponding symbols used in AVK API calls...
#define AVK_YUV_BM(format) \
	((format == AVK_BM_9)  ? AVK_YUV9  : \
	((format == AVK_BM_16) ? AVK_YUV16 : \
	((format == AVK_BM_24) ? AVK_YUV24 : \
	((format == AVK_BM_12) ? AVK_YUV24 : 0)))) // 0 : undefined
// Extract Information From Format Data Fields
#define AVK_BM_BITS(fmt)	((fmt) & 0x00ff)	// Bits per pixel 
#define AVK_BM_RATIOX(fmt)	(((fmt) >> 12) & 0x0f)	// Horiz Subsampling
#define AVK_BM_RATIOY(fmt)	(((fmt) >>  8) & 0x0f)	// Vert  Subsampling

#define AVK_BM_PACKED(fmt)	(((fmt) & 0xff00) == 0)	// Packed Format
#define AVK_BM_PLANAR(fmt)	(((fmt) & 0xff00) != 0)	// Planar Format

//***********************************************************************
// Bitmap Plane Flags: Microcode pixel processor drawing & Video display 
//   Flag bits for multiple planes 
//     Y,V,U planes for planar (subsampled) and 32-bit pixels 
//     Low bits so can OR with LSB: Pixel value: AVK_PLANE | AVK_LSB 

#define AVK_PLANE_NONE	0	// No planes 

#define AVK_PLANE_1   	0x01	// 1st plane 
#define AVK_PLANE_2   	0x02	// 2nd plane 
#define AVK_PLANE_3	0x04	// 3rd plane 
#define AVK_PLANE_12	(AVK_PLANE_1 | AVK_PLANE_2)  // 1 & 2 
#define AVK_PLANE_13   	(AVK_PLANE_1 | AVK_PLANE_3)  // 1 & 3 
#define AVK_PLANE_23   	(AVK_PLANE_2 | AVK_PLANE_3)  // 2 & 3 
#define AVK_PLANE_123  	(AVK_PLANE_1 | AVK_PLANE_23) // All 3 
#define AVK_PLANE_ALL  	(AVK_PLANE_1 | AVK_PLANE_23) // All 3 

//There are synomyms for planes that are used by various
// software. These different spellings of the planes are guaranteed
// to remain synonymous as shown.
#define AVK_PLANE_Y   	AVK_PLANE_1  // 1st plane 
#define AVK_PLANE_V   	AVK_PLANE_2  // 2nd plane 
#define AVK_PLANE_U   	AVK_PLANE_3  // 3rd plane 
#define AVK_PLANE_YV   	AVK_PLANE_12  // 1st & 2nd plane 
#define AVK_PLANE_YU   	AVK_PLANE_13  // 1st & 3rd plane 
#define AVK_PLANE_VU   	AVK_PLANE_23  // 2nd & 3rd plane 
#define AVK_PLANE_YVU  	AVK_PLANE_123  // All 3 planes (packed) 

//These guaranteed synonyms are used in AVK API calls...
#define AVK_Y_PLANE	AVK_PLANE_Y
#define AVK_U_PLANE	AVK_PLANE_U
#define AVK_V_PLANE	AVK_PLANE_V
#define AVK_YUV_PLANE	AVK_PLANE_YVU



//***********************************************************************
// Video Data Definitions 
//***********************************************************************

//***********************************************************************
// Video Sync Flags: Used for video formats and monitor types 
// also returned by the AvkVideoIn call to identify input sync
//   Flag bits for multiple syncs (Multisync monitor) 
//	0x000X	- Standard video syncs: NTSC, PAL, SECAM
//	0x00X0	- VGA syncs
//	0x0X00	- XGA syncs

#define AVK_SYNC_DEFAULT	0	// Use default sync (monitor type) 

#define AVK_SYNC_NTSC		0x0001	// NTSC: 480 lines, 30 fr/sec inter
#define AVK_SYNC_PAL		0x0002	// PAL 
#define AVK_SYNC_SECAM   	0x0004	// SECAM (reserved; not supported)
#define AVK_SYNC_VGA18		0x0010	// VGA, Mode 18:  640 x 480 @ 60Hz
#define AVK_SYNC_VGA18_VESA	0x0020	// VGA, Mode 18:  640 x 480 @ 72Hz
#define AVK_SYNC_VGA18_EIBM	0x0040	// VGA, Mode 18:  640 x 480 @ 75Hz
#define AVK_SYNC_XGA		0x0100	// XGA, Mode ??: 1024 x 768

					// all possible VGA syncs
#define AVK_SYNC_VGA_BITS	\
			(AVK_SYNC_VGA18|AVK_SYNC_VGA18_VESA|AVK_SYNC_VGA18_EIBM)


// this bit is reserved and indicates genlocking.
// It does not describe a sync type:
#define AVK_SYNC_LOCK		0x8000	// External Genlock, Flag OR w/ above 

// AVK_VID_... view symbols are guaranteed synonymous with the above.

// video VBI rates expressed as microseconds per vbi
#define AVK_NTSC_USEC_RATE 16683 //59.94 per sec
#define AVK_XGA_USEC_RATE  11500 //86.96 per sec
#define AVK_PAL_USEC_RATE  20000 //25.00 exactly
#define AVK_VGA_USEC_RATE  16683 //same as ntsc but non-interlaced. 60hz!


//these are handy values to use in the
// AvkVidStrmFormat and AvkAudStrmFormat calls,
// for "fullspeed" NTSC and PAL...
#define AVK_NTSC_FULL_RATE 33367L //microsecs per frame for "30 fps" NTSC
#define AVK_PAL_FULL_RATE  40000L //microsecs per frame for "25 fps" PAL

// Sync Rates in floating point: frames per second
// these values are accurate to four significant figures.
// They should not be used to COMPUTE microseconds per frame,
// but are adequate for selecting among supported rates.
#define AVK_SYNC_RATE_NTSC	29.97	// NTSC almost 30 frames, interlaced
#define AVK_SYNC_RATE_PAL	25.00	// PAL  exactly 25 frames, interlaced
#define AVK_SYNC_RATE_VGA18	59.94	// VGA  2 * NTSC, non-interlaced, 60hz
#define AVK_SYNC_RATE_XGA	43.48	// XGA  frames, interlaced

// Sync Rates in floating point: vbis per second (interlaced fields)
#define AVK_SYNC_VBI_NTSC	59.94	// NTSC almost 60 fields
#define AVK_SYNC_VBI_PAL	50.00	// PAL  exactly 50 fields
#define AVK_SYNC_VBI_VGA18	AVK_SYNC_VBI_NTSC	// non-interlaced, 60hz
#define AVK_SYNC_VBI_VESA	72.809	// VGA @ 72hz
#define AVK_SYNC_VBI_EIBM	75.00	// VGA @ 75hz
#define AVK_SYNC_VBI_XGA	86.96	// XGA  interlaced


// Sync rates as 16-bit integer ratios.
// The following pairs of symbols express the frames per second
// as a ratio of two integers..
// _ratio_1_ / _ratio_2_  =   frames per second
#define AVK_SYNC_RATIO_1_NTSC	30000
#define AVK_SYNC_RATIO_2_NTSC	 1001
#define AVK_SYNC_RATIO_1_PAL	25
#define AVK_SYNC_RATIO_2_PAL	 1
#define AVK_SYNC_RATIO_1_VGA18	(AVK_SYNC_RATE_NTSC_RATIO_1 * 2)
#define AVK_SYNC_RATIO_2_VGA18	(AVK_SYNC_RATE_NTSC_RATIO_2 )
#define AVK_SYNC_RATIO_1_XGA	10000
#define AVK_SYNC_RATIO_2_XGA	115

//the following macros assist work with various sync types...

//given a sync value return an integer, the number microseconds per vbi
#define AVK_SYNC_USEC(sync)	\
	(((sync) & AVK_SYNC_NTSC)  ? AVK_NTSC_USEC_RATE :	\
	(((sync) & AVK_SYNC_PAL)   ? AVK_PAL_USEC_RATE :	\
	(((sync) & AVK_SYNC_VGA18) ? AVK_NTSC_USEC_RATE : AVK_XGA_USEC_RATE)))

//given a sync value return a real number -- frames per second.
#define AVK_SYNC_RATE(sync)	\
	(((sync) & AVK_SYNC_NTSC) ? AVK_SYNC_RATE_NTSC :	\
	(((sync) & AVK_SYNC_PAL)  ? AVK_SYNC_RATE_PAL :	\
	(((sync) & AVK_SYNC_VGA18)? AVK_SYNC_RATE_VGA18 : AVK_SYNC_RATE_XGA)))

//given a sync value return a real number -- vbis per second.
#define AVK_SYNC_VBI(sync)	\
	(((sync) & AVK_SYNC_NTSC)  ? AVK_SYNC_VBI_NTSC :	\
	(((sync) & AVK_SYNC_PAL)   ? AVK_SYNC_VBI_PAL :	\
	(((sync) & AVK_SYNC_VGA18) ? AVK_SYNC_VBI_VGA18 : AVK_SYNC_VBI_XGA)))

//given a sync value return the numerator of the integral fps calculation
#define AVK_SYNC_RATIO_1(sync)	\
	(((sync) & AVK_SYNC_NTSC)  ? AVK_SYNC_RATIO_1_NTSC :	\
	(((sync) & AVK_SYNC_PAL)   ? AVK_SYNC_RATIO_1_PAL :	\
	(((sync) & AVK_SYNC_VGA18) ? AVK_SYNC_RATIO_1_VGA18 :	\
					AVK_SYNC_RATIO_1_XGA)))

//given a sync value return the denominator of the integral fps calculation
#define AVK_SYNC_RATIO_2(sync)	\
	(((sync) & AVK_SYNC_NTSC)  ? AVK_SYNC_RATIO_2_NTSC :	\
	(((sync) & AVK_SYNC_PAL)   ? AVK_SYNC_RATIO_2_PAL :	\
	(((sync) & AVK_SYNC_VGA18) ? AVK_SYNC_RATIO_2_VGA18 :	\
					AVK_SYNC_RATIO_2_XGA)))
		

// Given a sync, and an x res, this macro produces a suitable Y res
// for specifying a view...
#define AVK_SYNC_DEFAULT_RES_Y(sync, xres)	\
(((xres) < 512) ?	\
 (((sync) == AVK_SYNC_PAL) ? 288 : ((sync) == AVK_SYNC_XGA) ? 192 : 240) :  \
 (((sync) == AVK_SYNC_PAL) ? 576 : ((sync) == AVK_SYNC_XGA) ? 384 : 480))

#endif       // End of #ifndef to control multiple includes
                                                         
