;
; 	          Diagnostic ROM for IBM-PC/XT and compatibles
;
;			   Written by Ruud Baltissen
;
;
; Notes:
; -	This ROM can be used to diagnose the IBM-PC, IBM-XT and compatible 
;	computers. 
; -	This version only supports the MDA and CGA video adapter. The reason: 
;	the EGA and VGA cards need to be initialized by their own ROMs and I'm
;	sure that they run subroutines and therefore need RAM for the stack. 
;	The Landmark diagnostic ROM supports EGA by using its own routines but
;	1) IMHO that needs more effort than it is worth the trouble and time
;	and 2) and EGA monitors are rare nowadays.

; -	This test simply expects either a MDA or a CGA video present. It will
;	beep after the checksum test to inform the user that the test is running
;	fine so far. But if there is nothing on the screen at that point, there
;	is either something wrong with the board or a part of the video system.
; -	For the future: a version that does beep at every stage for those 
;	people that don't have a MDA or CGA video system. 
; -	The beeping will represent the binary code: a long beep is an one, a
;	short beep is a zero. 

;%define SoundDebug		; sound the codes during OutDebug


; -	If the CPU is to be found bad, the test stops. Why continuing with a
;	bad processor? Remark: a bad EPROM can cause this error as well.
; -	If the checksum of the EPROM is not correct, the test stops. If I can't
;	trust the EPROM, can I trust the outcome of the following tests?
; -	If the program can't find 2 KB of good memory from 0000:0000h on, the
;	test will stop as well. The rest of the program needs RAM for the
;	interrupt vectors and subroutines. Without RAM that is not possible. 
;	I know Landmark can continue by using this ROM-Stack method but this is
;	a very labour intensive way of programming. I'm interested in repairing
;	PCs with this tool and if the memory is bad, then repair that problem
;	first before looking for possible other problems.
; -	General remark: broken glue logic and buffers can cause errors that are
;	interpreted wrong by this diagnostic ROM or errors that won't be 
;	detected at all. For example: a broken data buffer can cause that 
;	nothing shows up at the screen althought the card itself is fine.

; Versions
;	This is a project I do just for fun. So I have no intention to keep
;	track of any change I made. In case of a major change I will save a
;	dated copy but that is all.

; Disclaimer
; -	All names with a copyright are acknowledged.
; -	Some information is derived from deduction after reading a lot of
;	documents and can be unreliable due to the lack of real proof.
; -	If you use this source code and the resulting binary, then it is on
;	your own account. I cannot be held responsible for blowing up your
;	computer, mother-in-law or whatever; it will always be your own fault.
;	USING THIS SOURCE CODE AND ITS BINARY IS COMPLETELY AT YOUR OWN RISK !!!
; -	I'm a Dutchman and English is not my native language. So you can expect
;	to find several linguistic mistakes.



;-------------------------------------------------------------------------------
; I/O port information
;-------------------------------------------------------------------------------
DMA8237		equ	0		; 00h - 1Fh

PIC8259		equ	20h		; 20h - 21h

PIT8253A	equ	40h
PIT8253B	equ	41h
PIT8253C	equ	42h
PIT8253Ctrl	equ	43h

; Programmable Peripheral Interface 8255
i8255A		equ	60h	; read key board OR (PC only) read switches
i8255B		equ	61h
;0061	w	system control port
;		 bit 7 = 1  clear keyboard + (PC only) enable read dip switches
;		 bit 6 = 0  hold keyboard clock low
;		 bit 5 = 0  I/O check enable
;		 bit 4 = 0  RAM parity check enable
;		 bit 3 = 0  XT: read low switches - PC: cassette motor on
;		 bit 2 = 1  XT: reserved / turbo ON - PC: read switches
;		 bit 1 = 1  speaker data enable
;		 bit 0 = 1  timer 2 gate to speaker enable
i8255C		equ	62h
;		 bit 7      read/write memory parity check
;		 bit 6      I/O channel check
;		 bit 5      timer channel 2 OUT
;		 bit 4      spare + (PC only) cassette data in
; XT:
;		 bit 3      SW-4: RAM size 1        Floppy drives 0
;		 bit 2      SW-3: RAM size 0    OR  Floppy drives 1
;		 bit 1      SW-2: co-processor      Video displa 0
;		 bit 0      SW-2: loop on post      Video displa 1
; PC:
;		 bit 3      memory SW2-4
;		 bit 2      memory SW2-3
;		 bit 1      memory SW2-2
;		 bit 0      memory SW2-1  OR  spare SW2-5
i8255Ctrl	equ	63h

DebugIO		equ	80h

; DMA page register
DmaPageRegCh2	equ	81h
DmaPageRegCh3	equ	82h
DmaPageRegCh1	equ	83h

NmiPort		equ	0A0h

MDA_addr	equ	3B4h
MDA_data	equ	3B5h
MDA_ctrl	equ	3B8h

CGA_addr	equ	3D4h
CGA_data	equ	3D5h
CGA_ctrl	equ	3D8h

FdcMotor	equ	3F2h
FdcStatus	equ	3F4h


SegmentMDA	equ	0B000h
SegmentCGA	equ	0B800h


;-------------------------------------------------------------------------------
; Memory reserved for the counters
;-------------------------------------------------------------------------------
InvScreenRAM	equ	4000		; start of invisible screen RAM
ControlWord	equ	InvScreenRAM
Err8253Ch0	equ	InvScreenRAM+2
Err8253Ch1	equ	InvScreenRAM+3
Err8253Ch2	equ	InvScreenRAM+4
Err8237DMA	equ	InvScreenRAM+5
Err8255Parity	equ	InvScreenRAM+6
Err2KbRAM	equ	InvScreenRAM+7
Err8259PIC	equ	InvScreenRAM+8
ErrBadIRQ	equ	InvScreenRAM+9
ErrINT0		equ	InvScreenRAM+10
ErrNMI		equ	InvScreenRAM+11
ErrMemoryMDA	equ	InvScreenRAM+12
ErrMemoryCGA	equ	InvScreenRAM+13
Err8087		equ	InvScreenRAM+14
ErrKeybCntr	equ	InvScreenRAM+15
ErrKeybScan	equ	InvScreenRAM+16
ErrFDC		equ	InvScreenRAM+17
ErrFdcRead	equ	InvScreenRAM+18
ErrRomF400	equ	InvScreenRAM+19
ErrRomF600	equ	InvScreenRAM+20
ErrRomF800	equ	InvScreenRAM+21
ErrRomFA00	equ	InvScreenRAM+22
ErrRomFC00	equ	InvScreenRAM+23
TotalCount	equ	InvScreenRAM+24


;-------------------------------------------------------------------------------
; Create the needed size
;-------------------------------------------------------------------------------
%ifdef Size32KB
 START		equ 08000h		; BIOS starts at offset 8000h
%elifdef Size16KB
 START		equ 0C000h		; BIOS starts at offset C000h
%else
 START		equ 0E000h		; BIOS starts at offset E000h
%endif



;=========================================================================
; setloc - Set location. Insert 0FFh bytes until specifed location is reached.
;-------------------------------------------------------------------------
%imacro setloc  1.nolist
%assign pad_bytes (%1-($-$$)-START)
%if pad_bytes < 0
%assign over_bytes -pad_bytes
%error Preceding code extends beyond setloc location by over_bytes bytes
%endif
%if pad_bytes > 0
%warning Inserting pad_bytes bytes
 times  pad_bytes db 0FFh
%endif
%endm



;**  Update the error counter in front of the text FAILED
;	using ROM stack
; %1 = pointer to line number
; %2 = pointer to error counter
; %3 = pointer to the next test
%macro MacFailed 3
	mov	si,%1
	mov	di,26			; position on the line
	mov	sp,.m10
	jmp	CalcSpotScreen

.m10:
	dw .m12


; DI now points to the correct possition on the screen
.m12:
	mov	sp,.m20
	mov	bx,%2
	jmp	GetErrorValue

.m20:
	dw .m22



; DX/BL contains decimal number
; DI contains location on the screem
.m22:
	mov	sp,.m24
	jmp	DisplayValue

.m24:
	dw .m26

.m26:
	mov	sp,%3
	mov	si,%1
	jmp	DisplayFailed
%endmacro



;**  Output a debug code to the two main LPT ports and IBM's debug port
; REMARK: meant only for parts where RAM for subroutines is NOT available !!!
%macro OutDebug 1
	mov	al,%1

%ifndef SoundDebug
	out	80h,al			; IBM's debug port
	
	mov	dx,378h			; LPT 1
	out	dx,al

	dec	dh
	out	dx,al			; LPT 2
%else
	mov	al,%1
	mov	sp,%%O20
	jmp	OutBeeep

%%O20:
	dw %%O40
%%O40:
	xor	cx,cx
%%O50:
	loop	%%O50
%%O60:
	loop	%%O60
%%O70:
	loop	%%O70
%endif

%endmacro



;**  Output a debug code to the two main LPT ports and IBM's debug port
; REMARK: meant only for parts where RAM for subroutines is available !!!
%macro OutDebugB 1
	mov	al,%1
	call	sOutDebug

 %ifdef SoundDebug
	call	Beeep
	call	Delay3			; between complete code beeps delay
 %endif
%endmacro




 

BITS 16

CPU 8086



;-------------------------------------------------------------------------------
; Start of the actual ROM
;-------------------------------------------------------------------------------

	org	START

%ifdef Size32KB
	setloc	08000h			; 32 KB ROM BIOS
%endif

%ifdef Size16KB
	setloc	0C000h			; 16 KB ROM BIOS
%endif


	setloc	0E000h			; 8 KB ROM BIOS


L_E000:

;------------------------------------------------------------------------------
; Version code
;------------------------------------------------------------------------------

	db 'No copyright, all open source.  '
	db 'Author: Ruud Baltissen  '
	db 'Version: '
	db __DATE__, 32		; Release date (YYYY-MM-DD)
	db __TIME__ 		; Release time (uu:mm:ss)
	db '  '
 

;**  Entry point after an hard reset
ColdStart:
	cli	

	OutDebug 0


;**  As very first we test the CPU
; Jump means error
	xor	ax,ax
	jb	ErrorCPU
	jo	ErrorCPU
	js	ErrorCPU
	jnz	ErrorCPU
	jpo	ErrorCPU

	add	ax,1
	jz	ErrorCPU
	jpe	ErrorCPU

	sub	ax,8002h
	js	ErrorCPU

	inc	ax
	jno	ErrorCPU

	shl	ax,1
	jnb	ErrorCPU
	jnz	ErrorCPU

	shl	ax,1
	jb	ErrorCPU

	xor	ax,ax
	jnz	ErrorCPU		; Jump if not zero

	add	al,80h
	jns	ErrorCPU		; Jump if not sign
	jc	ErrorCPU		; Jump if carry Set
	jp	ErrorCPU		; Jump if parity=1

	add	al,80h
	jnz	ErrorCPU		; Jump if not zero
	jno	ErrorCPU		; Jump if not overflw
	jnc	ErrorCPU		; Jump if carry=0

	lahf				; Load ah from flags
	or	ah,41h
	sahf				; Store ah into flags
	jnz	ErrorCPU		; Jump if not zero
	jnc	ErrorCPU		; Jump if carry=0

	cld				; Clear direction
	xor	si,si			; Zero register
	lodsb				; copy contents of [SI] into AL

	dec	si
	or	si,si			; SI = zero?
	jz	TestCPU1		; yes, -> continue test

; Error in the CPU found
ErrorCPU:
	OutDebug 80h

	cli
	hlt				; Halt processor


;**  Test CPU part 2: test the REGISTERS of the CPU
TestCPU1:
	OutDebug 1

	mov	bx,5555h
.L10:
	mov	bp,bx
	mov	cx,bp
	mov	sp,cx
	mov	dx,sp
	mov	ss,dx
	mov	si,ss
	mov	es,si
	mov	di,es
	mov	ds,di
	mov	ax,ds
	cmp	ax,5555h		; AX = 5555h?
	jnz	.L20			; no, -> 

	not	ax			; AX := 0AAAAh
	mov	bx,ax
	jmp	.L10

.L20:
	xor	ax,0AAAAh		; AX = 0AAAAh?
	jnz	ErrorCPU		; no, -> error


;**  Check the ROM by calculating its checksum
	OutDebug 2

	mov	cx,2000h
	
	mov	ax,cs
	mov	ds,ax			; DS := CS

	mov	bx,0E000h
	xor	al,al			; AL := zero

; Add the content of every byte in this ROM
.L30:
	add	al,[bx]
	inc	bx
	loop	.L30

	or	al,al			; result is zero?
	je	InitIO		; yes, -> OK

; No. Probably bad ROM or maybe hardware that cannot be trusted, so stop.
	OutDebug 81h

	cli
	hlt				; Halt processor


;**   Initialise the 8255 multi-port chip
InitIO:
	OutDebug 3

	mov	al,99h			; program 8255 PIA chip
	out	i8255Ctrl,al		;   Ports A & C, inputs

; Make sure that turbo is off and disable memory parity
	mov	al,0F8h
	out	i8255B,al


;**  Disable NMI interrupts
	xor	al,al
	out	NmiPort,al


;**  Enable the expansion unit
	mov	dx,0213h		; Expansion unit port
	out	dx,al


;**  Disable the CGA and MDA video system ...
	xor	al,al
	mov	dx,CGA_ctrl		; Color Graphic port
	out	dx,al			;   no video display

	inc	al			; AL := 1
	mov	dx,MDA_ctrl		; Monochrome port
	out	dx,al			;   no video display

; ... and initialise them both, no matter what card is present
	mov	ax,cs
	mov	ds,ax			; DS := CS
	mov	ss,ax			; SS := CS

	cld	

; Initialise the MDA port
	mov	dx,MDA_addr		; Monochrome port
	mov	si,Tbl_dataMDA
	mov	cx,16
	xor	bx,bx			; register counter
.L010:
	mov	al,bl
	out	dx,al			; output number of register

	lodsb				; read dat from table
	inc	dl
	out	dx,al			; output data

	dec	dl
	inc	bx
	loop	.L010


;**  Initialise the CGA port
	mov	dx,CGA_addr			; Color Graphic port
	mov	si,Tbl_dataCGA
	mov	cx,16
	xor	bx,bx			; register counter
.L020:
	mov	al,bl
	out	dx,al			; output number of register

	lodsb				; read dat from table
	inc	dx
	out	dx,al			; output data

	dec	dx
	inc	bx
	loop	.L020

	OutDebug 4

; Clear MDA screen
	mov	ax,SegmentMDA
	mov	es,ax

	mov	ax,0720h

	xor	di,di
	mov	cx,4096
	repz	stosw

; Set the correct mode
	mov	dx,MDA_ctrl
	mov	al,29h
	out	dx,al

; Clear CGA screen
	mov	ax,SegmentCGA
	mov	es,ax

	mov	ax,0720h

	xor	di,di
	mov	cx,4096
	repz	stosw

; Set the correct mode
	mov	dx,CGA_ctrl
	mov	al,29h
	out	dx,al

	inc	dx			; 3D9h = CGA palette register
	mov	al,30h
	out	dx,al

	OutDebug 5


;**  Make the cursor invisible
; For MDA
	mov	cx,2607h
	mov	ah,10			; 6845 register

	mov	al,ch			; Send ch, cl through CRT register ah

	mov	dx,3B4h			;   mono port

	xchg	al,ah
	out	dx,al			; send register number

	xchg	al,ah
	inc	dl
	out	dx,al			; send data byte for that register


	inc	ah			;   increment
	mov	al,cl			;   send cl

	mov	dx,3B4h			;   mono port
	xchg	al,ah
	out	dx,al			; send register number

	xchg	al,ah
	inc	dl
	out	dx,al			; send data byte for that register

; For CGA
	mov	cx,2607h
	mov	ah,10			; 6845 register

	mov	al,ch			; Send ch, cl through CRT register ah

	mov	dx,3D4h			;   CGA port

	xchg	al,ah
	out	dx,al			; send register number

	xchg	al,ah
	inc	dl
	out	dx,al			; send data byte for that register


	inc	ah			;   increment
	mov	al,cl			;   send cl

	mov	dx,3D4h			;   CGA port
	xchg	al,ah
	out	dx,al			; send register number

	xchg	al,ah
	inc	dl
	out	dx,al			; send data byte for that register


;**  Reserve the invisible part of the screen memory for the counters
; Invisible RAM =  4 KB - (visible screen + attributes)
;		   4096 -      (25 * 80 * 2)            = 96
CounterRAM:
	OutDebug 6

; Prepare the RAM for storage, first check if already done
	mov	ax,SegmentMDA		; MDA
.L10:
	mov	ds,ax
	mov	es,ax

; Set control word
	mov	ax,5C3Ah
	mov	[ControlWord],ax

; Clear (future) data holders
	xor	ax,ax			; AX := zero
	mov	di,InvScreenRAM+2
	mov	cx,47			; 47 words = 94 bytes
	rep	stosw
.L20:
	mov	ax,ds
	cmp	ax, SegmentMDA		; did we just check MDA?
	jne	DiagLoop		; no, -> was CGA, so continue with test

	mov	ax,SegmentCGA
	jmp	.L10


;**  Output the name of the program and the version
DiagLoop:
	OutDebug 7

	mov	ax,cs
	mov	ds,ax			; DS := CS
	mov	ss,ax			; SS := CS

	mov	sp,.L01
	mov	si,TxtVersion
	jmp	TextToScreen

.L01:
	dw .L02

.L02:
	mov	sp,TestCPU2
	mov	al,1
	jmp	Beeep


;**  We do the previous tests of the CPU and ROM again but now withe the 
;    according text. These tests are included in the loop, the previous ones
;    not. Begin FLAG test of CPU
TestCPU2:
	dw	.L04

.L04:
	OutDebug 8

	mov	sp,.L08
	mov	si,TxtTestCPU
	jmp	TextToScreen

.L08:
	dw	.L12

.L12:
	mov	sp,.L16
	jmp	ShowArrow

.L16:
	dw	.L20

.L20:
	OutDebug 9


;* Jump means error
	xor	ax,ax
	jb	ErrorCPU2
	jo	ErrorCPU2
	js	ErrorCPU2
	jnz	ErrorCPU2
	jpo	ErrorCPU2

	add	ax,1
	jz	ErrorCPU2
	jpe	ErrorCPU2

	sub	ax,8002h
	js	ErrorCPU2

	inc	ax
	jno	ErrorCPU2

	shl	ax,1
	jnb	ErrorCPU2
	jnz	ErrorCPU2

	shl	ax,1
	jb	ErrorCPU2

	xor	ax,ax
	jnz	ErrorCPU2		; Jump if not zero

	add	al,80h
	jns	ErrorCPU2		; Jump if not sign
	jc	ErrorCPU2		; Jump if carry Set
	jp	ErrorCPU2		; Jump if parity=1

	add	al,80h
	jnz	ErrorCPU2		; Jump if not zero
	jno	ErrorCPU2		; Jump if not overflw
	jnc	ErrorCPU2		; Jump if carry=0

	lahf				; Load ah from flags
	or	ah,41h
	sahf				; Store ah into flags
	jnz	ErrorCPU2		; Jump if not zero
	jnc	ErrorCPU2		; Jump if carry=0

	cld				; Clear direction
	xor	si,si			; Zero register
	lodsb				; copy contents of [SI] into AL

	dec	si
	or	si,si			; SI = zero?
	jz	TestCPU4		; yes, -> continue test

; Error in the CPU found
ErrorCPU2:
	OutDebug 82h

	mov	sp,.L10
	mov	si,TxtTestCPU
	jmp	DispCritical

.L10:
	dw .L20

.L20:
	mov	al,82h

; IMHO it makes no sense to continue the test with a bad CPU
HaltCPU:
%ifndef SoundDebug
	cli
	hlt				; Halt processor
%else
.L10:
	mov	al,1
	mov	sp,.L20
	jmp	Beeep

.L20:
	dw .L10				; just loop to produce a continous beep
%endif


;**  Test CPU part 2: test the REGISTERS of the CPU
TestCPU4:
	OutDebug 10

	mov	bx,5555h
.L10:
	mov	bp,bx
	mov	cx,bp
	mov	sp,cx
	mov	dx,sp
	mov	ss,dx
	mov	si,ss
	mov	es,si
	mov	di,es
	mov	ds,di
	mov	ax,ds
	cmp	ax,5555h		; AX = 5555h?
	jnz	.L20			; no, -> 

	not	ax			; AX := 0AAAAh
	mov	bx,ax
	jmp	.L10

.L20:
	xor	ax,0AAAAh		; AX = 0AAAAh?
	jnz	ErrorCPU2		; no, -> error

; Set SS again, has been changed here above
	mov	ax,cs
	mov	ss,ax			; SS := CS

	mov	sp,.L30
	mov	si,TxtTestCPU
	jmp	DisplayPassed

.L30:
	dw	.L40

.L40:
	mov	sp,ChecksumROM
	mov	si,TxtTestCPU
	jmp	RemoveArrow


;**  Check the ROM by calculating its checksum
ChecksumROM:
	dw .L08

.L08:
	OutDebug 12

	mov	sp,.L10
	mov	si,TxtChecksumROM
	jmp	TextToScreen

.L10:
	dw .L12

.L12:
	mov	sp,.L14
	jmp	ShowArrow

.L14:
	dw .L20

.L20:
	OutDebug 14

	mov	cx,2000h
	
	mov	ax,cs
	mov	ds,ax			; DS := CS

	mov	bx,0E000h
	xor	al,al			; AL := zero

; Add the content of every byte in this ROM
.L30:
	add	al,[bx]
	inc	bx
	loop	.L30

	or	al,al			; result is zero?
	je	.L40			; yes, -> OK

; No. Probably bad ROM or maybe hardware that cannot be trusted, so stop.
	OutDebug 83h

	mov	sp,.L33
	mov	si,TxtChecksumROM
	jmp	DispCritical

.L33:
	dw .L34

.L34:
	mov	al,83h
	jmp	HaltCPU


.L40:
	mov	sp,.L97
	mov	si,TxtChecksumROM
	jmp	DisplayPassed

.L97:
	dw	.L98

.L98:
	mov	sp,.L99
	mov	si,TxtChecksumROM
	jmp	RemoveArrow

.L99:
	dw CheckTimers


;**  Check the 8253 timer
CheckTimers:
	OutDebug 18

	mov	sp, .L12
	mov	si,Txt8253Ch0
	jmp	TextToScreen

.L12:
	dw .L14

.L14:
	mov	sp,.L16
	jmp	ShowArrow

.L16:
	dw .L18

.L18:
	OutDebug 20

	mov	al,4
	out	DMA8237+8,al		; disable the DMA controller

; Disable the speakers and enable timer 2
	in	al,i8255B
	or	al,1
	and	al,0FDh
	out	i8255B, al


	OutDebug 22

	mov	dx,0040h		; check counter 0


;**  Common routine for checking the channels of the 8253 timer
; in:	DX = address of channel
Check8253:
	mov	ax,dx			; AX := address of channel to be tested
	and	al,3			; we only need the first two bits
	ror	al,1
	ror	al,1			; bit 1..0 -> bits 7..6 = counter
 
	mov	ah,al			; save counter bits
	or	al,34h			; read/write counter bits 0-7 first,
					;    then 8-15
					; mode 2 select - rate generator
	out	43h,al

	xor	cx,cx			; CX := zero
	mov	bx,cx
	
	mov	al,cl			; AL := zero
	out	dx,al			; write bits 0..7

	nop
	nop

	out	dx,al			; write bits 8..15

	mov	di,4
.L20:
	nop
	nop

	mov	al,ah			; AL := counter
	out	43h,al			; latch command

	nop
	nop

	in	al,dx			; read counter bits 0..7
	or	bl,al			; save read value

	nop
	nop

	in	al,dx			; read counter bits 8..15
	or	bh,al
	cmp	bx,byte -01h		; result is OK?
	je	.L30			; yes, -> 

	loop	.L20			; another try ->

	dec	di			; another try?
	jne	.L20			; yes, -> 

; No, we have an error
	jmp	Error8253

.L30:
	mov	bx,0FFFFh
.L40:
	mov	al,ah			; AH = counter
	out	43h,al

	nop
	nop

	in	al,dx
	and	bl,al

	nop
	nop

	in	al,dx
	and	bh,al
	and	bx,bx			; result is OK?
	je	NoError8253		; yes, -> 

	loop	.L40			; another try ->

; Again, we found an error
	jmp	Error8253


; We found no errors, go to next counter or test
NoError8253:
	mov	si,dx
	and	si,3
	shl	si,1
	jmp	word [cs:si+.L10]

.L10:
	dw Check8253Ch0P
	dw Check8253Ch1P
	dw Check8253Ch2P


; Channel 0 passed the test
Check8253Ch0P:
	mov	sp,Check8253Ch0C
	mov	si,Txt8253Ch0
	jmp	DisplayPassed


Check8253Ch0C:
	dw	.L98

.L98:
	mov	sp,.L99
	mov	si,Txt8253Ch0
	jmp	RemoveArrow

.L99:
	dw Check8253Ch1


;**  Prepare for testing channel 1 of the 8253 timer
Check8253Ch1:
	mov	sp,.L96
	mov	si,Txt8253Ch1
	jmp	TextToScreen

.L96:
	dw	.L97

.L97:
	mov	sp,.L98
	jmp	ShowArrow

.L98:
	dw .L99

.L99:
	OutDebug 24

	mov	dx,0041h
	jmp	Check8253

 
; Channel 1 passed the test
Check8253Ch1P:
	mov	sp,Check8253Ch1C
	mov	si,Txt8253Ch1
	jmp	DisplayPassed

Check8253Ch1C:
	dw	.L98

.L98:
	mov	sp,.L99
	mov	si,Txt8253Ch1
	jmp	RemoveArrow

.L99:
	dw Check8253Ch2


;**  Prepare for testing channel 2 of the 8253 timer
Check8253Ch2:
	mov	sp,.L96
	mov	si,Txt8253Ch2
	jmp	TextToScreen

.L96:
	dw	.L97

.L97:
	mov	sp,.L98
	jmp	ShowArrow

.L98:
	dw .L99

.L99:
	OutDebug 26

	mov	dx,0042h
	jmp	Check8253


; Channel 2 passed the test
Check8253Ch2P:
	mov	sp,Check8253Ch2C
	mov	si,Txt8253Ch2
	jmp	DisplayPassed

Check8253Ch2C:
	dw	.L98

.L98:
	mov	sp,.L99
	mov	si,Txt8253Ch2
	jmp	RemoveArrow

.L99:
	dw	Check8237DMA


; We found an error, display FAILED and go to the next counter or test
Error8253:
	mov	si,dx
	and	si,3
	shl	si,1
	jmp	word [cs:si+.L10]

.L10:
	dw Check8253Ch0E
	dw Check8253Ch1E
	dw Check8253Ch2E


Check8253Ch0E:
	OutDebug 84h

	MacFailed	Txt8253Ch0,Err8253Ch0,Check8253Ch0C


Check8253Ch1E:
	OutDebug 86h

	MacFailed	Txt8253Ch1,Err8253Ch1,Check8253Ch1C


Check8253Ch2E:
	OutDebug 88h

	MacFailed	Txt8253Ch2,Err8253Ch2,Check8253Ch2C


;**  Test the 8237 DMA chip
Check8237DMA:
	OutDebug 28

	mov	sp,.L02
	mov	si,Txt8237DMA
	jmp	TextToScreen

.L02:
	dw	.L04

.L04:
	mov	sp,.L06
	jmp	ShowArrow

.L06:
	dw .L08

.L08:
	OutDebug 30

	mov	al,0
	out	DMA8237+0Dh,al		; stop all DMA activities

	nop
	nop
	nop

	out	DMA8237+8,al

	mov	ax,0FF00h
	mov	di,0001h

; Test with value in AX
.L10:
	xor	dx,dx
	mov	cx,0008h

; First fill registers 0..7 with value in AL
.L20:
	out	dx,al

	nop
	nop
	nop

	out	dx,al

	inc	dx
	loop	.L20

	xor	si,si

	xchg	ah,al
	mov	bx,ax

; Now test with value that was original in AH
.L24:
	mov	dx,si
	out	dx,al

	nop
	nop
	nop

	out	dx,al

	mov	cx,0008h
	xor	dx,dx
.L28:
	cmp	dx,si			; DX = SI?
	je	.L36			; yes, ->

	in	al,dx
	cmp	al,bh			; same as the original value?
	je	.L32			; yes, -> = OK, continue

	jmp	DmaError

.L32:
	in	al,dx
	cmp	al,bh			; same as the original value?
	je	.L40			; yes, -> = OK, continue

	jmp	DmaError

 .L36:
	in	al,dx
	cmp	al,bl			; same as the original value?
	jne	DmaError		; no, -> error

	in	al,dx
	cmp	al,bl			; same as the original value?
	jne	DmaError		; no, -> error

.L40:
	inc	dx
	loop	.L28

	OutDebug 34

	mov	dx,si

	mov	al,bh
	out	dx,al

	nop
	nop
	nop

	out	dx,al

	mov	ax,bx
	inc	si
	cmp	si,8			; we tested all eight registers?
	je	.L44			; yes, -> next phase

	jmp	.L24			; next register

.L44:
	mov	ax,00FFh
	dec	di			; did we test with 00FFh already?
	je	.L10			; no, -> test again with new value

	in	al,08h

	nop
	nop
	nop

	in	al,08h
	test	al,0Fh			; low nible all zero?
	je	DmaOK			; yes, -> 

DmaError:
	OutDebug 8Ah

	mov	sp,.L10
	mov	si,Txt8237DMA
	jmp	DispCritical
; Without DMA the memory cannot be refreshed and therefore I halt the test.

.L10:
	dw .L20

.L20:
	mov	al,8Ah
	jmp	HaltCPU
	

DmaOK:
	mov	sp,.L98
	mov	si,Txt8237DMA
	jmp	DisplayPassed

.L98:
	dw	.L99

.L99:
	mov	sp,InitRefresh0
	mov	si,Txt8237DMA
	jmp	RemoveArrow


;**  Init the DMA and 8253 so the DRAM will be refreshed from now on
InitRefresh0:
	dw InitRefresh


InitRefresh:
	OutDebug 36

	xor	al,al
	out	DMA8237+0Dh,al		; stop DMA on 8237 chip

	nop
	mov	al,58h			; refresh auto-init dummy read
	out	DMA8237+0Bh,al		;   on channel 0 of DMA chip

	nop
	mov	al,41h			; block verify
	out	DMA8237+0Bh,al		;   on channel 1 of DMA chip

	nop
	inc	al
	out	DMA8237+0Bh,al		;   on channel 2 of DMA chip

	nop
	inc	al
	out	DMA8237+0Bh,al		;   on channel 3 of DMA chip

	nop
	mov	al,0FFh			; refresh byte count with 0FFh
	out	DMA8237+1,al

	nop
	nop
	nop

	out	DMA8237+1,al

	nop
	inc	al
	out	DMA8237+0Ah,al		; enable DMA on all channels

	nop
	nop
	nop

	out	DMA8237+8,al		; initialize 8237 command reg with zero

; Set the 8253 timer as needed
	mov	al,54h			; IC 8253 inits memory refresh
	out	PIT8253Ctrl,al		;   chan 1 pulses IC 8237 to

	mov	al,12h			; DMA every 12h clock ticks
	out	PIT8253B,al		;   64K done in 1 millisecond


;**  Check for parity errors
	OutDebug 38

	mov	si,Txt8255Parity
	mov	sp,Check8255Par0
	jmp	TextToScreen

Check8255Par0:
	dw .L14

.L14:
	mov	sp,.L16
	jmp	ShowArrow

.L16:
	dw Check8255Par


Check8255Par:

; Disable memory parity
	OutDebug 40

	mov	dx,i8255B
	in	al,dx
	or	al,30h
	out	dx,al

; Delay
	xor	cx,cx
.L10:
	loop	.L10
.L12:
	loop	.L12

	mov	sp,.L96
	mov	si,Txt8255Parity

	in	al,i8255C
	test	al,0C0h			; parity error found?
	jne	.L20			; yes, -> 

	jmp	DisplayPassed

.L20:
	OutDebug 8Ch

	MacFailed	Txt8255Parity,Err8255Parity,.L96

.L96:
	dw	.L97

.L97:
	mov	sp,.L98
	mov	si,Txt8255Parity
	jmp	RemoveArrow

.L98:
	dw Check2KbRAM


;**  Check the first two KB of memory. We need this for the stack
Check2KbRAM:
	OutDebug 42

	mov	sp,.L00
	mov	si,TxtChk2KbRAM
	jmp	TextToScreen

.L00:
	dw .L01

.L01:
	mov	sp,.L02
	jmp	ShowArrow

.L02:
	dw .L03

.L03:
	cld	

; Enable memory parity
	OutDebug 46

	in	al,61h
	and	al,0CFh
	out	61h,al

; Clear 2 KB of memory
	xor	ax,ax
	mov	es,ax
	mov	ds,ax

	mov	di,ax
	mov	si,ax

	mov	cx,0400h		; 1024 words = 2048 bytes = 2 KB
	repz	stosw

	mov	cx,0400h
	rep	lodsw

	mov	ax,0FF7Fh
	xor	di,di
	mov	cx,0400h
	repz	stosw

	xor	si,si
	mov	cx,0400h
	rep	lodsw

; Toggle memory parity to clear possible errors caused by the above action
	in	al,61h
	or	al,30h			; clear parity
	out	61h,al

	nop
	nop

	and	al,0CFh			;  and enable it again
	out	61h,al

; Prepare for the actual check
	xor	ax,ax
	mov	es,ax			; very first segment

	mov	cx,0800h		; check 2 KB of RAM
	mov	sp,.L04
	jmp	CheckMemory	

.L04:
	dw .L05

.L05:
	mov	si,TxtChk2KbRAM
	jc	.L10			; if error found, ->

	mov	sp,.L96
	jmp	DisplayPassed

.L10:
	OutDebug 8Eh

	mov	sp,.L15
	jmp	DispCritical
; Note: The rest of the routines use subroutines, which means I need good
;	working RAM. If that is not present, I better can stop. A PC wouldn't
;	work as well so why should we continue? :)

.L15:
	dw .L20

.L20:
	mov	al,8Eh
	jmp	HaltCPU
	

.L96:
	dw	.L97

.L97:
	OutDebug 48

	mov	sp,.L98
	mov	si,TxtChk2KbRAM
	jmp	RemoveArrow

.L98:
	dw	CheckAllRAM


;**  Check all RAM until 640 KB or less in chunks of 1 KB
CheckAllRAM:
	xor	ax,ax
	mov	ss,ax			; stack segment := first segment
	mov	sp,05FFh		; set stack pointer

	OutDebugB 50

	mov	si,TxtChkAllRAM
	call	TextToScreen
	call	ShowArrow

	mov	bp,80h			; BP = KB counter, start with 3rd KB
.L10:
	mov	es,bp
	mov	cx,0400h		; test a block of 1 KB
	call	CheckMemory		; error?
	jc	.L80			; yes, ->

; No error found, display amount in decimal numbers
	OutDebugB 52

	add	bp,40h			; increment counter
	mov	dx,3030h		; hundreds and tens

	mov	ax,bp
	mov	cl,6
	shr	ax,cl			; AX := AX div 64 = number of KB
.L20:
	cmp	ax,100			; smaller than 99?
	jb	.L30			; yes, ->

	inc	dh
	sub	ax,100			; subtract 100
	jmp	.L20			; another check for hundreds

.L30:
	cmp	ax,10			; smaller than 9?
	jb	.L40			; yes, ->

	inc	dl
	sub	ax,10			; subtract 10
	jmp	.L30			; another check for tens

;* Display results
.L40:
	or	al,30h			; number -> ASCII
	mov	bl,al			; save AL

; Locate the spot on the screen where to display the results
	mov	si,TxtChkAllRAM
	mov	di,13			; position on the line
	call	CalcSpotScreen

	call	DisplayValue

	cmp	bp,0A000h		; 640 KB?
	jb	.L10			; no, ->

	mov	si,TxtNoMemError
	call	TextToScreen

	mov	si,TxtChkAllRAM
	call	RemoveArrow

	jmp	CheckPIC8259


;**  Bad memory, parity error or less than 640 KB found
; in:	AH = original value
;	DI = pointer within the tested 1 KB of memory
;	ES = current 1 KB segment
; Note:	My idea of "no memory" is that after a test 3 or more bits don't match.
;	I won't check for "good" memory further on to see if other banks are
;	OK again. This is because the hardware can range from 64 KB banks only,
;	a combination of 256 KB plus 64 KB banks or just one 1 MB SIM. IMHO it
;	is up to the user how to interprete the given information.
.L80:
	OutDebugB 54

	mov	ax,5555h
.L81:
	xor	cx,cx
	mov	[es:di],al		; write test value ...
.L82:
	loop	.L82

	mov	al,[es:di]		; ... and read it again after a delay
	xor	al,ah			; find the difference

	cmp	ah,55h			; did we do a second test?
	jne	.L83			; yes, -> 

	mov	bh,al			; save result
	mov	ax,0AAAAh
	jmp	.L81

.L83:
	or	bh,al			; combine results
; Note:	at this point I expect BH to be not zero. If it is, then what went
;	wrong with the original test?

	OutDebugB 56

	xor	ah,ah			; clear counter
	mov	cx,8			; 8 shifts needed
	mov	al,bh			; make a copy
	clc
.L84:
	shr	al,1			; shift bit 0 into Carry, set?
	jnc	.L85			; no, -> skip count

	inc	ah			; increment counter
.L85:
	loop	.L84

	push	ax			; save counter

	call	ES2DEC			; DH/DL/BL contains decimal # of KBs

	pop	ax			; restore counter

	push	bx			; save bad bit pattern / ones
	push	dx			; save hundreds and tens

	cmp	ah,3			; more than 3 bits?
	ja	.L92			; yes, -> consider it as "no memory" 

; Bad memory found
	OutDebugB 58

	mov	si,TxtBadMemory
	call	TextToScreen

	mov	si,TxtBadMemory
	mov	di,13			; position on the line
.L86:
	call	CalcSpotScreen
	push	di			; save place on screen

	mov	si,TxtBadBits
	call	TextToScreen

	pop	di			; restore place on screen
	pop	dx			; restore hundreds and tens
	pop	bx			; restore bad bit pattern / ones

	call	DisplayValue

; SI = TxtBadBits / TxtEndOfMemory, display individual bad bits
	mov	di,80+11		; position on the line
	call	CalcSpotScreen

	mov	cx,8
.L87:
	shl	bh,1			; bit is set?
	jnc	.L88			; no, -> 

	mov	ax,7000h+'X'
	jmp	.L89

.L88:
	mov	ax,0700h+'0'
.L89:
	mov	[es:di],ax		; write result MDA
	add	di,8000h
	mov	[es:di],ax		; write result CGA
	sub	di,7FFCh		; back to MDA and skip space
	loop	.L87

	add	di,2			; skip space

	in	al,i8255C
	test	al,0C0h			; parity error?
	jne	.L90			; yes, ->

	mov	ax,0700h+'O'
	jmp	.L91

.L90:
	mov	ax,7000h+'X'
.L91:
	mov	[es:di],ax		; write result MDA
	add	di,8000h
	mov	[es:di],ax		; write result CGA

	mov	si,TxtChkAllRAM
	call	RemoveArrow

	jmp	CheckPIC8259


; End of memory < 640 KB found
.L92:
	OutDebugB 60

	mov	si,TxtEndOfMemory
	call	TextToScreen

; SI = TxtEndOfMemory
	mov	di,17
	jmp	.L86


;**  Check the 8259 Programmable Interrupt Controller (PIC)
CheckPIC8259:
	mov	si,TxtPIC8259
	call	TextToScreenA

	OutDebugB 62

; Init the 8259
	mov	al,13h			; ICW1 = 
					;  ICW4 needed, single 8259, call
	out	PIC8259,al		;  address interval = 8, edge triggered

	nop
	nop

	mov	al,8			; ICW2 = IRQs start at INT 8
	out	PIC8259+1,al		;   INT 8 = 0000:0020h

	nop
	nop

	mov	al,9			; ICW4 = 
	out	PIC8259+1,al		; non-buffered, normal EOI, 8086 mode

	nop
	nop

; Check the In Service Register (ISR)
	mov	al,0Bh			; OCW3 =
	out	PIC8259,al		;  read IS register on next RD pulse

	in	al,PIC8259		; should be zero
	and	al,al			; is it?
	jne	.L10			; no, -> error

; Check the Interrupt Enable Register
	out	PIC8259+1,al		; AL = zero

	nop
	nop

	in	al,PIC8259+1		; should be zero
	and	al,al			; is it?
	jne	.L10			; no, -> error

	dec	al			; AL := 0FFh
	out	PIC8259+1,al		;  also disable all interrupts

	nop
	nop

	in	al,PIC8259+1		; should be 0FFh
	cmp	al,0FFh			; is it?
	je	L8259OK			; yes, -> 

; Error found in 8259
.L10:
	OutDebugB 90h

	mov	bx,Err8259PIC
	call	DisplayFailedA

	jmp	CheckBadIRQ


; 8259 is OK
L8259OK:
	call	DisplayPassedA


;**  Generate a real interrupt and check the reaction of the 8259 
CheckBadIRQ:
	OutDebugB 64

	mov	si,TxtCheckBadIRQ
	call	TextToScreenA

; Fill the range for INT 8..15 with a vector to the routine RealINT
	xor	ax,ax
	mov	es,ax			; ES := first segment

	mov	di,0020h		; start at ES:0020h
	mov	cx,0008h
.L10:
	mov	ax,RealINT
	stosw

	mov	ax,cs
	stosw

	loop	.L10

	xor	ah,ah			; clear check byte
	sti				; enable interrupts

; Delay
.L20:
	loop	.L20
.L30:
	loop	.L30

	cli				; disable interrupts
	or	ah,ah			; did any interrupt occur?
	jne	.L80			; yes, -> error
; Note: AH contains the number of the bad interrupt. Future use maybe?

	call	DisplayPassedA
	jmp	CheckINT0

.L80:
	OutDebugB 92h

	mov	bx,ErrBadIRQ
	call	DisplayFailedA


;**  Check if INT0 is recognised = timer interrupt
CheckINT0:
	OutDebugB 66

	mov	si,TxtCheckINT0
	call	TextToScreenA

	mov	ax,00FEh		; clear counter and
	out	PIC8259+1,al		;  only ensable timer interrupt

	mov	al,10h			; select timer 0, LSB, mode 0, binary
	out	PIT8253Ctrl,al

	mov	ax,24
	out	PIT8253A,al

	mov	cx,ax

	sti	
.L10:
	or	ah,ah			; did an interrupt occur?
	jne	.L30			; yes, -> is good

	loop	.L10

; No interrupt occurred = error
.L20:
	OutDebugB 94h

	mov	bx,ErrINT0
	call	DisplayFailedA
	jmp	CheckNMI


; Interrupt occurred = OK
.L30:
	OutDebugB 68

	cli				; disable interrupts
	mov	cl,20

	mov	al,0FFh			; set timer
	out	PIT8253A,al

	mov	ax,00FEh		; clear counter and
	out	PIC8259+1,al		;  reensable timer interrupt

	sti
.L40:
	or	ah,ah			; did an interrupt occur?
	jne	.L20			; yes, -> 

	loop	.L40

; No interrupt occurred = OK
.L50:
	cli				; disable interrupts

	call	DisplayPassedA


;**  Check the non maskable interrupt
CheckNMI:
	cli				; disable interrupts

	OutDebugB 70

	mov	si,TxtCheckNMI
	call	TextToScreenA

	xor	ax,ax
	mov	es,ax

	mov	di,0008h

	mov	ax,RealNMI
	stosw

	mov	ax,cs
	stosw

	xor	ah,ah

; Disable memory parity
	mov	dx,i8255B
	in	al,dx
	or	al,30h
	out	dx,al

	mov	al,80h
	out	0A0h,al			; enable NMI

; Delay
	xor	cx,cx
.L10:
	loop	.L10
.L12:
	loop	.L12
.L14:
	loop	.L14

	xor	al,al
	out	NmiPort,al		; disable NMI interrupts

	or	ah,ah			; interrupt found?
	je	.L20			; no, -> continue

; Found one = error
	OutDebugB 96h

	mov	bx,ErrNMI
	call	DisplayFailedA
	jmp	CheckScrMemory

.L20:
	call	DisplayPassedA


;**  Check the screen memory of the video card (if present)
CheckScrMemory:
	OutDebugB 72

	mov	si,TxtCheckMDA
	call	TextToScreenA

	mov	dx,MDA_addr
.L04:
	push	dx

	mov	al,14			; HB cursor locaion
	out	dx,al

	inc	dx
	in	al,dx			; read register
	mov	bl,al			;  and save it

	xor	al,al			; AL := zero
	out	dx,al

	nop

	in	al,dx			; read it again
	or	al,al			; still zero?
	jne	.L10			; no, -> error

	mov	al,5
	out	dx,al			; save another test value

	nop

	in	al,dx			;  and read it again
	cmp	al,5			; the same?
	je	.L20			; yes, -> OK

; Display the "FAILED" message
.L10:
	pop	ax			; get I/O address
	push	ax			; and save it again

	cmp	ax,MDA_addr		; MDA?
	jne	.L12			; no, -> CGA

	mov	bx,ErrMemoryMDA
	mov	si,TxtCheckMDA
	jmp	.L14

.L12:
	mov	bx,ErrMemoryCGA
	mov	si,TxtCheckCGA
.L14:
	OutDebugB 96h

	call	DisplayFailedA

	pop	ax			; get I/O address
	cmp	ax,MDA_addr		; MDA?
	jne	.L16			; no, -> was CGA, next test

	jmp	.L58			; do CGA

.L16:
	jmp	Check8087


; 6845 found, restore register 14
.L20:
	mov	al,bl
	out	dx,al

; Now check the first two words
	pop	ax
	push	ax

	cmp	ax,MDA_addr		; MDA?
	jne	.L22			; no, -> CGA

	mov	ax,SegmentMDA		; MDA memory
	jmp	.L24

.L22:
	mov	ax,SegmentCGA		; CGA memory
.L24:
	mov	ds,ax
	mov	es,ax

	xor	si,si

; Save original content
	lodsw
	mov	bx,ax

	lodsw
	mov	cx,ax

; Write two test words
	mov	bp,5555h

; Reset pointers
.L30:
	xor	di,di
	xor	si,si

	mov	ax,bp
	stosw
	stosw

; And test the words
	lodsw
	cmp	ax,bp
	jne	.L10

	lodsw
	cmp	ax,bp
	jne	.L10

	cmp	bp,5555h		; only first test?
	jne	.L40			; no, ->

	not	bp			; BP := 0AAAAh
	jmp	.L30

; Restore original content
.L40:
	xor	di,di

	mov	ax,bx
	stosw

	mov	ax,cx
	stosw

;* Two words have been found OK, check the whole 80*25 words
; Note:	MDA has 4 KB = 4096 bytes and CGA 32 KB of RAM. Idea: use some RAM
;	beyond the 2000 bytes needed for the screen to save counters etc.

; Make a copy of the screen to the normal memory
	cld				; direction := upwards
.L50:
	mov	ax,0080h
	mov	es,ax			; destination segment

; Source segmnt has already been set but save its value now
	mov	dx,ds

	xor	di,di
	xor	si,si

	mov	cx,2010			; 80*25 + 10 words
	rep	movsw

; Now fill it with test values
	mov	bp,0AAAAh
	mov	es,dx			; destination segment := MDA/CGA screen
.L52:
	xor	di,di
	xor	si,si

	mov	cx,2010			; 80*25 + 10 words
	mov	ax,bp
	rep	stosw

; Delay
.L53:
	loop	.L53

; Test the written value
	mov	cx,2010			; 80*25 + 10 words
.L54:
	lodsw
	cmp	ax,bp			; same value?
	jne	.L10			; no, -> error

	loop	.L54

	cmp	bp,0AAAAh		; second test has been done?
	jne	.L56			; yes, ->

	not	bp
	jmp	.L52

; Restore the screen
.L56:
	mov	ax,0080h
	mov	ds,ax			; source segment

	xor	di,di
	xor	si,si

	mov	cx,2010			; 80*25 + 10 words
	rep	movsw

	pop	dx
	cmp	dx,MDA_addr		; CGA has been tested?
	jne	.L90			; yes, ->

	mov	si,TxtCheckMDA
	call	DisplayPassedA
.L58:
	OutDebugB 74

	mov	si,TxtCheckCGA
	call	TextToScreenA

	mov	dx,CGA_addr
	jmp	.L04

.L90:
	mov	si,TxtCheckCGA
	call	DisplayPassedA


;**  Looking for (##### and checking) the 8087 coprocessor
Check8087:
	OutDebugB 76

	mov	si,TxtCheck8087
	call	TextToScreenA
 
	xor	ax,ax
	mov	ds,ax			; first segment

	fninit				; try to init FPU

	mov	bx,0420h
	mov	[bx],ax			; clear memory byte
.wait:
	loop 	.wait 			; wait for coprocessor to initialize

	fnstcw	[bx]			; put control word in memory

	mov	ax,[bx]
	cmp	ax,03FFh		; 8087 present?
	jne	.L10			; no, ->

	mov	si,TxtCheck8087
	call	DisplayPassedA
	jmp	CheckKeyboard

.L10:
	OutDebugB 98h

	mov	bx,Err8087
	mov	si,TxtCheck8087
	call	DisplayFailedA


;**  Check if a keyboard is present
CheckKeyboard:
	OutDebugB 78

	mov	si,TxtChkKeybCont
	call	TextToScreenA

	in	al,i8255A		; clear buffer

; Enable the keyboard
	mov	al,0Ch			; set keyboard CLK line low
	out	i8255B,al

; Delay
	xor	cx,cx
.L04:
	loop	.L04

	mov	al,0CCh			; set CLK, enable lines high
	out	i8255B,al

	nop
	nop

	mov	al,4Ch			; keyboard CLK high, enable low
	out	i8255B,al

; Now wait for an interrupt
	mov	al,0FDh
	out	PIC8259+1,al		; enable keyboard interrupt

	sti	

	call	CheckForINT		; has there been an interrupt?
	jne	.L20			; yes, -> OK
.L10:
	OutDebugB 9Ah

	mov	bx,ErrKeybCntr
	call	DisplayFailedA
	jmp	.L30

.L20:
	in	al,i8255A		; read value
	cmp	al,0AAh			; correct init code?
	jne	.L10			; no, -> error

	call	DisplayPassedA


; Disable all interrupts
.L30:
	mov	al,0FFh
	out	PIC8259+1,al

	cli


;**  Scan the keyboard for not expected output
CheckKeybScan:
	OutDebugB 80

	mov	si,TxtChkKeybScan
	call	TextToScreenA

; Disable the keyboard
	mov	al,0F8h
	out	i8255B,al

	nop
	nop

	mov	al,78h
	out	i8255B,al

	mov	al,0FDh
	out	PIC8259+1,al		; enable keyboard interrupt

; Now wait for an interrupt
	call	CheckForINT		; has there been an interrupt?
	jne	.L20			; yes, -> error

	call	DisplayPassedA
	jmp	.L30

.L20:
	in	al,i8255A		; read value
	push	ax

	mov	si,TxtStuckKey
	call	TextToScreen

	mov	si,TxtStuckKey2
	call	PlaceToScreen

	pop ax				; restore found key
	mov	ch,al			;  and move to CH

	mov	cl,4
	shr	al,cl			; get hi-nibble

	mov	bx,Tbl_ASCII
	xlatb				; transfer into ASCII

	mov	dx,SegmentMDA		; MDA screen
	mov	es,dx
	mov	ah,7			; attribute
	mov	[es:di],ax		; write to the MDA screen
	mov	[es:di+8000h],ax	; write to the CGA screen

	mov	al,ch			; get read byte again
	and	al,0Fh			; get lo-nibble

	mov	bx,Tbl_ASCII
	xlatb				; transfer into ASCII

	mov	ah,7			; attribute
	mov	[es:di+2],ax		; write to the MDA screen
	mov	[es:di+8002h],ax	; write to the CGA screen


	OutDebugB 9Ch

	mov	si,TxtChkKeybScan
	mov	bx,ErrKeybScan
	call	DisplayFailedA

; Disable all interrupts
.L30:
	mov	al,0FFh
	out	PIC8259+1,al

	cli				; disable interrupts

	mov	al,0FCh
	out	i8255A,al		; reset the keyboard


;**  Look for and check the Floppy Disk Controller
CheckFDC:
	OutDebugB 82

	mov	si,TxtChkFDC
	call	TextToScreenA

; Enable interrupts for the FDC
	mov	al,0BFh
	out	PIC8259+1,al

	mov	dx,FdcMotor

	xor	ax,ax
	mov	ds,ax			; DS := first segment

; Normal mode
	mov	al,0Ch
	out	dx,al

	nop
	nop

; Reset FDC
	mov	al,8
	out	dx,al

	nop
	nop

; Normal mode again
	mov	al,0Ch
	out	dx,al

	call	SenseIntStatus		; went OK?
	jb	.L99			; no, -> error

;* Perform the Specify command
	OutDebugB 84

	mov	ah,3			; specify
	call	ProgramFDC		; went OK?
	jb	.L99			; no, -> error found

	mov	ah,0CFh			; step rate time: 12 ms.
					; head unload time: 240 ms
	call	ProgramFDC		; went OK?
	jb	.L99			; no, -> error found

	mov	ah,2			; head load time: 6 ms
					; mode = DMA
	call	ProgramFDC		; went OK?
	jb	.L99			; no, -> error found

	mov	si,TxtChkFDC
	call	DisplayPassedA

	jmp	ChkReadFloppy

.L99:
	OutDebugB 9Eh

	mov	bx,ErrFDC
	mov	si,TxtChkFDC
	call	DisplayFailedA


;**  Checke if a floppy can be read
ChkReadFloppy:
	mov	si,TxtReadFloppy
	call	TextToScreenA

; Activate the motor of drive 0
	OutDebugB 86

	mov	dx,FdcMotor
	mov	al,1Ch
	out	dx,al

; Delay
	xor	cx,cx
.L10:
	loop	.L10

; Program the 8237
	mov	al,46h			; read command for DMA
	out	DMA8237+0Ch,al		;   diddle LSB/MSB flip-flop

	nop
	nop

	out	DMA8237+0Bh,al		;   send it to the 8237

; Set address where to store the read data = 0000:2000h
	xor	al,al
	out	DMA8237+4,al

	nop
	nop

	mov	al,20h			; store at 2000h
	out	DMA8237+4,al

	xor	al,al			; 0000
	out	DmaPageRegCh2,al	; 64K page number to DMA page reg

; Number of bytes to read, 512 = 01FFh 
	mov	al,0FFh
	out	DMA8237+5,al

	mov	al,1
	out	DMA8237+5,al

; Use channel 2
	mov	al,2
	out	DMA8237+10,al

; Start with drive 0
	xor	ax,ax			; first segment / drive A
	mov	ds,ax

; Recalibrate the drive
.L20:
	OutDebugB 88

	mov	[0480h],ah		; save AH

	mov	ah,7
	call	ProgramFDC

	mov	ah,[0480h]		; AH := drive
	call	ProgramFDC		; enter AH = drive

	call	SenseIntStatus		; went OK?
	jb	.L24			; no, -> 

	test	byte [0400h],0C0h	; status = OK?
	je	.L30			; yes, ->

.L24:
	or	byte [0480h],0		; drive A?
	jne	.error			; no, -> error

	mov	ah,1			; try next drive
	jmp	.L20

; Seek the drive
.L30:
	OutDebugB 90

	mov	ah,0Fh
	call	ProgramFDC

	mov	ah,[0480h]		; AH := drive
	call	ProgramFDC

	mov	ah,5			; goto track 5
	call	ProgramFDC

	call	SenseIntStatus		; went OK?
	jb	.error			; no, -> error

	test	byte [0400h],0C0h	; status = OK?
	jne	.error			; no, -> error

; Settle the head. For IBM: 25 * 550 = 13750. For turbo PCs I use 8000h
	mov	cx,8000h
.SettleHead:
	loop	.SettleHead

; Read data from the floppy
	mov	ah,66h			; one side, MFM, skip deleted data
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,[0480h]		; AH := drive, head 0
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,5			; goto track 5
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	xor	ah,ah			; head 0
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,1			; sector 1
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,2			; 512 bytes/sector
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,9			; last sector on the track
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,2Ah			; gap length
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

	mov	ah,0FFh			; data length = non-user defined
	call	ProgramFDC		; error found?
	jb	.error			; no, -> error

; Now wait for an interrupt
	OutDebugB 92

	call	CheckForINT
	je	.error			; no, -> error

	call	ReadDataFDC		; error found?
	jb	.error			; no, -> error

	test	byte [0400h], 0C0h	; status OK?
	jne	.error			; no, -> error

; Floppy read test = OK
	mov	si,TxtReadFloppy
	call	DisplayPassedA

	jmp	.L50

.error:
	OutDebugB 0A0h

	mov	bx,ErrFdcRead
	mov	si,TxtReadFloppy
	call	DisplayFailedA

; Stop the motors
.L50:
	OutDebugB 94

	mov	al,0Ch
	mov	dx,FdcMotor
	out	dx,al

	cli				;disable interrupts


;**  Check the extra ROMs, if present
	OutDebugB 96

	mov	si,TxtRomF4000
	call	TextToScreenA

	mov	bx,ErrRomF400
	mov	di,0F400h
	call	DoChecksumRes

	OutDebugB 98

	mov	si,TxtRomF6000
	call	TextToScreenA

	mov	bx,ErrRomF600
	mov	di,0F600h
	call	DoChecksumRes

	OutDebugB 100

	mov	si,TxtRomF8000
	call	TextToScreenA

	mov	bx,ErrRomF800
	mov	di,0F800h
	call	DoChecksumRes

	OutDebugB 102

	mov	si,TxtRomFA000
	call	TextToScreenA

	mov	bx,ErrRomFA00
	mov	di,0FA00h
	call	DoChecksumRes

	OutDebugB 104

	mov	si,TxtRomFC000
	call	TextToScreenA

	mov	bx,ErrRomFC00
	mov	di,0FC00h
	call	DoChecksumRes


;**  Display the switch settings
; Note:	How to distinct between the PC and the XT? The settings of the first 
;	switch of the PC are obtained by reading port A of the 8255. Reading
;	port A of the 8255 of the XT should give 0FFh but that is not 
;	guaranteed. Assuming the used has one or two drives, switch 8 of SW1 is
;	set to "ON" and will set bit 7 to zero. The same can be said when 
;	reading the dip switches for the drives on a XT: here bit 7 will also
;	be zero and on a PC it will be one.
ReadSwitches:
	OutDebugB 106

	mov	al,0F8h			; select SW1 (if PC) and hi-nibble of
	out	i8255B,al		;  XT's dip switch block

	xor	ah,ah			; clear collector
	mov	cx,10			; read SW1 ten times
.L04:
	in	al,i8255A
	or	ah,al			; OR result with collector
	loop	.L04


	and	ah,80h			; filter bit 7, zero = PC?
	je	SwitchPC		; yes, ->

	OutDebugB 108

	xor	ah,ah			; clear collector
	mov	cx,10			; read port C ten times
.L08:
	in	al,i8255A
	or	ah,al			; OR result with collector
	loop	.L08

	and	ah,8			; filter bit 3, zero = XT?
	je	SwitchXT		; yes, ->

; Incorrect dip switch setting found
	mov	si,TxtBadSwitch
	mov	dh,70h			; set attribute character = inverted
	call	TextToScreen2

	jmp	DispTotalCount


SwitchPC:
	OutDebugB 110

	mov	si,TxtSwitchPC
	call	TextToScreen

	mov	si,TxtSwitchPC2
	call	TextToScreen

	mov	si,TxtSwitchPC3
	call	TextToScreen

	mov	si,TxtSwitchPC4
	call	TextToScreen

	mov	si,TxtSwitchPC5
	call	TextToScreen

	mov	si,TxtSwitchPC6
	call	TextToScreen

	mov	si,TxtSwitchPC7
	call	TextToScreen

	mov	ax,SegmentMDA		; start with MDA
.L04:
	mov	ds,ax
	
	mov	al,0F4h			; read first four switches
	out	i8255B,al

	in	al,i8255A		; read SW1

	mov	si,TxtSwitchXT5
	mov	di,5			; start at offset 5
	call	SubSwitches

	call	SubSwitches2

	mov	si,TxtSwitchXT5
	mov	di,19			; start at offset 5
	call	SubSwitches

	jmp	DispTotalCount



SwitchXT:
	OutDebugB 112

	mov	si,TxtSwitchXT
	call	TextToScreen

	mov	si,TxtSwitchXT2
	call	TextToScreen

	mov	si,TxtSwitchXT3
	call	TextToScreen

	mov	si,TxtSwitchXT4
	call	TextToScreen

	mov	si,TxtSwitchXT5
	call	TextToScreen

	mov	si,TxtSwitchXT6
	call	TextToScreen

	mov	si,TxtSwitchXT7
	call	TextToScreen

	mov	ax,SegmentMDA		; start with MDA
.L04:
	mov	ds,ax

	mov	al,0F4h			; read first four switches
	out	i8255B,al

	call	SubSwitches2

	mov	si,TxtSwitchXT5
	mov	di,12			; start at offset 12
	call	SubSwitches


;**  Display the total count
DispTotalCount:
	OutDebugB 124

	mov	si,TxtTotalCount
	call	TextToScreen

	mov	bx,TotalCount
	call	GetErrorValue

	mov	si,TxtTotalCount
	mov	di,18			; position on the line
	call	CalcSpotScreen

	call	DisplayValue

	OutDebugB 126

	jmp	DiagLoop




;-------------------------------------------------------------------------------
; Subroutines
;-------------------------------------------------------------------------------


OutBeeep:
	out	80h,al			; IBM's debug port
	
	mov	dx,378h			; LPT 1
	out	dx,al

	dec	dh
	out	dx,al			; LPT 2

	cmp	al,6			; ready with screen?
	jb	Beeep.L50		; no, -> 

;**  PC speaker beep
; in:	AL = bit code: 1 = long beep, 0 = short beep
; Note:	The calling program must take care of a silent delay between two calls.
Beeep:
	or	al,al			; valid value?
	je	.L50			; no, -> exit

	mov	dl,al			; save original value
	mov	bl,al			; use BL as work value
	mov	bh,8			; counter
.L04:
	test	bl,80h			; bit 7 set?
	jne	.L10			; yes, -> 

	dec	bh
	shl	bl,1			; rotate next bit into bit 7
	jmp	.L04

.L10:
	mov	al,10110110b		; Timer IC 8253 square waves
	out	PIT8253Ctrl,al		;   channel 2, speaker

	mov	ax,528h			; Get countdown constant word
	out	PIT8253C,al		;   send low order

	mov	al,ah			;   load high order
	out	PIT8253C,al		;   send high order

	in	al,i8255B		; Read IC 8255 machine status
	mov	ah,al			; save settings of speaker port

	or	al,3
	out	i8255B,al		; Turn speaker on

	xor	cx,cx			; clear counter

	mov	al,1			; short beep

	test	bl,80h			; short beep indeed?
	je	.L20			; yes, ->

	mov	al,3			; long beep
.L20:
	loop	.L20

	dec	al
	jnz	.L20

	mov	al,ah			; restore original value
	out	i8255B,al		; Turn speaker off

	dec	bh			; more beeps?
	je	.L50			; no, -> exit

	shl	bl,1			; rotate BL

; silent delay
.L30:
	loop	.L30
.L40:
;	loop	.L40
	jmp	short .L10

.L50:
	mov	al,dl			; restore original AL
	ret



;**  Calculate the spot on the screen where to place some text
; in:	DI = offset on the line in number of chars
;	SI = points to the line number
; out:	DI = offset on the screen
; used:	AX, CH, DS
CalcSpotScreen:
	push	ax			; save AX

	shl	di,1			; DI := DI * 2 because of attribute

	mov	al,[cs:si]		; get row
	inc	si

	xor	ah,ah
	mov	ch,160			; = 2 * 80 (character + attribute)
	mul	ch			; AX := start of the line
	add	di,ax			; save result

	mov	al,[cs:si]		; get column

	xor	ah,ah
	shl	ax,1			; AX := AX * 2 because of attribute
	add	di,ax			; save end result

	pop	ax			; restore AX
	ret



;** Check if an interrupt has ocured
; out:	Zero flag, clear = none, set = yes
CheckForINT:
	xor	ah,ah			; clear AH = flag

	sti				; enable interrupts

; Delay
	mov	bl,4
	xor	cx,cx
.wait:
	or	ah,ah			; has there been an interrupt?
	jne	.end			; yes, -> 

	loop 	.wait 			; Keayboard must send init code now

	dec	bl
	jne	.wait
.end:
	cli	
	ret	


 
;**  Check a block of memory on write/read and pairty errors
; in:	CX = amount of memory in bytes to be tested
;	ES = segment
; out:	Carry: set = error, clear = OK
CheckMemory:
	mov	dx,i8255B
	in	al,dx
	or	al,30h			; clear old memory parity error
	out	dx,al

	mov	dx,cx			; save CX
	mov	ax,es
	mov	ds,ax

;* First "clear" the memory to make sure the parity bit is set right
; Write to memory
	xor	ax,ax
	mov	di,ax
	mov	si,ax
	repz	stosw

; Now read it
	mov	cx,dx
	rep	lodsw

; Write to memory again with other parity value
	mov	ax,0FF7Fh
	xor	di,di
	mov	cx,dx
	repz	stosw

; And now read it again
	xor	si,si
	mov	cx,0400h		; 
	rep	lodsw

; Enable parity
	mov	dx,i8255B
	in	al,dx
	and 	al,0CFh
	out	dx,al

;* Now the actual test
	mov	ah,55h

; Write the memory
.L10:
	mov	al,ah

	mov	cx,dx
	xor	di,di
	rep	stosb

; Long delay to check if refresh works as it should
	mov	al,1
 mov cx,2	
.L20:
	loop	.L20

	dec	al			; more waiting to do?
	jne	.L20			; yes, ->

	xor	di,di
	mov	cx,dx			; restore CX
.L30:
	cmp	ah,[es:di]		; same as original written value?
	jne	.L90			; no, -> error

	in	al,i8255C
	test	al,0C0h			; parity error?
	jne	.L96			; yes, ->

	loop	.L30	

	cmp	ah,0AAh			; 2nd check done?
	je	.L40			; yes, -> exit

	not	ah			; AH := 0AAh
	jmp	.L10

; Test went well completely
.L40:
	clc
	ret

.L90:
	stc
	ret

 
.L96:
; Parity error found
	in	al,61h
	or	al,30h			; disable parity
	out	61h,al

	stc
	ret



;**  Output a debug code to the last positions of the first line
; in:	AL = code
DebugScreen:
	push	ax
	push	cx
	push	ds

	mov	cx,SegmentMDA
	mov	ds,cx			; ES points to MDA screen

	mov	ah,7			; attribute
	mov	ch,al			; save AL
	
	mov	cl,4
	shr	al,cl			; 1234 xxxx -> 0000 1234
	or	al,30h
	cmp	al,3Ah			; 0..9 ?
	jb	.L20			; yes, -> 

	add	al,7			; A..F
.L20:
	mov	[9Ch],ax
	mov	[809Ch],ax		; update CGA screen

	mov	al,ch			; restore AL
	and	al,0Fh			; xxxx 1234 -> 0000 1234
	or	al,30h
	cmp	al,3Ah			; 0..9 ?
	jb	.L40			; yes, -> 

	add	al,7			; A..F
.L40:
	mov	[9Eh],ax
	mov	[809Eh],ax		; update CGA screen

	pop	ds
	pop	cx
	pop	ax

	ret



;**  Delay three complete CX loops
Delay3:
	xor	cx,cx
.L10:
	loop	.L10
.L20:
	loop	.L20
.L30:
	loop	.L30

	ret



;**  Display critical message on the screen
; in:	SI points to original message
DispCritical:
	mov	ax,cs
	mov	ds,ax

; Calculate position on the screen
	cld	
	lodsb				; read row of original message

	add	al,2			; skip two lines

	mov	ch,0A0h
	mul	ch
	mov	di,ax			; DI := start of line on screen

	add	di,4			; start of critical message

	mov	si,TxtCritical
	mov	dh,70h			; char attribute = inverted

	jmp	TextToScreen3



;**  Display the text "FAILED" behind message in DX
; in:	SI points to original message
DisplayFailed:
	mov	dx,TxtFailed
	mov	bl,70h			; char attribute = inverted
	jmp	DispSecondMsg2


 
;**  Display the text "FAILED" behind message in DX and remove the arrow
; in:	SI points to original message
;	BX points to the byte containing the value 
DisplayFailedA:
	push	si
	push	bx

	mov	di,26			; position on the line
	call	CalcSpotScreen

; DI now points to the correct possition on the screen
	pop	bx
	call	GetErrorValue

; DX/BL contains decimal number
; DI contains location on the screem
	pop	si
	push	si
	call	DisplayValue

	mov	dx,TxtFailed
	mov	bl,70h			; char attribute = inverted
	call	DispSecondMsg2

	pop	si

	jmp	RemoveArrow


 
;**  Display the text "PASSED" behind message in DX
; in:	SI points to original message
DisplayPassed:
	mov	dx,TxtPassed
	jmp	DispSecondMsg



;**  Display the text "PASSED" behind message in DX and remove the arrow
; in:	SI points to original message
DisplayPassedA:
	push	si

	mov	dx,TxtPassed
	call	DispSecondMsg

	pop	si

	jmp	RemoveArrow



;**  Display decimal value in DH/DL/BL on CGA and MDA screen
; in:	BL = ones
;	DL = tens
;	DH = hundreds
;	DI = place on screen
DisplayValue:
	mov	ax,SegmentMDA		; MDA
	mov	es,ax

	cmp	dh,'0'			; skip hundreds?
	ja	.L10			; no, -> display number

	mov	dh,' '			; fill hundreds with a space

	cmp	dl,'0'			; skip tens?
	jne	.L10			; no, ->

	mov	dl,' '			; fill tens with a space
.L10:
	mov	[es:di],dh		; hundreds -> MDA
	add	di,8000h
	mov	[es:di],dh		; hundreds -> CGA
	sub	di,7FFEh

	mov	[es:di],dl		; tens -> MDA
	add	di,8000h
	mov	[es:di],dl		; tens -> CGA
	sub	di,7FFEh

	mov	[es:di],bl		; ones -> MDA
	add	di,8000h
	mov	[es:di],bl		; ones -> CGA

	ret


 
;**  Display a status message behind a previous one
; in:	SI points to original message
;	DX points to Passed / FAILED that should be written after
;	   the SI message
DispSecondMsg:
	mov	bl,07h			; char attribute = normal
DispSecondMsg2:
	mov	ax,cs
	mov	ds,ax

; Calculate position on the screen
	cld	
	lodsb				; read row, inc SI

	mov	ch,0A0h
	mul	ch
	mov	di,ax			; DI := start of line on screen

	lodsb				; read column
	cbw				; AX := AL
	shl	ax,1			; double for attribute
	add	di,ax			; DI := start of 2nd message on screen
	add	di,30*2			; start of Passed / FAILED

	mov	si,dx			; SI := Passed / FAILED / NOT PRESENT
	mov	dh,bl			; char attribute

	jmp	TextToScreen3



;**  Convert ES to decimal for displaying amount of RAM
; in:	ES = segment
; out:	BL = ones
;	DL = tens
;	DH = hundreds
; used:	AX, CL
ES2DEC:
	mov	ax,es
	mov	cl,6
	shr	ax,cl			; AX := AX div 64 = number of KB



;**  Convert AL to decimal 
; in:	AL = number
; out:	BL = ones
;	DL = tens
;	DH = hundreds
; used:	AX
AL2DEC:
	mov	dx,3030h		; hundreds and tens

	cmp	al,100			; smaller than 99?
	jb	.L30			; yes, ->

	inc	dh
	sub	al,100			; subtract 100
	jmp	AL2DEC			; another check for hundreds

.L30:
	cmp	al,10			; smaller than 9?
	jb	.L40			; yes, ->

	inc	dl
	sub	al,10			; subtract 10
	jmp	.L30			; another check for tens

;* Display results
.L40:
	or	al,30h			; number -> ASCII
	mov	bl,al			; save AL

	ret



;** Perform a checksum on a 8 KB ROM in the F000 segemnt
; in:	DI = start of 8 KB range
; Note:	A range full with 0FFh bytes will end up as well with a checksum of 
;	zero. If we find 2000h FFs, then there is no ROM at all. This is 
;	something the Landmark ROM does not notice.
DoChecksum:
	push	si

	mov	cx,2000h
	xor	dx,dx			; use as FF counter
	xor	si,si			; clear source pointer
	
	mov	ax,cs
	mov	ds,ax			; DS := CS = 0F000h

	xor	ah,ah			; AH := zero, clear the checksum

; Add the content of every byte in this ROM
.L30:
	lodsb				; read from the possible ROM
	add	ah,al			; add to the checksum

	cmp	al,0FFh			; AL = 0FFH?
	jne	.L40			; no, -> skip counter

	inc	dx			; increment 0FFh counter
.L40:
	loop	.L30

	or	ah,ah			; result is zero?
	jne	.L99			; no, -> bad checksum, exit
	
	cmp	dx,2000h		; all bytes are 0FFh?
	jne	.L50			; no, -> good chance it is a ROM

	inc	ah			; make sure zero flag will reset
.L50:
	or	ah,ah			; set or reset zero flag
.L99:
	pop	si
	ret



;**  Perform a checksum and display the result
; in:	DI = start of 8 KB range
;	SI = text info
DoChecksumRes:
	call	DoChecksum
	je	.L10

	jmp	DisplayFailedA

.L10:
	jmp	DisplayPassedA



;**  Retrieve the error counter value from the invisible screen memory
; in: 	BX points to the byte containing the value
GetErrorValue:
	mov	ax,SegmentMDA
.L10:
	mov	ds,ax

	cmp	word [ControlWord],5C3Ah
	je	.L20

	mov	ax,ds
	cmp	ax,SegmentMDA		; just checked MDA?
	je	.L14			; yes, -> do CGA now

	xor	al,al			; display zero
	jmp	.L30

.L14:
	mov	ax,SegmentCGA
	jmp	.L10

.L20:
	inc	byte [bx]
	mov	al,[bx]			; read increased counter

.L30:
	jmp	AL2DEC			; convert AL to decimal



;**  Wait for the 765 to be ready and prgram diskette command/data register 0
; in:	AH = byte for 03F5h = diskette command/data register 0
; out:	Carry, clear = OK, set = bad
ProgramFDC:
	mov	dx,FdcStatus
	xor	cx,cx

; Check if "transfer is from controller to system" and "RQM data register is
; ready".
	mov	bl,4
.L10:
	in	al,dx
	and	al,0C0h
	cmp	al,80h			; Check = OK?
	je	.L20			; yes, -> 

	loop	.L10

	dec	bl
	jne	.L10

	stc	
	ret	

; Program diskette command/data register 0
.L20:
	inc	dx			; DX := 03F5h

	mov	al,ah
	out	dx,al

	clc	
	ret	



;**  Read data from the 765 floppy controller
; out:	Carry, clear = OK, set = bad
ReadDataFDC:
	mov	dx,FdcStatus

	xor	cx,cx
	mov	ds,cx			; DS points to first segment
.L10:
	in	al,dx
	and	al,0C0h
	cmp	al,0C0h			; RQM data register is ready?
					; transfer = from controller to system?
	je	.L20			; yes, -> OK

	loop	.L10

	jmp	.error


 ; Read diskette command/data register 0 
.L20:
	mov	bl,7			; maximal 7 bytes to be read
	xor	si,si

	inc	dx			; DX := 03F5
.L40:
	in	al,dx
	mov	[si+0400h],al		; store read value

	inc	si

; Delay for NEC
	mov	cx,10
.L50:
	loop	.L50

	dec	dx
	in	al,dx

	inc	dx
	test	al,10h			; seek completed?
	jne	.L60			; no, -> continue

; Exit
	clc	
	ret

.L60:
	dec	bl			; another byte to read?
	jne	.L40			; yes, ->

; Error
.error:
	stc	
	ret	



;**  Handle a real test interrupt
RealINT:
	in	al,PIC8259
	or	ah,al

	mov	al,20h			; end-of-interrupt command
	out	PIC8259,al

	iret	



;**  Handle a real NMI
RealNMI:
	xor	al,al
	out	NmiPort,al		; disable NMI interrupts

	mov	ah,1
	iret



;**  FDC command: Sense interrupt status
SenseIntStatus:
	xor	ax,ax
	mov	ds,ax

; Now wait for an interrupt
	call	CheckForINT		; has there been an interrupt?
	je	.L98			; no, -> error = exit

	mov	ah,8			; sense interupt status
	call	ProgramFDC		; went OK?
	jb	.L99			; no, -> error found = exit

	call	ReadDataFDC		; error found?
	jb	.L99			; yes, -> error found = exit

	clc
	ret

.L98:
	stc
.L99:
	ret



;**  Remove the arrow placed at the start of the test
; in:	SI points to parameters of original text 
RemoveArrow:
	mov	dl,1
	jmp	SubArrow

;**  Show an arrow on the screen so the user knows where the ROM is busy
; in:	SI points to parameters of original text 
ShowArrow:
	xor	dl,dl			; show arrow

SubArrow:
	mov	ax,cs
	mov	ds,ax

; Calculate position on the screen
	cld	
	lodsb				; read row, inc SI

	mov	ch,0A0h
	mul	ch
	mov	di,ax			; DI := start of line on screen

	lodsb				; read column
	cbw				; AX := AL
	shl	ax,1			; double for attribute
	add	di,ax			; DI := start of 2nd message on screen
	sub	di,4			; start of '->'

	mov	dh,7			; char attribute

	or	dl,dl			; show arrow?
	jne	.L10			; no, -> 

	mov	si,TxtShowArrow
	jmp	.L20

.L10:
	mov	si,TxtRemoveArrow
.L20:
	jmp	TextToScreen3

TxtRemoveArrow:
	db '  ', 0

TxtShowArrow:
	db '->', 0



;**  Subroutine for outputting debug code on the LPT port and IBM's debug port
sOutDebug:
	out	80h,al			; IBM's debug port
	
	mov	dx,378h			; LPT 1
	out	dx,al

	dec	dh
	out	dx,al			; LPT 2

	ret



SubSwitches:
	call	CalcSpotScreen

	mov	cx,8			; switch counter
.L08:
	shr	al,1			; shift momentary bit 1 into Carry
	jb	.L12			; if set, -> bottom row

; Place 'X' on top row
	mov	byte [di],'X'
	jmp	.L16


; Place 'X' on bottom row
.L12:
	mov	byte [di+160],'X'
.L16:
	inc	di
	inc	di
	loop	.L08

	ret



SubSwitches2:
	in	al,i8255C
	and	al,15			; isolate the four switches
	mov	bl,al			; temporary storage

	mov	al,0FCh			; read second four switches
	out	i8255B,al

	in	al,i8255C

	mov	cl,4
	shl	al,cl			; xxxx 8765  -->  8765 0000
	or	al,bl			; combine results

	ret



;**  Output text, with preset or custom attribute.
; in:	SI points to parameters for the text to be displayed
; used:	AX, BP, BX, CX, DI, DS, DX, ES
; Parameters:
; 1st byte:	row
; 2nd byte:	column
; The text starts at the 3rd byte
TextToScreen:
	mov	dh,07h			; set attribute for character

TextToScreen2:
	mov	ax,cs
	mov	ds,ax

	cld	
	lodsb				; get row

	xor	ah,ah
	mov	ch,160			; = 2 * 80 (character + attribute)
	mul	ch
	mov	di,ax			; save result

	lodsb				; get column

	xor	ah,ah
	shl	ax,1
	add	di,ax			; DI := place where to copy the text to

TextToScreen3:
	mov	ah,dh			; set attribute for character

; Copy the text to the screen
	mov	dx,SegmentMDA		; MDA screen
	mov	es,dx

	mov	bp,di			; save DI
	mov	bx,si			; save SI
	mov	dx,cx			; save CX
.L10:
	lodsb				; read character
	and	al,al			; end of the text?
	je	.L20			; yes, -> 

	stosw				; write character plus attribute
	jmp	.L10

.L20:
	mov	cx,es
	cmp	cx,SegmentCGA		; did we do CGA already?
	je	.L30			; yes, -> exit

; Copy the text to the CGA screen
	mov	di,bp			; restore DI
	mov	si,bx			; restore SI
	mov	cx,dx			; restore CX

	mov	dx,SegmentCGA		; CGA screen
	mov	es,dx

	jmp	.L10

.L30:
	mov	si,bx
	sub	si,2			; restore original SI

	ret	



;**  Output text, with preset or custom attribute and display an arrow
; in:	SI points to parameters for the text to be displayed
; used:	AX, BP, BX, CX, DI, DS, DX, ES
; Parameters:
; 1st byte:	row
; 2nd byte:	column
; The text starts at the 3rd byte
TextToScreenA:
	push	si

	call	TextToScreen
	call	ShowArrow

	pop	si
	ret



;**  Calculate the place on the screen to output characters
; Note: cannot be used by TextToScreen because no subroutines are allowed
;	during the ROM-only-mode
PlaceToScreen:
	mov	ax,cs
	mov	ds,ax

	cld	
	lodsb				; get row

	xor	ah,ah
	mov	ch,160			; = 2 * 80 (character + attribute)
	mul	ch
	mov	di,ax			; save result

	lodsb				; get column

	xor	ah,ah
	shl	ax,1
	add	di,ax			; DI := place where to copy the text to

	ret



;**  Update the error counter in front of the text FAILED
;	using RAM stack
; in:	BX points to the correct counter
;	SI points to the line number
UpdErrorCountB:
	mov	di,26			; position on the line
	call	CalcSpotScreen



	ret



;-------------------------------------------------------------------------------
; Data
;-------------------------------------------------------------------------------


Tbl_ASCII:
	db '0123456789ABCDEF'
	 

;**  Tables with data to initialise the MDA and CGA screen
Tbl_dataMDA:
	db 061h, 050h, 052h, 00Fh, 019h, 006h, 019h, 019h
	db 002h, 00Dh, 00Bh, 00Ch, 000h, 000h, 000h, 000h


Tbl_dataCGA:
	db 071h, 050h, 05Ah, 00Ah, 01Fh, 006h, 019h, 01Ch
	db 002h, 007h, 006h, 007h, 000h, 000h, 000h, 000h
 


;**  Various message outputted during the test 
TxtVersion:
	db  0,  8, "Ruud's Diagbostics ROM for PC/XT    "
	db 'Version: '
	db __DATE__, 32		; Release date (YYYY-MM-DD)
	db __TIME__, 0		; Release time (uu:mm:ss)

TxtTestCPU:
	db  2,  2, 'Testing CPU', 0

TxtChecksumROM:
	db  3,  2, 'Diagnostic ROM checksum', 0

Txt8253Ch0:
	db  4,  2, '8253 timer channel 0', 0

Txt8253Ch1:
	db  5,  2, '8253 timer channel 1', 0

Txt8253Ch2:
	db  6,  2, '8253 timer channel 2', 0

Txt8237DMA:
	db  7,  2, '8237A DMA controller', 0
	
Txt8255Parity:
	db  8,  2, '8255 parity check', 0

TxtChk2KbRAM:
	db  9,  2, 'Check first 2 KB of RAM', 0

TxtChkAllRAM:
	db 10,  2, 'Checked RAM:   2 KB', 0

TxtBadMemory:
	db 11,  2, 'Bad memory at     KB', 0

TxtEndOfMemory:
	db 11,  2, 'End of memory at     KB', 0

TxtNoMemError:
	db 11,  2, 'No memory error found', 0

TxtBadBits:
	db 12,  2, 'Bad bits:  7 6 5 4 3 2 1 0  P', 0

TxtPIC8259:
	db 14,  2, '8259 interrupt controller', 0

TxtCheckBadIRQ:
	db 15,  2, 'Looking for bad interrupt', 0

TxtCheckINT0:
	db 16,  2, 'Checking INT 0', 0

TxtCheckNMI:
	db 17,  2, 'Checking nonmaskable INT', 0

TxtCheckMDA:
	db 18,  2, 'Checking MDA + memory', 0

TxtCheckCGA:
	db 19,  2, 'Checking CGA + memory', 0

TxtCheck8087:
	db 20,  2, 'Checking 8087 coprocessor', 0

TxtChkKeybCont:
	db 21,  2, 'Check keyboard controller', 0

TxtChkKeybScan:
	db 22,  2, 'Check keyboard scan lines', 0

TxtChkFDC:
	db 23,  2, 'Check floppy controller', 0

TxtReadFloppy:
	db 24,  2, 'Trying to read a floppy', 0






TxtRomF4000:
	db  2, 42, 'Check ROM at F4000', 0

TxtRomF6000:
	db  3, 42, 'Check ROM at F6000', 0

TxtRomF8000:
	db  4, 42, 'Check ROM at F8000', 0

TxtRomFA000:
	db  5, 42, 'Check ROM at FA000', 0

TxtRomFC000:
	db  6, 42, 'Check ROM at FC000', 0

TxtBadSwitch:
	db  8, 42, 'BAD SWITCH SETTING!', 0

TxtSwitchPC:
	db  8, 42, 'Switch settings for the PC', 0

TxtSwitchXT:
	db  8, 42, 'Switch settings for the XT', 0

TxtSwitchPC2:
	db 10, 42, '       SW1           SW2   ', 0

TxtSwitchXT2:
	db 10, 42, '               SW          ', 0

TxtSwitchPC3:
	db 11, 42, '    '
	db 218, 196, 196, 196, 196, 196, 196, 196, 196, 191,
	db '    '
	db 218, 196, 196, 196, 196, 196, 196, 196, 196, 191, 0
;	db 11, 42, '    ┌────────┐    ┌────────┐', 0

TxtSwitchXT3:
	db 11, 42, '           '
	db 218, 196, 196, 196, 196, 196, 196, 196, 196, 191, 0
;	db 11, 42, '           ┌────────┐       ', 0

TxtSwitchPC4:
	db 12, 42, '    ', 179, '12345678', 179
	db '    ', 179, '12345---', 179, 0
;	db 12, 42, '    │12345678│    │12345---│', 0

TxtSwitchXT4:
	db 12, 42, '           ', 179, '12345678', 179, '       ', 0
;	db 12, 42, '           │12345678│       ', 0

TxtSwitchPC5:
	db 13, 42, ' ON ', 179, '        ', 179
	db '    ', 179, '        ', 179, '', 0
;	db 13, 42, ' ON │        │    │        │', 0

TxtSwitchXT5:
	db 13, 42, '        ON ', 179, '        ', 179, '       ', 0
;	db 13, 42, '        ON │        │       ', 0

TxtSwitchPC6:
	db 14, 42, '    ', 179, '        ', 179
	db '    ', 179, '        ', 179, '', 0
;	db 14, 42, '    │        │    │        │', 0

TxtSwitchXT6:
	db 14, 42, '           ', 179, '        ', 179, '       ', 0
;	db 14, 42, '           │        │       ', 0

TxtSwitchPC7:
	db 15, 42, '    '
	db 192, 196, 196, 196, 196, 196, 196, 196, 196, 217,
	db '    '
	db 192, 196, 196, 196, 196, 196, 196, 196, 196, 217, 0
;	db 15, 42, '    └────────┘    └────────┘', 0

TxtSwitchXT7:
	db 15, 42, '           '
	db 192, 196, 196, 196, 196, 196, 196, 196, 196, 217, 0
;	db 15, 42, '           └────────┘       ', 0

TxtTotalCount:
	db 18, 42, 'Completed passes:', 0

TxtStuckKey:
	db 22, 42, 'Stuck key found:          ', 0

TxtStuckKey2:
	db 22, 59



TxtCritical:
	db 'Critical error, diagnostics heave been stopped!', 0

TxtFailed:
	db 'FAILED', 0

TxtPassed:
	db 'Passed', 0



;-------------------------------------------------------------------------------


	setloc	0FFF0h

;------------------------------------------------------------------------------
; Power-On Entry Point
;------------------------------------------------------------------------------
PowerOn:
	jmp	0F000h:ColdStart

 
S_FFF5:
	db __DATE__		; Release date (YYYY-MM-DD)

	db 0			; checksum
	
 

