
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit SBMixer   : Eine Unit, die eine Vielfalt von Routinen zur Program-   }
{                  mierung des Mischer-Chips des Sound Blaster Pro (Version }
{                  2 und 3) und des Sound Blaster 16 ASP bzw. einer ver-    }
{                  gleichbaren Karte (z.B. Audio Blaster Pro 4) zur Verf-  }
{                  gung stellt.                                             }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : SBMIXER.PAS                                              }
{ entwickelt am  : 08.08.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.02                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT SBMixer;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

INTERFACE

USES CRT, SBDrv;                            { Units CRT und SBDrv einbinden }

TYPE

  {=========================================================================}
  { TSides: linker und rechter Stereo-Kanal                                 }
  {=========================================================================}

  TSides = (Left_, Right_);

  {=========================================================================}
  { TInputFilters: Eingabe-Filter: kein, Tieftne, Hochtne                 }
  {=========================================================================}

  TInputFilters = (NoFilter, LowFilter, HighFilter);

VAR
  SBBoardType : Word;              { Typ der Soundkarte: SB Pro / SB 16 ASP }

{===========================================================================}
{ Prozedur InitSBMixer: Initialisiert die Unit. Zunchst ruft diese Routine }
{                       InitSBDrv, um die Basis-I/O-Adresse, die Interrupt- }
{                       und die DMA-Nummer aus der BLASTER-Umgebungsvariab- }
{                       len auszulesen. Danach testet sie, ob eine Sound    }
{                       Blaster Pro oder eine Sound Blaster 16 ASP vorhan-  }
{                       den ist.                                            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE InitSBMixer;

{===========================================================================}
{ Funktion SBMixerIn: Liest den Wert eines Mischer-Registers aus (nur       }
{                     Reset-Register 0 ist "Write Only").                   }
{===========================================================================}
{ Eingabe: Reg = Nummer des Registers                                       }
{ Ausgabe: ein Datenbyte                                                    }
{---------------------------------------------------------------------------}

FUNCTION SBMixerIn(Reg : Byte) : Byte;

{===========================================================================}
{ Prozedur SBMixerOut: Gibt einen Wert an ein Mischer-Register aus.         }
{===========================================================================}
{ Eingabe: Reg  = Nummer des Registers (bei SB Pro zwischen 0 und 2E hex.,  }
{                 bei SB 16 zwischen 0 und 47 hex.)                         }
{          Data = Datenbyte                                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBMixerOut(Reg, Data : Byte);

{===========================================================================}
{ Prozedur MixerReset: Fhrt einen Reset des Mischer-Chips aus. Danach wer- }
{                      den alle Registerwerte auf die Standardwerte einge-  }
{                      stellt.                                              }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE MixerReset;

{===========================================================================}
{ Prozedur SetMasterVolume: Setzt die Gesamtlautstrke fr VOICE, FM, CD    }
{                           und LINE-IN.                                    }
{===========================================================================}
{ Eingabe: Side  = linker oder rechter Stereo-Kanal (Left_, Right_)         }
{          Param = 0-15 bei SB Pro (Version 2 und 3), 0-255 bei SB 16       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetMasterVolume(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetMasterVolume: Liest die Gesamtlautstrke aus.                 }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: Gesamtlautstrke                                                 }
{---------------------------------------------------------------------------}

FUNCTION GetMasterVolume(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetVoiceVolume: Setzt die VOICE-Lautstrke.                      }
{===========================================================================}
{ Eingabe: Side  = linker oder rechter Stereo-Kanal (Left_, Right_)         }
{          Param = 0-15 bei SB Pro (Version 2 und 3), 0-255 bei SB 16       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetVoiceVolume(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetVoiceVolume: Liefert den Wert fr die VOICE-Lautstrke zu-    }
{                          rck.                                            }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: VOICE-Lautstrke                                                 }
{---------------------------------------------------------------------------}

FUNCTION GetVoiceVolume(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetFMVolume: Setzt die FM-Lautstrke.                            }
{===========================================================================}
{ Eingabe: Side  = linker oder rechter Stereo-Kanal (Left_, Right_)         }
{          Param = 0-15 bei SB Pro (Version 2 und 3), 0-255 bei SB 16       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetFMVolume(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetFMVolume: Liefert den Wert fr die FM-Lautstrke zurck.      }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: FM-Lautstrke                                                    }
{---------------------------------------------------------------------------}

FUNCTION GetFMVolume(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetCDVolume: Setzt die Lautstrke fr CD-Audio.                  }
{===========================================================================}
{ Eingabe: Side  = linker oder rechter Stereo-Kanal (Left_, Right_)         }
{          Param = 0-15 bei SB Pro (Version 2 und 3), 0-255 bei SB 16       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetCDVolume(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetCDVolume: Liefert den Wert fr die CD-Lautstrke zurck.      }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: CD-Lautstrke                                                    }
{---------------------------------------------------------------------------}

FUNCTION GetCDVolume(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetLineVolume: Setzt die Lautstrke fr den LINE-Eingang.        }
{===========================================================================}
{ Eingabe: Side  = linker oder rechter Stereo-Kanal (Left_, Right_)         }
{          Param = 0-15 bei SB Pro (Version 2 und 3), 0-255 bei SB 16       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetLineVolume(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetLineVolume: Gibt den Wert fr die LINE-IN Lautstrke zurck.  }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: LINE-IN Lautstrke                                               }
{---------------------------------------------------------------------------}

FUNCTION GetLineVolume(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetMicVolume: Setzt die Lautstrke fr den Mikrofon-Eingang.     }
{===========================================================================}
{ Eingabe: Param = 0-7 bei SB Pro (Ver. 2 und 3), 0-255 bei SB 16 oder SB   }
{                  Pro 4                                                    }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetMicVolume(Param : Byte);

{===========================================================================}
{ Funktion GetMicVolume: Liefert die Lautstrke fr den Mikrofon-Eingang    }
{                        zurck.                                            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Mikrofon-Lautstrke                                              }
{---------------------------------------------------------------------------}

FUNCTION GetMicVolume : Byte;

{===========================================================================}
{ Prozedur SetPCSpkVolume: Setzt die Lautstrke fr den PC-Lautsprecher,    }
{                          falls dieser ber die Sound-Blaster-Karte ge-    }
{                          steuert wird. Diese Routine kann nur bei SB 16   }
{                          aufgerufen werden, sonst hat sie keine Wirkung.  }
{===========================================================================}
{ Eingabe: Param = Laustrke des PC-Lautsprechers (0-255)                   }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetPCSpkVolume(Param : Byte);

{===========================================================================}
{ Funktion GetPCSpkVolume: Liefert die Lautstrke des PC-Lautsprechers,     }
{                          wenn dieser ber die Soundkarte gesteuert wird.  }
{                          Die Funktion kann nur bei SB 16 ASP aufgerufen   }
{                          werden, sonst liefert sie immer den Wert 0 zu-   }
{                          rck.                                            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Lautstrke des PC-Lautsprechers                                  }
{---------------------------------------------------------------------------}

FUNCTION GetPCSpkVolume : Byte;

{===========================================================================}
{ Prozedur SetInputFilter: Setzt einen Input-Filter.                        }
{===========================================================================}
{ Eingabe: WhatType = Filterart (NoFilter, LowFilter, HighFilter)           }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetInputFilter(WhatType : TInputFilters);

{===========================================================================}
{ Funktion LowInputFilterOn: Liefert TRUE zurck, wenn der Input-Filter fr }
{                            Tieftne aktiv ist, sonst FALSE.               }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION LowInputFilterOn : Boolean;

{===========================================================================}
{ Funktion HighInputFilterOn: Gibt TRUE zurck, wenn der Input-Filter fr   }
{                             Hochtne gesetzt ist, sonst FALSE.            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION HighInputFilterOn : Boolean;

{===========================================================================}
{ Prozedur SetOutputFilter: Schaltet den Output-Filter ein bzw. aus.        }
{===========================================================================}
{ Einagbe: On = TRUE, wenn der Filter angeschaltet werden soll, FALSE fr   }
{               Ausschalten                                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetOutputFilter(On : Boolean);

{===========================================================================}
{ Funktion OutputFilterOn: Liefert TRUE zurck, wenn der Output-Filter an   }
{                          ist, sonst FALSE.                                }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION OutputFilterOn : Boolean;

{===========================================================================}
{ Prozedur SetStereo: Setzt den Stereo- bzw- Mono-Modus.                    }
{===========================================================================}
{ Eingabe: On = TRUE fr Stereo, FALSE fr Mono                             }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetStereo(On : Boolean);

{===========================================================================}
{ Funktion StereoOn: Gibt TRUE zurck, wenn der Stereo-Modus aktiv ist und  }
{                    FALSE, wenn der Mono-Modus eingeschaltet ist.          }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION StereoOn : Boolean;

{===========================================================================}
{ Prozedur SetTrebleLevel: Setzt den Parameter fr die Hhen-Regelung (nur  }
{                          Sound Blaster 16 bzw. Audio Blaster Pro 4).      }
{===========================================================================}
{ Eingabe: Side  = linker bzw. rechter Stereo-Kanal (Left_, Right_)         }
{          Param = Hhenparameter (0-255)                                   }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetTrebleLevel(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetTrebleLevel: Gibt den Wert fr die Hhen-Regelung zurck (nur }
{                          Sound Blaster 16 bzw. Audio Blaster Pro 4).      }
{===========================================================================}
{ Eingabe: Side = linker bzw. rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: Hhenparameter                                                   }
{---------------------------------------------------------------------------}

FUNCTION GetTrebleLevel(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetBassLevel: Setzt den Parameter fr die Bass-Regelung (nur SB  }
{                        16 bzw. Audio Blaster Pro 4).                      }
{===========================================================================}
{ Eingabe: Side  = linker oder rechter Stereo-Kanal (Left_, Right_)         }
{          Param = Bass-Parameter (0-255)                                   }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetBassLevel(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetBassLevel: Liefert den Parameter fr die Bass-Regelung zu-    }
{                        rck. Die Option ist nur bei Sound Blaster 16 ASP  }
{                        bzw. hher verfgbar.                              }
{===========================================================================}
{ Eingabe: Side = linker bzw. rechter Stereo-Kanal (Left_, Right_)          }
{ Ausgabe: Bass-Parameter                                                   }
{---------------------------------------------------------------------------}

FUNCTION GetBassLevel(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetInputGain: Setzt den Parameter fr die Eingangsverstrkung    }
{                        eines Kanals. Die Option ist nur bei Sound Blaster }
{                        16 ASP bzw. hher verfgbar.                       }
{===========================================================================}
{ Eingabe: Side  = linker bzw. rechter Stereo-Kanal (Left_, Right_)         }
{          Param = Parameter fr die Verstrkung (255 bedeutet jedoch nicht }
{                  die 255-fache, sondern 8-fache Verstrkung)              }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetInputGain(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetInputGain: Liefert den Wert fr die Eingangsverstrkung des   }
{                        angegebenen Kanals zurck (nur SB 16 ASP bzw. ver- }
{                        gleichbare Karte).                                 }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Kanal (Left_, Right_)                 }
{ Ausgabe: Verstrkungsparameter                                            }
{---------------------------------------------------------------------------}

FUNCTION GetInputGain(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetOutputGain: Setzt den Parameter fr die Ausgangsverstrkung   }
{                         des angegebenen Kanals (nur SB 16 ASP oder hnli- }
{                         che Karte).                                       }
{===========================================================================}
{ Eingabe: Side  = linker bzw. rechter Stereo-Kanal (Left_, Right_)         }
{          Param = Parameter fr die Verstrkung (255 bedeutet jedoch nicht }
{                  die 255-fache, sondern 8-fache Verstrkung)              }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetOutputGain(Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetOutputGain: Gibt den Parameter fr die Ausgangsverstrkung    }
{                         eines Kanals zurck (nur SB 16 oder vergleichbare }
{                         Soundkarte).                                      }
{===========================================================================}
{ Eingabe: Side = linker oder rechter Kanal (Left_, Right_)                 }
{ Ausgabe: Verstrkungsparameter                                            }
{---------------------------------------------------------------------------}

FUNCTION GetOutputGain(Side : TSides) : Byte;

IMPLEMENTATION

VAR

  {-------------------------------------------------------------------------}
  { Portadressen des Mischer-Chips                                          }
  {-------------------------------------------------------------------------}

  Mixer_Addr : Word;               { Register-Portadresse des Mischer-Chips }
  Mixer_Data : Word;                  { Daten-Portadresse des Mischer-Chips }

PROCEDURE InitSBMixer;

VAR
  Tmp : Byte;

BEGIN
  InitSBDrv;

  { Mischer-Portadressen setzen }

  Mixer_Addr := SB_BaseAddress+$04;
  Mixer_Data := SB_BaseAddress+$05;

  { Mischer-Chip testen und Typ des Sound Blaster feststellen }

  Tmp := SBMixerIn($30);
  SBMixerOut($30, 80);
  IF SBMixerIn($30) = 80 THEN
    BEGIN
      SBMixerOut($30, Tmp);
      SBBoardType := type_SB16;      { Sound Blaster 16 oder hnliche Karte }
    END
  ELSE
    BEGIN
      Tmp := SBMixerIn($04);
      SBMixerOut($04, 81);
      IF SBMixerIn($04) = 81 THEN
        BEGIN
          SBMixerOut($04, Tmp);
          SBBoardType := type_SBPro; { Sound Blaster Pro (Version 2 oder 3) }
        END
      ELSE         { Sound Blaster Pro oder Sound Blaster 16 nicht gefunden }
        BEGIN
          WriteLn;
          WriteLn('Fehler: Sound Blaster Pro oder Sound Blaster 16 nicht '+
            'vorhanden.');
          Halt;
        END;
    END;
END;

FUNCTION SBMixerIn(Reg : Byte) : Byte;

BEGIN
  Port[Mixer_Addr] := Reg;   { Nummer des Register zum Adreregister senden }
  SBMixerIn := Port[Mixer_Data];  { und Inhalt des Daten-Registers auslesen }
END;

PROCEDURE SBMixerOut(Reg, Data : Byte);

BEGIN
  Port[Mixer_Addr] := Reg;         { Nummer des Registers zum Adreregister }
  Port[Mixer_Data] := Data;        { und Datenbyte zum Datenregister senden }
END;

PROCEDURE MixerReset;

BEGIN
  SBMixerOut(0, 0);
END;

PROCEDURE SetMasterVolume(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN         { SB 16 oder Audio Blaster Pro 4? }
    IF Side = Left_ THEN                         { ja, linker Stereo-Kanal? }
      SBMixerOut($30, Param)                                           { ja }
    ELSE                                                             { nein }
      SBMixerOut($31, Param)                         { rechter Stereo-Kanal }
  ELSE                                         { Sound Blaster Pro 2 oder 3 }
    IF Side = Left_ THEN
      SBMixerOut($22, (SBMixerIn($22) AND 15) OR ((Param AND 15) SHL 4))
    ELSE
      SBMixerOut($22, (SBMixerIn($22) AND 240) OR (Param AND 15));
END;

FUNCTION GetMasterVolume(Side : TSides) : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN         { SB 16 oder Audio Blaster Pro 4? }
    IF Side = Left_ THEN                         { ja, linker Stereo-Kanal? }
      GetMasterVolume := SBMixerIn($30)                                { ja }
    ELSE                                                             { nein }
      GetMasterVolume := SBMixerIn($31)                     { rechter Kanal }
  ELSE                                         { Sound Blaster Pro 2 oder 3 }
    IF Side = Left_ THEN
      GetMasterVolume := (SBMixerIn($22) AND 15)
    ELSE
      GetMasterVolume := (SBMixerIn($22) AND 240) SHR 4;
END;

PROCEDURE SetVoiceVolume(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      SBMixerOut($32, Param)
    ELSE
      SBMixerOut($33, Param)
  ELSE
    IF Side = Left_ THEN
      SBMixerOut($04, (SBMixerIn($04) AND 15) OR ((Param AND 15) SHL 4))
    ELSE
      SBMixerOut($04, (SBMixerIn($04) AND 240) OR (Param AND 15));
END;

FUNCTION GetVoiceVolume(Side : TSides) : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      GetVoiceVolume := SBMixerIn($32)
    ELSE
      GetVoiceVolume := SBMixerIn($33)
  ELSE
    IF Side = Left_ THEN
      GetVoiceVolume := (SBMixerIn($04) AND 15)
    ELSE
      GetVoiceVolume := (SBMixerIn($04) AND 240) SHR 4;
END;

PROCEDURE SetFMVolume(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      SBMixerOut($34, Param)
    ELSE
      SBMixerOut($35, Param)
  ELSE
    IF Side = Left_ THEN
      SBMixerOut($26, (SBMixerIn($26) AND 15) OR ((Param AND 15) SHL 4))
    ELSE
      SBMixerOut($26, (SBMixerIn($26) AND 240) OR (Param AND 15));
END;

FUNCTION GetFMVolume(Side : TSides) : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      GetFMVolume := SBMixerIn($34)
    ELSE
      GetFMVolume := SBMixerIn($35)
  ELSE
    IF Side = Left_ THEN
      GetFMVolume := (SBMixerIn($26) AND 15)
    ELSE
      GetFMVolume := (SBMixerIn($26) AND 240) SHR 4;
END;

PROCEDURE SetCDVolume(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      SBMixerOut($36, Param)
    ELSE
      SBMixerOut($37, Param)
  ELSE
    IF Side = Left_ THEN
      SBMixerOut($28, (SBMixerIn($28) AND 15) OR ((Param AND 15) SHL 4))
    ELSE
      SBMixerOut($28, (SBMixerIn($28) AND 240) OR (Param AND 15));
END;

FUNCTION GetCDVolume(Side : TSides) : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      GetCDVolume := SBMixerIn($36)
    ELSE
      GetCDVolume := SBMixerIn($37)
  ELSE
    IF Side = Left_ THEN
      GetCDVolume := (SBMixerIn($28) AND 15)
    ELSE
      GetCDVolume := (SBMixerIn($28) AND 240) SHR 4;
END;

PROCEDURE SetLineVolume(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      SBMixerOut($38, Param)
    ELSE
      SBMixerOut($39, Param)
  ELSE
    IF Side = Left_ THEN
      SBMixerOut($2E, (SBMixerIn($2E) AND 15) OR ((Param AND 15) SHL 4))
    ELSE
      SBMixerOut($2E, (SBMixerIn($2E) AND 240) OR (Param AND 15));
END;

FUNCTION GetLineVolume(Side : TSides) : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      GetLineVolume := SBMixerIn($38)
    ELSE
      GetLineVolume := SBMixerIn($39)
  ELSE
    IF Side = Left_ THEN
      GetLineVolume := (SBMixerIn($2E) AND 15)
    ELSE
      GetLineVolume := (SBMixerIn($2E) AND 240) SHR 4;
END;

PROCEDURE SetMicVolume(Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN         { SB 16 oder Audio Blaster Pro 4? }
    SBMixerOut($3A, Param)                                             { ja }
  ELSE                                              { nein, SB Pro 2 oder 3 }
    SBMixerOut($0A, (SBMixerIn($0A) AND 248) OR (Param AND 7));
END;

FUNCTION GetMicVolume : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN
    GetMicVolume := SBMixerIn($3A)
  ELSE
    GetMicVolume := (SBMixerIn($0A) AND 7);
END;

PROCEDURE SetPCSpkVolume(Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    SBMixerOut($3B, Param);
END;

FUNCTION GetPCSpkVolume : Byte;

BEGIN
  IF SBBoardType = type_SB16 THEN
    GetPCSpkVolume := SBMixerIn($3B)
  ELSE
    GetPCSpkVolume := 0;
END;

PROCEDURE SetInputFilter(WhatType : TInputFilters);

BEGIN
  CASE WhatType OF
    NoFilter :                                                { kein Filter }
      SBMixerOut($0C, SBMixerIn($0C) OR 32);
    LowFilter :                                       { Filter fr Tieftne }
      SBMixerOut($0C, SBMixerIn($0C) AND NOT 40);
    HighFilter :                                      { Filter fr Hochtne }
      BEGIN
        SBMixerOut($0C, SBMixerIn($0C) AND NOT 32);
        SBMixerOut($0C, SBMixerIn($0C) OR 8);
      END;
  END;
END;

FUNCTION LowInputFilterOn : Boolean;

BEGIN
  LowInputFilterOn := (SBMixerIn($0C) AND 40) = 0;
END;

FUNCTION HighInputFilterOn : Boolean;

BEGIN
  HighInputFilterOn := ((SBMixerIn($0C) AND 32) = 0)
    AND ((SBMixerIn($0C) AND 8) = 8);
END;

PROCEDURE SetOutputFilter(On : Boolean);

BEGIN
  IF On THEN                                           { Filter aktivieren? }
    SBMixerOut($0E, SBMixerIn($0E) AND NOT 32)                         { ja }
  ELSE                                                               { nein }
    SBMixerOut($0E, SBMixerIn($0E) OR 32);             { Filter ausschalten }
END;

FUNCTION OutputFilterOn : Boolean;

BEGIN
  OutputFilterOn := (SBMixerIn($0E) AND 32) = 0;
END;

PROCEDURE SetStereo(On : Boolean);

BEGIN
  IF On THEN                                                      { Stereo? }
    SBMixerOut($0E, SBMixerIn($0E) OR 2)                               { ja }
  ELSE                                                         { nein, Mono }
    SBMixerOut($0E, SBMixerIn($0E) AND NOT 2);
END;

FUNCTION StereoOn : Boolean;

BEGIN
  StereoOn := (SBMixerIn($0E) AND 2) = 2;
END;

PROCEDURE SetTrebleLevel(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN     { SB 16 ASP bzw. Audio Blaster Pro 4? }
    IF Side = Left_ THEN                                { ja, linker Kanal? }
      SBMixerOut($44, Param)  { ja, Parameter fr die Hhen-Regelung setzen }
    ELSE                                              { nein, rechter Kanal }
      SBMixerOut($45, Param);                            { Parameter setzen }
END;

FUNCTION GetTrebleLevel(Side : TSides) : Byte;

BEGIN
  GetTrebleLevel := 0;
  IF SBBoardType = type_SB16 THEN     { SB 16 ASP bzw. Audio Blaster Pro 4? }
    IF Side = Left_ THEN                         { ja, linker Stereo-Kanal? }
      GetTrebleLevel := SBMixerIn($44)                                 { ja }
    ELSE                                              { nein, rechter Kanal }
      GetTrebleLevel := SBMixerIn($45);
END;

PROCEDURE SetBassLevel(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      SBMixerOut($46, Param)
    ELSE
      SBMixerOut($47, Param);
END;

FUNCTION GetBassLevel(Side : TSides) : Byte;

BEGIN
  GetBassLevel := 0;
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      GetBassLevel := SBMixerIn($46)
    ELSE
      GetBassLevel := SBMixerIn($47);
END;

PROCEDURE SetInputGain(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN     { SB 16 ASP oder vergleichbare Karte? }
    IF Side = Left_ THEN                                { ja, linker Kanal? }
      SBMixerOut($3F, Param)             { ja, Verstrkungsparameter setzen }
    ELSE                                              { nein, rechter Kanal }
      SBMixerOut($40, Param);                            { Parameter setzen }
END;

FUNCTION GetInputGain(Side : TSides) : Byte;

BEGIN
  GetInputGain := 0;
  IF SBBoardType = type_SB16 THEN          { SB 16 ASP oder hnliche Karte? }
    IF Side = Left_ THEN                         { ja, linker Stereo-Kanal? }
      GetInputGain := SBMixerIn($3F)                                   { ja }
    ELSE                                              { nein, rechter Kanal }
      GetInputGain := SBMixerIn($40);
END;

PROCEDURE SetOutputGain(Side : TSides; Param : Byte);

BEGIN
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      SBMixerOut($41, Param)
    ELSE
      SBMixerOut($42, Param);
END;

FUNCTION GetOutputGain(Side : TSides) : Byte;

BEGIN
  GetOutputGain := 0;
  IF SBBoardType = type_SB16 THEN
    IF Side = Left_ THEN
      GetOutputGain := SBMixerIn($41)
    ELSE
      GetOutputGain := SBMixerIn($42);
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

BEGIN
  InitSBMixer;
END.
