/*
******************************************************************
******************************************************************
******************************************************************
***                    Copyright (c) 1996                      ***
***               Rockwell Semiconductor Systems               ***
***            Multimedia Communications Division              ***
***                     All Rights Reserved                    ***
***                                                            ***
***                 CONFIDENTIAL -- No Dissemination or        ***
***              use without prior written permission.         ***
***                                                            ***
******************************************************************
******************************************************************
 
 MODULE NAME:    RACPI.H
 
 AUTHOR:    K.R. Snyder 


 HISTORY:     Major Revision                   Date       By
          -----------------------------      --------    -----
          Created from                       06/14/96     KRS



  DESCRIPTION:

                This module contains C Language include
                defs for the Boca SE336 Rockwell Audio Capture/Playback
                Interface (RACPI)
        

  NOTES:        None.
******************************************************************
******************************************************************
******************************************************************
*/

//Boca SE336 VxD Name for use in GetDDB call
//NOTE: This is case dependent
#define STR_RWA95VXD  "rwa95vxd"


//The private DDB message for the Client VxD to send to RWA95VXD
#define RACPI_RegisterAudioClient BEGIN_RESERVED_PRIVATE_SYSTEM_CONTROL


#define RACPI_ERROR  (DWORD) NULL           //error return
#define RACPI_OK     (DWORD) 1              //valid return


//Structure that RACPI_GetInfo fills in. Note, the DMA buffers will always be
//a multiple of 4Kbytes.  Default for current Boca SE336 is 64K bytes each.  Either
//8 or 16-bit audio data will start at the playback and capture buffers' "start" addresses.
//But for 8-bit samples, the "End" addresses point to the last slot for a sample, whereas for 16-bit
//samples, the "End" addresses point to the second byte of the last slot for a sample.  Thus
//for 16-bit samples, "dwPbBufEndAddress-1" and "dwcapBufEndAddress-1"should be used by the Client as the
//buffer wrap points.

typedef struct
{
  DWORD  dwPbBufStartAddress;          //address of first byte of the DMA Playback buffer
  DWORD  dwPbBufEndAddress;            //address of last byte of the DMA playback buffer
  DWORD  dwCapBufStartAddress;         //address of first byte of the DMA capture buffer
  DWORD  dwCapBufEndAddress;           //address of last byte of the DMA capture buffer
} RACPIINFO, *PRACPIINFO;



//
//  RACPI Function table structure
//

typedef struct
{
    
    //VxD version (bits 03=minor, bits 4-7 = major, bits 8-31 unused
    //eg: return value of 0x00000010 would be current number for 1.0. 
    DWORD (__stdcall *RACPI_GetVersion) ();    

    //Get DMA buffer addresses. Returns 0 for bad.  Structure assumed to be
    //as defined above (RACPIINFO) and filled in for use by client.
    DWORD (__stdcall *RACPI_GetInfo)( PRACPIINFO pRACPIInfo );

    //Begin DMA Input from Boca SE336 to PC DMA buffer.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_StartCapture)();
    
    //Begin DMA Output from PC to Boca SE336.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_StartPlayback)();
    
    //Stop DMA Input from Boca SE336 to PC DMA buffer.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_StopCapture)();
    
    //Stop DMA Output to Boca SE336.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_StopPlayback)();
    
    //Pause DMA Input from Boca SE336 to PC DMA buffer.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_PauseCapture)();
    
    //Pause DMA Output to Boca SE336.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_PausePlayback)();

    //Resume DMA Input from Boca SE336 to PC DMA buffer.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_ResumeCapture)();
    
    //Resume DMA Output to Boca SE336.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_ResumePlayback)();

    
    //Set sample rate and data format for input DMA.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_SetCaptureFmtFreq)
                         ( 
                           DWORD dwDataFmt,        //0=signed 8, 1=Signed 16, 2=Unsigned 8
                           DWORD dwNumChannels, //1 or 2
                           DWORD dwSampleRate    //4000 to 44100
                         );

    
    //Set sample rate and data format for output DMA.
    //Non Null return means OK
    DWORD (__stdcall *RACPI_SetPlaybackFmtFreq)
                         ( 
                           DWORD dwDataFmt,     //0=signed 8, 1=Signed 16, 2=Unsigned 8
                           DWORD dwNumChannels, //1 or 2
                           DWORD dwSampleRate   //4000 to 44100
                         );


    
    //Get current DMA Input position
    //Non Null return means OK
    DWORD (__stdcall *RACPI_GetCapPosition)(PDWORD pDMAInPosition);

    
    //Get current DMA output position
    //Non Null return means OK
    DWORD (__stdcall *RACPI_GetPlayPosition)(PDWORD pDMAOutPosition);
       
       
    //Quit
    //Non Null return means OK
    DWORD (__stdcall *RACPI_Terminate)();




} RACPIFUNCTBL, *PRACPIFUNCTBL;



/*


		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////
         ////////  RACPI  SAMPLE USAGE CODE   ///////////////////////
		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////
		 {

		  struct VxD_Desc_Block *RwaDDB;
		  PRACPIFUNCTBL FuncTbl;
		  RACPIINFO DmaPtrs;
		  DWORD dwCapCur, dwPlayCur, dwCapPos, dwT, dwLastCapPos, dwNextPlayPos;
		  DWORD i;

          //Here is an example of how to access the RACPI interface.  Note, this example was
		  //put together in haste, and thus is NOT optimized nor beautiful.
          
          //Using the name of the Boca SE336 VxD, get the address of its DDB.  This will let us
		  //send it a message stating we wish to use RACPI
          RwaDDB = Get_DDB(0,STR_RWA95VXD);


          //register with Boca SE336 RACPI and save table address in FuncTbl.  A zero return here indicates
		  //failure.  Otherwise, the RACPI entry points are subsequently available as ptrs to funcs.
          if (FuncTbl = (PRACPIFUNCTBL) Directed_Sys_Control(RwaDDB, RACPI_RegisterAudioClient,0, 0, 0,0))
		  {
		   //version is currently 0x00000010,  check this
		   if(FuncTbl -> RACPI_GetVersion() == (DWORD)0x10)
		   {
		     //ask and check that the DMA ptrs are properly filled into DmaPtrs
		     if(FuncTbl -> RACPI_GetInfo( &DmaPtrs))
			 {
			    //setup 8bit, unsigned, 16Khz, mono for both capture and playback.  These numbers are taken
				//from the hat.  You can use whatever you like.  But Boca SE336 cannot currently handle bidirectional
				//16-bit stereo at 44.1Khz.  It can come close.
			    FuncTbl -> RACPI_SetCaptureFmtFreq(2, 1, 16000);
			    FuncTbl -> RACPI_SetPlaybackFmtFreq(2, 1, 16000);
				
				//start capture.  For my test, I am using a microphone here and I speak nonsense into it for about
				//17 seconds.
				FuncTbl -> RACPI_StartCapture();

				//delay 2 seconds while gathering data in the capture buffer
				Wait(2000);


				//get current capture position
				//and use it as place to copy up to
				FuncTbl -> RACPI_GetCapPosition(&dwCapPos);
				
				//lazy man's memcopy, copy the 2 seconds worth of initial data
				for( dwCapCur=DmaPtrs.dwCapBufStartAddress, dwPlayCur=DmaPtrs.dwPbBufStartAddress;
				     dwCapCur < dwCapPos; 
				     dwCapCur++, dwPlayCur++ )
				{
				  * (PBYTE) dwPlayCur =  * (PBYTE) dwCapCur;
				}

				//start playback of what we captured, we are now hearing a delayed playback of what
				//we recorded
				FuncTbl -> RACPI_StartPlayback();
				

			   	//Now keep it going.  Every 50ms, copy data from the last capture position to the current one, into the
				//the playback buffer.  If we do this 300 times, it is about 15 seconds of delayed audio output.
			   	for (i=0; i< 300; i++)
				{
			   	  //delay 50 mseconds
				  Wait(50);

				  //remember where we left off before we see where input DMA is now.
				  dwLastCapPos = dwCapPos;
                  dwNextPlayPos = (dwPlayCur == DmaPtrs.dwPbBufEndAddress)? DmaPtrs.dwPbBufStartAddress : dwPlayCur++;
							
				  //see where 50ms of waiting has brought the capture position.
				  FuncTbl -> RACPI_GetCapPosition(&dwCapPos);

				  //lazy man's memcopy of data from previous capture position to current capture position into playback
				  //buffer.  Take care of wrap points too.
				  for( dwCapCur=dwLastCapPos, dwPlayCur=dwNextPlayPos;
				     dwCapCur != ( (dwCapPos == DmaPtrs.dwCapBufStartAddress)?DmaPtrs.dwCapBufEndAddress :(dwCapPos -1)); 
				     dwCapCur++, dwPlayCur++ )
				    {
				      if (dwCapCur > DmaPtrs.dwCapBufEndAddress) dwCapCur = DmaPtrs.dwCapBufStartAddress;
				      if (dwPlayCur > DmaPtrs.dwPbBufEndAddress) dwPlayCur = DmaPtrs.dwPbBufStartAddress;
				      * (PBYTE) dwPlayCur =  * (PBYTE) dwCapCur;
				    }

				}

				
				//stop recording after the 15 seconds. Note, playback is still looping what's in
				//the DMA buffer
				FuncTbl -> RACPI_StopCapture();

				
				//frequency test.  While playing back, increase sample rate 5Hz every 10 ms through the whole
				//range.
				for (i=4000; i <= 44100; i +=5)
				{
			   	  //delay 10 mseconds
				  Wait(10);
				  FuncTbl -> RACPI_SetPlaybackFmtFreq(2, 1, i);

				}
				
				//frequency and pause/resume test.  Every half second, pause playback for a half second and reduce the
				//frequency by 700hz through the whole range.
				for (i=44100; i>4000; i -=700)
				{
				  FuncTbl -> RACPI_PausePlayback();
				  Wait(500);
				  FuncTbl -> RACPI_ResumePlayback();
				  
				  FuncTbl -> RACPI_SetPlaybackFmtFreq(2, 1, i);
				  
				  Wait(500);
				}

				//Done.  This cleans up.
				FuncTbl -> RACPI_Terminate();

			 }
			}
		   }
	      }
		 
		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////
		 ////////////////////////////////////////////////////////////



*/
