#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include "stm32f469xx.h"
#include "lz.h"


#define fatal(...)		do { while(1) {} } while(0)


struct _MPU_Type
{
	volatile uint32_t TYPE;                         /*!< Offset: 0x00  MPU Type Register                              */
	volatile uint32_t CTRL;                         /*!< Offset: 0x04  MPU Control Register                           */
	volatile uint32_t RNR;                          /*!< Offset: 0x08  MPU Region RNRber Register                     */
	volatile uint32_t RBAR;                         /*!< Offset: 0x0C  MPU Region Base Address Register               */
	volatile uint32_t RASR;                         /*!< Offset: 0x10  MPU Region Attribute and Size Register         */
};

#define _MPU               (( struct _MPU_Type*)0xE000ED90)


static void entry(void);
static void hwrInit(void);

__attribute__ ((aligned (512), section(".vec"), used)) void (*__ISR_VECTORS[]) (void) =
{
	(void*)0x20000400,
	entry,
	(void*)0x11111111,		//destination address
	(void*)0x22222222,		//compressed length
	(void*)0x33333333,		//compressed data start
};

static void mpuInit(void)
{
	uint32_t i, nreg;
	
	if (_MPU->TYPE & 1) {
		//Separate I/D MPU not supported
		while(1);
	}
	
	nreg = (_MPU->TYPE & 0xff00) >> 8;
	for (i = 0; i < nreg; i++) {
		
		_MPU->RNR = i;
		_MPU->RASR = 0;	//disable this region for now
		_MPU->RBAR = 0;
		_MPU->RASR = 0x030f003f;	//all perms, on, 4GB
	}
	_MPU->CTRL = 7;	//enable mpu
}

static void entry(void)
{
	const uint32_t *vecs = (const uint32_t*)(((uint32_t)&entry) & 0xffff0000);
	const uint8_t *src = ((uint8_t*)(vecs[4])) + (uint32_t)vecs;
	uint8_t *dst = (uint8_t*)(vecs[2]);
	uint32_t comprSz = vecs[3];
	
	mpuInit();
	hwrInit();
	lzUncompress(src, dst, comprSz);
	
	asm volatile(
		"mov sp, %0			\n\t"
		"bx  %1				\n\t"
		::"r"(((uint32_t*)dst)[0]), "r"(((uint32_t*)dst)[1])
		:
	);
}




static void clockTreeInitStm32f429(void)
{
	//first go to safe settings: HSI
	RCC->CR = RCC_CR_HSION;													//HSI on, PLL off
	RCC->CFGR = (RCC->CFGR &~ RCC_CFGR_SW_Msk) | RCC_CFGR_SW_HSI;			//switch to HSI
	
	//turn on HSI and wait for it
	RCC->CR |= RCC_CR_HSEON;
	while (!(RCC->CR & RCC_CR_HSERDY));
	
	//HSE / 4 as source (2MHz), VCO outputs 384MHZ, P output (main clock) is 192MHz, Q output (usb&SDIO is 48MHz)
	RCC->PLLCFGR = (192 << RCC_PLLCFGR_PLLN_Pos) | (8 << RCC_PLLCFGR_PLLQ_Pos) | (4 << RCC_PLLCFGR_PLLM_Pos) | RCC_PLLCFGR_PLLSRC_HSE;
	
	//we need PLLSAI for LCD. CFG: HSE / 4 as source (2MHz), VCO outputs 192MHZ, SAI1 clock min speed, SAIDLCD clock = VCO/4 = 48MHz
	RCC->PLLSAICFGR = (96 << RCC_PLLSAICFGR_PLLSAIN_Pos) | (15 << RCC_PLLSAICFGR_PLLSAIQ_Pos) | (4 << RCC_PLLSAICFGR_PLLSAIR_Pos);
	
	//set up lcd clock
	RCC->DCKCFGR = 2 << RCC_DCKCFGR_PLLSAIDIVR_Pos;	//LCD = SAILCD / 8 = 6MHz
	
	//turn on the main PLL and wait for it
	RCC->CR |= RCC_CR_PLLON;
	while (!(RCC->CR & RCC_CR_PLLRDY));
	
	//turnon SAI PLL & wait for it
	RCC->CR |= RCC_CR_PLLSAION;
	while (!(RCC->CR & RCC_CR_PLLSAIRDY));
	
	//setup flash wait states and optimizations
	FLASH->ACR = FLASH_ACR_LATENCY_5WS | FLASH_ACR_PRFTEN | FLASH_ACR_ICEN | FLASH_ACR_DCEN | FLASH_ACR_ICRST | FLASH_ACR_DCRST;
	
	//set up other divisors (RTC = HSE/8 = 1MHz, APB2 = sysclk/2 = 96MZ, APB1 = sysclk/4 = 48MZ, AHB = sysclk = 192MHz) and go to it
	RCC->CFGR = RCC_CFGR_I2SSRC | (8 << RCC_CFGR_RTCPRE_Pos) | RCC_CFGR_PPRE2_DIV2 | RCC_CFGR_PPRE1_DIV4 | RCC_CFGR_SW_PLL;
	while (((RCC->CFGR & RCC_CFGR_SW_Msk) >> RCC_CFGR_SW_Pos) != ((RCC->CFGR & RCC_CFGR_SWS_Msk) >> RCC_CFGR_SWS_Pos));
	
	//enable periphs we care about
	RCC->AHB1ENR = RCC_AHB1ENR_CCMDATARAMEN | RCC_AHB1ENR_GPIOAEN | RCC_AHB1ENR_GPIOBEN | RCC_AHB1ENR_GPIOCEN | RCC_AHB1ENR_GPIODEN | RCC_AHB1ENR_GPIOEEN | RCC_AHB1ENR_GPIOFEN | RCC_AHB1ENR_GPIOGEN;
	RCC->AHB3ENR = RCC_AHB3ENR_FMCEN;
	RCC->APB2ENR = RCC_APB2ENR_SYSCFGEN;
	RCC->APB1ENR = 0;
}

static void gpiosInitStm32f429(void)
{
	//SDRAM = AF12,push-pull, high speed, no pullup
	
	/*-- GPIOs Configuration -----------------------------------------------------*/
	/*
	 +-------------------+--------------------+--------------------+--------------------+
	 +                       SDRAM pins assignment                                      +
	 +-------------------+--------------------+--------------------+--------------------+
	 | PD0  <-> FMC_D2   | PE0  <-> FMC_NBL0  | PF0  <-> FMC_A0    | PG0  <-> FMC_A10   |
	 | PD1  <-> FMC_D3   | PE1  <-> FMC_NBL1  | PF1  <-> FMC_A1    | PG1  <-> FMC_A11   |
	 | PD8  <-> FMC_D13  | PE7  <-> FMC_D4    | PF2  <-> FMC_A2    | PG8  <-> FMC_SDCLK |
	 | PD9  <-> FMC_D14  | PE8  <-> FMC_D5    | PF3  <-> FMC_A3    | PG15 <-> FMC_NCAS  |
	 | PD10 <-> FMC_D15  | PE9  <-> FMC_D6    | PF4  <-> FMC_A4    |--------------------+ 
	 | PD14 <-> FMC_D0   | PE10 <-> FMC_D7    | PF5  <-> FMC_A5    |   
	 | PD15 <-> FMC_D1   | PE11 <-> FMC_D8    | PF11 <-> FMC_NRAS  | 
	 +-------------------| PE12 <-> FMC_D9    | PF12 <-> FMC_A6    | 
	                     | PE13 <-> FMC_D10   | PF13 <-> FMC_A7    |    
	                     | PE14 <-> FMC_D11   | PF14 <-> FMC_A8    |
	                     | PE15 <-> FMC_D12   | PF15 <-> FMC_A9    |
	 +-------------------+--------------------+--------------------+
	 | PB5 <-> FMC_SDCKE1| 
	 | PB6 <-> FMC_SDNE1 | 
	 | PC0 <-> FMC_SDNWE |
	 +-------------------+  

	
		SWD:	AF0, very high speed
			A13 - SWDIO (pullup)
			A14 - SWCLK (pulldown)


	*/

	
	GPIOA->MODER	= 0x28000000;	//A13,A14 - SWD
	GPIOA->OTYPER	= 0x00000000;
	GPIOA->OSPEEDR	= 0x3c000000;
	GPIOA->PUPDR	= 0x24000000;
	GPIOA->AFR[0]	= 0x00000000;
	GPIOA->AFR[1]	= 0x00000000;
	
	GPIOB->MODER	= 0x00002800;	//B5,B6 - SDRAM
	GPIOB->OTYPER	= 0x00000000;
	GPIOB->OSPEEDR	= 0x00002800;
	GPIOB->PUPDR	= 0x00000000;
	GPIOB->AFR[0]	= 0x0CC00000;
	GPIOB->AFR[1]	= 0x00000000;
	
	GPIOC->MODER	= 0x00000002;	//C0 - SDRAM
	GPIOC->OTYPER	= 0x00000000;
	GPIOC->OSPEEDR	= 0x00000002;
	GPIOC->PUPDR	= 0x00000000;
	GPIOC->AFR[0]	= 0x0000000C;
	GPIOC->AFR[1]	= 0x00000000;
	
	GPIOD->MODER	= 0xA02A000A;	//D0,D1,D8,D9,D10,D14,D15 - SDRAM
	GPIOD->OTYPER	= 0x00000000;
	GPIOD->OSPEEDR	= 0xA02A000A;
	GPIOD->PUPDR	= 0x00000000;
	GPIOD->AFR[0]	= 0x000000CC;
	GPIOD->AFR[1]	= 0xCC000CCC;
	
	GPIOE->MODER	= 0xAAAA800A;	//E0,E1,E7,E8,E9,E10,E11,E12,E13,E14,E15 - SDRAM
	GPIOE->OTYPER	= 0x00000000;
	GPIOE->OSPEEDR	= 0xAAAA800A;
	GPIOE->PUPDR	= 0x00000000;
	GPIOE->AFR[0]	= 0xC00000CC;
	GPIOE->AFR[1]	= 0xCCCCCCCC;
	
	GPIOF->MODER	= 0xAA800AAA;	//F0,F1,F2,F3,F4,F5,F11,F12,F13,F14,F15 - SDRAM
	GPIOF->OTYPER	= 0x00000000;
	GPIOF->OSPEEDR	= 0xAA800AAA;
	GPIOF->PUPDR	= 0x00000000;
	GPIOF->AFR[0]	= 0x00CCCCCC;
	GPIOF->AFR[1]	= 0xCCCCC000;
	
	GPIOG->MODER	= 0x80020A0A;	//G0,G1,G4,G5,G8,G15 = SDRAM; G6,G7,G10-G12
	GPIOG->OTYPER	= 0x00000000;
	GPIOG->OSPEEDR	= 0x80020A0A;
	GPIOG->PUPDR	= 0x00000000;
	GPIOG->AFR[0]	= 0x00CC00CC;
	GPIOG->AFR[1]	= 0xC000000C;
	
	//setup which GPIOs are used for external ints
	SYSCFG->EXTICR[0] = 0x00000000;
	SYSCFG->EXTICR[1] = 0x00000000;
	SYSCFG->EXTICR[2] = 0x00000000;
	SYSCFG->EXTICR[3] = 0x00000000;
}

static bool sdramSendCmdStm32f429(uint32_t cmdModeAndTarget, uint32_t autoRefreshNmbr, uint32_t modeRegisterDef, uint32_t timeout)
{
	uint32_t ticks;
	
	FMC_Bank5_6->SDCMR = cmdModeAndTarget | ((autoRefreshNmbr - 1) << FMC_SDCMR_NRFS_Pos) | (modeRegisterDef << FMC_SDCMR_MRD_Pos);
	
	for (ticks = 0; ticks < timeout; ticks++) {
		if (FMC_Bank5_6->SDSR & FMC_SDSR_BUSY)
			continue;
		return true;
	}
	
	return false;
}

static void sdramInitStm32f429(void)
{
	volatile uint32_t delay;
	uint32_t t;
	
	#define FMC_SDRAM_CMD_CLK_ENABLE				1
	#define FMC_SDRAM_CMD_PALL						2
	#define FMC_SDRAM_CMD_AUTOREFRESH_MODE			3
	#define FMC_SDRAM_CMD_LOAD_MODE					4
	
	#define SDRAM_MODEREG_BURST_LENGTH_2			1
	#define SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL		0
	#define SDRAM_MODEREG_CAS_LATENCY_3				0x30
	#define SDRAM_MODEREG_OPERATING_MODE_STANDARD	0
	#define SDRAM_MODEREG_WRITEBURST_MODE_SINGLE	0x200
	
	#define FMC_SDRAM_CLOCK_PERIOD_2				0x00000800
	#define FMC_SDRAM_RBURST_CFG					FMC_SDCR1_RBURST
	#define FMC_SDRAM_RPIPE_DELAY_1					0x00002000
	#define FMC_SDRAM_COLUMN_BITS_NUM_8				0x00000000
	#define FMC_SDRAM_ROW_BITS_NUM_12				0x00000004
	#define FMC_SDRAM_INTERN_BANKS_NUM_4			0x00000040
	#define FMC_SDRAM_CAS_LATENCY_3					0x00000180
	#define FMC_SDRAM_WRITE_PROTECTION_DISABLE		0x00000000
	#define FMC_SDRAM_MEM_BUS_WIDTH_16				0x00000010
	#define FMC_SDRAM_CMD_TARGET_BANK2				FMC_SDCMR_CTB2
	
	//SDRAM config
	FMC_Bank5_6->SDCR[0] = (FMC_Bank5_6->SDCR[0] &~ (FMC_SDCR1_SDCLK | FMC_SDCR1_RBURST | FMC_SDCR1_RPIPE)) | FMC_SDRAM_CLOCK_PERIOD_2 | FMC_SDRAM_RBURST_CFG | FMC_SDRAM_RPIPE_DELAY_1;;
	FMC_Bank5_6->SDCR[1] = (FMC_Bank5_6->SDCR[1] &~ (FMC_SDCR1_NC | FMC_SDCR1_NR | FMC_SDCR1_MWID| FMC_SDCR1_NB | FMC_SDCR1_CAS | FMC_SDCR1_WP | FMC_SDCR1_SDCLK | FMC_SDCR1_RBURST | FMC_SDCR1_RPIPE)) | FMC_SDRAM_COLUMN_BITS_NUM_8 | FMC_SDRAM_ROW_BITS_NUM_12 | FMC_SDRAM_MEM_BUS_WIDTH_16 | FMC_SDRAM_INTERN_BANKS_NUM_4 | FMC_SDRAM_CAS_LATENCY_3 | FMC_SDRAM_WRITE_PROTECTION_DISABLE;

	//SDRAM timing config
	FMC_Bank5_6->SDTR[0] = (FMC_Bank5_6->SDTR[0] &~ (FMC_SDTR1_TRC | FMC_SDTR1_TRP)) | (((7 /* TRC:  min=63 (6x11.90ns) -> 7*/)-1) << FMC_SDTR1_TRC_Pos) | (((2 /* TRP:  15ns => 2x11.90ns -> 2 */)-1) << FMC_SDTR1_TRP_Pos);
	FMC_Bank5_6->SDTR[1] = (FMC_Bank5_6->SDTR[1] &~ (FMC_SDTR1_TMRD | FMC_SDTR1_TXSR | FMC_SDTR1_TRAS | FMC_SDTR1_TRC | FMC_SDTR1_TWR | FMC_SDTR1_TRP | FMC_SDTR1_TRCD)) | (((2 /* TMRD - load to active delay - 2 cy */) - 1) << FMC_SDTR1_TMRD_Pos) | (((7 /* TXSR: min=70ns (6x11.90ns) -> 7 */) - 1) << FMC_SDTR1_TXSR_Pos) | (((4 /* TRAS: min=42ns (4x11.90ns) max=120k (ns) -> 4 */) - 1) << FMC_SDTR1_TRAS_Pos) | (((2 /* TWR:  2 Clock cycles -> 2 */) - 1) << FMC_SDTR1_TWR_Pos) | (((2 /* TRCD: 15ns => 2x11.90ns -> 2 */) - 1) << FMC_SDTR1_TRCD_Pos);

	//Clock configuration enable command
	if (!sdramSendCmdStm32f429(FMC_SDRAM_CMD_CLK_ENABLE | FMC_SDRAM_CMD_TARGET_BANK2, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "Cfg Ena");
	
	//100 ms delay or so
	for (delay = 1920000; delay; delay--);
	
	//PALL (precharge all) command
	if (!sdramSendCmdStm32f429(FMC_SDRAM_CMD_PALL | FMC_SDRAM_CMD_TARGET_BANK2, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "PALL");
	
	//Auto-Refresh command
	if (!sdramSendCmdStm32f429(FMC_SDRAM_CMD_AUTOREFRESH_MODE | FMC_SDRAM_CMD_TARGET_BANK2, 4, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "AutoRefresh");

	//Program external memory mode register
	if (!sdramSendCmdStm32f429(FMC_SDRAM_CMD_LOAD_MODE | FMC_SDRAM_CMD_TARGET_BANK2, 1, SDRAM_MODEREG_BURST_LENGTH_2 | SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL | SDRAM_MODEREG_CAS_LATENCY_3 | SDRAM_MODEREG_OPERATING_MODE_STANDARD | SDRAM_MODEREG_WRITEBURST_MODE_SINGLE, 0x1000))
		fatal("SDRAM init failed: %s\n", "External Memory Mode");

	//Set the refresh rate counter => (15.62 us x Freq) - 20, Set the device refresh counter */
	FMC_Bank5_6->SDRTR |= (0x0569 << FMC_SDRTR_COUNT_Pos);
	
	//for now, clear all of it excepr where rom is/will be/will have been/whatever
}

static void clockTreeInitStm32f469(void)
{
	//setup flash wait states and optimizations, reset caches
	FLASH->ACR = FLASH_ACR_LATENCY_5WS;
	FLASH->ACR |= FLASH_ACR_DCRST | FLASH_ACR_ICRST;
	FLASH->ACR &=~ (FLASH_ACR_DCRST | FLASH_ACR_ICRST);
	FLASH->ACR |= FLASH_ACR_PRFTEN | FLASH_ACR_ICEN | FLASH_ACR_DCEN;

	//first go to safe settings: HSI
	RCC->CR = RCC_CR_HSION;													//HSI on, PLL off
	RCC->CFGR = (RCC->CFGR &~ RCC_CFGR_SW_Msk) | RCC_CFGR_SW_HSI;			//switch to HSI
	
	
	//turn on HSE and wait for it
	RCC->CR |= RCC_CR_HSEON;
	while (!(RCC->CR & RCC_CR_HSERDY));
	
	//HSE / 4 as source (2MHz), VCO outputs 384MHZ, P output (main clock) is 192MHz, Q output (usb&SDIO is 48MHz)
	RCC->PLLCFGR = (192 << RCC_PLLCFGR_PLLN_Pos) | (8 << RCC_PLLCFGR_PLLQ_Pos) | (4 << RCC_PLLCFGR_PLLM_Pos) | RCC_PLLCFGR_PLLSRC_HSE;
	
	//we need PLLSAI for LCD at precisely thid freq. CFG: HSE / 4 as source (2MHz), VCO outputs 384MHZ, SAI1 clock min speed, SAIDLCD clock = VCO/7 = 54.857 MHz
	RCC->PLLSAICFGR = (192 << RCC_PLLSAICFGR_PLLSAIN_Pos) | (15 << RCC_PLLSAICFGR_PLLSAIQ_Pos) | (7 << RCC_PLLSAICFGR_PLLSAIR_Pos);
	
	//set up lcd clock
	RCC->DCKCFGR = 0 << RCC_DCKCFGR_PLLSAIDIVR_Pos;	//LCD = SAILCD / 2 = 27.429 MHz
	
	//turn on the main PLL and wait for it
	RCC->CR |= RCC_CR_PLLON;
	while (!(RCC->CR & RCC_CR_PLLRDY));
	
	//turnon SAI PLL & wait for it
	RCC->CR |= RCC_CR_PLLSAION;
	while (!(RCC->CR & RCC_CR_PLLSAIRDY));
	
	//set up other divisors (RTC = HSE/8 = 1MHz, APB2 = sysclk/2 = 96MZ, APB1 = sysclk/4 = 48MZ, AHB = sysclk = 192MHz) and go to it
	RCC->CFGR = RCC_CFGR_I2SSRC | (8 << RCC_CFGR_RTCPRE_Pos) | RCC_CFGR_PPRE2_DIV2 | RCC_CFGR_PPRE1_DIV4 | RCC_CFGR_SW_PLL;
	while (((RCC->CFGR & RCC_CFGR_SW_Msk) >> RCC_CFGR_SW_Pos) != ((RCC->CFGR & RCC_CFGR_SWS_Msk) >> RCC_CFGR_SWS_Pos));
	
	RCC->AHB1ENR = RCC_AHB1ENR_CCMDATARAMEN | RCC_AHB1ENR_GPIOBEN | RCC_AHB1ENR_GPIOCEN | RCC_AHB1ENR_GPIODEN | RCC_AHB1ENR_GPIOEEN | RCC_AHB1ENR_GPIOFEN | RCC_AHB1ENR_GPIOGEN | RCC_AHB1ENR_GPIOHEN | RCC_AHB1ENR_GPIOIEN | RCC_AHB1ENR_GPIOJEN | RCC_AHB1ENR_GPIOKEN;
	RCC->AHB3ENR = RCC_AHB3ENR_FMCEN;
	RCC->APB2ENR = RCC_APB2ENR_SYSCFGEN;
	RCC->APB1ENR = RCC_APB1ENR_PWREN;
}

static void gpiosInitStm32f469(void)
{

	/*-- GPIOs Configuration -----------------------------------------------------*/
	/*
	 SDRAM:
	 		C: 0
	 		D: 0 1 8 9 10 14 15
	 		E: 0 1 7 8 9 10 11 12 13 14 15
	 		F: 0 1 2 3 4 5 11 12 13 14 15
	 		G: 0 1 4 5 8 15
	 		H: 2 3 8 9 10 11 12 13 14 15
	 		I: 0 1 2 3 4 5 6 7 9 10
	SWD:
			A: 13 14

	*/
	
	GPIOA->MODER	= 0x28000000;	//A13,A14 - SWD
	GPIOA->OTYPER	= 0x00000000;
	GPIOA->OSPEEDR	= 0x3C000000;
	GPIOA->PUPDR	= 0x24000000;
	GPIOA->AFR[0]	= 0x00000000;
	GPIOA->AFR[1]	= 0x00000000;
	
	GPIOB->MODER	= 0x00000000;
	GPIOB->OTYPER	= 0x00000000;
	GPIOB->OSPEEDR	= 0x00000000;
	GPIOB->PUPDR	= 0x00000000;
	GPIOB->AFR[0]	= 0x00000000;
	GPIOB->AFR[1]	= 0x00000000;

	GPIOC->MODER	= 0x00000002;	//C0 - SDRAM
	GPIOC->OTYPER	= 0x00000000;
	GPIOC->OSPEEDR	= 0x00000002;
	GPIOC->PUPDR	= 0x00000001;
	GPIOC->AFR[0]	= 0x0000000C;
	GPIOC->AFR[1]	= 0x00000000;

	GPIOD->MODER	= 0xA02A000A;	//D0,D1,D8,D9,D10,D14,D15 - SDRAM
	GPIOD->OTYPER	= 0x00000000;
	GPIOD->OSPEEDR	= 0xA02A000A;
	GPIOD->PUPDR	= 0x50150005;
	GPIOD->AFR[0]	= 0x000000CC;
	GPIOD->AFR[1]	= 0xCC000CCC;

	GPIOE->MODER	= 0xAAAA800A;	//E0,E1,E7,E8,E9,E10,E11,E12,E13,E14,E15 - SDRAM
	GPIOE->OTYPER	= 0x00000000;
	GPIOE->OSPEEDR	= 0xAAAA800A;
	GPIOE->PUPDR	= 0x55554005;
	GPIOE->AFR[0]	= 0xC00000CC;
	GPIOE->AFR[1]	= 0xCCCCCCCC;

	GPIOF->MODER	= 0xAA800AAA;	//F0,F1,F2,F3,F4,F5,F11,F12,F13,F14,F15 - SDRAM
	GPIOF->OTYPER	= 0x00000000;
	GPIOF->OSPEEDR	= 0xAA800AAA;
	GPIOF->PUPDR	= 0x55400555;
	GPIOF->AFR[0]	= 0x00CCCCCC;
	GPIOF->AFR[1]	= 0xCCCCC000;
	
	GPIOG->MODER	= 0x80020A0A;	//G0,G1,G4,G5,G8,G15 = SDRAM
	GPIOG->OTYPER	= 0x00000000;
	GPIOG->OSPEEDR	= 0x80020A0A;
	GPIOG->PUPDR	= 0x40010505;
	GPIOG->AFR[0]	= 0x00CC00CC;
	GPIOG->AFR[1]	= 0xC000000C;
	
	GPIOH->MODER	= 0xAAAA00A0;	//H2,H3,H8,H9,H10,H11,H12,H13,H14,H15 = SDRAM, H7-lcd_reset
	GPIOH->OTYPER	= 0x00000080;
	GPIOH->OSPEEDR	= 0xAAAA00A0;
	GPIOH->PUPDR	= 0x55550050;
	GPIOH->AFR[0]	= 0x0000CC00;
	GPIOH->AFR[1]	= 0xCCCCCCCC;
	
	GPIOI->MODER	= 0x0028AAAA;	//I0,I1,I2,I3,I4,I5,I6,I7,I9,I10 = SDRAM
	GPIOI->OTYPER	= 0x00000000;
	GPIOI->OSPEEDR	= 0x0028AAAA;
	GPIOI->PUPDR	= 0x00145555;
	GPIOI->AFR[0]	= 0xCCCCCCCC;
	GPIOI->AFR[1]	= 0x00000CC0;
	
	GPIOJ->MODER	= 0x00000000;
	GPIOJ->OTYPER	= 0x00000000;
	GPIOJ->OSPEEDR	= 0x00000000;
	GPIOJ->PUPDR	= 0x00000000;
	GPIOJ->AFR[0]	= 0x00000000;
	GPIOJ->AFR[1]	= 0x00000000;
		
	//setup which GPIOs are used for external ints
	SYSCFG->EXTICR[0] = 0x00000000;
	SYSCFG->EXTICR[1] = 0x00000000;
	SYSCFG->EXTICR[2] = 0x00000000;
	SYSCFG->EXTICR[3] = 0x00000000;
}

static bool sdramSendCmdStm32f469(uint32_t cmdModeAndTarget, uint32_t autoRefreshNmbr, uint32_t modeRegisterDef, uint32_t timeout)
{
	uint32_t ticks;
	
	FMC_Bank5_6->SDCMR = cmdModeAndTarget | ((autoRefreshNmbr - 1) << FMC_SDCMR_NRFS_Pos) | (modeRegisterDef << FMC_SDCMR_MRD_Pos);
	
	for (ticks = 0; ticks < timeout; ticks++) {
		if (FMC_Bank5_6->SDSR & FMC_SDSR_BUSY)
			continue;
		return true;
	}
	
	return false;
}

static void sdramInitStm32f469(void)
{
	volatile uint32_t delay;
	uint32_t cfg;
	
	#define FMC_SDRAM_CMD_CLK_ENABLE				1
	#define FMC_SDRAM_CMD_PALL						2
	#define FMC_SDRAM_CMD_AUTOREFRESH_MODE			3
	#define FMC_SDRAM_CMD_LOAD_MODE					4
	
	#define SDRAM_MODEREG_BURST_LENGTH_1			0
	#define SDRAM_MODEREG_BURST_LENGTH_2			1
	#define SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL		0
	#define SDRAM_MODEREG_CAS_LATENCY_3				0x30
	#define SDRAM_MODEREG_OPERATING_MODE_STANDARD	0
	#define SDRAM_MODEREG_WRITEBURST_MODE_SINGLE	0x200
	
	
	//SDRAM config
	cfg = FMC_SDCR1_SDCLK_1 | FMC_SDCR1_RBURST | FMC_SDCR1_NB | (3 << FMC_SDCR1_CAS_Pos) | FMC_SDCR1_MWID_1 | FMC_SDCR1_NR_0;
	FMC_Bank5_6->SDCR[0] = cfg;
	FMC_Bank5_6->SDCR[1] = cfg;

	//SDRAM timing config
	cfg =	((2 /* LoadToActiveDelay*/ - 1) << FMC_SDTR1_TMRD_Pos) |
			((6 /* ExitSelfRefreshDelay */ - 1) << FMC_SDTR1_TXSR_Pos) |
			((4 /* SelfRefreshTime */ - 1) << FMC_SDTR1_TRAS_Pos) |
			((6 /* RowCycleDelay */ - 1) << FMC_SDTR1_TRC_Pos) |
			((2 /* WriteRecoveryTime */ - 1) << FMC_SDTR1_TWR_Pos) |
			((2 /* RPDelay */ - 1) << FMC_SDTR1_TRP_Pos) |
			((2 /* RCDDelay */ - 1) << FMC_SDTR1_TRCD_Pos);
	
	FMC_Bank5_6->SDTR[0] = cfg;
	FMC_Bank5_6->SDTR[1] = cfg;

	//Clock configuration enable command
	if (!sdramSendCmdStm32f469(FMC_SDRAM_CMD_CLK_ENABLE | FMC_SDCMR_CTB1, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "Cfg Ena");
	
	//100 ms delay or so
	for (delay = 1920000; delay; delay--);

	//PALL (precharge all) command
	if (!sdramSendCmdStm32f469(FMC_SDRAM_CMD_PALL | FMC_SDCMR_CTB1, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "PALL");
	
	//Auto-Refresh command
	if (!sdramSendCmdStm32f469(FMC_SDRAM_CMD_AUTOREFRESH_MODE | FMC_SDCMR_CTB1, 8, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "AutoRefresh");

	//Program external memory mode register
	if (!sdramSendCmdStm32f469(FMC_SDRAM_CMD_LOAD_MODE | FMC_SDCMR_CTB1, 1, SDRAM_MODEREG_BURST_LENGTH_1 | SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL | SDRAM_MODEREG_CAS_LATENCY_3 | SDRAM_MODEREG_OPERATING_MODE_STANDARD | SDRAM_MODEREG_WRITEBURST_MODE_SINGLE, 0x1000))
		fatal("SDRAM init failed: %s\n", "External Memory Mode");

	//Set the refresh rate counter => (15.62 us x Freq) - 20, Set the device refresh counter */
	FMC_Bank5_6->SDRTR |= (0x050C << FMC_SDRTR_COUNT_Pos);
	
	//map it at 0x60000000 instead of 0xc0000000
	SYSCFG->MEMRMP = (SYSCFG->MEMRMP &~ SYSCFG_MEMRMP_SWP_FMC_Msk) | SYSCFG_MEMRMP_SWP_FMC_0;
	
	
	//for now, clear it
	memset((void*)0x60000000, 0, 16*1024*1024);
}

static void hwrInitStm32f429(void)
{
	clockTreeInitStm32f429();
	gpiosInitStm32f429();
	sdramInitStm32f429();
}

static void hwrInitStm32f469(void)
{
	clockTreeInitStm32f469();
	gpiosInitStm32f469();
	sdramInitStm32f469();
}

static void hwrInitCortexEmu(void)
{
	
}

static void hwrInit(void)
{
	uint32_t coreid = *(volatile uint32_t*)0xE000ED00UL;
	
	if (coreid == 0x412FC231UL || coreid == 0x410FC240 || coreid == 0x410CC601 || coreid == 0x411CD200) {

		hwrInitCortexEmu();
	}
	else if (((*(volatile uint32_t*)0xE0042000) & 0xFFF) == 0x419) {
		
		hwrInitStm32f429();
	}
	else if (((*(volatile uint32_t*)0xE0042000) & 0xFFF) == 0x434) {
		
		hwrInitStm32f469();
	}
}
