#ifndef _OS_STRUCTS_H_
#define _OS_STRUCTS_H_

#include <stdint.h>


#define HEAP_FLAGS_RO					0x0001
#define HEAP_FLAGS_IS_V2				0x8000
#define HEAP_FLAGS_IS_V3				0x4000
#define HEAP_FLAGS_IS_V4				0x2000	//first and last master pointer table entries are special
#define HEAP_FLAGS_IS_V5				0x1000
#define HEAP_FLAGS_IS_V6				0x0800

#define CARD_FLAGS_68328				0x0010	//targets original 68k processor
#define CARD_FLAGS_68K_EZ				0x0020	//targets the 68k EZ
#define CARD_FLAGS_68K_VZ				0x0040	//targets the 68k VZ
#define CARD_FLAGS_ARM					0x0080	//targets arm processors
#define CARD_FLAGS_WORD_ALIGN			0x2000	//everything is word aligned everywhere (implied by CARD_FLAGS_ARM)


struct ResourceListEntryOS5 {
	uint32_t					type;
	uint16_t					id;
	uint16_t					zero;
	uint32_t					dataPtr;
} __attribute__((packed));

struct ResourceListEntryOS4 {
	uint32_t					type;
	uint16_t					id;
	uint32_t					dataPtr;
} __attribute__((packed));

struct RecordListEntry {
	uint32_t					dataPtr;
	uint8_t						attr;
	uint8_t						uniqId[3];			//LE
};

struct DatabaseHeaderCommon {
	char						name[32];			// 0x00		//can actually be 32 chars with no termination....m520 does this
	uint16_t					attributes;			// 0x20
	uint16_t					version;			// 0x22
	uint32_t					creationDate;			// 0x24
	uint32_t					modificationDate;		// 0x28
	uint32_t					lastBackupDate;			// 0x2C - zero for ROM DBs
	uint32_t					modificationNumber;		// 0x30
	uint32_t					appInfoPtr;			// 0x34
	uint32_t					sortInfoPtr;			// 0x38
	uint32_t					type;				// 0x3C
	uint32_t					creator;			// 0x40
	uint32_t					uniqueIDSeed;			// 0x44
	uint32_t					nextRecordListPtr;		// 0x48 (always zero for us)
	uint16_t					numChildren;			// 0x4C
} __attribute__((packed));

struct DatabaseHeaderOS5 {
	struct DatabaseHeaderCommon	hdr;				// 0x00
	uint16_t					val0xD2CE;			// 0x4E	always 0xD2CE in our rom
} __attribute__((packed));

struct DatabaseHeaderOS4 {
	struct DatabaseHeaderCommon	hdr;				// 0x00
} __attribute__((packed));


struct MemMstrPtrTableV1 {		//OS1+

	uint16_t					numEntries;			// 0x00 num entries ?? (0 for us)
	uint16_t					nextTblOffset;			// 0x02 offset from start of heap to next table NOTE: not from this struct)
	void *						mstrP[];			// 0x04 array of pointers to movable chunks
} __attribute__((packed));


struct MemMstrPtrTableV234 {		//OS2+

	uint16_t					numEntries;			// 0x00 num entries ?? (0 for us)
	uint32_t					nextTblOffset;			// 0x02 offset from start of heap to next table NOTE: not from this struct)
	void *						mstrP[];			// 0x06 array of pointers to movable chunks
} __attribute__((packed));

struct MemMstrPtrTableV4_ARM {		//OS5

	uint16_t					numEntries;				// 0x00 num entries ?? (0 for us)
	uint16_t					ffff;					// 0x02 0xffff always
	uint32_t					nextTblOffset;			// 0x04 offset from start of heap to next table NOTE: not from this struct)
	void *						mstrP[];				// 0x0C array of pointers to movable chunks
} __attribute__((packed));

struct HeapHeaderV1 {			//OS1+
	
	uint16_t					flags;				// 0x00
	uint16_t					size;				// 0x02
	struct MemMstrPtrTableV1			mstrPtrTbl;			// 0x04
} __attribute__((packed));

struct HeapHeaderV2 {			//OS?+
	
	uint16_t					flags;				// 0x00
	uint32_t					size;				// 0x02
	struct MemMstrPtrTableV234			mstrPtrTbl;			// 0x06
} __attribute__((packed));

struct HeapHeaderV3 {			//OS3+
	
	uint16_t					flags;				// 0x00
	uint32_t					size;				// 0x02
	uint32_t					firstFreeChunkOffset;		// 0x06 offset from heap start to chunk header. cannot be zero. cannot point to non free chunk
	struct MemMstrPtrTableV234			mstrPtrTbl;			// 0x0a
} __attribute__((packed));

struct HeapHeaderV4_68k {		//OS4 (same as HeapHeaderV3, but first and last master pointer table entries are special in a way that does not matter for ROM heaps)
	
	uint16_t					flags;				// 0x00 0x2001 for us
	uint32_t					size;				// 0x02 size of heap, ours goes to end of rom, size is from start of this header
	uint32_t					firstFreeChunkOffset;		// 0x06 offset from heap start to chunk header. cannot be zero. cannot point to non free chunk
	struct MemMstrPtrTableV234			mstrPtrTbl;			// 0x0A Master pointer table
} __attribute__((packed));

struct HeapHeaderV4_ARM {		//OS5	first and last master pointer table entries are special
	
	uint16_t					flags;				// 0x00 0x2001 for us
	uint16_t					ffff;				// 0x02 0xffff always
	uint32_t					size;				// 0x04 size of heap, ours goes to end of rom, size is from start of this header
	uint32_t					firstFreeChunkOffset;		// 0x08 offset from heap start to chunk header. cannot be zero. cannot point to non free chunk. in WORDS
	struct MemMstrPtrTableV4_ARM			mstrPtrTbl;			// 0x0C Master pointer table
} __attribute__((packed));

struct HeapChunkHeader {
/*
	//assumes compiler allocates from bottm to top bits with no holes

	//WORD @ 0x00
	uint32_t size				: 24;					// 0x00FFFFFF - size of the allocated chunk including this header
	uint32_t sizeExtra			:  4;					// 0x0F000000 - how many bytes were allocated over what was requested (excluding header)
	uint32_t unused				:  2;					// 0x30000000 - not used it seems
	uint32_t irrelevant			:  1;					// 0x40000000 - weird one (docs say this is called "moved")
	uint32_t free				:  1;					// 0x80000000 - set if chunk is currently free
	//WORD @ 0x04
	uint32_t handleOfst			: 24;					// 0x00FFFFFF - if free chunk, offset to next free chunk divided by 2; if not free, 0 if nonmovable, else negative offset to memhandle divided by 2 (memhandle is chunk header minus 2 times this)
	uint32_t owner				:  4;					// 0x0F000000 - ownerID of the owner
	uint32_t lockCt				:  4;					// 0xF0000000 - lock count (15 for seemingly all things in rom)
*/

	uint32_t w0;
	uint32_t w1;
} __attribute__((packed));

struct HeapChunkHeaderV1 {		//for very old heaps
	uint16_t					size;				// 0x00
	uint8_t						lockCtAndOwner;			// 0x02 (lock count in hi4, owner id in lo4)
	uint8_t						flagsAndSizeExtra;		// 0x03 (flags in hi4, sizeExtra in lo4). flag 0x08 == free
	int16_t						handleOfst;			// 0x04 same as for newer chunks
} __attribute__((packed));

//warning: these bitmasks change on BE PalmOS
#define CHUNK_HDR_W0_SIZE_MASK				0x00FFFFFF
#define CHUNK_HDR_W0_SIZE_SHIFT				0
#define CHUNK_HDR_W0_SIZE_XTRA_MASK			0x0F000000
#define CHUNK_HDR_W0_SIZE_XTRA_SHIFT			24
#define CHUNK_HDR_W0_MOVED_MASK				0x40000000
#define CHUNK_HDR_W0_MOVED_SHIFT			30
#define CHUNK_HDR_W0_FREE_MASK				0x80000000
#define CHUNK_HDR_W0_FREE_SHIFT				31

#define CHUNK_HDR_W1_HANDLE_OFST_MASK			0x00FFFFFF
#define CHUNK_HDR_W1_HANDLE_OFST_SHIFT			0
#define CHUNK_HDR_W1_OWNER_MASK				0x0F000000
#define CHUNK_HDR_W1_OWNER_SHIFT			24
#define CHUNK_HDR_W1_LOCK_CT_MASK			0xF0000000
#define CHUNK_HDR_W1_LOCK_CT_SHIFT			28

#define CHUNK_OWNER_NOBODY				0x00				//chunks in dynamic heap and free chunks everywhere are owned by this owner
#define CHUNK_OWNER_DM_MGMNT_DATA			0x01				//Data manager's chunks (db list and headers) are owned by this owner
#define CHUNK_OWNER_DM_REC_AND_RES			0x02				//database records and resources are owned by this owner
#define CHUNK_OWNER_MASTER_PTR_TAB			0x0F				//first one is part of heap header, others are in chunks with this owner

#define CHUNK_LOCK_CT_NONMOVABLE			15				//lock count of all non-movable chunks

//cobalt masks are different
#define OS6_CHUNK_HDR_W0_SIZE_MASK			0x03FFFFFF
#define OS6_CHUNK_HDR_W0_SIZE_SHIFT			0
#define OS6_CHUNK_HDR_W0_SIZE_XTRA_MASK			0x1C000000
#define OS6_CHUNK_HDR_W0_SIZE_XTRA_SHIFT		26
#define OS6_CHUNK_HDR_W0_MOVED_MASK			0x40000000
#define OS6_CHUNK_HDR_W0_MOVED_SHIFT			30
#define OS6_CHUNK_HDR_W0_FREE_MASK			0x80000000
#define OS6_CHUNK_HDR_W0_FREE_SHIFT			31


#define OS6_CHUNK_HDR_W1_OWNER_MASK			0xF0000000
#define OS6_CHUNK_HDR_W1_OWNER_SHIFT			28
#define OS6_CHUNK_HDR_W1_LOCK_CT_MASK			0x0F000000
#define OS6_CHUNK_HDR_W1_LOCK_CT_SHIFT			24

//so are owners
#define OS6_CHUNK_OWNER_DM_MGMNT_DATA			0x03				//Data manager's chunks (db headers) are owned by this owner
#define OS6_CHUNK_OWNER_DM_REC_AND_RES			0x01				//database records and resources are owned by this owner


#define CARD_HEADER_SIG					0xFEEDBEEF
#define STORE_HEADER_SIG				0xFEEDFACE
#define CARD_HEADER_COBALT_BIGROM		0xDADAFACE
#define CARD_HEADER_COBALT_SMALLROM		0xFEEDBABE


//broken out because otherwise gcc does not align things reight (who knows why)
struct CardHeaderDiff1os4 {
	uint16_t					version;			// 0x50 always 1
	uint32_t					creationDate;			// 0x52 V2: card creation date in palm format
	uint16_t					numRAMBlocks;			// 0x56 always 1
} __attribute__((packed));

struct CardHeaderDiff1os5 {
	uint16_t					version;			// 0x50 always 1
	uint16_t					numRAMBlocks;			// 0x52 always 1
	uint32_t					creationDate;			// 0x54 card creation date in palm format
} __attribute__((packed));

struct CardHeaderDiff2os4 {
	uint16_t 					romChecksum;			// 0x70 probably checksum
//V3+
	uint32_t					readWriteWorkingOffset;		// 0x72 always 0
	uint32_t					readWriteWorkingSize;		// 0x76 always 0
//V4+
	uint32_t					halCodeOffset;		// 0x7A offset from this header to ('boot' 19000) resource in HAL
//V5+
	uint32_t					companyID;			// 0x7E 4cc
	uint32_t					halID;				// 0x82 4cc
	uint32_t					romVersion;			// 0x86 seems to be zero always
	char						romVersionString[32];		// 0x8A rom name string
	uint32_t					halCode2offset;				// 0xaa offset from this header to ('boot' 19001) resource in DAL	
} __attribute__((packed));

struct CardHeaderDiff2os5 {
	uint32_t					readWriteWorkingOffset;		// 0x70 always 0
	uint32_t					readWriteWorkingSize;		// 0x74 always 0
	uint32_t					dalAmddRsrcOfst;		// 0x78 offset from this header to ('amdd' 19000) resource in DAL
	uint32_t 					romChecksum;			// 0x7C probably checksum
	uint32_t					companyID;			// 0x80 4cc
	uint32_t					halID;				// 0x84 4cc
	uint32_t					romVersion;			// 0x88 seems to be zero always
	char						romVersionString[32];		// 0x8C rom name string
	uint32_t					dalAmdiRsrcOfst;		// 0xAC offset from this header to ('amdi' 19000) resource in DAL	
} __attribute__((packed));

struct CardHeader {
	union {
		struct {
			uint32_t			jumpToCode;			// 0x00 jump instr to code ('boot' 19000) res in DAL
			uint32_t			zeroNum0;			// 0x04 always 0
		} os5 __attribute__((packed));
		struct {
			uint32_t			initialSp;			// 0x00 SP for boot code
			uint32_t			initialPc;			// 0x04 PC for boot code (pointer into dal)
		} os4 __attribute__((packed));
		struct {
			uint32_t			initialSp;			// 0x00 initial SP val
			uint32_t			initialPc;			// 0x04 pc ptr incl bottom bit for thumb
		} rePalm __attribute__((packed));
	} entry;
	uint32_t					signature;			// 0x08 0xFEEDBEEF
	uint16_t					hdrVersion;			// 0x0C 5 for small rom, 6 for big in OS5, max 5 in OS4
	uint16_t					flags;				// 0x0E 0x2080 for big rom, 0x20A0 for small
	char						name[32];			// 0x10 card name
	char						manuf[32];			// 0x30 card manufacturer's name
	union {
		struct CardHeaderDiff1os4 os4;
		struct CardHeaderDiff1os5 os5;
	} diff1;
	uint32_t					blockListPtr;			// 0x58 RAM block list
//V2+ only
	uint32_t					readWriteParmsOffset;		// 0x5C always 0
	uint32_t					readWriteParmsSize;		// 0x60 always 0
	uint32_t					romTokenStorePtr;		// 0x64 pointer to rom token storage (RO)
	uint32_t					bigRomPtr;			// 0x68 VA of big rom (in both small and big rom)
	uint32_t					totalRomSz;			// 0x6C offset form this header to end of this ROM (small or big)
	union {
		struct CardHeaderDiff2os4 os4;
		struct CardHeaderDiff2os5 os5;
	} diff2;

} __attribute__((packed));	//0xB0 bytes


struct CardHeaderCobalt {
	uint32_t					jumpToCode;			// 0x000 jump to ???
	uint32_t					rfu[63];			// 0x004 padding
	uint32_t					signature;			// 0x100 0xDADAFACE (big rom) or FEEDBABE (small rom)
	uint32_t					unsure1;			// 0x104 - 0x00018000
	uint32_t					zero1;				// 0x108 - zero for us
	uint32_t					romType;			// 0x10C - 'psys'
	uint32_t					romCrid;			// 0x110 - 'MX21'
	uint32_t					unsure2;			// 0x114 - 0x00010000
	uint32_t					bigRomSearchStartOffset;	// 0x118 - 0xc8440000 - offset from this header to look for the big rom (used by small rom), and then every 256K is searched
	uint32_t					romBase;			// 0x11c - 0x01040000 - maybe bigrom base
	char						romVersionString[32];		// 0x120 rom name string
	uint32_t					osCrid;				// 0x140 - 'Palm'
	uint32_t					unsure5;			// 0x144 - 0x00F6FE00 (rom size?)
	uint32_t					unsure6;			// 0x148 - 0x0000CD94 (checksum?)
	uint32_t					osVersionBcd;			// 0x14c - 0x06113003 = 6.1.1 release 3 bugfix 3
	char						osVersionString[32];		// 0x150 OS version string
} __attribute__((packed));


struct HeapListOS5 {
	uint16_t 					numItems;
	uint16_t					unused;
	uint32_t 					heapHeaderPtrs[1];		//sized to fit one :)
};

struct HeapListOS4 {
	uint16_t 					numItems;
	uint32_t 					heapHeaderPtrs[1];		//sized to fit one :)
} __attribute__((packed));

struct DatabaseListOS5 {
	uint32_t					nextDatabaseDir;		// 0x00 - always zero
	uint32_t					numDatabases;			// 0x04 - how many entries follow
	uint32_t					databaseHdr[];			// 0x08 - pointers to headers, exactly 'numDatabases' of them
} __attribute__((packed));

struct DatabaseListOS4 {
	uint32_t					nextDatabaseDir;		// 0x00 - always zero
	uint16_t					numDatabases;			// 0x04 - how many entries follow
	uint32_t					databaseHdr[];			// 0x06 - pointers to headers, exactly 'numDatabases' of them
} __attribute__((packed));

struct StoreHeaderDiff1os4 {
	uint16_t					zero3;				// 0x54
	uint32_t					bootScreen0ptr;			// 0x56 pointer to "Palm OS Data_enUS" ('Tbsb' 19000) resource (normal boot screen)
	uint32_t					bootScreen1ptr;			// 0x5A pointer to "Palm OS Data_enUS" ('Bbsb' 19001) resource (request for hard reset screen)
	uint16_t					romLanguage;			// 0x5E - see locale mgr. 0x00 = english (lEnglish)
	uint16_t					romCountry;				// 0x60 - see locale mgr. 0x17 = USA (cUnitedStates)
	uint16_t					zero4;					// 0x62

} __attribute__((packed));

struct StoreHeaderDiff1os5 {
	uint32_t					zero3;				// 0x54
	uint32_t					bootScreen0ptr;			// 0x58 pointer to "Palm OS Data" ('absb' 19000) resource (normal boot screen)
	uint32_t					bootScreen1ptr;			// 0x5C pointer to "Palm OS Data_enUS" ('absb' 19001) resource (request for hard reset screen)
	uint16_t					romLanguage;			// 0x60 - see locale mgr. 0x00 = english (lEnglish)
	uint16_t					romCountry;				// 0x62 - see locale mgr. 0x17 = USA (cUnitedStates)
} __attribute__((packed));

struct StoreHeader {
	uint32_t					signature;			// 0x00 0xFEEDFACE
	uint16_t					version;			// 0x04 always 1 or 2
	uint16_t					flags;				// 0x06 always 0
	char						name[32];			// 0x08 name of store (usually unused)
	uint32_t					zeroNum0;			// 0x28 always 0
	uint32_t					zeroNum1;			// 0x2C always 0
	uint32_t					heapListPtr;			// 0x30 pointer to struct HeapList
	uint32_t					bootRsrsBoot10001;		// 0x34 pointer to Boot.prc ('boot' 10001) resource (data)
	uint32_t					bootRsrsBoot10002;		// 0x38 pointer to Boot.prc ('boot' 10002) resource (info)
	uint32_t					databaseListPtr;		// 0x3C database directory pointer
	uint32_t					zeros1[5];			// 0x40 zeroes
	union {
		struct StoreHeaderDiff1os4 os4;
		struct StoreHeaderDiff1os5 os5;
	};
	
	uint32_t					zeroNum2;			// 0x64 always 0 pre-OS6, there it is 3 in english version 0x53 in chinese???
	uint32_t					zeroNum3;			// 0x68 always 0 
//OS5+
	uint32_t					bootScreen2ptr;			// 0x6C pointer to "Palm OS Data_enUS" ('absb' 19002) resource (soem sort of a boot screen - a full real palmos bitmap 320x320)
	uint32_t					zeros2[34];			// 0x70 zeroes
	uint32_t					bootBootRsrcPtr;		// 0xF8 pointer to Boot.prc ('boot' 10003) resource
	uint32_t					crcUnused;			// 0xFC always 0 
};


#endif
