#include <endian.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include "parsePrcPdb.h"
#include "util.h"

#define FMT4CC		"%c%c%c%c"
#define CVT4CC(x)	((x) >> 24) & 0xff, ((x) >> 16) & 0xff, ((x) >> 8) & 0xff, (x) & 0xff
#define MK4CC(_a,_b,_c,_d)		(((_a) << 24) | ((_b) << 16) | ((_c) << 8) | (_d))



struct BmpInfo {
	uint16_t density;
	uint8_t version;
	uint8_t depth;
	uint32_t ofst;
	uint32_t length;
};


typedef uint_fast8_t (*BmpsFilterFunc)(struct BmpInfo *bmps, uint_fast8_t nBmps);	//return new number of bitmaps, do not mess up order

struct FilterOption {
	BmpsFilterFunc func;		//NULL is sentinel
	const char *name;
	const char *descr;
};

extern struct FilterOption mFilterOptions[];


static void usage(const char* me)
{
	const struct FilterOption *fo;
	
	fprintf(stderr, "USAGE: %s file.prc file.out.prc <FLAGS>\n"
					" FLAGS:\n"
					"  'fixtrunc' - fix bitmaps whose \"next bitmap\" pointers dangle poiting @ end\n", me);
	
	for (fo = mFilterOptions; fo->func; fo++)
		fprintf(stderr, "  '%s' - %s\n", fo->name, fo->descr);
	
	exit(-1);
}


#define VER_MASK_LE			0x80

struct BmpGeneric {
	int16_t w, h;
	uint16_t rowBytes;
	uint16_t flags;
	uint8_t rfu0, ver;
} __attribute__((packed));

struct BmpV0 {
	int16_t w, h;
	uint16_t rowBytes;
	uint16_t flags;
	uint16_t rfu[4];	//mandatorily zero filled
} __attribute__((packed));

struct BmpV1 {
	int16_t w, h;
	uint16_t rowBytes;
	uint16_t flags;
	uint8_t pixSz, ver;
	uint16_t nextDepthOfst;	//in units of 4 bytes
	uint16_t rfu[2];
} __attribute__((packed));

struct BmpV2 {
	int16_t w, h;
	uint16_t rowBytes;
	uint16_t flags;
	uint8_t pixSz, ver;
	uint16_t nextDepthOfst;	//in units of 4 bytes
	uint8_t transIdx, comprTyp;
	uint16_t rfu[1];
} __attribute__((packed));

struct BmpV3 {
	int16_t w, h;
	uint16_t rowBytes;
	uint16_t flags;
	uint8_t pixSz, ver, structSz, pixFmt, rfu0, comprTyp;
	uint16_t density;
	uint32_t transVal, nextDepthOfst; 
} __attribute__((packed));


static void bmpCull(uint32_t resType, uint_fast16_t resIdx, struct Buffer *buf, bool leExpected, BmpsFilterFunc *filters, bool fixTrunc)
{
	bool first = true, hadPreV3spacer = false;
	uint32_t curOfst = 0, newBmpSz = 0;
	uint_fast8_t i, ver, nBmps = 0;
	struct BmpInfo bmps[32];
	BmpsFilterFunc filt;
	uint8_t *newRes;
	
	fprintf(stderr, " ("FMT4CC", 0x%04x): ", CVT4CC(resType), (unsigned)resIdx);
	
	while(1) {
		struct BmpGeneric *bmpG = (struct BmpGeneric*)(((char*)buf->data) + curOfst);
		struct BmpV0 *bmp0 = (struct BmpV0*)bmpG;
		struct BmpV1 *bmp1 = (struct BmpV1*)bmpG;
		struct BmpV2 *bmp2 = (struct BmpV2*)bmpG;
		struct BmpV3 *bmp3 = (struct BmpV3*)bmpG;
		uint8_t thisDepth, thisDensity, hdrSz;
		bool isPreV3Spacer = false;
		uint32_t thisImgSz;
		
		if (curOfst == buf->sz) {
			
			if (fixTrunc) {
				
				fprintf(stderr, "ends at res end. likely truncated");
				break;
			}
			else {
				
				fprintf(stderr, "ends at res end. likely truncated\n");
				return;
			}
		}
		
		if (curOfst > buf->sz || buf->sz - curOfst < sizeof(*bmpG)) {
			
			fprintf(stderr, "cur bmp offset is %u which is implausible in a res of size %u\n", curOfst, buf->sz);
			break;
			return;
		}
		
		ver = bmpG->ver;
		
		if (!(ver & VER_MASK_LE) != !leExpected) {
			
			fprintf(stderr, "unexpected endianness resource found\n");
			return;
		}
		
		ver &=~ VER_MASK_LE;
		
		switch (ver) {
			case 0:
				if (!first) {
					fprintf(stderr, "V0 bitmap must be first\n");
					return;
				}
				thisImgSz = 0;
				thisDepth = 1;
				thisDensity = 72;
				break;
			
			case 1:
				thisImgSz = (leExpected ? le16toh(bmp1->nextDepthOfst) : be16toh(bmp1->nextDepthOfst)) * 4;
				thisDepth = bmp1->pixSz;
				hdrSz = sizeof(*bmp1);
				thisDensity = 72;
				
				if (thisDepth == 0xff) {
					
					isPreV3Spacer = true;
					thisImgSz = sizeof(*bmp1);
					
					if (bmp1->w || bmp1->h || bmp1->flags || bmp1->nextDepthOfst || bmp1->rfu[0] || bmp1->rfu[1]) {
						
						fprintf(stderr, "invalid v3 spacer\n");
						return;
					}
				}
				break;
			
			case 2:
				thisImgSz = (leExpected ? le16toh(bmp2->nextDepthOfst) : be16toh(bmp2->nextDepthOfst)) * 4;
				thisDepth = bmp2->pixSz;
				hdrSz = sizeof(*bmp2);
				thisDensity = 72;
				break;
			
			case 3:
				thisImgSz = leExpected ? le32toh(bmp3->nextDepthOfst) : be32toh(bmp3->nextDepthOfst);
				thisDensity = leExpected ? le16toh(bmp3->density) : be16toh(bmp3->density);
				thisDepth = bmp3->pixSz;
				hdrSz = sizeof(*bmp3);
				break;
			
			default:
				fprintf(stderr, "version %u not supported\n", ver);
				return;
		}
		
		if (isPreV3Spacer) {
			
			if (hadPreV3spacer) {
				
				fprintf(stderr, "too many pre-v3 spacers\n");
				return;
			}
			else
				hadPreV3spacer = true;
		}
		else {
			
			struct BmpInfo *bmpi = bmps + nBmps++;
			
			if (nBmps > sizeof(bmps) / sizeof(*bmps)) {
				
				fprintf(stderr, "too many bitmap components\n");
				return;
			}
			
			if (ver >= 3 && !hadPreV3spacer) {
				
				fprintf(stderr, "V3+ bitmap seen without a pre-v3-spacer\n");
				return;
			}
		
			if (!thisDepth || thisDepth > 16 || (thisDepth & (thisDepth - 1))) {
				
				fprintf(stderr, "invalid depth %u\n", thisDepth);
				return;
			}
			if (!thisDensity || (thisDensity != 108 && (thisDensity % 72))) {
				
				fprintf(stderr, "invalid density %u\n", thisDensity);
				return;
			}
			
			bmpi->density = thisDensity;
			bmpi->version = ver;
			bmpi->depth = thisDepth;
			bmpi->ofst = curOfst;
			bmpi->length = thisImgSz ? thisImgSz : buf->sz - curOfst;
		
			if (bmpi->length & 3) {
				
				fprintf(stderr, "length not a multiple of 4\n");
				return;
			}
		
			fprintf(stderr, "(v%u: %cE %ubpp, %uppi @ %08xh + %08xh) ", bmpi->version, leExpected ? 'L' : 'B', bmpi->depth, bmpi->density, bmpi->ofst, bmpi->length);
		}
		
		if (!ver || !thisImgSz)
			break;
		
		if (thisImgSz < hdrSz) {
			
			fprintf(stderr, "cur img sz < hdr sz (%u %u)\n", thisImgSz, hdrSz);
			return;
		}
		
		curOfst += thisImgSz;
	}
	
	while (filt = *filters++)
		nBmps = filt(bmps, nBmps);
	
	if (!nBmps) {
		
		fprintf(stderr, "  no bitmaps left after filtering\n");
		return;
	}
	
	//now assemble the new resource. max size is sum of sizes plus one v3 spacer
	newBmpSz = 0;
	for (i = 0; i < nBmps; i++)
		newBmpSz += bmps[i].length;
	newRes = malloc(newBmpSz + sizeof(struct BmpV1));
	if (!newRes) {
		fprintf(stderr, "  cannot alloc new res\n");
		return;
	}
	newBmpSz = 0;
	hadPreV3spacer = false;
	
	for (i = 0; i < nBmps; i++) {
		
		char *dst = newRes + newBmpSz;
		uint32_t nextImgOfst;
		
		if (bmps[i].version >= 3 && !hadPreV3spacer) {
			struct BmpV1 *spacer = (struct BmpV1*)dst;
			
			dst = (char*)(spacer + 1);
			newBmpSz += sizeof(*spacer);
			
			memset(spacer, 0, sizeof(*spacer));
			spacer->pixSz = 0xff;
			spacer->ver = 1 + (leExpected ? VER_MASK_LE : 0);
			hadPreV3spacer = true;
		}
		
		nextImgOfst = (i == nBmps - 1) ? 0 : bmps[i].length;
		memcpy(dst, ((char*)buf->data) + bmps[i].ofst, bmps[i].length);
		ver = ((struct BmpGeneric*)dst)->ver &~ VER_MASK_LE;
		
		switch (ver) {
			
			case 1:
				((struct BmpV1*)dst)->nextDepthOfst = leExpected ? htole16(nextImgOfst / 4) : htobe16(nextImgOfst / 4);
				break;
			
			case 2:
				((struct BmpV2*)dst)->nextDepthOfst = leExpected ? htole16(nextImgOfst / 4) : htobe16(nextImgOfst / 4);
				break;
			
			case 3:
				((struct BmpV3*)dst)->nextDepthOfst = leExpected ? htole32(nextImgOfst) : htobe32(nextImgOfst);
				break;
			
			default:
				fprintf(stderr, "not sure how to link bmp v %u in\n", ver);
				free(newRes);
				return;
		}
		
		newBmpSz += bmps[i].length;
	}
	
	if (newBmpSz > buf->sz) {
		
		fprintf(stderr, "  somehow image got bigger\n");
		free(newRes);
		return;
	}
	
	memcpy(buf->data, newRes, newBmpSz);
	
	fprintf(stderr, "(%u -> %u b)", buf->sz, newBmpSz);
	
	buf->sz = newBmpSz;
	
	fprintf(stderr, "\n");
}

int main(int argc, const char** argv)
{
	const char* self = *argv++, **flags = argv + 2;
	uint_fast16_t i, nFlags = argc - 3, nFilters = 0;
	BmpsFilterFunc filters[64] = {};
	bool fixTrunc = false;
	struct PalmDb *db;
	
	if (argc < 3 || !(db = parsePrcPdb(argv[0])) || !db->isPrc)
		usage(self);
	
	while (nFlags--) {
		
		const struct FilterOption *fo;
		const char *flag = *flags++;
		
		if (!strcmp(flag, "fixtrunc"))
			fixTrunc = true;
		else {
			
			for (fo = mFilterOptions; fo->func; fo++) {
				
				if (!strcmp(fo->name, flag)) {
					
					if (nFilters + 1 >= sizeof(filters) / sizeof(*filters)) {
						
						fprintf(stderr, "too many filters\n");
						abort();
					}
					
					filters[nFilters++] = fo->func;
					break;
				}
			}
			if (!fo->func) {
				
				fprintf(stderr, "unknown filter '%s'\n", flag);
				abort();
			}
		}
	}
	
	for (i = 0; i < db->numChildren; i++) {
		
		uint32_t typ = db->res[i].type;
		uint16_t id = db->res[i].id;
		
		if (typ == MK4CC('T','b','m','p') || typ == MK4CC('t','A','I','B'))
			bmpCull(typ, id, &db->res[i].buf, false, filters, fixTrunc);
		else if (typ == MK4CC('a','b','m','p') || typ == MK4CC('a','a','i','b'))
			bmpCull(typ, id, &db->res[i].buf, true, filters, fixTrunc);
	}
	
	if (!writePrcPdb(db, argv[1])) {
		fprintf(stderr, "failed to write '%s'\n", argv[1]);
		abort();
	}
	
	freePrcPdb(db);
	
	return 0;
}


//filters

static uint_fast8_t bmpsFilterFuncLrOnly(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	uint_fast8_t i = 0;
	
	while (i < nBmps) {
		
		if (i && bmps[i].density > 72) {	//leave first even if HR
			
			memmove(bmps + i, bmps + i + 1, sizeof(*bmps) * (nBmps - i - 1));
			nBmps--;
			continue;
		}
		i++;
	}
	
	return nBmps;
}

static uint_fast8_t bmpsFilterFuncHrOnly(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	uint_fast8_t i = 0;
	struct BmpInfo first = *bmps;
	
	while (i < nBmps) {
		
		if (bmps[i].density != 144) {	//leave first even if LR
			
			memmove(bmps + i, bmps + i + 1, sizeof(*bmps) * (nBmps - i - 1));
			nBmps--;
			continue;
		}
		i++;
	}
	
	if (!nBmps) {
		*bmps = first;
		nBmps = 1;
	}
	
	return nBmps;
}

static uint_fast8_t bmpsFilterFuncDelXbpp(struct BmpInfo *bmps, uint_fast8_t nBmps, uint_fast8_t delDepth)
{
	uint_fast8_t i = 0;
	
	while (i < nBmps) {
		
		if (bmps[i].depth == delDepth) {
			
			//see if image before or after is same density
			if ((i && bmps[i - 1].density == bmps[i].density) || (i != nBmps - 1 && bmps[i + 1].density == bmps[i].density)) {
				
				memmove(bmps + i, bmps + i + 1, sizeof(*bmps) * (nBmps - i - 1));
				nBmps--;
				continue;
			}
		}
		i++;
	}
	
	
	return nBmps;
}

static uint_fast8_t bmpsFilterFuncDel1bpp(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	return bmpsFilterFuncDelXbpp(bmps, nBmps, 1);
}

static uint_fast8_t bmpsFilterFuncDel2bpp(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	return bmpsFilterFuncDelXbpp(bmps, nBmps, 2);
}

static uint_fast8_t bmpsFilterFuncDel4bpp(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	return bmpsFilterFuncDelXbpp(bmps, nBmps, 4);
}

static uint_fast8_t bmpsFilterFuncDel8bpp(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	return bmpsFilterFuncDelXbpp(bmps, nBmps, 8);
}

static uint_fast8_t bmpsFilterFuncDel16bpp(struct BmpInfo *bmps, uint_fast8_t nBmps)
{
	return bmpsFilterFuncDelXbpp(bmps, nBmps, 16);
}

struct FilterOption mFilterOptions[] = {
	{bmpsFilterFuncLrOnly, "lronly", "remove all non-low-res images",},
	{bmpsFilterFuncHrOnly, "hronly", "remove all non-hi-res images",},
	{bmpsFilterFuncDel1bpp, "del1bpp", "remove all 1bpp images unless the only image at that density",},
	{bmpsFilterFuncDel2bpp, "del2bpp", "remove all 2bpp images unless the only image at that density",},
	{bmpsFilterFuncDel4bpp, "del4bpp", "remove all 4bpp images unless the only image at that density",},
	{bmpsFilterFuncDel8bpp, "del8bpp", "remove all 8bpp images unless the only image at that density",},
	{bmpsFilterFuncDel16bpp, "del16bpp", "remove all 16bpp images unless the only image at that density",},
	{0,},	
};








