#ifndef _VDRV_V5_H_
#define _VDRV_V5_H_

#include <stdint.h>
#include <MemoryMgr.h>
#include <LibTraps.h>
#include <SerialMgr.h>

#define VDRV_VERSION_5							5

//main entries to the driver
#define VDRV_ENTRY_NUM_GET_NUM_PORTS			0			//data is uint16_t* that we write with number of ports
#define VDRV_ENTRY_NUM_GET_PORT_FUNCS			1			//data is struct VdrvPortFuncs* that we fill
#define VDRV_ENTRY_NUM_GET_PORT_INFO			2			//data is struct VdrvPortInfo* that we fill

//port flags
#define VDRV_PORT_FLAG_PHYSICAL_PORT			0x00000001	//never seen as used
#define VDRV_PORT_FLAG_CAN_RUN_IN_BACKGROUND	0x00000002	//always set seemingly
#define VDRV_PORT_FLAG_RS232_CAPABLE			0x00000004	//set on non-ir non-usb ports
#define VDRV_PORT_FLAG_IRDA_CAPABLE				0x00000008	//set on IrDA ports
#define VDRV_PORT_FLAG_CRADLE_PORT				0x00000010	//set on USB and serial ports that are on the cradle
#define VDRV_PORT_FLAG_EXTERNAL					0x00000020	//set on ports that are not integated into the device (eg: on a memory card)
#define VDRV_PORT_FLAG_MODEM_PORT				0x00000040	//set on ports that connect to in-device modem
#define VDRV_PORT_FLAG_VISIBLE_IN_CNC_MGR		0x00000080	//set on ports that may be shown to the user (seemingly always)
#define VDRV_PORT_FLAG_CONSOLE_PORT				0x00000100	//set on ports that canbe used for console output
#define VDRV_PORT_FLAG_USB_CAPABLE				0x00000200	//set on USB ports
#define VDRV_PORT_FLAG_PRIVATE_USE				0x00001000	//set on ports that are for special use (likely not enumerateable)

//port status bits
#define VDRV_STA_CTS							0x00000001
#define VDRV_STA_RTS							0x00000002	//never seen in the wild
#define VDRV_STA_DSR							0x00000004
#define VDRV_STA_TX_FIFO_FULL					0x00000008
#define VDRV_STA_TX_FIFO_EMPTY					0x00000010
#define VDRV_STA_BREAK_ASSERTED					0x00000020

//line error values
#define VDRV_LINE_ERR_PARITY					0x00000001	//parity error
#define VDRV_LINE_ERR_HW_RX_OVERFLOW			0x00000002	//hw buffer overflowed - data was lost
#define VDRV_LINE_ERR_FRAMING					0x00000004	//framing error
#define VDRV_LINE_ERR_BREAK						0x00000008	//break received
#define VDRV_LINE_ERR_HANDSHAKE					0x00000010	//hw handshaking error
#define VDRV_LINE_ERR_SW_RX_OVERFLOW			0x00000020	//sw buffer overflowed - data was lost
#define VDRV_LINE_ERR_CARRIER_LOST				0x00000040

//uart port settigns
#define VDRV_UART_SETTING_2_STOP_BITS			0x00000001	//else 1
#define VDRV_UART_SETTING_PARITY_ON				0x00000002	//else off
#define VDRV_UART_SETTING_PARITY_EVEN			0x00000004	//else odd
#define VDRV_UART_SETTING_RTS_RX_FLOW			0x00000010	//if set, HW will control RTS pin to prevent HW RX FIFO overflow
#define VDRV_UART_SETTING_CTS_TX_FLOW			0x00000020	//if set, HW will honour incoming CTS signal to not send whne remote isn't ready
#define VDRV_UART_SETTING_5_BIT_CHARS			0x00000000
#define VDRV_UART_SETTING_6_BIT_CHARS			0x00000040
#define VDRV_UART_SETTING_7_BIT_CHARS			0x00000080
#define VDRV_UART_SETTING_8_BIT_CHARS			0x000000c0
#define VDRV_UART_SETTING_CHAR_SZ_MASK			0x000000c0
#define VDRV_UART_SETTING_RTS_SW_PROTECT		0x00000100	//if set, and VDRV_UART_SETTING_RTS_RX_FLOW is set, when HW is ready to RX, but SW fifo is full, RTS will signal "do not sent" so remote does not cause us to overlow our SW RX FIFO
#define VDRV_UART_SETTING_RTS_OFF				0x00000200	//if set, and VDRV_UART_SETTING_RTS_RX_FLOW is clear, RTS is kept in the "DO NOT send me data" state


//port control values
#define VDRV_CTL_OP_SET_BAUDRATE				0x1000		// param: const uint32_t* baudRateP
#define VDRV_CTL_OP_SET_SETTINGS				0x1001		// param: const uint32_t* settingsValP			configures UART basic settings (bits per char, stop bits, parity, etc) Settings is a bitmask of VDRV_UART_SETTING_*
#define VDRV_CTL_OP_SET_CTS_TIMEOUT				0x1002		// param: const uint32_t* ctsTimeoutInMsecP		used to limit how long TX takes
#define VDRV_CTL_OP_CLEAR_ERR					0x1003		// param: ignored								clears current port error state
#define VDRV_CTL_OP_SLEEP						0x1004		// param: ignored								put the port to sleep
#define VDRV_CTL_OP_WAKE						0x1005		// param: ignored								wake port up after VDRV_CTL_OP_SLEEP
#define VDRV_CTL_OP_GET_TX_FIFO_FREE			0x1006		// param: uint32_t *txFifoFreeBytes;			writes into the provided ptr how many bytes are DEFEINITELY free in TX fifo (it is OK to under-estimate, not OK to over-estimate)
#define VDRV_CTL_OP_START_BREAK					0x1007		// param: ignored								begin sending a break (lower TX line)
#define VDRV_CTL_OP_STOP_BREAK					0x1008		// param: ignored								finish sending a break (re-raise TX line)
#define VDRV_CTL_OP_LOOPBACK_ENABLE				0x1009		// param: ignored								enable loopback mode
#define VDRV_CTL_OP_LOOPBACK_DISABLE			0x100a		// param: ignored								disable loopback mode
#define VDRV_CTL_OP_FLUSH_TX_FIFO				0x100b		// param: ignored								flush HW TX FIFO. Ignores all timouts. Blocks till done
#define VDRV_CTL_OP_FLUSH_RX_FIFO				0x100c		// param: ignored								flush HW RX FIFO. Data is discarded
#define VDRV_CTL_OP_SEND_QUEUED_DATA			0x100d		// param: ignored								same as VDRV_CTL_OP_FLUSH_TX_FIFO, but honors CTS timeout and may return serErrTimeOut
//0x100e not seen
#define VDRV_CTL_OP_GET_BEST_TX_BLOCK_SZ		0x100f		// param: uint32_t *bestTxBufSzP;				writes into the provided ptr how many bytes at a time the driver prefers to get in a call to VdrvWriteF(). Advisory at best
//0x1010 not seen
#define VDRV_CTL_OP_GET_MAX_RX_BLOCK_SZ			0x1011		// param: uint32_t *maxRxBufSzP;				writes into the provided ptr how many bytes at most the RX HW will accept
#define VDRV_CTL_OP_NOTIF_RX_BYTES_CONSUMED		0x1012		// param: ignored								tells driver that SW RX queue may now have more free space (someone consumed some bytes). A call to RecvQueueGetQueueSpaceF() is expected
#define VDRV_CTL_OP_SET_DTR						0x1013		// param: const uint8_t *dtrStateP				sets DTR pin to passed state. palm serial dtr doesnt seem to support this. Will return serErrNotSupported if value is zero
#define VDRV_CTL_OP_GET_DTR						0x1014		// param: uint8_t *dtrStateP					gets DTR state. palm serial dtr doesnt seem to support this. Always sets value to 1
#define VDRV_CTL_OP_WAIT_FOR_CONFIG				0x1015		// param: ignored								block until port is configured. not used for hardware ports. used for things like usb that need external config
#define VDRV_CTL_OP_USB_GET_DEVICE_DESCR		0x1016		// param: not sure								allegedly used for usb. palm's usb driver returns serErrNotSupported
#define VDRV_CTL_OP_USB_GET_CONFIG_DESCR		0x1017		// param: not sure								allegedly used for usb. palm's usb driver returns serErrNotSupported
#define VDRV_CTL_OP_IRDA_ENABLE					0x1018		// param: ignored								enable IrDA mode on the port
#define VDRV_CTL_OP_IRDA_DISABLE				0x1019		// param: ignored								disable IrDA mode on the port
#define VDRV_CTL_OP_IR_RX_ENA					0x101a		// param: ignored								used for IrDA
#define VDRV_CTL_OP_IR_RX_DIS					0x101b		// param: ignored								used for IrDA
#define VDRV_CTL_OP_IRQ_DISABLE					0x101c		// param: ignored								disables port IRQ and thus all forms of RX
#define VDRV_CTL_OP_IRQ_ENABLE					0x101d		// param: ignored								enabled port IRQ and thus enables RX. Only useful after VDRV_CTL_OP_IRQ_DISABLE
#define VDRV_CTL_OP_SET_NEW_RECV_QUEUE			0x101e		// param: struct VdrvRecvQueue*					immediately replces the SW RX quque used by the driver 
#define VDRV_CTL_OP_GET_IR_SPEED_BITMAPS		0x101f		// param: uint16_t* outValP						IrDA driver writes 62 into it, all others return serErrNotSupported. bitmask of VDRV_IR_SPEED_*

#define VDRV_CTL_OP_USER_DEF					0x2000		// param depends on driver and request

//custom ops palm implemented
#define VDRV_CTL_OP_RTS_RAW_STOP				0x8001		// param: ignored								disable RTS flow control, and signal RTS into "DO NOT send me data" state
#define VDRV_CTL_OP_RTS_RAW_START				0x8002		// param: ignored								disable RTS flow control, and signal RTS into "DO send me data" state

//custom ops palm implemented for BT UART
#define VDRV_CTL_OP_BT_ANOTHER_GPIO_ON			0x8003		// param: ignored								sets GPIO 83 (on z72)
#define VDRV_CTL_OP_BT_ANOTHER_GPIO_OFF			0x8004		// param: ignored								clears GPIO 83 (on z72)
#define VDRV_CTL_OP_BT_POWER_OFF				0x8005		// param: ignored								disables power to BT unit
#define VDRV_CTL_OP_BT_POWER_ON					0x8006		// param: ignored								enables power to BT unit
#define VDRV_CTL_OP_BT_SOME_GPIO_ON				0x8007		// param: ignored								sets GPIO 33 (on z72)
#define VDRV_CTL_OP_BT_SOME_GPIO_OFF			0x8008		// param: ignored								clears GPIO 33 (on z72)



#define VDRV_IR_SPEED_2K2				0x00000001
#define VDRV_IR_SPEED_9K6				0x00000002
#define VDRV_IR_SPEED_19K2				0x00000004
#define VDRV_IR_SPEED_38K4				0x00000008
#define VDRV_IR_SPEED_57K6				0x00000010
#define VDRV_IR_SPEED_115K				0x00000020
#define VDRV_IR_SPEED_576K				0x00000040
#define VDRV_IR_SPEED_1M152				0x00000080
#define VDRV_IR_SPEED_4M				0x00000100


struct VdrvRecvQueue {
	void *userData;		//passed to each callback
	Err (*RecvQueueByteRxedF)(void *userData, uint8_t byte, uint32_t lineErrors /* VDRV_LINE_ERR_* bitfield */);
	Err (*RecvQueueBlockRxedF)(void *userData, const uint8_t *bytes, uint32_t nbytes, uint32_t lineErrors /* VDRV_LINE_ERR_* bitfield */);
	uint32_t (*RecvQueueGetQueueSizeF)(void *userData);	//how big is the queue?
	uint32_t (*RecvQueueGetQueueSpaceF)(void *userData);	//how much space is there in the queue? MUST BE at most getQueueSizeF() - 1
};

struct VdrvConfig {
	uint32_t baudrate;			//requested baudrate
	uint32_t portName;			//device "name". Selects the port to open
	uint32_t function;			//extra param for port opening type. Used by USB driver sometimes
	void *drvData;				//driver specific an dunused
	uint16_t drvDataSz;			//size of the above data (also seems to be unused)
};

struct VdrvPortFuncs {
	Err (*VdrvOpenF)(void **drvrDataP /* OUT */, const struct VdrvConfig *cfg, const struct VdrvRecvQueue* recvQ);
	Err (*VdrvCloseF)(void *drvrData);
	Err (*VdrvControlF)(void *drvrData, uint32_t controlOp /* VDRV_CTL_OP_* */, void* data, uint16_t *controlDataLenP /* never used */);
	Err (*VdrvGetStatusF)(void *drvrData, uint16_t* statusP /* VDRV_STA_* bitfield */ );
	uint32_t (*VdrvWriteF)(void *drvrData, const void *buf, uint32_t size, Err *errP);
	Err (*VdrvCustomControlF)(void *drvrData, ...);	//always returns serErrNotSupported
};

struct VdrvPortInfo {
	uint32_t portName;			//OUT:	device "name"
	uint32_t driverVer;			//OUT:	driver version: VDRV_VERSION_5
	uint32_t maxBaudrate;		//OUT:	maximum baudrate supported by the port
	uint32_t handshakeBaudrate;	//OUT:	baudrate at which HW handshaking is suggested
	uint32_t portFlags;			//OUT:	VDRV_PORT_FLAG* bitmask
	char *portNameStr;			//IN:	pointer to buffer where we'll write a zero-terminated port name if NON-NULL. Buffer is 64 bytes
	uint32_t dbCrid;			//OUT:	driver databse creator ID
	uint8_t portIdx;			//IN:	tells what port undex we want the info on
};



//main entry point (Exported as enrty #0). PilotMain() is unused
typedef Err (*VdrvExportedFunc0)(uint32_t code /* VDRV_ENTRY_NUM_* */, void* data);


#endif
