#include <boot.h>	//must be first

#include <Extensions/ExpansionMgr/SlotDrvrLib.h>
#include <kal.h>
#include <dal.h>
#include <StringMgr.h>
#include <string.h>
#include "../LibFTL/ftl.h"
#include "nandPublic.h"
#include "common.h"
#include "printf.h"
#include "slot.h"
#include "nand.h"


struct Globals {								//limited to 0x2c bytes by 'amdi' resource. change that to get more
	
	uint32_t			mutex;					//only protects "curSec" & "buf"
	uint8_t 			*buf;
	uint32_t			curSec;
	
	MemHandle		 	ftl;
	struct NandFuncs	*funcs;					//just so we can free it
	uint16_t			slotRefNum;
	uint16_t			ftlSecSize;				//aka 512 << g->stdSecPerFtlSecShift
	uint8_t				stdSecPerFtlSecShift;
	uint8_t				stdSecPerFtlSec;		//aka 1 << stdSecPerFtlSecShift
	
	uint8_t				uid[17];				//one extra to make over-read easy
	uint8_t 			haveUid	: 1;
	uint8_t				mounted	: 1;
};


//from disassembly of TX's FAT32
#define TRIM_CUSTOM_CONTROL_CRID		CREATE_4CC('e','x','v','f')
#define TRIM_CUSTOM_CONTROL_SEL			0x1000

struct TrimRequest {		//an array of these is passed, length detemined by length passed to custom control
	uint32_t firstSec;
	uint32_t numSec;
};


static struct Globals* globalsGet(void)
{
	void**** ret;
	
	asm ("mov %0, r9":"=r"(ret));
	return (struct Globals*)&ret[0][MY_LIB_ID / 4][0x20/4];
}

static bool slotPrvNandBlockErase(struct NandFuncs *funcs, uint32_t firstPageNum)
{
	return nandBlockErase(firstPageNum);
}

static int_fast8_t slotPrvNandPageReadPartial(struct NandFuncs *funcs, uint32_t pageNum, void *data, uint32_t ofst, uint32_t len, bool withEcc)
{
	return nandPageReadPartial(pageNum, data, ofst, len, withEcc);
}

static bool slotPrvNandPageWritePartial(struct NandFuncs *funcs, uint32_t pageNum, const void *data, uint32_t ofst, uint32_t len, bool withEcc)
{
	return nandPageWritePartial(pageNum, data, ofst, len, withEcc);
}

Err impl_SlotLibOpen(uint16_t slotLibRefNum)
{
	uint32_t ftlSecSz, stdSecPerFtlSec;
	struct Globals *g = globalsGet();
	struct NandSpec *spec;
	Err e;
	
	g->curSec = 0xffffffff;
	
	e = KALMutexCreate(&g->mutex, CREATE_4CC('s','d','D','r'));
	if (e != errNone)
		goto out_err;

	logt("%s(slotLibRefNum=0x%04x)\n", __func__, slotLibRefNum);
	if (!nandInit()) {
		logw("nand init failed\n");
		goto out_nand;
	}

	logt("flash inited\n");
	
	//read uid while nobody is using the nand
	g->haveUid = nandReadUid(g->uid);
	g->uid[16] = 0;
	
	g->funcs = MemChunkNew(0, sizeof(struct NandFuncs) + sizeof(struct NandSpec), 0x200);
	if (!g->funcs)
		goto out_alloc;
	
	spec = (struct NandSpec*)(g->funcs + 1);
	g->funcs->NandBlockErase = slotPrvNandBlockErase;
	g->funcs->NandPageReadPartial = slotPrvNandPageReadPartial;
	g->funcs->NandPageWritePartial = slotPrvNandPageWritePartial;
	nandFillSpec(spec);
	
	g->ftl = ftlInit(spec, g->funcs);
	if (!g->ftl)
		goto out_ftl;
	
	g->ftlSecSize = ftlSecSz = (uint16_t)ftlGetInfo(g->ftl, FtlGeometry);
	if (!ftlSecSz || (ftlSecSz % slotSectorSize))
		goto out_ftl;
	g->stdSecPerFtlSec = stdSecPerFtlSec = ftlSecSz / slotSectorSize;
	if (stdSecPerFtlSec & (stdSecPerFtlSec - 1))		//must be a power of 2
		goto out_ftl;
	
	g->stdSecPerFtlSecShift = 31 - __builtin_clz(stdSecPerFtlSec);
	logt("stdSecPerFtlSecShift = %u\n", g->stdSecPerFtlSecShift);
	
	//alloc our buf
	g->buf = MemChunkNew(0, ftlSecSz, 0x1200);
	if (!g->buf)
		goto out_mem;
	
	e = ExpSlotRegister(slotLibRefNum, &g->slotRefNum);
	if (e != errNone)
		goto out_exp;
	
	logt("slot registered: slot handle %u\n", g->slotRefNum);
	
	g->mounted = ftlMount(g->ftl);
	if (!g->mounted)
		logw("ftl mount failed\n");
	
	(void)ExpCardInserted(g->slotRefNum);

	return errNone;

out_exp:
	MemChunkFree(g->buf);
	g->buf = NULL;
	
out_mem:
	MemChunkFree(g->funcs);
	g->funcs = NULL;

out_ftl:
	//nothing

out_alloc:
	nandDeinit();

out_nand:
	if (g->mutex)
		KALMutexDelete(g->mutex);
	g->mutex = 0;

out_err:
	return 0xffff;
}

Err impl_SlotLibClose(uint16_t slotLibRefNum)
{
	struct Globals *g = globalsGet();
	
	if (g->mutex)
		KALMutexReserve(g->mutex, -1);
	
	if (g->mounted) {
		
		ftlUnmount(g->ftl, true);
		g->mounted = false;
	}
	
	if (g->ftl) {
		
		ftlFree(g->ftl);
		g->ftl = NULL;
	}

	if (g->slotRefNum)
		ExpSlotUnregister(g->slotRefNum);
	g->slotRefNum = 0;
	
	MemChunkFree(g->buf);
	g->buf = NULL;
	
	MemChunkFree(g->funcs);
	g->funcs = NULL;
	
	nandDeinit();
	
	if (g->mutex)
		KALMutexDelete(g->mutex);
	g->mutex = 0;

	return errNone;
}

Err impl_SlotLibSleep(uint16_t slotLibRefNum)
{
	struct Globals *g = globalsGet();
	
	logt("%s\n", __func__);
	if (g->mounted)
		ftlSleepNotify(g->ftl);
	nandSleep();
	return errNone;
}

Err impl_SlotLibWake(uint16_t slotLibRefNum)
{
	logt("%s\n", __func__);
	nandWake();
	return errNone;
}
	
uint32_t impl_SlotLibLibAPIVersion(uint16_t slotLibRefNum)
{
	logt("%s\n", __func__);
	return slotDrvrAPIVersion;
}

Err slotPrvTrim(uint32_t firstSec, uint32_t numSec)
{
	struct Globals *g = globalsGet();
	
	uint32_t firstFtlSec = (firstSec + g->stdSecPerFtlSec - 1) >> g->stdSecPerFtlSecShift;
	uint32_t pastEndFtlSec = (firstSec + numSec) >> g->stdSecPerFtlSecShift;
	uint32_t numFtlSecs = pastEndFtlSec - firstFtlSec;
	
	if (!g->mounted)
		return expErrCardBadSector;
	
	logt("%s(%u+%u)\n", __func__, firstSec, numSec);
	
	while (numFtlSecs--) {
		uint32_t sec = firstFtlSec++;
		
		if (!ftlTrim(g->ftl, sec))
			logi("Failed to trim FTL sec %u\n", sec);
	}

	return errNone;
}

static uint32_t slotPrvRead32from68k(const void *ptr)	//68k uint32s are 2-byte aligned
{
	const uint16_t *v = (const uint16_t*)ptr;
	
	return __builtin_bswap32((((uint32_t)v[1]) << 16) + v[0]);
}

//valueLenP will be swapped and aligned, valueP may not be
static Err slotPrvCustomControl(uint16_t slotLibRefNum, uint32_t apiCreator, uint16_t apiSelector, void *valueP, uint16_t *valueLenP, bool fromPace)
{
	struct Globals *g = globalsGet();
	
	logt("%s("FMT_4CC",0x%04x,ptr,&(0x%04x), %s)\n", __func__, CNV_4CC(apiCreator), apiSelector, valueLenP ? *valueLenP : 0xffff, fromPace ? "from68k" : "fromARM");
	
	if (apiCreator == TRIM_CUSTOM_CONTROL_CRID && apiSelector == TRIM_CUSTOM_CONTROL_SEL && valueP && valueLenP && !(*valueLenP % sizeof(struct TrimRequest))) {
		
		const struct TrimRequest *req = (const struct TrimRequest*)valueP;
		uint32_t numReqd = *valueLenP / sizeof(struct TrimRequest);
		Err e;
		
		while (numReqd--) {
			
			uint32_t first = fromPace ? slotPrvRead32from68k(&req->firstSec) : req->firstSec;
			uint32_t num = fromPace ? slotPrvRead32from68k(&req->numSec) : req->numSec;
			
			e = slotPrvTrim(first, num);
			if (e != errNone)
				break;
			
			req++;
		}
		
		return e;
	}
	if (apiCreator == NAND_SLOT_CUSTOM_CONTROL_API_CRID && apiSelector == NAND_SLOT_CUSTOM_CONTROL_SEL_GET_INFO && valueP && valueLenP && *valueLenP >= sizeof(struct NandGetInfoRequest)) {
		
		struct NandGetInfoRequest *req = (struct NandGetInfoRequest*)valueP;
		uint_fast8_t i;
		uint64_t ret;
		
		*valueLenP = sizeof(struct NandGetInfoRequest);
		
		if (!g->mounted)
			return expErrCardBadSector;
	
		ret = ftlGetInfo(g->ftl, (enum FtlDataType)req->infoType);

		if (fromPace) {
			
			uint8_t *dst = (uint8_t*)&req->value;
			
			for (i = 0; i < 8; i++, ret >>= 8)
				dst[7 - i] = ret;
		}
		else
			req->value = ret;
		
		return errNone;
	}
	
	logt("Unexpected custom control %u\n", apiSelector);
	
	return expErrUnsupportedOperation;
}

Err slotCustomControlFrom68k(uint16_t slotLibRefNum, uint32_t apiCreator, uint16_t apiSelector, void *valueP, uint16_t *valueLenP)
{
	return slotPrvCustomControl(slotLibRefNum, apiCreator, apiSelector, valueP, valueLenP, true);
}

Err impl_SlotLibCustomControl(uint16_t slotLibRefNum, uint32_t apiCreator, uint16_t apiSelector, void *valueP, uint16_t *valueLenP)
{
	return slotPrvCustomControl(slotLibRefNum, apiCreator, apiSelector, valueP, valueLenP, false);
}

Err impl_SlotLibCardPresent(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	logt("%s\n", __func__);
	
	return errNone;
}

Err impl_SlotLibCardInfo(uint16_t slotLibRefNum, uint16_t slotRefNum, ExpCardInfoType *infoP)
{
	struct Globals *g = globalsGet();
	uint16_t life;
	
	logt("%s\n", __func__);
	
	life = ((uint32_t)ftlGetInfo(g->ftl, FtlGetLifetimeLeft) + 50) / 100;

	infoP->capabilityFlags = expCapabilityHasStorage;
	StrCopy(infoP->manufacturerStr, "Dmitry.GR");
	StrPrintF(infoP->productStr, g->mounted ? "%u.%02u%% NAND wear left" : "", life / (uint16_t)100, life % (uint16_t)100);
	StrCopy(infoP->deviceClassStr, NAND_SLOT_DEVICE_CLASS);
	
	if (!g->haveUid)
		infoP->deviceUniqueIDStr[0] = 0;
	else {
		static const char snumChars[48] = "123456789ACDEFGHJKLMNPQRSTUVWXYZabdefghjknqrtwyz";
		uint_fast8_t haveBytes = 16, curByte = 0, curPosOut = 0;
		uint_fast16_t curReservoir = 0, curTracker = 0;
		
		while (haveBytes || curTracker) {
			
			if (curTracker < sizeof(snumChars) && haveBytes) {
				
				curTracker = curTracker * 256 + 255;
				curReservoir = curReservoir * 256 + g->uid[curByte++];
				haveBytes--;
			}
			
			infoP->deviceUniqueIDStr[curPosOut++] = snumChars[curReservoir % sizeof(snumChars)];
			curReservoir /= sizeof(snumChars);
			curTracker /= sizeof(snumChars);
		}
		
		infoP->deviceUniqueIDStr[curPosOut++] = 0;
	}
	
	return errNone;
}

Err impl_SlotLibCardMediaType(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t *mediaTypeP)
{
	logt("%s\n", __func__);
	
	if (mediaTypeP)
		*mediaTypeP = NAND_SLOT_MEDIA_TYPE;
	
	return errNone;
}

Err impl_SlotLibMediaType(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t *mediaTypeP)
{
	logt("%s\n", __func__);
	
	if (mediaTypeP)
		*mediaTypeP = NAND_SLOT_MEDIA_TYPE;
	
	return errNone;
}

Err impl_SlotLibCardReserve(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	logt("%s\n", __func__);
	return errNone;
}

Err impl_SlotLibCardRelease(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	logt("%s\n", __func__);
	return errNone;
}

Err impl_SlotLibCardGetSerialPort(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t* portP)
{
	if (!portP)
		return sysErrParamErr;
	
	*portP = 0;
	return expErrUnsupportedOperation;
}

Boolean impl_SlotLibCardIsFilesystemSupported(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t filesystemType)
{
	logt("%s\n", __func__);
	return filesystemType == 'vfat' || filesystemType == 'fatf';
}

Err impl_SlotLibCardMetrics(uint16_t slotLibRefNum, uint16_t slotRefNum, CardMetricsPtr cardMetricsP)
{
	struct Globals *g = globalsGet();
	uint32_t nsec;
	
	logt("%s\n", __func__);
	if (g->mounted)
		nsec = ((uint32_t)(ftlGetInfo(g->ftl, FtlGeometry) >> 32)) << g->stdSecPerFtlSecShift;
	else
		nsec = 128;	//eh, why not?
	
	//this is a pain since palm's FS driver does not permit us to get it to properly align things
	//we are forced to reverse-engineer it to make ti all work. Things we CAN drect affect:
	// cardMetricsP->sectorsPerBlock will become the cluster size
	//that is all we have direct control over. But also we can affect:
	// cardMetricsP->partitionStart and cardMetricsP->partitionSize will be honoured
	//we need to work backwards from the math in the driver to set them
	//the math differs for fat12 vs fat16 too, for more fun
	//Palm driver DOES always create 512 root dir entries (32 sectors)
	//the logic also differes between drivers. this is best suited for palm's fat32 driver
	
	//we want clusters to align with flash
	uint32_t secPerCluster = g->stdSecPerFtlSec;
	
	//figure out the first sector past the end of our device and align it
	uint32_t pastEnd = (nsec / secPerCluster) * secPerCluster;
	
	//how many sectors of data COULD we have? remember that the MBR needs one
	uint32_t totalAvailSecsInPartition = pastEnd - 1;
	
	//sort out sectors left over for FAT tables and data clusters (subtract other partition necessities)
	uint32_t secsForFatsAndData = totalAvailSecsInPartition - 32 /* root dir size*/ - 1 /*PBR */;
	
	//pick a fat type based on this heuristic
	bool fat12 = nsec <= 0x8000;
	
	//calc num sectors each fat will use
	uint32_t intermediate = fat12 ? 
								secPerCluster * slotSectorSize * 2 :
								secPerCluster * (slotSectorSize / 2);
	
	uint32_t secsPerFat = fat12 ?
								(intermediate + 5 + 3 * (secsForFatsAndData + 2 * secPerCluster)) / (intermediate + 6):
								(intermediate + 1 + secsForFatsAndData + secPerCluster * 2) / (intermediate + 2);
	
	//calc how many data clusters there'll be
	uint32_t numDataClusters = (secsForFatsAndData - secsPerFat * 2) / secPerCluster;
	
	//calc where first data cluster should start
	uint32_t dataStartSec = pastEnd - numDataClusters * secPerCluster;
	
	//before it comes the root directory
	uint32_t rootDirStartSec = dataStartSec - 32;
	
	//before that come two fats
	uint32_t fatsStartSec = rootDirStartSec - 2 * secsPerFat;
	
	//before that comes the PBR
	uint32_t pbrSec = fatsStartSec - 1;
	
	//and now we can calc the partition size
	uint32_t partSize = pastEnd - pbrSec;
	
	//export metrics such that we align best with 
	MemSet(cardMetricsP, sizeof(*cardMetricsP), 0);
	cardMetricsP->totalSectors		= nsec;
	cardMetricsP->bytesPerSector	= slotSectorSize;
	cardMetricsP->bootIndicator		= 0xFF;
	cardMetricsP->partitionStart	= pbrSec;
	cardMetricsP->partitionSize		= partSize;
	cardMetricsP->sectorsPerBlock	= secPerCluster;
	
	return errNone;
}


Err impl_SlotLibCardLowLevelFormat(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	struct Globals *g = globalsGet();
	
	logt("%s\n", __func__);
	
	if (g->mounted) {
		logt(" first unmount\n");
		ftlUnmount(g->ftl, false /* why bother? */);		//if they existed, stats will be kept
	}
	
	if (!ftlLowLevelFormat(g->ftl)) {
		logt(" LL format failed\n");
		return expErrCardBadSector;
	}
	
	g->mounted = ftlMount(g->ftl);
	logt(" remount = %d\n", g->mounted);
	
	return g->mounted ? errNone : expErrCardBadSector;
}

Err impl_SlotLibPowerCheck(uint16_t slotLibRefNum, uint16_t slotRefNum, uint16_t operationFlags, uint16_t readBlocks, uint16_t writeBlocks)
{
	return errNone;
}

static bool slotPrvFtlWrite(struct Globals *g, uint32_t ftlSecNum, const void *buf)
{
	const uint8_t *src = (const uint8_t*)buf;
	uint32_t i;
	
	if (buf) {
		for (i = 0; i < g->ftlSecSize; i++) {
			
			if (*src++) { //nozero found
				
				logt(" W %u\n", ftlSecNum);
				
				return ftlWrite(g->ftl, ftlSecNum, buf);
			}
		}
	}
	
	//none found - trim
	logt(" T %u\n", ftlSecNum);
	return ftlTrim(g->ftl, ftlSecNum);
}

static bool slotPrvFtlRead(struct Globals *g, uint32_t ftlSecNum, void *buf)
{
	logt(" R %u\n", ftlSecNum);
	
	return ftlRead(g->ftl, ftlSecNum, buf);
}

static bool slotPrvReadIntoBuf(struct Globals *g, uint32_t ftlSecNum)
{
	if (g->curSec == ftlSecNum)
		return true;
	
	if (slotPrvFtlRead(g, ftlSecNum, g->buf)) {
		
		g->curSec = ftlSecNum;
		return true;
	}
	else {
		
		g->curSec = 0xffffffff;
		return false;
	}
}

static bool slotPrvWriteFromBuf(struct Globals *g, uint32_t ftlSecNum)
{
	if (slotPrvFtlWrite(g, ftlSecNum, g->buf)) {
		
		g->curSec = ftlSecNum;
		return true;
	}
	else {
		
		g->curSec = 0xffffffff;
		return false;
	}
}

static char* tohex(char *p, uint32_t val, uint32_t len)
{
	char *ret = p + len;
	
	while (len) {
		
		static const char* hexch = "0123456789abcdef";
		
		p[--len] = hexch[val % 16];
		val /= 16;
	}
	
	return ret;
}

static void showSec(const void *ptr)
{
	const uint8_t *src = ptr;
	char str[64], *p;
	uint32_t len;
	
	for (len = 0; len < 512; len++) {
		
		if (!(len % 16))
			p = tohex(str, len, 4);
		
		*p++ = ' ';
		p = tohex(p, *src++, 2);
		
		if (len % 16 == 15) {
			*p = 0;
			logt("%s\n", str);
		}
	}
}

Err impl_SlotLibCardSectorRead(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t sectorNumber, uint8_t *bufferP, uint32_t *numSectorsP)
{
	uint32_t cardNsec, numDesired, numDone, ofst, now;
	struct Globals *g = globalsGet();
	bool ok;
	
	//sanity
	if (!bufferP || !numSectorsP)
		return sysErrParamErr;
	
	logt("%s(%u+%u)\n", __func__, sectorNumber, *numSectorsP);
	
	//mount check
	if (!g->mounted)
		return expErrCardBadSector;
	
	//boundaries
	cardNsec = ((uint32_t)(ftlGetInfo(g->ftl, FtlGeometry) >> 32)) << g->stdSecPerFtlSecShift;
	if (sectorNumber >= cardNsec)
		*numSectorsP = 0;
	else if (*numSectorsP > cardNsec - sectorNumber)
		*numSectorsP = cardNsec - sectorNumber;
	if (!*numSectorsP)
		return errNone;
	
	numDesired = *numSectorsP;
	numDone = 0;
	
	ofst = sectorNumber & (g->stdSecPerFtlSec - 1);
	if (ofst) {
		
		now = g->stdSecPerFtlSec - ofst;
		if (now > numDesired)
			now = numDesired;
		
		KALMutexReserve(g->mutex, -1);
		ok = slotPrvReadIntoBuf(g, sectorNumber >> g->stdSecPerFtlSecShift);
		if (ok) {
			MemMove(bufferP, g->buf + slotSectorSize * ofst, now * slotSectorSize);
			bufferP += slotSectorSize * now;
			numDone += now;
			numDesired -= now;
			sectorNumber += now;
		}
		KALMutexRelease(g->mutex);
		
		if (!ok)
			goto out;
	}
	
	//fully aligned blocks
	while (numDesired >= g->stdSecPerFtlSec) {
		
		ok = slotPrvFtlRead(g, sectorNumber >> g->stdSecPerFtlSecShift, bufferP);
		if (!ok)
			goto out;
		
		bufferP += slotSectorSize * g->stdSecPerFtlSec;
		numDone += g->stdSecPerFtlSec;
		numDesired -= g->stdSecPerFtlSec;
		sectorNumber += g->stdSecPerFtlSec;
	}
	
	//any leftover partial blocks?
	if (numDesired) {
		
		KALMutexReserve(g->mutex, -1);
		ok = slotPrvReadIntoBuf(g, sectorNumber >> g->stdSecPerFtlSecShift);
		if (ok) {
			MemMove(bufferP, g->buf, numDesired * slotSectorSize);
			bufferP += slotSectorSize * numDesired;
			numDone += numDesired;
			numDesired -= numDesired;
			sectorNumber += numDesired;
		}
		KALMutexRelease(g->mutex);
	}
	
out:
	*numSectorsP = numDone;
	return numDone ? errNone : 0xffff;
}

Err impl_SlotLibCardSectorWrite(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t sectorNumber, const uint8_t *bufferP, uint32_t *numSectorsP)
{
	uint32_t cardNsec, numDesired, numDone, ofst, now;
	struct Globals *g = globalsGet();
	bool ok;
	
	//sanity
	if (!numSectorsP)
		return sysErrParamErr;

	logt("%s(%u+%u)\n", __func__, sectorNumber, *numSectorsP);
	
	//mount check
	if (!g->mounted)
		return expErrCardBadSector;
	
	//boundaries
	cardNsec = ((uint32_t)(ftlGetInfo(g->ftl, FtlGeometry) >> 32)) << g->stdSecPerFtlSecShift;
	if (sectorNumber >= cardNsec)
		*numSectorsP = 0;
	else if (*numSectorsP > cardNsec - sectorNumber)
		*numSectorsP = cardNsec - sectorNumber;
	if (!*numSectorsP)
		return errNone;
	
	numDesired = *numSectorsP;
	numDone = 0;
	
	logt("WRITE DATA:\n");
	if (bufferP)
		showSec(bufferP);
	else
		logt("NULLS\n");
	
	//unaligned (with ftl sector size) at start?
	ofst = sectorNumber & (g->stdSecPerFtlSec - 1);
	if (ofst) {
		
		now = g->stdSecPerFtlSec - ofst;
		if (now > numDesired)
			now = numDesired;
		
		KALMutexReserve(g->mutex, -1);
		ok = slotPrvReadIntoBuf(g, sectorNumber >> g->stdSecPerFtlSecShift);
		if (ok) {
			
			if (bufferP)
				MemMove(g->buf + slotSectorSize * ofst, bufferP, now * slotSectorSize);
			else
				MemSet(g->buf + slotSectorSize * ofst, now * slotSectorSize, 0);
			
			ok = slotPrvWriteFromBuf(g, sectorNumber >> g->stdSecPerFtlSecShift);
			if (ok) {
				if (bufferP)
					bufferP += slotSectorSize * now;
				numDone += now;
				numDesired -= now;
				sectorNumber += now;
			}
		}
		KALMutexRelease(g->mutex);
		
		if (!ok)
			goto out;
	}
	
	//fully aligned blocks
	while (numDesired >= g->stdSecPerFtlSec) {
		
		uint32_t ftlSec = sectorNumber >> g->stdSecPerFtlSecShift;
		
		if (ftlSec == g->curSec)
			g->curSec = 0xffffffff;	//our buffer is no longer valid
		
		ok = slotPrvFtlWrite(g, ftlSec, bufferP);
		if (!ok)
			goto out;
		
		if (bufferP)
			bufferP += slotSectorSize * g->stdSecPerFtlSec;
		numDone += g->stdSecPerFtlSec;
		numDesired -= g->stdSecPerFtlSec;
		sectorNumber += g->stdSecPerFtlSec;
	}
	
	//any leftover partial blocks?
	if (numDesired) {
		
		KALMutexReserve(g->mutex, -1);
		ok = slotPrvReadIntoBuf(g, sectorNumber >> g->stdSecPerFtlSecShift);
		if (ok) {
			if (bufferP)
				MemMove(g->buf, bufferP, numDesired * slotSectorSize);
			else
				MemSet(g->buf, numDesired * slotSectorSize, 0);
			
			ok = slotPrvWriteFromBuf(g, sectorNumber >> g->stdSecPerFtlSecShift);
			if (ok) {
				if (bufferP)
					bufferP += slotSectorSize * numDesired;
				numDone += numDesired;
				numDesired -= numDesired;
				sectorNumber += numDesired;
			}
		}
		KALMutexRelease(g->mutex);
	}
	
out:
	*numSectorsP = numDone;
	return numDone ? errNone : 0xffff;
}


