#include <boot.h>
#include "stm32f469xx.h"
#include <kal.h>
#include <string.h>
#include "printf.h"
#include "nand.h"


#define GPIO_PULL_NONE		0
#define GPIO_PULL_UP		1
#define GPIO_PULL_DOWN		2

#define QSPI_MICRON_CFG_MODE_NORMAL				0x00
#define QSPI_MICRON_CFG_MODE_OTP_PARAM_UNIQ		0x02
#define QSPI_MICRON_CFG_MODE_NOR_ENA			0x05
#define QSPI_MICRON_CFG_MODE_OTP_PROT			0x06
#define QSPI_MICRON_CFG_MODE_PERM_BLOCK_LOCK	0x07

#define QSPI_MICRON_PAGE_NO_UID					0x00
#define QSPI_MICRON_PAGE_NO_PARAMS				0x01

#define QSPI_MICRON_FTR_BLOCK_LOCK				0xA0
#define QSPI_MICRON_FTR_CONFIG					0xB0
#define QSPI_MICRON_FTR_STATUS					0xC0
#define QSPI_MICRON_FTR_DIE_SELECT				0xD0

#define QSPI_MICRON_CMD_WRITE_DISABLE			0x04
#define QSPI_MICRON_CMD_WRITE_ENABLE			0x06
#define QSPI_MICRON_CMD_GET_FEATURES			0x0F
#define QSPI_MICRON_CMD_PROGRAM_EXECUTE			0x10
#define QSPI_MICRON_CMD_PAGE_READ				0x13
#define QSPI_MICRON_CMD_SET_FEATURES			0x1F
#define QSPI_MICRON_CMD_PRGM_LOAD_x4			0x32
#define QSPI_MICRON_CMD_READ_ID					0x9F
#define QSPI_MICRON_CMD_ERASE_BLOCK				0xD8
#define QSPI_MICRON_CMD_READ_CACHE_QUAD			0xEB

//please remember the erratum "First nibble of data is not written after dummy phase"!!!
#define QSPI_MODE_READ					QUADSPI_CCR_FMODE_0
#define QSPI_MODE_WRITE					0
#define QSPI_MODE_AUTO_POLLING			QUADSPI_CCR_FMODE_1
#define QSPI_NO_DATA					0
#define QSPI_DATA_ON_1_LINE				QUADSPI_CCR_DMODE_0
#define QSPI_DATA_ON_4_LINES			(QUADSPI_CCR_DMODE_0 | QUADSPI_CCR_DMODE_1)
#define QSPI_NO_DUMMY_CYCLES			0
#define QSPI_2_DUMMY_CYCLES				QUADSPI_CCR_DCYC_1
#define QSPI_4_DUMMY_CYCLES				QUADSPI_CCR_DCYC_2
#define QSPI_8_DUMMY_CYCLES				QUADSPI_CCR_DCYC_3
#define QSPI_NO_ALTERNATE_BYTES			0
#define QSPI_1_ALTERNATE_BYTE			0
#define QSPI_2_ALTERNATE_BYTES			QUADSPI_CCR_ABSIZE_0
#define QSPI_3_ALTERNATE_BYTES			QUADSPI_CCR_ABSIZE_1
#define QSPI_4_ALTERNATE_BYTES			(QUADSPI_CCR_ABSIZE_0 | QUADSPI_CCR_ABSIZE_1)
#define QSPI_ALT_BYTES_ON_1_LINE		QUADSPI_CCR_ABMODE_0
#define QSPI_ALT_BYTES_ON_4_LINES		(QUADSPI_CCR_ABMODE_0 | QUADSPI_CCR_ABMODE_1)
#define QSPI_NO_ADDR_BYTES				0
#define QSPI_1_ADDR_BYTE				0
#define QSPI_2_ADDR_BYTES				QUADSPI_CCR_ADSIZE_0
#define QSPI_3_ADDR_BYTES				QUADSPI_CCR_ADSIZE_1
#define QSPI_4_ADDR_BYTES				(QUADSPI_CCR_ADSIZE_0 | QUADSPI_CCR_ADSIZE_1)
#define QSPI_ADDR_BYTES_ON_1_LINE		QUADSPI_CCR_ADMODE_0
#define QSPI_ADDR_BYTES_ON_2_LINES		QUADSPI_CCR_ADMODE_1
#define QSPI_ADDR_BYTES_ON_4_LINES		(QUADSPI_CCR_ADMODE_0 | QUADSPI_CCR_ADMODE_1)
#define QSPI_NO_INSTR					0
#define QSPI_INSTR_ON_1_LINE			QUADSPI_CCR_IMODE_0
#define QSPI_INSTR_ON_4_LINES			(QUADSPI_CCR_IMODE_0 | QUADSPI_CCR_IMODE_1)


static void nandPrvGpioConfigOne(GPIO_TypeDef* gpio, uint32_t idx, bool output, uint32_t pull, uint32_t afr)
{
	gpio->MODER = (gpio->MODER &~ (3 << (idx * 2))) | ((afr ? 2 : (output ? 1 : 0)) << (idx * 2));
	gpio->OTYPER &=~ (1 << idx);
	gpio->OSPEEDR = (gpio->OSPEEDR &~ (3 << (idx * 2))) | (3 << (idx * 2));		//very high speed
	gpio->PUPDR = (gpio->PUPDR &~ (3 << (idx * 2)))  | (pull << (idx * 2));
	gpio->AFR[idx / 8] = (gpio->AFR[idx / 8] &~ (0x0F << ((idx % 8) * 4))) | afr << ((idx % 8) * 4);
}

static void nandPrvPowerOn(void)
{
	RCC->AHB3ENR |= RCC_AHB3ENR_QSPIEN;
	
	nandPrvGpioConfigOne(GPIOF, 10, true, GPIO_PULL_DOWN, 9);	//CK
	nandPrvGpioConfigOne(GPIOF, 8, true, GPIO_PULL_DOWN, 10);	//D0
	nandPrvGpioConfigOne(GPIOF, 9, true, GPIO_PULL_DOWN, 10);	//D1
	nandPrvGpioConfigOne(GPIOF, 7, true, GPIO_PULL_DOWN, 9);	//D2
	nandPrvGpioConfigOne(GPIOF, 6, true, GPIO_PULL_DOWN, 9);	//D3
	nandPrvGpioConfigOne(GPIOB, 6, true, GPIO_PULL_UP, 10);		//CS
	
	QUADSPI->CR = 0;
	QUADSPI->DCR =	(31 << QUADSPI_DCR_FSIZE_Pos)	|	//we need 27 bits to address a byte in our memory
					(4 << QUADSPI_DCR_CSHT_Pos);		//given our clock we need 3.05 cycles between commands. make it 5 to round up safely
	QUADSPI->PSMKR = 0x01;								//only low bit matters when polling
	QUADSPI->PSMAR = 0x00;								//and it must be zero for polling to complete
	QUADSPI->PIR = 4;									//poll with 4 clocks between polls
	
	QUADSPI->CR =	(2 << QUADSPI_CR_PRESCALER_Pos)	|	//AHB/3
					QUADSPI_CR_APMS					|	//stop busy polling once not busy
					(4 << QUADSPI_CR_FTHRES_Pos)	|	//tell me when 4 bytes can be read/written
					QUADSPI_CR_SSHIFT				|	//sample data late to accound for long lines
					QUADSPI_CR_EN;
	
	KALTaskDelay(1);
	
	
	
	
	
}

void nandPrvPowerOff(void)
{
	QUADSPI->CR = 0;
	RCC->AHB3ENR &=~ RCC_AHB3ENR_QSPIEN;
	
	nandPrvGpioConfigOne(GPIOF, 10, false, GPIO_PULL_DOWN, 0);	//CK
	nandPrvGpioConfigOne(GPIOF, 8, false, GPIO_PULL_DOWN, 0);	//D0
	nandPrvGpioConfigOne(GPIOF, 9, false, GPIO_PULL_DOWN, 0);	//D1
	nandPrvGpioConfigOne(GPIOF, 7, false, GPIO_PULL_DOWN, 0);	//D2
	nandPrvGpioConfigOne(GPIOF, 6, false, GPIO_PULL_DOWN, 0);	//D3
	nandPrvGpioConfigOne(GPIOB, 6, false, GPIO_PULL_UP, 0);	//CS
}


static void qspiControllerBusyWait(void)
{
	while (QUADSPI->SR & QUADSPI_SR_BUSY);
}

static void qspiCommand(uint32_t cmdAndFlags, uint32_t addr, uint32_t alt, uint32_t nBytes)
{
	if (nBytes)
		nBytes--;
	
	qspiControllerBusyWait();
	
	QUADSPI->FCR = QUADSPI_FCR_CTEF | QUADSPI_FCR_CTCF | QUADSPI_FCR_CSMF | QUADSPI_FCR_CTOF;
	QUADSPI->DLR = nBytes;
	QUADSPI->ABR = alt;
	QUADSPI->CCR = cmdAndFlags;
	if (cmdAndFlags & (QUADSPI_CCR_ADMODE_0 | QUADSPI_CCR_ADMODE_1))
		QUADSPI->AR = addr;
}

static void qspiRead(void* dstP, uint32_t len)	//all buffers over 4 bytes must be 4 byte aligned
{
	uint32_t *dst32 = (uint32_t*)dstP;
	uint8_t *dst8;
	
	while (!(QUADSPI->SR & QUADSPI_SR_FLEVEL_Msk));	//wait for fifo to not be empty, else we can get a zero instead of data
	
	while (len >= 4) {
		
		len -= 4;
		*dst32++ = QUADSPI->DR;
	}
	dst8 = (uint8_t*)dst32;
	while (len--)
		*dst8++ = *(volatile uint8_t*)&QUADSPI->DR;
	
	while (QUADSPI->SR & QUADSPI_SR_FLEVEL_Msk) {
		(void)*(volatile uint8_t*)&QUADSPI->DR;
	}
}

static void qspiWrite(const void* srcP, uint32_t len)	//all buffers over 4 bytes must be 4 byte aligned
{
	const uint32_t *src32 = (const uint32_t*)srcP;
	const uint8_t *src8;
	
	while (len >= 4) {
		len -= 4;
		QUADSPI->DR = *src32++;
	}
	src8 = (const uint8_t*)src32;
	while (len--)
		*(volatile uint8_t*)&QUADSPI->DR = *src8++;
}

static void qspiWriteEnableDisable(bool enable)
{
	qspiCommand(	(enable ? QSPI_MICRON_CMD_WRITE_ENABLE : QSPI_MICRON_CMD_WRITE_DISABLE) |
					QSPI_MODE_READ |
					QSPI_NO_DATA |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_NO_ADDR_BYTES |
					QSPI_INSTR_ON_1_LINE,
				0, 0, 0);
}

static void qspiReadId(uint8_t *id)
{
	qspiCommand(	QSPI_MICRON_CMD_READ_ID |
					QSPI_MODE_READ |
					QSPI_DATA_ON_1_LINE |
					QSPI_8_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_NO_ADDR_BYTES |
					QSPI_INSTR_ON_1_LINE,
				0, 0, 2);
	
	qspiRead(id, 2);
}

static uint_fast8_t qspiGetFeature(uint_fast8_t feature)
{
	uint8_t reply;
	
	qspiCommand(	QSPI_MICRON_CMD_GET_FEATURES |
					QSPI_MODE_READ |
					QSPI_DATA_ON_1_LINE |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_1_ADDR_BYTE |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				feature, 0, sizeof(reply));
			
	qspiRead(&reply, sizeof(reply));
	
	return reply;
}

static void qspiSetFeature(uint_fast8_t feature, uint8_t val)
{
	qspiCommand(	QSPI_MICRON_CMD_SET_FEATURES |
					QSPI_MODE_WRITE |
					QSPI_DATA_ON_1_LINE |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_1_ADDR_BYTE |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				feature, 0, sizeof(val));
			
	qspiWrite(&val, sizeof(val));
}

static uint_fast8_t __attribute__((unused)) qspiReadStatus(void)
{
	return qspiGetFeature(QSPI_MICRON_FTR_STATUS);
}

static uint8_t qspiBusyWait(void)	//return status
{
	qspiCommand(	QSPI_MICRON_CMD_GET_FEATURES |
					QSPI_MODE_AUTO_POLLING |
					QSPI_DATA_ON_1_LINE |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_1_ADDR_BYTE |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				QSPI_MICRON_FTR_STATUS, 0, 1);
	
	qspiControllerBusyWait();
	
	return *(volatile uint8_t*)&QUADSPI->DR;
}

static void qspiPageReadCmd(uint32_t addr)
{
	qspiCommand(	QSPI_MICRON_CMD_PAGE_READ |
					QSPI_MODE_WRITE |		//yup...
					QSPI_NO_DATA |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_3_ADDR_BYTES |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				addr, 0, 0);
}

static void qspiReadFromCache(uint32_t colStart, uint32_t nBytes, void* dst)
{
	qspiCommand(	QSPI_MICRON_CMD_READ_CACHE_QUAD |
					QSPI_MODE_READ |
					QSPI_DATA_ON_4_LINES |
					QSPI_4_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_2_ADDR_BYTES |
					QSPI_ADDR_BYTES_ON_4_LINES |
					QSPI_INSTR_ON_1_LINE,
				colStart, 0, nBytes);
	
	qspiRead(dst, nBytes);
}

static void qspiProgramLoad(uint32_t colStart, uint32_t nBytes, const void* src)
{
	qspiCommand(	QSPI_MICRON_CMD_PRGM_LOAD_x4 |
					QSPI_MODE_WRITE |
					QSPI_DATA_ON_4_LINES |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_2_ADDR_BYTES |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				colStart, 0, nBytes);
	
	qspiWrite(src, nBytes);
}

static void qspiProgramExecute(uint32_t pageNum)
{
	qspiCommand(	QSPI_MICRON_CMD_PROGRAM_EXECUTE |
					QSPI_MODE_READ |
					QSPI_NO_DATA |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_3_ADDR_BYTES |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				pageNum, 0, 0);
}

static void qspiBlockErase(uint32_t pageNum)
{
	qspiCommand(	QSPI_MICRON_CMD_ERASE_BLOCK |
					QSPI_MODE_READ |
					QSPI_NO_DATA |
					QSPI_NO_DUMMY_CYCLES |
					QSPI_NO_ALTERNATE_BYTES |
					QSPI_3_ADDR_BYTES |
					QSPI_ADDR_BYTES_ON_1_LINE |
					QSPI_INSTR_ON_1_LINE,
				pageNum, 0, 0);
}

static uint_fast8_t qspiMlSetCfgMode(uint_fast8_t cfgMode)	//returns old mode
{
	uint_fast8_t cfg = qspiGetFeature(QSPI_MICRON_FTR_CONFIG);
	uint_fast8_t oldMode = cfg & 0xc2;
	
	qspiSetFeature(QSPI_MICRON_FTR_CONFIG, (cfg &~ 0xc2) | ((cfgMode & 6) << 5) | ((cfgMode & 1) << 1));
	
	return ((oldMode & 0xc0) >> 5) | ((oldMode & 2) >> 1);
}

static int_fast8_t qspiHlReadPage(uint32_t pageNo, uint32_t ofst, uint32_t len, uint8_t *dst)	//return num corrected ECC bits as pectent of possible (negative if uncorrectable)
{
	uint_fast8_t sta;
	
	qspiPageReadCmd(pageNo);
	sta = qspiBusyWait();
	qspiReadFromCache(ofst + ((pageNo & 64) ? 0x1000 : 0), len, dst);
	
	switch ((sta >> 4) & 3) {
		case 0: return 0;
		case 1: return 100 * 3 / 8;
		default:
		case 2: return -1;
		case 3: return 100 * 6 / 8;
		case 5: return 100 * 8 / 8;
	}
}

static void qspiMlGetParamsOrUidPage(uint8_t pageNo, uint32_t ofst, uint32_t len, uint8_t *dst)
{
	uint_fast8_t oldCfgMode = qspiMlSetCfgMode(QSPI_MICRON_CFG_MODE_OTP_PARAM_UNIQ);
	
	(void)qspiHlReadPage(pageNo, ofst, len, dst);
	
	qspiMlSetCfgMode(oldCfgMode);
}

static void __attribute__((unused)) qspiHlGetParamsPage(uint32_t ofst, uint32_t len, uint8_t *dst)
{
	qspiMlGetParamsOrUidPage(QSPI_MICRON_PAGE_NO_PARAMS, ofst, len, dst);
}

static void qspiHlGetUidPage(uint32_t ofst, uint32_t len, uint8_t *dst)
{
	qspiMlGetParamsOrUidPage(QSPI_MICRON_PAGE_NO_UID, ofst, len, dst);
}

static bool qspiHlWritePage(uint32_t pageNo, uint32_t ofst, uint32_t len, const uint8_t *src)	//remember: max 4 partial progams per page
{
	uint_fast8_t sta;
	
	qspiWriteEnableDisable(true);
	qspiProgramLoad(ofst + ((pageNo & 64) ? 0x1000 : 0), len, src);
	qspiProgramExecute(pageNo);
	sta = qspiBusyWait();
	
	return !(sta & 0x08);
}

static bool qspiHlEraseBlock(uint32_t pageNo)	//erases 128 pages
{
	uint_fast8_t sta;
	
	qspiWriteEnableDisable(true);
	qspiBlockErase(pageNo);
	sta = qspiBusyWait();
	
	return !(sta & 0x04);
}

bool nandInit(void)
{
	uint8_t id[2];
	
	nandPrvPowerOn();
	
	qspiReadId(id);
	
	logt("ID reports %02x %02x\n", id[0], id[1]);
	if (id[0] != 0x2c || id[1] != 0x24)
		return false;
	
	qspiSetFeature(QSPI_MICRON_FTR_BLOCK_LOCK, 0x00);	//unblock the device
	
	return true;
}

void nandSleep(void)
{
	nandPrvPowerOff();
}

void nandWake(void)
{
	nandPrvPowerOn();
}

void nandDeinit(void)
{
	nandPrvPowerOff();
}

bool nandBlockErase(uint32_t firstPageNum)
{
	return qspiHlEraseBlock(firstPageNum);
}

int_fast8_t nandPageRead(uint32_t pageNum, void *data)
{
	return qspiHlReadPage(pageNum, 0, NAND_PAGE_SIZE + NAND_SPARE_SIZE, data);
}

bool nandPageWrite(uint32_t pageNum, const void *data)
{
	return qspiHlWritePage(pageNum, 0, NAND_PAGE_SIZE + NAND_SPARE_SIZE, data);
}

bool nandPageWritePartial(uint32_t pageNum, const void *data, uint32_t ofst, uint32_t len)
{
	return qspiHlWritePage(pageNum, ofst, len, data);
}

bool nandIsBlockBad(uint32_t firstPageNum)
{
	uint8_t marker;
	
	(void)qspiHlReadPage(firstPageNum, 2048, 1, &marker);
	
	return marker == 0;
}

void nandBlockMarkBad(uint32_t firstPageNum)
{
	uint8_t zero = 0;
	
	(void)qspiHlWritePage(firstPageNum, 2048, 1, &zero);
}

bool nandReadUid(uint8_t uid[static 32])
{
	uint32_t i, j, potential[8];
	
	for (i = 0; i < 16; i++) {
		
		qspiHlGetUidPage(sizeof(potential) * i, sizeof(potential), (uint8_t*)potential);
		
		for (j = 0; j < 4; j++) {
			
			if ((potential[j] ^ potential[j + 4]) != 0xffffffff)
				break;
		}
		if (j == 4) {
			MemMove(uid, potential, 16);
			return true;
		}
	}
	
	return false;
}



