#define _BUILDING_DMA_LIB

#include <stdint.h>
#include <MemoryMgr.h>
#include <LibTraps.h>
#include <SerialMgr.h>
#include <FatalAlert.h>
#include <StringMgr.h>
#include <string.h>
#include "common.h"
#include "printf.h"
#include <dal.h>
#include <ral.h>
#include "DmaDriver.h"
#include "stm32f469xx.h"


#define NUM_STREAMS_PER_CONTROLLER		8
#define NUM_DMA_CONTROLERS				2
#define POINTER_MANGLE_FACTOR			0x69696969

#define DMA_STRM_IRQ_ALL				(DMA_STRM_IRQ_DONE | DMA_STRM_IRQ_HALF | DMA_STRM_IRQ_XFER_ERR | DMA_STRM_IRQ_DIRECT_ERR | DMA_STRM_IRQ_FIFO_ERR)


union DmaStreamCfg {
	
	struct {
		//irq selection
		uint32_t 					irqOnComplete	: 1;
		uint32_t 					irqOnHalf		: 1;
		uint32_t					irqOnError		: 1;
		uint32_t					irqOnDirectErr	: 1;
		uint32_t					irqOnFifoErr	: 1;
		
		//config
		uint32_t					chan			: 3;
		uint32_t					memBurstSz		: 2;
		uint32_t					perBurstSz		: 2;
		uint32_t					dblBuf			: 1;
		uint32_t					circBuf			: 1;
		uint32_t 					prio			: 2;
		uint32_t					perIncr			: 1;	//autoincrement periph addr?
		uint32_t					perSz			: 2;
		uint32_t					memIncr			: 1;	//autoincrement periph addr?
		uint32_t					memSz			: 2;
		uint32_t					toMem			: 1;	//else from mem
		uint32_t					perControlsFlow	: 1;
		uint32_t					useFifo			: 1;
		uint32_t					fifoThresh		: 2;
	
		//enable
		uint32_t					enabled			: 1;
		
		//internal state not for chip but fo rus
		uint32_t					reserved		: 1;
	};
	uint32_t						raw;
};
struct Globals {						//size set to 0x288 bytes by amdi resource...
	//global state: precisely 8 words
	uint32_t rfu[8];
	
	//per controller state: exactly 256 bytes each, x2
	struct DmaControlerState {
		
		//per-stream state: exactly 32 bytes each, x8
		struct DmaStreamState {
			DmaLibIrqHandlerF	irqF;
			void*				irqD;
			uint32_t			addrP;
			uint32_t			addrM[2];
			union DmaStreamCfg	cfg;
			uint32_t			numItems;
			uint32_t			rfu;
		} strm[NUM_STREAMS_PER_CONTROLLER];
		
	} ctrl[NUM_DMA_CONTROLERS];
};

static const uint8_t mIrqsInOrder[] = {
	DMA1_Stream0_IRQn, DMA1_Stream1_IRQn, DMA1_Stream2_IRQn, DMA1_Stream3_IRQn,
	DMA1_Stream4_IRQn, DMA1_Stream5_IRQn, DMA1_Stream6_IRQn, DMA1_Stream7_IRQn,
	DMA2_Stream0_IRQn, DMA2_Stream1_IRQn, DMA2_Stream2_IRQn, DMA2_Stream3_IRQn,
	DMA2_Stream4_IRQn, DMA2_Stream5_IRQn, DMA2_Stream6_IRQn, DMA2_Stream7_IRQn,
};


static struct Globals* __attribute__((const)) dmaGlobalsGet(void)
{
	void*** ret;
	
	asm ("ldr %0, [r9]":"=r"(ret));
	return (struct Globals*)&ret[MY_LIB_ID / 4][0x10/4];
}

static struct DmaStreamState* dmaPointerUnmangle(DmaStream strm)						//preserves NULL
{
	uint32_t t = (uint32_t)strm;
	
	t += POINTER_MANGLE_FACTOR;
	t ^= POINTER_MANGLE_FACTOR;
	
	return (struct DmaStreamState*)t;
}

static DmaStream dmaPointerMangle(struct DmaStreamState* s)								//preserves NULL
{
	return (DmaStream)((((uint32_t)s) ^ POINTER_MANGLE_FACTOR) - POINTER_MANGLE_FACTOR);
}

static int32_t __attribute__((pure)) dmaGetIndex(struct DmaStreamState* s)	//direct index of unit returned
{
	struct Globals *g = dmaGlobalsGet();
	uint32_t i;
	
	for (i = 0; i < NUM_DMA_CONTROLERS; i++) {
		if (s >= g->ctrl[i].strm && s < g->ctrl[i].strm + NUM_STREAMS_PER_CONTROLLER)
			return (i * NUM_STREAMS_PER_CONTROLLER) + (s - g->ctrl[i].strm);
	}
	
	loge("Cannot find index for 0x%08x\n", s);
	return -1;
}

static DMA_Stream_TypeDef* __attribute__((pure)) dmaHwGetStreamBase(struct DmaStreamState* s)
{
	static DMA_Stream_TypeDef *const addrs[] = {
		DMA1_Stream0, DMA1_Stream1, DMA1_Stream2, DMA1_Stream3,
		DMA1_Stream4, DMA1_Stream5, DMA1_Stream6, DMA1_Stream7,
		DMA2_Stream0, DMA2_Stream1, DMA2_Stream2, DMA2_Stream3,
		DMA2_Stream4, DMA2_Stream5, DMA2_Stream6, DMA2_Stream7,
	};
		
	int32_t i = dmaGetIndex(s);
	
	if (i < 0 || i >= sizeof (addrs) / sizeof(*addrs))
		return NULL;
	
	return addrs[i];
}

static uint32_t __attribute__((pure)) dmaHwGetIrqNo(struct DmaStreamState* s)
{
	int32_t i = dmaGetIndex(s);
	
	if (i < 0 || i >= sizeof (mIrqsInOrder) / sizeof(*mIrqsInOrder))
		return 0;
	
	return REPALM_IRQ_NO_MANGLE(mIrqsInOrder[i]);
}

static uint32_t dmaIrqFlagsGet(struct DmaStreamState* s)
{
	int32_t i = dmaGetIndex(s);
	volatile uint32_t *reg;
	DMA_TypeDef *unit;
	uint32_t shift;
	
	unit = (i < NUM_STREAMS_PER_CONTROLLER) ? DMA1 : DMA2;
	i %= NUM_STREAMS_PER_CONTROLLER;
	
	reg = (i < 4) ? &unit->LISR : &unit->HISR;
	i %= 4;
	
	shift = (i < 2) ? 0 : 16;
	shift += (i & 1) ? 6 : 0;
	
	return ((*reg) >> shift) & DMA_STRM_IRQ_ALL;
}

static void dmaIrqFlagsClear(struct DmaStreamState* s, uint32_t irqs)
{
	int32_t i = dmaGetIndex(s);
	volatile uint32_t *reg;
	DMA_TypeDef *unit;
	uint32_t shift;
	
	unit = (i < NUM_STREAMS_PER_CONTROLLER) ? DMA1 : DMA2;
	i %= NUM_STREAMS_PER_CONTROLLER;
	
	reg = (i < 4) ? &unit->LIFCR : &unit->HIFCR;
	i %= 4;
	
	shift = (i < 2) ? 0 : 16;
	shift += (i & 1) ? 6 : 0;
	
	*reg = (irqs & DMA_STRM_IRQ_ALL) << shift;
}

void __attribute__((used)) impl_DmaLibGetInfo(uint32_t *numControllersP, uint32_t *numStreamsPerControllerP)
{
	if (numControllersP)
		*numControllersP = NUM_DMA_CONTROLERS;
	
	if (numStreamsPerControllerP)
		*numStreamsPerControllerP = NUM_STREAMS_PER_CONTROLLER;
}

DmaStream __attribute__((used)) impl_DmaLibStreamReserve(uint32_t controller, uint32_t stream)
{
	struct Globals *g = dmaGlobalsGet();
	struct DmaStreamState *ret = NULL;
	
	controller--;	//they are 1-indexed for humans (and streams are not, go figure)
	if (controller < NUM_DMA_CONTROLERS && stream < NUM_STREAMS_PER_CONTROLLER) {
		
		union DmaStreamCfg cfgRead, cfgWritten;
		uint32_t storeFailed;
		
		do {
			asm volatile("ldrex %0, %1":"=r"(cfgRead.raw):"m"(g->ctrl[controller].strm[stream].cfg.raw):"memory");
			cfgWritten.raw = cfgRead.raw;
			cfgWritten.reserved = true;
			asm volatile("strex %0, %1, %2":"=r"(storeFailed):"r"(cfgWritten.raw),"m"(g->ctrl[controller].strm[stream].cfg.raw):"memory");
		} while (storeFailed);
		
		if (!cfgRead.reserved)
			ret = &g->ctrl[controller].strm[stream];
	}
	
	return dmaPointerMangle(ret);
}

bool __attribute__((used)) impl_DmaLibStreamRelease(DmaStream strm)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	bool ret = false;
	
	if (s->cfg.reserved) {
	
		if (s->cfg.enabled)
			dmaHwGetStreamBase(s)->CR = 0;
		
		s->cfg.reserved = false;
		ret = true;
	}
	
	return ret;
}

bool __attribute__((used)) impl_DmaLibStreamSetIrqHandler(DmaStream strm, DmaLibIrqHandlerF irqFunc, void* irqHandlerData)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	
	if (s->cfg.enabled)
		return false;
		
	s->irqF = irqFunc;
	s->irqD = irqHandlerData;

	return true;
}

bool __attribute__((used)) impl_DmaLibStreamSetIrqState(DmaStream strm, uint32_t irqsEnabled /* mask of DMA_STRM_IRQ_* */)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	DMA_Stream_TypeDef *hw = dmaHwGetStreamBase(s);
	uint32_t cr, fcr, irqNo = dmaHwGetIrqNo(s);
	
	if (irqsEnabled &~ DMA_STRM_IRQ_ALL)
		return false;
	
	s->cfg.irqOnComplete = (irqsEnabled & DMA_STRM_IRQ_DONE) ? 1 : 0;
	s->cfg.irqOnHalf = (irqsEnabled & DMA_STRM_IRQ_HALF) ? 1 : 0;
	s->cfg.irqOnError = (irqsEnabled & DMA_STRM_IRQ_XFER_ERR) ? 1 : 0;
	s->cfg.irqOnDirectErr = (irqsEnabled & DMA_STRM_IRQ_DIRECT_ERR) ? 1 : 0;
	s->cfg.irqOnFifoErr = (irqsEnabled & DMA_STRM_IRQ_FIFO_ERR) ? 1 : 0;
	
	if (s->cfg.enabled) {	//set while enabled is more complex
		cr = hw->CR;
		fcr = hw->FCR;
		
		cr &=~ (DMA_SxCR_TCIE | DMA_SxCR_HTIE | DMA_SxCR_TEIE | DMA_SxCR_DMEIE);
		fcr &=~ DMA_SxFCR_FEIE;
		
		if (irqsEnabled & DMA_STRM_IRQ_DONE)
			cr |= DMA_SxCR_TCIE;
		if (irqsEnabled & DMA_STRM_IRQ_HALF)
			cr |= DMA_SxCR_HTIE;
		if (irqsEnabled & DMA_STRM_IRQ_XFER_ERR)
			cr |= DMA_SxCR_TEIE;
		if (irqsEnabled & DMA_STRM_IRQ_DIRECT_ERR)
			cr |= DMA_SxCR_DMEIE;
		if (irqsEnabled & DMA_STRM_IRQ_FIFO_ERR)
			fcr |= DMA_SxFCR_FEIE;
		
		HALInterruptSetState(irqNo, false);
		
		hw->CR = cr;
		hw->FCR = fcr;
		dmaIrqFlagsClear(s, irqsEnabled);
		
		HALInterruptSetState(irqNo, !!irqsEnabled);
	}
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamConfigure(DmaStream strm, const struct DmaStreamUserCfg* cfg)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	
	//ABI check
	if (cfg->magic != CFG_STRUCT_MAGIX)
		return false;
	
	if (s->cfg.enabled)
		return false;
	
	s->cfg.chan = cfg->chan;
	s->cfg.memBurstSz = cfg->memBurstSz;
	s->cfg.perBurstSz = cfg->perBurstSz;
	s->cfg.dblBuf = cfg->dblBuf;
	s->cfg.circBuf = cfg->circBuf;
	s->cfg.prio = cfg->prio;
	s->cfg.perIncr = cfg->perIncr;
	s->cfg.perSz = cfg->perSz;
	s->cfg.memIncr = cfg->memIncr;
	s->cfg.memSz = cfg->memSz;
	s->cfg.toMem = cfg->toMem;
	s->cfg.perControlsFlow = cfg->perControlsFlow;
	s->cfg.useFifo = cfg->useFifo;
	s->cfg.fifoThresh = cfg->fifoThresh;
	s->numItems = cfg->numItems;
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamSetEnabled(DmaStream strm, bool on)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	uint32_t cr = 0, fcr = 0, irqNo = dmaHwGetIrqNo(s);
	DMA_Stream_TypeDef *hw = dmaHwGetStreamBase(s);
	
	if ((s->cfg.enabled && on) || (!s->cfg.enabled && !on))
		return true;
	
	if (!on) {
		HALInterruptSetState(irqNo, 0);
		s->cfg.enabled = 0;
		hw->CR = 0;
	}
	else {		//to enable: calc CR, FCR, set them, enable stream
		
		cr |= s->cfg.chan << DMA_SxCR_CHSEL_Pos;
		cr |= s->cfg.memBurstSz << DMA_SxCR_MBURST_Pos;
		cr |= s->cfg.perBurstSz << DMA_SxCR_PBURST_Pos;
		cr |= s->cfg.dblBuf << DMA_SxCR_DBM_Pos;
		cr |= s->cfg.prio << DMA_SxCR_PL_Pos;
		cr |= s->cfg.memSz << DMA_SxCR_MSIZE_Pos;
		cr |= s->cfg.perSz << DMA_SxCR_PSIZE_Pos;
		cr |= s->cfg.memIncr << DMA_SxCR_MINC_Pos;
		cr |= s->cfg.perIncr << DMA_SxCR_PINC_Pos;
		cr |= s->cfg.circBuf << DMA_SxCR_CIRC_Pos;
		cr |= s->cfg.toMem ? 0 : DMA_SxCR_DIR_0;
		cr |= s->cfg.perControlsFlow << DMA_SxCR_PFCTRL_Pos;
		cr |= s->cfg.irqOnComplete << DMA_SxCR_TCIE_Pos;
		cr |= s->cfg.irqOnHalf << DMA_SxCR_HTIE_Pos;
		cr |= s->cfg.irqOnError << DMA_SxCR_TEIE_Pos;
		cr |= s->cfg.irqOnDirectErr	<< DMA_SxCR_DMEIE_Pos;
		
		fcr |= s->cfg.irqOnFifoErr << DMA_SxFCR_FEIE_Pos;
		fcr |= s->cfg.useFifo << DMA_SxFCR_DMDIS_Pos;
		fcr |= s->cfg.fifoThresh << DMA_SxFCR_FTH_Pos;
		
		s->cfg.enabled = 1;
		
		HALInterruptSetState(irqNo, false);
		dmaIrqFlagsClear(s, DMA_STRM_IRQ_ALL);
		
		hw->CR = cr;
		hw->FCR = fcr;
		hw->NDTR = s->numItems;
		hw->CR = cr | DMA_SxCR_EN;
		
		logt("DMA: hw=0x%08x, CR=0x%08x FCR=0x%08x NDTR=0x%08x (expect %u) PAR=0x%08x M0AR=0x%08x\n",
			hw, hw->CR, hw->FCR, hw->NDTR, s->numItems, hw->PAR, hw->M0AR);
		
		HALInterruptSetState(irqNo, s->cfg.irqOnComplete || s->cfg.irqOnHalf || s->cfg.irqOnError || s->cfg.irqOnDirectErr || s->cfg.irqOnFifoErr);
	}
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamSetPeriphAddr(DmaStream strm, uint32_t addr)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	DMA_Stream_TypeDef *hw = dmaHwGetStreamBase(s);
	
	if (s->cfg.enabled)
		return false;
	
	s->addrP = addr;
	hw->PAR = addr;
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamSetMemAddr(DmaStream strm, uint32_t bufferIdx, uint32_t addr)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	DMA_Stream_TypeDef *hw = dmaHwGetStreamBase(s);
	
	if (s->cfg.enabled || bufferIdx > 1)
		return false;
	
	s->addrM[bufferIdx] = addr;
	if (bufferIdx)
		hw->M1AR = addr;
	else
		hw->M0AR = addr;
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamGetItemsLeftToTransfer(DmaStream strm, uint32_t *numItemsToTransferP)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	DMA_Stream_TypeDef *hw = dmaHwGetStreamBase(s);
	
	if (!s->cfg.enabled)
		return false;
	
	if (numItemsToTransferP)
		*numItemsToTransferP = hw->NDTR;
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamGetCurrentTargetBuffer(DmaStream strm, uint32_t *targetBufferIdxP)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	DMA_Stream_TypeDef *hw = dmaHwGetStreamBase(s);
	
	if (!s->cfg.enabled)
		return false;
	
	if (targetBufferIdxP)
		*targetBufferIdxP = (hw->CR & DMA_SxCR_CT) ? 1 : 0;
	
	return true;
}

bool __attribute__((used)) impl_DmaLibStreamGetEnabled(DmaStream strm, bool *enabledP)
{
	struct DmaStreamState *s = dmaPointerUnmangle(strm);
	
	return !!s->cfg.enabled;
}

static void dmaIrqHandler(void *irqNoP)
{
	uint32_t stream = ((uint32_t)irqNoP) % NUM_STREAMS_PER_CONTROLLER;
	uint32_t unit = ((uint32_t)irqNoP) / NUM_STREAMS_PER_CONTROLLER;
	struct Globals *g = dmaGlobalsGet();
	struct DmaStreamState *strm;
	uint32_t sr;
	
	//loge("dma irqh\n");
	
	strm = &g->ctrl[unit].strm[stream];
	sr = dmaIrqFlagsGet(strm);
	dmaIrqFlagsClear(strm, sr);
	
	strm->irqF(strm->irqD, sr);
}

uint32_t __attribute__((used)) PilotMain(uint16_t cmd, void* cmdPBP, uint16_t flags)
{
	struct Globals *g;
	uint32_t i, j;

	loge("dma main 0x%04x in\n", cmd);
		
	if (cmd == RAL_CMD_LOAD) {
		g = dmaGlobalsGet();
		
		for (i = 0; i < sizeof(mIrqsInOrder) / sizeof(*mIrqsInOrder); i++) {
			
			uint32_t irq = REPALM_IRQ_NO_MANGLE(mIrqsInOrder[i]);
			
			RCC->AHB1ENR |= RCC_AHB1ENR_DMA1EN | RCC_AHB1ENR_DMA2EN;
			HALInterruptSetHandler(irq, dmaIrqHandler, (void*)i);
			HALInterruptSetState(irq, false);
		}
	}
	else if (cmd == RAL_CMD_UNLOAD) {
		
		g = dmaGlobalsGet();
		
		for (i = 0; i < NUM_DMA_CONTROLERS; i++) {
			for (j = 0; j < NUM_STREAMS_PER_CONTROLLER; j++) {
				if (g->ctrl[i].strm[j].cfg.enabled) {
					
					char x[64];
					StrPrintF(x, "DMA%u Stream %u enabled while DmaDriver being unloaded!\n", i, j);
					(void)SysFatalAlert(x);
					dmaHwGetStreamBase(&g->ctrl[i].strm[j])->CR = 0;
				}
			}
		}
		
		for (i = 0; i < sizeof(mIrqsInOrder) / sizeof(*mIrqsInOrder); i++) {

			uint32_t irq = REPALM_IRQ_NO_MANGLE(mIrqsInOrder[i]);
			
			HALInterruptSetState(irq, false);
			HALInterruptSetHandler(irq, NULL, NULL);
		}
		
		RCC->AHB1ENR &=~ (RCC_AHB1ENR_DMA1EN | RCC_AHB1ENR_DMA2EN);
	}
	
	loge("dma main 0x%04x out\n", cmd);
	
	return errNone;
}



