#include "zodiac.h"
#include "timers.h"
#include "printf.h"
#include <string.h>
#include "boot.h"
#include "dal.h"
#include "cpu.h"


enum TwIfaceNum {
	TwIfaceGfxReset = 7,	//part of UIReset - close graphics if not yet closed
	TwIfaceSound = 9,
	
	
	
	//MIDI 0x2000..0x2023
	TwIfaceMidiUnknown0x2000 = 0x2000,
	
	TwIfaceMidiSmfOpen = 0x2004,
	TwIfaceMidiSmfClose,
	TwIfaceMidiSmfPlay,
	TwIfaceMidiSmfStop,
	
	TwIfaceMidiOpen = 0x2009,
	TwIfaceMidiClose,
	TwIfaceMidiNoteOn,
	TwIfaceMidiNoteOff,
	TwIfaceMidiControlChange,
	TwIfaceMidiProgramChange,
	TwIfaceMidiPitchBend,
	TwIfaceMidiSysEx,
	
	TwIfaceMidiVolumeControlHandleGrab = 0x201d,
	TwIfaceMidiVolumeControlHandleRelease,
	TwIfaceMidiVolumeSetMaster,
	TwIfaceMidiVolumeGetMaster,
	
	TwIfaceMidiUnknown0x2023 = 0x2023,
};

struct TwTombstoneAppDescr {
	uint32_t dbType;			//0x00
	uint32_t dbCrid;			//0x04
	char dbName[32];			//0x08
};

struct TwTombstone {
	uint32_t resetReason;						//0x00	twResetReason*
	struct TwTombstoneAppDescr mainApp;			//0x04
	struct TwTombstoneAppDescr sublaunchedApp;	//0x2C
	uint32_t something;							//0x54	//if set, "mainApp" is valid
};	//0x58



void DALEXPORT impl_TwReadTombstoneStruct(struct TwTombstone *dst)
{
	logi("%s\n", __func__);
	
	if (dst)
		memset(dst, 0, sizeof(*dst));
}

void DALEXPORT impl_TwWriteRombstoneStruct(const struct TwTombstone *src)
{
	logi("%s\n", __func__);
}

static Err zodPrvAppExitedNotifHandler(struct SysNotifyParamType *notifyParamsP)
{
	logi("%s\n", __func__);
	
	zodPrvAppExitNotifTwInput();
	
	return errNone;
}

void zodSetInitStage(uint32_t stage)
{
	if (stage == DAL_INIT_STAGE_MEM_AND_DM) {
		
		zodTwGfxInit();
		zodDevicesInit();
		zodVibratorInit();
	}
	else if (stage == DAL_INIT_STAGE_ALL_UP) {
		
		SysNotifyRegister(NOTIF_REGISTER_NO_DB, CREATE_4CC('t','w','A','X'), zodPrvAppExitedNotifHandler, 0 , NULL);
		zodTwInputInit();
	}
}

void* DALEXPORT impl_TwGetInterfaceFuncs(enum TwIfaceNum iface)
{
	logi("%s(%u)\n", __func__, iface);
	
	switch (iface) {
		case TwIfaceSound:
			return zodPrvGetSoundIface();
		
		case TwIfaceGfxReset:
			return zodPrvGetGfxResetIface();
		
		case 0x2000 ... 0x2023:
			logi("Midi not yet supported, returning NULL for interface request\n");
			return NULL;
		
		
		default:
			fatal("we do not know about interface 0x%04x, requested from 0x%08x\n", iface, __builtin_return_address(0));
	}
	
	return NULL;
	
	/*
		existent ifaces on real HW:
			0x0000..0x0008	//general
			0x1000..0x103b	//gfx
			0x2000..0x2010	//midi
			0x2013..0x2023	//midi
		
		on sim only these exist: 5, 7, 9
		
		NULL is returned for others
		
		some just re-export existing funcs:
		0->TwUnknown528
		1->OEMReserved5
		2->OEMReserved6
		3->OEMReserved7
		0x1037->TwUnknown3fc
		0x103a->TwUnknown3f8
		
		0x0007 is something related to Gfx. no args. it will check if TwGfx is properly closed and warn if not, then close it. likely for app-exit-time checking
		
		TwMidi table is
	idx: 00  04  08  0c  10  14  18  1c  20  24  28  2c  30  34  38  3c
		[04, 05, 06, 07, 09, 0a, 0b, 0c, 0d, 0e, 0f, 10, 1d, 1f, 20, 1e]
		
		0x2004 - smfopen. called as InternalErrorType (?, ?, ?, ?)
		0x2005 - smfclose. called as (u32 ref?)
		0x2006 - SmfPlay. called as InternalErrorType (?, ?, ?, ?, ?)
		0x2007 - SmfStop. called as (u32 ref?)
		0x2009 - midiopen. called as called as InternalErrorType (zero, u8 *handleOutP)
		0x200a - midiclose. called as (u8 handle)
		0x200b - note on. called as InternalErrorType (u8 handle, UInt8 aChannel, UInt8 aKey, UInt8 aVelocity)
		0x200c - note off. called as InternalErrorType (u8 handle, UInt8 aChannel, UInt8 aKey, UInt8 aVelocity)
		0x200d - controlChange. called as InternalErrorType (u8 handle, UInt8 aChannel, u8 aControlChange, u8 aValue)
		0x200e - program change. called as InternalErrorType (u8 handle, UInt8 aChannel, UInt8 aProgramChange)
		0x200f - pitch bend. called as InternalErrorType (u8 handle, UInt8 aChannel, i16 aBend)
		0x2010 - SysEx. called as InternalErrorType (u8 handle, UInt8 aChannel, UInt8* aData, UInt16 aSize)
		0x201e - ReleaseColumeControlHandle called as (u32 volHandle?)
		0x201d - GetVolumeControlHandle called as Err (u32 *volHandle?)
		0x201f - SetMasterVolume. called as (volHandle?, 0x10, volume)
		0x2020 - GetMasterVolume. called as (volHandle?, 0x10, u16 *volumeP)
	*/
}

uint32_t DALEXPORT impl_TwGetMicroSeconds(void)
{
	return timerGetTime() / (TIMER_TICKS_PER_MSEC / 1000);
}