#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "kernel_int.h"
#include "ral_export.h"
#include "platform.h"
#include "printf.h"
#include "kernel.h"
#include "timers.h"
#include "entry.h"
#include "heap.h"
#include "irqs.h"
#include "emu.h"
#include "mpu.h"
#include "dal.h"
#include "ral.h"


void kernelPrvHandleSwi(void);	//handled by kernelLegacy.c

struct LegacyExcFrame {
	uint32_t sr, r8_to_r14[7];
	uint32_t r0_to_r7[8], pc;
};

enum ExceptionType {
	ExceptionTypeUndefInstr,		//data is {instr, -}
	ExceptionTypePrefetchAbt,		//data is {FSR, FAR}
	ExceptionTypeDataAbt,			//data is {FSR, FAR}
	
};

void emuCoreInit(void)
{
	//nothing - no emu needed on chips supporting ARM instructions
}

#ifdef CUSTOM_MMU_FAULT_HANDLER

	extern bool CUSTOM_MMU_FAULT_HANDLER(struct LegacyExcFrame *exc, uint32_t addr);	//true to continue execution as if no fault happened

#endif

static bool handleFaultSilently(struct LegacyExcFrame *exc, enum ExceptionType what, uint32_t data1, uint32_t data2)
{
	extern uint8_t user_access_0[], user_access_1[], user_access_2[], user_access_3[];
	extern bool userAccessFaultReturnPath(void);
	
	if (what == ExceptionTypeDataAbt) {
		
		uint32_t pc = exc->pc;
	
		if ((pc == (uintptr_t)&user_access_0) || (pc == (uintptr_t)&user_access_1) || (pc == (uintptr_t)&user_access_2) || (pc == (uintptr_t)&user_access_3)) {
			
			exc->pc = ((uint32_t)&userAccessFaultReturnPath) &~ 1;
			return true;
		}
	}
	
	return false;
}

static void generalFaultHandlerShowSpWords(uint32_t *words, uint32_t nwords)
{
	uint32_t i;
	
	for (i = 0; i < nwords; i++)
		loge("    [SP + 0x%03x ( == 0x%08x) ] = 0x%08x\n", i * 4, words + i, words[i]);
}

static void generalFaultHandlerLogSpWords(char *line, uint32_t *words, uint32_t nwords)
{
	while (nwords >= 3) {
		spr(line + strlen(line), "%08x %08x %08x\n", words[0], words[1], words[2]);
		nwords -= 3;
		words +=3;
	}
}

void __attribute__((used)) kernelPrvHandleExc(struct LegacyExcFrame *exc, enum ExceptionType what, uint32_t data1, uint32_t data2)
{
	uint32_t i, *sp = (uint32_t*)exc->r8_to_r14[13 - 8];
	char *msg, *line;
	
	msg = halErrorGetBuffer();
	
	#ifdef CUSTOM_MMU_FAULT_HANDLER
	
		if (what == ExceptionTypeDataAbt && (data1 & 0x10d) == 0x0d && CUSTOM_MMU_FAULT_HANDLER(exc, data2))
			return;
	
	#endif
	
	if (handleFaultSilently(exc, what, data1, data2))
		return;
	
	switch (what) {
		case ExceptionTypeUndefInstr:
			loge("UNDEFINED INSTR HIT\n");
			spr(msg, "FAULT: %s", "UND");
			line = msg + strlen(msg) + 1;
			line[0] = 0;
			spr(line + strlen(line), (exc->sr & LEGACY_SR_FLAG_T)  ? "INSTR: %04x\n" : "INSTR: %08x\n", data1);
			loge("instr 0x%08x\n", data1);
			break;
		
		case ExceptionTypePrefetchAbt:
			loge("PREFETCH ABORT\n");
			spr(msg, "FAULT: %s", "PABT");
			goto abt_common;
		
		case ExceptionTypeDataAbt:
			loge("DATA ABORT\n");
			spr(msg, "FAULT: %s", "DABT");
	abt_common:
			line = msg + strlen(msg) + 1;
			line[0] = 0;
			spr(line + strlen(line), "CODE: %03xh %08xh\n", data1 & 0x1ff, data2);
			loge("FSR 0x%08x FAR 0x%08x\n", data1, data2);
			break;
		
		default:
			loge("EXCEPTION %u\n", what);
			spr(msg, "FAULT: %u", what);
			line = msg + strlen(msg) + 1;
			line[0] = 0;
			spr(line + strlen(line), "INFO: %08xh %08xh\n", data1, data2);
			loge("INFO 0x%08x 0x%08x\n", data1, data2);
			break;
	}
	
	loge("  SR  = 0x%08x\n", exc->sr);
	for (i = 0; i < 7; i++)
		loge("  R%2u = 0x%08x    R%2u = 0x%08x\n", i, exc->r0_to_r7[i], i + 8, exc->r8_to_r14[i]);
	loge("  R%2u = 0x%08x    R%2u = 0x%08x\n", i, exc->r0_to_r7[i], i + 8, exc->pc);
		
	spr(line + strlen(line), "sr %08x regs: %08x\n", exc->sr, exc->r0_to_r7[0]);
	spr(line + strlen(line), "%08x %08x %08x\n", exc->r0_to_r7[1], exc->r0_to_r7[2], exc->r0_to_r7[3]);
	spr(line + strlen(line), "%08x %08x %08x\n", exc->r0_to_r7[4], exc->r0_to_r7[5], exc->r0_to_r7[6]);
	spr(line + strlen(line), "%08x %08x %08x\n", exc->r0_to_r7[7], exc->r8_to_r14[8 - 8], exc->r8_to_r14[9 - 8]);
	spr(line + strlen(line), "%08x %08x %08x\n", exc->r8_to_r14[10 - 8], exc->r8_to_r14[11 - 8], exc->r8_to_r14[12 - 8]);
	spr(line + strlen(line), "%08x %08x %08x\n", exc->r8_to_r14[13 - 8], exc->r8_to_r14[14 - 8], exc->pc);
	
	spr(line + strlen(line), "@ %08x %08x %08x %08x\n", *(volatile uint32_t*)0xffff0000, *(volatile uint32_t*)0xffff0004, *(volatile uint32_t*)0xffff0008, *(volatile uint32_t*)0xffff000c);
	spr(line + strlen(line), "@ %08x %08x %08x %08x\n", *(volatile uint32_t*)0xffff0010, *(volatile uint32_t*)0xffff0014, *(volatile uint32_t*)0xffff0018, *(volatile uint32_t*)0xffff001c);
	spr(line + strlen(line), "@ %08x %08x %08x %08x\n", *(volatile uint32_t*)0xffff0020, *(volatile uint32_t*)0xffff0024, *(volatile uint32_t*)0xffff0028, *(volatile uint32_t*)0xffff002c);
	spr(line + strlen(line), "@ %08x %08x %08x %08x\n", *(volatile uint32_t*)0xffff0030, *(volatile uint32_t*)0xffff0034, *(volatile uint32_t*)0xffff0038, *(volatile uint32_t*)0xffff003c);
	
	
	loge("  some words at SP:\n");
	generalFaultHandlerShowSpWords(sp, 64);
	generalFaultHandlerLogSpWords(line, sp, 96);
	
	impl_HALErrDisplay(msg, false, NULL, false);	//let it draw
	
	kernelLogCurTaskForExc();
	
	irqsAllOff();
	while(1);
}


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

static void __attribute__((naked)) kernelPrvSetExcStacks(void)
{
	asm volatile(
		"	msr  cpsr_c, #0xD1			\n\t"	//FIQ
		"	nop							\n\t"
		"	ldr  sp, =__stack_top_fiq	\n\t"
		"	nop							\n\t"
		
		"	msr  cpsr_c, #0xD2			\n\t"	//IRQ
		"	nop							\n\t"
		"	ldr  sp, =__stack_top_irq	\n\t"
		"	nop							\n\t"
		
		"	msr  cpsr_c, #0xD7			\n\t"	//ABT
		"	nop							\n\t"
		"	ldr  sp, =__stack_top_abt	\n\t"
		"	nop							\n\t"
		
		"	msr  cpsr_c, #0xDB			\n\t"	//UND
		"	nop							\n\t"
		"	ldr  sp, =__stack_top_und	\n\t"
		"	nop							\n\t"
		
		"	msr  cpsr_c, #0xD3			\n\t"
		"	nop							\n\t"
		"	bx   lr						\n\t"
		:
		:
		:"memory", "cc"
	);
}

		
static void __attribute__((naked, used)) kernelPrvStashRegs(void)		//leaves spsr in r0
{
	asm volatile(
		"	mrs    r0, spsr							\n\t"
		"	sub    sp, #4 * 8						\n\t"
		"	mov    r3, sp							\n\t"
		"	ands   r2, r0, #0x0f					\n\t"
		"	cmpne  r2, #0x0f						\n\t"
		"	beq    need_user_regs					\n\t"
		"nonuser_regs:								\n\t"
		"	mrs    r1, cpsr							\n\t"
		"	orr    r2, r2, #0xd0					\n\t"
		"	msr    cpsr_c, r2						\n\t"
		"	nop										\n\t"
		"	stmia  r3, {r0, r8-r14}					\n\t"
		"	nop										\n\t"
		"	msr    cpsr_c, r1						\n\t"
		"	nop										\n\t"
		"	bx     lr								\n\t"
		"need_user_regs:							\n\t"
		"	stmia  r3, {r0, r8-r14}^				\n\t"
		"	nop										\n\t"
		"	bx     lr								\n\t"
	);
}

static void __attribute__((naked, used)) kernelPrvUnstashRegs(void)	//leaves PC pushed
{
	asm volatile(
		"	ldr    r0, [sp], #4						\n\t"
		"	msr    spsr, r0							\n\t"
		"	mov    r3, sp							\n\t"
		"	ands   r0, r0, #0x0f					\n\t"
		"	cmpne  r0, #0x0f						\n\t"
		"	beq    restore_user_regs				\n\t"
		"restore_nonuser_regs:						\n\t"
		"	mrs    r1, cpsr							\n\t"
		"	orr    r0, r0, #0xd0					\n\t"
		"	msr    cpsr_c, r0						\n\t"
		"	nop										\n\t"
		"	ldmia  r3, {r8-r14}						\n\t"
		"	nop										\n\t"
		"	msr    cpsr_c, r1						\n\t"
		"	nop										\n\t"
		"	b      restore_continue					\n\t"
		"restore_user_regs:							\n\t"
		"	ldmia  r3, {r8-r14}^					\n\t"
		"	nop										\n\t"
		"restore_continue:							\n\t"
		"	add    sp, #4 * 7						\n\t"
		"	ldmia  sp!, {r0-r7}						\n\t"
		"	b      kernelMaybeContextSwitch_pushed	\n\t"
	);
}

static void __attribute__((naked)) kernelPrvHandleUnd(void)
{
	asm volatile(
		"	stmfd  sp!, {r0-r7, lr}					\n\t"
		"	mov    r4, lr							\n\t"
		"	bl     kernelPrvStashRegs				\n\t"
		"	tst    r0, #0x20						\n\t"
		"	mov    r0, sp							\n\t"
		"	mov    r1, %0							\n\t"
		"	ldreq  r2, [r4, #-4]					\n\t"
		"	ldrneh r2, [r4, #-2]					\n\t"
		"	bl     kernelPrvHandleExc				\n\t"
		"	b      kernelPrvUnstashRegs				\n\t"
		:
		:"i"(ExceptionTypeUndefInstr)
		:"memory", "cc"
	);
}


static void  __attribute__((naked)) kernelPrvHandlePabt(void)
{
	asm volatile(
		"	sub    lr, #4							\n\t"	//by default we'll retry the instr
		"	stmfd  sp!, {r0-r7, lr}					\n\t"
		"	bl     kernelPrvStashRegs				\n\t"
		"	mov    r0, sp							\n\t"
		"	mov    r1, %0							\n\t"
		"	mrc    p15, 0, r2, c5, c0, 0			\n\t"	//FSR
		"	mrc    p15, 0, r3, c6, c0, 0			\n\t"	//FAR
		"	bl     kernelPrvHandleExc				\n\t"
		"	b      kernelPrvUnstashRegs				\n\t"
		:
		:"i"(ExceptionTypePrefetchAbt)
		:"memory", "cc"
	);
}

static void  __attribute__((naked)) kernelPrvHandleDabt(void)
{
	asm volatile(
		"	sub    lr, #8							\n\t"	//by default we'll retry the instr
		"	stmfd  sp!, {r0-r7, lr}					\n\t"
		"	bl     kernelPrvStashRegs				\n\t"
		"	mov    r0, sp							\n\t"
		"	mov    r1, %0							\n\t"
		"	mrc    p15, 0, r2, c5, c0, 0			\n\t"	//FSR
		"	mrc    p15, 0, r3, c6, c0, 0			\n\t"	//FAR
		"	bl     kernelPrvHandleExc				\n\t"
		"	b      kernelPrvUnstashRegs				\n\t"
		:
		:"i"(ExceptionTypeDataAbt)
		:"memory", "cc"
	);
}

static void  __attribute__((naked)) kernelPrvHandleIrq(void)	//disabled in SWI mode so should never be hit when not using user regs
{
	asm volatile(
		"	sub    lr, #4							\n\t"	//so we can return to LR
		"	stmfd  sp!, {r0-r3, r9, r12, lr}		\n\t"
		"	bl     ralSetSafeR9						\n\t"	//ensure irq code and handlers have a sane R9 value
		"	bl     kernelDispatchIrq				\n\t"
		"	ldmfd  sp!, {r0-r3, r9, r12}			\n\t"
		"	b      kernelMaybeContextSwitch_pushed	\n\t"
		:
		:
		:"memory", "cc"
	);
}

static void kernelFirstTask(void *param)
{
	void (*entryFunc)(void) = (void (*)())param;
	irq_state_t sta;
	uint32_t dummy;
	
	//re-give SVC mode the full svc stack
	sta = irqsAllOff();
	asm volatile(
		"	mrs %0, CPSR				\n\t"
		"	msr CPSR_c, %1				\n\t"
		"	ldr sp, =__stack_top		\n\t"
		"	msr CPSR_c, %0				\n\t"
		:"=&r"(dummy)
		:"r"(0xd3)
		:"memory"
	);
	irqsRestoreState(sta);
	
	entryFunc();
	__builtin_unreachable();
}

static void __attribute__((naked)) kernelGotoSysModeWithIntsOn(void)	//return in system mode with ints/fiqs on. using same stack as we are now
{
	asm volatile(
		"	mov r0, lr			\n\t"
		"	mov r1, sp			\n\t"
		"	msr cpsr_c, #0x1f	\n\t"
		"	mov sp, r1			\n\t"
		"	bx  r0				\n\t"
		:
		:
		:"memory"
	);
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif


static void kernelPrvSetExceptionHandlers(void)
{
	void **handlers = (void**)0xffff0020;
	
	handlers[1] = kernelPrvHandleUnd;
	handlers[2] = kernelPrvHandleSwi;
	handlers[3] = kernelPrvHandlePabt;
	handlers[4] = kernelPrvHandleDabt;
	handlers[6] = kernelPrvHandleIrq;
}


void kernelInit(void (*entryFunc)(void), void* hyperFuncIfAny)
{
	kstatus_t sta;
	void* stack;
	tid_t tid;
	
	kernelPrvSetExcStacks();
	kernelPrvSetExceptionHandlers();
	
	machInit(STAGE_INIT_SET_VTOR, NULL);
	machInit(STAGE_SETUP_HEAPS, NULL);
	machInit(STAGE_INIT_INTERRUPTS, NULL);

	timersInit();
	
	//export funcs
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_GET_CLOCK_RATE, cpuGetClockRate) ||
			!ralSetRePalmTabFunc(REPALM_FUNC_IDX_GET_TIMER_VAL, timerGetTime))
		fatal("cannot export DAL funcs\n");
	
	
	kernelGotoSysModeWithIntsOn();
	
	if (!mpuInit())
		fatal("MPU init failed\n");
	
	machInit(STAGE_INIT_MPU, NULL);
	
	machInit(STAGE_INIT_PRE_SCHED, NULL);
	if (KERN_STATUS_OK != schedInit())
		fatal("SCHED init failed\n");
	
	stack = kheapAlloc(ZEROTH_TASK_STACK_SZ);
	if (!stack)
		fatal("cannot allocate initial stack\n");
	
	sta = KTaskCreate(CREATE_4CC('t','s','k','0'), (void*)kernelFirstTask, stack, ZEROTH_TASK_STACK_SZ, NULL, SCHED_DEFAULT_PRIO, false, &tid);
	if (sta != KERN_STATUS_OK)
		fatal("cannot create first task with code %u\n", sta);

	sta = schedStart(tid, (void*)entryFunc);
	if (sta != KERN_STATUS_OK)
		fatal("sched start failed with code %u\n", sta);
	
	fatal("kernel sched failed to init\n");
}

kstatus_t kernelInitLate(void)
{
	//nothing yet
	
	return KERN_STATUS_OK;
}