#include "printf.h"
#include "rtc.h"
#include "cpu.h"
#include "kal.h"
#include "dal.h"


#define NUM_SECONDS_IN_DAY		(60UL * 60UL * 24UL)
#define ALARM_NEVER				(0xFFFFFFFFUL)

static DrvRtcCbk mAlmCbk = NULL, mHzCbk = NULL;
static uint32_t mCurTimeOfst = 0xDAB00000, mCurAlarmLeft = ALARM_NEVER;


static uint32_t rtcReadRaw(void)
{
	uint32_t val = RTC->TR;
	
	return	((val >> 20) & 0x03) * 10 * 3600 + 			//tens of hours
			((val >> 16) & 0x0f) * 3600 +				//digits of hours
			((val >> 12) & 0x07) * 10 * 60 +			//tens of minutes
			((val >>  8) & 0x0f) * 60 + 				//digits of minutes
			((val >>  4) & 0x07) * 10 + 				//tens of seconds
			((val >>  0) & 0x0f);						//digits of seconds
}

static uint32_t rtcToBcd(uint32_t secs)
{
	uint32_t ret = 0;
	
	ret += (secs % 10) << 0;
	secs /= 10;
	ret += (secs % 6) << 4;
	secs /= 6;
	ret += (secs % 10) << 8;
	secs /= 10;
	ret += (secs % 6) << 12;
	secs /= 6;
	ret += (secs % 10) << 16;
	secs /= 10;
	ret += secs << 20;
	
	return ret;
}

static uint32_t rtcGetCurTime(uint32_t rawRtc)
{
	return rawRtc + mCurTimeOfst;
}

static void rtcSetAlmRaw(uint32_t at)
{
	if (at != ALARM_NEVER)
		at = rtcToBcd(at) | 0x80000000;	//date do not care
	else
		at = 0x10000000;		//match on 10th day (never happens
	
	RTC->CR &=~ 0x100;			//disable alarm a
	while(!(RTC->ISR & 1));		//wait for alarm to be writeable
	RTC->ALRMAR = at;
	RTC->CR |= 0x100;			//wnable alarm a
}

static void rtcSetInitial(void)
{
	//enter init mode
	RTC->ISR = 0x80;
	while (!(RTC->ISR & 0x40));
	
	//setup prescaler
	RTC->PRER = 0x0063270F;		//aclock is 10KHz, rtc counter is 1Hz
	
	RTC->CR = 0;				//disable all things
	
	//config RTC and set time to zero
	RTC->DR = 0x00002101;		//monday, jan 1 2000 :)
	RTC->TR = 0;				//midnight
	RTC->ALRMBR = 0x02000000;	//alarm B at midnight to jan 2
	RTC->CR = 0x3304;			//alarm A&B on, alarm A&B ints on, 1hz clock is wakeup clock
	RTC->ISR &=~ 0x80;			//exit init mode
}

static void rtcAlarmFires(void)
{
	if(mAlmCbk)
		mAlmCbk();
}

void __attribute__((used)) RTC_Alarm_IRQHandler(void)
{
	uint32_t isr = RTC->ISR;
	
	EXTI->PR = 0x20000;	//clear int in exti
	
	if (isr & 0x200) {	//alarm b - day overflowed
		rtcSetInitial();
		mCurTimeOfst += NUM_SECONDS_IN_DAY;
		
		if (mCurAlarmLeft != ALARM_NEVER) {
			
			if (mCurAlarmLeft >= NUM_SECONDS_IN_DAY)
				mCurAlarmLeft -= NUM_SECONDS_IN_DAY;
			else if (mCurAlarmLeft)
				rtcSetAlmRaw(mCurAlarmLeft);
			else
				rtcAlarmFires();
		}
	}
	if (isr & 0x100) {	//alarm a
		rtcAlarmFires();
		mCurAlarmLeft = ALARM_NEVER;
		rtcSetAlmRaw(ALARM_NEVER);
	}
	
	RTC->ISR = isr;
}

static void rtcThreadFunc(void *param)
{
	while (1) {
		KALTaskDelay(1000);
		if(mHzCbk)
			mHzCbk();
	}
}
		
kstatus_t drvRtcInit(DrvRtcCbk hzCbk, DrvRtcCbk almCbk)
{
	static const struct KalTaskCreateInfo rtcThCrNfo = {		//this is a shitty way to do this, but since stm32f429 HW is walking dead, who cares?
		.codeToRun = rtcThreadFunc,
		.stackSz = 512,
		.prio = 50,	//low-ish prio
		.tag = CREATE_4CC('_','r','t','c'),
	};
	
	static tid_t tid;
	
	mAlmCbk = almCbk;
	mHzCbk = hzCbk;
	
	PWR->CR |= 0x100;		//allow write to backup domain
	RCC->BDCR = 0x10000;	//backup domain reset
	RCC->BDCR = 0x10300;	//backup domain reset, clock is HSE
	RCC->BDCR = 0x00300;	//RTC clock is HSE
	RCC->BDCR = 0x08300;	//RTC is on, clock is HSE
	
	//unlock regs
	RTC->WPR = 0xCA;
	RTC->WPR = 0x53;
	
	EXTI->RTSR |= 0x20000;	//detect rising edge on EXTI17 (RTC int)
	EXTI->PR = 0x20000;		//clear current status on EXTI17 (RTC int)
	EXTI->IMR |= 0x20000;	//interrupt is on for EXTI17 (RTC int)
	
	rtcSetInitial();
	rtcSetAlmRaw(ALARM_NEVER);	//alarm A off
	NVIC_EnableIRQ(RTC_Alarm_IRQn);
	
	if (errNone != KALTaskCreate(&tid, &rtcThCrNfo))
		fatal("Cannot create RTC thread\n");

	if (errNone != KALTaskStart(tid, NULL))
		fatal("Cannot start RTC thread\n");
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcPreSleep(void)
{
	//todo
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcPostWake(void)
{
	//todo
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcGet(uint32_t *valP)
{
	*valP = rtcGetCurTime(rtcReadRaw());
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcSet(uint32_t val)
{
	mCurTimeOfst = val - rtcReadRaw();
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcSetAlarm(uint32_t at)
{
	uint32_t raw = rtcReadRaw(), now = rtcGetCurTime(raw);
	
	if (now > at) {
		rtcSetAlmRaw(ALARM_NEVER);
		mCurAlarmLeft = ALARM_NEVER;
	}
	else {
		at -= now;
		
		if (at < NUM_SECONDS_IN_DAY - raw) {
			rtcSetAlmRaw(at + raw);
			mCurAlarmLeft = ALARM_NEVER;
		}
		else {
			rtcSetAlmRaw(ALARM_NEVER);
			mCurAlarmLeft = at + raw;
		}	
	}
	
	return KERN_STATUS_OK;
}
