#include "machSpecific.h"
#include "printf.h"
#include "input.h"
#include "cpu.h"
#include "kal.h"
#include "ral.h"
#include "dal.h"


static DrvInputKeyCbk mKeyCbk = NULL;
static DrvInputBtnCbk mBtnCbk = NULL;
static DrvInputPenCbk mPenCbk = NULL;
static bool mPrevBtnState;
static tid_t mI2cThread;

#define REG_CHIP_ID			0x00	//16b
#define REG_ID_VER			0x02
#define REG_SYS_CTRL1		0x03
#define REG_SYS_CTRL2		0x04
#define REG_INT_CTRL		0x09
#define REG_INT_EN			0x0A
#define REG_INT_STA			0x0B
#define REG_GPIO_INT_EN		0x0C
#define REG_ADC_INT_EN		0x0E
#define REG_GPIO_AF			0x17
#define REG_TSC_CTRL		0x40
#define REG_TSC_CFG			0x41
#define REG_WDW_TR_X		0x42	//16b
#define REG_WDW_TR_Y		0x44	//16b
#define REG_WDW_BL_X		0x46	//16b
#define REG_WDW_BL_Y		0x48	//16b
#define REG_FIFO_TH			0x4A
#define REG_FIFO_STA		0x4B
#define REG_FIFO_SIZE		0x4C
#define REG_TSC_DATA_X		0x4D	//16b
#define REG_TSC_DATA_Y		0x4F	//16b
#define REG_TSC_DATA_Z		0x51
#define REG_TSC_DATA_XYZ	0x52	//32b
#define REG_TSC_FRACTION_Z	0x56
#define REG_TSC_DATA		0x57
#define REG_TSC_I_DRIVE		0x58
#define REG_TSC_SHIELD		0x59
#define REG_TSC_DATA_NOINC	0xD7	//same as REG_TSC_DATA but no autoincrement on address



static void i2cDelay(void)
{
	uint32_t tmp;
	
	asm volatile(
		"1:						\n\t"
		"	subs %0, #1			\n\t"
		"	bne 1b				\n\t"
		:"=l"(tmp)
		:"0"(32)
		:"memory", "cc"
	);
}

static void i2cSdaHi(void)
{
	GPIOC->BSRR = 1 << 9;
}

static void i2cSdaLo(void)
{
	GPIOC->BSRR = 1 << (16 + 9);
}

static bool i2cSdaRead(void)
{
	return !!(GPIOC->IDR & (1 << 9));
}

static bool i2cSclRead(void)
{
	return !!(GPIOA->IDR & (1 << 8));
}

static void i2cSclHi(void)
{
	GPIOA->BSRR = 1 << 8;
	while(!i2cSclRead());	//wait for clock stretching to end
}

static void i2cSclLo(void)
{
	GPIOA->BSRR = 1 << (16 + 8);
}

static void i2cBitW(bool hi)
{
	if (hi)
		i2cSdaHi();
	else
		i2cSdaLo();
	i2cDelay();
	i2cSclHi();
	i2cDelay();
	i2cSclLo();
}

static bool i2cBitR(void)
{
	bool ret;
	
	i2cSdaHi();
	i2cDelay();
	i2cSclHi();
	i2cDelay();
	
	ret = i2cSdaRead();
	
	i2cSclLo();
	
	return ret;
}

static void i2cStart(void)
{
	i2cSclHi();
	i2cSdaHi();
	i2cDelay();
	i2cSdaLo();
	i2cDelay();
	i2cSclLo();
}

static void i2cStop(void)
{
	i2cSdaLo();
	i2cDelay();
	i2cSclHi();
	i2cDelay();
	i2cSdaHi();
	i2cDelay();
}

static bool i2cByteW(uint32_t val)
{
	uint32_t i;
	
	for (i = 0; i < 8; i++, val <<= 1)
		i2cBitW(!!(val & 0x80));
	
	return !i2cBitR();
}

static uint32_t i2cByteR(bool ack)
{
	uint32_t i, v = 0;
	
	for (i = 0; i < 8; i++)
		v = (v << 1) | (i2cBitR() ? 1 : 0);
	
	i2cBitW(!ack);
	
	return v;
}

static bool i2cWrite(uint8_t dev, uint8_t reg, const uint8_t *bytes, uint32_t len)	//guaranteed to permanently fuck up the I2C unit internal state machine if you get a NAK. FUCK ST for this shitty I2C unit!
{
	i2cStart();
	if (!i2cByteW(dev * 2 + 0)) {
		loge("write address naked\n");
		return false;
	}
	if (!i2cByteW(reg)) {
		loge("write reg naked\n");
		return false;
	}
	
	while (len) {
		
		len--;
		if (!i2cByteW(*bytes++)) {
			
			if (len) {
				loge("early nak\n");
				i2cStop();
				return false;
			}
		}
	}
	
	i2cStop();
	return true;
}

static bool i2cRead(uint8_t dev, uint8_t reg, uint8_t *bytes, uint32_t len)	//guaranteed to permanently fuck up the I2C unit internal state machine if you get a NAK. FUCK ST for this shitty I2C unit!
{
	i2cStart();
	if (!i2cByteW(dev * 2 + 0)) {
		loge("write address naked\n");
		return false;
	}
	if (!i2cByteW(reg)) {
		loge("write reg naked\n");
		return false;
	}
	i2cStart();
	if (!i2cByteW(dev * 2 + 1)) {
		loge("read address naked\n");
		return false;
	}
	
	while (len) {
		
		len--;
		*bytes++ = i2cByteR(!!len);
	}
	
	i2cStop();
	return true;
}

static int64_t touchRegRead(uint8_t regAddr, uint32_t regSz)
{
	uint32_t reg = 0;
	
	if (!i2cRead(0x41, regAddr, ((uint8_t*)&reg) + (4 - regSz), regSz))
		return -1;
	
	return (uint64_t)(uint32_t)__builtin_bswap32(reg);
}

static bool touchRegWrite(uint32_t regAddr, uint32_t regSz, uint32_t val)
{
	uint32_t reg = __builtin_bswap32(val);
	
	return i2cWrite(0x41, regAddr, ((uint8_t*)&reg) + (4 - regSz), regSz);
}

void __attribute__((used)) EXTI15_10_IRQHandler(void)
{
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	
	if (EXTI->PR & 0x8000) {
		EXTI->PR = 0x8000;
		KALTaskWake(mI2cThread);
	}
	
	ralRestoreR9(r9);
}

void __attribute__((used)) EXTI0_IRQHandler(void)
{
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	bool down;
	
	while (EXTI->PR & 0x0001) {
		EXTI->PR = 0x0001;
		down = !!(GPIOA->IDR & 1);
		
		if (!mPrevBtnState != !down) {
			
			if (mBtnCbk)
				mBtnCbk(0x00000001 /* power button */, down);
			mPrevBtnState = down;
		}
	}
	ralRestoreR9(r9);
}


static void i2cThreadFunc(void *param)
{
	bool penWasDown = false;
	
	logi("TOUCH: CHIP_ID = 0x%04llx, ID_VER=0x%02x\n", (signed long long)touchRegRead(REG_CHIP_ID, 2), (signed long long)touchRegRead(REG_ID_VER, 1));

	//generic config
	touchRegWrite(REG_SYS_CTRL2, 1, 0x08);		//touch, gpio & adc on, temp sensor off
	touchRegWrite(REG_INT_EN, 1, 0x03);			//ints on: fifo above threshold, touch
	touchRegWrite(REG_GPIO_INT_EN, 1, 0x00);	//gpio ints off
	touchRegWrite(REG_ADC_INT_EN, 1, 0x00);		//adc ints off
	
	//TSC cfg
	touchRegWrite(REG_TSC_CFG, 1, 0xE3);		//8 sample average, touch detect delay of 5ms, settling time 1ms
	touchRegWrite(REG_FIFO_TH, 1, 1);			//if more than one sample present, alert me
	touchRegWrite(REG_TSC_I_DRIVE, 1, 0);		//lowere current TSC drive
	touchRegWrite(REG_TSC_CTRL, 1, 0x12);		//tracking window of 4, X&Y only, TSC off
	touchRegWrite(REG_TSC_CTRL, 1, 0x13);		//tracking window of 4, X&Y only, TSC on
	
	//ints on
	touchRegWrite(REG_INT_STA, 1, 0xFF);		//clear all ints
	touchRegWrite(REG_INT_CTRL, 1, 0x01);		//active low level interrupts, enabled
	
	
	while(1) {
		
		KALTaskWait(-1);
		
		do {
			
			//while fifo isnt empty, read data
			while (!(touchRegRead(REG_FIFO_STA, 1) & 0x20)) {
				uint32_t data = touchRegRead(REG_TSC_DATA_NOINC, 3);
				uint32_t x = data >> 12, y = data & 0xfff;
				
				if (mPenCbk)
					mPenCbk(x / 16, y / 16);
				
				penWasDown = true;
				
			}
			
			//IRQ handler, basically: 
			//clear int status (so we know when it re-happens)
			touchRegWrite(REG_INT_STA, 1, 0xff);
			
			//now check if pen was released
			if (mPenCbk && penWasDown && !(touchRegRead(REG_TSC_CTRL, 1) & 0x80)) {
				penWasDown = false;
				mPenCbk(-1, -1);
			}
		} while(!(GPIOA->IDR & 0x8000));
	}
}

kstatus_t drvInputInit(DrvInputKeyCbk keyCbk, DrvInputBtnCbk btnCbk, DrvInputPenCbk penCbk)
{
	static const struct KalTaskCreateInfo i2cThCrNfo = {
		.codeToRun = i2cThreadFunc,
		.stackSz = 512,
		.prio = 5,	//hi prio
		.tag = CREATE_4CC('_','i','2','c'),
	};
	
	mKeyCbk = keyCbk;
	mBtnCbk = btnCbk;
	mPenCbk = penCbk;
	
	GPIOA->BSRR = 1 << 8;
	GPIOC->BSRR = 1 << 9;
	
	EXTI->FTSR |= 0x8001;	//detect falling edge on EXTI15 (aka A15) and EXTI0 (A0)
	EXTI->RTSR |= 0x0001;	//detect rising edge on EXTI0 (A0)
	EXTI->IMR |= 0x8001;	//interrupt is on for EXTI15 (aka A15) and EXTI0 (A0)
	EXTI->PR = 0x8001;		//clear current status on EXTI15 (aka A15) and EXTI0 (A0)
	machBusyWaitDelayMsec(1);
	
	if (errNone != KALTaskCreate(&mI2cThread, &i2cThCrNfo))
		fatal("Cannot create i2c thread\n");

	if (errNone != KALTaskStart(mI2cThread, NULL))
		fatal("Cannot start i2c thread\n");
	
	logi("i2c Thread is up\n");
	
	NVIC_EnableIRQ(EXTI15_10_IRQn);
	NVIC_EnableIRQ(EXTI0_IRQn);

	return KERN_STATUS_OK;
}

kstatus_t drvInputPreSleep(void)
{
	return KERN_STATUS_OK;
}

kstatus_t drvInputPostWake(void)
{
	return KERN_STATUS_OK;
}

uint32_t drvInputReadKeysEarly(void)
{
	return 0;
}
