#include "disp.h"
#include "printf.h"
#include "machSpecific.h"
#include <MemoryMgr.h>
#include <string.h>
#include "memmap.h"
#include "timers.h"
#include "boot.h"
#include "heap.h"
#include "cpu.h"
#include "dal.h"
#include "kal.h"


#define DISP_WIDTH				240			//should be divisile by 8
#define DISP_HEIGHT				320
#define DISP_DENSITY			108


static void* mFb;
static uint8_t mCurDepth;
static bool haveGrafArea = false;


#define USE_DMA2D			1


static void lcdCsLo(void)
{
	GPIOC->BSRR = 1 << (16 + 2);
}

static void lcdCsHi(void)
{
	GPIOC->BSRR = 1 << 2;
}

static void lcdWrite(uint8_t byte)	//spi 1-wire
{
	uint64_t ticks;
	
	SPI5->CR1 &= ~SPI_CR1_SPE;
	SPI5->CR1 |= SPI_CR1_BIDIMODE;
	SPI5->CR1 |= SPI_CR1_BIDIOE;	//1-wire mode output
	
	lcdCsLo();
	SPI5->CR1 |= SPI_CR1_SPE;
	
	SPI5->DR = byte;
	
	ticks = timerGetTime();
	while (!(SPI5->SR & SPI_SR_TXE)) {
		if (timerGetTime() - ticks >= TIMER_TICKS_PER_MSEC / 100)	//10ms total
			fatal("timeout TXE\n");
	}
	while (SPI5->SR & SPI_SR_BSY) {
		if (timerGetTime() - ticks >= TIMER_TICKS_PER_MSEC / 100)	//10ms total
			fatal("timeout BSY\n");
	}
	
	SPI5->CR1 &=~ SPI_CR1_SPE;
	lcdCsHi();
	
	(void)SPI5->DR;
	(void)SPI5->SR;
}

/*
static uint8_t lcdRead(void)	//spi 1-wire
{
	uint64_t ticks;
	uint8_t b;
	
	SPI5->CR1 &=~ SPI_CR1_SPE;
	SPI5->CR1 |= SPI_CR1_BIDIMODE;
	SPI5->CR1 &=~ SPI_CR1_BIDIOE;	//1-wire mode input
	
	lcdCsLo();
	SPI5->CR1 |= SPI_CR1_SPE;
	
	ticks = timerGetTime();
	while (!(SPI5->SR & SPI_SR_RXNE)) {
		if (timerGetTime() - ticks >= TIMER_TICKS_PER_MSEC / 100)	//10ms total
			fatal("timeout RXNE\n");
	}
	b = SPI5->DR;
	SPI5->CR1 &=~ SPI_CR1_SPE;
	
	// clear rx statuses
	(void)SPI5->DR;
	(void)SPI5->SR;
	
	lcdCsHi();
	
	return b;
}
*/

static void lcdWriteReg(uint8_t reg)
{
	GPIOD->BSRR = 1 << (16 + 13);
	lcdWrite(reg);
}

static void lcdWriteData(uint8_t reg)
{
	GPIOD->BSRR = 1 << 13;
	lcdWrite(reg);
}

static void lcdInitSeq(const uint16_t* data, uint32_t items)
{
	uint32_t v;
	
	while(items--) {
		v = *data++;
		
		if (v & 0x8000)
			lcdWriteReg((uint8_t)v);
		else
			lcdWriteData((uint8_t)v);
	}
}

static void dispInitLtdc(void)	//setup LCD interface
{
	/* Timing configuration  (Typical configuration from ILI9341 datasheet)
		HSYNC=10 (9+1)
		HBP=20 (29-10+1)
		ActiveW=240 (269-20-10+1)
		HFP=10 (279-240-20-10+1)
		
		VSYNC=2 (1+1)
		VBP=2 (3-2+1)
		ActiveH=320 (323-2-2+1)
		VFP=4 (327-320-2-2+1)
	*/
	
	#define LTDC_HSPOLARITY_AL                ((uint32_t)0x00000000U)                /*!< Horizontal Synchronization is active low. */
	#define LTDC_HSPOLARITY_AH                LTDC_GCR_HSPOL                        /*!< Horizontal Synchronization is active high. */
	#define LTDC_VSPOLARITY_AL                ((uint32_t)0x00000000U)                /*!< Vertical Synchronization is active low. */
	#define LTDC_VSPOLARITY_AH                LTDC_GCR_VSPOL                        /*!< Vertical Synchronization is active high. */
	#define LTDC_DEPOLARITY_AL                ((uint32_t)0x00000000U)                /*!< Data Enable, is active low. */
	#define LTDC_DEPOLARITY_AH                LTDC_GCR_DEPOL                        /*!< Data Enable, is active high. */
	#define LTDC_PCPOLARITY_IPC               ((uint32_t)0x00000000U)                /*!< input pixel clock. */
	#define LTDC_PCPOLARITY_IIPC              LTDC_GCR_PCPOL                        /*!< inverted input pixel clock. */

	#define  ILI9341_HSYNC            ((uint32_t)35)   /* Horizontal synchronization */
	#define  ILI9341_HBP              ((uint32_t)29)    /* Horizontal back porch      */ 
	#define  ILI9341_HFP              ((uint32_t)2)    /* Horizontal front porch     */
	#define  ILI9341_VSYNC            ((uint32_t)1)   /* Vertical synchronization   */
	#define  ILI9341_VBP              ((uint32_t)3)    /* Vertical back porch        */
	#define  ILI9341_VFP              ((uint32_t)2)    /* Vertical front porch       */

	
	// Configure the HS, VS, DE and PC polarity
	LTDC->GCR = (LTDC->GCR &~ (LTDC_GCR_HSPOL | LTDC_GCR_VSPOL | LTDC_GCR_DEPOL | LTDC_GCR_PCPOL)) | LTDC_HSPOLARITY_AL | LTDC_VSPOLARITY_AL | LTDC_DEPOLARITY_AL | LTDC_PCPOLARITY_IPC;
	
	// Set Synchronization size
	LTDC->SSCR = (ILI9341_HSYNC << 16U) | ILI9341_VSYNC;
	
	// Set Accumulated Back porch
	LTDC->BPCR = (ILI9341_HBP << 16U) | ILI9341_VBP;
	
	// Set Accumulated Active Width
	LTDC->AWCR = (269 << 16U) | 323;
	
	// Set Total Width
	LTDC->TWCR = (279 << 16U) | 327;
	
	// Set the background color to black (format is XRGB8888)
	LTDC->BCCR = 0x00000000;
	
	// Enable LTDC
	LTDC->GCR |= LTDC_GCR_LTDCEN;
	
	// set CS
	lcdCsLo();
	lcdCsHi();
}

static void dispInitSpi(void)
{
	#define SPI_MODE_SLAVE                  ((uint32_t)0x00000000U)
	#define SPI_MODE_MASTER                 (SPI_CR1_MSTR | SPI_CR1_SSI)
	#define SPI_DIRECTION_2LINES            ((uint32_t)0x00000000U)
	#define SPI_DIRECTION_2LINES_RXONLY     SPI_CR1_RXONLY
	#define SPI_DIRECTION_1LINE             SPI_CR1_BIDIMODE
	#define SPI_DATASIZE_8BIT               ((uint32_t)0x00000000U)
	#define SPI_DATASIZE_16BIT              SPI_CR1_DFF
	#define SPI_POLARITY_LOW                ((uint32_t)0x00000000U)
	#define SPI_POLARITY_HIGH               SPI_CR1_CPOL
	#define SPI_PHASE_1EDGE                 ((uint32_t)0x00000000U)
	#define SPI_PHASE_2EDGE                 SPI_CR1_CPHA
	#define SPI_NSS_SOFT                    SPI_CR1_SSM
	#define SPI_NSS_HARD_INPUT              ((uint32_t)0x00000000U)
	#define SPI_NSS_HARD_OUTPUT             ((uint32_t)0x00040000U)
	#define SPI_BAUDRATEPRESCALER_2         ((uint32_t)0x00000000U)
	#define SPI_BAUDRATEPRESCALER_4         ((uint32_t)0x00000008U)
	#define SPI_BAUDRATEPRESCALER_8         ((uint32_t)0x00000010U)
	#define SPI_BAUDRATEPRESCALER_16        ((uint32_t)0x00000018U)
	#define SPI_BAUDRATEPRESCALER_32        ((uint32_t)0x00000020U)
	#define SPI_BAUDRATEPRESCALER_64        ((uint32_t)0x00000028U)
	#define SPI_BAUDRATEPRESCALER_128       ((uint32_t)0x00000030U)
	#define SPI_BAUDRATEPRESCALER_256       ((uint32_t)0x00000038U)
	#define SPI_FIRSTBIT_MSB                ((uint32_t)0x00000000U)
	#define SPI_FIRSTBIT_LSB                SPI_CR1_LSBFIRST
	#define SPI_CRCCALCULATION_DISABLE     ((uint32_t)0x00000000U)
	#define SPI_CRCCALCULATION_ENABLE      SPI_CR1_CRCEN
	#define SPI_TIMODE_DISABLE             ((uint32_t)0x00000000U)
	#define SPI_TIMODE_ENABLE              SPI_CR2_FRF

	//spi off
	SPI5->CR1 = 0;
	
	//spi config (direction is 2 for gyro and 1 for lcd, keep 2 for now)
	SPI5->CR1 = SPI_MODE_MASTER | SPI_DIRECTION_2LINES | SPI_DATASIZE_8BIT | SPI_POLARITY_LOW | SPI_PHASE_1EDGE | (SPI_NSS_SOFT & SPI_CR1_SSM) | SPI_BAUDRATEPRESCALER_16 | SPI_FIRSTBIT_MSB | SPI_CRCCALCULATION_DISABLE;
	
	SPI5->CR2 = ((SPI_NSS_SOFT >> 16) & SPI_CR2_SSOE) | SPI_TIMODE_DISABLE;
}

static void dispInitController(void)	//setup Display's registers
{
	#define LCD_SWRESET             0x01   /* Software Reset */
	#define LCD_READ_DISPLAY_ID     0x04   /* Read display identification information */
	#define LCD_RDDST               0x09   /* Read Display Status */
	#define LCD_RDDPM               0x0A   /* Read Display Power Mode */
	#define LCD_RDDMADCTL           0x0B   /* Read Display MADCTL */
	#define LCD_RDDCOLMOD           0x0C   /* Read Display Pixel Format */
	#define LCD_RDDIM               0x0D   /* Read Display Image Format */
	#define LCD_RDDSM               0x0E   /* Read Display Signal Mode */
	#define LCD_RDDSDR              0x0F   /* Read Display Self-Diagnostic Result */
	#define LCD_SPLIN               0x10   /* Enter Sleep Mode */
	#define LCD_SLEEP_OUT           0x11   /* Sleep out register */
	#define LCD_PTLON               0x12   /* Partial Mode ON */
	#define LCD_NORMAL_MODE_ON      0x13   /* Normal Display Mode ON */
	#define LCD_DINVOFF             0x20   /* Display Inversion OFF */
	#define LCD_DINVON              0x21   /* Display Inversion ON */
	#define LCD_GAMMA               0x26   /* Gamma register */
	#define LCD_DISPLAY_OFF         0x28   /* Display off register */
	#define LCD_DISPLAY_ON          0x29   /* Display on register */
	#define LCD_COLUMN_ADDR         0x2A   /* Colomn address register */ 
	#define LCD_PAGE_ADDR           0x2B   /* Page address register */ 
	#define LCD_GRAM                0x2C   /* GRAM register */   
	#define LCD_RGBSET              0x2D   /* Color SET */   
	#define LCD_RAMRD               0x2E   /* Memory Read */   
	#define LCD_PLTAR               0x30   /* Partial Area */   
	#define LCD_VSCRDEF             0x33   /* Vertical Scrolling Definition */   
	#define LCD_TEOFF               0x34   /* Tearing Effect Line OFF */   
	#define LCD_TEON                0x35   /* Tearing Effect Line ON */   
	#define LCD_MAC                 0x36   /* Memory Access Control register*/
	#define LCD_VSCRSADD            0x37   /* Vertical Scrolling Start Address */   
	#define LCD_IDMOFF              0x38   /* Idle Mode OFF */   
	#define LCD_IDMON               0x39   /* Idle Mode ON */   
	#define LCD_PIXEL_FORMAT        0x3A   /* Pixel Format register */
	#define LCD_WRITE_MEM_CONTINUE  0x3C   /* Write Memory Continue */   
	#define LCD_READ_MEM_CONTINUE   0x3E   /* Read Memory Continue */   
	#define LCD_SET_TEAR_SCANLINE   0x44   /* Set Tear Scanline */   
	#define LCD_GET_SCANLINE        0x45   /* Get Scanline */   
	#define LCD_WDB                 0x51   /* Write Brightness Display register */
	#define LCD_RDDISBV             0x52   /* Read Display Brightness */   
	#define LCD_WCD                 0x53   /* Write Control Display register*/
	#define LCD_RDCTRLD             0x54   /* Read CTRL Display */   
	#define LCD_WRCABC              0x55   /* Write Content Adaptive Brightness Control */   
	#define LCD_RDCABC              0x56   /* Read Content Adaptive Brightness Control */   
	#define LCD_WRITE_CABC          0x5E   /* Write CABC Minimum Brightness */   
	#define LCD_READ_CABC           0x5F   /* Read CABC Minimum Brightness */   
	#define LCD_READ_ID1            0xDA   /* Read ID1 */
	#define LCD_READ_ID2            0xDB   /* Read ID2 */
	#define LCD_READ_ID3            0xDC   /* Read ID3 */
	
	/* Level 2 Commands */
	#define LCD_RGB_INTERFACE       0xB0   /* RGB Interface Signal Control */
	#define LCD_FRMCTR1             0xB1   /* Frame Rate Control (In Normal Mode) */
	#define LCD_FRMCTR2             0xB2   /* Frame Rate Control (In Idle Mode) */
	#define LCD_FRMCTR3             0xB3   /* Frame Rate Control (In Partial Mode) */
	#define LCD_INVTR               0xB4   /* Display Inversion Control */
	#define LCD_BPC                 0xB5   /* Blanking Porch Control register */
	#define LCD_DFC                 0xB6   /* Display Function Control register */
	#define LCD_ETMOD               0xB7   /* Entry Mode Set */
	#define LCD_BACKLIGHT1          0xB8   /* Backlight Control 1 */
	#define LCD_BACKLIGHT2          0xB9   /* Backlight Control 2 */
	#define LCD_BACKLIGHT3          0xBA   /* Backlight Control 3 */
	#define LCD_BACKLIGHT4          0xBB   /* Backlight Control 4 */
	#define LCD_BACKLIGHT5          0xBC   /* Backlight Control 5 */
	#define LCD_BACKLIGHT7          0xBE   /* Backlight Control 7 */
	#define LCD_BACKLIGHT8          0xBF   /* Backlight Control 8 */
	#define LCD_POWER1              0xC0   /* Power Control 1 register */
	#define LCD_POWER2              0xC1   /* Power Control 2 register */
	#define LCD_VCOM1               0xC5   /* VCOM Control 1 register */
	#define LCD_VCOM2               0xC7   /* VCOM Control 2 register */
	#define LCD_NVMWR               0xD0   /* NV Memory Write */
	#define LCD_NVMPKEY             0xD1   /* NV Memory Protection Key */
	#define LCD_RDNVM               0xD2   /* NV Memory Status Read */
	#define LCD_READ_ID4            0xD3   /* Read ID4 */
	#define LCD_PGAMMA              0xE0   /* Positive Gamma Correction register */
	#define LCD_NGAMMA              0xE1   /* Negative Gamma Correction register */
	#define LCD_DGAMCTRL1           0xE2   /* Digital Gamma Control 1 */
	#define LCD_DGAMCTRL2           0xE3   /* Digital Gamma Control 2 */
	#define LCD_INTERFACE           0xF6   /* Interface control register */
	
	/* Extend register commands */
	#define LCD_POWERA               0xCB   /* Power control A register */
	#define LCD_POWERB               0xCF   /* Power control B register */
	#define LCD_DTCA                 0xE8   /* Driver timing control A */
	#define LCD_DTCB                 0xEA   /* Driver timing control B */
	#define LCD_POWER_SEQ            0xED   /* Power on sequence register */
	#define LCD_3GAMMA_EN            0xF2   /* 3 Gamma enable register */
	#define LCD_PRC                  0xF7   /* Pump ratio control register */
	
	static const uint16_t init1[] = {0x8000 | 0xCA, 0xC3, 0x08, 0x50, 0x8000 | LCD_POWERB, 0x00, 0xC1, 0x30, 0x8000 | LCD_POWER_SEQ, 0x64, 0x03, 0x12, 0x81, 0x8000 | LCD_DTCA, 0x85, 0x00, 0x78, 0x8000 | LCD_POWERA, 0x39, 0x2C, 0x00, 0x34, 0x02, 0x8000 | LCD_PRC, 0x20, 0x8000 | LCD_DTCB, 0x00, 0x00, 0x8000 | LCD_FRMCTR1, 0x00, 0x1B, 0x8000 | LCD_DFC, 0x0A, 0xA2, 0x8000 | LCD_POWER1, 0x10, 0x8000 | LCD_POWER2, 0x10, 0x8000 | LCD_VCOM1, 0x45, 0x15, 0x8000 | LCD_VCOM2, 0x90, 0x8000 | LCD_MAC, 0xC8, 0x8000 | LCD_3GAMMA_EN, 0x00, 0x8000 | LCD_RGB_INTERFACE, 0xC2, 0x8000 | LCD_DFC, 0x0A, 0xA7, 0x27, 0x04, 0x8000 | LCD_COLUMN_ADDR, 0x00, 0x00, 0x00, 0xEF, 0x8000 | LCD_PAGE_ADDR, 0x00, 0x00, 0x01, 0x3F, 0x8000 | LCD_INTERFACE, 0x01, 0x00, 0x06, 0x8000 | LCD_GRAM};
	static const uint16_t init2[] = {0x8000 | LCD_GAMMA, 0x01, 0x8000 | LCD_PGAMMA, 0x0F, 0x29, 0x24, 0x0C, 0x0E, 0x09, 0x4E, 0x78, 0x3C, 0x09, 0x13, 0x05, 0x17, 0x11, 0x00, 0x8000 | LCD_NGAMMA, 0x00, 0x16, 0x1B, 0x04, 0x11, 0x07, 0x31, 0x33, 0x42, 0x05, 0x0C, 0x0A, 0x28, 0x2F, 0x0F, 0x8000 | LCD_SLEEP_OUT};
	static const uint16_t init3[] = {0x8000 | LCD_DISPLAY_ON, 0x8000 | LCD_GRAM};
	
	lcdInitSeq(init1, sizeof(init1) / sizeof(*init1));
	machBusyWaitDelayMsec(200);
	lcdInitSeq(init2, sizeof(init2) / sizeof(*init2));
	machBusyWaitDelayMsec(200);
	lcdInitSeq(init3, sizeof(init3) / sizeof(*init3));
}

static bool dispSetupLayer(bool second, void* data, uint32_t w, uint32_t h, uint32_t lineBytes, bool bpp16)	//second layer always below first (for graf area)
{
	int32_t lcdW = 240, lcdH = 320, spaceLeft, spaceRight, spaceTop, spaceBottom, pixelSz = bpp16 ? 2 : 1;
	LTDC_Layer_TypeDef* layer = second ? LTDC_Layer2 : LTDC_Layer1;
	
	spaceLeft = (lcdW - w) / 2;
	spaceRight = lcdW - w - spaceLeft;
	spaceTop = 0;	//to vertically center use this instead: (lcdH - h) / 2;
	if (second)
		spaceTop += DISP_HEIGHT;
	spaceBottom = lcdH - h - spaceTop;
	
	if (spaceBottom < 0) {		//do not allow fake grafiti to go below screen bottom
		h += spaceBottom;
		spaceBottom = 0;
	}
	
	if (layer->CR & LTDC_LxCR_LEN) {		//if it is on, turn it off
		layer->CR = 0;
		LTDC->SRCR = 2;		//make it happen & wait
		while(LTDC->SRCR);
	}
	
	layer->WHPCR = ((269 - spaceRight) << 16) | ((ILI9341_HBP + spaceLeft + 1) << 0);
	layer->WVPCR = ((323 - spaceBottom) << 16) | ((ILI9341_VBP + spaceTop + 1) << 0);
	layer->CFBLR = (lineBytes << 16) | ((w * pixelSz) + 3);
	layer->CFBLNR = h;
	
	layer->PFCR = bpp16 ? 2 : 5;	//RGB565 or L8
	layer->CACR = 255;
	layer->DCCR = 0;
	layer->BFCR = 0x607;			//constant alpha
	layer->CFBAR = (uintptr_t)data;
	
	if (second) {	//must be teansparent outside it sboundaries..somehow that is NOT the default
		
		layer->CR = 0x01;			//layer on, clut off, color keying on
	}
	else{
	
		layer->CR = bpp16 ? 1 : 0x11;	//clut on for 8-bit main layer, not for second
	}
	
	//make it happen
	LTDC->SRCR = 2;
	
	return true;
}

void dispSetBri(uint8_t bri)
{
	//not doable on this board
}

void dispSetContrast(uint8_t bri)
{
	//not doable on this board	
}

bool dispSetBacklight(bool on)
{
	return false;	//not switchable
}

bool dispGetBacklight(void)
{
	return false;	//always off
}

bool dispDrvInit(uint16_t *wP, uint16_t* hP, uint16_t* densityP, uint16_t *realDpiP, uint16_t *supportedDepthMapP, void** framebufferP, bool *indexedFmtIsLEP)
{
	uint32_t vramSz = DISP_WIDTH * DISP_HEIGHT * sizeof(uint16_t);
	
	#ifdef CPU_HARDWIRED_VRAM_ADDR
		if (CPU_HARDWIRED_VRAM_SIZE >= vramSz)
			mFb = (void*)CPU_HARDWIRED_VRAM_ADDR;
		else
			fatal("VRAM is too small %u < %u\n", CPU_HARDWIRED_VRAM_SIZE, vramSz);
	#else
		mFb = kheapAlloc(vramSz);
		if (!mFb) {
			loge("Cannot alloc fb\n");
			return false;
		}
	#endif
	
	*wP = DISP_WIDTH;
	*hP = DISP_HEIGHT;
	*densityP = DISP_DENSITY;
	*supportedDepthMapP = ((1 << 16) | (1 << 8) | (1 << 4) | (1 << 2) | (1 << 1)) >> 1;
	*framebufferP = mFb;
	*realDpiP = 160;
	*indexedFmtIsLEP = false;
	
	dispInitLtdc();
	dispInitSpi();
	dispInitController();
	mCurDepth = 16;
	return dispSetupLayer(false, mFb, DISP_WIDTH, DISP_HEIGHT, DISP_WIDTH * 2, true);
}

void dispSetClut(int32_t firstIdx, uint32_t numEntries, const struct PalmClutEntry *entries)
{
	uint32_t i;
	
	if (firstIdx == -1) {
		
		if (numEntries > 256)
			return;
	}
	else if (firstIdx < 0)
		return;
	else if (firstIdx >= 256 || numEntries + firstIdx > 256)
		return;

	for (i = 0; i < numEntries; i++) {
		
		uint32_t where = (firstIdx == -1) ? entries[i].idx : i + firstIdx;
		uint32_t r = entries[i].r;
		uint32_t g = entries[i].g;
		uint32_t b = entries[i].b;
		
		LTDC_Layer1->CLUTWR = (where << LTDC_LxCLUTWR_CLUTADD_Pos) | (r << LTDC_LxCLUTWR_RED_Pos) | (g << LTDC_LxCLUTWR_GREEN_Pos) | (b << LTDC_LxCLUTWR_BLUE_Pos);
	}
}

static void dispConfigResampler(uint32_t depth)
{
	static const uint32_t mOneBppFirstStageLut[] = {0xff000000, 0xff002000, 0xff008000, 0xff00a000, 0xff100000, 0xff102000, 0xff108000, 0xff10a000, 0xff400000, 0xff402000, 0xff408000, 0xff40a000, 0xff500000, 0xff502000, 0xff508000, 0xff50a000, 0xff000008, 0xff002008, 0xff008008, 0xff00a008, 0xff100008, 0xff102008, 0xff108008, 0xff10a008, 0xff400008, 0xff402008, 0xff408008, 0xff40a008, 0xff500008, 0xff502008, 0xff508008, 0xff50a008, 0xff000020, 0xff002020, 0xff008020, 0xff00a020, 0xff100020, 0xff102020, 0xff108020, 0xff10a020, 0xff400020, 0xff402020, 0xff408020, 0xff40a020, 0xff500020, 0xff502020, 0xff508020, 0xff50a020, 0xff000028, 0xff002028, 0xff008028, 0xff00a028, 0xff100028, 0xff102028, 0xff108028, 0xff10a028, 0xff400028, 0xff402028, 0xff408028, 0xff40a028, 0xff500028, 0xff502028, 0xff508028, 0xff50a028, 0xff000080, 0xff002080, 0xff008080, 0xff00a080, 0xff100080, 0xff102080, 0xff108080, 0xff10a080, 0xff400080, 0xff402080, 0xff408080, 0xff40a080, 0xff500080, 0xff502080, 0xff508080, 0xff50a080, 0xff000088, 0xff002088, 0xff008088, 0xff00a088, 0xff100088, 0xff102088, 0xff108088, 0xff10a088, 0xff400088, 0xff402088, 0xff408088, 0xff40a088, 0xff500088, 0xff502088, 0xff508088, 0xff50a088, 0xff0000a0, 0xff0020a0, 0xff0080a0, 0xff00a0a0, 0xff1000a0, 0xff1020a0, 0xff1080a0, 0xff10a0a0, 0xff4000a0, 0xff4020a0, 0xff4080a0, 0xff40a0a0, 0xff5000a0, 0xff5020a0, 0xff5080a0, 0xff50a0a0, 0xff0000a8, 0xff0020a8, 0xff0080a8, 0xff00a0a8, 0xff1000a8, 0xff1020a8, 0xff1080a8, 0xff10a0a8, 0xff4000a8, 0xff4020a8, 0xff4080a8, 0xff40a0a8, 0xff5000a8, 0xff5020a8, 0xff5080a8, 0xff50a0a8, 0xff000800, 0xff002800, 0xff008800, 0xff00a800, 0xff100800, 0xff102800, 0xff108800, 0xff10a800, 0xff400800, 0xff402800, 0xff408800, 0xff40a800, 0xff500800, 0xff502800, 0xff508800, 0xff50a800, 0xff000808, 0xff002808, 0xff008808, 0xff00a808, 0xff100808, 0xff102808, 0xff108808, 0xff10a808, 0xff400808, 0xff402808, 0xff408808, 0xff40a808, 0xff500808, 0xff502808, 0xff508808, 0xff50a808, 0xff000820, 0xff002820, 0xff008820, 0xff00a820, 0xff100820, 0xff102820, 0xff108820, 0xff10a820, 0xff400820, 0xff402820, 0xff408820, 0xff40a820, 0xff500820, 0xff502820, 0xff508820, 0xff50a820, 0xff000828, 0xff002828, 0xff008828, 0xff00a828, 0xff100828, 0xff102828, 0xff108828, 0xff10a828, 0xff400828, 0xff402828, 0xff408828, 0xff40a828, 0xff500828, 0xff502828, 0xff508828, 0xff50a828, 0xff000880, 0xff002880, 0xff008880, 0xff00a880, 0xff100880, 0xff102880, 0xff108880, 0xff10a880, 0xff400880, 0xff402880, 0xff408880, 0xff40a880, 0xff500880, 0xff502880, 0xff508880, 0xff50a880, 0xff000888, 0xff002888, 0xff008888, 0xff00a888, 0xff100888, 0xff102888, 0xff108888, 0xff10a888, 0xff400888, 0xff402888, 0xff408888, 0xff40a888, 0xff500888, 0xff502888, 0xff508888, 0xff50a888, 0xff0008a0, 0xff0028a0, 0xff0088a0, 0xff00a8a0, 0xff1008a0, 0xff1028a0, 0xff1088a0, 0xff10a8a0, 0xff4008a0, 0xff4028a0, 0xff4088a0, 0xff40a8a0, 0xff5008a0, 0xff5028a0, 0xff5088a0, 0xff50a8a0, 0xff0008a8, 0xff0028a8, 0xff0088a8, 0xff00a8a8, 0xff1008a8, 0xff1028a8, 0xff1088a8, 0xff10a8a8, 0xff4008a8, 0xff4028a8, 0xff4088a8, 0xff40a8a8, 0xff5008a8, 0xff5028a8, 0xff5088a8, 0xff50a8a8, };
	static const uint32_t mTwoBppLut[] = {0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x00010000, 0x01010000, 0x02010000, 0x03010000, 0x00020000, 0x01020000, 0x02020000, 0x03020000, 0x00030000, 0x01030000, 0x02030000, 0x03030000, 0x00000100, 0x01000100, 0x02000100, 0x03000100, 0x00010100, 0x01010100, 0x02010100, 0x03010100, 0x00020100, 0x01020100, 0x02020100, 0x03020100, 0x00030100, 0x01030100, 0x02030100, 0x03030100, 0x00000200, 0x01000200, 0x02000200, 0x03000200, 0x00010200, 0x01010200, 0x02010200, 0x03010200, 0x00020200, 0x01020200, 0x02020200, 0x03020200, 0x00030200, 0x01030200, 0x02030200, 0x03030200, 0x00000300, 0x01000300, 0x02000300, 0x03000300, 0x00010300, 0x01010300, 0x02010300, 0x03010300, 0x00020300, 0x01020300, 0x02020300, 0x03020300, 0x00030300, 0x01030300, 0x02030300, 0x03030300, 0x00000001, 0x01000001, 0x02000001, 0x03000001, 0x00010001, 0x01010001, 0x02010001, 0x03010001, 0x00020001, 0x01020001, 0x02020001, 0x03020001, 0x00030001, 0x01030001, 0x02030001, 0x03030001, 0x00000101, 0x01000101, 0x02000101, 0x03000101, 0x00010101, 0x01010101, 0x02010101, 0x03010101, 0x00020101, 0x01020101, 0x02020101, 0x03020101, 0x00030101, 0x01030101, 0x02030101, 0x03030101, 0x00000201, 0x01000201, 0x02000201, 0x03000201, 0x00010201, 0x01010201, 0x02010201, 0x03010201, 0x00020201, 0x01020201, 0x02020201, 0x03020201, 0x00030201, 0x01030201, 0x02030201, 0x03030201, 0x00000301, 0x01000301, 0x02000301, 0x03000301, 0x00010301, 0x01010301, 0x02010301, 0x03010301, 0x00020301, 0x01020301, 0x02020301, 0x03020301, 0x00030301, 0x01030301, 0x02030301, 0x03030301, 0x00000002, 0x01000002, 0x02000002, 0x03000002, 0x00010002, 0x01010002, 0x02010002, 0x03010002, 0x00020002, 0x01020002, 0x02020002, 0x03020002, 0x00030002, 0x01030002, 0x02030002, 0x03030002, 0x00000102, 0x01000102, 0x02000102, 0x03000102, 0x00010102, 0x01010102, 0x02010102, 0x03010102, 0x00020102, 0x01020102, 0x02020102, 0x03020102, 0x00030102, 0x01030102, 0x02030102, 0x03030102, 0x00000202, 0x01000202, 0x02000202, 0x03000202, 0x00010202, 0x01010202, 0x02010202, 0x03010202, 0x00020202, 0x01020202, 0x02020202, 0x03020202, 0x00030202, 0x01030202, 0x02030202, 0x03030202, 0x00000302, 0x01000302, 0x02000302, 0x03000302, 0x00010302, 0x01010302, 0x02010302, 0x03010302, 0x00020302, 0x01020302, 0x02020302, 0x03020302, 0x00030302, 0x01030302, 0x02030302, 0x03030302, 0x00000003, 0x01000003, 0x02000003, 0x03000003, 0x00010003, 0x01010003, 0x02010003, 0x03010003, 0x00020003, 0x01020003, 0x02020003, 0x03020003, 0x00030003, 0x01030003, 0x02030003, 0x03030003, 0x00000103, 0x01000103, 0x02000103, 0x03000103, 0x00010103, 0x01010103, 0x02010103, 0x03010103, 0x00020103, 0x01020103, 0x02020103, 0x03020103, 0x00030103, 0x01030103, 0x02030103, 0x03030103, 0x00000203, 0x01000203, 0x02000203, 0x03000203, 0x00010203, 0x01010203, 0x02010203, 0x03010203, 0x00020203, 0x01020203, 0x02020203, 0x03020203, 0x00030203, 0x01030203, 0x02030203, 0x03030203, 0x00000303, 0x01000303, 0x02000303, 0x03000303, 0x00010303, 0x01010303, 0x02010303, 0x03010303, 0x00020303, 0x01020303, 0x02020303, 0x03020303, 0x00030303, 0x01030303, 0x02030303, 0x03030303, };
	static const uint32_t mFourBppLut[] = {0xff000000, 0xff002000, 0xff004000, 0xff006000, 0xff008000, 0xff00a000, 0xff00c000, 0xff00e000, 0xff080000, 0xff082000, 0xff084000, 0xff086000, 0xff088000, 0xff08a000, 0xff08c000, 0xff08e000, 0xff000008, 0xff002008, 0xff004008, 0xff006008, 0xff008008, 0xff00a008, 0xff00c008, 0xff00e008, 0xff080008, 0xff082008, 0xff084008, 0xff086008, 0xff088008, 0xff08a008, 0xff08c008, 0xff08e008, 0xff000010, 0xff002010, 0xff004010, 0xff006010, 0xff008010, 0xff00a010, 0xff00c010, 0xff00e010, 0xff080010, 0xff082010, 0xff084010, 0xff086010, 0xff088010, 0xff08a010, 0xff08c010, 0xff08e010, 0xff000018, 0xff002018, 0xff004018, 0xff006018, 0xff008018, 0xff00a018, 0xff00c018, 0xff00e018, 0xff080018, 0xff082018, 0xff084018, 0xff086018, 0xff088018, 0xff08a018, 0xff08c018, 0xff08e018, 0xff000020, 0xff002020, 0xff004020, 0xff006020, 0xff008020, 0xff00a020, 0xff00c020, 0xff00e020, 0xff080020, 0xff082020, 0xff084020, 0xff086020, 0xff088020, 0xff08a020, 0xff08c020, 0xff08e020, 0xff000028, 0xff002028, 0xff004028, 0xff006028, 0xff008028, 0xff00a028, 0xff00c028, 0xff00e028, 0xff080028, 0xff082028, 0xff084028, 0xff086028, 0xff088028, 0xff08a028, 0xff08c028, 0xff08e028, 0xff000030, 0xff002030, 0xff004030, 0xff006030, 0xff008030, 0xff00a030, 0xff00c030, 0xff00e030, 0xff080030, 0xff082030, 0xff084030, 0xff086030, 0xff088030, 0xff08a030, 0xff08c030, 0xff08e030, 0xff000038, 0xff002038, 0xff004038, 0xff006038, 0xff008038, 0xff00a038, 0xff00c038, 0xff00e038, 0xff080038, 0xff082038, 0xff084038, 0xff086038, 0xff088038, 0xff08a038, 0xff08c038, 0xff08e038, 0xff000040, 0xff002040, 0xff004040, 0xff006040, 0xff008040, 0xff00a040, 0xff00c040, 0xff00e040, 0xff080040, 0xff082040, 0xff084040, 0xff086040, 0xff088040, 0xff08a040, 0xff08c040, 0xff08e040, 0xff000048, 0xff002048, 0xff004048, 0xff006048, 0xff008048, 0xff00a048, 0xff00c048, 0xff00e048, 0xff080048, 0xff082048, 0xff084048, 0xff086048, 0xff088048, 0xff08a048, 0xff08c048, 0xff08e048, 0xff000050, 0xff002050, 0xff004050, 0xff006050, 0xff008050, 0xff00a050, 0xff00c050, 0xff00e050, 0xff080050, 0xff082050, 0xff084050, 0xff086050, 0xff088050, 0xff08a050, 0xff08c050, 0xff08e050, 0xff000058, 0xff002058, 0xff004058, 0xff006058, 0xff008058, 0xff00a058, 0xff00c058, 0xff00e058, 0xff080058, 0xff082058, 0xff084058, 0xff086058, 0xff088058, 0xff08a058, 0xff08c058, 0xff08e058, 0xff000060, 0xff002060, 0xff004060, 0xff006060, 0xff008060, 0xff00a060, 0xff00c060, 0xff00e060, 0xff080060, 0xff082060, 0xff084060, 0xff086060, 0xff088060, 0xff08a060, 0xff08c060, 0xff08e060, 0xff000068, 0xff002068, 0xff004068, 0xff006068, 0xff008068, 0xff00a068, 0xff00c068, 0xff00e068, 0xff080068, 0xff082068, 0xff084068, 0xff086068, 0xff088068, 0xff08a068, 0xff08c068, 0xff08e068, 0xff000070, 0xff002070, 0xff004070, 0xff006070, 0xff008070, 0xff00a070, 0xff00c070, 0xff00e070, 0xff080070, 0xff082070, 0xff084070, 0xff086070, 0xff088070, 0xff08a070, 0xff08c070, 0xff08e070, 0xff000078, 0xff002078, 0xff004078, 0xff006078, 0xff008078, 0xff00a078, 0xff00c078, 0xff00e078, 0xff080078, 0xff082078, 0xff084078, 0xff086078, 0xff088078, 0xff08a078, 0xff08c078, 0xff08e078, };
	
	const uint32_t *lut;
	uint32_t i, amtcr;
	
	switch (depth) {
		case 1:		//1bbp stage 1
			//two stages, first a 2x expansion with field reversal, and then a 4x expansion
			//stage is picked based on value of "DMA2D->OMAR" 
			
			if (DMA2D->OMAR == ((uintptr_t)mFb) + DISP_WIDTH * DISP_HEIGHT) {	//need stage 1
			
				DMA2D->FGMAR = (uintptr_t)mFb;
				DMA2D->OMAR = (uintptr_t)mFb + DISP_WIDTH * DISP_HEIGHT / 8;
			
				DMA2D->FGPFCCR = 0xff05;	//256-entry argb8888 clut. input data is L8 format
				DMA2D->OPFCCR = 0x02;		//output RGB565
				lut = mOneBppFirstStageLut;
				DMA2D->NLR = ((DISP_WIDTH / 8) << 16) + DISP_HEIGHT;
				break;
			}
			
			//need stage 2
			DMA2D->OMAR = ((uintptr_t)mFb) + DISP_WIDTH * DISP_HEIGHT;
			DMA2D->FGMAR = (uintptr_t)mFb + DISP_WIDTH * DISP_HEIGHT / 8;
			//now same as 2bpp
			goto _2bpp_common;
			
		case 2:		//L8 -> argb8888
			DMA2D->FGMAR = (uintptr_t)mFb;
			DMA2D->OMAR = ((uintptr_t)mFb) + DISP_WIDTH * DISP_HEIGHT;
	_2bpp_common:
			DMA2D->FGPFCCR = 0xff05;	//16-entry argb8888 clut. input data is L8 format
			DMA2D->OPFCCR = 0x00;		//output ARGB8888
			lut = mTwoBppLut;
			DMA2D->NLR = ((DISP_WIDTH / 4) << 16) + DISP_HEIGHT;
			break;
		case 4:		//L8 -> rgb565
			DMA2D->FGMAR = (uintptr_t)mFb;
			DMA2D->OMAR = ((uintptr_t)mFb) + DISP_WIDTH * DISP_HEIGHT;
			DMA2D->FGPFCCR = 0xff05;	//256-entry argb8888 clut. input data is L8 format
			DMA2D->OPFCCR = 0x02;		//output RGB565
			lut = mFourBppLut;
			DMA2D->NLR = ((DISP_WIDTH / 2) << 16) + DISP_HEIGHT;
			break;
		default:
			return;
	}
	
	//load CLUT using DMA (tested faster than SW)
	DMA2D->FGCMAR = (uintptr_t)lut;
	amtcr = DMA2D->AMTCR;
	DMA2D->AMTCR = 0;				//disable DMA2D's AHB access dead time module for speed - let DMA2D use AHB all it wants for fast CLUT loading
	DMA2D->FGPFCCR |= 0x20;			//start loading
	while (DMA2D->FGPFCCR & 0x20);	//wait for it
	DMA2D->AMTCR = amtcr;			//restore dead time config
}

void __attribute__((used)) TIM5_IRQHandler(void)	//high bits come first
{
	TIM5->SR = 0;
	if (USE_DMA2D) {
		
		if (mCurDepth == 1)		///every other frame we need to reconfigure in the 1bpp case. halves the framerate, but no 1bpp ap needs a lot of ramerate anyways
			dispConfigResampler(1);
		
		DMA2D->CR |= 1;			//start
		return;
	}
	else {
		uint32_t t, pixels = DISP_WIDTH * DISP_HEIGHT;
		uint8_t *src = mFb, *dst = src + pixels;
		uint16_t *dst16 = (uint16_t*)dst;
		uint32_t *dst32 = (uint32_t*)dst;

		switch (mCurDepth) {
			case 4:
				pixels /= 2;
				do {
					t = *src++;
					*dst++ = (t >> 4) & 0x0F;
					*dst++ = (t >> 0) & 0x0F;
				} while (--pixels);
				break;
			case 2:
				pixels /= 4;
				do {
					static const uint16_t lut2b[] = {0x0000, 0x0100, 0x0200, 0x0300, 0x0001, 0x0101, 0x0201, 0x0301, 0x0002, 0x0102, 0x0202, 0x0302, 0x0003, 0x0103, 0x0203, 0x0303, };
					
					t = *src++;
					*dst16++ = lut2b[(t >> 4) & 0x0F];
					*dst16++ = lut2b[(t >> 0) & 0x0F];
				} while (--pixels);
				break;
			case 1:
				pixels /= 8;
				do {
					static const uint32_t lut1b[] = {0x00000000, 0x01000000, 0x00010000, 0x01010000, 0x00000100, 0x01000100, 0x00010100, 0x01010100, 0x00000001, 0x01000001, 0x00010001, 0x01010001, 0x00000101, 0x01000101, 0x00010101, 0x01010101};
					
					t = *src++;
					*dst32++ = lut1b[(t >> 4) & 0x0F];
					*dst32++ = lut1b[(t >> 0) & 0x0F];
				} while (--pixels);
				break;
		}
	}
}

void dispSetDepth(uint32_t depth)
{
	uint32_t i;
	
	TIM5->CR1 = 0;
	NVIC_DisableIRQ(TIM5_IRQn);
	
	mCurDepth = depth;
	
	if (depth == 16 || depth == 8) {
		if (!dispSetupLayer(false, mFb, DISP_WIDTH, DISP_HEIGHT, DISP_WIDTH * depth / 8, depth == 16))
			fatal("Cannot set disp depth\n");
		return;
	}
	
	if (depth != 1 && depth != 2 && depth != 4) {
		fatal("depth %u is invalid\n", depth);
		return;
	}
	
	//set up display to be in second half of real FB
	dispSetupLayer(false, ((uint8_t*)mFb) + DISP_WIDTH * DISP_HEIGHT, DISP_WIDTH, DISP_HEIGHT, DISP_WIDTH, false);
	
	//start timer to update LCD constantly
	TIM5->PSC = 0;
	TIM5->DIER = TIM_DIER_UIE;
	TIM5->SR = -1;
	TIM5->CNT = TIMER_TICKS_PER_MSEC * 1000 / 60;
	TIM5->ARR = TIMER_TICKS_PER_MSEC * 1000 / 60;
	TIM5->CR1 = TIM_CR1_DIR | TIM_CR1_URS | TIM_CR1_CEN;
	NVIC_EnableIRQ(TIM5_IRQn);
	
	if (USE_DMA2D) {
		
		DMA2D->CR |= 4;			//abort
		while (DMA2D->CR & 1);	//wait
		
		DMA2D->CR = 0x10000 | DMA2D_CR_TCIE;	//mem-to-mem with pfc, int on complete
		DMA2D->FGOR = 0;	//input stride == width
		DMA2D->OOR = 0;		//output stride == width
		DMA2D->AMTCR = 0x1301;	//19 AHB cycles between each access by DMA2D (limits DMA2D's AHB use to 5%)
		
		dispConfigResampler(depth);
	}
}

void dispManualUpdate(void)
{
	//nothing
}

void dispRequestUpdate(void)
{
	//nothing
}


void dispSleep(void)
{
	TIM5->CR1 &=~ TIM_CR1_CEN;
	LTDC_Layer1->CR &=~ 1;
	LTDC->SRCR = 2;
	
	//we can do more to shut more things down, but why bother since this is a dev board
}

void dispWake(void)
{
	LTDC_Layer1->CR |= 1;
	LTDC->SRCR = 2;
	
	if (mCurDepth < 8)
		TIM5->CR1 |= TIM_CR1_CEN;
	
	//we can do more to shut more things down, but why bother since this is a dev board
}
