#define WEAK __attribute__ ((weak))
#define remoteioComms(f) __attribute__ ((weak, alias (#f)))

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "machSpecific.h"
#include "stm32f469xx.h"
#include "remoteioComms.h"
#include "memmap.h"
#include "printf.h"
#include "timers.h"
#include "entry.h"
#include "boot.h"
#include "heap.h"
#include "irqs.h"
#include "mpu.h"


void __attribute__((used)) IntDefaultHandler(void)
{
	cpuIrqDefaultHandler();
	asm volatile("DSB 0x0f");		//c-m4f erratum
}

#define VEC_(nm, pfx)	void nm##pfx(void) __attribute__ ((weak, alias ("IntDefaultHandler"))) 
#define VEC(nm)		VEC_(nm, Handler)
#define VECI(nm)	VEC_(nm, IRQHandler)


VEC(NMI_);
VEC(HardFault_);
VEC(MemManage_);
VEC(BusFault_);
VEC(UsageFault_);
VEC(SVC_);
VEC(PendSV_);
VEC(SysTick_);

VECI(WWDG_);
VECI(PVD_);
VECI(TAMP_STAMP_);
VECI(RTC_WKUP_);
VECI(FLASH_);
VECI(RCC_);
VECI(EXTI0_);
VECI(EXTI1_);
VECI(EXTI2_);
VECI(EXTI3_);
VECI(EXTI4_);
VECI(DMA1_Stream0_);
VECI(DMA1_Stream1_);
VECI(DMA1_Stream2_);
VECI(DMA1_Stream3_);
VECI(DMA1_Stream4_);
VECI(DMA1_Stream5_);
VECI(DMA1_Stream6_);
VECI(ADC_);
VECI(CAN1_TX_);
VECI(CAN1_RX0_);
VECI(CAN1_RX1_);
VECI(CAN1_SCE_);
VECI(EXTI9_5_);
VECI(TIM1_BRK_TIM9_);
VECI(TIM1_UP_TIM10_);
VECI(TIM1_TRG_COM_TIM11_);
VECI(TIM1_CC_);
VECI(TIM2_);
VECI(TIM3_);
VECI(TIM4_);
VECI(I2C1_EV_);
VECI(I2C1_ER_);
VECI(I2C2_EV_);
VECI(I2C2_ER_);
VECI(SPI1_);
VECI(SPI2_);
VECI(USART1_);
VECI(USART2_);
VECI(USART3_);
VECI(EXTI15_10_);
VECI(RTC_Alarm_);
VECI(OTG_FS_WKUP_);
VECI(TIM8_BRK_TIM12_);
VECI(TIM8_UP_TIM13_);
VECI(TIM8_TRG_COM_TIM14_);
VECI(TIM8_CC_);
VECI(DMA1_Stream7_);
VECI(FMC_);
VECI(SDIO_);
VECI(TIM5_);
VECI(SPI3_);
VECI(UART4_);
VECI(UART5_);
VECI(TIM6_DAC_);
VECI(TIM7_);
VECI(DMA2_Stream0_);
VECI(DMA2_Stream1_);
VECI(DMA2_Stream2_);
VECI(DMA2_Stream3_);
VECI(DMA2_Stream4_);
VECI(ETH_);
VECI(ETH_WKUP_);
VECI(CAN2_TX_);
VECI(CAN2_RX0_);
VECI(CAN2_RX1_);
VECI(CAN2_SCE_);
VECI(OTG_FS_);
VECI(DMA2_Stream5_);
VECI(DMA2_Stream6_);
VECI(DMA2_Stream7_);
VECI(USART6_);
VECI(I2C3_EV_);
VECI(I2C3_ER_);
VECI(OTG_HS_EP1_OUT_);
VECI(OTG_HS_EP1_IN_);
VECI(OTG_HS_WKUP_);
VECI(OTG_HS_);
VECI(DCMI_);
VECI(HASH_RNG_);
VECI(FPU_);
VECI(UART7_);
VECI(UART8_);
VECI(SPI4_);
VECI(SPI5_);
VECI(SPI6_);
VECI(SAI1_);
VECI(LTDC_);
VECI(LTDC_ER_);
VECI(DMA2D_);
VECI(CRYP_);
VECI(QSPI_);
VECI(DSI_);


//aligned by linker script as needed
__attribute__ ((section(".ramvecs"))) void (*__ISR_VECTORS[]) (void) =
{
	0,		// unused: initial sp
	0,		// unused: reset handler
	NMI_Handler,
	HardFault_Handler,
	MemManage_Handler,
	BusFault_Handler,
	UsageFault_Handler,
	0,
	0,
	0,
	0,
	SVC_Handler,		// SVCall handler
	0,					// Reserved
	0,					// Reserved
	PendSV_Handler,		// The PendSV handler
	SysTick_Handler,	// The SysTick handler
	
	// Chip Level - STM32F469
	WWDG_IRQHandler,
	PVD_IRQHandler,
	TAMP_STAMP_IRQHandler,
	RTC_WKUP_IRQHandler,
	FLASH_IRQHandler,
	RCC_IRQHandler,
	EXTI0_IRQHandler,
	EXTI1_IRQHandler,
	EXTI2_IRQHandler,
	EXTI3_IRQHandler,
	EXTI4_IRQHandler,
	DMA1_Stream0_IRQHandler,
	DMA1_Stream1_IRQHandler,
	DMA1_Stream2_IRQHandler,
	DMA1_Stream3_IRQHandler,
	DMA1_Stream4_IRQHandler,
	DMA1_Stream5_IRQHandler,
	DMA1_Stream6_IRQHandler,
	ADC_IRQHandler,
	CAN1_TX_IRQHandler,
	CAN1_RX0_IRQHandler,
	CAN1_RX1_IRQHandler,
	CAN1_SCE_IRQHandler,
	EXTI9_5_IRQHandler,
	TIM1_BRK_TIM9_IRQHandler,
	TIM1_UP_TIM10_IRQHandler,
	TIM1_TRG_COM_TIM11_IRQHandler,
	TIM1_CC_IRQHandler,
	TIM2_IRQHandler,
	TIM3_IRQHandler,
	TIM4_IRQHandler,
	I2C1_EV_IRQHandler,
	I2C1_ER_IRQHandler,
	I2C2_EV_IRQHandler,
	I2C2_ER_IRQHandler,
	SPI1_IRQHandler,
	SPI2_IRQHandler,
	USART1_IRQHandler,
	USART2_IRQHandler,
	USART3_IRQHandler,
	EXTI15_10_IRQHandler,
	RTC_Alarm_IRQHandler,
	OTG_FS_WKUP_IRQHandler,
	TIM8_BRK_TIM12_IRQHandler,
	TIM8_UP_TIM13_IRQHandler,
	TIM8_TRG_COM_TIM14_IRQHandler,
	TIM8_CC_IRQHandler,
	DMA1_Stream7_IRQHandler,
	FMC_IRQHandler,
	SDIO_IRQHandler,
	TIM5_IRQHandler,
	SPI3_IRQHandler,
	UART4_IRQHandler,
	UART5_IRQHandler,
	TIM6_DAC_IRQHandler,
	TIM7_IRQHandler,
	DMA2_Stream0_IRQHandler,
	DMA2_Stream1_IRQHandler,
	DMA2_Stream2_IRQHandler,
	DMA2_Stream3_IRQHandler,
	DMA2_Stream4_IRQHandler,
	ETH_IRQHandler,
	ETH_WKUP_IRQHandler,
	CAN2_TX_IRQHandler,
	CAN2_RX0_IRQHandler,
	CAN2_RX1_IRQHandler,
	CAN2_SCE_IRQHandler,
	OTG_FS_IRQHandler,
	DMA2_Stream5_IRQHandler,
	DMA2_Stream6_IRQHandler,
	DMA2_Stream7_IRQHandler,
	USART6_IRQHandler,
	I2C3_EV_IRQHandler,
	I2C3_ER_IRQHandler,
	OTG_HS_EP1_OUT_IRQHandler,
	OTG_HS_EP1_IN_IRQHandler,
	OTG_HS_WKUP_IRQHandler,
	OTG_HS_IRQHandler,
	DCMI_IRQHandler,
	CRYP_IRQHandler,
	HASH_RNG_IRQHandler,
	FPU_IRQHandler,
	UART7_IRQHandler,
	UART8_IRQHandler,
	SPI4_IRQHandler,
	SPI5_IRQHandler,
	SPI6_IRQHandler,
	SAI1_IRQHandler,
	LTDC_IRQHandler,
	LTDC_ER_IRQHandler,
	DMA2D_IRQHandler,
	QSPI_IRQHandler,
	DSI_IRQHandler,
};

static void clockTreeInit(void)
{
	uint32_t basePllSetting;
	
	
	#ifdef OVERCLOCK
		basePllSetting = 344;
		//HSE / 7 as PLL source (8/7MHz), VCO outputs 393.143MHz, P output (main clock) is 196.57MHz, Q output (SDIO is 49.143MHz)
		//RTC = HSE/8 = 1MHz, APB2 = sysclk/2 = 98.2857MHz, APB1 = sysclk/4 = 49.1429MZ, AHB = sysclk = 196.57MHz
	#else
		basePllSetting = 316;
		//HSE / 7 as PLL source (8/7MHz), VCO outputs 361.143MHz, P output (main clock) is 180.57MHz, Q output (SDIO is 45.143MHz)
		//RTC = HSE/8 = 1MHz, APB2 = sysclk/2 = 90.2857MHz, APB1 = sysclk/4 = 45.1429MZ, AHB = sysclk = 180.57MHz
	#endif
	
	//turn on PWR
	RCC->APB1ENR |= RCC_APB1ENR_PWREN;
	
	//setup flash wait states and optimizations, reset caches
	FLASH->ACR = FLASH_ACR_LATENCY_6WS;
	FLASH->ACR |= FLASH_ACR_DCRST | FLASH_ACR_ICRST;
	FLASH->ACR &=~ (FLASH_ACR_DCRST | FLASH_ACR_ICRST);
	FLASH->ACR |= FLASH_ACR_PRFTEN | FLASH_ACR_ICEN | FLASH_ACR_DCEN;

	//first go to safe settings: HSI
	RCC->CR = RCC_CR_HSION;													//HSI on, PLL off
	RCC->CFGR = (RCC->CFGR &~ RCC_CFGR_SW_Msk) | RCC_CFGR_SW_HSI;			//switch to HSI
	
	//set voltage scaling
	if ((PWR->CR & PWR_CR_VOS) != (PWR_CR_VOS_0 | PWR_CR_VOS_1)) {
		PWR->CR |= PWR_CR_VOS_0 | PWR_CR_VOS_1;
		while (!(PWR->CSR & PWR_CSR_VOSRDY));
	}
	
	//turn on overdrive
	PWR->CR |= PWR_CR_ODEN;
	while (!(PWR->CSR & PWR_CSR_ODRDY));
	
	//enable overdrive
	PWR->CR |= PWR_CR_ODSWEN;
	while (!(PWR->CSR & PWR_CSR_ODSWRDY));
	
	//turn on HSE and wait for it
	RCC->CR |= RCC_CR_HSEON;
	while (!(RCC->CR & RCC_CR_HSERDY));
	
	//HSE / 7 as source (8/7MHz), P output (main clock) is VCO/2, Q output (SDIO) is VCO/8
	RCC->PLLCFGR = (basePllSetting << RCC_PLLCFGR_PLLN_Pos) | (8 << RCC_PLLCFGR_PLLQ_Pos) | (7 << RCC_PLLCFGR_PLLM_Pos) | RCC_PLLCFGR_PLLSRC_HSE;
	
	//turn on the main PLL and wait for it
	RCC->CR |= RCC_CR_PLLON;
	while (!(RCC->CR & RCC_CR_PLLRDY));
	
	//set up divisors (RTC = HSE/8 = 1MHz, APB2 = sysclk/2 = VCO/4, APB1 = sysclk/4 = VCO/8, AHB = sysclk = VCO/2) and go to it
	RCC->CFGR = (8 << RCC_CFGR_RTCPRE_Pos) | RCC_CFGR_PPRE2_DIV2 | RCC_CFGR_PPRE1_DIV4 | RCC_CFGR_SW_PLL;
	while (((RCC->CFGR & RCC_CFGR_SW_Msk) >> RCC_CFGR_SW_Pos) != ((RCC->CFGR & RCC_CFGR_SWS_Msk) >> RCC_CFGR_SWS_Pos));
}

static void clockUpPeriphs(void)
{
	RCC->AHB1ENR = RCC_AHB1ENR_GPIOAEN | RCC_AHB1ENR_GPIOBEN | RCC_AHB1ENR_GPIOCEN | RCC_AHB1ENR_GPIODEN | RCC_AHB1ENR_GPIOEEN | RCC_AHB1ENR_GPIOFEN | RCC_AHB1ENR_GPIOGEN | RCC_AHB1ENR_GPIOHEN | RCC_AHB1ENR_GPIOIEN | RCC_AHB1ENR_GPIOJEN | RCC_AHB1ENR_GPIOKEN | RCC_AHB1ENR_CCMDATARAMEN;
	RCC->AHB3ENR = RCC_AHB3ENR_FMCEN;
	RCC->APB2ENR = RCC_APB2ENR_SYSCFGEN | RCC_APB2ENR_TIM1EN;
	RCC->APB1ENR = RCC_APB1ENR_TIM2EN | RCC_APB1ENR_TIM5EN | RCC_APB1ENR_PWREN;
}

static void gpiosInit(void)
{

	/*-- GPIOs Configuration -----------------------------------------------------*/
	/*
	 FMC:
			C 0
			D 0 1 4 5 7 8 9 10 11 12 13 14 15
			E 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15
			F 0 1 2 3 4 5 11 12 13 14 15
			G 0 1 2 4 5 8 15 
			H 2 3 8 9 10 11 12 13 14 15
			I 0 1 2 3 4 5 6 7 9 10

			D7 - SCE for sram


	SWD:
			A: 13 14

	SQPI:
			F10 - CK (AF9)
			F6 - D3 (AF9)
			F7 - D2 (AF9)
			F8 - D0 (AF10)
			F9 - D1 (AF10)
			B6 - CS (AF10)

	MISC
			A0 - irq from visor, active low
			A2 - nSD_CARD_INSERTED
			A4 - irq to visor, active low
			B10 - nSD_CARD_POWER
			C1 - microphone input analog
			C2 - microphone amp enable
			C3 - vref enable (goes to vref+

	AUDIO OUT
			B13 & A8, output, low speed, AF1
				(not configured here)
	
	AUDIO IN
			C1 ADC in
			C2 AMP_ON
			C3 VREF ON

	*/
	
	#define ONE_PER_PIN_ALL(_v0,_v1,_v2,_v3,_v4,_v5,_v6,_v7,_v8,_v9,_vA,_vB,_vC,_vD,_vE,_vF)													(	\
						(((_v0) & 1) << (0x0 * 1)) | (((_v1) & 1) << (0x1 * 1)) | (((_v2) & 1) << (0x2 * 1)) | (((_v3) & 1) << (0x3 * 1)) | 		\
						(((_v4) & 1) << (0x4 * 1)) | (((_v5) & 1) << (0x5 * 1)) | (((_v6) & 1) << (0x6 * 1)) | (((_v7) & 1) << (0x7 * 1)) | 		\
						(((_v8) & 1) << (0x8 * 1)) | (((_v9) & 1) << (0x9 * 1)) | (((_vA) & 1) << (0xA * 1)) | (((_vB) & 1) << (0xB * 1)) |			\
						(((_vC) & 1) << (0xC * 1)) | (((_vD) & 1) << (0xD * 1)) | (((_vE) & 1) << (0xE * 1)) | (((_vF) & 1) << (0xF * 1))		)
						
	#define TWO_PER_PIN_ALL(_v0,_v1,_v2,_v3,_v4,_v5,_v6,_v7,_v8,_v9,_vA,_vB,_vC,_vD,_vE,_vF)													(	\
						(((_v0) & 3) << (0x0 * 2)) | (((_v1) & 3) << (0x1 * 2)) | (((_v2) & 3) << (0x2 * 2)) | (((_v3) & 3) << (0x3 * 2)) | 		\
						(((_v4) & 3) << (0x4 * 2)) | (((_v5) & 3) << (0x5 * 2)) | (((_v6) & 3) << (0x6 * 2)) | (((_v7) & 3) << (0x7 * 2)) | 		\
						(((_v8) & 3) << (0x8 * 2)) | (((_v9) & 3) << (0x9 * 2)) | (((_vA) & 3) << (0xA * 2)) | (((_vB) & 3) << (0xB * 2)) |			\
						(((_vC) & 3) << (0xC * 2)) | (((_vD) & 3) << (0xD * 2)) | (((_vE) & 3) << (0xE * 2)) | (((_vF) & 3) << (0xF * 2))		)
						
	#define FOUR_PER_PIN_LO(_v0,_v1,_v2,_v3,_v4,_v5,_v6,_v7,_v8,_v9,_vA,_vB,_vC,_vD,_vE,_vF)													(	\
						(((_v0) & 15) << (0x0 * 4)) | (((_v1) & 15) << (0x1 * 4)) | (((_v2) & 15) << (0x2 * 4)) | (((_v3) & 15) << (0x3 * 4)) | 	\
						(((_v4) & 15) << (0x4 * 4)) | (((_v5) & 15) << (0x5 * 4)) | (((_v6) & 15) << (0x6 * 4)) | (((_v7) & 15) << (0x7 * 4))	)
	
	#define FOUR_PER_PIN_HI(_v0,_v1,_v2,_v3,_v4,_v5,_v6,_v7,_v8,_v9,_vA,_vB,_vC,_vD,_vE,_vF)													(	\
						(((_v8) & 15) << (0x0 * 4)) | (((_v9) & 15) << (0x1 * 4)) | (((_vA) & 15) << (0x2 * 4)) | (((_vB) & 15) << (0x3 * 4)) | 	\
						(((_vC) & 15) << (0x4 * 4)) | (((_vD) & 15) << (0x5 * 4)) | (((_vE) & 15) << (0x6 * 4)) | (((_vF) & 15) << (0x7 * 4))	)
	
	//mode
	#define I		0				//input
	#define O		1				//output
	#define A		2				//AFR
	#define Z		3				//analog in
	
	//pull
	#define X		0				//none
	#define U		1				//up
	#define D		2				//down
	
	//speed
	#define L		0				//low
	#define M		1				//medium
	#define H		2				//high
	#define V		3				//very high

	//convenience
	#define C		12

	//A13,A14 - SWD  (pull: clock is down, io is up), A0 is irq from visor, pulled up, A4 is irq to visor, output, high
	GPIOA->MODER	= TWO_PER_PIN_ALL(I,I,I,I,O,I,I,I,I,I,I,I,I,A,A,I);
	GPIOA->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOA->OSPEEDR	= TWO_PER_PIN_ALL(M,L,L,L,M,L,L,L,L,L,L,L,L,V,V,L);
	GPIOA->PUPDR	= TWO_PER_PIN_ALL(U,D,D,D,X,D,D,D,D,D,D,D,D,U,D,D);
	GPIOA->AFR[0]	= FOUR_PER_PIN_LO(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOA->AFR[1]	= FOUR_PER_PIN_HI(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOA->BSRR		= ONE_PER_PIN_ALL(0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0);

	//B13 - AF1 - audio out
	GPIOB->MODER	= TWO_PER_PIN_ALL(I,I,I,I,I,I,I,I,I,I,I,I,I,I,I,I);
	GPIOB->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOB->OSPEEDR	= TWO_PER_PIN_ALL(L,L,L,L,L,L,L,L,L,L,L,L,L,L,L,L);
	GPIOB->PUPDR	= TWO_PER_PIN_ALL(D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D);
	GPIOB->AFR[0]	= FOUR_PER_PIN_LO(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOB->AFR[1]	= FOUR_PER_PIN_HI(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);

	//FMC: 0; mic digital control: 2 (low), 3(low); mic analog input: 1
	GPIOC->MODER	= TWO_PER_PIN_ALL(A,Z,O,O,I,I,I,I,I,I,I,I,I,I,I,I);
	GPIOC->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOC->OSPEEDR	= TWO_PER_PIN_ALL(V,V,L,L,L,L,L,L,L,L,L,L,L,L,L,L);
	GPIOC->PUPDR	= TWO_PER_PIN_ALL(U,X,X,X,D,D,D,D,D,D,D,D,D,D,D,D);
	GPIOC->AFR[0]	= FOUR_PER_PIN_LO(C,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOC->AFR[1]	= FOUR_PER_PIN_HI(C,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOA->BSRR		= (ONE_PER_PIN_ALL(0,0,1,1,0,0,0,0,0,0,0,0,0,0,0,0)) << 16;
	
	//FMC: 0 1 4 5 7 8 9 10 11 12 13 14 15
	GPIOD->MODER	= TWO_PER_PIN_ALL(A,A,I,I,A,A,I,A,A,A,A,A,A,A,A,A);
	GPIOD->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOD->OSPEEDR	= TWO_PER_PIN_ALL(V,V,L,L,V,V,L,V,V,V,V,V,V,V,V,V);
	GPIOD->PUPDR	= TWO_PER_PIN_ALL(U,U,D,D,U,U,D,U,U,U,U,U,U,U,U,U);
	GPIOD->AFR[0]	= FOUR_PER_PIN_LO(C,C,0,0,C,C,0,C,C,C,C,C,C,C,C,C);
	GPIOD->AFR[1]	= FOUR_PER_PIN_HI(C,C,0,0,C,C,0,C,C,C,C,C,C,C,C,C);

	//FMC: 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15
	GPIOE->MODER	= TWO_PER_PIN_ALL(A,A,A,A,A,A,A,A,A,A,A,A,A,A,A,A);
	GPIOE->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOE->OSPEEDR	= TWO_PER_PIN_ALL(V,V,V,V,V,V,V,V,V,V,V,V,V,V,V,V);
	GPIOE->PUPDR	= TWO_PER_PIN_ALL(U,U,U,U,U,U,U,U,U,U,U,U,U,U,U,U);
	GPIOE->AFR[0]	= FOUR_PER_PIN_LO(C,C,C,C,C,C,C,C,C,C,C,C,C,C,C,C);
	GPIOE->AFR[1]	= FOUR_PER_PIN_HI(C,C,C,C,C,C,C,C,C,C,C,C,C,C,C,C);
	
	//FMC: 0 1 2 3 4 5 11 12 13 14 15
	GPIOF->MODER	= TWO_PER_PIN_ALL(A,A,A,A,A,A,I,I,I,I,I,A,A,A,A,A);
	GPIOF->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOF->OSPEEDR	= TWO_PER_PIN_ALL(V,V,V,V,V,V,L,L,L,L,L,V,V,V,V,V);
	GPIOF->PUPDR	= TWO_PER_PIN_ALL(U,U,U,U,U,U,D,D,D,D,D,U,U,U,U,U);
	GPIOF->AFR[0]	= FOUR_PER_PIN_LO(C,C,C,C,C,C,0,0,0,0,0,C,C,C,C,C);
	GPIOF->AFR[1]	= FOUR_PER_PIN_HI(C,C,C,C,C,C,0,0,0,0,0,C,C,C,C,C);
	
	//FMC: 0 1 2 4 5 8 15
	GPIOG->MODER	= TWO_PER_PIN_ALL(A,A,A,I,A,A,I,I,A,I,I,I,I,I,I,A);
	GPIOG->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOG->OSPEEDR	= TWO_PER_PIN_ALL(V,V,V,L,V,V,L,L,V,L,L,L,L,L,L,V);
	GPIOG->PUPDR	= TWO_PER_PIN_ALL(U,U,U,D,U,U,D,D,U,D,D,D,D,D,D,U);
	GPIOG->AFR[0]	= FOUR_PER_PIN_LO(C,C,C,0,C,C,0,0,C,0,0,0,0,0,0,C);
	GPIOG->AFR[1]	= FOUR_PER_PIN_HI(C,C,C,0,C,C,0,0,C,0,0,0,0,0,0,C);
	
	//FMC: 2 3 8 9 10 11 12 13 14 15
	GPIOH->MODER	= TWO_PER_PIN_ALL(I,I,A,A,I,I,I,I,A,A,A,A,A,A,A,A);
	GPIOH->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOH->OSPEEDR	= TWO_PER_PIN_ALL(L,L,V,V,L,L,L,L,V,V,V,V,V,V,V,V);
	GPIOH->PUPDR	= TWO_PER_PIN_ALL(D,D,U,U,D,D,D,D,U,U,U,U,U,U,U,U);
	GPIOH->AFR[0]	= FOUR_PER_PIN_LO(0,0,C,C,0,0,0,0,C,C,C,C,C,C,C,C);
	GPIOH->AFR[1]	= FOUR_PER_PIN_HI(0,0,C,C,0,0,0,0,C,C,C,C,C,C,C,C);
	
	//FMC: 0 1 2 3 4 5 6 7 9 10
	GPIOI->MODER	= TWO_PER_PIN_ALL(A,A,A,A,A,A,A,A,I,A,A,I,I,I,I,I);
	GPIOI->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOI->OSPEEDR	= TWO_PER_PIN_ALL(V,V,V,V,V,V,V,V,L,V,V,L,L,L,L,L);
	GPIOI->PUPDR	= TWO_PER_PIN_ALL(U,U,U,U,U,U,U,U,D,U,U,D,D,D,D,D);
	GPIOI->AFR[0]	= FOUR_PER_PIN_LO(C,C,C,C,C,C,C,C,0,C,C,0,0,0,0,0);
	GPIOI->AFR[1]	= FOUR_PER_PIN_HI(C,C,C,C,C,C,C,C,0,C,C,0,0,0,0,0);
	
	GPIOJ->MODER	= TWO_PER_PIN_ALL(I,I,I,I,I,I,I,I,I,I,I,I,I,I,I,I);
	GPIOJ->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOJ->OSPEEDR	= TWO_PER_PIN_ALL(L,L,L,L,L,L,L,L,L,L,L,L,L,L,L,L);
	GPIOJ->PUPDR	= TWO_PER_PIN_ALL(D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D);
	GPIOJ->AFR[0]	= FOUR_PER_PIN_LO(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOJ->AFR[1]	= FOUR_PER_PIN_HI(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	
	GPIOK->MODER	= TWO_PER_PIN_ALL(I,I,I,I,I,I,I,I,I,I,I,I,I,I,I,I);
	GPIOK->OTYPER	= ONE_PER_PIN_ALL(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOK->OSPEEDR	= TWO_PER_PIN_ALL(L,L,L,L,L,L,L,L,L,L,L,L,L,L,L,L);
	GPIOK->PUPDR	= TWO_PER_PIN_ALL(D,D,D,D,D,D,D,D,D,D,D,D,D,D,D,D);
	GPIOK->AFR[0]	= FOUR_PER_PIN_LO(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	GPIOK->AFR[1]	= FOUR_PER_PIN_HI(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);
	
	
	#undef TWO_PER_PIN
	#undef FOUR_PER_PIN_LO
	#undef FOUR_PER_PIN_HI
	#undef I
	#undef O
	#undef A
	#undef Z
	#undef X
	#undef U
	#undef D
	#undef L
	#undef M
	#undef H
	#undef V
		
	//setup which GPIOs are used for external ints
	SYSCFG->EXTICR[0] = 0x00000000;
	SYSCFG->EXTICR[1] = 0x00000000;
	SYSCFG->EXTICR[2] = 0x00000000;
	SYSCFG->EXTICR[3] = 0x00000000;
}

static bool sdramSendCmd(uint32_t cmdModeAndTarget, uint32_t autoRefreshNmbr, uint32_t modeRegisterDef, uint32_t timeout)
{
	uint32_t ticks;
	
	FMC_Bank5_6->SDCMR = cmdModeAndTarget | ((autoRefreshNmbr - 1) << FMC_SDCMR_NRFS_Pos) | (modeRegisterDef << FMC_SDCMR_MRD_Pos);
	
	for (ticks = 0; ticks < timeout; ticks++) {
		if (FMC_Bank5_6->SDSR & FMC_SDSR_BUSY)
			continue;
		return true;
	}
	
	return false;
}

static void sramInit(void)
{
	//we need mode 1, DATAST >= 40ns, ADDSET >= 25ns, BUSTURN >= 15ns. HCLK here is AHBCLOCK
	uint32_t hclk = TIMER_TICKS_PER_MSEC;	//in khz

	#define NS_TO_SRAM_CLOCKS_ROUNDED_UP(_x)		((hclk * (_x) + 999999) / 1000000)
	
	//set bank 1 BTR (read timings)
	FMC_Bank1->BTCR[1] = (NS_TO_SRAM_CLOCKS_ROUNDED_UP(20) << FMC_BTR1_DATAST_Pos) | (NS_TO_SRAM_CLOCKS_ROUNDED_UP(40) << FMC_BTR1_ADDSET_Pos) | (NS_TO_SRAM_CLOCKS_ROUNDED_UP(20) << FMC_BTR1_BUSTURN_Pos);
	
	//set bank 1 BCR, DISABLE WRITE FIFO
	FMC_Bank1->BTCR[0] = FMC_BCR1_MWID_0 | FMC_BCR1_WREN | FMC_BCR1_MBKEN;
	
	//map sram at 0xc0000000 instead of 0x60000000 (sdram is then at 0x60000000)
	SYSCFG->MEMRMP = (SYSCFG->MEMRMP &~ SYSCFG_MEMRMP_SWP_FMC_Msk) | SYSCFG_MEMRMP_SWP_FMC_0;
	
	#undef NS_TO_SRAM_CLOCKS_ROUNDED_UP
}

static void sdramInit(void)
{
	uint32_t sdramclk = TIMER_TICKS_PER_MSEC / 2;	//in khz
	uint32_t cfg;
	
	#define FMC_SDRAM_CMD_CLK_ENABLE				1
	#define FMC_SDRAM_CMD_PALL						2
	#define FMC_SDRAM_CMD_AUTOREFRESH_MODE			3
	#define FMC_SDRAM_CMD_LOAD_MODE					4
	
	#define SDRAM_MODEREG_BURST_LENGTH_1			0
	#define SDRAM_MODEREG_BURST_LENGTH_2			1
	#define SDRAM_MODEREG_BURST_LENGTH_4			5
	#define SDRAM_MODEREG_BURST_LENGTH_8			3
	#define SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL		0
	#define SDRAM_MODEREG_CAS_LATENCY_2				0x20
	#define SDRAM_MODEREG_OPERATING_MODE_STANDARD	0
	#define SDRAM_MODEREG_WRITEBURST_MODE_SINGLE	0x000
	#define SDRAM_MODEREG_WRITEBURST_MODE_BURST		0x200
	
	//SDRAM config
	cfg = FMC_SDCR1_SDCLK_1 | FMC_SDCR1_RBURST | FMC_SDCR1_NB | (2 << FMC_SDCR1_CAS_Pos) | FMC_SDCR1_NB | FMC_SDCR1_MWID_1;
	FMC_Bank5_6->SDCR[0] = cfg;
	FMC_Bank5_6->SDCR[1] = cfg;

	#define NS_TO_SDRAM_CLOCKS_ROUNDED_UP(_x)		((sdramclk * (_x) + 999999) / 1000000)

	//SDRAM timing config
	cfg =	((2										/* LoadToActiveDelay: 2 cycles			*/ - 1) << FMC_SDTR1_TMRD_Pos) |
			((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(77)		/* ExitSelfRefreshDelay: 77ns			*/ - 1) << FMC_SDTR1_TXSR_Pos) |
			((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(42)		/* SelfRefreshTime: 42ns				*/ - 1) << FMC_SDTR1_TRAS_Pos) |
			((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(70)		/* RowCycleDelay: 70ns					*/ - 1) << FMC_SDTR1_TRC_Pos) |
			((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(7) + 1	/* WriteRecoveryTime: (7ns + 1 clock)	*/ - 1) << FMC_SDTR1_TWR_Pos) |
			((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(20)		/* RPDelay: 20ns						*/ - 1) << FMC_SDTR1_TRP_Pos) |
			((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(20)		/* RCDDelay: 20ns						*/ - 1) << FMC_SDTR1_TRCD_Pos);
	
	FMC_Bank5_6->SDTR[0] = cfg;
	FMC_Bank5_6->SDTR[1] = cfg;

	//Clock configuration enable command
	if (!sdramSendCmd(FMC_SDRAM_CMD_CLK_ENABLE | FMC_SDCMR_CTB1, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "Cfg Ena");
	
	//100 ms delay or so
	asm volatile(
		"1:					\n\t"
		"	subs %0, #1		\n\t"
		"	bne 1b			\n\t"
		:"=l"(cfg)
		:"0"(7000000)
		:"cc"
	);

	//PALL (precharge all) command
	if (!sdramSendCmd(FMC_SDRAM_CMD_PALL | FMC_SDCMR_CTB1, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "PALL");
	
	//Auto-Refresh command
	if (!sdramSendCmd(FMC_SDRAM_CMD_AUTOREFRESH_MODE | FMC_SDCMR_CTB1, 8, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "AutoRefresh");

	//Program external memory mode register
	if (!sdramSendCmd(FMC_SDRAM_CMD_LOAD_MODE | FMC_SDCMR_CTB1, 1, SDRAM_MODEREG_BURST_LENGTH_1 | SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL | SDRAM_MODEREG_CAS_LATENCY_2 | SDRAM_MODEREG_OPERATING_MODE_STANDARD | SDRAM_MODEREG_WRITEBURST_MODE_SINGLE, 0x1000))
		fatal("SDRAM init failed: %s\n", "External Memory Mode");

	//Set the refresh rate counter => (15.625 us x Freq) - 20, Set the device refresh counter */
	FMC_Bank5_6->SDRTR = (FMC_Bank5_6->SDRTR &~ FMC_SDRTR_COUNT_Msk) | ((NS_TO_SDRAM_CLOCKS_ROUNDED_UP(15625) - 20) << FMC_SDRTR_COUNT_Pos);
	
	#undef NS_TO_SDRAM_CLOCKS_ROUNDED_UP
	
	//Auto-Refresh command
	if (!sdramSendCmd(FMC_SDRAM_CMD_AUTOREFRESH_MODE | FMC_SDCMR_CTB1, 8, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "AutoRefresh");

	//map sdram at 0x60000000 instead of 0xc0000000 (sram is then at 0xc0000000)
	SYSCFG->MEMRMP = (SYSCFG->MEMRMP &~ SYSCFG_MEMRMP_SWP_FMC_Msk) | SYSCFG_MEMRMP_SWP_FMC_0;
	
	//for now, clear it
	//memset((void*)0x60000000, 0, 8*1024*1024);

	if (LOG_INFO || LOG_WARNING || LOG_ERROR || HAL_TRACE_ENABLED) {	//assume no debugger if all this is off
	
		loge("xxx: debug helper\n");
		DBGMCU->CR |= DBGMCU_CR_DBG_SLEEP;	//do not sleep in sleep mode
	}
}

void machIdle(void)
{
	asm volatile("wfi\n\tnop\n\tnop\n\tnop");		//three nops after WFI: an erratum for STM32F469
}

void mpuRegCfg(uint32_t idx, uint32_t addr, uint32_t cfg)
{
	MPU->RBAR = addr | 0x10 | idx;
	MPU->RASR = cfg;
}

//configures mmu for storage ram
void machSetStorageAreaWriteable(bool writeable)
{
	mpuRegCfg(2, 0x60000000, (writeable ? MPU_PERM_U_RW_S_RW : MPU_PERM_U_RO_S_RO) | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (22 << 1) | 0x0000);
}

static void __attribute__((naked)) dropPriv(void)
{
	asm volatile(
		"	mrs  r1, ipsr			\n\t"
		"	ubfx r1, r1, #0, #9		\n\t"
		"	cmp  r1, #3				\n\t"	//hard fault
		"	it   ne					\n\t"
		"	bxne lr					\n\t"
		"	ldr  r0, =0x01000005	\n\t"	//bus fault
		"	bic  lr, #1				\n\t"
		"	push {r0}				\n\t"
		"	push {lr}				\n\t"
		"	sub  sp, sp, #4*6		\n\t"
		"	ldr  r0, =0xfffffff1	\n\r"
		"	bx   r0					\n\t"
		".ltorg						\n\t"
	);
}

void __attribute__((used)) machInit(uint32_t stage, const void* data)
{
	if (stage == STAGE_INIT_EARLY) {
		
		//compressed roms not supported on reSpring
		clockTreeInit();
		clockUpPeriphs();
		gpiosInit();
		sramInit();
		remoteioCommsEarlyInit();
		sdramInit();
	}
	else if (stage == STAGE_INIT_SET_VTOR) {
		
		SCB->VTOR = (uint32_t)__ISR_VECTORS;
	}
	else if (stage == STAGE_SETUP_HEAPS) {
		
		//register CCM first so it is checked first
		kheapRegisterHeap(HAL_CCM_MEM_BASE, HAL_CCM_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_FAST);
		kheapRegisterHeap(HAL_STATIC_MEM_BASE, HAL_STATIC_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_USABLE_FOR_DMA | MEM_USABLE_FOR_EXEC | MEM_FAST);
	}
	else if (stage == STAGE_INIT_MPU) {
		// * CCM 0x10000000 + 0x00010000. cannot be DMAd or executed from
		// * internal ram (0x20000000) is jit TC (160K), .data&.bss (32K) and heap (128K) (320K total)
		// * rom 0x08000000
		// * ystsem memory at 0x1fff0000
		//configured together to save regions, SRD *IS* correct
		mpuRegCfg(0, 0x00000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (29 << 1) | 0xe100);

		//a region for NULL faulting (0x08000000 in size)
		mpuRegCfg(1, 0x00000000, MPU_PERM_U_XX_S_XX | MPU_PERM_NX | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (26 << 1) | 0x0000);
		
		//8MB region for storage ram (dyn ram overlays it so we do not need SRD here)
		machSetStorageAreaWriteable(false);
		
		//2MB region for dynamic ram & vram
		mpuRegCfg(3, 0x60000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (20 << 1) | 0x0000);
		
		// * FMC & QSPI regs (0xA0000000 + 8K, device, nx)
		// * shared SRAM (0xc0000000 + 4K, device, nx)
		mpuRegCfg(4, 0x80000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (30 << 1) | 0xeb00);
		
		//Periphs (0x40000000 + 0x10000000, device, nx)
		mpuRegCfg(5, 0x40000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (27 << 1) | 0x0000);
		
		//region 6 is used for display tracking (the high region number is needed to overrule all other rules)
		mpuRegCfg(6, 0, 0);
		
		//SCS/SCB/etc in 0xE0000000..0xE0100000 is always acessed using the default map
		
		//mpu on
		MPU->CTRL = MPU_CTRL_ENABLE_Msk;
	}
	else if (stage == STAGE_INIT_INTERRUPTS) {
		
		const struct MachInitDataInterrupts *info = (const struct MachInitDataInterrupts*)data;
		uint32_t mediumPrio = (info->lowestAllowablePrio + info->highestAllowablePrio) / 2;
		
		//scheduler timer interrupt is high prio too so nobody else can interrupt it (and more importantly - it will not interrupt syscalls and vise-versa)
		NVIC_SetPriority(TIM2_IRQn, info->schedulingTimerPrio);
		
		//audio is realtime and thus highest prio
		NVIC_SetPriority(DMA2_Stream5_IRQn, info->lowestAllowablePrio);
		
		//lcd copying int is high prio as well. safe since it doesnt touch any structs
		//visor comms is same cause it is important and they shouldnt interrupt each toher
		NVIC_SetPriority(TIM5_IRQn, info->lowestAllowablePrio + 1);
		NVIC_SetPriority(EXTI0_IRQn, info->lowestAllowablePrio + 1);
		
		//set all HW ints to medium prio
		NVIC_SetPriority(EXTI15_10_IRQn, mediumPrio);
		NVIC_SetPriority(RTC_WKUP_IRQn, mediumPrio);
		NVIC_SetPriority(RTC_Alarm_IRQn, mediumPrio);
		
		remoteioCommsLateInit();
	}
	else if (stage == STAGE_CRASH_LOCKS_BREAK) {
		
		//tell it to not send input to PalmOS (who knows what state it's in)
		remoteioCommsBreakLocks();
		
		//scheduling timer off and lcd refresh timer off
		NVIC_DisableIRQ(TIM2_IRQn);
		NVIC_DisableIRQ(TIM5_IRQn);
		NVIC_ClearPendingIRQ(TIM2_IRQn);
		NVIC_ClearPendingIRQ(TIM5_IRQn);
		
		//set comms irq prio high enough
		dropPriv();	//if hard fault, drop to bus-fault prio to allow interrupts
		
		NVIC_SetPriority(EXTI0_IRQn, 0);
		
		//trigger a refresh (yes, i know this is dirty)
		TIM5_IRQHandler();
	}
}

static void machBusyWaitDelay(uint64_t ticks)
{
	uint64_t start = timerGetTime();
	
	while (timerGetTime() - start < ticks);
}

void machBusyWaitDelayMsec(uint32_t msec)
{
	machBusyWaitDelay((uint64_t)msec * TIMER_TICKS_PER_MSEC);
}

void machBusyWaitDelayUsec(uint32_t usec)
{
	machBusyWaitDelay((uint64_t)usec * (TIMER_TICKS_PER_MSEC / 1000));
}

bool hwMaybeGetRomToken(uint32_t name, const void **dataP, uint16_t *szP)
{
	if (name == CREATE_4CC('s','n','u','m')) {
		
		static char snumStr[12];
		
		if (!snumStr[0]) {
			
			static const char *base32 = "23456789ABCDEFGHJKLMNPQRSTUVWXYZ";
			volatile uint32_t *uid = (volatile uint32_t*)UID_BASE;
			uint32_t snumLo, snumHi, i;
			uint64_t snum;
			
			//get the values
			snumLo = uid[0] ^ uid[1];
			snumHi = uid[1] ^ uid[2];
			
			snum = (((uint64_t)snumHi) << 32) + snumLo;
			
			for (i = 0; i < sizeof(snumStr); i++, snum >>= 5)
				snumStr[i] = base32[snum & 31];
		}
		if (dataP)
			*dataP = snumStr;
		if(szP)
			*szP = sizeof(snumStr);
		
		return true;
	}
	
	return false;
}

void hwGetMiscFlags(uint16_t *miscFlagsP, uint16_t *extMiscFlagsP)
{
	const struct MsgContinueBoot *cbm = remoteioCommsWaitForContinueBooot();

	if (miscFlagsP)
		*miscFlagsP = cbm->hwrMiscFlags;
	
	if (extMiscFlagsP)
		*extMiscFlagsP = cbm->hwrMiscExtFlags;
}

int32_t cpuGetClockRate(enum ClockRateDevice dev)
{
	switch (dev) {
		case CpuClockRate:
		case TimerClockRate:
			return CPU_CLOCK_RATE;
		
		case SdioUnitClockRate:
			return CPU_CLOCK_RATE / 4;
		
		case NandSpiUnitClockRate:
			return CPU_CLOCK_RATE / 2;
		
		default:
			return -1;
	}
}

bool hwPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{
	return false;
}

void machSleep(void)
{
	logi("pretending to sleep\n");
	SysTaskDelay(10000);
	logi("waking up\n");
	dalModifyWakeFlags(DAL_WAKE_FLAG_GENERAL, 0);
	//nothing yet
}

Err machinePaceDispatch(EmulStateRef ref, uint16_t call, Err *ret68kP)
{
	return sysErrNotAllowed;
}


