#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "audiohw.h"
#include "printf.h"
#include "atomic.h"
#include "irqs.h"
#include "heap.h"
#include "ral.h"
#include "cpu.h"
#include "../dma_driver/DmaDriver.h"


#define AUDIO_IN_BUF_SZ					1024

#define ADC_BIAS						0x8000

#define ADC_EXT_VREF_OVER_VCC			35634	//65536 * vref / vcc

static AudioInHwSamplesReadyF mSampleWriteF;
static DmaStream mDmaStream;
static uint16_t *mMicDmaBuf;

static void audioInHwPrvSendData(bool secondHalf)
{
	uint32_t r9state;
	
	r9state = ralSetSafeR9();
	mSampleWriteF(mMicDmaBuf + (secondHalf ? (AUDIO_IN_BUF_SZ / 2) : 0), ADC_BIAS);
	ralRestoreR9(r9state);
}

static void audioInHwPrvDmaIrq(void* userData, uint32_t strmSta)
{
	if (strmSta & DMA_STRM_IRQ_HALF)
		audioInHwPrvSendData(false);
	else if (strmSta & DMA_STRM_IRQ_DONE)
		audioInHwPrvSendData(true);
}

static void adcVrefUseAdjust(int32_t by)	//refcounted
{
	static int32_t refCnt = 0;
	
	irq_state_t sta = irqsAllOff();
	
	if (!refCnt && by < 0) {
		irqsRestoreState(sta);
		fatal("unref adc ref\n");
		return;
	}
	
	refCnt += by;
	
	if (refCnt)
		GPIOC->BSRR = (1 << 3);	//VREF on
	else
		GPIOC->BSRR = (1 << ( 16 + 3));	//VREF off
	
	irqsRestoreState(sta);
}

bool audioInHwInit(AudioInHwSamplesReadyF acceptSamplesF, uint32_t *numSamplesPerBufP)
{
	uint32_t i;
	static const struct DmaStreamUserCfg audioDmaCfg = {
		.magic = CFG_STRUCT_MAGIX,
		.chan = 2,
		.circBuf = 1,
		.prio = 2,
		.perSz = __builtin_ctz(sizeof(*mMicDmaBuf)),
		.memSz = __builtin_ctz(sizeof(*mMicDmaBuf)),
		.memIncr = true,
		.toMem = true,
		.numItems = AUDIO_IN_BUF_SZ,
	};
	
	//todo: configure adc to trigger on timer

	if ((AUDIO_IN_BUF_SZ / 2) & 7)
		fatal("we REQUIRE hardware buffer to be a multiple of 8 samples in size for speed\n");
	
	mDmaStream = DmaLibStreamReserve(2, 0);
	if (!mDmaStream) {
		logw("Audio.in failed to grab our DMA stream\n");
		goto out_err;
	}
	
	logi("r mDmaStream=0x%08x\n", mDmaStream);
	
	if (!DmaLibStreamConfigure(mDmaStream, &audioDmaCfg)) {
		logw("Audio.in failed to configure our stream\n");
		goto out_free_dma;
	}

	if (!DmaLibStreamSetIrqHandler(mDmaStream, audioInHwPrvDmaIrq, NULL)) {
		logw("Audio.in failed to configure irq handler\n");
		goto out_free_dma;
	}

	if (!DmaLibStreamSetPeriphAddr(mDmaStream, (uintptr_t)&ADC3->DR)) {
		logw("Audio.in failed to configure irq DADDR\n");
		goto out_free_dma;
	}

	mMicDmaBuf = kheapAllocEx(AUDIO_IN_BUF_SZ * sizeof(*mMicDmaBuf), MEM_USABLE_FOR_DMA);
	if (!mMicDmaBuf) {
		logw("Audio failed to get mic DMA memory\n");
		goto out_free_dma;
	}
	
	mSampleWriteF = acceptSamplesF;
	if (numSamplesPerBufP)
		*numSamplesPerBufP = AUDIO_IN_BUF_SZ / 2;
	
	return true;

out_free_dma:
	if (!DmaLibStreamRelease(mDmaStream))
		logw("Audio.in failed to release our dma stream\n");

out_err:
	return false;
}

bool audioInHwSetState(bool on, enum AudioSampleRate rate)
{
	uint32_t i;
	
	if (on) {
		
		logi("Audio.in enable at rate %d\n", rate);
		
		GPIOC->BSRR = (1 << 2);	//amp on
		adcVrefUseAdjust(1);
		
		RCC->APB2ENR |= RCC_APB2ENR_ADC3EN;
		RCC->APB1ENR |= RCC_APB1ENR_TIM3EN;
		
		TIM3->CR1 = TIM_CR1_URS; 			//upcount, edger mode, dma req only on overflow, 
		TIM3->CR2 = TIM_CR2_MMS_1;			//update event is trigger out
		TIM3->SMCR = 0;
		TIM3->CCMR1 = 0;
		TIM3->CCMR2 = 0;
		TIM3->PSC = 0;
		TIM3->CNT = 0;
		
		switch (rate) {		//keep in mind APB1 is CPUCLK/4 and timer runs at 2x that
			case AudioRate8000:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 8000 / 2) / 8000 - 1;
				break;

			case AudioRate11025:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 11025 / 2) / 11025 - 1;
				break;
			
			case AudioRate16000:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 16000 / 2) / 16000 - 1;
				break;
			
			case AudioRate22050:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 22050 / 2) / 22050 - 1;
				break;
			
			case AudioRate24000:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 24000 / 2) / 24000 - 1;
				break;
			
			case AudioRate32000:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 32000 / 2) / 32000 - 1;
				break;
			
			case AudioRate44100:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 44100 / 2) / 44100 - 1;
				break;
			
			case AudioRate48000:
				TIM3->ARR = (CPU_CLOCK_RATE / 2 + 48000 / 2) / 48000 - 1;
				break;
			
			default:
				fatal("microphone does not support this rate\n");
				return false;
		}
		
		ADC->CCR = ADC_CCR_ADCPRE_1;	//clock is APB2CLK / 6 = ~ 16.38MHz (lower ADC freq produces better accuracy)
		ADC3->CR1 = 0;
		ADC3->CR2 = ADC_CR2_ALIGN | ADC_CR2_EXTEN_0 | ADC_CR2_DDS | ADC_CR2_DMA | ADC_CR2_EXTSEL_3;			//trigger on trigger rising edge of TIM3_TRO, DMA forever
		ADC3->SMPR1 = (ADC2->SMPR1 &~ ADC_SMPR1_SMP11_Msk) | ADC_SMPR1_SMP11_2 | ADC_SMPR1_SMP11_1;					//sample for 144 cycles
	
		ADC3->SQR1 = 0;		//sample just one channel
		ADC3->SQR3 = 11;	//and that channel is #11
		
		ADC3->CR2 |= ADC_CR2_ADON;
		
		loge("s mDmaStream=0x%08x\n", mDmaStream);
		
		//every turn-on we do this to make sure it starts from start of buffer
		if (!DmaLibStreamSetMemAddr(mDmaStream, 0, (uintptr_t)mMicDmaBuf))
			logw("Audio.in failed to configure irq SADDR\n");
		
		//fill buffer with silence
		for (i = 0; i < AUDIO_IN_BUF_SZ; i++)
			mMicDmaBuf[i] = ADC_BIAS;
		
		if (!DmaLibStreamSetEnabled(mDmaStream, true))
			logw("Audio.in failed to enable dma\n");

		if (!DmaLibStreamSetIrqState(mDmaStream, DMA_STRM_IRQ_HALF | DMA_STRM_IRQ_DONE))
			logw("Audio.in failed to enable dma irqs\n");
		
		//timer on
		TIM3->CR1 |= TIM_CR1_CEN;
	}
	else {
		
		logi("Audio.in disable\n");
		
		TIM3->CR1 &=~ TIM_CR1_CEN;	//timer off
		
		ADC3->CR2 = 0;	//ADC off
		
		loge("e mDmaStream=0x%08x\n", mDmaStream);
		
		if (!DmaLibStreamSetIrqState(mDmaStream, false))
			logw("Audio.in failed to disable dma irqs\n");
		
		if (!DmaLibStreamSetEnabled(mDmaStream, false))
			logw("Audio.in failed to disable dma\n");
		
		//read out last possible result
		(void)ADC3->DR;
		
		GPIOC->BSRR = (1 << (2 + 16));	//amp off
		adcVrefUseAdjust(-1);
		
		RCC->APB2ENR &=~ RCC_APB2ENR_ADC3EN;
		RCC->APB1ENR &=~ RCC_APB1ENR_TIM3EN;
	}
	
	return true;
}



//we do mono input
// Pins we care about:
//	C1 - ADC in from mic, AC coupled, centered on VREF / 2
//	C2 - AMP_ON	active high signal to enable the audio emplifier
//	C3 - VREF_EN - active high signal to enable VREF
// Our data is ADC input #11 on any of the ADCs. We'll use ADC3
// we need to trigger the ADC on time. for that we'll use timer3_trgo
// we'll use DMA to move data out to a circular buffer
//  and we'll get interrupt shalf way into the buffer and all way (twice per buffer)
//  DMA 2 stream 0 (channel 2) will do



static bool adcStartAdc1(void)
{
	uint32_t prevVal;
	
	do {
		prevVal = RCC->APB2ENR;
	} while(!atomicTestAndSet32(&RCC->APB2ENR, prevVal, prevVal | RCC_APB2ENR_ADC1EN));

	return !(prevVal & RCC_APB2ENR_ADC1EN);
}

static void adcStopAdc1(void)
{
	atomicLogicOp32(&RCC->APB2ENR, ~RCC_APB2ENR_ADC1EN, 0);
}

static int32_t adcAcquireAdc1(uint32_t chNum, uint32_t oversampTimes, bool needVref)
{
	uint32_t ret = 0;
	
	if (!adcStartAdc1())
		return DAL_ADC_VAL_INVALID;
	
	adcVrefUseAdjust(1);
	
	if (needVref)
		ADC->CCR |= ADC_CCR_TSVREFE;
	
	ADC1->CR2 = 0;
	ADC1->CR1 = 0;
	
	//slowest possible sampling plz
	if (chNum >= 9)
		ADC1->SMPR1 |= 7 << ((chNum - 10) * 3);
	else
		ADC1->SMPR2 |= 7 << (chNum * 3);
	ADC1->JOFR1 = 0;
	ADC1->JSQR = chNum << ADC_JSQR_JSQ4_Pos;
	ADC1->CR2 = ADC_CR2_ADON;
	KALTaskDelay(10);	//10 ms is a good time for this
	
	while (oversampTimes--) {
	
		ADC1->SR = 0;	//clear satusses
		ADC1->CR2 |= ADC_CR2_JSWSTART;
		while (ADC1->CR2 & ADC_CR2_JSWSTART);	//"cleared by hardware when conversion starts", so this wait should be short
		
		while (!(ADC1->SR & ADC_SR_JEOC))
			KALTaskDelay(10);
	
		ret += ADC1->JDR1 & 0xfff;
	}
	
	if (needVref)
		ADC->CCR &=~ ADC_CCR_TSVREFE;
	
	ADC1->CR2 = 0;
	
	adcVrefUseAdjust(-1);
	
	adcStopAdc1();
	
	return ret;
}

static int32_t adcGetExtVrefMv(void)
{
	int32_t ret;
	
	ret = adcAcquireAdc1(17, 10, true);
	if (ret != DAL_ADC_VAL_INVALID)
		ret = 49561600 / ret;	//internal ref is 1.21v, this calculates voltage of external vref
	
	return ret;
}

static uint32_t scaleToOurVref(uint32_t val, uint32_t ourVrevMv)
{
	return (val * 3300 + ourVrevMv / 2) / ourVrevMv;
}

static int32_t adcGetTemp(void)
{
	const int32_t vrefMv = adcGetExtVrefMv();
	const uint32_t valAt110 = scaleToOurVref(*(uint16_t*)0x1FFF7A2E, vrefMv);
	const uint32_t valAt30 = scaleToOurVref(*(uint16_t*)0x1FFF7A2C, vrefMv);
	const int32_t slope = (valAt110 - valAt30) * 16384 / (110 - 30);	//slope * 16384
	const int32_t yint = valAt30 - 30 * slope / 16384;
	int32_t val = adcAcquireAdc1(18, 16, true);
	
	if (vrefMv != DAL_ADC_VAL_INVALID && val != DAL_ADC_VAL_INVALID)
		val = (val - yint * 16) * 102400 / slope;
	
	return val;
}

int32_t adcGetValue(enum AdcValueIdx which)
{
	int32_t ret;
	
	switch (which) {
		case AdcValVcpu:
			ret = adcGetExtVrefMv();
			if (ret != DAL_ADC_VAL_INVALID) {
				ret = ret * 65536 / ADC_EXT_VREF_OVER_VCC;	//convert to vcc
			}
			break;
			
		case AdcValCpuTemp:
			ret = adcGetTemp();
			break;
		
		default:
			ret = DAL_ADC_VAL_INVALID;
	}
	
	return ret;
}



