#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "audiohw.h"
#include "printf.h"
#include "cpu.h"

#define USE_DTS_TEMP_SENSOR			0		//it only has one point of calibration while the analog one has two, so we prefer that one



#define VREF_MV						1216
#define TS_CALIB_PT_0				3000	//30 degrees C
#define TS_CALIB_PT_1				12000	//120 degrees C
#define ADC_RESULT_BITS				20
#define TS_CALIB_BITS				16
#define TS_CALIB_VREF				3300


bool audioInHwInit(AudioInHwSamplesReadyF acceptSamplesF, uint32_t *numSamplesPerBufP)
{
	loge("Audio input not available on this HW\n");
	return false;
}

bool audioInHwSetState(bool on, enum AudioSampleRate rate)
{
	return false;
}

static uint32_t adcPrvRead(uint_fast8_t inputID, uint32_t commonCrrOrr)		//20-bit result, ADC2 only
{
	uint32_t times = 2, ret = 0;
	
	//clock to ADC on
	RCC->AHB1ENR |= RCC_AHB1ENR_ADC12EN;
	
	//ADCs use AHB clock
	RCC->SRDCCIPR = (RCC->SRDCCIPR &~ RCC_SRDCCIPR_ADCSEL_Msk) | RCC_SRDCCIPR_ADCSEL_1;	//use peripheral clock for ADC (AHB clock, same as cpu clock)
	
	//Fadc_ker_ck x 2 = adc_sclk / 4 = sysclock / 4, thus analog bits see sysclock / 8 = 40MHz which is less than the 50 MHz max
	ADC12_COMMON->CCR = (ADC12_COMMON->CCR &~ ADC_CCR_CKMODE_Msk) | ADC_CCR_CKMODE_0 | ADC_CCR_CKMODE_1;
	
	//enable any custom things we need
	ADC12_COMMON->CCR |= commonCrrOrr;
	
	//To start ADC operations, it is first needed to exit deep-power-down mode by clearing bit DEEPPWD=0
	ADC2->CR &=~ ADC_CR_DEEPPWD;
	
	//Then, it is mandatory to enable the ADC internal voltage regulator by setting the bit ADVREGEN=1 into ADC_CR register.
	ADC2->CR |= ADC_CR_ADVREGEN;
	
	//wait for LDO to come up
	while (!(ADC2->ISR & ADC_ISR_LDORDY));
	ADC2->ISR = ADC_ISR_LDORDY;
	
	//BOOST
	ADC2->CR |= ADC_CR_BOOST_0 | ADC_CR_BOOST_1;
	
	//calibrate the ADC
	ADC2->CR &=~ ADC_CR_ADCALDIF;
	ADC2->CR |= ADC_CR_ADCALLIN;
	ADC2->CR |= ADC_CR_ADCAL;
	while (ADC2->CR & ADC_CR_ADCAL);
	
	//channel is single-ended
	ADC2->DIFSEL &=~ (1UL << inputID);
	
	//set up just this channel to be converted in the normal sequence
	ADC2->SQR1 = inputID << ADC_SQR1_SQ1_Pos;
	
	//config the resolution, queueing, overrun mode, oversampling by 16
	ADC2->CFGR = ADC_CFGR_JQDIS;
	ADC2->CFGR2 = ((16 - 1) << ADC_CFGR2_OVSR_Pos) | ADC_CFGR2_ROVSE;
	
	//config a nice long sampling time
	if (inputID < 10)
		ADC2->SMPR1 = (ADC2->SMPR1 &~ (7 << ((inputID - 0) * 3))) | (6 << ((inputID - 0) * 3));
	else
		ADC2->SMPR2 = (ADC2->SMPR2 &~ (7 << ((inputID - 10) * 3))) | (6 << ((inputID - 10) * 3));
	
	//ADEN=1 enables the ADC. The flag ADRDY will be set once the ADC is ready for operation.
	ADC2->CR |= ADC_CR_ADEN;
	while (!(ADC2->ISR & ADC_ISR_ADRDY));
	ADC2->ISR = ADC_ISR_ADRDY;
	
	while (times--) {
		
		//go
		ADC2->CR |= ADC_CR_ADSTART;
		while (ADC2->CR & ADC_CR_ADSTART);
		
		//get result
		ret = ADC2->DR;
	}
	
	//turn off any custom things we turned on
	ADC12_COMMON->CCR &=~ commonCrrOrr;
	
	//After ADC operations are complete, the ADC can be disabled (ADEN=0)
	ADC2->CR |= ADC_CR_ADDIS;
	
	//It is possible to save power by also disabling the ADC voltage regulator.
	ADC2->CR &=~ ADC_CR_ADVREGEN;
	
	//to save more power by reducing the leakage currents, it is also possible to re-enter in
	//ADC deep-power-down mode by setting bit DEEPPWD=1 into ADC_CR register
	ADC2->CR |= ADC_CR_DEEPPWD;
	
	//clock off to ADCs
	RCC->AHB1ENR&=~ RCC_AHB1ENR_ADC12EN;
	
	
	return ret;
}

static inline int32_t dtsPrvMeasureTemp(void)	//only has one-point claibration, analog sensor has two so we prefer it!
{
	int32_t ret, rawVal, calibVal, calibTemp, periphClkRate = CPU_CLOCK_RATE / 2, slope, measuredHz, oversample = 8/* limits apply*/;
	
	RCC->APB4ENR |= RCC_APB4ENR_DTSEN;		//APB4 is sysclk/2, thus div2 above
	
	DTS->CFGR1 = (DTS->CFGR1 &~ (DTS_CFGR1_REFCLK_SEL | DTS_CFGR1_TS1_SMP_TIME_Msk)) | (127 << DTS_CFGR1_HSREF_CLK_DIV_Pos) | (oversample << DTS_CFGR1_TS1_SMP_TIME_Pos);	//slow but steady (8x oversample)
	DTS->CFGR1 |= DTS_CFGR1_TS1_EN;	//slow but steady
	while (!(DTS->SR & DTS_SR_TS1_RDY));

	DTS->ICIFR = DTS_ICIFR_TS1_CITEF;
	DTS->ITENR = DTS_ITENR_TS1_ITEEN;		//we do not use interrupts, but not setting this will cause DTS->SR & DTS_SR_TS1_ITEF to never go up (even as data does become avail)
	DTS->CFGR1 |= DTS_CFGR1_TS1_START;
	
	calibVal = 100 * ((DTS->T0VALR1 & DTS_T0VALR1_TS1_FMT0_Msk) >> DTS_T0VALR1_TS1_FMT0_Pos);  			//in hz
	calibTemp = 3000 + 10000 * ((DTS->T0VALR1 & DTS_T0VALR1_TS1_T0_Msk) >> DTS_T0VALR1_TS1_T0_Pos);		//in 1/100 degrees C
	slope = (DTS->RAMPVALR & DTS_RAMPVALR_TS1_RAMP_COEFF_Msk) >> DTS_RAMPVALR_TS1_RAMP_COEFF_Pos;		//in Hz / degrees C

	while (!(DTS->SR & DTS_SR_TS1_ITEF));
	rawVal = DTS->DR;
		
	measuredHz = periphClkRate * oversample / rawVal;
	
	ret = calibTemp + 100 * (measuredHz - calibVal + slope / 2) / slope;
	
	DTS->CFGR1 &=~ DTS_CFGR1_TS1_START;
	while (DTS->CFGR1 & DTS_CFGR1_TS1_START);
	DTS->CFGR1 &=~ DTS_CFGR1_TS1_EN;
	while (DTS->CFGR1 & DTS_CFGR1_TS1_EN);
	RCC->APB4ENR &=~ RCC_APB4ENR_DTSEN;
		
	return ret;
}

static uint32_t adcPrvMeasureVref(void)
{
	return adcPrvRead(19, ADC_CCR_VREFEN);
}

static uint32_t adcPrvMeasureTempSens(void)
{
	return adcPrvRead(18, ADC_CCR_TSEN);
}

static int32_t adcPrvMeasureTemp(void)
{
	int32_t vRefMeas = adcPrvMeasureVref();
	int32_t tempMeas = adcPrvMeasureTempSens();
	int32_t calib0v = ((uint32_t)(*(const volatile uint16_t*)0x08FFF814)) << (ADC_RESULT_BITS - TS_CALIB_BITS);
	int32_t calib1v = ((uint32_t)(*(const volatile uint16_t*)0x08FFF818)) << (ADC_RESULT_BITS - TS_CALIB_BITS);
	int32_t calib0t = TS_CALIB_PT_0;
	int32_t calib1t = TS_CALIB_PT_1;
		
	//calibration was done with a vref=vcc=3.3, convert to our vref (=current_vcc)
	calib0v = (uint32_t)(((uint64_t)calib0v * TS_CALIB_VREF * vRefMeas) >> ADC_RESULT_BITS) / VREF_MV;
	calib1v = (uint32_t)(((uint64_t)calib1v * TS_CALIB_VREF * vRefMeas) >> ADC_RESULT_BITS) / VREF_MV;
		
	//calculate temp
	return (calib1t - calib0t) * (tempMeas - calib0v) / (calib1v - calib0v) + calib0t;
}

static uint32_t adcPrvMeasureVoltage(void)
{
	uint32_t raw = adcPrvMeasureVref();
	
	return (VREF_MV << ADC_RESULT_BITS) / raw;
}

int32_t adcGetValue(enum AdcValueIdx which)
{
	switch (which) {
		case AdcValCpuTemp:
			return USE_DTS_TEMP_SENSOR ? dtsPrvMeasureTemp() : adcPrvMeasureTemp();
		
		case AdcValVcpu:
			return adcPrvMeasureVoltage();
		
		default:
			return DAL_ADC_VAL_INVALID;
	}
}
