#include <stddef.h>
#include <PalmOS.h>
#include <VFSMgr.h>
#include <SonyCLIE.h>
#include <SlotDrvrLib.h>
#include <ExpansionMgr.h>
#include "msioCommsProto.h"
#include "PalmOSextra.h"
#include "SonyMSIO.h"
#include "deviceID.h"
#include "globals.h"
#include "serial.h"
#include "asm.h"




#define MS_RD_SDATA					0x03
#define REPALM_FUNC_ID				0x64


#define CustomControl(_libRef, _apiCrid, _apiIdx, _dataPtr, _sizePtr)						\
	(g->fSlotCustomControl ? 																\
		g->fSlotCustomControl(g->customControlData, _apiIdx, _dataPtr):						\
		SlotCustomControl(g->libRef, SONY_MSIO_API_CREATOR, _apiIdx, _dataPtr, _sizePtr)	\
	)


static Err msioPrvIrqControl(struct Globals *g, Boolean irqOn)
{
	struct SonyMsioIntControl ic = {.slotRef = g->slotRef, .slotIdx = 0, .functionID = REPALM_FUNC_ID, .irqOn = irqOn, };
	
	return CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_INT_CONTROL, &ic, NULL);
}

static Err msioPrvStartBoot(struct Globals *g, struct MsioPktBootParams *bp, UInt32 rxBufSz)
{
	static const struct RectangleType allOfScreen = {.topLeft = {0,0}, .extent = {500, 500} };
	struct SonyMsioIoWriteData wd;
	SysBatteryKind bk;
	Boolean plugged;
	

	//round up rxBufSz to account fo rheaders
	rxBufSz += sizeof(struct MsioScreenImgHdr);
	
	if (rxBufSz < MIN_RX_BUF_SZ)
		rxBufSz = MIN_RX_BUF_SZ;

	g->rxBuf = MemChunkNew(0, rxBufSz, 0x1200);
	if (!g->rxBuf) {
		
		char x[16];
		
		StrIToA(x, rxBufSz);
		
		ErrAlertCustom(0, "Cannot allocate screen RX buffer of size ", NULL, x);
		return memErrNotEnoughSpace;
	}
	
	bp->voltageInCv = SysBatteryInfo(false, NULL, NULL, NULL, &bk, &plugged, &bp->percent);
	bp->kind = bk;
	bp->isPluggedIn = plugged;
	bp->curRtc = TimGetSeconds();
	
	WinSetDrawWindow(WinGetDisplayWindow());
	g->dispBuffer = BmpGetBits(WinGetBitmap(WinGetDisplayWindow()));
	WinDrawRectangle(&allOfScreen, 0);
	
	wd.slotRef = g->slotRef;
	wd.slotIdx = 0;
	wd.functionID = REPALM_FUNC_ID;
	wd.dataInP = (const void*)bp;
	wd.dataLen = sizeof(*bp);
	wd.readTimeout = 0;
	
	//pre-populate some things
	g->getIntReqRawTpc.slotRef = g->slotRef;
	g->getIntReqRawTpc.slotIdx = 0;
	g->getIntReqRawTpc.functionID = REPALM_FUNC_ID;
	g->getIntReqRawTpc.tpc = MS_GET_INT;
	g->getIntReqRawTpc.rfu = 0;
	g->getIntReqRawTpc.dataOutP = (UInt8*)&g->lastSta;
	g->getIntReqRawTpc.dataLen = 1;
	g->getIntReqRawTpc.readTimeout = 0;

	g->getIntReqManaged.slotRef = g->slotRef;
	g->getIntReqManaged.slotIdx = 0;
	g->getIntReqManaged.functionID = REPALM_FUNC_ID;
	g->getIntReqManaged.intRegValOutP = (UInt8*)&g->lastSta;
	g->getIntReqManaged.numBytesToRead = 1;
	g->getIntReqManaged.readTimeout = 0;

	g->irqAckCmd.slotRef = g->slotRef;
	g->irqAckCmd.slotIdx = 0;
	g->irqAckCmd.functionID = REPALM_FUNC_ID;
	g->irqAckCmd.cmd = 0x00;
	g->irqAckCmd.tpcTimeout = 0;
	
	g->wdPen.slotRef = g->slotRef;
	g->wdPen.slotIdx = 0;
	g->wdPen.functionID = REPALM_FUNC_ID;
	g->wdPen.dataInP = (UInt8*)&g->pkPen;
	g->wdPen.dataLen = sizeof(g->pkPen);
	g->wdPen.readTimeout = 0;
	g->pkPen.hdr.pktTyp = MSIO_PKT_PEN;

	g->rdMisc.slotRef = g->slotRef;
	g->rdMisc.slotIdx = 0;
	g->rdMisc.functionID = REPALM_FUNC_ID;
	g->rdMisc.dataOutP = (UInt8*)g->rxBuf;
	g->rdMisc.dataLen = MSIO_SHORT_PKT_SZ;
	g->rdMisc.readTimeout = 0;
	
	g->rdMiscBig.slotRef = g->slotRef;
	g->rdMiscBig.slotIdx = 0;
	g->rdMiscBig.functionID = REPALM_FUNC_ID;
	g->rdMiscBig.dataOutP = (UInt8*)g->rxBuf;
	g->rdMiscBig.dataLen = MSIO_MAX_PACKET_LEN;
	g->rdMiscBig.readTimeout = 0;
	
	return CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_WRITE_DATA, &wd, NULL);
}

static Err msioPrvGetSta(struct Globals *g)
{
	if (g->deviceMsioFlags & DEVICE_MSIO_FLAG_RAW_TPCS_BROKEN)
		return CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_GET_INT, &g->getIntReqManaged, NULL);
	else
		return CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_TPC_R, &g->getIntReqRawTpc, NULL);
}

void msioIrqHandler(void *userData)
{
	struct Globals *g = (struct Globals*)userData;
	
	//DSP devices will signal irq forever until irq is gone
	//non-DSP devices dont, but we just do this for all
	CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_SET_CMD, &g->irqAckCmd, NULL);
	EvtWakeup();
}

static Err msioPrvFunctionEnable(struct Globals *g, Boolean allowCurrentCheck)
{
	struct SonyMsioFunctionEnable fe;
	
	fe.slotRef = g->slotRef;
	fe.slotIdx = 0;
	fe.functionID = REPALM_FUNC_ID;
	fe.irqHandler = msioIrqHandler;
	fe.irqHandlerData = g;
	
	return CustomControl(g->libRef, SONY_MSIO_API_CREATOR, allowCurrentCheck ? SONY_MSIO_OP_FUNCTION_ENABLE : SONY_MSIO_OP_FUNCTION_ENABLE_NO_CURRENT_CHECK, &fe, NULL);
}

static Err msioPrvFunctionDisable(struct Globals *g)
{
	struct SonyMsioFunctionDisable fd;
	
	fd.slotRef = g->slotRef;
	fd.slotIdx = 0;
	fd.functionID = REPALM_FUNC_ID;
	
	return CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_FUNCTION_DISABLE, &fd, NULL);
}

static void screenDecompressNewC(UInt32 *dst, const UInt8 *src, UInt32 srcLen)
{
	const UInt8 *srcEnd = src + srcLen, marker = *src++;
	
	while (src < srcEnd) {
		
		UInt8 b = *src++;
		
		if (b != marker) {
			
			((UInt8*)dst)[0] = b;
			((UInt8*)dst)[1] = *src++;
			((UInt8*)dst)[2] = *src++;
			((UInt8*)dst)[3] = *src++;
			dst++;
		}
		else if (!(b = *src++)) {
			
			((UInt8*)dst)[0] = marker;
			((UInt8*)dst)[1] = *src++;
			((UInt8*)dst)[2] = *src++;
			((UInt8*)dst)[3] = *src++;
			dst++;
		}
		else {
			UInt16 ofst, len;
			UInt32 *from;
			
			if (b < 0x80)
				ofst = b;
			else
				ofst = ((b & 0x7f) << 8) + *src++;
			
			b = *src++;
			if (b < 0x80)
				len = b;
			else
				len = ((b & 0x7f) << 8) + *src++;
			
			len++;
			from = dst - ofst;
			while (len--)
				*dst++ = *from++;
		}
	}
}

static void msioPrvHandleVibOrLedMsg(struct Globals *g, UInt16 sysTrap, const struct MsioVibLedControl *vlc)
{
	Err (*hwrF)(Boolean set, UInt32 attr, const void* data) = SysGetTrapAddress(sysTrap);
	
	hwrF(true, LED_VIB_ATTR_INTERRUPT_OFF, NULL);
	if (vlc->numTimes) {
		Boolean on = true;
		
		hwrF(true, LED_VIB_ATTR_REPEAT_COUNT, &vlc->numTimes);
		hwrF(true, LED_VIB_ATTR_RATE, &vlc->csecPerPiece);
		hwrF(true, LED_VIB_ATTR_DELAY, &vlc->csecBetween);
		hwrF(true, LED_VIB_ATTR_PATTERN, &vlc->pattern);
		hwrF(true, LED_VIB_ATTR_ACTIVE, &on);
	}
}

static void msioPrvHandleClutRx(struct Globals *g, const struct MsioClutHdr *ch)	//we trust incoming data - it is faster
{
	struct RGBColorType *clut = (struct RGBColorType*)g->rxBuf;
	UInt16 numEntries = ch->numEntries;
	Err e;
		
	g->rdMiscBig.dataOutP = (UInt8*)g->rxBuf;
	if (numEntries <= MSIO_MAX_PACKET_LEN / sizeof(struct RGBColorType)) {		//one packet
		
		g->rdMiscBig.dataLen = numEntries * sizeof(struct RGBColorType);
		e = CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_READ_DATA, &g->rdMiscBig, NULL);
		if (e != errNone)
			ErrAlertCustom(e, "failed to read short CLUT", NULL, NULL);
	}
	else {
	
		g->rdMiscBig.dataLen = MSIO_MAX_PACKET_LEN;
		e = CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_READ_DATA, &g->rdMiscBig, NULL);
		if (e != errNone)
			ErrAlertCustom(e, "failed to read long CLUT p1", NULL, NULL);
		
		g->rdMiscBig.dataOutP = ((UInt8*)g->rxBuf) + MSIO_MAX_PACKET_LEN;
		g->rdMiscBig.dataLen = numEntries * sizeof(struct RGBColorType) - MSIO_MAX_PACKET_LEN;
		e = CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_READ_DATA, &g->rdMiscBig, NULL);
		if (e != errNone)
			ErrAlertCustom(e, "failed to read long CLUT p2", NULL, NULL);
	}
	
	e = WinPalette(winPaletteSet, 0, numEntries, clut);
	if (e != errNone)
		ErrAlertCustom(e, "failed to set CLUT", NULL, NULL);
}

static void msioPrvHandleScreenRx(struct Globals *g, struct MsioScreenImgHdr *ih)
{
	UInt32 szLeft, totalSz;
	Err e;
	
	szLeft = ((UInt32)ih->totalSzHi) << 16;
	szLeft |= ih->totalSzLo;
	totalSz = szLeft;
	
	if (szLeft > sizeof(ih->data)) {
		szLeft -= sizeof(ih->data);
		
		g->rdMiscBig.dataOutP = (UInt8*)(ih + 1);
		do {
			
			UInt16 now = (szLeft >= MSIO_MAX_PACKET_LEN) ? MSIO_MAX_PACKET_LEN : szLeft;
			
			g->rdMiscBig.dataLen = now;
			e = CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_READ_DATA, &g->rdMiscBig, NULL);
			if (e != errNone)
				ErrAlertCustom(e, "failed to read screen data part", NULL, NULL);
			
			szLeft -= now;
			g->rdMiscBig.dataOutP += now;
		
		} while (szLeft);
	}
	screenDecompressNew(g->dispBuffer, ih->data, totalSz);
}

static void uiPrvAlarmCbk(UInt16 almProcCmd, SysAlarmTriggeredParamType *param)
{
	struct Globals *g = (struct Globals*)param->ref;
	
	if (almProcCmd == almProcCmdTriggered) {
		g->alarmTriggered = true;
		EvtWakeup();
	}
}

void msioSerRxedData(struct Globals *g, const void *data, UInt32 len, UInt8 lineErrors)
{
	if (len <= MSIO_MAX_PACKET_LEN - sizeof(struct MsioSerialData)) {
		union {
			struct MsioSerialData sdp;
			UInt16 forAlignment;
			UInt8 forSize[MSIO_MAX_PACKET_LEN];
		} tx;
		struct SonyMsioIoWriteData tpc = {
			.slotRef = g->slotRef,
			.slotIdx = 0,
			.functionID = REPALM_FUNC_ID,
			.dataInP = (const UInt8*)&tx.sdp,
			.dataLen = sizeof(struct MsioSerialData) + len,
		};
		
		tx.sdp.hdr.pktTyp = MSIO_PKT_SER_DATA;
		tx.sdp.bytesAndFlags = ((len << MSG_BYTES_AND_FLAGS_BYTES_SHIFT) & MSG_BYTES_AND_FLAGS_BYTES_MASK) +
								((lineErrors << MSG_BYTES_AND_FLAGS_FLAGS_SHIFT) & MSG_BYTES_AND_FLAGS_FLAGS_MASK);
		MemMove(tx.sdp.data, data, len);
		
		(void)CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_WRITE_DATA, &tpc, NULL);
	}
}

static void uiPrvGetAndHandleModuleMessages(struct Globals *g)
{
	const struct MsioShortMessage *sm = (const struct MsioShortMessage*)g->rxBuf;
	const struct MsioVibLedControl *vlc = (const struct MsioVibLedControl*)g->rxBuf;
	const struct MsioPktHeader *ph = (const struct MsioPktHeader*)g->rxBuf;
	struct MsioSerialDataTxReply serTxReply;	//XXX: todo send this
	struct MsioPktSerOp serOpReply;				//XXX: todo send this
	UInt32 tmp32;
	Err e;
	
	while (errNone == msioPrvGetSta(g) && (g->lastSta & MSIO_STA_HAVE_DATA)) {
		
		e = CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_READ_DATA, &g->rdMisc, NULL);
		if (e != errNone)
			ErrAlertCustom(e, "failed to read misc data", NULL, NULL);
		else {
			switch (ph->pktTyp) {
		
				case MSIO_PKT_SER_OP_REQUEST:
					msioSerHandleOpReq(g, (const struct MsioPktSerOp*)ph, &serOpReply);
					break;
				
				case MSIO_PKT_SER_DATA:
					msioSerHandleDataReq(g, (const struct MsioSerialData*)ph, &serTxReply);
					break;
		
				case MSIO_PKT_SET_BACKLIGHT:
					(void)HwrBacklight(true, sm->value);
					break;
				
				case MSIO_PKT_SET_BRI:
					(void)SysLCDBrightness(true, sm->value);
					break;
				
				case MSIO_PKT_SET_CONTRAST:
					(void)SysLCDContrast(true, sm->value);
					break;
				
				case MSIO_PKT_SET_DEPTH:
					tmp32 = sm->value;
					WinScreenMode(winScreenModeSet, NULL, NULL, &tmp32, NULL);
					g->dispBuffer = BmpGetBits(WinGetBitmap(WinGetDisplayWindow()));
					break;
				
				case MSIO_PKT_LED_CONTROL:
					msioPrvHandleVibOrLedMsg(g, sysTrapHwrLEDAttributes, vlc);
					break;
				
				case MSIO_PKT_VIB_CONTROL:
					msioPrvHandleVibOrLedMsg(g, sysTrapHwrVibrateAttributes, vlc);
					break;
				
				case MSIO_PKT_CLUT_HDR:
					msioPrvHandleClutRx(g, (const struct MsioClutHdr*)ph);
					break;
				
				case MSIO_PKT_DISPLAY_HDR:
					msioPrvHandleScreenRx(g, (struct MsioScreenImgHdr*)g->rxBuf);
					break;
				
				case MSIO_PKT_SET_ALARM:
					AlmSetProcAlarm(uiPrvAlarmCbk, (UInt32)g, ((struct MsioRtcSetData*)g->rxBuf)->value);
					break;
				
				case MSIO_PKT_SET_RTC:
					TimSetSeconds(((struct MsioRtcSetData*)g->rxBuf)->value);
					break;
			}
		}
	}
}

static void msioPrvSendKeys(struct Globals *g)
{
	UInt32 keys = KeyCurrentState();
	Boolean dn;
	
	if (g->prevKeyState != keys) {
		g->prevKeyState = keys;
		
		struct MsioPktButtons b = {
			.hdr = {
				.pktTyp = MSK_PKT_BUTTONS,
			},
			.palmOsBitMask = keys,
		};
		struct SonyMsioIoWriteData tpc = {
			.slotRef = g->slotRef,
			.slotIdx = 0,
			.functionID = REPALM_FUNC_ID,
			.dataInP = (const UInt8*)&b,
			.dataLen = sizeof(b),
		};
		
		(void)CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_WRITE_DATA, &tpc, NULL);
	}
}

static void msioPrvSendPen(struct Globals *g)
{
	Boolean dn;
	
	EvtGetPen(&g->pkPen.x, &g->pkPen.y, &dn);
	if (!dn)
		g->pkPen.x = g->pkPen.y = -1;
	
	if (g->prevPenX != g->pkPen.x || g->prevPenY != g->pkPen.y) {
		g->prevPenX = g->pkPen.x;
		g->prevPenY = g->pkPen.y;
	
		(void)CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_WRITE_DATA, &g->wdPen, NULL);
	}
}

static void msioPrvReportRtc(struct Globals *g)
{
	UInt32 curRtc = TimGetSeconds();
	struct MsioRtcData dp;
	struct SonyMsioIoWriteData tpc = {
		.slotRef = g->slotRef,
		.slotIdx = 0,
		.functionID = REPALM_FUNC_ID,
		.dataInP = (const UInt8*)&dp,
		.dataLen = sizeof(dp),
	};
	
	if (g->prevRtcReported == curRtc && !g->alarmTriggered)
		return;
	
	dp.hdr.pktTyp = MSIO_PKT_RTC_REPORT;
	dp.alarmTriggered = g->alarmTriggered;
	dp.curTime = curRtc;
	
	g->alarmTriggered = false;
	g->prevRtcReported = curRtc;
	
	(void)CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_IO_WRITE_DATA, &tpc, NULL);
}

static void ui(struct Globals *g, struct MsioPktBootParams *bp, UInt32 rxBufSz)
{
	volatile UInt8 i = 0;
	EventType evt;
	Err e;
	
	WinSetDrawWindow(WinGetDisplayWindow());
	
	e = msioPrvFunctionEnable(g, true);
	if (e == sonyMsioErrCannotSupplyCurrent)
		e = msioPrvFunctionEnable(g, false);
	
	if (e != errNone)
		ErrAlertCustom(e, "cannot enable func", NULL, NULL);
	else {
		
		if (errNone != (e = msioPrvIrqControl(g, true)))
			ErrAlertCustom(e, "cannot enable irq", NULL, NULL);
		else {
			if (errNone != (e = msioPrvStartBoot(g, bp, rxBufSz)))
				ErrAlertCustom(e, "cannot start boot", NULL, NULL);
			else {
				
				while (1) {
					
					uiPrvGetAndHandleModuleMessages(g);
					msioPrvReportRtc(g);
					msioSerPeriodic(g, MSIO_MAX_PACKET_LEN - sizeof(struct MsioSerialData));
					EvtGetEvent(&evt, g->prevKeyState ? 0 : 50);	//there is no keyUp sometimes, so we'll busy wait
					
					switch (evt.eType) {
						case penDownEvent:
						case penMoveEvent:
						case penUpEvent:
							msioPrvSendPen(g);
							break;
						
						case nilEvent:
							if (KeyCurrentState() == g->prevKeyState)
								break;
							//fallthrough
						case keyDownEvent:
						case keyUpEvent:
							msioPrvSendKeys(g);
							break;
						
						default:
							break;
					}
				}
			}
			if (g->hrLibRef != sysErrLibNotFound) {
				
				if (errNone == HRClose(g->hrLibRef))
					SysLibRemove(g->hrLibRef);
			}
			if (g->silkLibRef != sysErrLibNotFound) {
				
				if (errNone == SilkLibClose(g->silkLibRef))
					SysLibRemove(g->silkLibRef);
			}
			
			if (g->rxBuf)
				MemChunkFree(g->rxBuf);
			
			if (errNone != msioPrvIrqControl(g, false))
				ErrAlertCustom(0, "cannot disable irq", NULL, NULL);
		}
		if (errNone != msioPrvFunctionDisable(g))
			ErrAlertCustom(0, "cannot disable func", NULL, NULL);
	}
	
	AlmSetProcAlarm(uiPrvAlarmCbk, (UInt32)g, 0);
}

static Boolean verifySlotFirstPass(struct Globals *g)
{
	struct SonyMsioApiVersion av;
	UInt16 len;
	
	len = sizeof(av);
	if (errNone != CustomControl(g->libRef, SONY_MSIO_API_CREATOR, SONY_MSIO_OP_API_VERSION, &av, &len))
		return false;
	
	if (len != sizeof(av) || av.ver != SONY_MSIO_API_VER_1)
		return false;

	return true;
}

static Boolean findSlot(UInt16 *slotRefP, UInt16 *libRefP)
{
	UInt32 iter = expIteratorStart;
	UInt16 slotRef, slotLibRef;
	
	while (iter != expIteratorStop && errNone == ExpSlotEnumerate(slotRefP, &iter)) {
		
		UInt32 mediaType;
		
		if (errNone != ExpSlotLibFind(*slotRefP, libRefP))
			continue;
		
		if (errNone != SlotMediaType(*libRefP, *slotRefP, &mediaType))
			continue;
		
		if (mediaType != expMediaType_MemoryStick)
			continue;
		
		return true;
	}
	return false;
}

UInt32 __Startup__(void) {
	
	void *prevGlobalsP, *globalsP;
	SysAppInfoPtr appInfoP;
	struct Globals g = {};
	
	SysAppStartup(&appInfoP, &prevGlobalsP, &globalsP);
	
	if (appInfoP->cmd == sysAppLaunchCmdNormalLaunch) {
		
		UInt16 slotRef, libRef, cardNo, dbAttrs;
		LocalID LID;
		
		//sony devices leave us in ram - mark self as recyclable to avoid this
		if (errNone == SysCurAppDatabase(&cardNo, &LID) && errNone == DmDatabaseInfo(cardNo, LID, NULL, &dbAttrs, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL)) {
			
			dbAttrs |= dmHdrAttrRecyclable;
			(void)DmSetDatabaseInfo(cardNo, LID, NULL, &dbAttrs, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
		}
		
		if (!findSlot(&slotRef, &libRef))
			ErrAlertCustom(0, "Cannot find the slot", NULL, NULL);
		else {
			struct MsioPktBootParams bp;
			UInt32 rxBufSz;
			
			g.slotRef = slotRef;
			g.libRef = libRef;
			g.prevPenX = -1;
			g.prevPenY = -1;
			g.prevKeyState = 0;
			
			switch (deviceIdentify(&g, &bp, &rxBufSz)) {
				case DeviceNeedsMissingDriver:
					ErrAlertCustom(0, "This device needs an MSIO driver upgrade", NULL, NULL);
					break;
				
				case DeviceUnknown:
					ErrAlertCustom(0, "This device is currently unsupported", NULL, NULL);
					break;
				
				case DeviceIdentificationError:
					ErrAlertCustom(0, "This device cannot be identified", NULL, NULL);
					break;
				
				case DeviceOK:
					if (!verifySlotFirstPass(&g))
						ErrAlertCustom(0, "Slot does not appear to support MSIO or contain a rePalm card", NULL, NULL);
					else
						ui(&g, &bp, rxBufSz);
					break;
			}
		}
	}
	
	SysAppExit(appInfoP, prevGlobalsP, globalsP);
	return 0;
}

//use OS funcs for size
void* memset(void *s, int c, size_t n)
{
	MemSet(s, n, c);
	
	return s;
}

void *memcpy(void *restrict dest, const void *restrict src, size_t n)
{
	MemMove(dest, src, n);
	
	return dest;
}
