#include <ErrorBase.h>
#include <string.h>
#include <stdio.h>
#include "audiohw.h"
#include "printf.h"
#include "disasm.h"
#include "atomic.h"
#include "audio.h"
#include "heap.h"
#include "boot.h"
#include "mpu.h"
#include "kal.h"
#include "dal.h"

//XXX: consider waking audio thread before enabling audio hw so we have a buffer full of audio to play. maybe. might have cons...


#define BUFFER_DEPTH				3
#define NUM_BUFFERS_TILL_SHUTOFF	(5 + BUFFER_DEPTH)		//how many empty buffers till we shut off hardware

struct AudioStream {
	
	struct AudioStream *next, *prev;
	uintptr_t handle;

	enum AudioSampleRate rate;
	enum AudioSampleType sampTyp;
	enum AudioChannelConfig chans;
	uint16_t nextMixerBuf	: 3;		//must allow 0 .. BUFFER_DEPTH - 1
	uint16_t nextClientBuf	: 3;		//must allow 0 .. BUFFER_DEPTH - 1
	uint16_t loopMode		: 1;
	uint16_t inTeardown		: 1;
	uint16_t haveWaiters	: 2;
	volatile uint8_t nMixerBufs;		//must allow 0 .. BUFFER_DEPTH
	
	uint16_t volumeL;
	uint16_t volumeR;
	uint32_t resampleState[7];		//big enough for any resampling state	
	uint16_t bufUsedClient;			//in samples (L + R = 1) how much of the current buffer has been used byt he client
	uint16_t bufUsedMixer;			//in samples (L + R = 1) how much of the current buffer has been used byt he client
	
	#ifdef SONY_SUPPORT_ENABLED
		uint32_t sonyNumOutSamplesProduced;
		uint32_t sonyLastMixedTicks;
	#endif
	
	uint32_t nSamples;				//number of subsamples (L and R counts as one) per buffer
	uint32_t clientSema;			//semaphore clients can wait on
	uint8_t *buf[BUFFER_DEPTH];		//pointers into the data that follows this stream struct
	
	void *mixFunc;
	
	//samples here
};

// * src must point to first byte of first source subsample 
// * might read beyond buffer by one sample
// * keep in mind the counter here is in "samples" so {L+R} counts as one since this processes input into stereo output always
// * maxOutSamples is the mas number of samples to write. if this is reached we'll stop
// * numInSamples is max number of samples we can read. if this is reached we'll also stop
// * return pointer to next unfilled output sample
// * leave *srcP pointing to next unused input sample
// * DO NOT CALL with maxOutSamples or numInSamples as zero!!!
typedef int32_t* (*MixInF)(int32_t* dst, const uint8_t** srcP, uint32_t maxOutSamples, void* resampleStateP, uint32_t volumeL, uint32_t volumeR, uint32_t numInSamples);


static const uint16_t *mResampTabs[AudioRateNumRates][AudioRateNumRates];

static uint32_t mMixerTask, mListMutex = 0, mSamplesPerHwBuf = 0;
static struct AudioStream* mStreams = NULL;	//irq reads this with no locks so keep it consistent!
static enum AudioSampleRate mNativeRate;
static uint32_t mCyclesTillHwOff = 0;
static int32_t *mMixBuffer = NULL;
static uintptr_t mNextHandle = 0;
static bool mNativeStereo;

static bool audioPrvStreamCreateOutputMixFunc(struct AudioStream *strm);


#if MIXER_ALLOW_VFP_USE && !defined(HAVE_FPU)
#error "canot use vfp float reader in audio mixer with no fpu"
#endif




static struct AudioStream* audioPrvLookupOutputStreamByHandle(struct AudioOutputStrmHandle* h)	//call with lock held!
{
	struct AudioStream *t;
	
	for (t = mStreams; t && t->handle != (uintptr_t)h; t = t->next);
	
	return t;
}

static void audiostreamMaybeStartOutputHwLocked(struct AudioStream *strm)	//call with lock held!
{
	if (!mCyclesTillHwOff) {
		mCyclesTillHwOff = NUM_BUFFERS_TILL_SHUTOFF;
		audioOutHwSetState(true);
	}
}

struct AudioOutputStrmHandle* audiostreamCreateOutput(enum AudioSampleRate rate, enum AudioSampleType sampTyp, enum AudioChannelConfig chans, uint32_t samplesPerBuf, bool loopMode)
{
	uint32_t nSamples, nSubSamples, bufSz, allocSz, sema, i;
	struct AudioOutputStrmHandle* ret = NULL;
	struct AudioStream *strm;
	
	
	#ifdef ABSOLUTELY_NO_AUDIO_SUPORT
		return NULL;
	#endif
	
	logt("Creating output audio stream: rate %u sample type %u, channelMode %u\n",	rate, sampTyp, chans);
	
	switch (rate) {
		case AudioRate8000:
		case AudioRate11025:
		case AudioRate16000:
		case AudioRate22050:
		case AudioRate24000:
		case AudioRate32000:
		case AudioRate44100:
		case AudioRate48000:
			//nothing
			break;
		default:
			return NULL;
	}
	
	switch (sampTyp) {
		case AudioSampleU8:
		case AudioSampleS8:
		case AudioSampleU16LE:
		case AudioSampleU16BE:
		case AudioSampleS16LE:
		case AudioSampleS16BE:
		case AudioSampleU32LE:
		case AudioSampleU32BE:
		case AudioSampleS32LE:
		case AudioSampleS32BE:
		case AudioSampleFloatLE:
		case AudioSampleFloatBE:
			//nothing
			break;
		default:
			return NULL;
	}
	
	if (loopMode)	//we allow anything in loop mode since there i sonly one buffer and we do not care about how it ends
		nSamples = samplesPerBuf;
	else {
	
		//calculate how many subsamples we'll have per channel (round to 4 samples so every buffer starts at a word boundary (mix funcs expect this)
		for (nSamples = samplesPerBuf; nSamples & 3; nSamples++);
		
		//limit it to a sane window
		if (nSamples < 256)
			nSamples = 256;
		
		if (nSamples > 4096)
			nSamples = 4096;
	}
	
	switch (chans) {
		case AudioMono:
			nSubSamples = nSamples;
			break;
		case AudioStereo:
			nSubSamples = nSamples * 2;
			break;
		default:
			return NULL;
	}
	bufSz = nSubSamples << mSampleShifts[sampTyp];
	allocSz = bufSz * (loopMode ? 1 : BUFFER_DEPTH) + 8 /* to make sure that an over-read by one sample (as mix func can do) never goes past buffer end */;
	
	if (errNone != KALSemaphoreCreate(&sema, CREATE_4CC('a','S','T','R'), BUFFER_DEPTH))
		return NULL;
	
	strm = (struct AudioStream*)kheapAlloc(sizeof(struct AudioStream) + allocSz);
	if (!strm) {
		KALSemaphoreDelete(sema);
		return NULL;
	}
	memset(strm, 0, sizeof(struct AudioStream) + allocSz);
	
	strm->mixFunc = kheapAllocEx(OUT_MIX_FUNC_BYTES, MEM_USABLE_FOR_EXEC | MEM_FAST);
	if (!strm->mixFunc) {
		
		kheapFree(strm);
		KALSemaphoreDelete(sema);
		return NULL;
	}

	strm->rate = rate;
	strm->sampTyp = sampTyp;
	strm->chans = chans;
	strm->volumeL = AUDIO_VOLUME_UNITY;
	strm->volumeR = AUDIO_VOLUME_UNITY;
	strm->clientSema = sema;
	
	#ifdef SONY_SUPPORT_ENABLED
		strm->sonyLastMixedTicks = TimGetTicks();
	#endif
	
	strm->nSamples = nSamples;
	strm->loopMode = loopMode ? 1 : 0;
	strm->buf[0] = (uint8_t*)(strm + 1);
	for (i = 1; i < BUFFER_DEPTH; i++)
		strm->buf[i] = strm->buf[i - 1] + (loopMode ? 0 : bufSz);
	
	if (!audioPrvStreamCreateOutputMixFunc(strm)) {
		kheapFree(strm);
		return NULL;
	}

	KALMutexReserve(mListMutex, -1);
	do {
		strm->handle = ++mNextHandle;
	} while (strm->handle == 0 || strm->handle == 0xffffffff || audioPrvLookupOutputStreamByHandle((struct AudioOutputStrmHandle*)(strm->handle)));
	
	strm->next = mStreams;
	if (strm->next)
		strm->next->prev = strm;
	mStreams = strm;
	
	if (loopMode)
		audiostreamMaybeStartOutputHwLocked(strm);
	
	ret = (struct AudioOutputStrmHandle*)strm->handle;
	
	KALMutexRelease(mListMutex);
	
	return ret;
}

static void audiostreamPrvDestroyLockedOutput(struct AudioStream *strm)		//may unlock and relock mutex!!!
{
	strm->inTeardown = 1;
	
	if (!strm->haveWaiters) {
		
		if (strm->prev)
			strm->prev->next = strm->next;
		else
			mStreams = strm->next;
		
		if (strm->next)
			strm->next->prev = strm->prev;

		(void)KALSemaphoreDelete(strm->clientSema);

		//free memory
		kheapFree(strm->mixFunc);
		kheapFree(strm);
	}
}

void audiostreamDestroyOutput(struct AudioOutputStrmHandle* strmH)
{
	struct AudioStream *strm;
	
	KALMutexReserve(mListMutex, -1);
	strm = audioPrvLookupOutputStreamByHandle(strmH);
	if (strm)
		audiostreamPrvDestroyLockedOutput(strm);
	KALMutexRelease(mListMutex);
}

uint32_t audiostreamOutputGetOptimalBufSz(struct AudioOutputStrmHandle* strmH)	//in bytes
{
	struct AudioStream *strm;
	uint32_t ret = 0;
	
	KALMutexReserve(mListMutex, -1);
	strm = audioPrvLookupOutputStreamByHandle(strmH);
	
	ret = strm->buf[1] - strm->buf[0];
	
	KALMutexRelease(mListMutex);
	return ret;
}

#ifdef SONY_SUPPORT_ENABLED
	bool audiostreamOutputGetSonyStats(struct AudioOutputStrmHandle* strmH, uint32_t *numOutSamplesProducedP, uint32_t *lastMixedTicksP)
	{
		struct AudioStream *strm;
		bool ret = false;
		
		KALMutexReserve(mListMutex, -1);
		strm = audioPrvLookupOutputStreamByHandle(strmH);
		
		if (strm) {
			
			*numOutSamplesProducedP = strm->sonyNumOutSamplesProduced;
			*lastMixedTicksP = strm->sonyLastMixedTicks;
			ret = true;
		}
		
		KALMutexRelease(mListMutex);
		return ret;
	}
#endif

void audiostreamSetVolumesOutput(struct AudioOutputStrmHandle* strmH, uint32_t volumeL, uint32_t volumeR)
{
	struct AudioStream *strm;
	
	KALMutexReserve(mListMutex, -1);
	strm = audioPrvLookupOutputStreamByHandle(strmH);
	
	if (strm) {
		strm->volumeL = volumeL;
		strm->volumeR = volumeR;
	}
	KALMutexRelease(mListMutex);
}

void* audioOutputLoopStreamGetBuffer(struct AudioOutputStrmHandle* strmH)
{
	struct AudioStream *strm;
	void* ret = NULL;
	
	KALMutexReserve(mListMutex, -1);
	strm = audioPrvLookupOutputStreamByHandle(strmH);
	
	if (strm && strm->loopMode)
		ret = strm->buf[0];
	KALMutexRelease(mListMutex);

	return ret;
}

void audioOutputLoopStreamEnable(struct AudioOutputStrmHandle* strmH)
{
	struct AudioStream *strm;
	
	KALMutexReserve(mListMutex, -1);
	strm = audioPrvLookupOutputStreamByHandle(strmH);
	
	if (strm && strm->loopMode)
		strm->nMixerBufs = 1;
	KALMutexRelease(mListMutex);
}

uint32_t audiostreamAddSamplesOutput(struct AudioOutputStrmHandle* strmH, const void* samples, uint32_t size)
{
	const uint8_t *src = (const uint8_t*)samples;
	uint32_t i, ret, sizeOrig = size, sampleShift;
	struct AudioStream *strm;
	
	#ifdef ABSOLUTELY_NO_AUDIO_SUPORT
		return sizeOrig;
	#endif
	
	KALMutexReserve(mListMutex, -1);
	strm = audioPrvLookupOutputStreamByHandle(strmH);
	if (strm) {
		sampleShift = ((strm->chans == AudioStereo) ? 1 : 0) + mSampleShifts[strm->sampTyp];
	
		while (size >> sampleShift) {
		
			uint32_t dataAvail = size >> sampleShift, spaceAvail, now, nowBytes;
			bool completed = false;
			
			if (strm->bufUsedClient)			//there is a partially-filled buffer, use it, no need to wait
				spaceAvail = strm->nSamples - strm->bufUsedClient;
			else {								//no partially full buffer? wait for via semaphore
				if (!strm->loopMode) {
					uint32_t sema = strm->clientSema;
					
					strm->haveWaiters++;
					KALMutexRelease(mListMutex);
					if (errNone != KALSemaphoreWait(strm->clientSema, -1))
						goto out;
					KALMutexReserve(mListMutex, -1);	//stream CANNOT have gone away since we marked it as being waited on, but then we're on the hook to clean it up
					strm->haveWaiters--;
					if (strm->inTeardown && !strm->haveWaiters) {
						audiostreamPrvDestroyLockedOutput(strm);
						break;
					}
				}
				spaceAvail = strm->nSamples;
			}
			
			if (dataAvail < spaceAvail)
				now = dataAvail;
			else {
				now = spaceAvail;
				completed = true;
			}
			nowBytes = now << sampleShift;
			
			memcpy(strm->buf[strm->nextClientBuf] + (((uint32_t)strm->bufUsedClient) << sampleShift), src, nowBytes);
			size -= nowBytes;
			src += nowBytes;
			
			if (!completed)
				strm->bufUsedClient += now;
			else {
				strm->bufUsedClient = 0;
				if (!strm->loopMode) {
					
					strm->nextClientBuf = (strm->nextClientBuf == BUFFER_DEPTH - 1) ? 0 : strm->nextClientBuf + 1;
					atomicAdd8(&strm->nMixerBufs, 1);
					audiostreamMaybeStartOutputHwLocked(strm);
				}
			}
		}
	}
	KALMutexRelease(mListMutex);

out:
	return sizeOrig - size;
}

uint32_t audioGetNumOutStreams(void)
{
	struct AudioStream *t;
	uint32_t num = 0;
	
	KALMutexReserve(mListMutex, -1);
	
	for (t = mStreams; t; t = t->next)
		num++;
	
	KALMutexRelease(mListMutex);
	
	return num;
}

static const int32_t* audioPrvOutputHwReadyForMoreSamples(bool start)
{
	if (start)
		return mMixBuffer;
	
	KALTaskWake(mMixerTask);
	return NULL;
}

static int32_t* audioPrvMixInOutputStream(int32_t* dst, const uint8_t** srcP, uint32_t maxSampOut, struct AudioStream *strm, uint32_t maxSampIn)
{
	return ((MixInF)emitBufPtrToFuncPtr(strm->mixFunc))(dst, srcP, maxSampOut, strm->resampleState, strm->volumeL, strm->volumeR, maxSampIn);
}

static void audioPrvOutputMixerThread(void *param)
{
	uint32_t L = mSamplesPerHwBuf;
	
	while (1) {
		
		struct AudioStream *strm;
		
		memset(mMixBuffer, 0, L * sizeof(int32_t) * (mNativeStereo ? 2 : 1));
		KALMutexReserve(mListMutex, -1);
		
		for (strm = mStreams; strm; strm = strm->next) {
			
			uint32_t spaceAvail = L, dataAvail, sampleShift = ((strm->chans == AudioStereo) ? 1 : 0) + mSampleShifts[strm->sampTyp];
			int32_t *dst = mMixBuffer;
			
			if (strm->inTeardown)
				continue;
			
			while (spaceAvail) {
				
				if (!strm->bufUsedMixer && !strm->nMixerBufs && !strm->loopMode)		//no more half-used buf data and on full buffers? stop mixing the stream in
					break;
				
				dataAvail = strm->nSamples - strm->bufUsedMixer;
				
				if (spaceAvail && dataAvail) {
					
					uint32_t outSamplesProduced, inSamplesConsumed;
					const uint8_t *srcOut, *src;
					int32_t *dstOrig = dst;
					
					srcOut = src = strm->buf[strm->nextMixerBuf] + (((uint32_t)strm->bufUsedMixer) << sampleShift);
					dst = audioPrvMixInOutputStream(dst, &srcOut, spaceAvail, strm, dataAvail);
					
					outSamplesProduced = (dst - dstOrig) >> (mNativeStereo ? 1 : 0);
					inSamplesConsumed = (srcOut - src) >> sampleShift;
				
				#ifdef SONY_SUPPORT_ENABLED
					strm->sonyNumOutSamplesProduced += outSamplesProduced;
					strm->sonyLastMixedTicks = TimGetTicks();
				#endif
					
					spaceAvail -= outSamplesProduced;
				
					if (inSamplesConsumed < dataAvail) //we did not finish the buffer
						strm->bufUsedMixer += inSamplesConsumed;
					else {					//we finished a buffer - signal it
						
						strm->bufUsedMixer = 0;
						if (!strm->loopMode) {
						
							strm->nextMixerBuf = (strm->nextMixerBuf == BUFFER_DEPTH - 1) ? 0 : strm->nextMixerBuf + 1;
							strm->nMixerBufs--;
							(void)KALSemaphoreSignal(strm->clientSema);
						}
					}
				}
			}
		}
		
		if (mStreams)
			mCyclesTillHwOff = NUM_BUFFERS_TILL_SHUTOFF;
		else if (mCyclesTillHwOff) {
			if (!--mCyclesTillHwOff)
				audioOutHwSetState(false);
		}
		
		KALMutexRelease(mListMutex);
		KALTaskWait(-1);
	}
}

bool audioOutInit(enum AudioSampleRate *nativeRateP, bool *nativeStereoP)
{
	uint32_t i;
	Err e;
	struct KalTaskCreateInfo tc = {
		.codeToRun = audioPrvOutputMixerThread,
		.stackSz = 512,
		.prio = 4,			//very high prio
		.tag = CREATE_4CC('_','a','u','d'),
	};
	
	#ifdef ABSOLUTELY_NO_AUDIO_SUPORT
		return false;
	#endif
	
	if (!audioOutHwInit(audioPrvOutputHwReadyForMoreSamples, &mSamplesPerHwBuf, &mNativeRate, &mNativeStereo)) {
		loge("No audio output hw found\n");
		goto fail_hw_init;
	}

	if (errNone != KALMutexCreate(&mListMutex, CREATE_4CC('_', 'a', 'u', 'd')))
		fatal("Cannot create audio mutex\n");
	
	mMixBuffer = (int32_t*)kheapAlloc(mSamplesPerHwBuf * sizeof(int32_t) * (mNativeStereo ? 2 : 1));
	
	if (!mMixBuffer) {
		loge("Cannot allocate mix buffer\n");
		goto fail_mix_buffer;
	}
	
	e = KALTaskCreate(&mMixerTask, &tc);
	if (e) {
		loge("Failed to create the mixer task\n");
		goto fail_task;
	}
	
	e = KALTaskStart(mMixerTask, NULL);
	if (e) {
		loge("Failed to start the mixer task\n");
		goto fail_start;
	}
	
	if (nativeRateP)
		*nativeRateP = mNativeRate;
	if (nativeStereoP)
		*nativeStereoP = mNativeStereo;
	return true;

fail_start:
	KALTaskDelete(mMixerTask);

fail_task:
	kheapFree(mMixBuffer);

fail_mix_buffer:
fail_hw_init:
	
	return false;
}

static bool audioPrvStreamCreateOutputMixFunc(struct AudioStream *strm)
{
	const uint16_t *tab = mResampTabs[mNativeRate][strm->rate];
	enum EmitStatus ret;
	struct EmitBuf buf;
	
	emitBufferInit(&buf, strm->mixFunc, OUT_MIX_FUNC_BYTES);
	
	if (!tab)
		ret = audioPrvStreamCreateOutputMixFuncGutsNoResamp(&buf, strm->sampTyp, strm->chans, mNativeStereo);
	else if (mNativeRate > strm->rate) {
		strm->resampleState[2] = (uintptr_t)tab;
		ret = audioPrvStreamCreateOutputMixFuncGutsUpsample(&buf, tab, strm->sampTyp, strm->chans, mNativeStereo);
	}
	else {
		strm->resampleState[6] = (uintptr_t)tab;
		ret = audioPrvStreamCreateOutputMixFuncGutsDownsample(&buf, tab, strm->sampTyp, strm->chans, mNativeStereo);
	}
	
	emitBufCacheCoherencyManage(&buf);
	
	if (0) {
		
		char *src = buf.bufStart;
		
		loge(" AUDIO MIX FUNC:\n");
		while ((void*)src < buf.buf) {
			
			char x[64];
			
			src += disasm(x, src, ((uintptr_t)emitBufPtrToFuncPtr(src)) & 1, false);
			loge(" %s\n", x);
		}
	}
	
	if (ret != EmitErrNone) {
		
		fatal("stream mix func creation failed: %d\n", ret);
		return false;
	}
	
	return true;
}

///////////////////////////////////////////////////////////////////////////
//////////////////////////// conversion tables ////////////////////////////
///////////////////////////////////////////////////////////////////////////

//some conversions are rounded for table shortness within 1%
//some tables are replicated a few times to reducew resamp-tab-rewind-overhead
//names are of format resamp_$FROMFREQ_$TOFREQ

//common upsample tabs
static const uint16_t resamp_X_2X[] = {0xfffe, 0x8001, 0xfffe, 0x8001, 0xfffe, 0x8001, 0xfffe, 0x8001, 0xfffe, 0x8001, 0xfffe, 0x8001, 0};
static const uint16_t resamp_X_3X[] = {0xfffe, 0xaaaa, 0x5555, 0xfffe, 0xaaaa, 0x5555, 0xfffe, 0xaaaa, 0x5555, 0xfffe, 0xaaaa, 0x5555, 0};
static const uint16_t resamp_X_4X[] = {0xfffe, 0xc000, 0x8000, 0x4001, 0xfffe, 0xc000, 0x8000, 0x4001, 0xfffe, 0xc000, 0x8000, 0x4001, 0};
static const uint16_t resamp_X_6X[] = {0xfffe, 0xd554, 0xaaaa, 0x8000, 0x5554, 0x2aab, 0xfffe, 0xd554, 0xaaaa, 0x8000, 0x5554, 0x2aab, 0};
static const uint16_t resamp_2X_3X[] = {0xfffe, 0x5555, 0xaaab, 0xfffe, 0x5555, 0xaaab, 0xfffe, 0x5555, 0xaaab, 0xfffe, 0x5555, 0xaaab, 0};
static const uint16_t resamp_3X_4X[] = {0xfffe, 0x4001, 0x8001, 0xc001, 0xfffe, 0x4001, 0x8001, 0xc001, 0xfffe, 0x4001, 0x8001, 0xc001, 0};
static const uint16_t resamp_441X_480X[] = {0xfffe, 0x14cd, 0x2999, 0x3e67, 0x5333, 0x6801, 0x7ccd, 0x9199, 0xa667, 0xbb33, 0xd001, 0xe4cd, 0xf998, 0x0e67, 0x2333, 0x3801, 0x4ccd, 0x6199, 0x7667, 0x8b33, 0xa001, 0xb4cd, 0xc999, 0xde67, 0xf332, 0x0801, 0x1ccd, 0x3199, 0x4667, 0x5b33, 0x7001, 0x84cd, 0x9999, 0xae67, 0xc333, 0xd801, 0xeccc, 0x0199, 0x1667, 0x2b33, 0x4001, 0x54cd, 0x6999, 0x7e67, 0x9333, 0xa801, 0xbccd, 0xd199, 0xe667, 0xfb32, 0x1001, 0x24cd, 0x3999, 0x4e67, 0x6333, 0x7801, 0x8ccd, 0xa199, 0xb667, 0xcb33, 0xe001, 0xf4cc, 0x0999, 0x1e67, 0x3333, 0x4801, 0x5ccd, 0x7199, 0x8667, 0x9b33, 0xb001, 0xc4cd, 0xd999, 0xee66, 0x0333, 0x1801, 0x2ccd, 0x4199, 0x5667, 0x6b33, 0x8001, 0x94cd, 0xa999, 0xbe67, 0xd333, 0xe801, 0xfccc, 0x1199, 0x2667, 0x3b33, 0x5001, 0x64cd, 0x7999, 0x8e67, 0xa333, 0xb801, 0xcccd, 0xe199, 0xf666, 0x0b33, 0x2001, 0x34cd, 0x4999, 0x5e67, 0x7333, 0x8801, 0x9ccd, 0xb199, 0xc667, 0xdb33, 0xf000, 0x04cd, 0x1999, 0x2e67, 0x4333, 0x5801, 0x6ccd, 0x8199, 0x9667, 0xab33, 0xc001, 0xd4cd, 0xe999, 0xfe66, 0x1333, 0x2801, 0x3ccd, 0x5199, 0x6667, 0x7b33, 0x9001, 0xa4cd, 0xb999, 0xce67, 0xe333, 0xf800, 0x0ccd, 0x2199, 0x3667, 0x4b33, 0x6001, 0x74cd, 0x8999, 0x9e67, 0xb333, 0xc801, 0xdccd, 0xf198, 0x0667, 0x1b33, 0x3001, 0x44cd, 0x5999, 0x6e67, 0x8333, 0x9801, 0xaccd, 0xc199, 0xd667, 0xeb33, 0};
static const uint16_t resamp_22X_48X[] = {0xfffe, 0x8aaa, 0x1555, 0xa000, 0x2aab, 0xb554, 0x4001, 0xcaaa, 0x5555, 0xe000, 0x6aab, 0xf554, 0x8000, 0x0aab, 0x9554, 0x2001, 0xaaaa, 0x3555, 0xc000, 0x4aab, 0xd554, 0x6001, 0xeaaa, 0x7555, 0};
static const uint16_t resamp_16X_44X[] = {0xfffe, 0xa2e8, 0x45d1, 0xe8ba, 0x8ba2, 0x2e8b, 0xd174, 0x745c, 0x1745, 0xba2e, 0x5d17, 0};
static const uint16_t resamp_32X_44X[] = {0xfffe, 0x45d1, 0x8ba3, 0xd174, 0x1745, 0x5d17, 0xa2e9, 0xe8ba, 0x2e8b, 0x745d, 0xba2f, 0};
static const uint16_t resamp_22X_32X[] = {0xfffe, 0x5001, 0xa001, 0xf000, 0x4001, 0x9001, 0xe000, 0x3001, 0x8001, 0xd000, 0x2001, 0x7001, 0xc000, 0x1001, 0x6001, 0xb001, 0};

//common downsample tabs
static const uint16_t resamp_4X_3X[] = {0xe000, 0xa001, 0x4000, 0xc001, 0x2000, 0xe001, 0xe000, 0xa001, 0x4000, 0xc001, 0x2000, 0xe001, 0};
static const uint16_t resamp_3X_2X[] = {0xd556, 0xaaab, 0x2aaa, 0xd557, 0xd556, 0xaaab, 0x2aaa, 0xd557, 0xd556, 0xaaab, 0x2aaa, 0xd557, 0};
static const uint16_t resamp_2X_X[] = {0xc000, 0xc001, 0xc000, 0xc001, 0xc000, 0xc001, 0xc000, 0xc001, 0xc000, 0xc001, 0xc000, 0xc001, 0xc000, 0xc001, 0};
static const uint16_t resamp_3X_X[] = {0xaaac, 0xaaaa, 0xaaab, 0xaaac, 0xaaaa, 0xaaab, 0xaaac, 0xaaaa, 0xaaab, 0xaaac, 0xaaaa, 0xaaab, 0};
static const uint16_t resamp_4X_X[] = {0xa000, 0xa000, 0xa000, 0xa001, 0xa000, 0xa000, 0xa000, 0xa001, 0xa000, 0xa000, 0xa000, 0xa001, 0};
static const uint16_t resamp_6X_X[] = {0x9554, 0x9554, 0x9556, 0x9556, 0x9556, 0x9557, 0x9554, 0x9554, 0x9556, 0x9556, 0x9556, 0x9557, 0};
static const uint16_t resamp_480X_441X[] = {0xf59a, 0x8a67, 0x6b34, 0x94cd, 0x60cc, 0x9f35, 0x5666, 0xa99b, 0x4c00, 0xb401, 0x419a, 0xbe67, 0x3734, 0xc8cd, 0x2ccc, 0xd335, 0x2266, 0xdd9b, 0x1800, 0xe801, 0x0d9a, 0xf267, 0x0332, 0xf59a, 0x8735, 0x6e66, 0x919b, 0x6400, 0x9c01, 0x599a, 0xa667, 0x4f34, 0xb0cd, 0x44cc, 0xbb35, 0x3a66, 0xc59b, 0x3000, 0xd001, 0x259a, 0xda67, 0x1b34, 0xe4cd, 0x10cc, 0xef35, 0x0666, 0xf59a, 0x8401, 0x719a, 0x8e67, 0x6734, 0x98cd, 0x5ccc, 0xa335, 0x5266, 0xad9b, 0x4800, 0xb801, 0x3d9a, 0xc267, 0x3334, 0xcccd, 0x28cc, 0xd735, 0x1e66, 0xe19b, 0x1400, 0xec01, 0x099a, 0xf59a, 0x80cd, 0x74cc, 0x8b35, 0x6a66, 0x959b, 0x6000, 0xa001, 0x559a, 0xaa67, 0x4b34, 0xb4cd, 0x40cc, 0xbf35, 0x3666, 0xc99b, 0x2c00, 0xd401, 0x219a, 0xde67, 0x1734, 0xe8cd, 0x0ccc, 0xf335, 0x0266, 0xf59a, 0x8801, 0x6d9a, 0x9267, 0x6334, 0x9ccd, 0x58cc, 0xa735, 0x4e66, 0xb19b, 0x4400, 0xbc01, 0x399a, 0xc667, 0x2f34, 0xd0cd, 0x24cc, 0xdb35, 0x1a66, 0xe59b, 0x1000, 0xf001, 0x059a, 0xf59a, 0x84cd, 0x70cc, 0x8f35, 0x6666, 0x999b, 0x5c00, 0xa401, 0x519a, 0xae67, 0x4734, 0xb8cd, 0x3ccc, 0xc335, 0x3266, 0xcd9b, 0x2800, 0xd801, 0x1d9a, 0xe267, 0x1334, 0xeccd, 0x08cc, 0xf59a, 0x819b, 0x7400, 0x8c01, 0x699a, 0x9667, 0x5f34, 0xa0cd, 0x54cc, 0xab35, 0x4a66, 0xb59b, 0x4000, 0xc001, 0x359a, 0xca67, 0x2b34, 0xd4cd, 0x20cc, 0xdf35, 0x1666, 0xe99b, 0x0c00, 0xf401, 0x019a, 0xf59a, 0x88cd, 0x6ccc, 0x9335, 0x6266, 0x9d9b, 0x5800, 0xa801, 0x4d9a, 0xb267, 0x4334, 0xbccd, 0x38cc, 0xc735, 0x2e66, 0xd19b, 0x2400, 0xdc01, 0x199a, 0xe667, 0x0f34, 0xf0cd, 0x04cc, 0xf59a, 0x859b, 0x7000, 0x9001, 0x659a, 0x9a67, 0x5b34, 0xa4cd, 0x50cc, 0xaf35, 0x4666, 0xb99b, 0x3c00, 0xc401, 0x319a, 0xce67, 0x2734, 0xd8cd, 0x1ccc, 0xe335, 0x1266, 0xed9b, 0x0800, 0xf59a, 0x8267, 0x7334, 0x8ccd, 0x68cc, 0x9735, 0x5e66, 0xa19b, 0x5400, 0xac01, 0x499a, 0xb667, 0x3f34, 0xc0cd, 0x34cc, 0xcb35, 0x2a66, 0xd59b, 0x2000, 0xe001, 0x159a, 0xea67, 0x0b34, 0xf4cd, 0x00cc, 0xf59a, 0x899b, 0x6c00, 0x9401, 0x619a, 0x9e67, 0x5734, 0xa8cd, 0x4ccc, 0xb335, 0x4266, 0xbd9b, 0x3800, 0xc801, 0x2d9a, 0xd267, 0x2334, 0xdccd, 0x18cc, 0xe735, 0x0e66, 0xf19b, 0x0400, 0xf59a, 0x8667, 0x6f34, 0x90cd, 0x64cc, 0x9b35, 0x5a66, 0xa59b, 0x5000, 0xb001, 0x459a, 0xba67, 0x3b34, 0xc4cd, 0x30cc, 0xcf35, 0x2666, 0xd99b, 0x1c00, 0xe401, 0x119a, 0xee67, 0x0732, 0xf59a, 0x8335, 0x7266, 0x8d9b, 0x6800, 0x9801, 0x5d9a, 0xa267, 0x5334, 0xaccd, 0x48cc, 0xb735, 0x3e66, 0xc19b, 0x3400, 0xcc01, 0x299a, 0xd667, 0x1f34, 0xe0cd, 0x14cc, 0xeb35, 0x0a66, 0xf59b, 0};
static const uint16_t resamp_48X_22X[] = {0xbaac, 0xbaaa, 0x8aab, 0x3000, 0xbaaa, 0x9557, 0x2556, 0xbaaa, 0xa001, 0x1aac, 0xbaaa, 0xaaab, 0x1000, 0xbaaa, 0xb557, 0x0556, 0xbaaa, 0xbaaa, 0x8557, 0x3556, 0xbaaa, 0x9001, 0x2aac, 0xbaaa, 0x9aab, 0x2000, 0xbaaa, 0xa557, 0x1556, 0xbaaa, 0xb001, 0x0aac, 0xbaaa, 0xbaab, 0};
static const uint16_t resamp_44X_16X[] = {0xae8c, 0xae8c, 0xa2e9, 0x0ba2, 0xae8c, 0xae8c, 0x9747, 0x1746, 0xae8c, 0xae8c, 0x8ba3, 0x22e8, 0xae8c, 0xae8d, 0};
static const uint16_t resamp_44X_32X[] = {0xdd18, 0xa2e9, 0x3a2e, 0xc5d3, 0x1746, 0xdd18, 0x8ba3, 0x5174, 0xae8d, 0x2e8c, 0xd175, 0x0ba2, 0xdd18, 0x9747, 0x45d2, 0xba2f, 0x22e8, 0xdd19, 0};
static const uint16_t resamp_32X_22X[] = {0xd800, 0xa801, 0x3000, 0xd001, 0x0800, 0xd800, 0xa001, 0x3800, 0xc801, 0x1000, 0xd800, 0x9801, 0x4000, 0xc001, 0x1800, 0xd800, 0x9001, 0x4800, 0xb801, 0x2000, 0xd800, 0x8801, 0x5000, 0xb001, 0x2800, 0xd801, 0};

//uncommon upsample tabs
static const uint16_t resamp_11X_48X[] = {0xfffe, 0xc554, 0x8aaa, 0x5000, 0x1555, 0xdaaa, 0xa000, 0x6554, 0x2aab, 0xf000, 0xb554, 0x7aaa, 0x4000, 0x0555, 0xcaaa, 0x9000, 0x5554, 0x1aab, 0xe000, 0xa554, 0x6aaa, 0x3001, 0xf554, 0xbaaa, 0x8000, 0x4554, 0x0aab, 0xd000, 0x9554, 0x5aaa, 0x2001, 0xe554, 0xaaaa, 0x7000, 0x3555, 0xfaaa, 0xc000, 0x8554, 0x4aaa, 0x1001, 0xd554, 0x9aaa, 0x6000, 0x2555, 0xeaaa, 0xb000, 0x7554, 0x3aab, 0};
static const uint16_t resamp_8X_44X[] = {0xfffe, 0xd174, 0xa2e8, 0x745c, 0x45d0, 0x1745, 0xe8ba, 0xba2e, 0x8ba2, 0x5d16, 0x2e8b, 0};
static const uint16_t resamp_24X_44X[] = {0xfffe, 0x745d, 0xe8ba, 0x5d17, 0xd174, 0x45d1, 0xba2e, 0x2e8b, 0xa2e8, 0x1745, 0x8ba3, 0};
static const uint16_t resamp_11X_32X[] = {0xfffe, 0xa800, 0x5001, 0xf800, 0xa000, 0x4801, 0xf000, 0x9800, 0x4001, 0xe800, 0x9000, 0x3801, 0xe000, 0x8800, 0x3001, 0xd800, 0x8000, 0x2801, 0xd000, 0x7800, 0x2001, 0xc800, 0x7000, 0x1801, 0xc000, 0x6800, 0x1001, 0xb800, 0x6000, 0x0801, 0xb000, 0x5801, 0};

//uncommon downsample tabs
static const uint16_t resamp_44100_24000[] = {0xc5a8, 0xba59, 0x0b52, 0xc5a8, 0xaf07, 0x16a4, 0xc5a8, 0xa3b5, 0x21f6, 0xc5a8, 0x9863, 0x2d48, 0xc5a8, 0x8d11, 0x389a, 0xc5a8, 0x81bf, 0x43ec, 0xbc15, 0x0994, 0xc5a8, 0xb0c5, 0x14e6, 0xc5a8, 0xa573, 0x2038, 0xc5a8, 0x9a21, 0x2b8a, 0xc5a8, 0x8ecf, 0x36dc, 0xc5a8, 0x837d, 0x422e, 0xbdd3, 0x07d8, 0xc5a8, 0xb281, 0x1328, 0xc5a8, 0xa731, 0x1e7a, 0xc5a8, 0x9bdf, 0x29cc, 0xc5a8, 0x908d, 0x351e, 0xc5a8, 0x853b, 0x4070, 0xbf91, 0x061a, 0xc5a8, 0xb43f, 0x116c, 0xc5a8, 0xa8ed, 0x1cbe, 0xc5a8, 0x9d9b, 0x280e, 0xc5a8, 0x924b, 0x3360, 0xc5a8, 0x86f9, 0x3eb2, 0xc14f, 0x045c, 0xc5a8, 0xb5fd, 0x0fae, 0xc5a8, 0xaaab, 0x1b00, 0xc5a8, 0x9f59, 0x2652, 0xc5a8, 0x9407, 0x31a2, 0xc5a8, 0x88b7, 0x3cf4, 0xc30d, 0x029e, 0xc5a8, 0xb7bb, 0x0df0, 0xc5a8, 0xac69, 0x1942, 0xc5a8, 0xa117, 0x2494, 0xc5a8, 0x95c5, 0x2fe6, 0xc5a8, 0x8a73, 0x3b36, 0xc4cb, 0x00e0, 0xc5a8, 0xb979, 0x0c32, 0xc5a8, 0xae27, 0x1784, 0xc5a8, 0xa2d5, 0x22d6, 0xc5a8, 0x9783, 0x2e28, 0xc5a8, 0x8c31, 0x397a, 0xc5a8, 0x80df, 0x44ca, 0xbb37, 0x0a74, 0xc5a8, 0xafe5, 0x15c6, 0xc5a8, 0xa493, 0x2118, 0xc5a8, 0x9941, 0x2c6a, 0xc5a8, 0x8def, 0x37bc, 0xc5a8, 0x829d, 0x430c, 0xbcf5, 0x08b6, 0xc5a8, 0xb1a3, 0x1408, 0xc5a8, 0xa651, 0x1f5a, 0xc5a8, 0x9aff, 0x2aac, 0xc5a8, 0x8fad, 0x35fe, 0xc5a8, 0x845b, 0x414e, 0xbeb3, 0x06f8, 0xc5a8, 0xb361, 0x124a, 0xc5a8, 0xa80f, 0x1d9c, 0xc5a8, 0x9cbd, 0x28ee, 0xc5a8, 0x916b, 0x3440, 0xc5a8, 0x8619, 0x3f90, 0xc071, 0x053a, 0xc5a8, 0xb51f, 0x108c, 0xc5a8, 0xa9cd, 0x1bde, 0xc5a8, 0x9e7b, 0x2730, 0xc5a8, 0x9329, 0x3282, 0xc5a8, 0x87d7, 0x3dd2, 0xc22f, 0x037c, 0xc5a8, 0xb6dd, 0x0ece, 0xc5a8, 0xab8b, 0x1a20, 0xc5a8, 0xa039, 0x2572, 0xc5a8, 0x94e7, 0x30c4, 0xc5a8, 0x8995, 0x3c14, 0xc3ed, 0x01be, 0xc5a8, 0xb89b, 0x0d10, 0xc5a8, 0xad49, 0x1862, 0xc5a8, 0xa1f7, 0x23b4, 0xc5a8, 0x96a5, 0x2f06, 0xc5a8, 0x8b53, 0x3a58, 0xc5a9, 0};
static const uint16_t resamp_24000_22050[] = {0xf59a, 0x8a67, 0x6b34, 0x94cd, 0x60cc, 0x9f35, 0x5666, 0xa99b, 0x4c00, 0xb401, 0x419a, 0xbe67, 0x3734, 0xc8cd, 0x2ccc, 0xd335, 0x2266, 0xdd9b, 0x1800, 0xe801, 0x0d9a, 0xf267, 0x0332, 0xf59a, 0x8735, 0x6e66, 0x919b, 0x6400, 0x9c01, 0x599a, 0xa667, 0x4f34, 0xb0cd, 0x44cc, 0xbb35, 0x3a66, 0xc59b, 0x3000, 0xd001, 0x259a, 0xda67, 0x1b34, 0xe4cd, 0x10cc, 0xef35, 0x0666, 0xf59a, 0x8401, 0x719a, 0x8e67, 0x6734, 0x98cd, 0x5ccc, 0xa335, 0x5266, 0xad9b, 0x4800, 0xb801, 0x3d9a, 0xc267, 0x3334, 0xcccd, 0x28cc, 0xd735, 0x1e66, 0xe19b, 0x1400, 0xec01, 0x099a, 0xf59a, 0x80cd, 0x74cc, 0x8b35, 0x6a66, 0x959b, 0x6000, 0xa001, 0x559a, 0xaa67, 0x4b34, 0xb4cd, 0x40cc, 0xbf35, 0x3666, 0xc99b, 0x2c00, 0xd401, 0x219a, 0xde67, 0x1734, 0xe8cd, 0x0ccc, 0xf335, 0x0266, 0xf59a, 0x8801, 0x6d9a, 0x9267, 0x6334, 0x9ccd, 0x58cc, 0xa735, 0x4e66, 0xb19b, 0x4400, 0xbc01, 0x399a, 0xc667, 0x2f34, 0xd0cd, 0x24cc, 0xdb35, 0x1a66, 0xe59b, 0x1000, 0xf001, 0x059a, 0xf59a, 0x84cd, 0x70cc, 0x8f35, 0x6666, 0x999b, 0x5c00, 0xa401, 0x519a, 0xae67, 0x4734, 0xb8cd, 0x3ccc, 0xc335, 0x3266, 0xcd9b, 0x2800, 0xd801, 0x1d9a, 0xe267, 0x1334, 0xeccd, 0x08cc, 0xf59a, 0x819b, 0x7400, 0x8c01, 0x699a, 0x9667, 0x5f34, 0xa0cd, 0x54cc, 0xab35, 0x4a66, 0xb59b, 0x4000, 0xc001, 0x359a, 0xca67, 0x2b34, 0xd4cd, 0x20cc, 0xdf35, 0x1666, 0xe99b, 0x0c00, 0xf401, 0x019a, 0xf59a, 0x88cd, 0x6ccc, 0x9335, 0x6266, 0x9d9b, 0x5800, 0xa801, 0x4d9a, 0xb267, 0x4334, 0xbccd, 0x38cc, 0xc735, 0x2e66, 0xd19b, 0x2400, 0xdc01, 0x199a, 0xe667, 0x0f34, 0xf0cd, 0x04cc, 0xf59a, 0x859b, 0x7000, 0x9001, 0x659a, 0x9a67, 0x5b34, 0xa4cd, 0x50cc, 0xaf35, 0x4666, 0xb99b, 0x3c00, 0xc401, 0x319a, 0xce67, 0x2734, 0xd8cd, 0x1ccc, 0xe335, 0x1266, 0xed9b, 0x0800, 0xf59a, 0x8267, 0x7334, 0x8ccd, 0x68cc, 0x9735, 0x5e66, 0xa19b, 0x5400, 0xac01, 0x499a, 0xb667, 0x3f34, 0xc0cd, 0x34cc, 0xcb35, 0x2a66, 0xd59b, 0x2000, 0xe001, 0x159a, 0xea67, 0x0b34, 0xf4cd, 0x00cc, 0xf59a, 0x899b, 0x6c00, 0x9401, 0x619a, 0x9e67, 0x5734, 0xa8cd, 0x4ccc, 0xb335, 0x4266, 0xbd9b, 0x3800, 0xc801, 0x2d9a, 0xd267, 0x2334, 0xdccd, 0x18cc, 0xe735, 0x0e66, 0xf19b, 0x0400, 0xf59a, 0x8667, 0x6f34, 0x90cd, 0x64cc, 0x9b35, 0x5a66, 0xa59b, 0x5000, 0xb001, 0x459a, 0xba67, 0x3b34, 0xc4cd, 0x30cc, 0xcf35, 0x2666, 0xd99b, 0x1c00, 0xe401, 0x119a, 0xee67, 0x0732, 0xf59a, 0x8335, 0x7266, 0x8d9b, 0x6800, 0x9801, 0x5d9a, 0xa267, 0x5334, 0xaccd, 0x48cc, 0xb735, 0x3e66, 0xc19b, 0x3400, 0xcc01, 0x299a, 0xd667, 0x1f34, 0xe0cd, 0x14cc, 0xeb35, 0x0a66, 0xf59b, 0};
static const uint16_t resamp_32000_11025[] = {0xac00, 0xac00, 0xa801, 0x0400, 0xac00, 0xac00, 0xa401, 0x0800, 0xac00, 0xac00, 0xa001, 0x0c00, 0xac00, 0xac00, 0x9c01, 0x1000, 0xac00, 0xac00, 0x9801, 0x1400, 0xac00, 0xac00, 0x9401, 0x1800, 0xac00, 0xac00, 0x9001, 0x1c00, 0xac00, 0xac00, 0x8c01, 0x2000, 0xac00, 0xac00, 0x8801, 0x2400, 0xac00, 0xac00, 0x8401, 0x2800, 0xac00, 0xac01, 0};
static const uint16_t resamp_48000_11025[] = {0x9d54, 0x9d56, 0x9d56, 0x9d56, 0x8aab, 0x12a8, 0x9d56, 0x9d56, 0x9d56, 0x9557, 0x07fe, 0x9d56, 0x9d56, 0x9d56, 0x9d56, 0x82ab, 0x1aa8, 0x9d56, 0x9d56, 0x9d56, 0x8d57, 0x0ffe, 0x9d56, 0x9d56, 0x9d56, 0x9801, 0x0554, 0x9d54, 0x9d56, 0x9d56, 0x9d56, 0x8557, 0x17fe, 0x9d56, 0x9d56, 0x9d56, 0x9001, 0x0d54, 0x9d56, 0x9d56, 0x9d56, 0x9aab, 0x02a8, 0x9d56, 0x9d56, 0x9d56, 0x9d56, 0x8801, 0x1554, 0x9d56, 0x9d56, 0x9d56, 0x92ab, 0x0aa8, 0x9d56, 0x9d56, 0x9d56, 0x9d57, 0};
static const uint16_t resamp_44100_8000[] = {0x9746, 0x9746, 0x9746, 0x9746, 0x9746, 0x8ba3, 0x0ba2, 0x9746, 0x9746, 0x9746, 0x9746, 0x9747, 0};

static const uint16_t *mResampTabs[AudioRateNumRates][AudioRateNumRates] = {	//first index is TO, second is FROM
	[AudioRate8000] = {
		[AudioRate8000] = NULL,
		[AudioRate11025] = resamp_44X_32X,
		[AudioRate16000] = resamp_2X_X,
		[AudioRate22050] = resamp_44X_16X,
		[AudioRate24000] = resamp_3X_X,
		[AudioRate32000] = resamp_4X_X,
		[AudioRate44100] = resamp_44100_8000,
		[AudioRate48000] = resamp_6X_X,
	},
	[AudioRate11025] = {
		[AudioRate8000] = resamp_32X_44X,
		[AudioRate11025] = NULL,
		[AudioRate16000] = resamp_32X_22X,
		[AudioRate22050] = resamp_2X_X,
		[AudioRate24000] = resamp_48X_22X,
		[AudioRate32000] = resamp_32000_11025,
		[AudioRate44100] = resamp_4X_X,
		[AudioRate48000] = resamp_48000_11025,
	},
	[AudioRate16000] = {
		[AudioRate8000] = resamp_X_2X,
		[AudioRate11025] = resamp_22X_32X,
		[AudioRate16000] = NULL,
		[AudioRate22050] = resamp_44X_32X,
		[AudioRate24000] = resamp_3X_2X,
		[AudioRate32000] = resamp_2X_X,
		[AudioRate44100] = resamp_44X_16X,
		[AudioRate48000] = resamp_3X_X,
	},
	[AudioRate22050] = {
		[AudioRate8000] = resamp_16X_44X,
		[AudioRate11025] = resamp_X_2X,
		[AudioRate16000] = resamp_32X_44X,
		[AudioRate22050] = NULL,
		[AudioRate24000] = resamp_480X_441X,
		[AudioRate32000] = resamp_32X_22X,
		[AudioRate44100] = resamp_2X_X,
		[AudioRate48000] = resamp_48X_22X,
	},
	[AudioRate24000] = {
		[AudioRate8000] = resamp_X_3X,
		[AudioRate11025] = resamp_22X_48X,
		[AudioRate16000] = resamp_2X_3X,
		[AudioRate22050] = resamp_441X_480X,
		[AudioRate24000] = NULL,
		[AudioRate32000] = resamp_4X_3X,
		[AudioRate44100] = resamp_44100_24000,
		[AudioRate48000] = resamp_2X_X,
	},
	[AudioRate32000] = {
		[AudioRate8000] = resamp_X_4X,
		[AudioRate11025] = resamp_11X_32X,
		[AudioRate16000] = resamp_X_2X,
		[AudioRate22050] = resamp_22X_32X,
		[AudioRate24000] = resamp_3X_4X,
		[AudioRate32000] = NULL,
		[AudioRate44100] = resamp_44X_32X,
		[AudioRate48000] = resamp_3X_2X,
	},
	[AudioRate44100] = {
		[AudioRate8000] = resamp_8X_44X,
		[AudioRate11025] = resamp_X_4X,
		[AudioRate16000] = resamp_16X_44X,
		[AudioRate22050] = resamp_X_2X,
		[AudioRate24000] = resamp_24X_44X,
		[AudioRate32000] = resamp_32X_44X,
		[AudioRate44100] = NULL,
		[AudioRate48000] = resamp_480X_441X,
	},
	[AudioRate48000] = {
		[AudioRate8000] = resamp_X_6X,
		[AudioRate11025] = resamp_11X_48X,
		[AudioRate16000] = resamp_X_3X,
		[AudioRate22050] = resamp_22X_48X,
		[AudioRate24000] = resamp_X_2X,
		[AudioRate32000] = resamp_2X_3X,
		[AudioRate44100] = resamp_441X_480X,
		[AudioRate48000] = NULL,
	},
};

/*
	== downsampling table generation ==
	
	
	//	table entry {
	//		uint16_t emit	: 1;	//set if we should emit one at the end
	//		uint16_t mul	:14;
	//		uint16_t newsamp :1;	//set if we need a new sample at the start
	//	}
	
	
	
	var from = 44100, to = 24000;
			
	var _gc = gcd(from, to);
	var i, j, each = from / to, prevSamp = -1;
	
	document.write("{");
	for (i = 0; i < to / _gc; i++) {
		var beginsAt = i * from / to, begSamp = parseInt(beginsAt);
		var endsAt = (i + 1) * from / to, endSamp = parseInt(endsAt);
		var itms = [], weightTotal = 0, weightMax = 16384;
		
		if (begSamp == endSamp)
			alert("sorry, but no");
		
		//first is weighted
		itms[itms.length] = [begSamp, begSamp + 1 - beginsAt];
		
		//next arents
		for (j = begSamp + 1; j < endSamp; j++)
			itms[itms.length] = [j, 1];
		
		//last is weighted
		if (endsAt != endSamp)
			itms[itms.length] = [endSamp, endsAt - endSamp];
		
		//calc weights
		for (j = 0; j < itms.length; j++)
			weightTotal += (itms[j][2] = parseInt(weightMax * itms[j][1] / each + 0.5));
		
		//remove excess and add missing
		for (j = 0; j < itms.length && weightTotal != weightMax; j++) {
			if (weightTotal < weightMax && itms[j][2] < weightMax - 1) {
				itms[j][2]++;
				weightTotal++;
			}
			if (weightTotal > weightMax && itms[j][2] > 0) {
				itms[j][2]--;
				weightTotal--;
			}
		}
		if (weightTotal != weightMax) 
			alert("weighting failed");
		
		//move bits into middle position as needed
		for (j = 0; j < itms.length; j++)
			itms[j][2]<<= 1;
		
		//add "emit" bit to last ones
		itms[itms.length - 1][2] |= 1;
		
		//add "consume" bits where needed
		 for (j = 0; j < itms.length; j++) {
		 	if (itms[j][0] != prevSamp) {
				prevSamp = itms[j][0];
				itms[j][2] |= 0x8000;
			}
		 }
		
		//emit table entries
		for (j = 0; j < itms.length; j++) {
			var s = itms[j][2].toString(16);
			
			while (s.length < 4)
				s = "0" + s;
			
			document.write("0x" + s + ", ");
		}
	}
	document.write("0};");
	
	
	function gcd(a, b)
	{
		while (a != b) {
			if (a > b)
				a -= b;
			else
				b -= a;
		}
	
		return a;
	}


	== upsampling table generation ==
	each entry is of the form 15.1, bottom bit tells us if we should advance one input sample
	the other 15bits are weight to multiply input sample by, next sample is multiplied with (0x8000 - this value)
	
	var from = 11, to = 32;
	
	var _gc = gcd(from, to);
	var i;
	
	document.write("{");
	for (i = 0; i < to / _gc; i++) {
		var exact = i * from / to;
		var first = parseInt(exact), second = first + 1, nextFirst = parseInt((i + 1) * from / to);
		var mulFirst = (second - exact), mulSecond = (exact - first);
		var coeff = parseInt(0x8000 * mulFirst);
		
		if (coeff == 0x8000)
			coeff = 0x7fff;
		else if (coeff == 0)
			coeff = 1;
		
		coeff <<= 1;
		
		if (nextFirst != first)
			coeff++;
		
		coeff = coeff.toString(16);
		while (coeff.length < 4)
			coeff = "0" + coeff;
		coeff = "0x" + coeff;
		
		document.write(coeff + ", ");
	}
	document.write("0};");
	
	
	function gcd(a, b)
	{
		while (a != b) {
			if (a > b)
				a -= b;
			else
				b -= a;
		}
		
		return a;
	}





*/
