#include <ErrorBase.h>
#include <string.h>
#include <stdio.h>
#include "audiohw.h"
#include "printf.h"
#include "atomic.h"
#include "audio.h"
#include "heap.h"
#include "boot.h"
#include "mpu.h"
#include "kal.h"
#include "dal.h"

#define NUM_MIC_BUFFERS		4


static volatile uint8_t mBufW, mBufR, mNumBufsFree, mReadingBuf;
static uint32_t mInBufReadPos;
static int16_t *mBuffers[NUM_MIC_BUFFERS];
static uint32_t mNumSamplesPerHwBuf;
static void *mConvertFunc;
static uint8_t mSampSzShift;
static uint32_t mBufferSem;


// * keep in mind the counter here is in "samples" so {L+R} counts as one
// * return pointer to next unfilled output sample
// * DO NOT CALL with nSamples as zero!!!
typedef void* (*MicCvtF)(void* dst, const int16_t *src, uint32_t volumeL, uint32_t volumeR, uint32_t nSamplesOver2);



#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

static void __attribute__((naked)) audioMicPrvCopyToBuf(int16_t *dst, const uint16_t *src, uint32_t midpoint, uint32_t nsamp)
{
	#if defined(BUILD_FOR_THUMB_1)
	
		asm volatile(
			".syntax unified				\n\t"
			"	lsrs   r3, #2				\n\t"	//we process 4 at a time
			"	push   {r4-r7}				\n\t"
			"1:								\n\t"
			"	ldrh   r7, [r1, #0x06]		\n\t"
			"	ldrh   r6, [r1, #0x04]		\n\t"
			"	ldrh   r5, [r1, #0x02]		\n\t"
			"	ldrh   r4, [r1, #0x00]		\n\t"
			"	adds   r1, #0x08			\n\t"
			"	subs   r7, r2				\n\t"
			"	subs   r6, r2				\n\t"
			"	subs   r5, r2				\n\t"
			"	subs   r4, r2				\n\t"
			"	strh   r7, [r0, #0x06]		\n\t"
			"	strh   r6, [r0, #0x04]		\n\t"
			"	strh   r5, [r0, #0x02]		\n\t"
			"	strh   r4, [r0, #0x00]		\n\t"
			"	adds   r0, #0x08			\n\t"
			"	subs   r3, #1				\n\t"
			"	bne    1b					\n\t"
			"	pop    {r4-r7}				\n\t"
			"	bx     lr					\n\t"
			:
			:
			:"memory","cc"
		);
	
	#elif !defined(HAVE_v7E_SUPPORT)
	
		asm volatile(
			"	lsrs   r3, #2				\n\t"	//we process 4 at a time
			"	push   {r4-r7}				\n\t"
			"1:								\n\t"
			"	ldrh   r7, [r1, #0x06]		\n\t"
			"	ldrh   r6, [r1, #0x04]		\n\t"
			"	ldrh   r5, [r1, #0x02]		\n\t"
			"	ldrh   r4, [r1] ,#0x08		\n\t"
			"	subs   r7, r2				\n\t"
			"	subs   r6, r2				\n\t"
			"	subs   r5, r2				\n\t"
			"	subs   r4, r2				\n\t"
			"	strh   r7, [r0, #0x06]		\n\t"
			"	strh   r6, [r0, #0x04]		\n\t"
			"	strh   r5, [r0, #0x02]		\n\t"
			"	strh   r4, [r0], #0x08		\n\t"
			"	subs   r3, #1				\n\t"
			"	bne    1b					\n\t"
			"	pop    {r4-r7}				\n\t"
			"	bx     lr					\n\t"
			:
			:
			:"memory","cc"
		);
	
	#else
	
		asm volatile(
			"	lsrs   r3, #3				\n\t"	//we process 8 at a time
			"	push   {r4-r7}				\n\t"
			"	orr    r2, r2, r2, lsl #16	\n\t"
			"1:								\n\t"
			"	ldmia  r1!, {r4-r7}			\n\t"
			"	ssub16 r4, r4, r2			\n\t"
			"	ssub16 r5, r5, r2			\n\t"
			"	ssub16 r6, r6, r2			\n\t"
			"	ssub16 r7, r7, r2			\n\t"
			"	stmia  r0!, {r4-r7}			\n\t"
			"	subs   r3, #1				\n\t"
			"	bne    1b					\n\t"
			"	pop    {r4-r7}				\n\t"
			"	bx     lr					\n\t"
			:
			:
			:"memory","cc"
		);
	
	#endif
}


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif


static void audioMicHwSamplesReady(const uint16_t *samples, uint_fast16_t midpoint)
{
	bool post = true;
	
	if (!mNumBufsFree) {
		
		if (!mReadingBuf) {	//reader is not in the middle of a buffer and we can quietly move it
			
			if (++mBufR == NUM_MIC_BUFFERS)
				mBufR = 0;
			
			post = false;	//but do not post on the sem
		
			logt("mic soft overrun\n");
		}
		else {
		
			logt("mic overrun\n");
			return;
		}
	}
	audioMicPrvCopyToBuf(mBuffers[mBufW++], samples, midpoint, mNumSamplesPerHwBuf);
	if (mBufW == NUM_MIC_BUFFERS)
		mBufW = 0;
	if (post) {
		
		mNumBufsFree--;
		(void)KALSemaphoreSignal(mBufferSem);
	}
}

bool audioInInit(void)
{
	if (!audioInHwInit(&audioMicHwSamplesReady, &mNumSamplesPerHwBuf)) {
		loge("Audio.in: hw init\n");
		return false;
	}
	
	return true;
}

bool audioMicSetup(enum AudioSampleRate rate, uint32_t samplesPerBuf, enum AudioSampleType sampTyp, enum AudioChannelConfig chCfg)
{
	enum EmitStatus now;
	struct EmitBuf buf;
	uint_fast8_t i;
	
	mBufW = 0;
	mBufR = NUM_MIC_BUFFERS - 1;
	mNumBufsFree = 0;
	mInBufReadPos = mNumSamplesPerHwBuf;	//so it gets a new one!
	mSampSzShift = 0;
	mReadingBuf = false;
	
	#ifdef ABSOLUTELY_NO_AUDIO_SUPORT
		return false;
	#endif
	
	switch (sampTyp) {
		case AudioSampleU8:
		case AudioSampleS8:
			break;
		case AudioSampleU16LE:
		case AudioSampleU16BE:
		case AudioSampleS16LE:
		case AudioSampleS16BE:
			mSampSzShift++;
			break;
		case AudioSampleU32LE:
		case AudioSampleU32BE:
		case AudioSampleS32LE:
		case AudioSampleS32BE:
		case AudioSampleFloatLE:
		case AudioSampleFloatBE:
			mSampSzShift += 2;
			break;
		default:
			loge("Audio.in: invalid sample type %d\n", sampTyp);
			return false;
	}
	switch (chCfg) {
		case AudioMono:
			break;
		case AudioStereo:
			mSampSzShift++;
			break;
		default:
			loge("Audio.in: invalid sample ch cfg %d\n", chCfg);
			return false;
	}
	
	if (errNone != KALSemaphoreCreate(&mBufferSem, CREATE_4CC('a','M','I','C'), 0)) {
		loge("Audio.in: sem\n");
		goto out_sem;
	}
	
	mConvertFunc = (uint16_t*)kheapAllocEx(IN_CONVERT_FUNC_BYTES, MEM_USABLE_FOR_EXEC | MEM_FAST);
	if (!mConvertFunc) {
		loge("Audio.in: cvt f alloc\n");
		goto out_cvt_func;
	}
	
	for (i = 0; i < NUM_MIC_BUFFERS; i++)
		mBuffers[i] = NULL;
	
	for (i = 0; i < NUM_MIC_BUFFERS; i++) {
		
		mBuffers[i] = kheapAlloc(sizeof(**mBuffers) * mNumSamplesPerHwBuf);
		if (!mBuffers[i]) {
			loge("Audio.in: buf %u alloc\n", i);
			goto out_alloc;
		}
	}

	//create convert func
	emitBufferInit(&buf, mConvertFunc, IN_CONVERT_FUNC_BYTES);
	now = audioPrvMicCreateConvertFunc(&buf, sampTyp, chCfg);
	if (now != EmitErrNone) {
		loge("Audio.in: cvt f create: %d\n", now);
		goto out_create_cvt;
	}
	
	emitBufCacheCoherencyManage(&buf);
	
	//success, turn it on
	if (!audioInHwSetState(true, rate)) {
		loge("Audio.in: start\n");
		goto out_audio_start;
	}
	
	//give it buffers to fill (except one is "being used" by us)
	mNumBufsFree = NUM_MIC_BUFFERS;
	
	return true;

out_audio_start:
	//nothing

out_create_cvt:
	//nothing

out_alloc:
	for (i = 0; i < NUM_MIC_BUFFERS; i++) {
		
		if (mBuffers[i])
			kheapFree(mBuffers[i]);
		mBuffers[i] = NULL;
	}
	kheapFree(mConvertFunc);
	mConvertFunc = NULL;

out_cvt_func:
	(void)KALSemaphoreDelete(mBufferSem);
	mBufferSem = 0;

out_sem:
	return false;
}

void audioMicStop(void)
{
	uint_fast8_t i;
	
	audioInHwSetState(false, 0);
	
	for (i = 0; i < NUM_MIC_BUFFERS; i++) {
		
		if (mBuffers[i])
			kheapFree(mBuffers[i]);
		mBuffers[i] = NULL;
	}
	
	kheapFree(mConvertFunc);
	mConvertFunc = NULL;
	
	(void)KALSemaphoreDelete(mBufferSem);
	mBufferSem = 0;
}


//return bytes provided
uint32_t audioMicGetSamples(void* dst, uint32_t size, uint32_t volL, uint32_t volR)
{
	uint32_t nSampDesired;		//always a power of two so we store Lg2 of it
	
	nSampDesired = size >> mSampSzShift;
	
	#ifdef ABSOLUTELY_NO_AUDIO_SUPORT
		return 0;
	#endif
	
	while (nSampDesired) {
		
		uint32_t nSampAvail = mNumSamplesPerHwBuf - mInBufReadPos, nSampNow = nSampDesired;
		const int16_t *src;
		
		if (nSampAvail)
			src = mBuffers[mBufR] + mInBufReadPos;
		else {
			
			(void)KALSemaphoreWait(mBufferSem, -1);
			if (++mBufR == NUM_MIC_BUFFERS)
				mBufR = 0;
			mReadingBuf = 1;
			mInBufReadPos = 0;
			src = mBuffers[mBufR];
			nSampAvail = mNumSamplesPerHwBuf;
		}
		if (nSampNow > nSampAvail)
			nSampNow = nSampAvail;
		
		if (nSampNow & 1)
			fatal("mic samples must always be a multiple of 4\n");
		
		dst = ((MicCvtF)emitBufPtrToFuncPtr(mConvertFunc))(dst, src, volL, volR, nSampNow / 2);
		nSampDesired -= nSampNow;
		mInBufReadPos += nSampNow;
		
		if (mInBufReadPos == mNumSamplesPerHwBuf) { 	//we finished a buffer
			
			mReadingBuf = 0;
			atomicAdd8(&mNumBufsFree, 1);
		}
	}
	
	return size;
}
