#include "util.h"

//palm structs
struct PalmFontCharInfo {
	int8_t offset; // unused
	int8_t width; // negative one if char is missing
};

#define FNT_TAB_WIDTH		20

//our structs
struct FntWidthAndOffsetData {
	int16_t maxWidth, finalWidth;
	int8_t thisCharWidth;
};

struct FntWordWrapData {
	int16_t resultWidth, maxWidth;
	int8_t lastCharLen;
};

static int32_t fntPrvScaleFontValueForDensity(int32_t val)
{
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	
	if (w) {
	
		uint32_t scaleFactor;
		
		scaleFactor = w->drawState->flags.unscaledText ? w->drawState->scaling.preGarnet.nativeToActive : w->drawState->scaling.preGarnet.standardToActive;
		val = scaleCoord(val, scaleFactor, true);
	}
	
	return val;
}

static int32_t fntPrvScaleFontValueForDensityComplex(int32_t val, bool unscale)
{
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint32_t dispDensity;

	if (!w)
		return val;
	
	dispDensity = BmpGetDensity(WinGetBitmap(WinGetDisplayWindow()));
	if (dispDensity == kDensityLow)
		return val;
	
	if (ds->flags.unscaledText) {
		
		int64_t t;
		
		if (unscale)
			return val;

		t = (int64_t)(int32_t)val * (int64_t)(int32_t)scaleAnalize(ds->scaling.preGarnet.nativeToActive);
		t += (BIG_SCALE_UNITY * 2 + 1) / 3;
		return t / BIG_SCALE_UNITY;
	}
	else if (dispDensity == kDensityOneAndAHalf) {
		
		if (ds->flags.unpaddedText) {
			
			if (unscale) {
				
				if (SCALE_UNITY == ds->scaling.preGarnet.activeToNative)
					return WinScaleCoord(val, true);
				
				return scaleCoord(val, ds->scaling.preGarnet.activeToNative, true);
			}
			else {
				
				if (SCALE_UNITY == ds->scaling.preGarnet.activeToNative)
					return val;
				
				return scaleCoord(val, ds->scaling.preGarnet.nativeToActive, false);
			}
		}
		else {
			
			if (SCALE_UNITY == ds->scaling.preGarnet.activeToNative) {
				
				if (!unscale)
					return val;
				
				return WinScaleCoord(val, true);
			}
		}
	}
	else if (!unscale)
		return val;

	return WinScaleCoord(val, false);
}
static const struct PalmFontCharInfo* getFontCharsInfo(struct PalmFont *fnt)
{
	struct PalmFontCharInfo *ret = (struct PalmFontCharInfo*)&fnt->metrics.owTLoc;
	
	return ret + fnt->metrics.owTLoc;
}

static int32_t fntPrvGetFontMapCharsWidth(struct PalmFont *fnt, const uint8_t* chs, int8_t *charLengthP)
{
	struct PalmFont **fonts = getFontTableForFontMap(fnt->metrics.fontType);	//i kid you not
	struct PalmFontMap *map = (struct PalmFontMap*)fnt;							//yup, i still kid you not
	const struct PalmFontCharInfo *charWidthInfo;
	uint32_t charLen = 1, ch;
	int32_t width;
	
	ch= *chs;
	fnt = fonts[map->entries[ch].fontIdx];	//get font for this char
	charWidthInfo = getFontCharsInfo(fnt);
	if (map->entries[ch].unk_0x01 == 2) {
		chs++;
		charLen++;
	}
	ch = *chs;
	
	*charLengthP = charLen;
	if (ch >= fnt->metrics.firstChar && ch <= fnt->metrics.lastChar) {
		width = charWidthInfo[ch - fnt->metrics.firstChar].width;
		if (width != -1)
			return width;
	}
	
	width = charWidthInfo[fnt->metrics.lastChar + 1 - fnt->metrics.firstChar].width;
	if (charLen == 2 && !(TxtByteAttr(ch) & byteAttrLast))
		*charLengthP = 1;
	
	return width;
}

static int16_t pFntBaseLine(void)
{
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	
	return fntPrvScaleFontValueForDensity(fnt->metrics.ascent);
}
DEF_BOOT_PATCH(pFntBaseLine, 0x3E0);

static int16_t pFntCharHeight(void)
{
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	
	return fntPrvScaleFontValueForDensity(fnt->metrics.fRectHeight);
}
DEF_BOOT_PATCH(pFntCharHeight, 0x3E4);

static int16_t pFntLineHeight(void)
{
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	
	return fntPrvScaleFontValueForDensity(fnt->metrics.fRectHeight) + fntPrvScaleFontValueForDensity(fnt->metrics.leading);
}
DEF_BOOT_PATCH(pFntLineHeight, 0x408);

static int16_t pFntAverageCharWidth(void)
{
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	
	return fntPrvScaleFontValueForDensity(fnt->metrics.fRectWidth);
}
DEF_BOOT_PATCH(pFntAverageCharWidth, 0x3DC);

static int16_t pFntDescenderHeight(void)
{
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	
	return fntPrvScaleFontValueForDensity(fnt->metrics.descent);
}
DEF_BOOT_PATCH(pFntDescenderHeight, 0x3F8);

static int16_t pFntCharWidth(uint8_t ch)
{
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	int32_t ret = -1;
	
	if ((fnt->metrics.fontType & FONT_MAP_MASK) == FONT_MAP_MASK) {
		
		int8_t charLen;
		uint8_t chs[2] = {ch, 0};
		
		ret = fntPrvGetFontMapCharsWidth(fnt, chs, &charLen);
	}
	else {
		
		const struct PalmFontCharInfo *charWidthInfo = getFontCharsInfo(fnt);
		
		if (ch >= fnt->metrics.firstChar && ch <= fnt->metrics.lastChar)
			ret = charWidthInfo[ch - fnt->metrics.firstChar].width;
		
		if (ret == -1)	//char missing
			ret = charWidthInfo[fnt->metrics.lastChar + 1 - fnt->metrics.firstChar].width;
	}
	
	return fntPrvScaleFontValueForDensity(ret);
}
DEF_BOOT_PATCH(pFntCharWidth, 0x3F0);

static int16_t pFntWCharWidth(WChar wch)
{
	char str[8];
	
	return FntCharsWidth(str, TxtSetNextChar(str, 0, wch));
}
DEF_BOOT_PATCH(pFntWCharWidth, 0x414);

//iter funcs returns false to cancel iteration, this func returns final offset value
//width adjust func returns new width to use
//forcing this to be inlined also inlines callbacks, creating nice templated funcs for us without complex macros & C++
static inline __attribute__((always_inline)) int32_t fntPrvCharsForeach(const uint8_t *chrs, int32_t length, bool (*preIterF)(const uint8_t *chr, int32_t screenWidthSoFar, void* userData), int32_t (*widthAdjustF)(const uint8_t *chr, int8_t chrLen, int32_t propsedRawWidth, int32_t widthSoFarUnscaled, void* userData), bool (*postIterF)(const uint8_t *chr, int8_t chrLen, int32_t chrWidthRaw, int32_t screenWidthSoFar, void* userData), void* userData)
{
	int32_t ofstSoFar = 0, widthSoFar = 0, scaledFinalWidth = 0, thisWidth;
	struct PalmFont *fnt = (struct PalmFont*)FntGetFontPtr();
	const uint8_t *chrStart;
	uint32_t ch;
	
	//first, find the spot
	if ((fnt->metrics.fontType & FONT_MAP_MASK) == FONT_MAP_MASK) {
	
		while (ofstSoFar < length) {
			
			int8_t charLen;
			
			chrStart = &chrs[ofstSoFar];
			
			if (preIterF && !preIterF(chrStart, scaledFinalWidth, userData))
				break;
			
			thisWidth = fntPrvGetFontMapCharsWidth(fnt, chrStart, &charLen);
			if (widthAdjustF)
				thisWidth = widthAdjustF(chrStart, charLen, thisWidth, widthSoFar, userData);
			widthSoFar += fntPrvScaleFontValueForDensityComplex(thisWidth, true);
			scaledFinalWidth = fntPrvScaleFontValueForDensityComplex(widthSoFar, false);
			
			if (postIterF && !postIterF(chrStart, charLen, thisWidth, scaledFinalWidth, userData))
				break;
			
			ofstSoFar += charLen;
		}
	}
	else {
		
		uint32_t firstChar = fnt->metrics.firstChar, lastChar = fnt->metrics.lastChar;
		const struct PalmFontCharInfo *charWidthInfo = getFontCharsInfo(fnt);
		
		while (ofstSoFar < length) {
			
			chrStart = &chrs[ofstSoFar];
			ch = *chrStart;
			
			if (preIterF && !preIterF(chrStart, scaledFinalWidth, userData))
				break;
			
			thisWidth = -1;
			if (ch >= firstChar && ch <= lastChar)
				thisWidth = charWidthInfo[ch - firstChar].width;
			
			if (thisWidth == -1)
				thisWidth = charWidthInfo[lastChar - firstChar + 1].width;
			
			if (widthAdjustF)
				thisWidth = widthAdjustF(chrStart, 1, thisWidth, widthSoFar, userData);
			widthSoFar += fntPrvScaleFontValueForDensityComplex(thisWidth, true);
			scaledFinalWidth = fntPrvScaleFontValueForDensityComplex(widthSoFar, false);
			
			if (postIterF && !postIterF(chrStart, 1, thisWidth, scaledFinalWidth, userData))
				break;
			
			ofstSoFar++;
		}
	}
	
	return ofstSoFar;
}

static bool fntPrvJustStoreWidthPostIter(const uint8_t *chr, int8_t chrLen, int32_t chrWidthRaw, int32_t screenWidthSoFar, void* userData)
{
	*(int32_t*)userData = screenWidthSoFar;
	
	return true;
}

static int16_t pFntCharsWidth(const uint8_t *chrs, int16_t len)
{
	int32_t resultWidth = 0;
	
	fntPrvCharsForeach(chrs, len, NULL, NULL, &fntPrvJustStoreWidthPostIter, &resultWidth);

	return resultWidth;
}
DEF_BOOT_PATCH(pFntCharsWidth, 0x3EC);

static bool fntPrvFntWidthToOffsetCharPostIter(const uint8_t *chr, int8_t chrLen, int32_t chrWidthRaw, int32_t screenWidthSoFar, void* userData)
{
	struct FntWidthAndOffsetData *d = (struct FntWidthAndOffsetData*)userData;
	
	d->finalWidth = screenWidthSoFar;
	d->thisCharWidth = chrWidthRaw;
	
	return screenWidthSoFar <= d->maxWidth;
}

static int16_t pFntWidthToOffset(const uint8_t *chrs, uint16_t length, int16_t pixelWidth, bool *leadingEdgeP, int16_t *truncWidthP)
{
	struct FntWidthAndOffsetData d = {.finalWidth = 0, .maxWidth = pixelWidth};
	int32_t finalOfst;
	
	finalOfst = fntPrvCharsForeach(chrs, length, NULL, NULL, &fntPrvFntWidthToOffsetCharPostIter, &d);

	if (finalOfst >= length) {	//we used the entire string
		if (leadingEdgeP)
			*leadingEdgeP = true;
		
		if (truncWidthP)
			*truncWidthP = d.finalWidth;
		
		return length;
	}
	else {
		
		if (leadingEdgeP)
			*leadingEdgeP = d.thisCharWidth / 2 < d.finalWidth - pixelWidth;
		
		if (truncWidthP)
			*truncWidthP = d.finalWidth - d.thisCharWidth;
		
		return finalOfst;
	}
}
DEF_BOOT_PATCH(pFntWidthToOffset, 0x418);

static bool fntPrvStopOnNullOrLinefeedPreIter(const uint8_t *chr, int32_t screenWidthSoFar, void* userData)
{
	uint8_t ch = *chr;
	
	return ch != 0 && ch != '\n';
}

static bool fntPrvFntCharsInWidthPostIter(const uint8_t *chr, int8_t chrLen, int32_t chrWidthRaw, int32_t screenWidthSoFar, void* userData)
{
	struct FntWidthAndOffsetData *d = (struct FntWidthAndOffsetData*)userData;
	
	d->finalWidth = screenWidthSoFar;
	d->thisCharWidth = chrWidthRaw;
	
	return screenWidthSoFar <= d->maxWidth;
}

static void pFntCharsInWidth(const uint8_t *chrs, int16_t *widthP, int16_t *lengthP, bool *fitsP)
{
	struct FntWidthAndOffsetData d = {.finalWidth = 0, .maxWidth = *widthP};
	int32_t finalOfst, t, strLen = *lengthP, width;
	
	finalOfst = fntPrvCharsForeach(chrs, *lengthP, &fntPrvStopOnNullOrLinefeedPreIter, NULL, &fntPrvFntCharsInWidthPostIter, &d);
	width = d.finalWidth;
	
	//did it fit?
	if (finalOfst >= strLen || chrs[finalOfst] == 0)
		*fitsP = true;
	else if (chrs[finalOfst] == '\n')
		*fitsP = false;
	else {
		
		//skip nondrawable chars as they do not count really and see if there is more to draw or sting end in there
		for (t = finalOfst; t < strLen && (chrs[t] == ' ' || chrs[t] == '\t'); t++);
		
		*fitsP = t >= strLen || chrs[t] == 0;
	}
	
	//if the last char we drew is invisible, remove it from the calculation
	if (finalOfst && (chrs[finalOfst - 1] == ' ' || chrs[finalOfst - 1] == '\n' || chrs[finalOfst - 1] == '\t')) {
		width -= FntCharWidth(chrs[finalOfst - 1]);
		finalOfst--;
	}
	
	//report
	*widthP = width;
	*lengthP = finalOfst;
}
DEF_BOOT_PATCH(pFntCharsInWidth, 0x3E8);

static int32_t fntPrvFntTabProcessingWidthAdjust(const uint8_t *chr, int8_t chrLen, int32_t propsedRawWidth, int32_t widthSoFarUnscaled, void* userData)
{
	if (*chr == '\t')
		return FNT_TAB_WIDTH - widthSoFarUnscaled % FNT_TAB_WIDTH;
	else
		return propsedRawWidth;
}

//same as FntCharsWidth, but accounts for tabs assuming first char is first in the line of text (tabs are position-based)
static int16_t pFntLineWidth(const uint8_t* chrs, uint16_t length)
{
	int32_t resultWidth = 0;
	
	fntPrvCharsForeach(chrs, length, NULL, &fntPrvFntTabProcessingWidthAdjust, &fntPrvJustStoreWidthPostIter, &resultWidth);

	return resultWidth;
}
DEF_BOOT_PATCH(pFntLineWidth, 0x40C);

static bool fntPrvFntWordWrapPostIter(const uint8_t *chr, int8_t chrLen, int32_t chrWidthRaw, int32_t screenWidthSoFar, void* userData)
{
	struct FntWordWrapData *d = (struct FntWordWrapData*)userData;
	
	d->resultWidth = screenWidthSoFar;
	d->lastCharLen = chrLen;
	
	return screenWidthSoFar <= d->maxWidth;
}

static uint16_t pFntWordWrap(const uint8_t* chrs, uint16_t maxWidth)
{
	struct FntWordWrapData d = {.maxWidth = maxWidth, };
	int32_t ofst;
	
	if (!maxWidth)
		return 0;
	
	ofst = fntPrvCharsForeach(chrs, 0x7fffffff, &fntPrvStopOnNullOrLinefeedPreIter, &fntPrvFntTabProcessingWidthAdjust, &fntPrvFntWordWrapPostIter, &d);
	//we'll only stop on null, linefeed, or over-width event
	//in case of null or linefeed, ofst points to that char that stopped us
	//if neither did, then ofst points to the last char that is entirely "offscreen"
	// but due to the above, never past a NULL, so it is safe to access
	
	//special-case terminations
	if (!chrs[ofst])
		return ofst;		//returned len does not include null-terminator
	else if (chrs[ofst] == '\n')
		return ofst + 1;	//returned len does include newline
	
	//if we went over our allowed width, backtrack one char, but never do so if we'd then try to claim we can write zero chars
	if (d.resultWidth > maxWidth && ofst > d.lastCharLen)
		ofst -= d.lastCharLen;
	
	return TxtGetWordWrapOffset((char*)chrs, ofst);
}
DEF_BOOT_PATCH(pFntWordWrap, 0x41C);
