/* MathEng.c, the math engine for LCARS 24 Math Service

	First load input to the string MathInput[1], then call the
	function LcMath(). Output goes to the string MathOutput.
	If there is a problem, output first goes to ErrMessage but
	is then copied to MathOutput for easy programming of the
	side of the calling interface.

	Adapted from a program whose history is described below
*/

/*
* By Dale Holt	 3945 Vondelpark Place, Colorado Springs, CO 80907
* AmigaBasic. November 1986, March 1991
* Conversion to C.  May 1993
* Fixed fmtcomma.  23 Oct 1993  (memmove ...l -> l+1)
* Fixed checks for 'e' and 'E' for scientific notation.  7 Mar 1994
* Added pi as a constant.  2 Sept 1994
* Fixed unary - and + after =.  6 Feb 1995
* Further unary - and + checks.  23 Aug 1995
* Remove spaces, better unary - check.  6 Oct 1997
* Fix handling of [,],{,} and remove terminating decimal point.  20 May 1999
* Fix handling of [,],{,} and add option for no comma in output.  7 Jul 1999
* Implement angles in degrees instead of radians.  22 Jan 2004
*/
/* This program is free software. You may redistribute and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License or
 * (at your option) any later version.
 *
 * Refer to the file C:\LCARS24\DATA\COPYING.TXT for details.
 */

#include "lcars24.h"
#include "matheng.h"

int MathEngine()
{
	int NumArgs=2;

char
	*infunct,
	var,
	value[80];

int
	*postfix,
	stack_size,
	numops,
	j,
	maxlen=0,
	iter_ct,
	iter_total,
	solve=0,
	repeat=0,
	use_x=0,
	use_y=0,
	use_z=0,
	use_equal=0,
	use_from=0,
		  do_comma=1;

double
	*constant,
	*var_ptr,
	x=0,
	y=0,
	z=0,
	var_inc,
	err,
	err_sign,
	prev_err,
	prev_err_sign,
	var_inc_mult,
	answer,
	search_start = 1.0,
	range_min,
	range_max;

/* format the output with commas? (not relevant to LCARS usage) */
	if (!strcmp(MathInput[1], "mathnc")) do_comma = 0;

/* determine the length of the input algebraic expression */
	j = 1;
	while (j < NumArgs) {
		if (!strcasecmp(MathInput[j],"from")) {
			if (NumArgs != j+2) {
				strcpy(MathOutput,"math: Format error.  There should be one number after 'from.'");
				goto ListInstruct;
			}
			use_from = 1;
			search_start = atof(MathInput[++j]);
			NumArgs = j-1;
		} else maxlen += strlen(MathInput[j++]);
	}

/* is there anything to calculate? */

	if ( NumArgs == 1 || ( NumArgs == 2 && !strcmp(MathInput[1],"-help") ) ) {
ListInstruct:
		err = 1.0 DEGREES;
		if (err < 0.5) strcpy(ErrMessage,"The trigonometric functions operate with degrees, not radians.");
			else strcpy(ErrMessage,"The trigonometric functions operate with radians, not degrees.");
		return(-1);
	}
	if ( (infunct = (char *)malloc(maxlen+4+NumArgs)) == NULL ) {
		strcpy(ErrMessage,"\aCould not allocate memory!");
		return(-1);
	}

	infunct[0] = '\0';
	for (j=1; j<NumArgs; j++) {
		strcat(infunct,MeStrTrim(MathInput[j]));
		strcat(infunct," ");		  /* for error checking of separate numbers */
	}

	j = strlen(infunct)+5;

	if ( (postfix = (int *)malloc(j*sizeof(int))) == NULL ) {
		sprintf(ErrMessage,"Could not allocate %d bytes of memory!",j*sizeof(int));
		free(infunct);
		return(-1);
	}

	if ( (constant = (double *)malloc(j*sizeof(double)/2)) == NULL ) {
		free(infunct);
		free(postfix);
		sprintf(ErrMessage,"Could not allocate %d bytes of memory!",j*sizeof(double)/2);
		return(-1);
	}

/*
Send function. Gets array of postfix operators and constants.
Returns number of PostFix operations. Negative is error.
*/

numops = MeCompile(infunct, postfix, constant,
				&use_x, &use_y, &use_z, &use_equal, &stack_size);
if (numops>0)
	 {
	 if ( use_equal && (use_x || use_y || use_z) )
	{
	if (use_x + use_y + use_z > 1)
		 {
		 strcpy(ErrMessage,"Can only solve equations with one variable.");
		 return(-1);
		 }
	solve = 1;
	}
	 else
	 if ( use_equal && !(use_x || use_y || use_z) )
	{
	strcpy(ErrMessage,"Please don't use the equal sign.");
	return(-1);
	}
	 else
	if (use_x || use_y || use_z) {
		strcpy(ErrMessage,"Please don't use variables.");
		return(-1);
		repeat = 1;
	}

	 if (use_from && !solve)
	strcpy(ErrMessage,"Ignoring from value.");

	 if (solve)
	{
	if (use_x) { var = 'x'; var_ptr = &x; }
	if (use_y) { var = 'y'; var_ptr = &y; }
	if (use_z) { var = 'z'; var_ptr = &z; }
	sprintf(ErrMessage,"Solving from %c = %G ...",var,search_start);
	iter_ct = iter_total = 0;
	range_min = range_max = *var_ptr = search_start;
	var_inc = *var_ptr / 100.0;
	var_inc_mult = 10.0;	 /* only changes sign */
	prev_err = err = MeEvaluate(postfix,constant,numops,x,y,z,stack_size);
	prev_err_sign = (prev_err >= 0 ? 1 : -1);

	while ( err != 0 && iter_total < 1000 && fabs(var_inc) > 1.0e-30)
		 {
		 *var_ptr += var_inc;
		 range_min = min(range_min,*var_ptr);
		 range_max = max(range_max,*var_ptr);
		 iter_ct++;
		 iter_total++;
		 prev_err = err;
		 prev_err_sign = err_sign;
		 err = MeEvaluate(postfix,constant,numops,x,y,z,stack_size);
		 if (err == 0) break;
		 err_sign = (err >= 0 ? 1 : -1);
		 if ( (err_sign == prev_err_sign) &&
		 (err_sign * err) < (prev_err_sign * prev_err) )
		{		/* keep going this direction */
		if (++iter_ct > 2 * var_inc_mult)	/* need bigger increment */
			 {
			 var_inc *= var_inc_mult;
			 iter_ct = 0;
			 }
		}
		 else	  /* go the other direction */
		{
		var_inc /= (-var_inc_mult);
		iter_ct = 0;
		}
		 }
	sprintf(ErrMessage,"%c = %.10G, with error = %G in %d iterations.",
		var,*var_ptr,err,iter_total);
	sprintf(ErrMessage,"Search range %G to %G",range_min,range_max);
	if (fabs(err) > 0.0001)
		strcpy(ErrMessage,"NOTICE: Solution was probably NOT found!");
	}
	 else
	 if (repeat)
	{
	while (repeat)
		 {
		 if (use_x)
		{
		strcpy(ErrMessage,"  x val? "); /* fflush(stdout); */
/*	 if (!fgets(value,20,stdin)) goto quit; */
		if (!strlen(MeStrTrim(value))) goto quit;
		x = atof(value);
		}
		 if (use_y)
		{
		strcpy(ErrMessage,"  y val? "); /* fflush(stdout); */
/*	 if (!fgets(value,20,stdin)) goto quit; */
		if (!strlen(MeStrTrim(value))) goto quit;
		y = atof(value);
		}
		 if (use_z)
		{
		strcpy(ErrMessage,"  z val? "); /* fflush(stdout); */
/*	 if (!fgets(value,20,stdin)) goto quit; */
		if (!strlen(MeStrTrim(value))) goto quit;
		z = atof(value);
		}
		 answer = MeEvaluate(postfix,constant,numops,x,y,z,stack_size);
		 MeFmtComma(value,answer,do_comma);
		 sprintf(ErrMessage,"result = %s for", value, x, y, z);
		 if (use_x) sprintf(ErrMessage,"  x = %G",x);
		 if (use_y) sprintf(ErrMessage,"  y = %G",y);
		 if (use_z) sprintf(ErrMessage,"  z = %G",z);
/*	  /* fflush(stdout); */
				}
	}
	 else
	{
	answer = MeEvaluate(postfix,constant,numops,x,y,z,stack_size);
	MeFmtComma(value,answer, do_comma);
	sprintf(MathOutput,"%s",value);
	return(0);
	}
	 }

quit:
		  /* free the memory */

	free(infunct);
	free(postfix);
	free(constant);
	return(1);
}

/**************** infix to postfix compiler ***************/

int MeCompile(char *infunct, int *postfix_array, double *constant,
	int *use_x, int *use_y, int *use_z, int *use_equal,
		  int *stack)
{

int
	j,
	k,
	l,
	c,
	no	 = 0,
	yes	= 1,
	EqErr = 0,
	infix = 0,
	funct_offset = 30,
	this_op,
	next_op,
	digit,
	decimal,
	exponent,
	operator,
	in_const,
	num_const,
	Lparen,
	Rparen,
	infixlen,
	err_pos,
	*infptr,
	*pendop,
	*level,
	*infix_array,
		  plevel,			/* parenthesis level */
		  pif,				/* infix string position */
		  ppf,				/* postfix array position */
		  pop,				/* pending operation stack position */
	num_functs=26,
	opriority[30]={0,3,3,2,2,1,1,4,4,4,4,4,4,4,
				 4,4,4,4,4,4,4,4,4,4,4,4,4,4,1},
	opfunctlen[30]={1,2,1,1,1,1,1,4,4,4,4,4,4,3,
			3,3,3,2,3,4,3,3,4,1,1,1,1,1,1};

char
	*ptr1,
	*ptr2,
	pchar,
	nchar,
	tchar,
	*funct,
	tempstr[64],
	opfunct[30][5]={" ", "**", "^", "*", "/",
				  "+", "-", "sinh", "cosh", "tanh",
			"asin", "acos", "atan", "sin", "cos",
			"tan", "log", "ln", "exp", "sqrt",
			"abs", "int", "frac", "x", "y",
			"z", "(", ")", "="},
			 /* 1st must be blank */

	ErrMsg2[80] = "";

	j = strlen(infunct);
	if (!j) {
		strcpy(ErrMessage,"Zero length input string!");
		return(0);
	}
/* printf("Expression: %s",infunct); */

/* allocate memory */

	j++;
	if (j<4) j = 4;
	if ( (funct = (char *)malloc(j)) == NULL ) {
		sprintf(ErrMessage,"Could not allocate %d bytes of memory!",j);
		return(-1);
	}
	if ( (infix_array = (int *)malloc(j*sizeof(int))) == NULL ) {
		sprintf(ErrMessage,"Could not allocate %d bytes of memory!",j*sizeof(int));
		return(-1);
	}
	if ( (pendop = (int *)malloc(j*sizeof(int)/2)) == NULL ) {
		sprintf(ErrMessage,"Could not allocate %d bytes of memory!",j*sizeof(int)/2);
		return(-1);
	}
	if ( (level = (int *)malloc(j*sizeof(int)/2)) == NULL )  {
		sprintf(ErrMessage,"Could not allocate %d bytes of memory!",j*sizeof(int)/2);
		return(-1);
	}

/* remove spaces, tabs, commas, line feeds, returns */
/* convert [,],{,} to (,)									*/

	ptr1 = ptr2 = infunct;
	while (*ptr2) {
		if (*ptr2 == '[') *ptr2 = '(';
		else
		if (*ptr2 == '{') *ptr2 = '(';
		else
		if (*ptr2 == ']') *ptr2 = ')';
		else
		if (*ptr2 == '}') *ptr2 = ')';
		else
		if (*ptr2 == '\t') *ptr2 = ' ';

		if (*ptr2 != ' ' && *ptr2 != '\n' && *ptr2 != '\r' && *ptr2 != ',')
			*ptr1++ = *ptr2;
		ptr2++;
	}
	*ptr1 = '\0';
	if (ptr1 == infunct) return 0;		/* all blanks */

/* check that parentheses match */

Lparen = Rparen = 0;
k = strlen(infunct);
for (j=0; j<k; j++)
	{
	if (infunct[j] == '(') Lparen++;
	else
	if (infunct[j] == ')')
		 {
		 if (++Rparen > Lparen)
		{
		strcpy(ErrMsg2,"Unmatched right parenthesis.");
		strcpy(funct,infunct);
		ptr1 = funct + j;
		goto CompileErr;
		}
		 }
	else
	if (infunct[j] == '=')
		 {
		 if ((*use_equal)++)
		{
		strcpy(ErrMsg2,"Second equal sign.");
		strcpy(funct,infunct);
		ptr1 = funct + j;
		goto CompileErr;
		}
		 if (Lparen != Rparen)
		{
		strcpy(ErrMsg2,"Unmatched left parenthesis at equal sign.");
		strcpy(funct,infunct);
		ptr1 = funct + j;
		goto CompileErr;
		}
		 }
	}
	if (Lparen != Rparen) {
		strcpy(ErrMsg2,"Unmatched left parenthesis.");
		strcpy(funct,infunct);
		ptr1 = funct + j;
		goto CompileErr;
	}

/* check for unary + and - signs */

	 ptr1 = infunct;
	 ptr2 = funct;
	 pchar = '(';
	 operator = 0;
	 digit = 0;
	 while (tchar = *ptr1)
	{
	if ( (nchar = *(ptr1+1)) == '.' || isdigit(nchar) )
		 digit = yes; else digit = no;
	switch (pchar)
		 {
		 case '=':
		 case '>':
		 case '<':
		 case '^':
		 case '*':
		 case '/':
		operator = 1;
		break;
		 default:
		operator = 0;
		 }
	switch (tchar)
		 {
		 case '~':	  /* my special character */
		strcpy(ErrMsg2,"Invalid character: ~");
		goto CompileErr;

		 case '+':
		if ( (operator && digit) ||
			  ( (pchar == '(') || ( (pchar == 'e' || pchar == 'E') && digit) ) )
			 tchar = '\0';
		break;

		 case '-':
		if ( (operator && digit) ||
			  ( (pchar == '(') || ( (pchar == 'e' || pchar == 'E') && digit) ) )
			 tchar = '~';	  /* signal negative constant */
		if ( (pchar == '(' || pchar == '=') && (!digit) )
			 {
			 *ptr2++ = '0';
			 tchar = '-';
			 }
		break;

		 default:
		break;
		 }
	if (tchar) *ptr2++ = tchar;
	pchar = tchar;
	ptr1++;
	}
*ptr2 = '\0';

/* convert to tokens */

ptr1 = funct;
infptr = infix_array;
num_const = 0;
digit = 0;
exponent = 0;
operator = 0;

while (*ptr1)
	{
	/* skip spaces */
	if (*ptr1 == ' ') goto next_char;

	/* is this a defined operation or function? */
	for (j=1; j<num_functs+3; j++)
		{
		if (!strncasecmp(ptr1,opfunct[j],opfunctlen[j]))
			{
			*infptr++ = j;
			ptr1 += (opfunctlen[j] - 1);
			if (j == 23) *use_x = yes;
			if (j == 24) *use_y = yes;
			if (j == 25) *use_z = yes;
			goto next_char;
			}
		}

	if (!strncasecmp(ptr1, "pi", 2)) {
		constant[++num_const] = M_PI;
		*infptr++ = funct_offset + num_const;
		ptr1++;
		goto next_char;
	}

	/* then it should be a constant */
	for (j=0; j<64; j++) tempstr[j] = '\0';
	in_const = yes;
	decimal = exponent = no;
	while (in_const)
		{
		tchar = *ptr1;
		if (tchar == '.')
			{
			if (decimal)
				{
				strcpy(ErrMsg2,"Second decimal point in constant.");
				goto CompileErr;
				}
			if (exponent)
				{
				strcpy(ErrMsg2,"Decimal point in exponent.");
				goto CompileErr;
				}
			decimal = yes;
			}
		else
		if (tchar == 'e' || tchar == 'E')
			{
			if (exponent)
				{
				strcpy(ErrMsg2,"Exponent specified twice.");
				goto CompileErr;
				}
			if (strlen(tempstr) == 0
				 || (strlen(tempstr) == 1 && tempstr[0] == '.'))
				{
				strcpy(ErrMsg2,"Invalid constant.");
				goto CompileErr;
				}
			exponent = yes;
			}
		else
		if (isdigit(tchar))
			{
			;
			}
		else
		if (tchar == '~')
			{
			if (strlen(tempstr) == 0
				 || tempstr[strlen(tempstr)-1] == 'e'
				 || tempstr[strlen(tempstr)-1] == 'E')
				tchar = '-';
			else
				{
				strcpy(ErrMsg2,"Misplaced unary minus sign.");
				goto CompileErr;
				}
			}
		else
			{
			in_const = no;
			if (strlen(tempstr) == 0)
				{
				strcpy(ErrMsg2,"Invalid character or function.");
				goto CompileErr;
				}
			if (strlen(tempstr) == 1 && !isdigit(*tempstr))
				{
				strcpy(ErrMsg2,"Unconnected decimal point.");
				ptr1--;
				goto CompileErr;
				}
			ptr1--;	/* save this character */
			}

		if (in_const)
			{
			tempstr[strlen(tempstr)] = tchar;
			ptr1++;
			}
		}
	tchar = tempstr[strlen(tempstr)-1];
	if (!isdigit(tchar) && tchar != '.')
		{
		strcpy(ErrMsg2,"Constant did not end with a digit.");
		goto CompileErr;
		}
	constant[++num_const] = strtod(tempstr,NULL);
	*infptr++ = funct_offset + num_const;
	decimal = exponent = no;

  next_char:
	++ptr1;
	*infptr = 0;
	}
infixlen = (int)(infptr - infix_array);
infix = yes;

/* check that the math rules are obeyed */

/************************************************************

RuleChecking:
1. Constants and variables must be followed by ) or = or operator.
2. Operator cannot be followed by operator or ) or =.
3. Function must be followed by (.
4. ( cannot be followed by operator or ) or =.
5. ) must be followed by operator or ) or =.
6. Expression must end with an constant, variable, or ).
Variables: X, Y, Z		--- not used in calc ---
Operators: + - * / ^ **

************************************************************/

if (infix_array[infixlen-1] < 23)
	{
	strcpy(ErrMsg2,"Expression must end with constant, variable, or ')'.");
	j = infixlen-1;	/* error position */
	goto CompileErr;
	}

if (infix_array[0] < 7)
	{
	strcpy(ErrMsg2,"Expression may not start with an operator.");
	j = 0;	/* error position */
	goto CompileErr;
	}

for (j=0; j<infixlen-1; j++)
	{
	this_op = infix_array[j];
	next_op = infix_array[j+1];
	if (this_op > funct_offset || this_op == 23 || this_op ==24 || this_op == 25)
		{	  /* constants and variables */
		if (next_op > 6 && next_op != 27 && next_op != 28)	 /*  27 is ), 28 is = */
			{
			strcpy(ErrMsg2,"Constant or variable must be followed by operator or ')'.");
			goto CompileErr;
			}
		}
	else
	if (this_op < 7)	  /* operators */
		{
		if (next_op < 7 || next_op == 27 || next_op == 28)	 /*  27 is ), 28 is = */
			{
			strcpy(ErrMsg2,"Operator must be followed by constant, variable, or function.");
			goto CompileErr;
			}
		}
	else
	if (this_op > 6 && this_op < 23)	  /* functions, 23 = X, 24 = Y, 25 = Z, 26 = ( */
		{
		if (next_op != 26)
			{
			strcpy(ErrMsg2,"Function must be followed by '('.");
			goto CompileErr;
			}
		}
	else
	if (this_op == 26)	  /* 26 = ( */
		{
		if (next_op == 27)
			{
			strcpy(ErrMsg2,"Empty set of parentheses.");
			goto CompileErr;
			}
		else
		if (next_op < 7)
			{
			strcpy(ErrMsg2,"'(' cannot be followed by an operator.");
			goto CompileErr;
			}
		else
		if (next_op == 28)
			{
			strcpy(ErrMsg2,"'(' cannot be followed by an equal sign.");
			goto CompileErr;
			}
		}
	else
	if (this_op == 27 && next_op > 6 && next_op != 27 && next_op != 28)	 /* 27 = ) */
		{
		strcpy(ErrMsg2,"An operator, ')', or '=' must follow ')'.");
		goto CompileErr;
		}
	}

/*
strcpy(MathOutput,("infix: ");

for (j=0; j<infixlen; j++)
	 {
	 if (infix_array[j]<funct_offset)
	sprintf(MathOut,"%s ",opfunct[infix_array[j]]);
	 else
	sprintf(MathOut,"%G ",constant[infix_array[j]-funct_offset]);
	 }
printf("\n");
*/

/* convert to postfix */

 plevel = 1;			 /* parenthesis level */
 pif = -1;				/* infix string position */
 ppf = -1;				/* postfix array position */
 pop = 0;				 /* pending operation stack position */
 pendop[0] = 0;		 /* signal bottom of stack */
 level[0] = 0;		  /* pending operator parenthesis level */

while (++pif < infixlen)
 {
 c = infix_array[pif];

 if (c>funct_offset || c==23 || c==24)	  /* constant or variable */
	  postfix_array[++ppf] = c;

 else
 if (c>6 && c<23)					/* function */
	 {
	 pendop[++pop] = c;
	 level[pop] = plevel;
	 }

 else
 if (c==26)					/* left parenthesis */
	 plevel++;

 else
 if (c==27 || c==28)				  /* right parenthesis or equal */
	 {
	 while (level[pop] == plevel)
	postfix_array[++ppf] = pendop[pop--];
	 plevel--;		  /* reduce parenthesis level */
	 }

 else			 /* must be operator */
	 {
	 while (level[pop] == plevel && opriority[pendop[pop]] >= opriority[c])
	postfix_array[++ppf] = pendop[pop--];
	 pendop[++pop] = c;
	 level[pop] = plevel;
	 }
 }

 while (pop>-1)
	 postfix_array[++ppf] = pendop[pop--];

 if (*use_equal)
	 postfix_array[ppf++] = 6;  /* append a subtraction operator */

 /* determine the required stack size for evaluating the expression */

 *stack = 0;
 for (k=j=0; j<=ppf; j++)
	 {
	 if (postfix_array[j] > 22)	/* variable or constant */
	{
	k++;
	}
	 else
	 if (postfix_array[j] < 7)  /* operator */
	{
	*stack = max(*stack,k);
	k--;
	}
	 }

/*
printf("postfix: ");
for (j=0; j<ppf; j++)
	 {
	 if (postfix_array[j]<funct_offset)
	printf("%s ",opfunct[postfix_array[j]]);
	 else
	printf("%G ",constant[postfix_array[j]-funct_offset]);
	 }
printf("\nrequired stack %d\n",*stack);
*/

	free(funct);
	free(infix_array);
	free(pendop);
	free(level);

	return ppf;


CompileErr:
	if (!infix) {
		err_pos = (int)(ptr1 - funct);
		ptr1 = funct;
		while (*ptr1) {
			if (*ptr1 == '~') *ptr1 = '-';	/* unary sign conversion */
			ptr1++;
		}
/*
	printf("%s\n",funct);
	printf("%.*s^\n",err_pos,"--------------------------------------------------------------------");
*/
	} else {
		k=0;
		for (l=0; l<infixlen; l++) {
			if (infix_array[l] < funct_offset) {
				sprintf(tempstr,"%s",opfunct[infix_array[l]]);
				sprintf(MathOutput,"%s",tempstr);
				k += strlen(tempstr);
			}
			else
			if (infix_array[l] > funct_offset) {
				sprintf(tempstr,"%G",constant[infix_array[l]-funct_offset]);
				sprintf(MathOutput,"%s",tempstr);
				k += strlen(tempstr);
			}
			if (l == j) err_pos = k;
	}
/*
	printf("\n%.*s^\n",err_pos,"--------------------------------------------------------------------");
*/
	}

	sprintf(MathOutput,"Error: %s",ErrMsg2);

	ppf = -1;

	free(funct);
	free(infix_array);
	free(pendop);
	free(level);

	return ppf;
/*
LcarsTrap:
	strcpy(ErrMessage,"Please don't use variables.");
	free(stack);
	return(-1);
*/
}


/************* evaluate the postfix expression *******************/

double MeEvaluate(int *postfix, double *constant, int numops, double xpt,
	 double ypt, double zpt, int stack_size)
{
int
	l,
	ls,
	funct_offset = 30;

double
	result,
	*stack;

/* allocate memory */

if ( (stack = (double *)malloc(stack_size*sizeof(double))) == NULL )
	{
	sprintf(ErrMessage,"Could not allocate %d bytes of memory!",numops*sizeof(double));
	return(-1);
	}

for (l=0, ls=-1; l<numops; l++)
	 {
	 switch (postfix[l])
	{
	case 1:
	case 2:
		 stack[ls-1] = pow(stack[ls-1],stack[ls]);
		 ls--;
		 break;

	case 3:
		 stack[ls-1] *= stack[ls];
		 ls--;
		 break;

	case 4:
		 if (stack[ls] == 0) goto DivZero;
		 stack[ls-1] /= stack[ls];
		 ls--;
		 break;

	case 5:
		 stack[ls-1] += stack[ls];
		 ls--;
		 break;

	case 6:
		 stack[ls-1] -= stack[ls];
		 ls--;
		 break;

	case 7:
		 stack[ls] = sinh(stack[ls] DEGREES);
		 break;

	case 8:
		 stack[ls] = cosh(stack[ls] DEGREES);
		 break;

	case 9:
		 stack[ls] = tanh(stack[ls] DEGREES);
		 break;

	case 10:
		 if (fabs(stack[ls]) > 1.0) goto TrigRange;
		 stack[ls] = asin(stack[ls]);
				stack[ls] = stack[ls] / (1.0 DEGREES);
		 break;

	case 11:
		 if (fabs(stack[ls]) > 1.0) goto TrigRange;
		 stack[ls] = acos(stack[ls]);
				stack[ls] = stack[ls] / (1.0 DEGREES);
		 break;

	case 12:
		 stack[ls] = atan(stack[ls]);
				stack[ls] = stack[ls] / (1.0 DEGREES);
		 break;

	case 13:
		 stack[ls] = sin(stack[ls] DEGREES);
		 break;

	case 14:
		 stack[ls] = cos(stack[ls] DEGREES);
		 break;

	case 15:
		 stack[ls] = tan(stack[ls] DEGREES);
		 break;

	case 16:
		 if (stack[ls] <= 0) goto LogNeg;
		 stack[ls] = log10(stack[ls]);
		 break;

	case 17:
		 if (stack[ls] <= 0) goto LogNeg;
		 stack[ls] = log(stack[ls]);
		 break;

	case 18:
		 stack[ls] = exp(stack[ls]);
		 break;

	case 19:
		 if (stack[ls] < 0) goto SqrtNeg;
		 stack[ls] = sqrt(stack[ls]);
		 break;

	case 20:
		 stack[ls] = fabs(stack[ls]);
		 break;

	case 21:
		 stack[ls] = floor(stack[ls]);
		 break;

	case 22:
		 stack[ls] = stack[ls] - floor(stack[ls]);
		 break;

	case 23:
		 stack[++ls] = xpt;
		 break;

	case 24:
		 stack[++ls] = ypt;
		 break;

	case 25:
		 stack[++ls] = zpt;
		 break;

	default:
		 stack[++ls] = constant[postfix[l]-funct_offset];
	}

	 }
if (ls) sprintf(ErrMessage,"Nonzero stack ptr: %d",ls);
result = stack[0];
free(stack);
return(result);

DivZero:
strcpy(ErrMessage,"Error: Division by zero.");
free(stack);
return(-1);

TrigRange:
strcpy(ErrMessage,"Error: Trig-function input out of range.");
free(stack);
return(-1);

SqrtNeg:
strcpy(ErrMessage,"Error: Square root of negative number.");
free(stack);
return(-1);

LogNeg:
strcpy(ErrMessage,"Error: Log of negative number.");
free(stack);
return(-1);
}

/***************************************************************/

void MeFmtComma(char *value, double number, int comma)
/*
 * 1995 Jan 10  Fixed for Negative numbers.  Dale Holt
 */
{
int  i,
	  j,
	  k,
	  m,
	  n,
	  sign;

	 sprintf(value,"%.10G",number);
	 /* only do nonexponential numbers */
	 if (strchr(value,'E')) return;

	 if (strchr(value,'.') == NULL) strcat(value,".");
	 /* format with commas */
	 if (comma)
	 {
		  i = strlen(value);
		  if (number < 0.0) sign=1; else sign=0;
		  j = 0;
		  m = sign;
		  while ( (m < i) && (value[m++] != '.') ) j++;  /* count the digits */
		  k = (--j) / 3;
		  memmove(value+k+sign,value+sign,i+1);  /* make room for the commas */
		  for (n=sign,m=k+sign; j > 2; m++ )
		  {
		 value[n++] = value[m];
		 if ( (j % 3 ) == 0)	value[n++] = ',';
		 j--;
		  }
	 }
	 /* remove trailing zeros */
	 i = strlen(value)-1;
	 while (value[i] == '0') value[i--] = '\0';
	 if (value[i] == '.') value[i] = '\0';
}

/***************************************************************/

/* function to convert an entire string to lower case */
char *MeStrLower(char *string)
{
	do {
		  if (isupper(*string)) *string = tolower(*string);
	  } while ( *++string != '\0' );
	return string;
}

/*-----------------------------------------------------------------*/

/* function to trim leading and trailing spaces from a string */

char *MeStrTrim(char *string)
{
	int j,k;

	j=0;
	while (string[j] == ' ') j++;
	k=strlen(string)-1;
	while (string[k] == ' ' || string[k] == '\n') k--;
	string[++k] = '\0';
	strcpy(string,&string[j]);
	return string;
}
int LcMath()
{
	int i,z,Result;
	char MathFormula[80];

	memset(MathInput[0],(null),80);
	memset(MathFormula,(null),80);
	memset(MathOutput,(null),80);
	memset(ErrMessage,(null),80);

	i=0;
	z=0;
	do {
		if(MathInput[1][z]==(null)) break;
		if( (MathInput[1][z] > 126) || (MathInput[1][z] < 32)) ++z;
		t=MathInput[1][z];
		if(t!=32) {
			MathFormula[i]=(t);
			++i;
		}
		++z;
	}  while(1);
	strcpy(MathInput[1],MathFormula);
	Result=MathEngine();
	if(ErrMessage[0] != (null)) strcpy(MathOutput,ErrMessage);
	return(Result);
}
/* eof */

