/***
 *** File: g3net.c
 ***
 *** Transport data over a faxmodem connection.
 ***/
char copyright[] = "
 * Copyright (c) 1996
 *     Luigi Rizzo (luigi@iet.unipi.it). All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement: 
 *      This product includes software developed by the Luigi Rizzo,
 *      and other contributors.
 * 4. Neither the name of the Author nor the names of other contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
";

/***
 *** Reed-Solomon encoding courtesy of Phil Karn.
 *** TIFF header files come from the "libtiff" package by Sam Leffler
 *** "getopt" comes from the BSD sources.
 ***/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>	/* bzero and friends */
#include <unistd.h>	/* read/write */
#include "rs.h"

#ifndef O_BINARY
#define O_BINARY 0	/* dos compatibility hack */
#endif

#ifndef min
#define min(a,b)	( (a) < (b) ? (a) : (b) )
#endif

#define	P(x) fprintf x
#define	Pv(x) if (verbose) fprintf x
#define	P2(x) if (verbose>1) fprintf x
#define	P3(x) if (verbose>2) fprintf x
#define	P4(x) if (verbose>3) fprintf x
#define	P1(x) 

#define	SRC_WIDTH		(64)
/*** a g3 line is actually preceded by 2 bytes (line #) */
#define	G3_WIDTH		(SRC_WIDTH+2)
#define	SRCLINES		KK
#define	ECCLINES		255 /* for RS codes */
#define	SRC_BLKSIZE		(SRC_WIDTH * SRCLINES)
#define	ECC_BLKSIZE		(SRC_WIDTH * ECCLINES)

#define u_char unsigned char
#define u_long unsigned long

int verbose=0;
#ifdef MSDOS
#define MSDOS_ON 1
#define UNIX_ON 0
#else
#define MSDOS_ON 0
#define UNIX_ON 1
#endif
int reverse=MSDOS_ON;
int add_tiff=MSDOS_ON;

u_long tot_lines = 0; /* reset once per output file */

#ifdef MISSING_BZERO	/* and other systems missing some functions */
    /* these are only here to make the program compile, and work
     * slowly. If you can, use/write better replacement functions.
     */

void
bzero(char *b, long len)
{
    for (; len ; len--) *b++ = 0; /* slow but who cares! */
}

void
bcopy(char *src, char *dst, long len)
{
    for (; len ; len--) *dst++ = *src++ ;
}

void
srandom(unsigned seed)
{
}

long
random(void)
{
    static long prev=1;
    prev = ( prev * 17 );
    return (prev);
}
char *__progname="g3net";
#endif

char usage[] =
    "%s [-e] [-d] [-r] [-t] [-v] [-f] [inputfile] [-o] [outputfile]\n"
    "    -d       decode input file (containing g3 data) -- default\n"
    "    -e       encode input file into g3 data\n"
    "    -r       [toggle] reverse input bits (default on MSDOS)\n"
    "    -t       [toggle] produce a TIFF file instead of a g3 file\n"
    "    -v       be verbose\n"
    "    -f       optional prefix for input file\n"
    "    -o       optional prefix for output file\n"
    "local defaults: decode, "
#ifdef MSDOS
    "reverse bits, tiff, "
#endif
    "input from stdin, output to stdout\n";

/***
 *** some info on data layout
 ***

 ENCODING

     Source data consists of a block of bytes, whose first 2 bytes
     (little endian) indicate the length of the block.  The max
     (source) blocksize is SRCLINES * SRC_WIDTH - 2.  The block is
     copied to eccbuf[], where each row of KK bytes is copied to
     a vector of size ECCLINES (the additional space is used for
     ecc codes, the last line is padded to an integer multiple of
     SRCLINES.

     Data in eccbuf[] is then encoded using RS codes, then transposes
     (column by column, in a temporary buffer), rotated, and encoded
     as g3 data.  Before g3-encoding, each line is prepended with
     2 copies of its index.

     The encoding process saves data to an array and the associated
     len. The array includes the leading and tailing EOL codes.

 DECODING

     The decoding process is passed a pointer to the received g3 data
     and the length of the block.
     Data is first g3-decoded, saving each line to a temporary buffer,
     and its length into the length[] vector.
     Then, the first 2 bytes of the temporary buffer are tested. If
     they match, we have a valid line, possibly partial, which is then
     transposed and copied to eccbuf[] keeping length information.
     Unused positions are zero-filled (but they are considered
     erasures by the RS decoding process).

     After g3-decoding, row by row we compute the erasures and invoke
     the RS decoder.

 ***
 ***/



u_char * g3_encode_byte(u_char data, u_char *dst, int cmd);

#define TESTING

#ifdef	TESTING
/***
 *** These functions are only used to test the encoder/decoder
 ***/

/***
 *** add_noise is used for testing purposes, it corrupts g3 data
 *** to test the robustness of the decoding algorithm.
 ***/
int
add_noise(u_char *data, int datalen, int errors)
{
    /***
     *** add errors percent of errors, including scrambled bytes
     *** and shortened/missing lines
     ***/
    int i, d;

    errors = (datalen / 100 ) * errors / 100 ;
    for (i=0; i < errors/2; i++) {
	d = random() % datalen;
	P4((stderr, "scrambling byte at g3 offset %d\n", d));
	data[d] = 0x5a;
    }
    for (i=0; i < errors/2; i++) {
	int j;
	d = random() % datalen;
	P4((stderr, "removing byte at g3 offset %d\n", d));
	for (j=d; j < datalen-1 ; j++)
	    data[j] = data[j+1];
	datalen-- ;
    }
    return datalen;
}

int
test_errs(u_char *src, u_char *dst, int len)
{
    int i, errs=0;
    for (i=0; i< len ; i++) {
	if (src[i] != dst[i]) {
	    errs++;
	    Pv((stderr,"error at pos %d - should be 0x%02x - 0x%02x\n",
		    i, src[i], dst[i]));
	}
    }
    if (errs)
	P((stderr,"%d decoding errors\n",errs));
    return 0;
}
#endif	/* TESTING */

int
copy_blk(u_char *src, long srclen, u_char *buf, long buflen)
{
    /***
     *** copy input data to the buffer, including the two length bytes
     *** at the beginning, and rs-encode. We use a (190,255) code, and
     *** exactly 64 columns. Hence, the source is first partitioned in
     *** 64 blocks, and lines are zero-padded up to position 190.
     ***
     ***/
    int w = 0, i ;
    int in_blks = (srclen + SRC_WIDTH - 1) / SRC_WIDTH ;

    bzero(buf, buflen);
    for (i=0, w=0 ; i < srclen; ) {
	buf[w++] = src[i++] ;
	if ( w == in_blks || i == srclen ) { /* end of line */
	    P3((stderr, "encoding block %d\n", in_blks));
	    encode_rs( buf, buf + SRCLINES );
	    w = 0 ;
	    buf += ECCLINES ;
	}
    }
    return in_blks;
}

u_char eccbuf[SRC_WIDTH][ECCLINES];	/* buffer for ECC data */
    /*** eccbuf is shared by encode and decode functions ***/

/***
 *** g3_encode encodes a block of max SRC_BLKSIZE bytes. The length of the
 *** encoded block is returned, or -1 if an error occurs. The only possible
 *** error at this point is passing invalid parameters.
 ***
 *** src: pointer to the data block.
 *** srclen: block size. srclen=0 means that an end of block
 ***	is produced.
 ***	NOTE NOTE NOTE: the first two bytes of the block are
 ***	overwritten with the length of the block, so they should
 ***	not contain user data.
 *** dst: pointer to the output block. Sufficient size is assumed,
 *** the block expands roughly by a factor of 2.
 ***/

int
g3_encode(u_char *src, int srclen, u_char *dst)
{
    int i, n1, n_ecc;
    u_char *dst0 = dst;

    if (srclen < 0 || srclen > SRC_BLKSIZE) {
	P((stderr,"invalid srclen %d\n", srclen));
	return -1;
    }
    if (srclen == 0) { /*** final padding of page ***/
	for (i=0; i<8 ; i++)
	    dst = g3_encode_byte(0, dst, 3); /*  EOL, padded */
	return dst - dst0;
    }

    src[0] = srclen & 0xff;
    src[1] = (srclen >>8) & 0xff;
    n1 = copy_blk( src, srclen, &eccbuf[0][0], sizeof(eccbuf) );
    n_ecc = min (ECCLINES-SRCLINES, (n1+1)/2 );
    Pv((stderr, "*** Encoding %d bytes (%d data blocks, %d ecc lines)\n",
	srclen, n1, n_ecc));
    n_ecc += SRCLINES ; /* easier for comparisons */

    /***
     *** at this point have blks rows, each containing 255 bytes of
     *** ECC data. Must now scramble and g3-encode.
     *** To protect from transmission errors, each line is also error-
     *** protected with a N,N+1 RS code.
     ***/

    dst = g3_encode_byte(0, dst, 0); /* reset encoder */

    for (i=0; i<2 ; i++)
	dst = g3_encode_byte(0, dst, 3); /* initial EOL, padded */

    for (i=0; i<ECCLINES; i++) {
	int j;
	int rot = (31*i) % (SRC_WIDTH);

	if ( (i >= n1 && i < SRCLINES) || i >= n_ecc) {
	    P3((stderr,"skipping line %d\n",i));
	    continue;
	}
	/***
	 *** send only non-zero data lines (say N1).
	 *** Fully-zero lines between N1 and SRCLINES are assumed
	 *** as correctly received and default to all-zeroes.
	 ***
	 *** send at most max( ECCLINES-SRCLINES, (N1*2)/5) ECC lines,
	 *** others default as erasures.
	 ***/
	/***
	 *** here we provide the g3 encoding for the line
	 ***/
	P3((stderr, "g3-encoding line %d, size %d starts with "
	    "0x%02x 0x%02x 0x%02x 0x%02x\n",i, G3_WIDTH,
	    eccbuf[rot][i], eccbuf[(rot+1)%SRC_WIDTH][i],
	    eccbuf[(rot+2)%SRC_WIDTH][i], eccbuf[(rot+3)%SRC_WIDTH][i]
	    ));

	dst = g3_encode_byte((u_char)i, dst, 1); /* index */
	dst = g3_encode_byte((u_char)i, dst, 1); /* index */

	for (j=0; j < SRC_WIDTH ; j++) {
	    dst = g3_encode_byte(eccbuf[rot++][i], dst, 1); /* data */
	    if (rot == SRC_WIDTH) rot = 0;
	}
	dst = g3_encode_byte(0, dst, 2); /* final EOL */
    }
    dst = g3_encode_byte(0, dst, 4); /* black line and one more EOL, padded */

#if 0
    for (i=0; i<6 ; i++)
	dst = g3_encode_byte(0, dst, 3); /* final EOL */
#endif
    Pv((stderr, "--- encoding process returns %d bytes\n", dst - dst0));
    return (dst - dst0);
}


/***
 *** these are the g3 codes used to encode bits.
 ***/
u_char w_len[16] = { /* length of g3 white codes */
   4,    4,    4,    4,    4,    4,
   5,    5,    5,    5,
   6,    6,    6,    6,    6,    6
};
u_char w_code[16] = { /* g3 white codes */
   0x0e, 0x01, 0x0d, 0x03, 0x07, 0x0f,	/* length 4 */
   0x19, 0x05, 0x1c, 0x02,		/* length 5 */
   0x04, 0x30, 0x0b, 0x2b, 0x15, 0x35	/* length 6 */
};
u_char w_g3_len[16] = { /* corresponding segment lengths in pixels */
   2,    3,    4,    5,    6,    7,
   8,    9,    10,   11,
   12,   13,   14,   15,   16,   17
};

u_char b_len[4] = { 2, 2, 3, 3 };
u_char b_code[4] = { 0x3, 0x1, 0x2, 0x6 };
u_char b_g3_len[4] = { 2, 3, 1, 4 };

/***********************************************************************
 ***
 *** these are the 'white' codes which are used to fill-up a line to
 *** its standard width (1728), so the file is a compliant g3 file.
 ***
 ***********************************************************************/

/* termina codes white, start 0, step 1, len 64 */

struct g3code { u_short code; u_short len ; } ;

struct g3code t_white[64] = {
{ 0x0ac,  8 }, { 0x038,  6 }, { 0x00e,  4 }, { 0x001,  4 },
{ 0x00d,  4 }, { 0x003,  4 }, { 0x007,  4 }, { 0x00f,  4 },
{ 0x019,  5 }, { 0x005,  5 }, { 0x01c,  5 }, { 0x002,  5 },
{ 0x004,  6 }, { 0x030,  6 }, { 0x00b,  6 }, { 0x02b,  6 },
{ 0x015,  6 }, { 0x035,  6 }, { 0x072,  7 }, { 0x018,  7 },
{ 0x008,  7 }, { 0x074,  7 }, { 0x060,  7 }, { 0x010,  7 },
{ 0x00a,  7 }, { 0x06a,  7 }, { 0x064,  7 }, { 0x012,  7 },
{ 0x00c,  7 }, { 0x040,  8 }, { 0x0c0,  8 }, { 0x058,  8 },
{ 0x0d8,  8 }, { 0x048,  8 }, { 0x0c8,  8 }, { 0x028,  8 },
{ 0x0a8,  8 }, { 0x068,  8 }, { 0x0e8,  8 }, { 0x014,  8 },
{ 0x094,  8 }, { 0x054,  8 }, { 0x0d4,  8 }, { 0x034,  8 },
{ 0x0b4,  8 }, { 0x020,  8 }, { 0x0a0,  8 }, { 0x050,  8 },
{ 0x0d0,  8 }, { 0x04a,  8 }, { 0x0ca,  8 }, { 0x02a,  8 },
{ 0x0aa,  8 }, { 0x024,  8 }, { 0x0a4,  8 }, { 0x01a,  8 },
{ 0x09a,  8 }, { 0x05a,  8 }, { 0x0da,  8 }, { 0x052,  8 },
{ 0x0d2,  8 }, { 0x04c,  8 }, { 0x0cc,  8 }, { 0x02c,  8 },
};

/* make-up codes white start 64, step 64, len 27 */
struct g3code m_white[27] = {
{ 0x01b,  5 }, { 0x009,  5 }, { 0x03a,  6 }, { 0x076,  7 },
{ 0x06c,  8 }, { 0x0ec,  8 }, { 0x026,  8 }, { 0x0a6,  8 },
{ 0x016,  8 }, { 0x0e6,  8 }, { 0x066,  9 }, { 0x166,  9 },
{ 0x096,  9 }, { 0x196,  9 }, { 0x056,  9 }, { 0x156,  9 },
{ 0x0d6,  9 }, { 0x1d6,  9 }, { 0x036,  9 }, { 0x136,  9 },
{ 0x0b6,  9 }, { 0x1b6,  9 }, { 0x032,  9 }, { 0x132,  9 },
{ 0x0b2,  9 }, { 0x006,  6 }, { 0x1b2,  9 },
};


/***
 *** encodes one byte. Returns the new value of the pointer,
 *** possibly advanced during the encoding process.
 ***
 *** cmd specifies the actual action, i.e:
 ***
 ***   0 : reset encoder, dst unused
 ***   1 : encode single byte
 ***   2 : finish line, no padding
 ***   3 : finish line, padding
 ***/
static u_char brev[256]; /* bit reversal */

void
init_brev()
{
    int i;
    if (reverse)
	for (i=0; i<256; i++) brev[i]=i;
    else
	for (i=0; i<256; i++)
	    brev[i] = ( ((i & 0x01) << 7) | ((i & 0x02) << 5) |
		    ((i & 0x04) << 3) | ((i & 0x08) << 1) |
		    ((i & 0x10) >> 1) | ((i & 0x20) >> 3) |
		    ((i & 0x40) >> 5) | ((i & 0x80) >> 7) );
}

u_char *
g3_encode_byte(u_char data, u_char *dst, int cmd)
{

#define	C_WHITE 4
#define	C_BLACK 2
#define	C_EOL   0

    static int c   = C_WHITE; /* start with white */
    static int in_bits =0, out_bits = 0 ;
    static u_long in_buf = 0, out_buf = 0 ;
    static u_char *dst0;

    static int this_line_len = 0;

    static long this_bytes = 0;

    if (cmd == 0) {
	P4((stderr,"g3_encode_byte: execute cmd 0 dst 0x%08lx\n", (u_long)dst));
	c   = C_WHITE;
	in_bits = out_bits = 0;
	in_buf = out_buf = 0;

	this_bytes = 0;
	this_line_len = 0;
	dst0 = dst;
	return dst;
    }
    if (dst == NULL || cmd < 1 || cmd > 4)
	return NULL;

    if (cmd == 1) {			/* encode byte */
	data &= 0xff ;
	in_buf |= (data << in_bits);
	in_bits += 8;

	this_bytes++;

        for (; in_bits >= c;) {
	    u_long d = in_buf & ( (1<< c) - 1 ) ;
	    in_bits -=  c;
	    in_buf >>=  c;
	    if (c == C_WHITE) {
		out_buf |= (w_code[d] << out_bits ) ;
		out_bits += w_len[d] ;
		this_line_len += w_g3_len[d];
		c = C_BLACK;
	    } else {
		out_buf |= (b_code[d] << out_bits ) ;
		out_bits += b_len[d] ;
		this_line_len += b_g3_len[d];
		c = C_WHITE;
	    }
	}
    } else {				/* close line */
	P4((stderr,"g3_encode_byte: execute cmd %d off %d\n",cmd,dst-dst0));
	if (in_bits != 0) { /* is an error */
	    P((stderr,"g3_encode_byte: error, have %d bits after %ld bytes\n",
		in_bits, this_bytes));
	    this_bytes = 0;
	    return NULL;
	}
	{ /* pad line */
	   int delta = 1728 - this_line_len;
	   int m = delta/64 - 1 ;
	   int t = delta & 63 ;

           P4((stderr,"line len %d, pad with %d, %d\n",
		this_line_len, m , t));
	   if (m>=0) {
	       out_buf |= (m_white[m].code << out_bits);
	       out_bits += m_white[m].len ;
	   }
	   out_buf |= (t_white[t].code << out_bits);
	   out_bits += t_white[t].len ;
	   this_line_len = 0;
	    while (out_bits >= 8) {
		P4((stderr,"emitting 0x%02x\n", (u_char)(out_buf & 0xff)));
		*dst++ = brev[ (u_char)(out_buf & 0xff) ];
		out_buf >>= 8;
		out_bits -= 8;
	    }
	}
	tot_lines ++;	/* count of total lines */
	this_bytes = 0;
	if (cmd == 4) {
#if 1	/* crude hack to add a black line */
	    static int black[] = { /* code/length */
#if 0
		0x800, 12,	/* code/len */
		0xAC, 8,	/* white segment */
		0xa40, 13,	/* black span */
		0x6, 3,		/* black fillup */
		0x800, 12,	/* code/len */
		0xAC, 8,	/* white segment */
		0xa40, 13,	/* black span */
		0x6, 3,		/* black fillup */
		0x800, 12,	/* code/len */
		0xAC, 8,	/* white segment */
		0xa40, 13,	/* black span */
		0x6, 3,		/* black fillup */
#endif
		0x0, 0	};	/* terminator */
	    int *ip = black;
	    u_long code, data;

		tot_lines --; /* XXX */
	    for (;;) {
		while (out_bits >= 8) {
		    P4((stderr,"emitting 0x%02x\n", (u_char)(out_buf & 0xff)));
		    *dst++ = brev[ (u_char)(out_buf & 0xff) ];
		    out_buf >>= 8;
		    out_bits -= 8;
		}
		code = *ip++;
		data = *ip++;
		if (data == 0) break;
		code &= (1<<data) - 1 ;
		out_buf |= (code << out_bits);
		out_bits += data ;
	    }
	    cmd = 3;
	}
#endif /*** hack ***/
	if (cmd == 3) {			/* eol, byte aligned */
	    if (out_bits > 4)
	       out_bits = 12 ;
	    else if (out_bits > 0)
	       out_bits = 4;
	}
	/* now is simply eol */

	out_buf |= ( 0x800 << out_bits ); /* add eol code */
	out_bits += 12;	/* max 24 now */
    }
    while (out_bits >= 8) {
	P4((stderr,"emitting 0x%02x\n", (u_char)(out_buf & 0xff)));
	*dst++ = brev[ (u_char)(out_buf & 0xff) ];
	out_buf >>= 8;
	out_bits -= 8;
    }
    return dst;
}    

void
repack(u_char *data)
{
    int i, len, blk;
    u_char * s, *d;

    len = data[0] + data[1]*256 ;
    blk = (len + SRC_WIDTH - 1)/SRC_WIDTH;

    Pv((stderr,"repacking in %d rows\n",blk));

    if (blk == SRCLINES || len==0) return; /* no need! */
    /*** fist line is ok ***/
    s = data + SRCLINES ;
    d = data + blk;
    for (i=1; i<SRC_WIDTH ; i++) {
	bcopy(s, d, blk);
	s += SRCLINES ;
	d += blk ;
    }
}



/***
 *** the decoder is a whole function.
 ***
 *** src: source of g3 data
 *** dst: destination buffer
 ***/

int
g3_decode(u_char *src, int srclen, u_char *dst, int dstlen)
{
    int fail = 0;
    int prev_eol = 0, curr_eol = 0; /* positions of eol */
    u_char *dst0 = dst;
    int line = 0;	/*** line currently processed ***/
    int i;

    u_char length[ECCLINES];

    bzero(eccbuf, sizeof(eccbuf) );
    bzero(length, sizeof(length) );
    dst[0] = dst[1] = 0;

  { 
    int real_line = 0;	/*** real line according to g3 data ***/
    int pos = 0;

    int c   = C_EOL; /*  wait for eol, or mode */
    int in_bits =0, out_bits = 0 ;
    u_long in_buf = 0, out_buf = 0 ;

    long this_bits = 0;
    long skipped_bits =  -1;
    int s = 0;

    u_char g3buf[G3_WIDTH + 1]; /* tmp buffer for g3 data */
    bzero(g3buf, sizeof(g3buf) );

    P2((stderr,"Starting g3_decode_block, srclen = %d\n", srclen));

    for (;;) {
	int len;
	int lim;

again:
	/***
	 *** first, try flushing pending output bits
	 ***/
	while (out_bits >= 8) {
	    if (pos < G3_WIDTH + 2 /* index */) {
		u_char d = out_buf & 0xff;
		g3buf[pos++] = d ;
		out_buf >>= 8;
		out_bits -= 8;
		if (pos < 10)
		    P4((stderr,"emitting byte 0x%02x (pos %d)\n",
			d, pos -1));
	    } else {
		/***
		 *** should not happen, unless the line is too long
		 ***/
		Pv((stderr,"g3 line too long (%d), maybe not a g3net file\n",
		    pos));
		goto g3_error;
	    }
	}

        if (s >= srclen) break;
	/***
	 *** if needed, read one more byte from the input stream
	 ***/
	if (in_bits < 12) {
	    in_buf |= brev[ src[s] ] << in_bits ;
	    in_bits += 8;
	    if (pos < 10 || in_buf & 0xfff == 0x800)
		P4((stderr,"state %d read 0x%02x --> 0x%08lx / %d\n",
		    c, src[s], in_buf, in_bits));
	    s++ ;
	}

	/***
	 *** run the state machine. Either wait for EOL, WHITE or BLACK
	 ***/
	switch (c) {
	case C_EOL:
	    /***
	     *** wait for eol, ignoring garbage. Does not start a new line.
	     ***/
	    while (in_bits >= 12) {
		if ( (in_buf & 0xfff) == 0x800 ) { /*** found EOL ***/
		    prev_eol = curr_eol;
		    curr_eol = s;
		    if (skipped_bits)
			P4((stderr,"found eol, line %d real %d "
			    "g3 offset %d skipped %ld bits\n",
			    line, real_line, s, skipped_bits));
		    in_buf >>= 12;
		    in_bits -= 12;
		    this_bits = 0;
		    skipped_bits = 0;
		    c = C_WHITE;	/*** next, wait for WHITE ***/
		    goto again;
		} else { /*** not found, shift and repeat ***/
		    P4((stderr,"not found eol, in_bits %d, shift by 1\n",
			in_bits));
		    in_buf >>= 1;
		    in_bits--;
		    skipped_bits ++;
		}
	    }
	    /*** need more data ***/
	    break;

	case C_WHITE:
	    /***
	     *** white codes can be 4..6 bits long
	     ***/
	    if (in_bits < 4) break;
	    lim = min(in_bits,6);
	    for (len = 4; len <= lim; len++) {
		u_char d = ( in_buf & ( (1 << len) - 1) );
		int i;
		for (i = 0; i < 16; i++) {
		    if (w_len[i] == len && w_code[i] == d) { /* found */
			this_bits += 4;
			out_buf |= ( i << out_bits );
			out_bits += 4 ;
			in_bits -= len ;
			in_buf >>= len ;
			if (pos < 10)
			    P4((stderr,"C_WHITE: 0x%1x/4 encoded as "
				"0x%02x/%d (0x%08lx), total %ld/%ld\n",
				i, d, len, out_buf,
				this_bits, this_bits/8));
			c = C_BLACK ;
			goto again;
		    }
		}
	    }

g3_error:
	    /***
	     *** if we get here, there is an error or an EOL.
	     *** For us the rest of the line does not contain anything.
	     *** need at least 2 bytes for a valid line
	     ***/
	    if (pos != G3_WIDTH) {
		/***
		 *** In case of error, try to skip corrupt bits.
		 ***/
		pos = pos - 2; /* XXX */
	    }
	    if (pos >= 2 && g3buf[0] == g3buf[1] && g3buf[0] >= real_line) {
		int rot ;
		line++;
		if (g3buf[0] >= SRCLINES && real_line < SRCLINES) {
		    int l;
		    for (l = real_line+1; l < SRCLINES; l++) {
			int j;
			P3((stderr,"rebuilding line %d\n",l));
			for (j=0; j < SRC_WIDTH; j++)
			    eccbuf[j][l] = 0;
			length[l] = SRC_WIDTH;
		    }
		}
		real_line = g3buf[0] ;
		length[real_line] = pos - 2; /*** skip first 2 bytes ***/
		rot = (31*real_line) % (SRC_WIDTH);

		P2((stderr, "g3-line %d (%d), size %d starts with "
		    "0x%02x 0x%02x 0x%02x 0x%02x\n",
		    real_line, line, pos,
		    g3buf[2], g3buf[3], g3buf[4], g3buf[5] ));

		/***
		 *** transpose
		 ***/
	        for (i = 2; i < pos ; i++) {
		    eccbuf[rot][real_line] = g3buf[i] ;
		    if ( ++rot == SRC_WIDTH ) rot = 0;
		}
		if (real_line == ECCLINES - 1) {
		    prev_eol = curr_eol = s;
		    goto end_of_block; /* no need to check further */
		}
	    } else {
		/*** maybe it is the beginning of a new block ? ***/
		if (pos >= 2 && g3buf[0] == g3buf[1] &&
			g3buf[0] < real_line && real_line >= SRCLINES) {
		    goto end_of_block; /* no need to check further */
		}
		if (pos > 1)
		    P3((stderr,"line not found, pos %d g3_ofs %d ix "
			"0x%02x 0x%02x in_buf 0x%08lx/%d\n",
			pos, s, g3buf[0], g3buf[1], in_buf, in_bits));
		g3buf[0] = g3buf[1] = 0xff;
	    }
	    out_bits = 0;
	    out_buf = 0;
	    pos = 0;
	    c = C_EOL;
	    break;

	case C_BLACK:
	    /***
	     *** black codes are 2 or 3 bits
	     ***/
	    if (in_bits < 2) break;
	    lim = min(in_bits,3);
	    for (len = 2; len <= lim; len++) {
		u_char d = ( in_buf & ( (1 << len) - 1) );
		int i;
		for (i = 0; i < 4; i++) {
		    if (b_len[i] == len && b_code[i] == d) {
			this_bits += 2;
			out_buf |= ( i << out_bits );
			out_bits += 2 ;
			in_bits -= len ;
			in_buf >>= len ;
			if (pos < 10)
			    P4((stderr,"C_BLACK: 0x%1x/4 encoded as "
				"0x%02x/%d (0x%08lx), total %ld/%ld\n",
				i, d, len, out_buf, this_bits, this_bits/8));
			c = C_WHITE ;
			goto again;
		    }
		}
	    }
	    goto g3_error;

	} /* switch */
    } /* outside for cycle */
  } /*** end of g3 decoder block ***/

end_of_block:
    /***
     *** now eccbuf[][] has the source data with erasures.
     *** Do RS decoding on each of the SRC_WIDTH lines
     ***/

    if (line ==0 ) {
	Pv((stderr,"no data found, consuming %d bytes\n", srclen));
	return srclen;
    }
    P2(( stderr, "starting RS decoding on %d lines\n", line ));

    if (1) {
	int broken = 0;
	int missing = 0;
	for (i=0; i< ECCLINES; i++) {
	    if (length[i] != SRC_WIDTH && length[i] > 0 && i < SRCLINES) {
		broken++;
		missing += SRC_WIDTH - length[i] ;
		Pv((stderr,"*** line %3d width %3d\n", i, length[i]));
	    }
	}
	if (broken || missing)
	    Pv((stderr,"%d corrupt lines, %d missing bytes\n",
		broken, missing));
    }
    for (i=0; i < SRC_WIDTH ; i++) {
	int eras_pos[ECCLINES];
	int eras = 0;
	int risu;
	int j;

	int first = 0;
	/***
	 *** compute erasures
	 ***/
	for (j = 0; j < ECCLINES ; j++) {
	    long last =  (first + length[j]) ;
	    int err = 0;
	    if ( length[j] == 0 /* empty line */ ) err = 1;
	    else if ( last <= SRC_WIDTH ) {
		if (i < first || i >= last) err = 1;
	    } else {
		last = last - SRC_WIDTH ;
		if ( i >= last && i < first ) err = 1;
	    }
	    if (err) {
		eras_pos[eras++] = j;
		P2((stderr,"warning, erasure %d at pos %d in line %d\n",
			eras, j, i));
	    }
	    if (++first == SRC_WIDTH) first = 0;
	}

	if (0 && eras) {
	    char b[ECCLINES+1];
	    memset(b, '*', eras);
	    b[eras]='\0';
	    P((stderr,"line %3d erasures %3d %s\n",i, eras, b));
	}
	 
	risu = eras_dec_rs(eccbuf[i], eras_pos, eras);
	if (risu == -1 ) {
	    Pv((stderr,"Too many errors decoding block %d (%d eras)\n", i,
		eras));
	    fail++;
	}
	if (dst + SRCLINES - dst0 > dstlen) {
	    P((stderr,"ouch! would overflow buffer (%d vs %d)\n",
		dst + SRCLINES - dst0, dstlen));
	    exit(0);
	}
	bcopy( eccbuf[i], dst, SRCLINES);
	dst += SRCLINES;
    }
    repack(dst0);
    if (fail)
	P((stderr,"%d blocks with decoding errors\n", fail));
    return fail ? -1 : prev_eol; /* this many chars were used */
}


/***
 *** routines to prepend a TIFF header to the file
 ***/

#include "tiff.h"

static u_char tiffdir[512];
static u_char *tiffdirp;
uint16 *tiffdircount;

/***
 *** tde writes a stripped-down directory entry, with only 1 element
 *** which is <= 4bytes in size.
 ***/
void
tde(u_short tag, u_short type, u_long val)
{
    TIFFDirEntry tde;
    tde.tdir_tag = tag;
    tde.tdir_type = type;
    tde.tdir_count = 1;
    tde.tdir_offset = val;

    (*tiffdircount)++;
    bcopy ((void *)&tde, (void *)tiffdirp, sizeof(tde));
    tiffdirp += sizeof(tde);
}

void
make_hdr(int out_file, unsigned length)
{
    TIFFHeader th;
    u_long cur_pos;

Pv((stderr, "%d total lines \n", length));
    cur_pos = (u_long)lseek(out_file, 0, SEEK_CUR); /* mark current file position */
    if (cur_pos & 1) {
	write(out_file, "\0", 1); /* pad... */
	cur_pos++;
    }
    th.tiff_magic = TIFF_LITTLEENDIAN ;
    th.tiff_version = TIFF_VERSION;
    th.tiff_diroff = (unsigned long)cur_pos;        /* filled in later */

    tiffdirp = tiffdir + 2;
    tiffdircount = (uint16 *)tiffdir;
    *tiffdircount = 0;

	/*** now make tiff tags ***/
    tde(TIFFTAG_IMAGEWIDTH, TIFF_SHORT, 1728);
    tde(TIFFTAG_IMAGELENGTH, TIFF_SHORT, length);
    tde(TIFFTAG_BITSPERSAMPLE, TIFF_SHORT, 1);
    tde(TIFFTAG_COMPRESSION, TIFF_SHORT, COMPRESSION_CCITTFAX3);
    tde(TIFFTAG_PHOTOMETRIC, TIFF_SHORT, PHOTOMETRIC_MINISWHITE);
    tde(TIFFTAG_FILLORDER, TIFF_SHORT,
	reverse ? FILLORDER_LSB2MSB : FILLORDER_MSB2LSB); /*XXX*/
    tde(TIFFTAG_STRIPOFFSETS, TIFF_LONG, 8);
    tde(TIFFTAG_ORIENTATION, TIFF_SHORT, ORIENTATION_TOPLEFT);
    tde(TIFFTAG_SAMPLESPERPIXEL, TIFF_SHORT, 1);/*XXX */
    tde(TIFFTAG_ROWSPERSTRIP, TIFF_SHORT, length);/*XXX */
    tde(TIFFTAG_STRIPBYTECOUNTS, TIFF_LONG, cur_pos-8);
    tde(TIFFTAG_PLANARCONFIG, TIFF_SHORT, PLANARCONFIG_CONTIG);/*XXX */

    bzero(tiffdirp, 4); /* trailing zero required */
    tiffdirp += 4;
    write(out_file, tiffdir, tiffdirp - tiffdir);
    lseek(out_file, 0, 0);
    write(out_file, &th, sizeof(th));
}

/***
 *** end of tiff routines
 ***/

u_char my_g3buf[ECC_BLKSIZE * 2]; /* oversized! */
u_char srcbuf[SRC_BLKSIZE];	/* buffer for test data */

/***
 *** open_out is used to open an output file. If the
 *** pathname has changed, the old handle is closed (if
 *** existed) and the new one is opened.
 ***
 *** A NULL filename is translated to stdout.
 *** Existing files are not overwritten.
 ***/

int
open_out(char *outfn)
{
    static char oldfn_buf[256];
    static char *oldfn=NULL;
    static int handle = -1;

    if (outfn==NULL) outfn="-";
    if (oldfn==NULL || strcmp(outfn, oldfn)) {
	/*** close old handle, open a new one ***/
	if (handle > 2 ) { /*** do not close std files ***/
	    close(handle);
	    handle = -1;
	}
	strncpy(oldfn_buf, outfn, sizeof(oldfn_buf) -1 );
	oldfn = oldfn_buf;
	if (!strcmp(outfn,"-"))
	    handle = 1;
	else
	    handle = open (oldfn, O_BINARY|O_EXCL|O_CREAT|O_WRONLY, 0777);
	if (handle < 0 ) {
	    P((stderr, "cannot create file %s\n",oldfn));
	}
    }
    return handle;
} 

/***
 *** decode an input file. A NULL outfn is assumed as a will to
 *** use the original filename. Otherwise, "-" means outoput
 *** to stdout, and a name ending with a "/" is prepended to
 *** the original filename.
 ***/
u_long
decode_file(int in_file, char *outfn)
{
    static char outfile[256];
    int out_file;
    u_long written=0;
    int len, g3len, delta = 0, ofs = 0;
    int data_offset;
    int isadir=0;

    if (in_file < 0)
	return 1; /* error */
    strcpy(outfile,"");
    if (outfn != NULL) {
	isadir = (outfn[strlen(outfn)-1] == '/' );
	strncpy(outfile, outfn, sizeof(outfile));
    }
    for (;;) {
	/***
	 *** read one block from the input file
	 ***/
again:
	g3len = read(in_file, my_g3buf + delta, sizeof(my_g3buf) - delta);
	if (g3len < 0) return 0;
	delta += g3len; /* add previous bytes */
	if (delta < sizeof(my_g3buf) && g3len > 0) goto again;
	g3len = delta;

	/***
	 *** try to decode the block
	 ***/
	bzero(srcbuf, sizeof(srcbuf));
	ofs = g3_decode(my_g3buf, g3len, srcbuf, sizeof(srcbuf) );
	if (ofs < 0) {
	    P((stderr,"decode failure! \n"));
	    exit(1);	/* decode failure ? */
	}

	/***
	 *** compute length, offset and possibly file name of data
	 ***/
	len = srcbuf[0] + 256*srcbuf[1];
	data_offset = srcbuf[2] ;
	if (len < 3 )
	    return written ; /* error in decoding */
	if (data_offset > 3) {
	    if (outfn == NULL)
		strncpy(outfile, srcbuf+3, sizeof(outfile));
	    else if (isadir)
		sprintf(outfile,"%s%s",outfn,srcbuf+3);
	    P((stderr,"original file name %s, saving to file %s\n",
		srcbuf+3, outfile));
	} else {
	    data_offset = 3;
	}
	out_file = open_out(outfile);
	if (out_file > 0) {
	    written += write(out_file, srcbuf+data_offset, len - data_offset);
	    Pv((stderr,"*** decode done, in %d (%d) g3 bytes, "
		"out %d data bytes ***\n",
		g3len, ofs, len - data_offset));
	} else {
	    written++; /* to make the program silent */
	}
	delta = g3len - ofs ;
	if (delta == 0) return written; /* ok */
	bcopy(my_g3buf+ofs, my_g3buf, delta) ;
    }
}

int
encode_file(int in_file, int out_file, char *in_fn, int nerrors)
{
    int test_len, g3len;
    int delta = 0;
    int first_block = 1;

    static u_char inbuf[SRC_BLKSIZE];	/* buffer in encode_file */

    if (out_file < 0) return 0;
    if (add_tiff)
	write(out_file, inbuf, 8); /* room for tiff header */

    for(;;) {
	int data_ofs = 3;
	delta = 0;
	if (first_block) {
	    first_block = 0 ;
	    if (in_fn != NULL && strlen(in_fn) < 60) {
		strcpy(inbuf+data_ofs, in_fn);
		data_ofs += strlen(in_fn) + 1;
	    }
	}
again:
	test_len = read(in_file, inbuf+data_ofs + delta,
		SRC_BLKSIZE - data_ofs - delta);
	if (test_len <0 || (test_len+delta == 0)) break;
	delta += test_len;
	if (delta < SRC_BLKSIZE - data_ofs && test_len > 0) goto again;
	test_len = delta + data_ofs;
	inbuf[0] = test_len & 0xff;
	inbuf[1] = (test_len >> 8)  & 0xff;
	inbuf[2] = data_ofs ;
	g3len = g3_encode(inbuf, test_len, my_g3buf);
	write(out_file, my_g3buf, g3len);
   
	if (verbose==0) P((stderr,"."));
	Pv((stderr,"*** encode done, in %d data bytes,"
		" out %d g3 bytes ***\n",
		test_len - data_ofs, g3len));
#ifdef	TESTING
	g3len = add_noise(my_g3buf, g3len, nerrors);
	g3_decode(my_g3buf, g3len, srcbuf, sizeof(srcbuf));

	g3len = srcbuf[0] + 256*srcbuf[1];
	if (g3len != test_len)
	    P((stderr,"+++ Length mismatch: source %d bytes, got %d\n",
		test_len, g3len));
	else test_errs(inbuf, srcbuf, g3len);
#endif	/* TESTING */
    }
    g3len = g3_encode(inbuf, 0, my_g3buf); /*** eof page ***/
    write(out_file, my_g3buf, g3len);
    if (add_tiff) make_hdr(out_file, tot_lines);
    return 0;
}

int
main(int argc,char *argv[])
{
    char *outfn = NULL;
    int in_file = 0;	/* default: stdin */
    int out_file = -1;
    char *in_file_name="-";

    int decode = 1;
    int i;
    long t;
    long nerrors = 0;
    long written = 0;

    extern char *optarg;

    verbose = 0;
    while ((i = getopt(argc,argv,"ergdtE:vo:")) != EOF) {
	switch(i){
	case 'g':	/* raw g3 data */
	    add_tiff = 0 ;
	    break;
	case 't':	/* add TIFF header */
	    add_tiff = 1 ;
	    break;
	case 'r':	/* reverse data bits */
	    reverse = ! reverse ;
	    break;
	case 'o':	/* output file */
	    outfn = optarg;
	    break;
	case 'e':	/* encode */
	    decode = 0;
	    break;
	case 'd':	/* Decode */
	    decode = 1;
	    break;
#ifdef	TESTING
	case 'E':	/* Number of errors per block */
	    nerrors = atoi(optarg);
	    break;
#endif
	case 'v':	/* Be verbose */
	    verbose++ ;
	    break;
	default:
	    P((stderr,usage, argv[0]));
	    exit(1);
	}
    }

    argc -= optind;
    argv += optind;

    P1((stderr, "Reed-Solomon code is (%d,%d) over GF(%d)\n", NN,KK,NN+1));
    time(&t);
    srandom(t);
    init_rs();

    init_brev();

    /***
     *** remaining arguments are assumed to be filenames
     ***/
    if (decode == 0) {
	out_file = open_out(outfn) ;
	if (out_file < 0) exit(1);
    }; 
    while (in_file_name || argc>0) {
	written = 0;
	if (argc > 0 ) {
	    in_file_name = argv[0];
	    argc--;
	    argv++;
	}
	P((stderr, "File %s ",in_file_name));
	if (!strcmp(in_file_name,"-")) {
	    in_file = 0; /* stdin */
	} else
	    in_file = open(in_file_name, O_BINARY | O_RDONLY);
	if (in_file < 0) {
	    P((stderr,"Cannot open input file %s\n",argv[0]));
	}
	if (decode) {
	    written = decode_file(in_file, outfn);
	    if (written == 0) {
		P((stderr,"Warning, no data written. "
			"Try again with '-r' flag\n"));
	    }
	} else {
	    encode_file(in_file, out_file, in_file_name, nerrors);
	}
	P((stderr, " done\n"));
	in_file_name = NULL;
    }
    return 0;
}
