/*
 * mini xml parse/encode/decode for xmlrpc
 * $Id: minixml.c 350 2008-06-01 21:57:20Z luigi $
 *
 * This librA small set a functions that takes a string that resembles
 * a scripting language, and encodes it in XML.
 * Basic data types are int, string (in quotes),
 * array [ a, b, c, ...] and structs { a=one, b=two, ... }.
 * Things can be nested. Finally, you can call functions
 * foo(a, b, c)
The encode routine also returns a binary description like this:

	
 */

#include "myhdr.h"
#include <ctype.h>

/* helper function to remove blanks */
static const char *skip_blanks(const char *s)
{
	if (s) {
		while (*s && index(" \t\r\n", *s))
			s++;
	}
	return s;
}

int obj_getlen(const struct __obj *o)
{
	if (!o)
		return 0;
	switch (o->type) {
	case TY_ARRAY:
		return o->v.a[0].len;
	case TY_MAP:
		return o->v.m[0].len;
	case TY_FUNCTION:
		return o->v.f[0].len;
	default:
		return 1;
	}
}

const struct __obj *obj_get(const struct __obj *o,
	uint32_t i, const char **key)
{
	if (!o)
		return NULL;
	switch (o->type) {
	default:
		return o;
	case TY_ARRAY:
		return (i < o->v.a[0].len) ? o->v.a[0].d[i] : NULL;
	case TY_MAP:
		if (i >= o->v.m[0].len)
			return NULL;
		if (key)
			*key = o->v.m[0].d[i].key;
		return o->v.m[0].d[i].value;

	case TY_FUNCTION:
		return (i < o->v.f[0].len) ? o->v.f[0].arg[i] : NULL;
	case TY_MAP_ENTRY:
		if (key)
			*key = o->v.me[0].key_value;
		return o->v.me[0].value;
	}
}

const struct __obj *obj_getfield(const struct __obj *o, const char *key, int keylen)
{
	int i;

	if (!o || o->type != TY_MAP || !key || !*key)
		return NULL;
	if (keylen == 0)
		keylen = strlen(key);
	for (i=0; i < o->v.m[0].len; i++)
		if (!strncmp(key, o->v.m[0].d[i].key, keylen) &&
			    strlen(o->v.m[0].d[i].key) == keylen)
			return o->v.m[0].d[i].value;
	return NULL;
}

/*
 * print an object. 0 = functional, 1 = xml, 2 = debugging
 */
const char *obj_print(dynstr *dst, const struct __obj *o, enum expr_fmt mode)
{
	char *sep = "";
	int i;
	enum expr_fmt mode_in = mode;

	if (dst == NULL)
		return NULL;
	if (!o)
		return ds_data(*dst);
	if (mode == FMT_FLAT)
		mode = FMT_FUNC;

    switch (mode) {
    case FMT_FLAT: /* functional form */
    case FMT_FUNC: /* functional form */
	switch (o->type) {
	case TY_INT:
		dsprintf(dst, "%d", o->v.i[0]);
		break;
	case TY_DOUBLE:
		dsprintf(dst, "%f", o->v.d[0]);
		break;
	case TY_STRING:
		dsprintf(dst, "\"%s\"", o->v.s);
		break;
	case TY_FUNCTION:
		i = o->v.f[0].len;
		dsprintf(dst, "%s(", (const char *)&(o->v.f[0].arg[i]));
		for (i=0; i < o->v.f[0].len; i++) {
		    dsprintf(dst, "%s", sep);
		    sep = ",";
		    obj_print(dst, o->v.f[0].arg[i], mode);
		}
		dsprintf(dst, ")");
		break;
	case TY_ARRAY:
		if (mode_in == FMT_FUNC)
		    dsprintf(dst, "[");
		for (i=0; i < o->v.a[0].len; i++) {
		    dsprintf(dst, "%s", sep);
		    sep = (mode_in == FMT_FUNC) ? "," : " ";
		    obj_print(dst, o->v.a[0].d[i], mode);
		}
		if (mode_in == FMT_FUNC)
		    dsprintf(dst, "]");
		break;
	case TY_MAP:
		if (mode_in == FMT_FUNC)
		    dsprintf(dst, "{");
		for (i=0; i < o->v.m[0].len; i++) {
		    dsprintf(dst, "%s%s=", sep, o->v.m[0].d[i].key);
		    sep = (mode_in == FMT_FUNC) ? "," : " ";
		    obj_print(dst, o->v.m[0].d[i].value, mode);
		}
		if (mode_in == FMT_FUNC)
		    dsprintf(dst, "}");
		break;

	case TY_MAP_ENTRY:
		dsprintf(dst, "%s=", o->v.me[0].key);
		obj_print(dst, o->v.me[0].value, mode);
	}
	break;

    case FMT_XML: /* XML */
	switch (o->type) {
	case TY_INT:
		dsprintf(dst, "<int>%d</int>", o->v.i[0]);
		break;
	case TY_DOUBLE:
		dsprintf(dst, "<double>%f</double>", o->v.d[0]);
		break;
	case TY_STRING:
		dsprintf(dst, "<string>%s</string>", o->v.s);
		break;

	case TY_FUNCTION:
		i = o->v.f[0].len;
		dsprintf(dst, "<methodCall>\n"
			"<methodName>%s</methodName>\n<params>\n",
			(const char *)&(o->v.f[0].arg[i]));
		for (i=0; i < o->v.f[0].len; i++) {
		    dsprintf(dst, "<param>\n<value>");
		    obj_print(dst, o->v.f[0].arg[i], mode);
		    dsprintf(dst, "</value>\n</param>\n");
		}
		dsprintf(dst, "</params>\n</methodCall>\n");
		break;

	case TY_ARRAY:
		dsprintf(dst, "<array><data>\n");
		for (i=0; i < o->v.a[0].len; i++) {
		    dsprintf(dst, "<value>");
		    obj_print(dst, o->v.a[0].d[i], mode);
		    dsprintf(dst, "</value>\n");
		}
		dsprintf(dst, "</data></array>\n");
		break;

	case TY_MAP:
		dsprintf(dst, "<struct>\n");
		for (i=0; i < o->v.m[0].len; i++) {
		    dsprintf(dst, "<member>\n<name>%s</name>\n<value>",
			o->v.m[0].d[i].key);
		    obj_print(dst, o->v.m[0].d[i].value, mode);
		    dsprintf(dst, "</value>\n</member>\n");
		}
		dsprintf(dst, "</struct>\n");
		break;
	case TY_MAP_ENTRY:
		dsprintf(dst, "%s=", o->v.me[0].key);
		obj_print(dst, o->v.me[0].value, mode);
	}
	break;

#if 0
    case 2: /* debugging */
	switch (o->type) {
	case TY_INT:
		dsprintf(dst, "integer: %d\n", o->v.i[0]);
		break;
	case TY_DOUBLE:
		dsprintf(dst, "double: %f\n", o->v.d[0]);
		break;
	case TY_STRING:
		dsprintf(dst, "string len %d [%s]\n", strlen(o->v.s), o->v.s);
		break;

	case TY_FUNCTION:
		i = o->v.f[0].len;
		dsprintf(dst, "function: [%s] (%d args)\n",
			(const char *)&(o->v.f[0].arg[i]), i);
		for (i=0; i < o->v.f[0].len; i++) {
		    dsprintf(dst, "argument %d: ", i);
		    obj_print(dst, o->v.f[0].arg[i], mode);
		    dsprintf(dst, "end argument %d: ", i);
		}
		dsprintf(dst, "end function: [%s]\n",
			(const char *)&(o->v.f[0].arg[i]));
		break;

	case TY_ARRAY:
		dsprintf(dst, "array: %d entries\n", o->v.a[0].len);
		for (i=0; i < o->v.a[0].len; i++) {
		    dsprintf(dst, "entry %d: ", i);
		    obj_print(dst, o->v.a[0].d[i], mode);
		    dsprintf(dst, "end entry %d: ", i);
		}
		dsprintf(dst, "end array\n");
		break;

	case TY_MAP:
		dsprintf(dst, "map: %d entries\n", o->v.m[0].len);
		for (i=0; i < o->v.m[0].len; i++) {
		    dsprintf(dst, "member %d [%s]:", i, o->v.m[0].d[i].key);
		    obj_print(dst, o->v.m[0].d[i].value, mode);
		    dsprintf(dst, "end member %d\n", i);
		}
		dsprintf(dst, "end map\n");
		break;
	case TY_MAP_ENTRY:
		dsprintf(dst, "map entry: [%s] : ", o->v.me[0].key);
		obj_print(dst, o->v.me[0].value, mode);
		dsprintf(dst, "end map entry: [%s] : ", o->v.me[0].key);
	}
	break;
#endif /* case 2 */
    }
	return ds_data(*dst);
}

/*
 * allocation and free routines for objects.
 * The allocator takes a size and optionally an initializer.
 * The free routine does a recursive free.
 */
struct __obj *obj_free(struct __obj *obj)
{
	static char *obj_names[TY_FUNCTION+1] = {
		"TY_INT", "TY_DOUBLE", "TY_STRING",
		"TY_ARRAY", "TY_MAP", "TY_MAP_ENTRY", "TY_FUNCTION"
	};
	char *ss = (obj && obj->type <= TY_FUNCTION ? obj_names[obj->type] : "??");
	int i;

	if (0) fprintf(stderr, "%s %p ty %d %s\n", __FUNCTION__, obj, obj ? obj->type : -1, ss);
	if (obj == NULL)
		return NULL;
	switch (obj->type) {
	case TY_INT:
		break;	// nothing special, all data in one chunk
	case TY_DOUBLE:
		break;	// nothing special, all data in one chunk
	case TY_STRING:
		//fprintf(stderr, "%s string value [%s]\n", __FUNCTION__, obj->v.s);
		break;	// nothing special, all data in one chunk

	case TY_MAP_ENTRY:
	    {
		struct __map_entry *me = obj->v.me;
		if (0) fprintf(stderr, "%s map entry for [%s]: %p\n", __FUNCTION__,
			me->key_value, me->value);
		me->value = obj_free(me->value);
		if (me->key && me->key != me->key_value)
		    free(me->key);
		me->key = NULL;
	    }
		break;

	case TY_FUNCTION:
		for (i = 0; i < obj->v.f[0].len; i++) {
			// fprintf(stderr, "argument %d %p\n", i, obj->v.f[0].arg[i]);
			obj->v.f[0].arg[i] = obj_free(obj->v.f[0].arg[i]);
		}
		break;

	case TY_ARRAY:
		// fprintf(stderr, "array of size %d\n", obj->v.a[0].len);
		for (i = 0; i < obj->v.a[0].len; i++)
			obj->v.a[0].d[i] = obj_free(obj->v.a[0].d[i]);
		break;

	case TY_MAP:
	    {
		struct __map_entry *me = obj->v.m[0].d;
		// fprintf(stderr, "map of size %d\n", obj->v.m[0].len);
		for (i = 0; i < obj->v.m[0].len; i++) {
			me[i].value = obj_free(me[i].value);
			if (me[i].key && me[i].key != me[i].key_value)
				free(me[i].key);
			me[i].key = NULL;
		}
	    }
		break;
	}
	free(obj);
	return NULL;
}

/*
 * Allocates and fills an object.
 * For string and map entry, copies the string in the descriptor.
 * For int, len is the value.
 * For double, src points to the value.
 * For arrays, copy also the object pointers.
 * For maps, copy also the map entries, (and frees the leftover stuff).
 * For functions, only copy the functor. Need extra argument for the pointers.
 */
struct __obj *obj_calloc(enum __xml_types type, int32_t len, const void *src)
{
    int l = 0;
    struct __obj *o;

    switch (type) {
    case TY_INT:
	l = sizeof(int32_t);
	break;

    case TY_DOUBLE:
	l = sizeof(double);
	break;

    case TY_STRING:
    case TY_MAP_ENTRY:
	l = len + 1;	/* length of the string. Add one extra byte for NUL */
	if (src) {
	    int n = strlen((const char *)src) + 1;
	    if (n > l)
		l = n;
	}
	if (type == TY_MAP_ENTRY)
	    l += sizeof(struct __map_entry);
	break;

    case TY_ARRAY:
	l = len * sizeof(struct __obj) + sizeof(struct __array_data);
	break;

    case TY_MAP:
	l = len * sizeof(struct __map_entry) + sizeof(struct __map_data);
	break;

    case TY_FUNCTION:
	l = len * sizeof(struct __obj) + sizeof(struct __func_data) + 1 /* name */;
	if (src)
	    l += strlen((const char *)src);
	break;
    }
    l += sizeof(struct __obj);
    o = calloc(1, l);
    if (!o)
	return NULL;
    o->type = type;
    switch (type) {
    case TY_INT:
	o->v.i[0] = len;
	break;

    case TY_DOUBLE:
	o->v.d[0] = *(const double *)src;
	break;

    case TY_STRING:
	if (src)
	    strcpy(o->v.s, (const char *)src);
	break;

    case TY_ARRAY:
	o->v.a[0].len = len;
	if (src)
	    bcopy(src, o->v.a[0].d, len * sizeof(struct __obj *));
	break;

    case TY_MAP_ENTRY: /* src is the key_value, obj must be copied */
	if (src) {
	    strcpy(o->v.me[0].key_value, (const char *)src);
	    o->v.me[0].key = o->v.me[0].key_value;
	}
	break;

    case TY_MAP:
	o->v.m[0].len = len;
	if (src) { // src is an array of TY_MAP_ENTRY
	    int i;
	    struct __obj **me = (struct __obj **)(void *)src;
	    for (i = 0; i < len; i++) {
		if (me[i]->type != TY_MAP_ENTRY)
			fprintf(stderr, "XXX XXX bad object ty %d %d %p to %s TY_MAP\n",
				me[i]->type,
				i, me[i], __FUNCTION__);
		o->v.m[0].d[i].value = me[i]->v.me[0].value;
		me[i]->v.me[0].value = NULL;
		o->v.m[0].d[i].key = strdup(me[i]->v.me[0].key_value);
		obj_free(me[i]);
	    }
	}
	break;

    case TY_FUNCTION:
	o->v.f[0].len = len;
	{ char *ss = (char *)&(o->v.f[0].arg[o->v.f[0].len]);
	strcpy(ss, (const char *)src);
	}
	// fprintf(stderr, "XXX %s TY_FUNCTION remember to copy the args\n", __FUNCTION__);
	break;
    }
    return o;
}

/*
 * Individual routines take a dynstr pointer and a pointer to
 * the next char to parse. They return a pointer to the next
 * char to parse.
 */
/*
 * encode_name() copies a string into the obj
 * with the usual quoting and escaping.
 * Returns the next non-whitespace char after the string.
 * Leading whitespace is skipped, trailing whitespace is not considered
 * part of the string.
 * Any char in [a-zA-Z0-9_.] is considered valid, others
 * must be escaped or quoted.
 * XXX later, add encoding
 */
static const char * encode_name(const char *d, struct __obj **obj)
{
	const char *cur;
#define NO_QUOTE ' '
	char quote = NO_QUOTE;
	int escape = 0;
	dynstr __s = NULL;
	dynstr *s = &__s;

	if (obj)
	    *obj = NULL;
	d = skip_blanks(d);
	for (cur = d; *cur; cur++) {
		if (escape) {
			ds_append(s, cur, 1);
			escape = 0;
			continue;
		}
		if (*cur == '\\') {
			escape = 1;
			continue;
		}
		if (quote != NO_QUOTE) { /* anything until closing quote */
			if (*cur == quote) // exit from quote mode
				quote = NO_QUOTE;
			else
				ds_append(s, cur, 1);
		} else if (*cur == '"' || *cur == '\'') {
			quote = *cur;
		} else if (isalnum(*cur) || (*cur && index("_.:/+-", *cur)) ) {
			ds_append(s, cur, 1);
		} else {
			break;
		}
	}
	/* skip the blanks */
	cur = skip_blanks(cur);
	if (0) fprintf(stderr, "%s [%s] gets [%s] rest [%s]\n", __FUNCTION__,
		d, ds_data(*s), cur);
	if (obj) {
		char *endp = NULL;
		long l = strtol(ds_data(*s), &endp, 10);
		if (*endp == '\0') { // a valid int
		    *obj = obj_calloc(TY_INT, l, NULL);
		} else {
		    *obj = obj_calloc(TY_STRING, 0, ds_data(*s));
		}
	}
	ds_free(*s);
	return cur;
}

static const char *_expr_parse(const char *s, struct __obj **obj);

/*
 * Helper routine to encode an argument for a list.
 *  'hint' is a hint on the type.
 * { -> struct (expect name-pair values)
 * ' ' -> everything else
 */
static const char *encode_arg(const char *d, char hint, struct __obj **obj)
{
	char c = *d;
	int closing_paren = 0;

	if (obj)
	    *obj = NULL;

	//fprintf(stderr, "-- %s arg [%s]\n", __FUNCTION__, d);
	if ( index("0123456789+-.", c)) { // a number
		c = '0';
	} else if (isalnum(c) || c == '"' || c == '\'') {
		c = 's';
	} else if (c == '[') {	// array
		closing_paren = 1;
	} else if (c == '{') { // struct
		closing_paren = 1;
	} else {
		c = '?';
		fprintf(stderr, "unknown arg [%s]\n", d);
	}
	if (hint == '{') { // struct, we expect 'name = value'
		struct __obj *entry_name, *entry;

		if (c != '0' && c != 's') {
			fprintf(stderr, "invalid field name %s\n", d);
			return d+1;
		}
		d = encode_name(d, obj ? &entry_name : NULL);
		if (*d == '=') {
			d = _expr_parse(skip_blanks(d+1), obj ? &entry : NULL);
		}
		if (obj) {
			*obj = obj_calloc(TY_MAP_ENTRY, 0, entry_name->v.s);
			if (*obj)
				(*obj)->v.me[0].value = entry;
			else
			    obj_free(entry);
			obj_free(entry_name);
		}
	} else if (hint == '(') {
		d = _expr_parse(d, obj);
	} else if (hint == '[') {
		d = _expr_parse(d, obj);
	} else { // scalar or function
		d = _expr_parse(d, obj);
	}
	return d;
}

/*
 * This function takes a compactly encoded XMLRPC call
 * and produces a suitable XML string.
 * Returns a pointer to the first non-blank char after the parsed text.

   MyFunc()
	--> <methodCall><methodName>MyFunc</MethodName>
		<params></params></methodCall>

 * By extension, it also encodes a plain object.
 * The last argument, if present, returns a pointer to the struct __obj
 * representation of the argument.
 */
static const char *_expr_parse(const char *s, struct __obj **obj)
{
    const char *orig = skip_blanks(s);
    char c;	/* initial char, or type of the object */

    s = orig;
    if (obj)
	*obj = NULL;
    if (!s || !*s)
	return s;
    c = *s;
    if (c != '[' && c != '{' && c != '(') {
	/* scalar or function. First take the name, then look for args. */
	s = encode_name(s, obj); /* function name */
	c = *s;
	if (c != '(') {	/* no arguments, just a name */
	    c = ' '; /* signal skip arguments */
	}
    }
    if (c != ' ') {
	dynstr args = NULL;	// build the array of arguments
	int argslen;
	// const char *startarg = s; // debugging
	s = skip_blanks(s+1); // skip opening parenthesis
	// fprintf(stderr, "--%s args %s\n", __FUNCTION__, s);
	if (*s != ']' && *s != '}' && *s != ')') {
	    for (;;) {
		struct __obj *entry;
		s = encode_arg(s, c, obj ? &entry : NULL);
		if (obj)
		    ds_append(&args, (void *)&entry, sizeof(entry));
		if (*s != ',')
		    break;
		s = skip_blanks(s+1);
	    }
	}
	argslen = ds_len(args) / sizeof(struct __obj *);
	// fprintf(stderr, "-- arglist %c %s size %d\n", c, startarg, argslen);
	// fprintf(stderr, "+++ open '%s' close '%s'---\n", orig, s);
	s = skip_blanks(s+1); // skip closing parenthesis
	if (c == '[') {
	    if (obj)
		(*obj) = obj_calloc(TY_ARRAY, argslen, ds_data(args));
	} else if (c == '(') {
	    if (obj) {
		struct __obj *functor = *obj;
		(*obj) = obj_calloc(TY_FUNCTION, argslen, functor->v.s);
		if (*obj)
		    bcopy( ds_data(args), (*obj)->v.f[0].arg, ds_len(args));
		obj_free(functor);
	    }
	} else if (c == '{') {
	    if (obj)
		(*obj) = obj_calloc(TY_MAP, argslen, ds_data(args));
	}
	ds_free(args);
    }
    return skip_blanks(s);
}

/*
 * extract the name of the xml token <foo>
 * Sets 'end' to 1 if the token starts or ends with '/>'
 */
static const char *xml_token(const char *s, const char **tok, int *l, int *end)
{
	*end = 0;
	*l = 0;
	*tok = NULL;
	s = skip_blanks(s);
	if (*s != '<')
		return s;
	s = skip_blanks(s+1);
	if (*s == '/') {
		*end = 1;
		s++;
	}
	*tok = s;
	while (*s && (isalnum(*s) || *s == '_')) {
		s++;
		(*l)++;
	}
	while (*s && *s != '>')
		s++;
	if (*s) {
		if (s[-1] == '/')
			*end = 1;
		s = skip_blanks(s+1);
	}
	return s;
}

static const char *xml_parse(const char *s, struct __obj **obj)
{
	const char *p;
	const char *orig = s;
	int l, end;
	char buf[50];
	struct __obj *my_obj;

	if (obj == NULL) {
		fprintf(stderr, "%s requires a non null obj\n", __FUNCTION__);
		obj = &my_obj;
	}

	//fprintf(stderr, "-- in %s [%s]\n", __FUNCTION__, s);
again:
	s = skip_blanks(s);
	if (*s != '<')
		return encode_name(s, obj);
	s = xml_token(s, &p, &l, &end);
	if (!p)
		return s;
	if (l == 4 && !strncasecmp(p, "?xml", 4)) {
		fprintf(stderr, "start found %s", orig);
		goto again;
	}
	if (l > sizeof(buf) - 1)
		l = sizeof(buf) - 1;
	strncpy(buf, p, l);
	buf[l] = '\0';
#define M(x) (!strncasecmp(p, x, strlen(x)))
	if (M("methodcall")) {
		struct __obj *functor, *args;
		int argslen;
		s = xml_parse(s, &functor); // methodname
		s = xml_parse(s, &args);	// these come as an array
		argslen = args->v.a[0].len;
		/* now pack functor and arg */
		(*obj) = obj_calloc(TY_FUNCTION, argslen, functor->v.s);
		if (*obj) {
		    bcopy(args->v.a[0].d, (*obj)->v.f[0].arg,
			argslen * sizeof(struct __obj *));
		    bzero(args->v.a[0].d, argslen * sizeof(struct __obj *));
		}
		obj_free(args);
		obj_free(functor);
	} else if (M("methodname")) {	/* pack as a string */
		s = xml_parse(s, obj);
	} else if (M("array")) {
		s = xml_parse(s, obj);
	} else if (M("member")) {
		struct __obj *name, *val;
		s = xml_parse(s, &name); // name
		s = xml_parse(s, &val); // value
		*obj = obj_calloc(TY_MAP_ENTRY, 0, name->v.s);
		if (*obj)
			(*obj)->v.me[0].value = val;
		else
			obj_free(val);
		obj_free(name);
	} else if (M("struct")) {
		dynstr args = NULL;
		int n;
		while (*s == '<' && s[1] && s[1] != '/') {
			struct __obj *entry;
			s = xml_parse(s, &entry);
			ds_append(&args, (const char *)&entry, sizeof(entry));
		}
		n = ds_len(args)/sizeof(struct __obj *);
		*obj = obj_calloc(TY_MAP, n, ds_data(args));
		ds_free(args);
	} else if (M("data")||M("params")) {
		dynstr args = NULL;
		int argslen;
		while (*s == '<' && s[1] && s[1] != '/') {
			struct __obj *entry;
			s = xml_parse(s, &entry);
			ds_append(&args, (const char *)&entry, sizeof(entry));
		}
		argslen = ds_len(args)/sizeof(struct __obj *);
		*obj = obj_calloc(TY_ARRAY, argslen, ds_data(args));
		ds_free(args);
	} else if (M("nil")) {	/* map to empty string */
		*obj = obj_calloc(TY_STRING, 0, NULL);
	} else if (M("string")) {
		const char *begin = s;
		while (*s && *s != '<')
			s++;
		*obj = obj_calloc(TY_STRING, s - begin, NULL);
		bcopy(begin, (*obj)->v.s, s - begin);
		(*obj)->v.s[s - begin] = '\0';
	} else { /* anything else, just dump the content */
		s = xml_parse(s, obj);
	} 
	if (!end)
		s = xml_token(s, &p, &l, &end);
	return s;
}

const char *expr_parse(const char *s, struct __obj **o, enum expr_fmt mode)
{
	switch (mode) {
	default:
		return s;
	case FMT_FUNC:
		return _expr_parse(s, o);
	case FMT_XML:
		return xml_parse(s, o);
	}
}

/*
 * apply a filter to a reply:
 *
 * .length()	-> returns object length
 * .type()	-> returns object type
 * .keys()	-> returns all keys
 * []		-> iterate on all elements
 * {}		-> return all keys
 * [n,...,m]	-> iterate on selected elements
 * {n,...,m}	-> iterate on selected keys
 * empty:	-> print object.
 */
const char *obj_filter(dynstr *s, const struct __obj *o, const char *filter, enum expr_fmt mode)
{
    int n;
    const char *nf;	/* next filter */

    if (!o || !filter)
	return NULL;

    n = obj_getlen(o);
    //dsprintf(s, "obj %p ty %d len %d\n", o, o->type, n);
    switch(*filter++) {
    case '\0':
	obj_print(s, o, mode);
	break;

    case '.':
	if (!strncmp(filter, "length()", 7)) {
	    dsprintf(s, "%d", n);
	} else if (!strncmp(filter, "type()", 6)) {
	    dsprintf(s, "%d", o->type);
	} else if (!strncmp(filter, "xml()", 5)) {
	    obj_print(s, o, FMT_XML);
	} else if (!strncmp(filter, "func()", 6)) {
	    obj_print(s, o, FMT_FUNC);
	} else if (!strncmp(filter, "keys()", 6)) {
	    int i;
	    if (o->type != TY_MAP) {
		dsprintf(s, "\"\"");
		break;
	    }
	    for (i=0; i < n; i++)
		dsprintf(s, "%s ", o->v.m[0].d[i].key);
	}
	// else other methods
	break;

    case '[':
    case '{':
	if (*filter == ']' || *filter == '}') { // all elements
	    int i;
	    for (i = 0; i < n; i++) {
		obj_filter(s, obj_get(o, i, NULL), filter + 1, mode);
		dsprintf(s, " ");
	    }
	    break;
	}
	// try selected elements on the portion past the ']' or '}'
	for (nf = filter; *nf && !index("}]", *nf) ; nf++)
	    ;
	if (*nf)
	    nf++;
	for (;;) {
	    int i;
	    char *endp = NULL;
	    enum expr_fmt mode2 = mode;

	    if (!*filter || *filter == ']' || *filter == '}')
		break;
	    i = strtol(filter, &endp, 10);
	    if (endp != filter) {
		if (*endp == '=')
		    mode2 = FMT_FLAT;
		filter = endp;
		obj_filter(s, obj_get(o, i, NULL), nf, mode2);
		dsprintf(s, " ");
	    } else {
	    	const char *begin = filter;
		// collect an index XXX use the string routines
		while (*filter && !index("}]=,", *filter))
		    filter++;
		if (begin != filter) {
		    if (*filter == '=')
			mode2 = FMT_FLAT;
		    obj_filter(s, obj_getfield(o, begin, filter - begin), nf, mode2);
		    dsprintf(s, " ");
		}
	    }
	    while (*filter && index(" \t=,", *filter))
		filter++;
	}
	break;
    }
    // dsprintf(s, "end obj %p -\n", o);
    return ds_data(*s);
}
/*--- end of file ---*/
