/*
 * receiver.c -- receiver routines for rmdp.
 *
 * This file is part of
 * 
 * rmdp -- Reliable Multicast data Distribution Protocol
 * 
 * (C) 1996-1998 Luigi Rizzo and Lorenzo Vicisano
 *     (luigi@iet.unipi.it, vicisano@cs.ucl.ac.uk)
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Luigi Rizzo,
 *      Lorenzo Vicisano and other contributors.
 * 4. Neither the name of the Authors nor the names of other contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * receiver main routines
 *
 * The receiver first sends requests with exponential backoff.
 * When data are missing, send continuation requests, again with
 * exponential backoff.
 */
/***
 *** TODO

- implement randomization of CONT requests
- implement exponential backoff on requests.
- make CONT request contain also the URL and original parameters;
- implement checkpoint/recovery of a session
- make the session id a fingerprint of the source.
- add a "-n" flag so that the output file name is got from the URL
- support a change of address/(session ID ?) upon a continuation request.

 ***
 ***/

#include "rmdp.h"
#include "net.h"

#define DEFAULT_RATE	(64*1024)
#define LOGGING

#ifdef LOGGING
#define LOG(x)	x;
FILE *logf;
#else
#define LOG(x)
#endif

EVLIST *eventlist;		/* callback attach. and timers */

#define ID_GETRESPONSE	23
#define ID_ASK_CONT	34
#define ID_SENDREQ	100
#define ID_INITRX	101
#define ID_GETDATA	104
#define MAX_RETRY	20

/***
 *** all these are `per session', and should be insulated, in case...
 ***/
char req_buf[1024];		/* requests to the server */

int ctrls;			/* control (out) socket */

int response_received;		/* set when the response is received */
int seen_last_round = 0;
int cont_pending = 0;		/* have a cont request pending */
int datas;			/* data (in) socket */
char response[2048];

void *fec_code ;

struct rx_session {
    void ***alldata;		/* received packets... */
    int **whichdata, *countdata;	/* info on received packets */
    n32	d_addr;			/* multicast address for data */
    n16	d_port;

    n32 myid;			/* appl. level id of this receiver */
    n32 myrequest;			/* id of the request issued */
    n32 session_id;			/* id of the assigned session */
    int B, N;			/* encoding parameters */
    int data_pkt_len;
    int missingblocks, duppackets;
    char outfn[256];
    ui32 fpkts, flength;
    ui32 starttime;
    int	data_rate;

    ui16 last_r;		/* last received */
    ui16 first_sn ;		/* first sequence number */
    ui32 last_time_s, last_time_r;
    ui32 tot_lost;
    int retries;
} rxs;

void decode();
int decode_block(int i);
int get_data(void *p);
int allocation();
int init_reception(void *p);
int get_response(void *p);

/* 
 * various auxiliary functions **
 */

void
badcmdline(char *arg)
{
    fprintf(stderr,
	"Bad argument %s\nCommand line:\n\t rmdp [options]\n"
	"options:\n"
	"\t-c <M-cast control address/port>\n"
	"\t-t <receiver ttl>\n"
	"\t-u -- use unicast UDP\n"
	"\t-r rate -- desired rx rate (bit/s)\n"
 	"\t<output file>\n", arg);
    exit(1);
}

#define REQ_TIMEOUT	5000000		/* 5 sec. */
#define S_REQ_ARG struct s_req_arg
S_REQ_ARG {
    int s;
    int req_len;
    u_char *request;
};

int
ask_cont(void *dummy)
{
    char *r;
    cont_pending = 0 ;

    if (seen_last_round) {
	DDB(fprintf(stderr, "\nsending continue request %d\n", rxs.retries));
	r = add_magic(req_buf);
	r = add_opt1(r, RMDP_T_R_CONT);
	r = add_opt4(r, RMDP_T_S_ID, rxs.session_id);
	if (send(ctrls, req_buf, r - req_buf, 0 ) == -1 ) {
	    perror("cont_request: send");
	}
    } else {
	DEB(fprintf(stderr, "ask_cont does not send, seen_last_round=0\n"));
	rxs.retries = 0;
    }
    return 0;
}

int
send_request(S_REQ_ARG * s_arg)
{
    if (response_received) {
	return 0;
    }
    DDB(fprintf(stderr,
	"send_request: sending req of size %d on desc %d, retry %d\n",
	s_arg->req_len, s_arg->s, rxs.retries) );
    if (send(s_arg->s, (char *)(s_arg->request),
	s_arg->req_len, 0) == -1) {
	perror("send_request: send");
	exit(1);
    }

    if (rxs.retries++ < MAX_RETRY)
	settimer(eventlist, ID_SENDREQ, REQ_TIMEOUT, (CALLBACK)send_request,
		(void *)s_arg, 0);
    else {
	fprintf(stderr,"Sorry, server not responding\n");
	exit(1);
    }
    return(0);
}

int
get_response(void *dummy)
{
    struct sockaddr_in srcaddr;
    int ch = showselected(eventlist);
    int len, srcaddrlen=sizeof(srcaddr);
    ui32 rate;

    n32 c_id = 0, r_id = 0, s_id = 0;
    int type = 0;
    u_char *p ;
    char *errp = NULL ;

    DEB(fprintf(stderr, "get_response handling response from fd = %d\n",
	ch));
    if ((len = recvfrom(ch, (char *)&response, sizeof(response), 0,
	(struct sockaddr *)&srcaddr, &srcaddrlen))<=0)
	perror("get_response: recvfrom");

    DEB(fprintf(stderr, "--- pkt len %d\n", len) );

    rxs.d_addr = 0 ;
    rxs.d_port = 0;
    if (!check_magic(response, len))
	    return 0;
    p = response + 4;
    len -= 4;
    fprintf(stderr,"\nREPLY: ");
    while (len > 0) {
        u_char opt = *p;
        i16 l = getoptlen(p, len) ;
        if (l == 0) return 0;
        len -= l;

        switch(opt) {
	case RMDP_T_RESP:
	    type = RMDP_T_RESP ;
	    break;
        case RMDP_T_PAD : 
            fprintf(stderr, " pad,");
            break;
        case RMDP_T_END :
            fprintf(stderr, " <end>\n");
            len = 0;
            break ;
        case RMDP_T_C_ID :
            bcopy(p+2, &c_id , 4);
            fprintf(stderr, " c_id 0x%08x,", c_id);
            break;
	case RMDP_T_R_RATE : {
		char c=' ';
	        struct timeval tv;

		bcopy(p+2, &rate , 4);
		rxs.data_rate = rate = ntohl(rate);
		if (rate > 8192) {
		    rate /= 1024;
		    c='K';
		}
		fprintf(stderr, " rate %d %cbit/s,", rate, c);
		tv.tv_sec = 5*8192/(1+rxs.data_rate);
		tv.tv_usec = 0;
		if (tv.tv_sec < 3)
		    tv.tv_sec = 3;
		else if (tv.tv_sec > 20)
		    tv.tv_sec = 20;
		set_default_timeout(eventlist, &tv);
	    }
            break;
	case RMDP_T_R_ID :
            bcopy(p+2, &r_id , 4);
            fprintf(stderr, " r_id 0x%08x,", r_id);
            break;
	case RMDP_T_S_ID :
            bcopy(p+2, &s_id , 4);
            fprintf(stderr, " s_id 0x%08x,", s_id);
            break;
        case RMDP_T_D_ADDR :
            bcopy(p+2, &rxs.d_addr , 4);
            fprintf(stderr, "\n\tdata addr %s,",
                inet_ntoa(*(struct in_addr *)&rxs.d_addr) );
            break;
        case RMDP_T_D_PORT :
            bcopy(p+2, &rxs.d_port , 2);
            fprintf(stderr, " data port %d,", ntohs(rxs.d_port)  );
            break;
        case RMDP_T_ERRMSG :
	    errp = p + 3;
	    break ;
	default :
            fprintf(stderr, "option 0x%02x len %d unknown\n", opt, l);
            break ;
        }   
        p += l;
    }
    if ( ! ( (type == RMDP_T_RESP) &&
	    (c_id == rxs.myid) && (r_id == rxs.myrequest) ) )
	return 0 ; /* not good or not mine! */
    if ( ! ( rxs.d_port && rxs.d_addr && s_id ) ) {
	if (!errp)
	    errp = "Server denied request";
	fprintf(stderr, "Data unavailable, reason: %s\n", errp) ;
	exit ( 0 );
    }

    response_received = 1;
    deletechan(eventlist, ch);

    /* use in net. order... XXX must handle s_id == 0 */
    rxs.session_id = s_id;

    rxs.B = rxs.duppackets = 0;
    datas = openrsock(rxs.d_addr, &rxs.d_port);
    insertchan(eventlist, datas, ID_GETDATA, get_data, NULL);
    return(0);
}

/*
 * now allocation does not purge fixed fields of rxs, so
 * the session can be easily resumed.
 */
int
allocation()
{
  int done, i, j;

    rxs.retries = 0;
    rxs.last_r = -1;
    rxs.tot_lost = 0;
    rxs.alldata = calloc(rxs.B+1, sizeof(void *));
    rxs.countdata = (int *)calloc(rxs.B, sizeof(int));
    rxs.whichdata = (int **)calloc(rxs.B+1, sizeof(void *));
    for (done=0, i=0; i <= rxs.B ; i++) {	/* one more for decoding... */
	rxs.whichdata[i] = (int *)calloc(rxs.N, sizeof(int));
	rxs.alldata[i] = (void **)calloc(rxs.N, sizeof(void *));
	for (j=0; j<rxs.N; j++) {
	    rxs.whichdata[i][j] = -1;
	    rxs.alldata[i][j]  = (void *)calloc(1, rxs.data_pkt_len);
	}
    }
    return(0);
}

int
get_data(void *p)
{
  int i;
  int chan, len, srcaddrlen, pkt_ix, blk_ix;
  struct sockaddr_in srcaddr;
  DATA_P packet;
  ui16 this_sn;
  ui32 this_time;
  ui32 this_lost;

    chan = showselected(eventlist);
    DEB(fprintf(stderr, "incoming data...\n"))
    srcaddrlen = sizeof(srcaddr);
    if ((len = recvfrom(chan, (char *)&packet, sizeof(packet), 0,
	(struct sockaddr *)&srcaddr, &srcaddrlen))<=0)
	perror("receiver: recvfrom");
    DEB(fprintf(stderr, "got %d bytes...\n", len))

    /* XXX need some checks */
    if ( packet.type != RMDP_DATA_T || packet.seid != rxs.session_id ) {
	DEB(fprintf(stderr, "get_data: incorrect data packet, discarding\n"));
	if (packet.seid != rxs.session_id ) {
	    DEB(fprintf(stderr, "(different sess. id (%x != %x)\n",
			packet.seid, rxs.session_id));
	}
	return(0);
    }

    /*** compute timing statistics ***/
#ifdef DO_TIMING
    this_time = timestamp_u();
    this_sn = ntohs(packet.sn);

    this_lost = (this_sn-rxs.last_r-1) & 0xffff ;
    rxs.tot_lost += this_lost ;
    /*
     *  avg delta (sender), avg delta(receiver), missing, tot_miss,
     *		rx time, tx time
     */
    { static int first_time = 1;
    if (first_time == 1)
	first_time = 0;
    else
    DEB(fprintf(logf, "%8lu %8lu %3d %3d 0x%04x %u %lu\n",
	(ntohl(packet.timestamp)-rxs.last_time_s)/(this_lost+1),
	(this_time-rxs.last_time_r)/(this_lost+1),
	this_lost, rxs.tot_lost,
	this_sn,
	this_time, ntohl(packet.timestamp)));
    }

    rxs.last_r = this_sn ;
    rxs.last_time_r = this_time;
    rxs.last_time_s = ntohl(packet.timestamp);
#endif

    DEB(fprintf(stderr, "pkt %lu sess %d from: %lx miss %d\n",
	ntohs(packet.sn), ntohs(packet.seid),
	srcaddr.sin_addr.s_addr, rxs.missingblocks))

    /* see what we have */
    pkt_ix = ntohs(packet.pkt_ix);
    blk_ix = ntohs(packet.blk_ix);

    if (packet.flags & RMDP_F_LAST_ROUND) {
	seen_last_round = 1;
	if (!cont_pending) {
	    fprintf(stderr,"... last round, schedule a cont_request\n");
	    cont_pending = 1;
	    settimer(eventlist, ID_ASK_CONT, 1000000 /* cont_timeout */,
		(CALLBACK)ask_cont, NULL, 0);
	}
    } else
	seen_last_round = 0;
    if (rxs.B == 0 ) { /* must allocate data buffers */
	rxs.N = ntohs(packet.the_N);
	rxs.flength=ntohl(packet.datasize);
	rxs.data_pkt_len = ntohs(packet.packetsize);
	rxs.fpkts = (rxs.flength+rxs.data_pkt_len-1)/rxs.data_pkt_len;
	rxs.B = (rxs.fpkts+rxs.N-1)/rxs.N;
	DDB(fprintf(stderr,"--- first packet, "
		"must allocate things (len %d B %d)\n", rxs.flength, rxs.B);)
	DDB(fprintf(stderr,"--- got B %d, k %d\n", rxs.B, rxs.N));
	rxs.missingblocks = rxs.B;
	rxs.duppackets = 0;
	if ( (rxs.B > 1000) || (rxs.flength > 10000000) ) {
		fprintf(stderr, "something wrong with data sizes,"
		" len 0x%08x B %d\n", rxs.flength , rxs.B);
		exit(1);
	}
	allocation();
	rxs.starttime = timestamp_m();
	rxs.first_sn = this_sn ;
    }

    rxs.retries = 0;
    if (rxs.countdata[blk_ix] >= rxs.N) {
	rxs.duppackets++;
	DDB(fprintf(stderr, "\nDUP! blk %d have already %d, %d totdup\r",
	    blk_ix, rxs.countdata[blk_ix] , rxs.duppackets))
    } else {
	int valid = 1;
	for (i=0; i < rxs.countdata[blk_ix]; i++)
	    if (pkt_ix == rxs.whichdata[blk_ix][i]) {
		valid = 0;
		DEB(fprintf(stderr, "not valid.. blk %d pkt %d\n",
		    blk_ix, pkt_ix))
		break;
	    }
	if (valid) {
	    rxs.whichdata[blk_ix][rxs.countdata[blk_ix]] = pkt_ix;
	    bcopy(packet.data, rxs.alldata[blk_ix][rxs.countdata[blk_ix]],
	        rxs.data_pkt_len);
	    rxs.countdata[blk_ix]++; 
	    DDB(fprintf(stderr, "got blk %4d pkt %4d (%4d pkts); "
		    "miss %4d pkts %4d blks    \r",
		    blk_ix, pkt_ix, rxs.countdata[blk_ix],
		    rxs.N-rxs.countdata[blk_ix],rxs.missingblocks))
	    if (rxs.countdata[blk_ix] == rxs.N) {
	        rxs.missingblocks--;
		decode_block(blk_ix);
	    }
	}
    }

    /* end reception ? */
    if (rxs.missingblocks == 0) {
	ui32 endtime;
	int j ;
	endtime = timestamp_m();
	deletechan(eventlist, chan);
	DDB(printf("Stats: %d useful, %d dup\n"
	    "in %d sec. (%f pkts/sec)\n",
	    rxs.B*rxs.N, rxs.duppackets, (endtime-rxs.starttime)/1000,
	    (double)rxs.B*rxs.N/(1+(endtime-rxs.starttime)/1000)));
	decode();
	/* XXX must free data structures */
	if (rxs.last_r <= rxs.first_sn)
	    i = (rxs.last_r + 65536 - rxs.first_sn) ;
	else
	    i = (rxs.last_r - rxs.first_sn) ;
	i++ ; /* must be incremented to count sent pkts! */
	j = rxs.B*rxs.N + rxs.duppackets ;
	    
	DDB(fprintf(stderr, "rx %d out of %d pkts (loss rate %8.2f %%)\n",
		j, i, (double)(100*(i-j))/(double)i );)
    }

    return(0);
}

void
decode()
{
    if (rxs.outfn[0]) {
	int i, j, len = rxs.data_pkt_len;
	FILE *of;
	if (NULL==(of=fopen(rxs.outfn, "w"))) {
	    perror("fopen");
	    exit(1);
	}
	for (i=0; i<rxs.B; i++) {
	    for (j=0; j<rxs.N && len > 0; j++) {
		len=rxs.data_pkt_len;
		if (rxs.flength < len) len=rxs.flength;
		rxs.flength -= len;
		fwrite(rxs.alldata[i][j], len, 1, of);
	    }
	}
	fclose(of);
    }
}

int
decode_block(int i)
{
    if (fec_code == NULL)
	fec_code = fec_new(rxs.N, 256 /* XXX */ );
    fec_decode(fec_code, rxs.alldata[i], rxs.whichdata[i], rxs.data_pkt_len);
    fprintf(stderr, "\n++ block %d done\n", i);
    return 0 ;
}

int
main(int argc, char *argv[])
{
    int i, id;
    n32 myaddr;
    struct sockaddr_in s;

    char ctrladdr[ADDRLEN];
			/* data & control m-cast addresses */
    n16 cport = 0;
    n32 caddr = 0;
			/* data & control ports # (net order) */
    int reqin;		/* request (in) udp sock */
    n16 reqin_p;		/* request (in) udp port (net order) */
    ui8 ttl = TTL;
    char req_str[512];
    int unicast = 0 ;

    S_REQ_ARG send_r_a;

    char *r;


    rxs.retries = 0 ;
    bzero(&rxs, sizeof(rxs) );
    strcpy(rxs.outfn,"fileout");
    rxs.data_rate = DEFAULT_RATE ;

    /* parse command line... */
    while ((i = getopt(argc,argv,"ur:t:d:c:o:")) != EOF) {
        switch(i) {
	case 'u': /* unicast reply */
	    unicast = 1;
	    break;
	case 'r': /* data rate */
	    { int l = strlen(optarg);
	      char c=' ';
	      if (l>0) c=optarg[l-1];
	    rxs.data_rate = atoi(optarg) ;
	    if (c=='K' || c=='k')
		rxs.data_rate *= 1024;
	    else if (c=='M')
		rxs.data_rate *= 1024*1024;
	    }
	    break;
        case 'c': /* command addr/port */
	    if (set_link_addr(optarg, ctrladdr, ADDRLEN, &cport, "control")<0)
		badcmdline(optarg);
            break;  
        case 't': /* TTL */
            ttl = atoi(optarg);
            if (ttl <= 0 || ttl > 255) badcmdline(optarg);
            break;
        case 'o': /* output file */
	    strncpy(rxs.outfn, optarg, sizeof(rxs.outfn));
	    break;
        }
    }
    if (cport == 0)
	set_link_addr(DEFAULT_CPORT, ctrladdr, ADDRLEN, &cport, "control");
    
    argc -= optind;
    argv += optind;
  
    if (argv[0])  {
	strncpy(req_str, argv[0], sizeof(req_str));
	DEB( fprintf(stderr,"--- request is \"%s\"\n", req_str) );
    } else {
	badcmdline("");
    }

    caddr = inet_addr(ctrladdr);

    if (unicast) {
	/***
	 *** extract hostip from the request, use that as
	 *** the address for the control session
	 ***/
	caddr = url_addr(req_str) ;
	if (caddr == 0) {
	    fprintf(stderr, "sorry, no server address available for %s\n",
		req_str);
	    exit(1);
	}
	fprintf(stderr, " addr is %s\n",
		inet_ntoa(*(struct in_addr *)&caddr) );
    }

    fec_code = NULL ;
    eventlist = createevlist();

#ifdef LOGGING
    if (NULL==(logf=fopen("logfile", "w"))) {
	perror("fopen");
	exit(1);
    }
#endif

    rxs.myid = compute_id();
    rxs.myrequest = compute_id();

    /* request (out) mcast socket */
    ctrls = openssock(caddr, cport, ttl);

    /* response (in) udp socket */
    localname(&s, ctrls);
    myaddr = s.sin_addr.s_addr ;
    reqin_p = 0;
    reqin = openrsock(INADDR_ANY, &reqin_p);

    insertchan(eventlist, reqin, ID_GETRESPONSE, get_response, NULL);

    r = add_magic(req_buf);
    r = add_opt1(r, RMDP_T_NEW_REQ);
    if (unicast)
	r = add_opt1(r, RMDP_T_R_UNICAST);
    r = add_opt4(r, RMDP_T_R_ID, rxs.myrequest);
    r = add_opt4(r, RMDP_T_C_ID, rxs.myid);
    r = add_opt4(r, RMDP_T_R_RATE, htonl(rxs.data_rate));
    r = add_opt4(r, RMDP_T_R_ADDR, myaddr);
    r = add_opt2(r, RMDP_T_R_PORT, reqin_p);
    r = add_opts(r, RMDP_T_R_FILE, req_str);
    r = add_opt1(r, RMDP_T_END);
    send_r_a.request = req_buf;
    send_r_a.req_len = r - req_buf ;

    response_received = 0;
    send_r_a.s = ctrls;
    fprintf(stderr, "--- schedule request...\n");
    {   struct timeval tv;
	tv.tv_sec = 5;
	tv.tv_usec = 0;
	set_default_timeout(eventlist, &tv);
    }
    settimer(eventlist, ID_SENDREQ, 0, (CALLBACK)send_request, &send_r_a, 1);

    while (EV_TERMINATE != (id = getevent(eventlist, 1))) {
	switch(id) {
	case EV_DEFTIMEOUT :
	    if (response_received && rxs.session_id) {
		rxs.retries++;
		if (rxs.retries >= MAX_RETRY) {
		    /* should we give up ? */
		    fprintf(stderr, "sorry couldn't complete the reception!\n");
		    exit(0);
		}
		/*** change default timeout... XXX ***/
		seen_last_round = 1;
		ask_cont(NULL);
	    } else {
		DEB(fprintf(stderr,"got timeout -- shouldn't happen!\n"));
		exit(1);
	    }
	    break;
	case ID_GETDATA :
	    break;
	default:
	    fprintf(stderr,"got event %d\n", id);
	    break;
	}
    }
    exit(0);
}
